/**
 * @file BtsSmfWrapper.cpp
 *
 * @par SW-Component
 * StateMachine
 *
 * @brief StateMachine wrapper.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Definition for StateMachine wrapper.
 */

#include "BtsSmfWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"
#include "FwErrorCodes.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_SM
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BtsSmfWrapper.cpp.trc.h"
#endif
#endif

namespace btstackif {

SmfWrapper::SmfWrapper()
{
   _createDone = false;
   _bts2IpcList.reserve(10); // reserve memory for 10 entries
   _bts2AppResultList.reserve(10); // reserve memory for 10 entries
   _bts2AppStatusList.reserve(10); // reserve memory for 10 entries
   _messageItem.reset();
   _messageItem.item.opCode = App2BtsOC_None;
   disableAllLocks();
}

SmfWrapper::~SmfWrapper()
{
   _bts2IpcList.clear();
   _bts2AppResultList.clear();
   _bts2AppStatusList.clear();
}

void SmfWrapper::resetSm(void)
{
   createSm();
   Init();
}

bool SmfWrapper::doSmProcessing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN BtStackIfCallback* callback /*= NULL*/, IN const BTSSessionHandle sessionHandle /*= 0*/)
{
   tResult result;
   bool msgProcessed = true;
   bool finished = false;

   while(true == msgProcessed)
   {
      msgProcessed = false;
      // ETG_TRACE_USR1((" doSmProcessing(): START"));
      result = StateMachine_Main_Passive(msgProcessed);
      /*
       *  0: SUCCESS
       * 14: FW_ERR_SM_NOT_CONSUMED
       * -2: STATE_MACHINE_FINISHED
       */
      if(0 == result)
      {
         // SUCCESS => OK
         ETG_TRACE_USR4((" doSmProcessing(): result=SUCCESS msgProcessed=%d", msgProcessed));
      }
      else if(STATE_MACHINE_FINISHED == result)
      {
         // STATE_MACHINE_FINISHED => OK
         ETG_TRACE_USR4((" doSmProcessing(): result=STATE_MACHINE_FINISHED msgProcessed=%d", msgProcessed));
         msgProcessed = false;
         finished = true;
      }
      else if(FW_ERR_SM_NOT_CONSUMED == result)
      {
         // FW_ERR_SM_NOT_CONSUMED => NOK (we have to check)
         ETG_TRACE_ERR((" doSmProcessing(): result=FW_ERR_SM_NOT_CONSUMED msgProcessed=%d", msgProcessed));
         FW_NORMAL_ASSERT_ALWAYS();
      }
      else
      {
         // any other unexpected error => NOK (we have to check)
         ETG_TRACE_ERR((" doSmProcessing(): result=%d msgProcessed=%d", result, msgProcessed));
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   // copy data
   bts2IpcMsgList.reserve(bts2IpcMsgList.size() + _bts2IpcList.size());
   bts2AppMsgList.reserve(bts2AppMsgList.size() + _bts2AppStatusList.size() + _bts2AppResultList.size());
   bts2IpcMsgList.insert(bts2IpcMsgList.end(), _bts2IpcList.begin(), _bts2IpcList.end());
   bts2AppMsgList.insert(bts2AppMsgList.end(), _bts2AppStatusList.begin(), _bts2AppStatusList.end()); // user and session handle have default values
   if(NULL != callback)
   {
      for(size_t i = 0; i < _bts2AppResultList.size(); i++)
      {
         if(NULL != _bts2AppResultList[i])
         {
            _bts2AppResultList[i]->setCallback(callback);
            _bts2AppResultList[i]->setSessionHandle(sessionHandle);
            bts2AppMsgList.push_back(_bts2AppResultList[i]);
         }
      }
   }
   else
   {
      for(size_t i = 0; i < _bts2AppResultList.size(); i++)
      {
         if(NULL != _bts2AppResultList[i])
         {
            delete _bts2AppResultList[i];
         }
      }
   }
   messageItem = _messageItem;

   // clear internal data
   _bts2IpcList.clear();
   _bts2AppResultList.clear();
   _bts2AppStatusList.clear();
   _messageItem.reset();
   _messageItem.item.opCode = App2BtsOC_None;

   return finished;
}

void SmfWrapper::createSm(void)
{
   if(false == _createDone)
   {
      _createDone = true;
      Create();
   }
}

void SmfWrapper::resetSmfWrapperData(void)
{
   _bts2IpcList.clear();
   _bts2AppResultList.clear();
   _bts2AppStatusList.clear();
}

void SmfWrapper::addBts2IpcMsg(IN Bts2Ipc_BaseMessage* message)
{
   if(NULL == message)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _bts2IpcList.push_back(message);
}

void SmfWrapper::addBts2AppMsg(IN Bts2App_BaseMessage* message, IN const bool isResultMsg)
{
   if(NULL == message)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(true == isResultMsg)
   {
      _bts2AppResultList.push_back(message);
   }
   else
   {
      _bts2AppStatusList.push_back(message);
   }
}

} //btstackif
