/**
 * @file BluetoothResetPin.cpp
 *
 * @par SW-Component
 * Base
 *
 * @brief Bluetooth Reset Pin.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of Bluetooth reset pin handling.
 */

#include <cstdlib>
#include <cstring>
#include <cerrno>
#include <sys/wait.h>
#include <unistd.h> // unix standard header, no C++ header available

#include "BluetoothResetPin.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BluetoothResetPin.cpp.trc.h"
#endif
#endif

namespace btstackif {

BluetoothResetPin::BluetoothResetPin()
{
   _pinId = 36; // default pin in G3G based projects
}

BluetoothResetPin::BluetoothResetPin(IN const unsigned int pinId)
{
   _pinId = pinId;
}

BluetoothResetPin::~BluetoothResetPin()
{
}

void BluetoothResetPin::setPinId(IN const unsigned int pinId)
{
   _pinId = pinId;
}

void BluetoothResetPin::performReset(void)
{
#ifdef GEN3X86
   // do all stuff for LSIM
   // we have to use UMFT2232H executable
   int result = system("/opt/bosch/connectivity/bluetoothservice/UMFT2232H >> /tmp/start_btservice.log");

   // check result
   if(0 <= result)
   {
      // success
      result = WEXITSTATUS(result);

      if(0 == result)
      {
         ETG_TRACE_USR1((" performReset(): system call success"));
      }
      else
      {
         ETG_TRACE_USR1((" performReset(): system call failed"));
      }
   }
   else
   {
      // failed
      int errNumber = errno;

      // print the error message
      ETG_TRACE_USR1((" performReset(): system call failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
   }

   // wait 100ms
   (void)usleep((useconds_t)(1000 * 100));

#else
   // do all stuff for target

#endif
#if 0
   void reset_cycle_bluetooth_chip(void)
   {
      FILE *fd;
      guint16 error_code = 0;
      struct stat sts;

      DEBUG("Performing reset cycle of the Bluetooth chip...");

      DEBUG_HIGH("is the GPIO line already exported ?");

      if (!((stat("/sys/class/gpio/gpio36", &sts) != -1) &&
                     S_ISDIR(sts.st_mode))) {

         /* no directory so try to enable the GPIO */
         DEBUG_HIGH("enabling the GPIO reset line");

         fd = fopen("/sys/class/gpio/export", "wb");

         if (fd == NULL) {
            error_code = 1;
            DEBUG("Reset cycle failed with code: %d", error_code);
            return;
         }

         fwrite("36", sizeof(char), 3, fd);

         fclose(fd);
      }

      DEBUG_HIGH("setting GPIO direction to output");

      fd = fopen("/sys/class/gpio/gpio36/direction", "wb");

      if (fd == NULL) {
         error_code = 2;
         DEBUG("Reset cycle failed with code: %d", error_code);
         return;
      }

      fwrite("out", sizeof(char), 3, fd);

      fclose(fd);
   #if 1 //Testing for Audio issue, Enabling now requested by volker
      DEBUG_HIGH("setting GPIO reset line to High for testing");

         fd = fopen("/sys/class/gpio/gpio36/value", "wb");

         if (fd == NULL) {
            error_code = 3;
            DEBUG("Reset cycle failed with code: %d", error_code);
            return;
         }

         fwrite("1", sizeof(char), 1, fd);

         fclose(fd);

         DEBUG_HIGH("keep reset High for 100ms then performs usal intialization by doing one more reset");
         usleep(100000);
   #endif

      DEBUG_HIGH("setting GPIO reset line to low");

      fd = fopen("/sys/class/gpio/gpio36/value", "wb");

      if (fd == NULL) {
         error_code = 3;
         DEBUG("Reset cycle failed with code: %d", error_code);
         return;
      }

      fwrite("0", sizeof(char), 1, fd);

      fclose(fd);

      /* FIXME: Reduce this period to a minimum */
      DEBUG_HIGH("keep reset low for 100ms");
      usleep(100000);

      DEBUG_HIGH("setting GPIO reset line to high");

      fd = fopen("/sys/class/gpio/gpio36/value", "wb");

      if (fd == NULL) {
         error_code = 4;
         DEBUG("Reset cycle failed with code: %d", error_code);
         return;
      }

      fwrite("1", sizeof(char), 1, fd);

      fclose(fd);

      DEBUG_FUNC("Exited");

      return;
   }
#endif
}

} //btstackif
