/**
 * @file StateMachineBase.h
 *
 * @par SW-Component
 * StateMachine
 *
 * @brief State machine base.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details StateMachine base class.
 */

#ifndef _STATE_MACHINE_BASE_H_
#define _STATE_MACHINE_BASE_H_

#include "IStateMachineBase.h"

namespace btstackif {

/**
 *
 */
class StateMachineBase : public IStateMachineBase
{
public:
   /**
    * Constructor.
    */
   StateMachineBase();

   /**
    * Destructor.
    */
   virtual ~StateMachineBase();

   /**
    * Reset internal data.
    */
   virtual void reset(void);

   /**
    * Set request active flag.
    *
    * @param[in] enable: flag
    */
   virtual void setRequestActive(IN const bool enable);

   /**
    * Get request active flag.
    *
    * @return = request active flag
    */
   virtual bool getRequestActive(void) const;

   /**
    * Sets App2Bts compare item.
    *
    * @param[in] item: compare item
    */
   virtual void setApp2BtsCompareItem(IN const BTSApp2BtsMessageCompareItem& item);

   /**
    * Gets App2Bts compare item.
    *
    * @param[out] item: compare item
    */
   virtual void getApp2BtsCompareItem(OUT BTSApp2BtsMessageCompareItem& item) const;

   /**
    * Gets App2Bts compare item.
    *
    * @return = compare item
    */
   virtual const BTSApp2BtsMessageCompareItem& getApp2BtsCompareItem(void) const;

   /**
    * Send status update message. Update is sent to all.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    */
   virtual void updateStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList) = 0;

   /**
    * Send status update message. Update is sent to specific user. If specific user is NULL update is sent to all.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] ptrUser: user
    * @param[in] sessionHandle: session handle
    */
   virtual void updateStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) = 0;

   /**
    * Send status update message. Use given status code if suitable. If specific user is NULL update is sent to all.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] ptrUser: user
    * @param[in] sessionHandle: session handle
    * @param[in] statusCode: status code to be used for update (if suitable)
    */
   virtual void updateStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle, IN const BTSCommonEnumClass statusCode) = 0;

   /**
    * Send status update as well as result message. If specific user is NULL status update is sent to all and no result message is sent.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] ptrUser: user
    * @param[in] sessionHandle: session handle
    * @param[in] sendStatusToAll: flag indicating if status shall be send to all
    */
   virtual void updateStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle, IN const bool sendStatusToAll) = 0;

   /**
    * Send status update as well as result message. Use given result and status code if suitable. If specific user is NULL status update is sent to all and no result message is sent.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] ptrUser: user
    * @param[in] sessionHandle: session handle
    * @param[in] sendStatusToAll: flag indicating if status shall be send to all
    * @param[in] resultCode: result code to be used for update (if suitable)
    * @param[in] statusCode: status code to be used for update (if suitable)
    */
   virtual void updateStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) = 0;

protected:
   bool _requestActive; /**< flag indicating active request */
   BTSApp2BtsMessageCompareItem _app2BtsCompareItem; /**< compare item to be used to identify a message */
};

/*
 * inline methods
 */
inline void StateMachineBase::setRequestActive(IN const bool enable) { _requestActive = enable; }
inline bool StateMachineBase::getRequestActive(void) const { return _requestActive; }
inline void StateMachineBase::setApp2BtsCompareItem(IN const BTSApp2BtsMessageCompareItem& item) { _app2BtsCompareItem = item; }
inline void StateMachineBase::getApp2BtsCompareItem(OUT BTSApp2BtsMessageCompareItem& item) const { item = _app2BtsCompareItem; }
inline const BTSApp2BtsMessageCompareItem& StateMachineBase::getApp2BtsCompareItem(void) const { return _app2BtsCompareItem; }

} //btstackif

#endif //_STATE_MACHINE_BASE_H_
