/**
 * @file BtsTimer.h
 *
 * @par SW-Component
 * Timer
 *
 * @brief Timer.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Timer handling. This timer handling is not thread safe.
 */

#ifndef _BTS_TIMER_H_
#define _BTS_TIMER_H_

#include "BtsTimerBase.h"

namespace btstackif {

// class forward declarations
class ITimeoutHandler;

/**
 *
 */
class Timer : public TimerBase
{
public:
   /**
    * Default constructor.
    */
   Timer();

   /**
    * Destructor.
    */
   virtual ~Timer();

   /**
    * Set timeout value in ms.
    *
    * @param[in] timeout: timeout in ms
    */
   inline void setTimeout(IN const BTSTimeValue timeout) { _timeout = timeout; }

   /**
    * Get timeout value in ms.
    *
    * @return = timeout in ms
    */
   inline BTSTimeValue getTimeout(void) const { return _timeout; }

   /**
    * Checks if timer is active.
    *
    * @return = timer active flag
    */
   inline bool isActive(void) const { return _active; }

   /**
    * Set callback function.
    *
    * @param[in] callback: callback function
    */
   inline void setCallback(IN BTSTimerCallBack callback) { _callback = callback; }

   /**
    * Set timeout function.
    *
    * @param[in] handler: timeout function
    */
   inline void setTimeoutHandler(IN ITimeoutHandler* handler) { _timeoutHandler = handler; }

   /**
    * Set cyclic flag.
    *
    * @param[in] cyclic: cyclic flag
    */
   inline void setCyclicFlag(IN const bool cyclic) { _cyclic = cyclic; }

   /**
    * Get cyclic flag.
    *
    * @return = cyclic flag
    */
   inline bool getCyclicFlag(void) const { return _cyclic; }

   /**
    * Set timer context.
    *
    * @param[in] context: timer context
    */
   inline void setContext(IN const unsigned int context) { _context = context; }

   /**
    * Start/restart timer.
    */
   void start(void);

   /**
    * Start/restart timer.
    *
    * @param[in] cyclic: cyclic flag
    */
   void start(IN const bool cyclic);

   /**
    * Start/restart timer.
    *
    * @param[in] timeout: timeout in ms
    */
   void start(IN const BTSTimeValue timeout);

   /**
    * Start/restart timer.
    *
    * @param[in] timeout: timeout in ms
    * @param[in] cyclic: cyclic flag
    */
   void start(IN const BTSTimeValue timeout, IN const bool cyclic);

   /**
    * Start/restart timer.
    *
    * @param[in] callback: callback function
    */
   void start(IN BTSTimerCallBack callback);

   /**
    * Start/restart timer.
    *
    * @param[in] handler: timeout function
    */
   void start(IN ITimeoutHandler* handler);

   /**
    * Start/restart timer.
    *
    * @param[in] callback: callback function
    * @param[in] cyclic: cyclic flag
    */
   void start(IN BTSTimerCallBack callback, IN const bool cyclic);

   /**
    * Start/restart timer.
    *
    * @param[in] handler: timeout function
    * @param[in] cyclic: cyclic flag
    */
   void start(IN ITimeoutHandler* handler, IN const bool cyclic);

   /**
    * Start/restart timer.
    *
    * @param[in] timeout: timeout in ms
    * @param[in] callback: callback function
    */
   void start(IN const BTSTimeValue timeout, IN BTSTimerCallBack callback);

   /**
    * Start/restart timer.
    *
    * @param[in] timeout: timeout in ms
    * @param[in] handler: timeout function
    */
   void start(IN const BTSTimeValue timeout, IN ITimeoutHandler* handler);

   /**
    * Start/restart timer.
    *
    * @param[in] timeout: timeout in ms
    * @param[in] callback: callback function
    * @param[in] cyclic: cyclic flag
    */
   void start(IN const BTSTimeValue timeout, IN BTSTimerCallBack callback, IN const bool cyclic);

   /**
    * Start/restart timer.
    *
    * @param[in] timeout: timeout in ms
    * @param[in] handler: timeout function
    * @param[in] cyclic: cyclic flag
    */
   void start(IN const BTSTimeValue timeout, IN ITimeoutHandler* handler, IN const bool cyclic);

   /**
    * Stop timer.
    */
   void stop(void);

   /**
    * Get timer id.
    *
    * @return = timer id
    */
   inline BTSTimerId getTimerId(void) const { return _id; }

private:
   BTSTimeValue _timeout; /**< timeout in ms */
   BTSTimerCallBack _callback; /**< callback function */
   ITimeoutHandler* _timeoutHandler; /**< timeout function */
   BTSTimerId _id; /**< timer id */
   bool _cyclic; /**< flag for cyclic timer */
};

} //btstackif

#endif //_BTS_TIMER_H_
