/**
 * @file Bts2Ipc_BaseMessage.h
 *
 * @par SW-Component
 * Messages
 *
 * @brief BtStackIf to IPC messages.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Base message declaration for BtStackIf to IPC messages.
 */

#ifndef _BTS_2_IPC_BASE_MESSAGE_H_
#define _BTS_2_IPC_BASE_MESSAGE_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

/**
 *
 */
class Bts2Ipc_BaseMessage
{
public:
   /**************************************************************************
    Constructor / destructor
    **************************************************************************/
   /**
    * Constructor. Do not use!!!
    */
   Bts2Ipc_BaseMessage();

   /**
    * Constructor.
    *
    * @param[in] functionBlock: function block
    * @param[in] opCode: opcode
    */
   Bts2Ipc_BaseMessage(IN const BTSCommonEnumClass functionBlock, IN const BTSCommonEnumClass opCode);

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   Bts2Ipc_BaseMessage(IN const Bts2Ipc_BaseMessage& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   Bts2Ipc_BaseMessage& operator=(IN const Bts2Ipc_BaseMessage& ref);

   /**
    * Destructor.
    */
   virtual ~Bts2Ipc_BaseMessage();

   /**
    * Trace function for output parameters.
    */
   virtual void doOutputTrace(void);

   /**
    * Returns function block identifier.
    *
    * @return = function block identifier
    */
   inline BTSCommonEnumClass getFunctionBlock(void) const { return _functionBlock; }

   /**
    * Returns message opcode.
    *
    * @return = message opcode
    */
   inline BTSCommonEnumClass getOpCode(void) const { return _opCode; }

   /**
    * Returns message opcode for tracing.
    *
    * @return = message opcode
    */
   inline BTSTraceOpCode getTraceOpCode(void) const { return (BTSTraceOpCode)_opCode; }

   /**
    * Set BTS source function block identifier.
    *
    * @param[in] btsFunctionBlock: BTS function block
    */
   inline void setBtsSourceFunctionBlock(IN const BTSFunctionBlock btsFunctionBlock) { _btsSourceFunctionBlock = btsFunctionBlock; }

   /**
    * Returns BTS source function block identifier.
    *
    * @return = BTS source function block identifier
    */
   inline BTSFunctionBlock getBtsSourceFunctionBlock(void) const { return _btsSourceFunctionBlock; }

   /**
    * Set BTS opcode.
    *
    * @param[in] opCode: BTS opcode
    */
   inline void setApp2BtsOpcode(IN const BTSApp2BtsOpcode opCode) { _app2BtsOpcode = opCode; }

   /**
    * Returns BTS opcode.
    *
    * @return = BTS opcode
    */
   inline BTSApp2BtsOpcode getApp2BtsOpcode(void) const { return _app2BtsOpcode; }

   /**
    * Returns BTS opcode for tracing.
    *
    * @return = BTS opcode
    */
   inline BTSTraceOpCode getTraceApp2BtsOpCode(void) const { return (BTSTraceOpCode)_app2BtsOpcode; }

   /**
    * Sets component.
    *
    * @param[in] component: component
    */
   inline void setComponent(IN const BTSFunctionBlock component) { _component = component; }

   /**
    * Returns component.
    *
    * @return = component
    */
   inline BTSFunctionBlock getComponent(void) const { return _component; }

   /**
    * Sets stack interface.
    *
    * @param[in] stackInterface: stack interface
    */
   inline void setStackInterface(IN const BTSInterfaceType stackInterface) { _stackInterface = stackInterface; }

   /**
    * Returns stack interface
    *
    * @return = stack interface
    */
   inline BTSInterfaceType getStackInterface(void) const { return _stackInterface; }

   /**
    * Sets BT device address.
    *
    * @param[in] deviceAddress: BT device address
    */
   inline void setBDAddress(IN const BTSBDAddress& deviceAddress) { _deviceAddress = deviceAddress; }

   /**
    * Gets BT device address.
    *
    * @param[out] deviceAddress: BT device address
    */
   inline void getBDAddress(OUT BTSBDAddress& deviceAddress) const { deviceAddress = _deviceAddress; }

   /**
    * Gets BT device address.
    *
    * @return = BT device address
    */
   inline const BTSBDAddress& getBDAddress(void) const { return _deviceAddress; }

   /**
    * Gets BT device address.
    *
    * @return = BT device address
    */
   inline BTSBDAddress& getBDAddressMutable(void) { return _deviceAddress; }

   /**
    * Sets object path.
    *
    * @param[in] objectPath: object path
    */
   inline void setObjectPath(IN const BTSObjectPath& objectPath) { _objectPath = objectPath; }

   /**
    * Gets object path.
    *
    * @param[out] objectPath: object path
    */
   inline void getObjectPath(OUT BTSObjectPath& objectPath) const { objectPath = _objectPath; }

   /**
    * Gets object path.
    *
    * @return = object path
    */
   inline const BTSObjectPath& getObjectPath(void) const { return _objectPath; }

   /**
    * Returns copy of the message object.
    *
    * @return = copy of the message object (pointer, created by new)
    */
   virtual Bts2Ipc_BaseMessage* clone(void) const = 0;

   /**
    * Sets D-Bus token (act).
    *
    * @param[in] token: D-Bus token (act)
    */
   inline void setDbusToken(IN const act_t token) { _token = token; }

   /**
    * Gets D-Bus token (act).
    *
    * @return = D-Bus token (act)
    */
   inline act_t getDbusToken(void) const { return _token; }

   /**
    * Sets App2Bts compare item.
    *
    * @param[in] item: compare item
    */
   inline void setApp2BtsCompareItem(IN const BTSApp2BtsMessageCompareItem& item) { _app2BtsCompareItem = item; }

   /**
    * Gets App2Bts compare item.
    *
    * @param[out] item: compare item
    */
   inline void getApp2BtsCompareItem(OUT BTSApp2BtsMessageCompareItem& item) const { item = _app2BtsCompareItem; }

   /**
    * Gets App2Bts compare item.
    *
    * @return = compare item
    */
   inline const BTSApp2BtsMessageCompareItem& getApp2BtsCompareItem(void) const { return _app2BtsCompareItem; }

   /**
    * Sets flag for response message.
    *
    * @param[in] enable: flag for response message
    */
   inline void setResponseMessageFlag(IN const bool enable) { _isResponseMessage = enable; }

   /**
    * Gets flag for response message.
    *
    * @return = flag for response message
    */
   inline bool getResponseMessageFlag(void) const { return _isResponseMessage; }

   /**
    * Sets timeout value.
    *
    * @param[in] value: timeout value
    */
   inline void setTimeout(IN const BTSTimeValue value) { _timeout = value; }

   /**
    * Gets timeout value.
    *
    * @return = timeout value
    */
   inline BTSTimeValue getTimeout(void) const { return _timeout; }

   /**
    * Gets timeout value.
    *
    * @return = timeout value
    */
   inline BTSTimeValue getMethodReturnTimeOut(void) const { return _timeout; }

   /**
    * Sets object id.
    *
    * @param[in] objectId: object id
    */
   inline void setObjectId(IN const BTSObjectPath& objectId) { _objectId = objectId; }

   /**
    * Gets object id.
    *
    * @param[out] objectId: object id
    */
   inline void getObjectId(OUT BTSObjectPath& objectId) const { objectId = _objectId; }

   /**
    * Gets object id.
    *
    * @return = object id
    */
   inline const BTSObjectPath& getObjectId(void) const { return _objectId; }

   /**
    * Get message trace id.
    *
    * @return = message trace id
    */
   uint32_t getMessageTraceId(void) const;

   /**
    * Returns number of existing BTS 2 IPC messages.
    *
    * @return = number of existing BTS 2 IPC messages
    */
   static inline unsigned int getBts2IpcMessageCount(void) { return _countBts2IpcMessages; }

protected:
   BTSCommonEnumClass _functionBlock; /**< identifies the function block the message relates to, depends on used supplier Bluetooth stack */

   BTSCommonEnumClass _opCode; /**< identifies the message itself, depends on used supplier Bluetooth stack */

   BTSFunctionBlock _btsSourceFunctionBlock; /**< identifies the BTS source function block that created this message */

   BTSApp2BtsOpcode _app2BtsOpcode; /**< identifies the BTS opcode that causes this message */

   BTSFunctionBlock _component; /**< component (function block) */

   BTSInterfaceType _stackInterface; /**< stack interface to be used (Bluetooth stack) */

   BTSBDAddress _deviceAddress; /**< parameter device address */

   BTSObjectPath _objectPath; /**< object path e.g. for device */

   act_t _token; /**< token to be used to identify a callback invocation */

   BTSApp2BtsMessageCompareItem _app2BtsCompareItem; /**< compare item to be used to identify a message */

   bool _isResponseMessage; /**< flag indicating if this is a response message or not */

   BTSTimeValue _timeout; /**< timeout value for receiving response/answer of this request */

   BTSObjectPath _objectId; /**< used to identify a message that belongs to a proxy using the object path */

   static unsigned int _countBts2IpcMessages; /**< number of existing BTS 2 IPC messages */
};

} //btstackif

#endif //_BTS_2_IPC_BASE_MESSAGE_H_
