/**
 * @file App2Bts_BaseMessage.h
 *
 * @par SW-Component
 * Messages
 *
 * @brief Application to BtStackIf messages.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Base message declaration for Application to BtStackIf messages.
 */

#ifndef _APP_2_BTS_BASE_MESSAGE_H_
#define _APP_2_BTS_BASE_MESSAGE_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class BtStackIfCallback;

/**
 *
 */
class App2Bts_BaseMessage
{
public:
   /**************************************************************************
    Constructor / destructor
    **************************************************************************/
   /**
    * Default constructor. Do not use!!!
    */
   App2Bts_BaseMessage();

   /**
    * Constructor.
    *
    * @param[in] messageType: message type
    */
   App2Bts_BaseMessage(IN const BTSFunctionBlock functionBlock, IN const BTSApp2BtsOpcode opCode);

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   App2Bts_BaseMessage(IN const App2Bts_BaseMessage& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   App2Bts_BaseMessage& operator=(IN const App2Bts_BaseMessage& ref);

   /**
    * Destructor.
    */
   virtual ~App2Bts_BaseMessage();

   /**
    * Trace function for input parameters.
    */
   virtual void doInputTrace(void);

   /**
    * Returns function block identifier.
    *
    * @return = function block identifier
    */
   inline BTSFunctionBlock getFunctionBlock(void) const { return _functionBlock; }

   /**
    * Returns message opcode.
    *
    * @return = message opcode
    */
   inline BTSApp2BtsOpcode getOpCode(void) const { return _opCode; }

   /**
    * Returns message opcode for tracing.
    *
    * @return = message opcode
    */
   inline BTSTraceOpCode getTraceOpCode(void) const { return (BTSTraceOpCode)_opCode; }

   /**
    * Sets user reference. NOTE: replace by setCallback.
    *
    * @param[in] ptrUser: user reference
    */
   inline void setUser(IN BtStackIfCallback* ptrUser) { _callback = ptrUser; }

   /**
    * Returns user reference. NOTE: replace by getCallback.
    *
    * @return = user reference
    */
   inline BtStackIfCallback* getUser(void) const { return _callback; }

   /**
    * Sets callback reference.
    *
    * @param[in] callback: callback reference
    */
   inline void setCallback(IN BtStackIfCallback* callback) { _callback = callback; }

   /**
    * Returns callback reference.
    *
    * @return = callback reference
    */
   inline BtStackIfCallback* getCallback(void) { return _callback; }

   /**
    * Sets session handle.
    *
    * @param[in] sessionHandle: session handle
    */
   inline void setSessionHandle(IN const BTSSessionHandle sessionHandle) { _sessionHandle = sessionHandle; }

   /**
    * Returns session handle.
    *
    * @return = session handle
    */
   inline BTSSessionHandle getSessionHandle(void) const { return _sessionHandle; }

   /**
    * Sets component.
    *
    * @param[in] component: component
    */
   inline void setComponent(IN const BTSFunctionBlock component) { _component = component; }

   /**
    * Returns component.
    *
    * @return = component
    */
   inline BTSFunctionBlock getComponent(void) const { return _component; }

   /**
    * Sets stack interface.
    *
    * @param[in] stackInterface: stack interface
    */
   inline void setStackInterface(IN const BTSInterfaceType stackInterface) { _stackInterface = stackInterface; }

   /**
    * Returns stack interface
    *
    * @return = stack interface
    */
   inline BTSInterfaceType getStackInterface(void) const { return _stackInterface; }

   /**
    * Sets BT device address.
    *
    * @param[in] deviceAddress: BT device address
    */
   inline void setBDAddress(IN const BTSBDAddress& deviceAddress) { _deviceAddress = deviceAddress; }

   /**
    * Gets BT device address.
    *
    * @param[out] deviceAddress: BT device address
    */
   inline void getBDAddress(OUT BTSBDAddress& deviceAddress) const { deviceAddress = _deviceAddress; }

   /**
    * Gets BT device address.
    *
    * @return = BT device address
    */
   inline const BTSBDAddress& getBDAddress(void) const { return _deviceAddress; }

   /**
    * Gets BT device address.
    *
    * @return = BT device address
    */
   inline BTSBDAddress& getBDAddressMutable(void) { return _deviceAddress; }

   /**
    * Returns copy of the message object.
    *
    * @return = copy of the message object (pointer, created by new)
    */
   virtual App2Bts_BaseMessage* clone(void) const = 0;

   /**
    * Gets compare item.
    *
    * @param[out] compareItem: message compare item
    */
   virtual void getCompareItem(OUT BTSApp2BtsMessageCompareItem& compareItem) const;

   /**
    * Set flag for internal message.
    *
    * @param[in] enable: flag for internal message
    */
   inline void setInternalMessageFlag(IN const bool enable) { _internalMsg = enable; }

   /**
    * Get flag for internal message.
    *
    * @return = flag for internal message
    */
   inline bool getInternalMessageFlag(void) const { return _internalMsg; }

   /**
    * Set flag for user data.
    *
    * @param[in] userData: user data
    */
   inline void setUserData(IN const unsigned int userData) { _userData = userData; }

   /**
    * Get flag for user data.
    *
    * @return = user data
    */
   inline unsigned int getUserData(void) const { return _userData; }

   /**
    * Get message trace id.
    *
    * @return = message trace id
    */
   uint32_t getMessageTraceId(void) const;

   /**
    * Get message pointer.
    *
    * @return = message pointer
    */
   App2Bts_BaseMessage* getMessagePtr(void);

   /**
    * Returns number of existing APP 2 BTS messages.
    *
    * @return = number of existing APP 2 BTS messages
    */
   static inline unsigned int getApp2BtsMessageCount(void) { return _countApp2BtsMessages; }

protected:
   BTSFunctionBlock _functionBlock; /**< identifies the function block the message relates to (equals sub component) */

   BTSApp2BtsOpcode _opCode; /**< identifies the message itself */

   BtStackIfCallback* _callback; /**< originator of this request */

   BTSSessionHandle _sessionHandle; /**< parameter session handle */

   BTSFunctionBlock _component; /**< component (function block) */

   BTSInterfaceType _stackInterface; /**< stack interface to be used (Bluetooth stack) */

   BTSBDAddress _deviceAddress; /**< parameter device address */

   bool _internalMsg; /**< flag for internal message */

   unsigned int _userData; /**< user data */

   static unsigned int _countApp2BtsMessages; /**< number of existing APP 2 BTS messages */
};

} //btstackif

#endif //_APP_2_BTS_BASE_MESSAGE_H_
