/**
 * @file FC_Bluetooth_clienthandler_SpmCore.cpp
 * @author
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 *
 * @addtogroup FC_Bluetooth
 *
 * @brief Public interface for CCABluetoothApplication
 *
 * @{
 */

/******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/
#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_TYPES
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_SERVICEINFO
#include "spm_fi_if.h"

#include "FC_Bluetooth_main.h"
#include "FC_Bluetooth_clienthandler_SpmCore.h"
#include "FC_Bluetooth_CCAService.h"
#include "IBmCoreRequestIf.h"
#include "FunctionTracer.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APPLICATION
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FC_Bluetooth_clienthandler_SpmCore.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APPLICATION
#endif // VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

/******************************************************************************/
/*									DEFINES									*/
/******************************************************************************/


/******************************************************************************/
/*									CCA MESSAGE MAP								*/
/******************************************************************************/
BEGIN_MSG_MAP(fc_bluetooth_tclClientHandler_SpmCore,ahl_tclBaseWork)

ON_MESSAGE_SVCDATA(SPM_COREFI_C_U16_BLUETOOTHSTATE, AMT_C_U8_CCAMSG_OPCODE_STATUS, OnBluetoothStateStatus)

END_MSG_MAP()

/*******************************************************************************
*
* FUNCTION     : fc_bluetooth_tclClientHandler_SpmCore::
*                fc_bluetooth_tclClientHandler_SpmCore(fc_phone_tclApp* poMainAppl)
*
* DESCRIPTION  : Constructor.
*
*                Create an object of the base class
*                ahl_tclBaseOneThreadClientHandler with a pointer to this
*                application, the to be used service identifier and the service
*                version as parameters.
*
* PARAMETER    : [IN] poMainAppl = Pointer to the object of this application.
*
* RETURNVALUE  : None.
*
******************************************************************************/
fc_bluetooth_tclClientHandler_SpmCore::fc_bluetooth_tclClientHandler_SpmCore(fc_bluetooth_tclApp* poMainAppl)
    : ahl_tclBaseOneThreadClientHandler(
      /* Application Pointer          */ poMainAppl,
      /* ID of used Service           */ /*CCA_C_U16_SRV_FB_SYSTEMSTATE*/ CCA_C_U16_SRV_SPM,
      /* MajorVersion of used Service */ SPM_COREFI_C_U16_SERVICE_MAJORVERSION,
      /* MinorVersion of used Service */ SPM_COREFI_C_U16_SERVICE_MINORVERSION)
{
	ENTRY

	vAddAutoRegisterForProperty(SPM_COREFI_C_U16_BLUETOOTHSTATE);

	_bluetoothApp = poMainAppl;
	m_u32BTState = (tU32)spm_fi_tcl_SPM_e32_BT_STATE::FI_EN_BLUETOOTHSTATE_NO_ACTION;
}
/*******************************************************************************
*
* FUNCTION     : fc_bluetooth_tclClientHandler_SpmCore::
*                ~fc_bluetooth_tclClientHandler_SpmCore(tVoid)
*
* DESCRIPTION  : Destructor.
*
* PARAMETER    : None.
*
* RETURNVALUE  : None.
*
*******************************************************************************/
fc_bluetooth_tclClientHandler_SpmCore::~fc_bluetooth_tclClientHandler_SpmCore(tVoid)
{
   ENTRY

   _bluetoothApp = 0;
}
/*******************************************************************************
*
* FUNCTION     : tVoid fc_bluetooth_tclClientHandler_SpmCore::vOnServiceAvailable()
*
* DESCRIPTION  : This function is called by the CCA framework when the service
*                this client-handler has registered for has become available.
*
* PARAMETER    : None.
*
* RETURNVALUE  : None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceAvailable().
*******************************************************************************/
tVoid fc_bluetooth_tclClientHandler_SpmCore::vOnServiceAvailable()
{
	ENTRY

	ETG_TRACE_USR2(("_BTSystemState__:vOnServiceAvailable() - AppID = %u.",u16GetServerAppID()));

	vGetBluetoothState();

	return;
}

/*******************************************************************************
*
* FUNCTION     : tVoid fc_bluetooth_tclClientHandler_SpmCore::vOnServiceUnavailable()
*
* DESCRIPTION  : This function is called by the CCA framework when the service
*                this client-handler has registered for has become unavailable.
*
* PARAMETER    : None.
*
* RETURNVALUE  : None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceUnavailable()
*******************************************************************************/
tVoid fc_bluetooth_tclClientHandler_SpmCore::vOnServiceUnavailable()
{
	ENTRY

	ETG_TRACE_USR2(("  _BTSystemState__ :vOnServiceUnavailable() entered. AppID = %u.",u16GetServerAppID()));

	return;
}

tVoid fc_bluetooth_tclClientHandler_SpmCore::vGetBluetoothState()
{
   ENTRY

   u32GetCurrentBluetoothState();

   ETG_TRACE_USR1(("_BTSystemState__ : vGetBluetoothState entered"));

   if (TRUE == bIfServiceAvailable())
   {
      ETG_TRACE_USR1(("_BTSystemState__ : vGetBluetoothState service is available"));

      spm_corefi_tclMsgBluetoothStateStatus oBTStateGet;

      fi_tclVisitorMessage oVisitorMsgOutput(oBTStateGet, SPM_COREFI_C_U16_SERVICE_MAJORVERSION);

      // Here the message related header information is added and by this the
      // creation of the message is completed.
      vInitServiceData(oVisitorMsgOutput, 0, SPM_COREFI_C_U16_BLUETOOTHSTATE, AMT_C_U8_CCAMSG_OPCODE_GET);

      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oVisitorMsgOutput, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vGetBluetoothState:enPostMessage() failed vGetPreferredPhonebookSortOrder message"));
      }

      oBTStateGet.vDestroy();
   }
   else
   {
      ETG_TRACE_ERR(("vGetBluetoothState: No Service available ( GET NOT POSTED for) vGetBluetoothState message)"));
   }

   return;
}

/*******************************************************************************
** FUNCTION    :   OnBluetoothStateStatus(..)
*******************************************************************************/
/* DESCRIPTION :This is a handler function for SystemState PropertyStatus
*
*  PARAMETERS  :
*     IN       :   poMessage: This is the received message from a client
*  RETURNVALUE : None
*******************************************************************************/
tVoid  fc_bluetooth_tclClientHandler_SpmCore::OnBluetoothStateStatus(amt_tclServiceData* poMessage)
{
	ENTRY

	// Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   spm_corefi_tclMsgBluetoothStateStatus oBTStateStatus;
   BmResult result = BM_RESULT_OK;

   if (oVisitorMsg.s32GetData(oBTStateStatus, SPM_COREFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR1(("OnBluetoothStateStatus: oBTStateStatus.BtState.enType :: %d"
            ,oBTStateStatus.BtState.enType));

      m_u32BTState = (tU32) oBTStateStatus.BtState.enType;
      BtSystemState btSystemState;

      switch (oBTStateStatus.BtState.enType)
      {
         case spm_fi_tcl_SPM_e32_BT_STATE::FI_EN_BLUETOOTHSTATE_BT_OFF:
         {
            ETG_TRACE_USR4(("OnBluetoothStateStatus: setting BtSystemState to OFF on SPM BTState change"));
            btSystemState._bluetoothSystemState = BM_BT_SYSTEM_STATE_OFF;
         }
         break;

         case spm_fi_tcl_SPM_e32_BT_STATE::FI_EN_BLUETOOTHSTATE_BLOCK:
         {
            ETG_TRACE_USR4(("OnBluetoothStateStatus: setting BtSystemState to BLOCK on SPM BTState change"));
            btSystemState._bluetoothSystemState = BM_BT_SYSTEM_STATE_BLOCK;
         }
         break;

         case spm_fi_tcl_SPM_e32_BT_STATE::FI_EN_BLUETOOTHSTATE_VALET_MODE:
         case spm_fi_tcl_SPM_e32_BT_STATE::FI_EN_BLUETOOTHSTATE_DISCONNECT:
         {
            ETG_TRACE_USR4(("OnBluetoothStateStatus: setting BtSystemState to DISCONNECT on SPM BTState change"));
            btSystemState._bluetoothSystemState = BM_BT_SYSTEM_STATE_DISCONNECT;
         }
         break;

         case spm_fi_tcl_SPM_e32_BT_STATE::FI_EN_BLUETOOTHSTATE_NORMAL_SWITCH_OFF_BT:
         {
             ETG_TRACE_USR4(("OnBluetoothStateStatus: setting BtSystemState to NORMAL_SWITCH_OFF_BT on SPM BTState change"));
             btSystemState._bluetoothSystemState = BM_BT_SYSTEM_STATE_NORMAL_SWITCH_OFF_BT;
         }
         break;

         case spm_fi_tcl_SPM_e32_BT_STATE::FI_EN_BLUETOOTHSTATE_NORMAL:
         {
            ETG_TRACE_USR4(("OnBluetoothStateStatus: setting BtSystemState to NORMAL on SPM BTState change"));
            btSystemState._bluetoothSystemState = BM_BT_SYSTEM_STATE_NORMAL;
         }
         break;

         case spm_fi_tcl_SPM_e32_BT_STATE::FI_EN_BLUETOOTHSTATE_NO_ACTION:
         {
            ETG_TRACE_USR4(("OnBluetoothStateStatus: setting BtSystemState to NO_ACTION on SPM BTState change"));
            btSystemState._bluetoothSystemState = BM_BT_SYSTEM_STATE_NO_ACTION;
         }
         break;

         default:
         {
            ETG_TRACE_ERR(("OnBluetoothStateStatus: Unhandled case"));
            result = BM_RESULT_ERR_GENERAL;
         }
         break;
      }//!End of switch (oBTStateStatus.BtState.enType)

      if(BM_RESULT_OK == result)
      {
          if (NULL != _bluetoothApp)
          {
             result = _bluetoothApp->getBmCoreRequestIf().setBtSystemState(btSystemState);

             if (BM_RESULT_OK != result)
             {
                ETG_TRACE_ERR(("OnBluetoothStateStatus: setBtSystemState is failed - %d ", result));
             }
          }
          else
          {
             ETG_TRACE_ERR(("OnBluetoothStateStatus: _bluetoothApp Is NULL"));
          }
      }//!End of if(BM_RESULT_OK == result)
   }//!End of if (oVisitorMsg.s32GetData(oBTStateStatus, SPM_COREFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)

   oBTStateStatus.vDestroy();
}



/*******************************************************************************
** FUNCTION:   u32GetCurrentBluetoothState(..)
*******************************************************************************/
/* DESCRIPTION:
*   Current System state
*
*  PARAMETERS:     
*  RETURNVALUE: m_u32BTState : Current System state enum
*******************************************************************************/
tU32  fc_bluetooth_tclClientHandler_SpmCore::u32GetCurrentBluetoothState()
{
   ENTRY

   ETG_TRACE_USR2(("u32GetCurrentBluetoothState : Current SysState is : %d", m_u32BTState));

   return m_u32BTState;
}
