/**
 * @file FC_Bluetooth_clienthandler_SystemState.cpp
 * @author
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 *
 * @addtogroup FC_Bluetooth
 *
 * @brief ClientHandler for the service of the Diagnosic (Diaglog)
 *
 * @{
 */
/******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/
#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"
#include <FC_Bluetooth_rootdaemon_client.h>
#include "FC_Bluetooth_main.h"
#include "FC_Bluetooth_clienthandler_Diaglog.h"
#include "FC_Bluetooth_CCAService.h"
#include "FunctionTracer.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APPLICATION
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FC_Bluetooth_clienthandler_Diaglog.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APPLICATION
#endif // VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

/*******************************************************************************
 Specific defines for this component
 *******************************************************************************/
// Local defines for the wanted version of the CCX_DMSVR_FI
#define DIAGLOG_FI_MAJOR_VERSION  1 //MIDW_CCX_DMSRVFI_C_U16_SERVICE_MAJORVERSION
#define DIAGLOG_FI_MINOR_VERSION  0 //MIDW_CCX_DMSRVFI_C_U16_SERVICE_MINORVERSION

#define VDL_ITC_GMG2_DOWNLOAD_HARD_PERF_INTERN_COMM_FAIL_BT    ITC_BLUETOOTH_SYSTEM_FAILURE
#define VDL_ITC_GMG2_DOWNLOAD_HARD_PERF_INTERN_COMM_FAIL_WIFI  ITC_WIFI_SYSTEM_FAILURE

/*******************************************************************************
 Message map for FunctionIDs to be handled
 *******************************************************************************/
/* Message map for FunctionIds (FIDs) which are handled by this clienthandler */

BEGIN_MSG_MAP(fc_bluetooth_tclClientHandler_Diaglog, ahl_tclBaseWork)

ON_MESSAGE_SVCDATA( MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT,
      AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,
      OnSavetestresultMethodresult )

ON_MESSAGE_SVCDATA( MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT,
      AMT_C_U8_CCAMSG_OPCODE_STATUS,
      OnSendnexttestresultStatus )

END_MSG_MAP()

/*******************************************************************************
 Methods of class
 *******************************************************************************/

/*******************************************************************************
 **  FUNCTION:
 *   fc_bluetooth_tclClientHandler_Diaglog(fc_bluetooth_tclApp* poMainAppl);
 *   Constructor
 *******************************************************************************/
fc_bluetooth_tclClientHandler_Diaglog::fc_bluetooth_tclClientHandler_Diaglog(fc_bluetooth_tclApp* poMainAppl) :
      ahl_tclBaseOneThreadClientHandler(poMainAppl // Main App Pointer
      , CCA_C_U16_SRV_DIAGLOG // Service Id
      , DIAGLOG_FI_MAJOR_VERSION // MajorVersion of Service
      , DIAGLOG_FI_MINOR_VERSION) // MinorVersion of Service
{
   ENTRY

   /* Add automated Upreg/RelUpreg handling for dedicated properties */
   vAddAutoRegisterForProperty(MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT);

   m_bExistTestDataToSend = FALSE;
   m_bStart = TRUE;
   m_s8BluetoothDTCState = BLUETOOTH_DTC_STATE_UNDEFINED;
   m_s8WiFiDTCState = BLUETOOTH_DTC_STATE_UNDEFINED;

   ETG_TRACE_USR2(("fc_bluetooth_tclClientHandler_Diaglog(): => initial state =%d", m_s8BluetoothDTCState));
}

/*******************************************************************************
 ** FUNCTION:   virtual ~fc_bluetooth_tclClientHandler_Diaglog();
 *  Destructor
 *******************************************************************************/
fc_bluetooth_tclClientHandler_Diaglog::~fc_bluetooth_tclClientHandler_Diaglog(tVoid)
{
   ENTRY
   m_s8BluetoothDTCState = BLUETOOTH_DTC_STATE_UNDEFINED;
   m_s8WiFiDTCState = BLUETOOTH_DTC_STATE_UNDEFINED;
}

/*******************************************************************************
 ** FUNCTION:   virtual tVoid vOnServiceAvailable();
 *******************************************************************************/
/* virtual */
// Overrides 'tVoid ahl_tclBaseOneThreadClientHandler::vOnServiceAvailable()'
tVoid fc_bluetooth_tclClientHandler_Diaglog::vOnServiceAvailable()
{
   ENTRY
   //only at the beginning of the application if the service is available has to send the TestResult to the diagnosis
   if (m_bStart)
   {
      ETG_TRACE_USR2(("vOnServiceAvailable: Service of AppID= %u became available.", u16GetServerAppID()));

      // this will also check for status==BLUETOOTH_DTC_STATE_UNDEFINED and set the trigger to send accordingly!
      vCheckAndSaveTestResult(VDL_ITC_GMG2_DOWNLOAD_HARD_PERF_INTERN_COMM_FAIL_BT, m_s8BluetoothDTCState);

      // this will also check for status==BLUETOOTH_DTC_STATE_UNDEFINED and set the trigger to send accordingly!
      vCheckAndSaveTestResult(VDL_ITC_GMG2_DOWNLOAD_HARD_PERF_INTERN_COMM_FAIL_WIFI, m_s8WiFiDTCState);

      //send all testresult to diaglog (only one cca message for all the results)
      vSendTestResultToDiaglog();

      m_bStart = FALSE;
   }
}

/*******************************************************************************
 ** FUNCTION:   virtual tVoid vOnServiceUnavailable();
 *******************************************************************************/
/* virtual */
// Overrides 'tVoid ahl_tclBaseOneThreadClientHandler::vOnServiceUnavailable()'
tVoid fc_bluetooth_tclClientHandler_Diaglog::vOnServiceUnavailable()
{
   ENTRY

   ETG_TRACE_USR2(( "vOnServiceUnavailable: Service of AppID= %u became unavailable.", u16GetServerAppID() ));
}

/*******************************************************************************
 Handler functions for FI messages
 *******************************************************************************/

/*******************************************************************************
 ** FUNCTION:   OnSavetestresultMethodresult(amt_tclServiceData* poMessage)
 *******************************************************************************/
tVoid fc_bluetooth_tclClientHandler_Diaglog::OnSavetestresultMethodresult(amt_tclServiceData* poMessage)
{
   ENTRY

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oResultMsg(poMessage);

   // Create the (message related) FI data object
   midw_diaglogfi_tclMsgSaveTestResultMethodResult oResultData;

   if (oResultMsg.s32GetData(oResultData) != OSAL_ERROR)
   {
      // if control setting accepted, composing positive response
      if (oResultData.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
      {
         ETG_TRACE_USR4(("OnSavetestresultMethodresult(): Accepted"));
      }
      else // test result rejected, composing negative response
      {
         ETG_TRACE_USR4(("OnSavetestresultMethodresult(): Rejected"));
      }
   }

   // Always destroy the FI data object before leaving its creation scope
   oResultData.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   OnSendnexttestresultStatus(..) property
 *******************************************************************************/
tVoid fc_bluetooth_tclClientHandler_Diaglog::OnSendnexttestresultStatus(amt_tclServiceData* poMessage)
{
   ENTRY
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   midw_diaglogfi_tclMsgSendNextTestResultStatus oFiDataObject;

   // Unmarshal the FI visitor message to the given FI data object. (The para-
   // meters are copied individually from shared memory to the FI data object.)
   if (oVisitorMsg.s32GetData(oFiDataObject) != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("Enter OnSendnexttestresultStatus..." ));
      // Extract details from the FI data object
      //midw_fi_tcl_SendNextTestResultStructure NextTestResultStruct;
      /*
       NextTestResultStruct.Status = True if
       client is required to send next test result to DiagLog.
       NextTestResultStruct.ITCList: This is the list of the
       DTCs for which the next test result has to be sent. if the
       List is empty and "NextTestResultStruct.Status == True"
       then the client should report all the DTCs for which he is
       reponsible.
       */
      if (oFiDataObject.NextTestResultStruct.Status == TRUE)
      {
         tBool bSend = FALSE;

         //calculate the testresult only if m_s8BluetoothDTCState is not BLUETOOTH_DTC_STATE_UNDEFINED, i.e. BT-Init was done.

         if (m_s8BluetoothDTCState != BLUETOOTH_DTC_STATE_UNDEFINED)
         {
            // this will also check for status==BLUETOOTH_DTC_STATE_UNDEFINED and set the trigger to send accordingly!
            vCheckAndSaveTestResult(VDL_ITC_GMG2_DOWNLOAD_HARD_PERF_INTERN_COMM_FAIL_BT, m_s8BluetoothDTCState);

            bSend = TRUE;
         }
         else
         {
            ETG_TRACE_USR4(("Bluetooth DTC State is still BLUETOOTH_DTC_STATE_UNDEFINED. No Msg will be sent." ));
         }

         if (m_s8WiFiDTCState != BLUETOOTH_DTC_STATE_UNDEFINED)
         {
            // this will also check for status==BLUETOOTH_DTC_STATE_UNDEFINED and set the trigger to send accordingly!
            vCheckAndSaveTestResult(VDL_ITC_GMG2_DOWNLOAD_HARD_PERF_INTERN_COMM_FAIL_WIFI, m_s8WiFiDTCState);

            bSend = TRUE;
         }
         else
         {
            ETG_TRACE_USR4(("WIFI DTC State is still BLUETOOTH_DTC_STATE_UNDEFINED. No Msg will be sent." ));
         }

         if (bSend) //send all testresult to diaglog (only one cca message for all the results)
            vSendTestResultToDiaglog();
      }
      // Status = false if this is a confirmation message from the Diaglog that the
      //UpReg has been succesful -> Do nothing
   }
   else
   {
      ETG_TRACE_ERR(( "vHandle_SetElapsedTimerRunmode_MethodResultLast: Received message invalid." ));
   }

   // Always destroy the FI data object before leaving its creation scope
   oFiDataObject.vDestroy();
}

/*******************************************************************************
 Specific functions of this application
 *******************************************************************************/

/*******************************************************************************
 ** FUNCTION:   tVoid vCheckAndSaveTestResult();
 * Save the Test Results to the CCA Msg for later sending to Diaglog
 *******************************************************************************/
tVoid fc_bluetooth_tclClientHandler_Diaglog::vCheckAndSaveTestResult(tU16 u16Dtc, tS8 s8State)
{
   ENTRY
   tBool bResult = FALSE;

   switch (u16Dtc)
   {
      case VDL_ITC_GMG2_DOWNLOAD_HARD_PERF_INTERN_COMM_FAIL_BT:
         switch (s8State)
         {
            case BLUETOOTH_DTC_STATE_OK_STARTUP:
            case BLUETOOTH_DTC_STATE_OK_NORMAL_OPERATION:
               bResult = TRUE;
               m_bExistTestDataToSend = TRUE;
            break;
            case BLUETOOTH_DTC_STATE_UNDEFINED:
               // DO NOT SAVE, I.E. DO NOT SEND DTC MESSAGE !!!
            break;
            default:
               bResult = FALSE;
               m_bExistTestDataToSend = TRUE;
            break;
         }
      break;

      case VDL_ITC_GMG2_DOWNLOAD_HARD_PERF_INTERN_COMM_FAIL_WIFI:
      {
         switch (s8State)
         {
            case WIFI_DTC_STATE_OK_STARTUP:
               bResult = TRUE;
               m_bExistTestDataToSend = TRUE;
            break;
            case WIFI_DTC_STATE_ERROR_STARTUP:
               bResult = FALSE;
               m_bExistTestDataToSend = TRUE;
            break;
            default:
               ETG_TRACE_ERR(("WiFi DTC default state"));
            break;
         }
      }
      break;

      default:
      break;
   }

   if (m_bExistTestDataToSend == TRUE)
   {

      midw_fi_tcl_TestResult oTestData;
      // Set the parameters of the FI data object

      oTestData.TroubleCode = u16Dtc;

      if (bResult)
      {
         oTestData.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
         ETG_TRACE_USR4(("vCheckAndSaveTestResult():=> Diagnosis Test Result = PASSED"));

      }
      else
      {
         oTestData.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_FAILED;
         ETG_TRACE_ERR(("vCheckAndSaveTestResult():=> Diagnosis Test Result = FAILED"));
      }

      //add the testData to the TestResultList
      m_oFiTestResultObject.TestResultList.TestResultList.push_back(oTestData);
   }
}

/*******************************************************************************
 ** FUNCTION:   tVoid vSendTestResultToDiaglog();
 * Send all the Test Results to Diaglog (1 CCA Msg for all results together)  (Method of this application)
 *******************************************************************************/
tVoid fc_bluetooth_tclClientHandler_Diaglog::vHandleNewBluetoothDTCState(tS8 s8State)
{
   ENTRY
   // This callback function is called by BT settings after BT initialisation,
   // therefore do not re-calculate the status again, just check ... !

   switch (s8State)
   {
      case BLUETOOTH_DTC_STATE_UNDEFINED:
      {
         m_s8BluetoothDTCState = s8State;
         // wait for bluetooth initialisation sequence to finish. This function will be triggered after BT initialisation again!
         ETG_TRACE_USR4(("vHandleNewBluetoothDTCState: Wait for trigger after BT init ..."));
      }
      break;
      case BLUETOOTH_DTC_STATE_OK_STARTUP:
      case BLUETOOTH_DTC_STATE_ERROR_NORMAL_OPERATION:
      case BLUETOOTH_DTC_STATE_ERROR_STARTUP:
      case BLUETOOTH_DTC_STATE_OK_NORMAL_OPERATION:
      {
         if(m_s8BluetoothDTCState != s8State)
         {
            m_s8BluetoothDTCState = s8State;
            // send the DTC status to Diaglog
            ETG_TRACE_USR2(("vHandleNewBluetoothDTCState: Process new Bluetooth DTC(%d) ...", s8State));

            // this will also check for status==BLUETOOTH_DTC_STATE_UNDEFINED and set the trigger to send accordingly!
            vCheckAndSaveTestResult(VDL_ITC_GMG2_DOWNLOAD_HARD_PERF_INTERN_COMM_FAIL_BT, s8State);

            //send all testresult to diaglog (only one cca message for all the results)
            vSendTestResultToDiaglog();
         }
         else
         {
            ETG_TRACE_USR2(("vHandleNewBluetoothDTCState: Bluetooth DTC(%d) is already udpated", s8State));
         }
      }
      break;
      case WIFI_DTC_STATE_ERROR_STARTUP:
      case WIFI_DTC_STATE_OK_STARTUP:
      {
         if(m_s8WiFiDTCState != s8State)
         {
            m_s8WiFiDTCState = s8State;
            // send the DTC status to Diaglog
            ETG_TRACE_USR2(("vHandleNewBluetoothDTCState: Process new WiFi DTC(%d) ...", s8State));

            // this will also check for status==BLUETOOTH_DTC_STATE_UNDEFINED and set the trigger to send accordingly!
            vCheckAndSaveTestResult(VDL_ITC_GMG2_DOWNLOAD_HARD_PERF_INTERN_COMM_FAIL_WIFI, s8State);

            //send all testresult to diaglog (only one cca message for all the results)
            vSendTestResultToDiaglog();
         }
         else
         {
            ETG_TRACE_USR2(("vHandleNewBluetoothDTCState: WiFi DTC(%d) is already udpated", s8State));
         }
      }
      break;
      default:
         ETG_TRACE_ERR(("vHandleNewBluetoothDTCState: default case..."));
      break;
   }
}

tVoid fc_bluetooth_tclClientHandler_Diaglog::vSendTestResultToDiaglog(tVoid)
{
   ENTRY
   if (m_bExistTestDataToSend)
   {
      ETG_TRACE_USR4(("vSendTestResultToDiaglog(): => exist Testresult to send to diaglog"));

      // Check if Service of server is available
      if (bIfServiceAvailable())
      {
         // ------------ SAVETESTRESULT - METHODSTART: begin ---------------
         m_bExistTestDataToSend = FALSE;

         //put the oFiDataObject into MethodStart message
         // Create FI VisitorMessage. (The FI data object will be streamed (each
         // parameter is copied individually) to shared memory.)
         fi_tclVisitorMessage oMsg(m_oFiTestResultObject);

         // Always destroy the FI data object before leaving its creation scope
         m_oFiTestResultObject.vDestroy();

         // Here the message related header information is added and by this the
         // creation of the message is completed.
         vInitServiceData(oMsg, // ServiceDataMsg
         0, // CmdCounter
         MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT, // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART // Opcode
         );

         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {
            // Posting of message has failed.
            // NOTE:  It is NOT necessary to call 'oMsg2.bDelete()' here as the
            // parameter 'TRUE' in "enPostMessage(&oMsg, TRUE)" will take care that
            // the message is deleted in a case of an error.
            ETG_TRACE_ERR(( "vCallSaveTestResult(): enPostMessage() failed for 'SAVETESTRESULT - METHODSTART'"));
         }

         // ------------ SAVETESTRESULT - METHODSTART: end -----------------

         m_bExistTestDataToSend = FALSE;

         ETG_TRACE_USR4(("vSendTestResultToDiaglog(): => SAVETESTRESULT Methodstart message is sended"));

      } // end: if( bIfServiceAvailable() )
      else
      {
         // Always destroy the FI data object before leaving its creation scope
         m_oFiTestResultObject.vDestroy();
         ETG_TRACE_USR4(("vSendTestResultToDiaglog(): => No Service available (not send SAVETESTRESULT Methodstart message)"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("vSendTestResultToDiaglog(): => NO exist Testresult to send to diaglog"));
   }
}
