/**
 * @file FC_Bluetooth_CCAMethodMap.cpp
 * @author 
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 *
 * @addtogroup FC_Bluetooth
 *
 * @brief Public interface for Bluetooth CCAMethodMap
 * @{
 */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "FC_Bluetooth_main.h"
#include "FC_Bluetooth_CCAMethodQueue.h"
#include "FunctionTracer.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APP_METHODQUEUE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FC_Bluetooth_CCAMethodQueue.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APP_METHODQUEUE
#endif
#endif

FC_Bluetooth_CCAMethodQueue *FC_Bluetooth_CCAMethodQueue::m_poMethodQueue;

FC_Bluetooth_CCAMethodQueue::FC_Bluetooth_CCAMethodQueue()
{
  ENTRY
  m_poMethodQueue = OSAL_NULL;
}

FC_Bluetooth_CCAMethodQueue::~FC_Bluetooth_CCAMethodQueue()
{
  ENTRY
}

bool _Job::operator==(const _Job& rhs) const
{
   return ((this->u16SourceAppID == rhs.u16SourceAppID)
         && (this->u16RegisterID == rhs.u16RegisterID)
         && (this->u16CmdCounter == rhs.u16CmdCounter)
         && (this->u16FunctionID == rhs.u16FunctionID));
}

/******************************************************************************/
/* poGetInstance()                                                    */
/******************************************************************************/
FC_Bluetooth_CCAMethodQueue* FC_Bluetooth_CCAMethodQueue::poGetInstance()
{
   ENTRY

   if(m_poMethodQueue == NULL)
   {
      m_poMethodQueue = OSAL_NEW FC_Bluetooth_CCAMethodQueue();
   }

   return (m_poMethodQueue);

}

void FC_Bluetooth_CCAMethodQueue::PushMethodContext(tU16 uiFunction_ID, Job f_rJob)
{
   JobList *joblist = OSAL_NULL;

   //Lock is used to avoid parallel queue update in CCA as well as BT core context
   _lock.lock();

   std::map<tU16, JobList*>::iterator It = MethodMap.find(uiFunction_ID);

   if(It != MethodMap.end())
   {
      joblist = (JobList*) (It->second);
      if (joblist)
      {
         joblist->push_back(f_rJob);
         ETG_TRACE_USR4(("Method Context is inserted the received: 'FID = %u'", uiFunction_ID));
      }
      else
      {
         ETG_TRACE_USR4(("Method Context is empty for the received: 'FID = %u'", uiFunction_ID));
      }
   }
   else
   {
      joblist = OSAL_NEW JobList;
      if (joblist)
      {
         joblist->clear();
         joblist->push_back(f_rJob);
         MethodMap.insert(std::make_pair(uiFunction_ID, joblist));
         ETG_TRACE_USR4(("Method Context is inserted the received: 'FID = %u'", uiFunction_ID));
      }
      else
      {
         ETG_TRACE_USR4(("Method Context is not created for the received: 'FID = %u'", uiFunction_ID));
      }
   }

   _lock.unlock();

   return;
}

JobList* FC_Bluetooth_CCAMethodQueue::PopMethodContext(tU16 uiFunction_ID)
{
   JobList *joblist = OSAL_NULL;

   //Lock is used to avoid parallel queue update in CCA as well as BT core context
   _lock.lock();

   std::map<tU16, JobList*>::iterator It = MethodMap.find(uiFunction_ID);

   if(It != MethodMap.end())
   {
      joblist = (JobList*)(It->second);

      MethodMap.erase(uiFunction_ID);

      ETG_TRACE_USR4(("Method Context is erased for the received: 'FID = %u'", uiFunction_ID));
   }
   else
   {
      ETG_TRACE_USR4(("Method Context is not available for the received: 'FID = %u'", uiFunction_ID));
   }

   _lock.unlock();

   return joblist;
}

JobList* FC_Bluetooth_CCAMethodQueue::GetMethodContext(tU16 uiFunction_ID)
{
   JobList *joblist = OSAL_NULL;

   //Lock is used to avoid parallel queue update in CCA as well as BT core context
   _lock.lock();

   std::map<tU16, JobList*>::iterator It = MethodMap.find(uiFunction_ID);

   if(It != MethodMap.end())
   {
      joblist = (JobList*)(It->second);

      ETG_TRACE_USR4(("Method Context is available for the received: 'FID = %u'", uiFunction_ID));
   }
   else
   {
      ETG_TRACE_USR4(("Method Context is not available for the received: 'FID = %u'", uiFunction_ID));
   }

   _lock.unlock();

   return joblist;
}

void FC_Bluetooth_CCAMethodQueue::PopParticularMethodContext(tU16 uiFunction_ID, Job f_rJob, tU16 u16ReceivedCnt)
{
   (tVoid) f_rJob;

   JobList *joblist = OSAL_NULL;
   tU16 u16Count = 0;

   //Lock is used to avoid parallel queue update in CCA as well as BT core context
   _lock.lock();

   std::map<tU16, JobList*>::iterator It = MethodMap.find(uiFunction_ID);

   if(It != MethodMap.end())
   {
      joblist = (JobList*)(It->second);

      if(joblist)
      {
         for (JobList::iterator it = joblist->begin(); it != joblist->end(); it++)
         {
            /* Temporary solution for MSG and PB */
            u16Count++;
            if(u16Count == u16ReceivedCnt)
            {
               joblist->erase(it);
               ETG_TRACE_USR4(("Job is erased for the received: 'FID = %u' and u16ReceivedCnt - %d", uiFunction_ID, u16ReceivedCnt));
               break;
            }
         }

         if(joblist->empty())
         {
            MethodMap.erase(uiFunction_ID);
         }
      }
   }

   _lock.unlock();

   return;
}

tBool FC_Bluetooth_CCAMethodQueue::GetMethodContext(tU16 uiFunction_ID, Job &f_rJob)
{
   JobList *joblist = OSAL_NULL;
   tBool bRet = false;

   //Lock is used to avoid parallel queue update in CCA as well as BT core context
   _lock.lock();

   std::map<tU16, JobList*>::iterator It = MethodMap.find(uiFunction_ID);

   if(It != MethodMap.end())
   {
      joblist = (JobList*)(It->second);

      if (joblist)
      {
         f_rJob = joblist->front();
         bRet = true;
      }
      else
      {
         ETG_TRACE_USR4(("Method Context is empty for the received: 'FID = %u'", uiFunction_ID));
      }
   }

   _lock.unlock();

   return bRet;
}

tU8 FC_Bluetooth_CCAMethodQueue::GetMethodContextListSize(tU16 uiFunction_ID)
{
   JobList *joblist = OSAL_NULL;
   tU8 u8ListSize = 0;

   //Lock is used to avoid parallel queue update in CCA as well as BT core context
   _lock.lock();

   std::map<tU16, JobList*>::iterator It = MethodMap.find(uiFunction_ID);

   if(It != MethodMap.end())
   {
      joblist = (JobList*)(It->second);

      if(joblist)
      {
         u8ListSize = static_cast<tU8>(joblist->size());
      }
   }

   _lock.unlock();

   ETG_TRACE_USR4(("Queue size for the received: 'FID = %u' is %d", uiFunction_ID, u8ListSize));

   return u8ListSize;
}
