/**
 * @file FC_Bluetooth_AudioRouting.cpp
 * @author
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 *
 * @addtogroup FC_Bluetooth
 *
 * @brief Implementation of the interface to ARL.
 * @{
 */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#ifndef OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#endif

#include "FC_Bluetooth_main.h"
#include "FC_Bluetooth_CCAService.h"
#include "FC_Bluetooth_AudioRouting.h"
#include "FC_Bluetooth_AudioPlayer.h"
#include "FunctionTracer.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APP_AUDIOROUTING
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FC_Bluetooth_AudioRouting.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APP_AUDIOROUTING
#endif
#endif

FC_Bluetooth_tclAudioRouting* FC_Bluetooth_tclAudioRouting::_audiorouting = OSAL_NULL;

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_tclAudioRouting::FC_Bluetooth_tclAudioRouting
 *
 * DESCRIPTION :  constructor, creates object FC_Bluetooth_tclAudioRouting
 *
 * PARAMETER   :  none
 *
 * RETURNVALUE :  none
 *
 *******************************************************************************/
FC_Bluetooth_tclAudioRouting::FC_Bluetooth_tclAudioRouting(fc_bluetooth_tclApp* poMainAppl) :
      arl_tclISource(poMainAppl)
{
   ENTRY

   /* copy the main pointer */
   _poMainAppl = poMainAppl;
   _audiorouting = this;
   m_u8SrcAvailabilty = ARL_EN_SRC_NOT_AVAILABLE;
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_tclAudioRouting::~FC_Bluetooth_tclAudioRouting
 *
 * DESCRIPTION :  destructor, destroys the object
 *
 * PARAMETER   :  none
 *
 * RETURNVALUE :  none
 *
 *******************************************************************************/
FC_Bluetooth_tclAudioRouting::~FC_Bluetooth_tclAudioRouting()
{
   ENTRY
   _poMainAppl = OSAL_NULL;
   _audiorouting = OSAL_NULL;
}

/*******************************************************************************
 *
 * FUNCTION    : poGetInstance()
 *
 * DESCRIPTION : To get instance of mediaPlayer Client Handler
 *
 * PARAMETER   :
 *
 * RETURNVALUE : - mediaPlayer Client Handler Instance
 *
 *******************************************************************************/
FC_Bluetooth_tclAudioRouting* FC_Bluetooth_tclAudioRouting::poGetInstance()
{
   ENTRY
   return _audiorouting;
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_tclAudioRouting::bOnAllocate
 *
 * DESCRIPTION :  Allocate request from audio routing.
 *
 * PARAMETER   :  enSrcNum:               (I)   source number
 *                rfcoAllocRoute:         (I)   reference to Allocate route result
 *
 * RETURNVALUE:   TRUE
 *
 *******************************************************************************/
tBool FC_Bluetooth_tclAudioRouting::bOnAllocate(arl_tenSource enSrcNum, const arl_tAllocRouteResult& rfcoAllocRoute)
{
   ENTRY

   if (rfcoAllocRoute.listOutputDev.strALSADev.size())
   {
      m_szAlsaDeviceName = rfcoAllocRoute.listOutputDev.strALSADev[0].szValue;
   }

   ETG_TRACE_USR2((" Device Name Received =%s", m_szAlsaDeviceName.c_str()));

   if (enSrcNum == ARL_SRC_PHONE_OUTBAND_RING)
   {
      FC_Bluetooth_AudioPlayer::setAlsaDevice(m_szAlsaDeviceName.c_str());
   }

   return TRUE;
}

/*******************************************************************************
 *
 * FUNCTION    :    FC_Bluetooth_tclAudioRouting::bOnDeAllocate
 *
 * DESCRIPTION : Deallocate request from audio routing.
 *
 * PARAMETER   :   enSrcNum: (I)   source number
 *
 * RETURNVALUE : TRUE

 *
 *******************************************************************************/
tBool FC_Bluetooth_tclAudioRouting::bOnDeAllocate(arl_tenSource enSrcNum)
{
   ENTRY

   (void) enSrcNum;

   m_szAlsaDeviceName = "";

   return TRUE;
}

void FC_Bluetooth_tclAudioRouting::SetSourceAvailable(tBool bSourceAvailable)
{
   ENTRY
   vSetSrcAvailable(bSourceAvailable);
}

void FC_Bluetooth_tclAudioRouting::vSetSourceAvailability(arl_tenSrcAvailability srcAvailability, arl_tenAvailabilityReason availabilityReason, arl_tenSource enSource, tU16 u16SubSource)
{
   ENTRY

   ETG_TRACE_USR4((" PreviousSrcAvailabilty ::%d , srcAvailability::%d", ETG_CENUM(tenSrcAvailability, m_u8SrcAvailabilty), ETG_CENUM(tenSrcAvailability, srcAvailability)));
   switch (enSource)
   {
      case ARL_SRC_PHONE_OUTBAND_RING:
         vHandle_SourceAvailibility(srcAvailability, availabilityReason, ARL_SRC_PHONE_OUTBAND_RING, u16SubSource);
      break;

      default:
         ETG_TRACE_ERR(("Invalid Source"));
      break;
   }
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_tclAudioRouting::bOnSrcActivity
 *
 * DESCRIPTION :  Source activity request from audio routing.
 *
 * PARAMETER   :  enSrcNum:               (I)   source number
 *                rfcoSrcActivity:        (I)   source activity
 *
 * RETURNVALUE : TRUE
 *
 *******************************************************************************/
tBool FC_Bluetooth_tclAudioRouting::bOnSrcActivity(arl_tenSource enSrcNum, const arl_tSrcActivity& rfcoSrcActivity)
{
   ENTRY

   ETG_TRACE_USR2(("SOURCE NUMBER: %u", enSrcNum));

   ETG_TRACE_USR2(("SOURCE ACTIVITY: %u", rfcoSrcActivity.enType));

   FC_Bluetooth_tclCCAService *poCCAService = FC_Bluetooth_tclCCAService::poGetInstance();

   switch (rfcoSrcActivity.enType)
   {
      case midw_fi_tcl_e8_SrcActivity::FI_EN_OFF:
      {
         if (enSrcNum == ARL_SRC_PHONE_OUTBAND_RING)
         {
            ETG_TRACE_USR4(("SA_OFF for VehicleRingtone"));

            if(poCCAService)
            {
               ETG_TRACE_USR4((" Stopping the Ringtone playback"));
               poCCAService->stopRingTone();
            }
         }
         else
         {
            ETG_TRACE_ERR(("Invalid Source"));
            return FALSE;
         }

         vSourceActivityResult(enSrcNum, ARL_EN_ISRC_ACT_OFF);
      }
      break;

      case midw_fi_tcl_e8_SrcActivity::FI_EN_PAUSE:
      {
         if (enSrcNum == ARL_SRC_PHONE_OUTBAND_RING)
         {
            ETG_TRACE_USR4(("SA_PAUSE for VehicleRingtone"));

            if(poCCAService)
            {
               // TBC: On the different flags set: Is it really required?
               ETG_TRACE_USR4((" Stopping the Ringtone playback"));
               poCCAService->stopRingTone();
            }
         }
         else
         {
            ETG_TRACE_ERR(("Invalid Source"));
            return FALSE;
         }

         vSourceActivityResult(enSrcNum, ARL_EN_ISRC_ACT_PAUSE);
      }
      break;

      case midw_fi_tcl_e8_SrcActivity::FI_EN_ON:
      {
         if (poCCAService)
         {
            if (enSrcNum == ARL_SRC_PHONE_OUTBAND_RING)
            {
               ETG_TRACE_USR4(("SA_ON for VehicleRingtone"));

               if(_poMainAppl)
               {
                  //! Stop timer if timer is active.
                  if (FALSE == _poMainAppl->bStopTimer(ARL_RESPONSE_TIMER_ID))
                  {
                     ETG_TRACE_ERR(("bOnSrcActivity : Unable to stop the timer"));
                  }
                  else
                  {
                     ETG_TRACE_USR4(("Timer is stopped"));
                  }
               }
               poCCAService->playRingTone();
            }
            else
            {
               ETG_TRACE_ERR(("Invalid Source"));
               return FALSE;
            }
         }

         vSourceActivityResult(enSrcNum, ARL_EN_ISRC_ACT_ON);
      }

      break;

      default:
      break;
   }

   return TRUE;
}

tBool FC_Bluetooth_tclAudioRouting::vTriggerARLAudio(tU16 u16AudioChannelId)
{
   ENTRY
   tBool bReturn = FALSE;

   if (u16AudioChannelId == FC_BLUETOOTH_AUDIOCHANNEL_MONORINGTONE)
   {
      SetSourceAvailable (TRUE);

      bReturn = bSetAudioRouteRequest(ARL_SRC_PHONE_OUTBAND_RING, ARL_EN_ISRC_ACT_ON);

      if (TRUE == bReturn)
      {
         ETG_TRACE_USR4(("Set Audio Request for ARL_SRC_PHONE_OUTBAND_RING is success"));

         if(_poMainAppl)
         {
            (void) _poMainAppl->bStopTimer(ARL_RESPONSE_TIMER_ID);

            // Timer introduced to release the channel in case no response from ARL after 3.5 secs
            bReturn = _poMainAppl->bStartTimer(ARL_RESPONSE_TIMER_ID, ARL_RESPONSE_MSEC, FC_BLUETOOTH_CCASERVICE_ZERO);

            if (FALSE == bReturn)
            {
               ETG_TRACE_ERR(("vTriggerARLAudio: Start of ARL Timer id is failed"));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("Set Audio Request for ARL_SRC_PHONE_OUTBAND_RING is failed"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("Invalid Channel ID"));
   }

   return bReturn;
}

//!Static
/***************************************************************************
 ** FUNCTION:  FC_Bluetooth_tclAudioRouting::vARLResponseTimerCb(...)
 ***************************************************************************/
void FC_Bluetooth_tclAudioRouting::vARLResponseTimerCb()
{
   ETG_TRACE_USR1(("vARLResponseTimerCb : Timeout for ARL Response"));

   vTriggerARLAudioDeallocation(FC_BLUETOOTH_AUDIOCHANNEL_MONORINGTONE);

   //MethodError
   FC_Bluetooth_tclCCAService* poCCAService = FC_Bluetooth_tclCCAService::poGetInstance();

   if(poCCAService)
   {
      poCCAService->PlayRingtoneMethodError();
   }

}//!End of bool FC_Bluetooth_tclAudioRouting::vARLResponseTimerCb(...)

void FC_Bluetooth_tclAudioRouting::vTriggerARLAudioDeallocation(tU16 u16AudioChannelId)
{
   ENTRY

   ETG_TRACE_USR1(("vTriggerARLAudioDeallocation : u16AudioChannelId - %d", u16AudioChannelId));

   if (u16AudioChannelId == FC_BLUETOOTH_AUDIOCHANNEL_MONORINGTONE)
   {
      if(_poMainAppl)
      {
         //! Stop timer if timer is active.
         if (FALSE == _poMainAppl->bStopTimer(ARL_RESPONSE_TIMER_ID))
         {
            ETG_TRACE_ERR(("vTriggerARLAudioDeallocation: Unable to stop the timer"));
         }
      }

      if (bSetAudioRouteRequest(ARL_SRC_PHONE_OUTBAND_RING, ARL_EN_ISRC_ACT_OFF))
      {
         ETG_TRACE_USR4(("vTriggerARLAudioDeallocation: Release Audio Request for ARL_SRC_PHONE_OUTBAND_RING is success"));
      }
      else
      {
         ETG_TRACE_ERR(("vTriggerARLAudioDeallocation: Release Audio Request for ARL_SRC_PHONE_OUTBAND_RING is failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Invalid Channel ID"));
   }
}

void FC_Bluetooth_tclAudioRouting::vHandle_SourceAvailibility(arl_tenSrcAvailability srcAvailability, arl_tenAvailabilityReason availabilityReason, arl_tenSource enSource, tU16 u16SubSource)
{
   ENTRY

   if (m_u8SrcAvailabilty != srcAvailability)
   {
      m_u8SrcAvailabilty = srcAvailability;

      tBool bResult = bSetSourceAvailability(srcAvailability, availabilityReason, enSource, u16SubSource);

      if (bResult)
      {
         ETG_TRACE_USR4((" bSetSourceAvailability() is success"));
      }
      else
      {
         ETG_TRACE_ERR((" bSetSourceAvailability() failed"));
      }
   }
   else
   {
      ETG_TRACE_USR4((" Source Availability is already set"));
   }
}

