/**
 * @file FC_Bluetooth_AudioPlayer.h
 * @author
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 *
 * @addtogroup FC_Bluetooth
 *
 * @brief Implementation to Play audio files through GStreamer.
 * @{
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "FC_Bluetooth_main.h"
#include "FC_Bluetooth_CCAService.h"
#include "FC_Bluetooth_AudioPlayer.h"
#include "FC_Bluetooth_AudioRouting.h"
#include "FunctionTracer.h"
#include <time.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APP_AUDIOPLAYER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FC_Bluetooth_AudioPlayer.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APP_AUDIOPLAYER
#endif
#endif

GMainLoop *FC_Bluetooth_AudioPlayer::gstMainLoop = NULL;
FC_Bluetooth_AudioPlayer* FC_Bluetooth_AudioPlayer::poAudioPlayer = NULL;
bool FC_Bluetooth_AudioPlayer::m_bPlayRepeated = true;
bool FC_Bluetooth_AudioPlayer::m_bIsPlayStarted = false;
bool FC_Bluetooth_AudioPlayer::m_bWaitforCondSignal = false;
std::stringstream FC_Bluetooth_AudioPlayer::m_szAlsaDeviceName;

pthread_mutex_t playerThread_Mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t threadWait_CondSignal = PTHREAD_COND_INITIALIZER;
pthread_attr_t playerThread_Attr;

FC_Bluetooth_AudioPlayer::FC_Bluetooth_AudioPlayer()
{
   ENTRY

   pthread_cond_init(&threadWait_CondSignal, NULL);
   pthread_mutex_init(&playerThread_Mutex, NULL);

   m_bSetAttributes = false;
   vSetAttributeToPthread();

}

FC_Bluetooth_AudioPlayer::~FC_Bluetooth_AudioPlayer()
{
   ENTRY

   pthread_cond_destroy(&threadWait_CondSignal);
   pthread_mutex_destroy(&playerThread_Mutex);

   // clean up memory
   pthread_attr_destroy(&playerThread_Attr);
}

FC_Bluetooth_AudioPlayer* FC_Bluetooth_AudioPlayer::getInstance()
{
   ENTRY

   if (NULL == poAudioPlayer)
   {
      poAudioPlayer = OSAL_NEW FC_Bluetooth_AudioPlayer();
   }

   return poAudioPlayer;
}

void FC_Bluetooth_AudioPlayer::deleteInstance()
{
   ENTRY

   if (NULL != poAudioPlayer)
   {
      delete poAudioPlayer;
      poAudioPlayer = NULL;
   }
}

void FC_Bluetooth_AudioPlayer::vSetAttributeToPthread()
{
   ENTRY_INTERNAL

   if(m_bSetAttributes == false)
   {
      sched_param param;
      int ret = 0;
      tBool bTemp = true;

      /* initialized with default attributes */
      ret = pthread_attr_init(&playerThread_Attr);
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_init(&playerThread_Attr) returned with ret= %d ", ret ));
         bTemp = false;
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_init(&playerThread_Attr) returned ret= %d ", ret ));
      }

      ret = pthread_attr_setschedpolicy(&playerThread_Attr, SCHED_FIFO);  // <-- RT prio policy
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_setschedpolicy(&playerThread_Attr, SCHED_FIFO) returned with ret= %d ", ret ));
         bTemp = false;
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_setschedpolicy(&playerThread_Attr, SCHED_FIFO) returned ret= %d ", ret ));
      }

      /* safe to get existing scheduling param */
      ret = 0;
      ret = pthread_attr_getschedparam(&playerThread_Attr, &param);
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_getschedparam(&playerThread_Attr, &param) returned with ret= %d ", ret ));
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_getschedparam(&playerThread_Attr, &param) returned ret= %d ", ret ));
      }

      /* set the priority; others are unchanged */
      param.sched_priority = PTHREAD_RT_PRIO;

      /* setting the new scheduling param */
      ret = pthread_attr_setschedparam(&playerThread_Attr, &param);
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_setschedparam(&playerThread_Attr, &param) returned with ret= %d ", ret ));
         bTemp = false;
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_setschedparam(&playerThread_Attr, &param) returned ret= %d ", ret ));
      }

      ret = pthread_attr_setinheritsched(&playerThread_Attr, PTHREAD_EXPLICIT_SCHED);  // <-- set RT prio to be inherited
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_setinheritsched(&playerThread_Attr, PTHREAD_EXPLICIT_SCHED) returned with ret= %d ", ret ));
         bTemp = false;
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_setinheritsched(&playerThread_Attr, PTHREAD_EXPLICIT_SCHED) returned ret= %d ", ret ));
      }

      m_bSetAttributes = bTemp;
   }
   else
   {
      ETG_TRACE_USR1((" SUCCESS:Attributes are already set "));
   }
   // set thread prio: END

   return;
}

bool FC_Bluetooth_AudioPlayer::playFile(const char *filePath, const tU16 u16RingtoneID, bool repeat)
{
   ENTRY

   bool playStarted = false;
   const char *filename = NULL;
   pthread_t threadHandle = 0;
   bool stopPlaybackIfTimerExpired = false;

   if (filePath != NULL)
   {
      vSetAttributeToPthread();

      if(m_bSetAttributes == true)
      {
         pthread_mutex_lock(&playerThread_Mutex);

         filename = strdup(filePath);

         ETG_TRACE_USR2(("playFile: File To Play: %50s and RintoneID - %d", filename, u16RingtoneID));

         int result = pthread_create(&threadHandle, &playerThread_Attr, FC_Bluetooth_AudioPlayer::startPlayback, (tVoid*)const_cast<char*>(filename));

         if(0 == result)
         {
            ETG_TRACE_USR2(("playFile: Wait for the conditional signal : %d", m_bWaitforCondSignal));

            struct timespec waitforGstreamerUpdate = {0, 0};

            waitforGstreamerUpdate.tv_sec = time(NULL) + PTHREAD_COND_WAIT_TIMER_IN_SEC;

            m_bWaitforCondSignal = false;

            // This condition check is used to overcome the Spurious wakeup of pthread_cond_wait
            while(!m_bWaitforCondSignal)
            {
                // This cond_timedwait for creates the G-Streamer pipeline and waiting for the playing update.
               // If no response is received within 5 sec from G-Streamer, timeout is occurred.
               int retVal = pthread_cond_timedwait(&threadWait_CondSignal, &playerThread_Mutex, &waitforGstreamerUpdate);

                ETG_TRACE_USR4(("playFile: pthread_cond_timedwait retValue - %d", retVal));

                if(ETIMEDOUT == retVal)
                {
                   ETG_TRACE_ERR(("playFile: Timeout is occured for Pthread_cond_TimedWait"));

                   //Set the m_bIsPlayStarted flag as TRUE if the RingtoneID is INCOMING_CALL if timeout is occurred.
                   // Because BMApp doesn't know ringtone is playing in HU or not. So Wait for StopRingtone MS from FC_Phone.
                   if(INCOMING_CALL_RINGTONE_ID == u16RingtoneID)
                   {
                      ETG_TRACE_USR4(("playFile: set the m_bIsPlayStarted flag as TRUE for RintoneID - %d", u16RingtoneID));
                      m_bIsPlayStarted = true;
                   }
                   //Set the stopPlaybackIfTimerExpired flag as TRUE if the RingtoneID is 2 to 11 and 255.
                   // Because BMApp doesn't know ringtone is playing in HU or not. So Stops the Playback internally.
                   else
                   {
                      ETG_TRACE_USR4(("playFile: set the stopPlaybackIfTimerExpired flag as TRUE for RintoneID - %d", u16RingtoneID));
                      stopPlaybackIfTimerExpired = true;
                   }
                   break;
                }
            }

            ETG_TRACE_USR4(("playFile: Condition signal received"));

            if(true == m_bIsPlayStarted)
            {
               playStarted = true;
               m_bPlayRepeated = repeat;
            }
            else
            {
               ETG_TRACE_ERR((" ERROR: Something went wrong with the Player thread "));
            }
         }
         else
         {
            ETG_TRACE_ERR((" ERROR: Player thread creation failed "));
            free((void*)filename);
         }

         pthread_mutex_unlock(&playerThread_Mutex);
      }
      else
      {
         ETG_TRACE_ERR((" ERROR: Attributes are not set properly "));
      }
   }
   else
   {
      ETG_TRACE_ERR((" ERROR: filePath is NULL "));
   }

   if(playStarted == false)
   {
      FC_Bluetooth_tclAudioRouting::poGetInstance()->vTriggerARLAudioDeallocation(FC_BLUETOOTH_AUDIOCHANNEL_MONORINGTONE);

      // Stops the Playback internally if stopPlaybackIfTimerExpired flag as TRUE
      if(true == stopPlaybackIfTimerExpired)
      {
         this->stopPlayback();
      }
   }

   return playStarted;
}

gboolean FC_Bluetooth_AudioPlayer::gstBusCallBack(GstBus *bus_local, GstMessage *msg, gpointer data)
{
   (tVoid) bus_local;

   ENTRY

   GstElement *playElement = (GstElement *)data;

   if (msg == NULL)
   {
      ETG_TRACE_USR4(("GstBusCallBack: msg is NULL"));
      return TRUE;
   }

   switch (GST_MESSAGE_TYPE(msg))
   {
      case GST_MESSAGE_BUFFERING:
      {
         gint percent = 0;
         gst_message_parse_buffering(msg, &percent);
         ETG_TRACE_USR4(("gstBusCallBack: Buffering"));
      }
      break;

      case GST_MESSAGE_EOS:
      {
         // PlayBack finished
         ETG_TRACE_USR4(("gstBusCallBack: EOS STREAM RECEIVED STOPPING THE PLAYER"));
         if (false == m_bPlayRepeated)
         {
            FC_Bluetooth_tclAudioRouting::poGetInstance()->vTriggerARLAudioDeallocation(FC_BLUETOOTH_AUDIOCHANNEL_MONORINGTONE);
         }
         else
         {
            gint64 time_nanoseconds = 0;
            gint64 stop_nanoseconds = (gint64)(GST_CLOCK_TIME_NONE);

            if (!gst_element_seek(playElement, 1.0, GST_FORMAT_TIME, GST_SEEK_FLAG_FLUSH, GST_SEEK_TYPE_SET,
                  time_nanoseconds, GST_SEEK_TYPE_NONE, stop_nanoseconds))
            {
               ETG_TRACE_ERR(("gstBusCallBack: Seeking failed"));
               FC_Bluetooth_tclAudioRouting::poGetInstance()->vTriggerARLAudioDeallocation(FC_BLUETOOTH_AUDIOCHANNEL_MONORINGTONE);
            }
            else
            {
               gst_element_set_state(playElement, GST_STATE_PLAYING);
            }
         }
      }
      break;

      case GST_MESSAGE_STATE_CHANGED:
      {
         // gst_play_pipeline changed state
         GstState oldState;
         GstState newState;
         GstState pending;
         gst_message_parse_state_changed(msg, &oldState, &newState, &pending);
         gstStateChanged(newState, oldState);
      }
      break;

      case GST_MESSAGE_WARNING:
      {
         // Warning received from GST
         gchar *debug = NULL;
         GError *error = NULL;
         gst_message_parse_warning(msg, &error, &debug);
         ETG_TRACE_ERR(("gstBusCallBack: Warning is =%s", error->message));
         g_free(debug);
         g_error_free(error);
      }
      break;

      case GST_MESSAGE_ERROR:
      {
         // Error received from GST
         gchar *debug = NULL;
         GError *error = NULL;
         gst_message_parse_error(msg, &error, &debug);
         ETG_TRACE_ERR(("GstBusCallBack: Error is  =%s", error->message));
         g_free(debug);
         g_error_free(error);

         // Stop the playback and kill the thread
         FC_Bluetooth_AudioPlayer* poAudioPlayer = FC_Bluetooth_AudioPlayer::getInstance();
         poAudioPlayer->vSendSignaltoWaitingThread(false);
      }
      break;

      default:
      {
         ETG_TRACE_USR2 (("GstBusCallBack :got %s message", GST_MESSAGE_TYPE_NAME (msg)));
         break;
      }
   }
   return TRUE;
}

void FC_Bluetooth_AudioPlayer::gstStateChanged(GstState new_state, GstState old_state)
{
   ENTRY

   ETG_TRACE_USR2(("gstStateChanged: %u => %u", ETG_ENUM(TR_GSTSTATE, old_state), ETG_ENUM(TR_GSTSTATE, new_state)));

   if (GST_STATE_PLAYING == new_state)
   {
      if(false == m_bIsPlayStarted)
      {
         // Set playing state
         FC_Bluetooth_AudioPlayer* poAudioPlayer = FC_Bluetooth_AudioPlayer::getInstance();
         poAudioPlayer->vSendSignaltoWaitingThread(true);
      }
   }
}

bool FC_Bluetooth_AudioPlayer::setAlsaDevice(const char *deviceName)
{
   ENTRY

   ETG_TRACE_USR2(("setAlsaDevice: device =%s", deviceName));

   bool returnVal = true;

   if(NULL == deviceName)
   {
      ETG_TRACE_ERR(("ALSA deviceName is received as Null"));
      returnVal = false;
   }
   else
   {
      m_szAlsaDeviceName.str("");
      m_szAlsaDeviceName << deviceName;
   }

   ETG_TRACE_USR2(("setAlsaDevice: Hardware name is %s", m_szAlsaDeviceName.str().c_str()));

   return returnVal;
}

void FC_Bluetooth_AudioPlayer::stopPlayback()
{
   ENTRY

   ETG_TRACE_USR4(("stopPlayback:entered"));

   pthread_mutex_lock(&playerThread_Mutex);

   ETG_TRACE_USR2(("stopPlayback: Wait for the conditional signal : %d", m_bWaitforCondSignal));

   if (NULL != gstMainLoop)
   {
      if (g_main_loop_is_running(gstMainLoop))
      {
         struct timespec waitforUpdate = {0, 0};
         waitforUpdate.tv_sec = time(NULL) + PTHREAD_COND_WAIT_TIMER_IN_SEC;

         g_main_loop_quit(gstMainLoop);

         m_bWaitforCondSignal = false;

         // This condition check is used to overcome the Spurious wakeup of pthread_cond_wait
         while(!m_bWaitforCondSignal)
         {
            // This cond_timedwait for clear all the ref(playbin element, etc,.) created in the startPlayback thread
            // If no response is received within 5 sec from Pthread, timeout is occurred.
            int retVal = pthread_cond_timedwait(&threadWait_CondSignal, &playerThread_Mutex, &waitforUpdate);

            ETG_TRACE_USR4(("stopPlayback: pthread_cond_timedwait retVal - %d", retVal));

            if(ETIMEDOUT == retVal)
            {
               ETG_TRACE_USR2(("stopPlayback: Timeout is occured"));
               break;
            }
         }

         ETG_TRACE_USR2(("stopPlayback: Condition signal received"));
      }
      else
      {
         ETG_TRACE_USR4(("stopPlayback: g_main loop is not running"));
      }

      m_bIsPlayStarted = false;

   }
   else
   {
      ETG_TRACE_USR4(("stopPlayback:gstMainLoop is NULL"));
   }

   pthread_mutex_unlock(&playerThread_Mutex);

   ETG_TRACE_USR4(("stopPlayback: end"));
}

void *FC_Bluetooth_AudioPlayer::startPlayback(void* fileInfo)
{
   ENTRY

   ETG_TRACE_USR4(("startPlayback: entered"));

   char* file = NULL;
   guint64 l_iAlsaSinkAlignementThreshold = ALSA_ALIGNMENT_THRESHOLD_NSEC;

   FC_Bluetooth_AudioPlayer* poAudioPlayer = FC_Bluetooth_AudioPlayer::getInstance();

   if (NULL == fileInfo)
   {
      ETG_TRACE_USR4(("startPlayback:fileInfo is null"));
      poAudioPlayer->vSendSignaltoWaitingThread(false);
      return NULL;
   }

   file = (char*)fileInfo;

   std::stringstream fileToPlay;

   fileToPlay.str("");
   fileToPlay<<"file://";
   fileToPlay<<file;

   ETG_TRACE_USR4(("startPlayback: filename:%s", file));
   ETG_TRACE_USR4(("startPlayback: filename to play:%s", fileToPlay.str().c_str()));

   free(fileInfo);

   if (m_szAlsaDeviceName.str().empty())
   {
      ETG_TRACE_USR4(("startPlayback:m_szAlsaDeviceName is empty"));
      poAudioPlayer->vSendSignaltoWaitingThread(false);
      return NULL;
   }
   else
   {
      ETG_TRACE_USR4(("startPlayback: device :%s", m_szAlsaDeviceName.str().c_str()));
   }

   /* Initialise GStreamer */
   gst_init(NULL, NULL);

   /* set up the playbin */
   GstElement *playBinElement = gst_element_factory_make("playbin", "play");
   if (NULL == playBinElement)
   {
      ETG_TRACE_USR4(("Not able to create playBinElement"));
      poAudioPlayer->vSendSignaltoWaitingThread(false);
      return NULL;
   }

   GstElement *sinkElement = gst_element_factory_make("alsasink", "sink");
   if (NULL == sinkElement)
   {
      ETG_TRACE_USR4(("Not able to create sinkElement"));
      gst_object_unref((GstObject*) playBinElement);
      poAudioPlayer->vSendSignaltoWaitingThread(false);
      return NULL;
   }

   g_object_set((GObject*) playBinElement, "uri", fileToPlay.str().c_str(), NULL);
   g_object_set((GObject*) sinkElement, "device", m_szAlsaDeviceName.str().c_str(), NULL);
   g_object_set((GObject*) sinkElement, "alignment-threshold", l_iAlsaSinkAlignementThreshold, NULL);
   g_object_set((GObject*) playBinElement, "audio-sink", sinkElement, NULL);

   /* Adding a message handler */
   GstBus* gstBus = gst_pipeline_get_bus(GST_PIPELINE_CAST((tVoid*) playBinElement));

   if (!gstBus)
   {
      ETG_TRACE_USR4(("startPlayback: gstBus is null: Unable to create pipeline"));
      gst_object_unref((GstObject*) playBinElement);
      poAudioPlayer->vSendSignaltoWaitingThread(false);
      return NULL;
   }

   gstMainLoop = g_main_loop_new(NULL, FALSE);

   if(NULL == gstMainLoop)
   {
      ETG_TRACE_USR4(("startPlayback: gstMainLoop is null"));
      gst_object_unref(gstBus);
      gst_object_unref((GstObject*) playBinElement);
      poAudioPlayer->vSendSignaltoWaitingThread(false);
      return NULL;
   }

   gst_bus_add_watch(gstBus, gstBusCallBack, playBinElement);

   gst_object_unref(gstBus);

   gst_element_set_state(playBinElement, GST_STATE_PLAYING);

   /* Run the PlayingLoop*/
   ETG_TRACE_USR4(("startPlayback: Playing the file"));
   g_main_loop_run(gstMainLoop);

   ETG_TRACE_USR4(("CLEANING PLAYING THREAD"));

   gst_element_set_state(playBinElement, GST_STATE_NULL);
   gst_object_unref((GstObject*) playBinElement);

   if(gstMainLoop != NULL)
   {
      g_main_loop_unref (gstMainLoop);
      gstMainLoop = NULL;
   }

   poAudioPlayer->vSendSignaltoWaitingThread(false);
   ETG_TRACE_USR4(("startPlayback: PLAYING THREAD STOPPED"));

   return NULL;
}

void FC_Bluetooth_AudioPlayer::vSendSignaltoWaitingThread(tBool bResponseStatus)
{
   ETG_TRACE_USR4(("vSendSignaltoWaitingThread:: bResponseStatus = %d", bResponseStatus));

   pthread_mutex_lock(&playerThread_Mutex);

   m_bIsPlayStarted = bResponseStatus;

   m_bWaitforCondSignal = true;

   pthread_cond_signal(&threadWait_CondSignal);

   pthread_mutex_unlock(&playerThread_Mutex);

   return;
}

