/**
 * @file FC_Bluetooth_CCAService.h
 * @author 
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 *
 * @addtogroup FC_Bluetooth
 *
 * @brief Public interface for BluetoothCCAService
 * @{
 */

#ifndef _FC_BLUETOOTH_CCASERVICE_H_
#define _FC_BLUETOOTH_CCASERVICE_H_

// Include Application Helper Library.
#define AHL_S_IMPORT_INTERFACE_GENERIC
#define AHL_S_IMPORT_INTERFACE_CCA_EXTENSION
#include "ahl_if.h"

// Include Message Framework interface (AMT, MSGFW).
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE

// Include public FI interface of this service.
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_BLUETOOTHSETTINGS_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_BLUETOOTHSETTINGS_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_BLUETOOTHSETTINGS_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_BLUETOOTHSETTINGS_SERVICEINFO
#include "midw_fi_if.h"

//Include FI interface of used service
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_TYPES
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_FUNCTIONIDS
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_ERRORCODES
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_SERVICEINFO
//Included for most_fi libraries Functions declaration.
#include "conn_most_fi_if.h"

#include <algorithm>
#include <map>

// Include BM Core interface
#include "BmCoreIfTypes.h"
#include "FC_Bluetooth_CCAMethodQueue.h"
#include "FC_Bluetooth_RingtoneList.h"

using namespace bmcore;

#define FC_BLUETOOTH_CCASERVICE_ZERO 0
#define FC_BLUETOOTH_CCASERVICE_ONE 1

#define VARIANT_CMC19_CABIN_CONTROLLER 4

// Default value set as 4 since pairing response type value is 0 to 3
#define FC_BLUETOOTH_CCASERVICE_PARIING_RESPONSE_NOTHING 4

#define FC_BLUETOOTH_SERVICE_METHOD_QUEUE_SIZE 32

#define FC_BLUETOOTH_DEVICEHANDLE_UNKNOWN                      0xFF
#define FC_BLUETOOTH_CALLINSTANCE_INCOMINGCALLEVENT_RINGING    1
#define FC_BLUETOOTH_CLEAR_PRIVATEDATA_SUCCESS   0
#define FC_BLUETOOTH_CLEAR_PRIVATEDATA_FAILURE  -1
#define FC_BLUETOOTH_VALID_BD_ADDRESS_LENGTH    12

#define FC_BLUETOOTH_AUDIOCHANNEL_MONORINGTONE     0x01
#define INCOMING_CALL_RINGTONE_ID 0xFFFF
#define VOLUME_SLIDING_RINGTONE_ID 0xFE
#define INVALID_DEVICE_HANDLE 0xFF

#define BT_SERVICE_FILEPATH         "/opt/bosch/connectivity/bluetoothservice/bluetoothservice_out.out"
#define BT_RINGTONE_FILEPATH        "/var/opt/bosch/static/connectivity/tones/"
#define WIFI_POWER_ATT_DIAG_FILEPATH "/opt/bosch/connectivity/wifi/ugkz7_wifi_pwr_att_diag.sh"
#define RINGTONE_INBAND_RINGTONEID  0x01
#define DEFAULT_RINGTONE_ID         0x02
#define SYSTEM_WIDE_DEVICEHANDLE    0x00
#define BD_CMD_STRING_LEN           200

#define BT_ONLY_MODULE_RESET_FILEPATH "/opt/bosch/connectivity/bt_module/use_gpio37_for_bt_only_module_reset.txt"
#define BT_STATE_OFF_FILEPATH         "/var/opt/bosch/dynamic/connectivity/bt_state_off"

#define BT_MAX_RINGTONE_SUPPORTED   4 // Device RingTone and 10 Ringtones
#define NO_RINGTONE                 0xFF

#define DEFAULT_RSSI_VALUE -128

#define BT_LOCALMODE_ON    1
#define BT_LOCALMODE_OFF   2
#define BT_LOCALMODE_AUTO  3

#define NO_CHANGE                   0
#define LOCALPAIRABLEMODECHANGED    1
#define LOCALCONNECTABLEMODECHANGED 2
#define PAIRINGSTATUSCHANGED        3

#define PAIRED_DEVICE_LIST_CHANGED              1
#define DEVICE_CONNECTION_STATUS_LIST_CHANGED   2

#define REMOVE_ALL_METHOD           0
#define REMOVE_PARTICULAR_METHOD    1

#define ALSA_ALIGNMENT_THRESHOLD_NSEC (120000000)
#define PTHREAD_RT_PRIO                30

#define FC_BT_ROOTDAEMON_CLIENT_NAME    "FC_Bluetooth"
#define FC_BT_KDS_BIT_CONVERT          2

#define PTHREAD_COND_WAIT_TIMER_IN_SEC   5

#define _CCA_LOOPBACK_MSG_

//Internal FunctionIds
enum enFunctionID {
   BLUETOOTHSETTINGS_C_U16_IFID_INTERNAL= 0xFFF1
};

//Internal Messages ID
enum enMessageContentID {

   //Callback IDs for BMCore
   BTSET_BMCORE_RESPONSE_MSG_INVALID  = 0,

   BTSET_BMCORE_MSG_SWITCHBTSTATUS_RESPONSE = 100,
   BTSET_BMCORE_MSG_SWITCHBTSTATUS_CHANGED,
   BTSET_BMCORE_MSG_SWITCHDISCOVERYSTATUS_RESPONSE,
   BTSET_BMCORE_MSG_SWITCHLOCALPAIRABLEMODERESPONSE,
   BTSET_BMCORE_MSG_SWITCHLOCALCONNECTABLEMODERESPONSE,
   BTSET_BMCORE_MSG_PAIRINGPIN,
   BTSET_BMCORE_MSG_PAIRINGRESPONSE,
   BTSET_BMCORE_MSG_STARTAUTOCONNECT,
   BTSET_BMCORE_MSG_SENDPAIRINGREQUESTRESPONSE,
   BTSET_BMCORE_MSG_CHANGEDEVICESTATERESPONSE,
   BTSET_BMCORE_MSG_BTLIMITATIONMODERESPONSE,
   BTSET_BMCORE_MSG_CONNECTPROFILERESPONSE,
   BTSET_BMCORE_MSG_DISCONNECTPROFILERESPONSE,
   BTSET_BMCORE_MSG_BLOCKPROFILERESPONSE,
   BTSET_BMCORE_MSG_UNBLOCKPROFILERESPONSE,
   BTSET_BMCORE_MSG_CLEARDEVICEINFORESPONSE,
   BTSET_BMCORE_MSG_ADDOOBPAIREDDEVICERESPONSE,
   BTSET_BMCORE_MSG_SETFAVORITERESPONSE,
   BTSET_BMCORE_MSG_CLEARFAVORITERESPONSE,
   BTSET_BMCORE_MSG_STARTSERVICESEARCHRESPONSE,
   BTSET_BMCORE_MSG_STOPSERVICESEARCHRESPONSE,
   BTSET_BMCORE_MSG_SWITCHMULTIHFPSUPPORTRESPONSE,
   BTSET_BMCORE_MSG_SETDEVICEUSAGEPREFERENCERESPONSE,
   BTSET_BMCORE_MSG_SETPRIMARYRESPONSE,
   BTSET_BMCORE_MSG_SETBTPROFILEUSAGEINFORESPONSE,
   BTSET_BMCORE_MSG_NOTIFYUSERDECISIONRESPONSE,
   BTSET_BMCORE_MSG_SETPROFILEUSERAVAILABILITYRESPONSE,
   BTSET_BMCORE_MSG_REPLACEBTLIMITATIONMODERESPONSE,
   BTSET_BMCORE_MSG_BLOCKDEVICERESPONSE,
   BTSET_BMCORE_MSG_UNBLOCKDEVICERSPONSE,
   BTSET_BMCORE_MSG_PAIRINGSTATUS_CHANGED,
   BTSET_BMCORE_MSG_LOCALPAIRABLEMODE_CHANGED,
   BTSET_BMCORE_MSG_LOCALCONNECTABLEMODE_CHANGED,
   BTSET_BMCORE_MSG_SWITCHDISCOVERYSTATUS_CHANGED,
   BTSET_BMCORE_MSG_BMCONFIGDATA_CHANGED,
   BTSET_BMCORE_MSG_BMSERVICEAVAILABILITY_CHANGED,
   BTSET_BMCORE_MSG_AUTOCONNECTIONTYPE_CHANGED,
   BTSET_BMCORE_MSG_LOCALFRIENDLYNAME_CHANGED,
   BTSET_BMCORE_MSG_BTLOCALINFO_CHANGED,
   BTSET_BMCORE_MSG_PAIREDDEVICELIST_CHANGED,
   BTSET_BMCORE_MSG_CONNECTIONDEVICELIST_CHANGED,
   BTSET_BMCORE_MSG_BLOCKSTATUS_CHANGED,
   BTSET_BMCORE_MSG_BTLIMITATIONMODE_CHANGED,
   BTSET_BMCORE_MSG_DISCOVERDEVICE_CHANGED,
   BTSET_BMCORE_MSG_DISCOVERDEVICELIST_CHANGED,
   BTSET_BMCORE_MSG_LINKQUALITY_CHANGED,
   BTSET_BMCORE_MSG_TESTMODE_LINKQUALITY_CHANGED,
   BTSET_BMCORE_MSG_UPDATE_CONFIGURATION,
   BTSET_BMCORE_MSG_MULTIHFPSUPPORT_CHANGED,
   BTSET_BMCORE_MSG_BTPROFILEUSAGEINFO_CHANGED,
   BTSET_BMCORE_MSG_USERDECISIONREQUIRED_CHANGED,
   BTSET_BMCORE_MSG_AUTOCONNECTIONSTATUS_CHANGED
};

template<class ItemType>
class MeetsDeviceHandle
{
public:

   explicit MeetsDeviceHandle(const DeviceId deviceHandle) : _deviceHandle(deviceHandle)
   {
   }

   bool operator()(const ItemType& item) const
   {
      return (item._deviceHandle == _deviceHandle);
   }

private:
   DeviceId _deviceHandle;
};

typedef struct tagLoopBackMessageData
{
public:
    tU32 u32MessageContentId;
    tU32 u32CmdCntr;
    BmResult result;
    tVoid* vpData;

    tagLoopBackMessageData()
    {
        u32MessageContentId = 0;
        u32CmdCntr = 0;
        result = BM_RESULT_ERR_GENERAL;
        vpData = OSAL_NULL;
    }
} LoopBackMessageData;

enum enBlockInfo {
   AVP_UNBLOCKED = 0,
   AVP_DISCONNECT_BLOCKED,
   AVP_BLOCKED
};

typedef struct stdiscoveredDevice
{
   BdAddress bdAddress;
   BdName btDeviceName;
}DiscoveredDevice;

typedef struct stConnectBluetoothProfile
{
   most_btsetfi_tclMsgConnectBluetoothProfileMethodStart CBPFiDataObjectMethodStart;
   most_fi_tcl_BTServiceType BTServiceType;
}connectBluetoothProfileStruct;

typedef struct stDisconnectBluetoothProfile
{
   most_btsetfi_tclMsgDisconnectBluetoothProfileMethodStart DBPFiDataObjectMethodStart;
   most_fi_tcl_BTServiceType BTServiceType;
}disconnectBluetoothProfileStruct;

/**
 *  class forward declarations.
 */
class ahl_tclBaseOneThreadService;
class fc_bluetooth_tclApp;

/**
 * class FC_Bluetooth_tclCCAService
 * This class implements the BluetoothCCAService component.
 */
class FC_Bluetooth_tclCCAService : public ahl_tclBaseOneThreadService
{
public:

   /**
    * Constructor of the service class
    * @return
    */
   FC_Bluetooth_tclCCAService();

   /**
    * Constructor of the service class
    * @param[in] poMainAppl: Pointer to the CCA application
    * @return
    */
   FC_Bluetooth_tclCCAService(fc_bluetooth_tclApp *poMainAppl);

   /**
    * Destructor of the service class
    * @return
    */
   virtual ~FC_Bluetooth_tclCCAService();

   static FC_Bluetooth_tclCCAService* poGetInstance(tVoid)
   {
         return m_poCCAService;
   }

   /**
    * This function is used to call the framework function eUpdateClients()
    * @param[in] u16FunctionId = Function ID of the updating property.
    */
   void updateClients(tU16 u16FunctionId)
   {
      eUpdateClients(u16FunctionId);
   };

   tVoid vOnLoopbackService(amt_tclServiceData* poMessage);

   tVoid vPostLoopBackMessageToCCA(tU16 u16CallbackFuncID, tU32 u32CommandCounter);

   /**
    * Method Result
    */
   //TODO: 9/10: one method result to be added
   tVoid AddNewDeviceMethodResult(Job,const PairingStatus& btPairingStatus);
   tVoid ChangeDeviceStateMethodResult(Job, most_fi_tcl_e8_BTSetConnectionResult e8_BTSetConnectionResult, tBool bReplaceDeviceRequired = false);
   tVoid ConnectBluetoothProfileMethodResult(Job, most_btsetfi_tclMsgConnectBluetoothProfileMethodStart);
   tVoid PairingResponseMethodResult(Job, PairingResult result);
   tVoid SwitchBluetoothOnOffExtendedMethodResult(Job, SwitchState switchState);
   tVoid SendPairingRequestMethodResult(Job, const PairingStatus& btPairingStatus, BmResult result = BM_RESULT_OK);
   tVoid ScanBluetoothDevicesMethodResult(Job, const SwitchStatus& btSwitchStatus, tBool);
   tVoid DisconnectBluetoothProfileMethodResult(Job, most_btsetfi_tclMsgDisconnectBluetoothProfileMethodStart);
   tVoid ChangePinMethodResult(Job, LocalFriendlyNameType localFriendlyName, PairingPinType pin);
   tVoid PlayRingtoneMethodResult(Job);
   tVoid StopRingtoneMethodResult(Job, tBool bStatus);
   tVoid ConnectSPPServiceMethodResult(Job, tBool bStatus);
   tVoid StartAutoConnectSeqMethodResult(Job, tBool bStatus);
   tVoid SwitchBTLocalModeMethodResult(Job, tBool bStatus);
   tVoid SetBTLimitationModeMethodResult(Job, tBool bStatus);
   tVoid ChangeWifiTransmitPowerMethodResult(Job methodcontext, tBool bStatus);
   tVoid DisconnectSPPServiceMethodResult(Job, tBool bStatus);
   tVoid StartSearchSPPServicesMethodResult(Job, tBool bStatus, tU8 u8SearchHandle);
   tVoid StopSearchSPPServicesMethodResult(Job, tBool bStatus);
   tVoid AddOobPairedDeviceMethodResult(Job methodcontext, tBool bStatus);
   tVoid SetFavoriteMethodResult(Job methodcontext, tBool bStatus);
   tVoid ClearFavoriteMethodResult(Job methodcontext, tBool bStatus);
   tVoid SwitchMultiHFPSupportMethodResult(Job methodcontext, tBool bStatus);
   tVoid SetPrimaryMethodResult(Job methodcontext, tBool bStatus);
   tVoid SetBtPrfoilesUsageMethodResult(Job methodcontext, tBool bStatus);
   tVoid ResolveConflictsMethodResult(Job methodcontext, tBool bStatus);
   tVoid SetDeviceUsagePreferenceMethodResult(Job methodcontext, tBool bStatus);
   tVoid SetProfileUserAvailabilityMethodResult(Job methodcontext, tBool bStatus);
   tVoid ReplaceBluetoothLimitationModeMethodResult(Job methodcontext, tBool bStatus);
   tVoid StartServiceSearchMethodResult(Job methodcontext, tBool bStatus);
   tVoid StopServiceSearchMethodResult(Job methodcontext, tBool bStatus);
   tVoid ClearDeviceInfoMethodResult(Job, tBool bStatus);

   /**
    * Method Error
    */
   template<typename T> tVoid MethodError(T oFiDataObjectMethodError,Job, tU16 u16ErrorCode);
   tVoid funcResponseError(tU16 uiFunction_ID, tU16 u16ErrorCode, tBool bRemove);
   tVoid sendMethodError(tU16 u16FunctionId, Job, tU16 u16ErrorCode);
   tVoid sendMethodResult(tU16 u16FunctionId);
   tVoid ConnectBluetoothProfileMethodError(Job& methodcontext, tU16& u16Count);
   tVoid DisconnectBluetoothProfileMethodError(Job& methodcontext, tU16& u16Count);
   tVoid ChangeDeviceStateMethodError(Job& methodcontext, tU16& u16Count);
   tVoid PlayRingtoneMethodError();

   tVoid ResetPairingResponseType();
   tS32  clearAllPrivateData();
   tVoid restoreRadioSettings();
   tVoid playRingTone();
   tVoid stopRingTone();

   //Outgoing Status
   tU16 u16GetDeviceHandleForCurrentOutgoingSource(const DeviceConnectionStatusList& deviceConnectionStatusList);
   tU16 u16GetDeviceHandleForNonOutgoingSource(const DeviceConnectionStatusList& deviceConnectionStatusList);
   tVoid vGetPairedDeviceList(PairedDeviceList& pairedDeviceList);
   tVoid vGetDeviceConnectionStatusList(DeviceConnectionStatusList& deviceConnectionStatusList);
   tVoid vGetVehicleAddress(BdAddress& bdAddress);
   tVoid vGetPairingPin(PairingPin& pairingPin);
   tVoid vGetLocalFriendlyName(LocalFriendlyNameType& btLocalFriendlyName);
   tVoid vGetLinkQualityTestMode(TestModeLinkQuality& linkQuality);
   tVoid vGetCurrentOutGoingSource(tU16& u16OutgoingSourceDeviceHandle);
   tVoid vGetBlockStatus(BlockStatus& blockStatus);
   tVoid vGetAutoConnectionType(tBool& bAutoconnectOnOff);
   tVoid vGetBluetoothOnOff(SwitchState& switchState);
   tVoid vGetDiscoveryStatus(SwitchStatus& switchStatus);
   tVoid vGetPairingState(PairingState &state);
   tVoid vGetmaxNumPairedDevices(tU16& maxNumPairedDevices);
   tVoid vGetLinkQuality(LinkQuality& linkQuality);
   tVoid vGetPairingStatus(PairingStatus& btPairingStatus);

   tVoid vMaxDevicesPairedReached(const PairedDeviceList& pairedDeviceList);
   tVoid vUpdateRssiValue(const LinkQuality& linkQuality);
   tVoid vUpdatePairingStatus(PairingResult result, BdName remoteBdName, BdAddress remoteBdAddress);
   tVoid vUpdatePairingRequest(const PairingStatus& btPairingStatus);
   tVoid vUpdatePairableMode(tU8 update = NO_CHANGE, SwitchState switchState = SWITCH_STATE_SWITCHED_OFF,
         PairingState state = BM_PAIRING_STATE_UNKNOWN, BdAddress remoteBdAddress = "");
   tVoid vUpdateBluetoothOnOff(SwitchState switchState);
   tVoid vUpdateBtLocalFriendlyNameSet(LocalFriendlyNameType localFriendlyName);
   tVoid vUpdateVehicleBtAddress(BdAddress bdAddress);
   tVoid vUpdateVehicleSupportedProfiles(ProtocolInfoMap protocolInfos);
   tVoid vUpdateVehicleFriendlyName(LocalFriendlyNameType localFriendlyName);
   tVoid vUpdateStackInitialized(bool stackInitialized);
   tVoid vUpdatePairingPin(PairingPinType pin);
   tVoid vUpdateAutoConnectOnOff(StandardAutoConnectionType type);
   tVoid vUpdateList();
   tVoid vUpdatePairedList(const PairedDeviceList& pairedDeviceList);
   tVoid vUpdateConnectionStatusList(const DeviceConnectionStatusList& deviceConnectionStatusList);
   tVoid vUpdateDeviceListExtended(const PairedDeviceList& pairedDeviceList, const DeviceConnectionStatusList& deviceConnectionStatusList);
   tVoid vUpdateBmConfigData(const BmConfigData& bmConfigData);
   tVoid vUpdateBmServiceAvailability(const BmServiceAvailability& bmServiceAvailability);
   tVoid vUpdateVehicleConfiguration(const LocalInfo& localInfo);
   tVoid vUpdateLinkQualityTestMode(const LinkQualityInfo& linkQualityInfo);
   tVoid vUpdateSppServiceConnection(const DeviceConnectionStatusList& deviceConnectionStatusList);
   tVoid vUpdateBlockStatus(const BlockStatus& blockStatus);
   tVoid vUpdateBluetoothAudioSource(tU8 u8DeviceHandle, const PairedDeviceList&);
   tVoid vUpdateBtLimitationMode(const BtLimitationMode& btLimitationMode);
   tVoid vUpdateDiscoveryStatus(SwitchStatus switchStatus);
   tVoid vPairingStatusChanged(const PairingStatus& btPairingStatus);
   tVoid vUpdateDiscoveredDeviceList(const DiscoveredDeviceList& discoveredDeviceList);
   tBool bCheckLinkQualityStatus(const char* bdAddress,LinkQualityInfo& linkQualityInfo);
   tVoid vUpdateConfiguration(const BmCoreConfiguration& bmCoreConfiguration);
   tVoid vUpdateMultiHFPSupportStatus(SwitchState switchState);
   tVoid vUpdateProfileUsageList(const ProfileUsageList& profileUsageList);
   tVoid vUpdateIssueInfoList(const IssueInfoList& issueInfoList);
   tVoid vUpdateAutoConnectionStatus(const AutoConnectionStatus& autoConnectionStatus);

   tVoid setBlockInfoStauts(DeviceId deviceHandle, enBlockInfo blockInfo);
   tVoid getBlockInfoStauts(DeviceId deviceHandle, enBlockInfo& blockInfo);

   //Device Ringtone details
   FC_Bluetooth_RingtoneList oAvailableRingtoneList;
   tVoid updateRingtoneValues(tBool bClearRingtoneValue);
   tBool bGetBTOnOffState();

   tBool _bBluetoothServiceIsAvail;

   std::map<DeviceId, enBlockInfo> _blockAVPInfo;

protected:

   /**
    * vOnServiceAvailable:
    * This function is called by the CCA framework when the service which is offered by this
    * server has become available.
    * 
    * @return void
    */
   virtual tVoid vOnServiceAvailable();

   /**
    * vOnServiceUnavailable:
    * This function is called by the CCA framework when the service which is offered by this
    * server has become unavailable.
    *
    * @return void
    */
   virtual tVoid vOnServiceUnavailable();

   /**
    * bStatusMessageFactory:
    * This function is called by the CCA framework to request ANY property which is offered by this
    * service. For each property accessed via parameter 'u16FunctionId' the user has to prepare the
    * corresponding FI data object which is then copied to the referenced parameter 'roOutMsg'.
    * 
    * Overrides method ahl_tclBaseOneThreadService::bStatusMessageFactory().
    *
    * @param[in]   u16FunctionId = Function ID of the requested property.
    * @param[in]   poInMsg = Selector message which is used to select dedicated
    *                           content to be copied to 'roOutMsg' instead of
    *                           updating the entire FI data object.
    * @param[out]  roOutMsg = Reference to the service data object to which the
    *                             content of the prepared FI data object should be
    *                             copied to.
    *
    * @return:     TRUE  = Status message for property successfully generated.
    *              FALSE = Failed to generate status message for property.
    */
   virtual tBool bStatusMessageFactory(tU16 u16FunctionId, amt_tclServiceData& roOutMsg, amt_tclServiceData* poInMsg);

   /**
    * bProcessSet:
    * This function is called by the CCA framework when it has received a message for a property with Opcode
    * 'SET' or 'PURESET' and there is no dedicated handler function defined in the message map for this pair
    * of FID and opcode. The user has to set the application specific property to the requested value and the
    * CCA framework then cares about informing the requesting client as well as other registered clients.
    *  
    * @param[in]  poMessage = Property to be set.
    * @param[out] bPropertyChanged = Property changed flag to be set to TRUE if property has changed. Otherwise to be set to FALSE (default).
    * @param[out] u16Error = Error code to be set if a CCA error occurs, otherwise don't touch.
    *
    * @return:    TRUE  = Send 'STATUS' message to the requesting client if opcode
    *                     was 'SET'. Send 'STATUS' message to other registered
    *                     clients as well if [OUT] parameter 'bPropertyChanged'
    *                     is set to TRUE.
    *             FALSE = Send an error message to the requesting client.
    */
   virtual tBool bProcessSet(amt_tclServiceData* poMessage, tBool& bPropertyChanged, tU16& u16Error);

   /**
    * vPostResult:
    * This function shall be called when an answer message is sent inside a handler function
    *
    * @param[in] rBaseMessage = Reference to the message related FI data object. All message specific parameters are already set
    *                           The parameters of the CCA header are set inside this functions. This function also deletes the FI data object.
    * @param[in] rServiceMethodStart = Reference to the received message
    *
    * @return void
    */
   tVoid vPostResult(most_btsetfi_tclMsgBaseMessage &rBaseMessage, amt_tclServiceData &rServiceMethodStart);

   /**
    * vPostError:
    * This function shall be called when an error is sent inside a handler function
    *
    * @param[in] rServiceMethodStart Reference to the received message
    * @param[in] errorCode  Error code
    * @param[in] errorParam Error Info
    *
    * @return void
    */
   tVoid vPostError(amt_tclServiceData &rServiceMethodStart, most_fi_tcl_e8_ErrorCode::tenType errorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE,tU8 errorParam = 0);

   /**
    *
    * Message map definition macro
    */

   DECLARE_MSG_MAP(FC_Bluetooth_tclCCAService)

   /**
    * Convert functions for MOST to Internal
    */

   /**
    * function converts bluetoothOnOff (boolean) to targetSwitchState
    * with if case statement
    */
   void MapBluetoothOnOffToInternal(TargetSwitchState &targetSwitchState, const tBool bluetoothOnOff);

   /**
    * function converts autoConnectOnOff (boolean) to autoConnectionType
    * with if case statement
    */
   void MapAutoConnectOnOffToInternal(AutoConnectionType &autoConnectionType, const tBool autoConnectOnOff);

   void MapLimitationFeatureToInternal(LimitationFeature &limitationFeature, most_fi_tcl_e8_BTTechnology e8Technology);

   void MapLimitationCommunicationIfToInternal(LimitationCommunicationIf& limitationCommunicationIf,
         most_fi_tcl_e8_BTCommunicationChannel e8CommunicationChannel);

   void MapLimitationActionToInternal(LimitationAction &limitationAction, most_fi_tcl_e8_BTLimitationAction e8Action);

   void MapLimitationActionStateToInternal(LimitationState &limitationActionState, most_fi_tcl_e8_BTLimitationActionState e8ActionState);

   void MapOobTypeToInternal(OobType& oobType, most_fi_tcl_e8_BTOobType e8OobType);

   void MapFavoriteTypeToInternal(FavoriteType& favoriteType, most_fi_tcl_e8_BTFavoriteType e8FavoriteType);

   void MapDeviceUsagePreferenceToInternal(UsagePreference& usagePreference, most_fi_tcl_e8_BTSetUsagePreference e8DeviceUsagePreference);

   void MapBTProfileToInternal(Protocol& protocol, most_fi_tcl_e8_BTProfile e8BTProfile, Uuid uuid);

   void MapProfileUsageTypeToInternal(BTProfileUsageType& profileUsageType, most_fi_tcl_e8_BTProfileUsageType e8BTProfileUsageType);

   void MapConflictTriggerToInternal(ConflictTrigger& conflictTrigger, most_fi_tcl_e8_BTConflictTrigger e8BTConflictTrigger);

   void MapResolutionTypeToInternal(UserDecision& userDecision, most_fi_tcl_e8_BTResolutionType e8BTResolutionType);

   void MapBTUserAvailabilityToInternal(bool& available, most_fi_tcl_e8_BTUserAvailable e8UserAvailable);

   /**
    * Convert functions for Internal to MOST
    */

   /**
    * function converts PairingType to most_fi_tcl_e8_BTSetPairingType
    * with switch case statement
    */
   void MapPairingTypeToMOST(most_fi_tcl_e8_BTSetPairingType &mostPairingType, const PairingType pairingType);

   /**
    * function converts PairingRequestOrigin to most_fi_tcl_e8_BTSetOrigin
    * with switch case statement
    */
   void MapOriginTypeToMOST(most_fi_tcl_e8_BTSetOrigin &mostOrigin, const PairingRequestOrigin pairingRequestOrigin);

   /**
    * function converts PairingStatus to most_fi_tcl_e8_BTSetPairingStatusType
    * with switch case statement
    */
   void MapPairingStatusTypeToMOST(most_fi_tcl_e8_BTSetPairingStatusType &mostPairingStatusType, const PairingResult pairingResult);

   /**
    * function converts PairingStatus to most_fi_tcl_e8_BTSetPairingStatusType
    * with switch case statement
    */
   void MapSetPairingResultToMOST(most_fi_tcl_e8_BTSetPairingResult &mostPairingResult, const PairingState pairingState);

   /**
    * function converts DeviceStatus to most_fi_tcl_e8_BTSetDeviceStatus
    * with switch case statement
    */
   void MapDeviceStatusTypeToMOST(most_fi_tcl_e8_BTSetDeviceStatus &mostDeviceStatus, const ConnectionStatus connectionStatus, tBool bHFP);

   /**
    * function converts DeviceStatus to most_fi_tcl_e8_BTSetDeviceStatus
    * with switch case statement
    */
   void MapConnectionStatusTypeToMOST(most_fi_tcl_e8_BTConnectionStatus &mostConnectionStatus,
         ConnectionStatus connectionStatus);

   /**
    * function converts SwitchState to most_btsetfi_tclMsgBluetoothOnOffStatus
    * with switch case statement
    */
   void MapBluetoothOnOffStatusToMOST(most_btsetfi_tclMsgBluetoothOnOffStatus &mostBTOnOff, const SwitchState switchState);

   /**
    * function converts SupportedProfiles to most_fi_tcl_BTSetSupportedProfiles
    * with switch case statement
    */
   void MapSupportedProfilesStatusToMOST(most_fi_tcl_BTSetSupportedProfiles &mostSupportedProfiles, const ProtocolInfoMap protocolInfoMap);

   /**
    * function converts DeviceProfiles to most_fi_tcl_BTServiceType
    * with if statement
    */
   void MapDeviceProfilesStatusToMOST(most_fi_tcl_BTServiceType &mostBTServiceType, most_fi_tcl_BTProfileConnectionStatus &mostBTProfileConnectionStatus,
         most_fi_tcl_BTProfileDisconnectedReason &mostBTDeviceProfileDisconnectedReason, const ProtocolConnectionInfoMap protocolConnectionInfoMap, DeviceId deviceHandle);

   /**
    * function converts protocolConnectionInfoMap to bOutgoingSourceStatus
    * with if statement
    */
   void MapOutgoingSourceStatusToMOST(tBool &bOutgoingSourceStatus, const ProtocolConnectionInfoMap protocolConnectionInfoMap);

   /**
    * function converts MajorDeviceClass to most_fi_tcl_e8_BTSetMajorDeviceClass
    * with switch case statement
    */
   void MapMajorDeviceClassTypeToMOST(most_fi_tcl_e8_BTSetMajorDeviceClass &mostMajorClass, const BTSMajorDeviceClass majorDeviceClass);

   /**
    * function converts DeviceIdentification to most_fi_tcl_BTSetDeviceIdentification
    * with switch case statement
    */
   void MapDeviceIdentificationToMOST(most_fi_tcl_BTSetDeviceIdentification &mostDeviceIdentification, const DeviceIdentification deviceIdentification);

   /**
    * function converts DisconnectedReason to MapDeviceDisconnectedReasonToMOST
    * with switch case statement
    */
   void MapDeviceDisconnectedReasonToMOST(most_fi_tcl_e8_BTSetDeviceDisconnectedReason &mostDeviceDisconnectedReason, const DisconnectedReason disconnectedReason);

   /**
    * function converts DisconnectedReason to MapDeviceDisconnectedReasonToMOST
    * with switch case statement
    */
   void MapSppDisconnectionReasonToMOST(most_fi_tcl_e8_BTDisconnectionReason &mostSppDisconnectedReason, DisconnectedReason disconnectedReason);

   /**
    * function converts seconds to most_fi_tcl_BTSetDateTimeStamp
    * with switch case statement
    */
   void MapsecondsToMOST(most_fi_tcl_BTSetDateTimeStamp &mostDateTimeStamp, const unsigned long long int seconds);

   /**
    * function converts switchStatus to scan (boolean)
    * with if case statement
    */
   void MapSwitchDiscoveryStatusToInternal(TargetSwitchState &switchState, const tBool scan);

   /**
    * function converts SwitchStatus to most_fi_tcl_e8_BTSetScanningState
    * with switch case statement
    */
   void MapSwitchStatusToMOST(most_fi_tcl_e8_BTSetScanningState &mostScanningState, const SwitchStatus& btSwitchStatus, tBool directresponse);

   /**
    * function converts BluetoothFeatureSupport to most_fi_tcl_BTSetBluetoothFeatureSupport
    * with switch case statement
    */
   void MapBluetoothFeatureSupportToMOST(most_fi_tcl_BTSetBluetoothFeatureSupport &mostBluetoothFeatureSupport, const ProtocolInfoMap protocolInfoMap, tU8 u8DeviceHandle);

   /**
    * function converts AudioStreamingSupportType to most_fi_tcl_e8_BTSetAudioStreamingSupportType
    * with switch case statement
    */
   void MapAudioStreamingSupportTypeToMOST(most_fi_tcl_e8_BTSetAudioStreamingSupportType &mostAudioStreamingSupport, const ProtocolVersion version);

   /**
    * function converts AudioStreamingSupportType to most_fi_tcl_e8_BTSetAudioStreamingSupportType
    * with switch case statement
    */
   void MapAutoConnectOnOffStatusToMOST(most_btsetfi_tclMsgAutoConnectOnOffStatus &mostAutoConnectOnOffStatus, const AutoConnectionType autoConnectionType);

   void MapLimitationFeatureToMost(most_fi_tcl_e8_BTTechnology& e8Technology, const LimitationFeature limitationFeature);

   void MapLimitationCommunicationIfToMost(most_fi_tcl_e8_BTCommunicationChannel& e8CommunicationChannel,
         const LimitationCommunicationIf limitationCommunicationIf);

   void MapLimitationActionStateToMost(most_fi_tcl_e8_BTLimitationActionState& e8ActionState, const LimitationState limitationAction);

   void MapVisibleDeviceConnectionStatusToMOST(most_fi_tcl_e8_BTSetVisibleDeviceConnectionStatus &moste8VisibleDeviceConnectionStatus,
         DeviceId deviceHandle, ConnectionStatus connectionStatus);


   void MapDeviceListExtendedConnectionStatusToMOST(tBool &bDeviceConnectionStatus, DeviceId deviceHandle, ConnectionStatus connectionStatus);

   void MapFavoriteProfilesIndexToMOST(most_fi_tcl_BTProfileFavoriteIndex &mostBTProfileFavoriteIndex, const ProtocolInfoMap protocolInfoMap);

   void MapMultiHFPSupportStatusToMOST(most_fi_tcl_e8_BTSwitchStatus &mostBTSwitchStatus, const SwitchState switchState);

   void MapDeviceUsagePreferenceToMOST(most_fi_tcl_e8_BTUsagePreference &mostDeviceUsagePreference, const UsagePreference usagePreference);

   void MapBtProfilesToMost(most_fi_tcl_e8_BTProfile& e8BTProfile, ProtocolId protocolId);

   void MapBTProfileUsageTypeToMost(most_fi_tcl_e8_BTProfileUsageType& e8BTProfileUsageType, BTProfileUsageType profileUsageType);

   void MapConflictStateToMost(most_fi_tcl_e8_BTConflictState& e8ConflictState, ConflictState conflictState);

   void MapConflictTypeToMost(most_fi_tcl_e8_BTConflictType& e8BTConflictType, ConflictType conflictType);

   void MapConflictTriggerToMost(most_fi_tcl_e8_BTConflictTrigger& e8BTConflictTrigger, ConflictTrigger conflictTrigger);

   void MapAutoConnectionStatusToMOST(most_fi_tcl_e8_BTAutoconnectionStatus& e8AutoConnectionStatus, AutoConnectionState autoConnectionState);

   void MapServiceAvailabilityToMOST(most_fi_tcl_e8_BTServiceAvailable& e8ServiceAvailable, ServiceAvailability serviceAvailability);

private:

   fc_bluetooth_tclApp* _bluetoothApp; /** Pointer to Main Application */

   tBool _bIsRingtonePlaying;
   tBool _bIsStopRingtoneinProgress;

   Job _playRingtone;
   Job _stopRingtone;
   BdAddress _bdAAPAddress;

   // indices of filtered devices
   ConnectionStatus  _e8connectionStatus;
   ConnectionStatus  _e8prevconnectionStatus;
   tU8 _u8DeviceHandle;
   tU8 _u8prevDeviceHandle;
   most_fi_tcl_BTSetDeviceListChangeExtended _oDeviceListChange;
   tU16 _outgoingSourceDeviceHandle;
   tU16 _u16PBAPDeviceHandle;
   tU16 _u16MAPDeviceHandle;
   tU16 _u16PANDeviceHandle;
   tBool _bVersionUpdate;
   tU8 _u8BluetoothAudioDeviceHandle;
   ProtocolVersion _bluetoothAudioVersion;

   tU16 _u16SystemWideRingtone;
   tU8 _u8SystemWideDeviceHandle;
   tU8 _u8PrevRingtoneIDDeviceHandle;
   InbandRingtoneSupport _ringtoneIDInBandSupport;
   PairingState _pairingState;
   PairingStatus _btPairingStatus;

   //Max device
   tU16 _maxNumPairedDevices;
   PairingPinType _pairingPin;
   LocalFriendlyNameType _btLocalFriendlyName;
   BdAddress _vehicleBtAddress;
   LinkQualityInfo _linkQualityTestModeInfo;

   //Switchbtlocalmode
   tU8 m_u8Pairable;
   tU8 m_u8Connectable;

   BtLimitationMode _btLimitationModeList;

   PairedDeviceList _pairedDeviceList;
   DeviceConnectionStatusList _deviceConnectionStatusList;
   BlockStatus _blockStatus;
   SwitchState _bluetoothswitchState;
   SwitchStatus _discoveryStatus;
   tBool _bAVPStatusUpdate;
   tBool _bPBAPStatusUpdate;
   tBool _bMAPStatusUpdate;
   tBool _bPANStatusUpdate;
   tBool _bPairedDevicelistUpdate;
   tBool _bHFPStatusUpdate;
   tBool _bDeviceListChangeUpdate;
   tBool _bProtocolStatusUpdate;
   tBool _initPairedDeviceListReceived;
   tBool _bUpdateRingtoneStatus;
   InbandRingtoneSupport _inbandRingtoneSupport;
   tBool _bUpdateBtDTCState;
   bool _bStackInitialized;

   BmCoreConfiguration _bmCoreConfiguration;

   //linkquality
   LinkQuality _linkQuality;

   typedef std::list<Job> JobList;

   static FC_Bluetooth_tclCCAService *m_poCCAService;

   most_btsetfi_tclMsgChangeRingtoneStatus _oFIDataChangeRingtone;
   most_btsetfi_tclMsgPairingStatusStatus _oFIDataPairingStatus;
   most_btsetfi_tclMsgPairingRequestStatus _oFIDataPairingRequest;
   most_btsetfi_tclMsgPairableModeStatus _oFIDataPairableMode;
   most_btsetfi_tclMsgBluetoothOnOffStatus _oFIDataBluetoothOnOff;
   most_btsetfi_tclMsgVehicleBTFriendlyNamesetStatus _oFIDataVehicleBTFriendlyNameset;
   most_btsetfi_tclMsgVehicleBTAddressStatus _oFIDataVehicleBTAddress;
   most_btsetfi_tclMsgVehicleSupportedBTProfilesStatus _oFIDataVehicleSupportedBTProfiles;
   most_btsetfi_tclMsgVehicleBTFriendlyNameStatus _oFIDataVehicleBTFriendlyName;
   most_btsetfi_tclMsgPairingPinStatus _oFIDataPairingPin;
   most_btsetfi_tclMsgAutoConnectOnOffStatus _oFIDataAutoConnectOnOff;
   most_btsetfi_tclMsgDeviceListExtendedStatus _oFIDataObjectDeviceListExtended;
   most_btsetfi_tclMsgDeviceListExtendedStatus _oFIDataObjectDeviceListExtendedPrev;
   most_btsetfi_tclMsgMaxDevicesPairedStatus _oFIDataMaxDevicesPaired;
   most_btsetfi_tclMsgSignalStrengthStatus _oFIDataSignalStrength;
   most_btsetfi_tclMsgSPPServiceConnectionStatus _oFIDataSPPServiceConnection;
   most_btsetfi_tclMsgBluetoothAudioSourceStatus _oFIDataBluetoothAudioSource;
   most_btsetfi_tclMsgBluetoothLimitationModeStatus _oFIDataBTLimitationMode;
   most_btsetfi_tclMsgVisibleDevicesStatus _oFIDataVisibleDevicesStatus;
   most_btsetfi_tclMsgMultiHFPSupportStatus _oFIDataMultiHFPSupport;
   most_btsetfi_tclMsgBTProfilesUsageInfoStatus _oFIDataBTProfilesUsageInfo;
   most_btsetfi_tclMsgConflictsDetectedStatus _oFIDataConflictsDetected;
   most_btsetfi_tclMsgAutoConnectionStatusStatus _oFIDataAutoConnectionStatus;
   most_btsetfi_tclMsgServiceAvailableStatus _oFIDataServiceAvailableStatus;
   most_btsetfi_tclMsgVehicleConfigurationStatus _oFIDataVehicleConfiguration;

   /**
    * Lock to avoid parallel access
    */
   Lock _sendMethodErrorLock;
   Lock _avpBlockInfoLock;

   /**
    * Method start used by Message Map.
    */

   /**
    * BT Local Info
    */

   //TODO: Comments should be added for all functions

   tVoid onSwitchBluetoothOnOffExtendedMethodStart(amt_tclServiceData* poMessage);

   tVoid onScanBluetoothDevicesMethodStart(amt_tclServiceData* poMessage);

   /**
    * Device Info
    */


   tVoid onGetDeviceInfoExtendedMethodStart(amt_tclServiceData* poMessage);

   /**
    * Device Pairing and Connection
    */

   tVoid onAddNewDeviceMethodStart(amt_tclServiceData* poMessage);

   tVoid onChangePinMethodStart(amt_tclServiceData* poMessage);

   tVoid onPairingResponseMethodStart(amt_tclServiceData* poMessage);

   tVoid onChangeDeviceStateMethodStart(amt_tclServiceData* poMessage);

   tVoid onReplaceDeviceMethodStart(amt_tclServiceData* poMessage);

   tVoid onClearDeviceInfoMethodStart(amt_tclServiceData* poMessage);

   tVoid onConnectBluetoothProfileMethodStart(amt_tclServiceData* poMessage);

   tVoid onDisconnectBluetoothProfileMethodStart(amt_tclServiceData* poMessage);

   tVoid onSendPairingRequestMethodStart(amt_tclServiceData* poMessage);

   tVoid OnPlayRingtoneMethodStart(amt_tclServiceData* poMessage);

   tVoid OnStopRingtoneMethodStart(amt_tclServiceData* poMessage);

   tVoid OnRequestAvailableRingtonesListMethodStart(amt_tclServiceData* poMessage);

   tVoid OnRequestDeviceRingtonesListMethodStart(amt_tclServiceData* poMessage);

   tVoid OnConnectSPPServiceMethodStart(amt_tclServiceData* poMessage);

   tVoid OnSwitchBTLocalModeMethodStart(amt_tclServiceData* poMessage);

   tVoid OnStartAutoConnectSequenceMethodStart(amt_tclServiceData* poMessage);

   tVoid OnSetBTLimitationModeMethodStart(amt_tclServiceData* poMessage);

   tVoid OnChangeWifiTransmitPowerMethodStart(amt_tclServiceData* poMessage);

   tVoid OnDisconnectSPPServiceMethodStart(amt_tclServiceData* poMessage);

   tVoid OnStartSearchSPPServicesMethodStart(amt_tclServiceData* poMessage);

   tVoid OnStopSearchSPPServicesMethodStart(amt_tclServiceData* poMessage);

   tVoid OnAddOobPairedDeviceMethodStart(amt_tclServiceData* poMessage);

   tVoid OnSetFavoriteMethodStart(amt_tclServiceData* poMessage);

   tVoid OnClearFavoriteMethodStart(amt_tclServiceData* poMessage);

   tVoid OnSwitchMultiHFPSupportMethodStart(amt_tclServiceData* poMessage);

   tVoid OnSetPrimaryMethodStart(amt_tclServiceData* poMessage);

   tVoid OnSetBtPrfoilesUsageMethodStart(amt_tclServiceData* poMessage);

   tVoid OnResolveConflictsMethodStart(amt_tclServiceData* poMessage);

   tVoid OnSetDeviceUsagePreferenceMethodStart(amt_tclServiceData* poMessage);

   tVoid OnSetProfileUserAvailabilityMethodStart(amt_tclServiceData* poMessage);

   tVoid OnReplaceBluetoothLimitationModeMethodStart(amt_tclServiceData* poMessage);

   /**
    * Handle functions for Get Properties
    */


   tBool handlePairingStatusGet(amt_tclServiceData& roOutMsg);

   tBool handleVehicleFriendlyNameGet(amt_tclServiceData& roOutMsg);

   tBool handleVehicleFriendlyNameSetStatus(amt_tclServiceData& roOutMsg);

   tBool handlePairingPinGet(amt_tclServiceData& roOutMsg);

   tBool handleMaxDevicesPairedGet(amt_tclServiceData& roOutMsg);

   tBool handleBluetoothOnOffGet(amt_tclServiceData& roOutMsg);

   tBool handleVehicleBtAddressGet(amt_tclServiceData& roOutMsg);

   tBool handleVehicleSupportedProfilesGet(amt_tclServiceData& roOutMsg);

   tBool handleAutoconnectOnOffGet(amt_tclServiceData& roOutMsg);

   tBool handlePairingRequestGet(amt_tclServiceData& roOutMsg);

   tBool handleDeviceListExtendedGet(amt_tclServiceData& roOutMsg);

   tBool handleVisibleDevicesGet(amt_tclServiceData& roOutMsg);

   tBool handleBluetoothAudioSourceGet(amt_tclServiceData& roOutMsg);

   tBool handleChangeRingtoneGet(amt_tclServiceData& roOutMsg);

   tBool handleSignalStrengthGet(amt_tclServiceData& roOutMsg);

   tBool handlePairableModeGet(amt_tclServiceData& roOutMsg);

   tBool handleSppServiceConnectionGet(amt_tclServiceData& roOutMsg);

   tBool handleBTLimitationModeGet(amt_tclServiceData& roOutMsg);

   tBool handleMultiHFPSupportGet(amt_tclServiceData& roOutMsg);

   tBool handleBtProfilesUsageInfoGet(amt_tclServiceData& roOutMsg);

   tBool handleConflictDetectedGet(amt_tclServiceData& roOutMsg);

   tBool handleAutoConnectionStatusGet(amt_tclServiceData& roOutMsg);

   tBool handleServiceAvailableGet(amt_tclServiceData& roOutMsg);

   tBool handleVehicleConfigurationGet(amt_tclServiceData& roOutMsg);

   tVoid vFillDeviceListChange(const PairedDeviceList& pairedDeviceList);

   tVoid vHandleChangedDeviceStatusOnDeviceListUpdate(tU16 u16OutgoingSourceDeviceHandle);

   tVoid updateChangeRingtoneStatus(tU8 u8DeviceHandle, tU16 u16RingtoneID, std::string sRingtoneName, tBool bUpdateClients = TRUE);

   tBool bIsDeviceAddressValid(const std::string& sDeviceAddress) const;

   tVoid vCheckAVPStatus(tBool& bAVPUpdate, const DeviceConnectionStatusList& deviceConnectionStatusList);

   tVoid vCheckHFPStatus(tBool& bHFPUpdate, const DeviceConnectionStatusList& deviceConnectionStatusList);

   tVoid vCheckPBAPStatus(tBool& bPBAPUpdate, const DeviceConnectionStatusList& deviceConnectionStatusList);

   tVoid vCheckMAPStatus(tBool& bMAPUpdate, const DeviceConnectionStatusList& deviceConnectionStatusList);

   tVoid vCheckPANStatus(tBool& bPANUpdate, const DeviceConnectionStatusList& deviceConnectionStatusList);

   tVoid vFindChangeProfileStatus(ProtocolId protocolId, tU16 u16InDeviceHandle,
         const DeviceConnectionStatusList& deviceConnectionStatusList, tBool& bUpdate, tBool& bProtocolStatusUpdate, tU16& u16OutDeviceHandle);

   tBool bCheckBlockStatus(tU8 u8DeviceHandle, most_fi_tcl_e8_BTSetChangeDeviceStateAction e8ChangeDeviceStateAction,
            most_fi_tcl_e8_BTSetConnectionResult& e8_BTSetConnectionResult);

   tVoid vCheckAVPConnectionStatus(tBool &bAVPStatus, ProtocolConnectionInfoMap protocolConnectionInfoMap);

   void updateConnectProfileMR(const DeviceConnectionStatusList&, Job& methodcontext, tBool& bJobFind, tU16& u16Count, tBool& bResponse);

   void updateDisconnectProfileMR(const DeviceConnectionStatusList&, Job& methodcontext, tBool& bJobFind, tU16& u16Count, tBool& bResponse);

   void ConnectProfileResponse(const DeviceConnectionStatusList&, tBool bResponse = FALSE, BmResult result = BM_RESULT_OK);

   void DisconnectProfileResponse(const DeviceConnectionStatusList&, tBool bResponse = FALSE, BmResult result = BM_RESULT_OK);

   void vBlockAndUnblockProfileResponse(const BlockStatus&, tBool bResponse = FALSE, BmResult result = BM_RESULT_OK);

   void updatePairingResponse(const PairingStatus&, tBool bResponse = FALSE, BmResult result = BM_RESULT_OK);

   void AddOobPairedDeviceResponse(BmResult result);

   void SetFavoriteResponse(BmResult result);

   void ClearFavoriteResponse(BmResult result);

   void MultiHFPSupportResponse(BmResult result);

   void SetDeviceUsagePreferenceResponse(BmResult result);

   void SetPrimaryResponse(BmResult result);

   void SetBtPrfoilesUsageInfoResponse(BmResult result);

   void ResolveConflictsResponse(BmResult result);

   void SetProfileUserAvailabilityResponse(BmResult result);

   void ReplaceBluetoothLimitationModeResponse(BmResult result);

   void StartServiceSearchResponse(BmResult result);

   void StopServiceSearchResponse(BmResult result);

   void StartAutoconnectSeqResponse(BmResult result);

   void ChangeDeviceStateResponse(const PairedDeviceList&, tBool bResponse = FALSE, BmResult result = BM_RESULT_OK);

   void updateChangeDeviceStateMR(const PairedDeviceList&, Job& methodcontext, tBool& bJobFind, tU16& u16Count, tBool& bResponse);

   void updateChangeDeviceStateBlockMR(const BlockStatus& blockStatus, Job& methodcontext, tBool& bJobFind, tU16& u16Count);


   tVoid updateBTOnOffExtenedMR(SwitchState, tBool bResponse);

   tVoid updateDiscoveryStatusOnOffMR(SwitchStatus, tBool);

   tVoid updatePairingPinMR(PairingPinType& pin);

   tVoid updateAddNewDeviceMR(const PairingStatus& btPairingStatus);

   tBool vCheckProfileStatus(DeviceId deviceHandle, ProtocolId protocolId, ConnectionStatus &connectionStatus, BlockState &blockState);

   tVoid vGetProfileBlockStatus(DeviceId deviceHandle, ProtocolId protocolId, BlockState &blockState);

   tVoid vCheckCurProfileStatusWithList(DeviceId deviceHandle, ProtocolId protocolId,
         const DeviceConnectionStatusList& deviceConnectionStatusList, ConnectionStatus &connectionStatus);

   tBool vCheckDeviceConnectionStatus(DeviceId deviceHandle, ConnectionStatus &connectionStatus);

   tVoid vCopyPairedDeviceList(const PairedDeviceList&);

   tVoid vCopyDeviceConnectionStatusList(const DeviceConnectionStatusList&);
   
   tVoid vCopyBlockStatus(const BlockStatus& blockStatus);

   tVoid vUpdateBlockInfo(const PairedDeviceList&);

   tBool bCheckSppServiceConnection(const DeviceConnectionStatusList& deviceConnectionStatusList);

   tVoid vPrintPairedDevicelist(const PairedDeviceList& pairedDeviceList);

   tVoid vPrintDeviceConnectionStatusList(const DeviceConnectionStatusList& deviceConnectionStatusList);

   tVoid vPrintBlockStatus(const BlockStatus& blockStatus);

   tVoid vCheckProtocoInfoChanged(ProtocolInfoMap curprotocolInfo, ProtocolInfoMap oldprotocolInfo);

   tVoid vCopyLinkQuality(const LinkQuality&);

   tVoid vCopyFIDataObjectDeviceListExtended();



   tVoid vSetPrevProfileStatus(DeviceId deviceHandle, ProtocolId protocolId, tBool& bProfileStatus);

   tVoid SwitchLocalPairableModeResponse(BmResult result);

   tVoid SwitchLocalConnectableModeResponse(BmResult result);

   tVoid vUpdateRingtoneSupport(DeviceId deviceHandle, InbandRingtoneSupport inbandRingtoneSupport);

   tVoid vCheckAndUpdateRingtoneSupport();

   tVoid vUpdateBtDTCState(BdAddress bdAddress);

   tVoid vUpdateWifiDTCState();

   bool bCheckStackIsRunning();

   bool bCheckBtStateOffFileIsExist();

   bool bCheckBtOnlyResetFileIsExist();

   tVoid checkForCreateOrDeleteBtStateOffFile(bool bBTOnOff, bool& delayBtOnRequest);

   BmResult GetInbandRingtoneSupport(tBool &binbandSupport, tU8 u8DeviceHandle);
};


// Common function
tS32 s32StringCompare(const char* ps8Str1, const char* ps8Str2);


#endif // _FC_BLUETOOTH_CCASERVICE_H_

