/**
 * @file FC_Bluetooth_AudioPlayer.h
 * @author
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 *
 * @addtogroup FC_Bluetooth
 *
 * @brief Implementation to Play audio files through GStreamer.
 * @{
 */

#ifndef FC_BLUETOOTH_AUDIOPLAYER_H_
#define FC_BLUETOOTH_AUDIOPLAYER_H_

//we need this define to avoid conflit between the overloaded
//new and delete operators when using Qt.
//Ref:- developer's handbook v0.17
#ifndef __PLACEMENT_NEW_INLINE
#define __PLACEMENT_NEW_INLINE 1
#endif

#include<unistd.h>
#include <sstream>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
//Required for ALSA
#include <alsa/asoundlib.h>
#include <alsa/version.h>
#include <string>
#include <gst/gst.h>
#include <glib.h>
#include <pthread.h>
#include <sched.h>

/**
 * @class FC_Bluetooth_AudioPlayer
 * @ingroup CCAInterfaceModule
 *
 * @brief This class is used for playing mp3 audio files using gstreamer.
 * This is a replacement for the player engine.
 *
 */
class FC_Bluetooth_AudioPlayer
{
public:
   /**
    * @brief Constructor of class FC_Bluetooth_AudioPlayer
    *
    * This class is used for intialising the member variables
    *
    *
    * @return void
    */
   FC_Bluetooth_AudioPlayer();

   /**
    * @brief Destructor of class FC_Bluetooth_AudioPlayer
    *
    * @return void
    */
   virtual ~FC_Bluetooth_AudioPlayer();

   /**
    * @brief Returns the instance of the class FC_Bluetooth_AudioPlayer
    *
    * This function returns the static instance of the class.
    *
    *
    * @return Static Instance of FC_Bluetooth_AudioPlayer
    */
   static FC_Bluetooth_AudioPlayer* getInstance();

   /**
    * @brief Deletes the static instance of the class FC_Bluetooth_AudioPlayer
    *
    * This should be called only once during the ignition cycle to
    * delete the static instance that is created.
    *
    * @return void
    */
   static void deleteInstance();

   /**
    * @brief Spawns the playback thread to start playback
    *
    * Receives the file name as input parameter and if valid spawns a new
    * thread to start the playback of the received audio file path.
    * The same function would be used for playing the ringtone selected either
    * from list or when an incoming call is received in the connected phone
    *
    * param[in] filePath File path of the audio file to be played
    * param[in] Repeat play the file repeatedly for incoming ringtone
    *
    * @return 1: success
    * @return 0: failure
    */
   bool playFile(const char *filePath, const tU16 u16RingtoneID, bool repeat = true);

   /**
    * @brief Thread handler for the thread spawned in playFile
    *
    * Once the thread is instantiated succesfully the thread handler is
    * executed. A gstreamer pipeline is created through playbin2 element.
    * Also a bus is attached to watch the pipeline and the main loop is run.
    *
    * param[in] fileInfo File path which was provided in pthread_create()
    *
    * @return void
    */
   static void *startPlayback(void* fileInfo);

   /**
    * @brief Stop the playback of the audio file
    *
    * This method would be invoked, when the EOS signal was indicated by the
    * GStreamer and the repeat option is not enabled; during source switch;
    * when the phone gets disconnected.
    *
    * param[in] fileInfo File path which was provided in pthread_create()
    *
    * @return void
    */
   static void stopPlayback();

   /**
    * @brief State change in the GstBus
    *
    * The state of the GstBus will be informed asynchronously. It will work
    * only if the the GMainLoop is run
    *
    * param[in] NewState The ALSA device name to be set
    * param[in] OldState The ALSA device name to be set
    *
    * @return void
    */
   static void gstStateChanged(GstState NewState, GstState OldState);

   /**
    * @brief The state of the GstBus
    *
    * The state of the GstBus will be informed asynchronously. It will work
    * only if the the GMainLoop is run
    *
    * @return boolean
    */
   static gboolean gstBusCallBack(GstBus *bus, GstMessage *msg, gpointer data);

   /**
    * @brief Set the alsa device name received from ARL
    *
    * This method would be invoked from bOnAllocate to set the ALSA device name
    * It is mandatory for this method to be called before the actual pipeline
    * is created and playback started.
    *
    * param[in] DeviceName The ALSA device name to be set
    *
    * @return boolean
    */
   static bool setAlsaDevice(const char *DeviceName);

   void vSetAttributeToPthread();

   void vSendSignaltoWaitingThread(tBool bResponseStatus);

private:
   static FC_Bluetooth_AudioPlayer* poAudioPlayer;

   static std::stringstream m_szAlsaDeviceName;

   static GMainLoop *gstMainLoop;

   static bool m_bPlayRepeated;

   static bool m_bIsPlayStarted;

   static bool m_bWaitforCondSignal;

   tBool m_bSetAttributes;
};

#endif // FC_BLUETOOTH_AUDIOPLAYER_H_
