/**
 * @defgroup PairingControllerModule Pairing Controller
 * @ingroup BmCoreModule
 * 
 * @brief This module comprises all parts related to pairing with a remote BT device
 */
 
/**
 * @file
 * @ingroup PairingControllerModule
 * 
 * @brief This file contains the declaration of the class PairingController
 * 
 * @author Stefan Reis
 */

#ifndef _PAIRING_CONTROLLER_H_
#define _PAIRING_CONTROLLER_H_

#include "PairingSm.h"
#include "ILocalSpm.h"
#include "FwISmTimerCallback.h"

namespace bmcore
{
   /**
    * @class PairingController
    * @ingroup PairingControllerModule
    * 
    * @brief The class PairingController implements the state machine PairingSm
    * 
    */
   class PairingController : public PairingSm, public ILocalSpm, public ::fw::ISmTimerCallback
   {
   
   public:
      /**
       * @brief Constructor of class PairingController which creates an instance
       *
       * No further initialization of internal components and resources
       * is performed in the constructor. See the explicit allocation
       * and initialization functions Create() and Init(). Hence, the
       * implementation is trivial.
       *
       * Synchronization: not re-entrant.
       *
       * Performance considerations: none.
       *
       * @return void
       */
      PairingController(ComponentId componentID);
   
      /**
       * @brief Destructor of class PairingController
       *
       * @return void
       */
      virtual ~PairingController();

      // ************************************
      // * Overridden methods of PairingSm *
      // ************************************
   
      Result initSm();
   
      Result handleStopSm();
   
      Result handleDone();

      /**
       * @brief Stores the pairing information available for pairing requests initiated by the local BT device
       * in corresponding members and updates Pairing Status property
       *
       * Calls PairingController::savePairingInfo(@a bdAddress, BM_PAIRING_REQUEST_ORIGIN_LOCAL, BM_PAIRING_TYPE_UNKNOWN, "").
       *
       * @param[in] bdAddress Bluetooth device address of remote BT device
       *
       * @return 0: success
       */
      Result savePairingInfoLocal(IN const BdAddressChrArr bdAddress);

      /**
       * @brief Stores the pairing information available for pairing requests initiated by the remote BT device
       * in corresponding members and updates Pairing Status property
       *
       * Calls PairingController::savePairingInfo(@a bdAddress, @ deviceName, BM_PAIRING_REQUEST_ORIGIN_REMOTE, @a pairingType, @a sspPinCode).
       *
       * @param[in] bdAddress Bluetooth device address of remote BT device
       * @param[in] pairingType Pairing type e.g. BM_PAIRING_TYPE_LEGACY (see PairingType)
       * @param[in] sspPinCode Simple Secure Pairing PIN code
       *
       * @return 0: success
       */
      Result savePairingInfoRemote(IN const BdAddressChrArr bdAddress, IN const DeviceNameChrArr deviceName, IN const PairingType pairingType, IN const SspPinCodeChrArr sspPinCode);

      Result startRemotePairingSession(void);

      /**
       * @brief Reserves unused Device Connection Controller
       *
       * Reserves unused Device Connection Controller by calling @a result = BMController::reserveDeviceConnectionController().
       *
       * If the BM Controller could successfully reserve a Device Connection Controller (@a result == 0):
       *
       *    If @p _pairingRequestOrigin == BM_PAIRING_REQUEST_ORIGIN_LOCAL:
       *
       *       The trigger PairingSm::START_PAIRING() is sent.
       *
       *    If @p _pairingRequestOrigin == BM_PAIRING_REQUEST_ORIGIN_REMOTE:
       *
       *       The trigger PairingSm::CHECK_AUTO_CONFIRMATION() is sent.
       *
       * If the BM Controller could not reserve a Device Connection Controller (@a result != 0):
       *
       *    If @p _pairingRequestOrigin == BM_PAIRING_REQUEST_ORIGIN_LOCAL
       *
       *       The trigger PairingSm::PAIRING_FINISHED(BM_PAIRING_RESULT_ERR_NO_RESSOURCE_AVAILABLE) is sent.
       *
       *    If @p _pairingRequestOrigin == BM_PAIRING_REQUEST_ORIGIN_REMOTE
       *
       *       Set @p _pairingResult to BM_PAIRING_RESULT_ERR_NO_RESSOURCE_AVAILABLE.
       *
       *       If @p _pairingType == BM_PAIRING_TYPE_LEGACY:
       *
       *          The pairing request is rejected by calling @a btsResult = BTStackIf::setPinCode(@p _bdAddress, "").
       *
       *       If @p _pairingType != BM_PAIRING_TYPE_LEGACY:
       *
       *          Get the appropriate SSP mode from the pairing type stored in @p _pairingType.
       *          The pairing request is rejected by calling
       *          @a btsResult = BTStackIf::confirmSecureSimplePairing(@p _bdAddress, @a sspMode, @p _sspPinCode, BTS_CONFIRM_SSP_REJECT).
       *
       * @return 0: success
       */
      Result checkDccReservation();

      /**
       * @brief Starts pairing with a remote BT device
       *
       * Starts pairing with a remote BT device by calling @a btsResult = BTStackIF::StartPairing(@p _bdAddress).\n
       * If @a btsResult != BTS_OK the message PairingSM::PAIRING_FINISHED(BM_PAIRING_RESULT_ERR_START_PAIRING_FAILED) is sent.
       *
       * @return 0: success
       */
      Result startPairing();

      /**
       * @brief Stores additional pairing information (pairing type and SSP PIN code) for pairing requests
       * initiated by the local BT device
       *
       * If @a bdAddress == @p _bdAddress:
       *
       *    @a pairingType and @a sspPinCode are stored in corresponding members @p _pairingType
       *    and @a sspPinCode and the message PairingSm::CHECK_AUTO_CONFIRMATION() is sent.
       *
       * If @a bdAddress != @p _bdAddress:
       *
       *    If @a pairingType == BM_PAIRING_TYPE_LEGACY:
       *
       *       The pairing request is rejected by calling @a btsResult = BTStackIf::setPinCode(@a bdAddress, "")
       *
       *    If @a pairingType != BM_PAIRING_TYPE_LEGACY:
       *
       *       Get the appropriate SSP mode from the pairing type given in @a pairingType.
       *       The pairing request is rejected by calling
       *       @a btsResult = BTStackIf::confirmSecureSimplePairing(@a bdAddress, @a sspMode, @a sspPinCode, BTS_CONFIRM_REJECT)
       *
       *
       * @param[in] bdAddress Bluetooth device address of remote BT device
       * @param[in] pairingType Pairing type e.g. BM_PAIRING_TYPE_LEGACY (see PairingType)
       * @param[in] sspPinCode Simple Secure Pairing PIN code
       *
       * @return 0: success
       */
      Result saveAdditionalPairingInfo(IN const BdAddressChrArr bdAddress, IN const DeviceNameChrArr deviceName, IN const PairingType pairingType, IN const SspPinCodeChrArr sspPinCode);

      /**
       * @brief Determines if a pairing request is to be confirmed automatically and updates the
       * PairingStatus property accordingly.
       *
       * Dependent on the pairing type and the pairing request origin currently stored in members
       * @p _pairingType and @p _pairingRequestOrigin one of the following configuration parameters is
       * checked:
       *
       *    AutoConfirmLocalLegacyPairingRequest.\n
       * or AutoConfirmRemoteLegacyPairingRequest.\n
       * or AutoConfirmLocalSspPairingRequest.\n
       * or AutoConfirmRemoteSspPairingRequest.\n
       *
       * If the value of the appropriate configuration parameter is true the trigger
       * PairingSm::AUTO_CONFIRM() is sent, if it is false the trigger PairingSm::USER_CONFIRMATION_NEEDED() is sent.
       *
       * The PairingStatus property is updated accordingly (userConfirmationRequired=YES/NO).
       *
       * @return 0: success
       */
      Result shallAutoConfirmPairingRequest();

      /**
       * @brief Accepts a pairing request
       *
       * If @p _pairingType == BM_PAIRING_TYPE_LEGACY:
       *
       *    Gets the legacy pairing PIN from BM Controller by calling @a pin = BMController::getLegacyPin()\n.
       *    Accepts pairing request by calling @a btsResult = BTStackIf::setPinCode(@p _bdAddress, @a pin).
       *
       * If @p _pairingType != BM_PAIRING_TYPE_LEGACY:
       *
       *    Get the appropriate SSP mode from the pairing type stored in @p _pairingType.
       *    Accepts pairing request by calling
       *    @a btsResult = BTStackIf::confirmSecureSimplePairing(@p _bdAddress, @a sspMode, @p _sspPinCode, BTS_CONFIRM_SSP_ACCEPT).
       *
       * @return 0: success
       */
      Result confirmPairing();

      /**
       * @brief Rejects a pairing request
       *
       * Set @p _pairingResult to BM_PAIRING_RESULT_ERR_PAIRING_REJECTED.
       *
       * If @p _pairingType == BM_PAIRING_TYPE_LEGACY:
       *
       *    The pairing request is rejected by calling @a btsResult = BTStackIf::setPinCode(@p _bdAddress, "").
       *
       * If @p _pairingType != BM_PAIRING_TYPE_LEGACY:
       *
       *    Get the appropriate SSP mode from the pairing type stored in @p _pairingType.
       *    The pairing request is rejected by calling
       *    @a btsResult = BTStackIf::confirmSecureSimplePairing(@p _bdAddress, @a sspMode, @p _sspPinCode, BTS_CONFIRM_SSP_REJECT).
       *
       * @return 0: success
       */
      Result rejectPairing();

      /**
       * @brief Rejects a pairing initiated by the remote BT device
       *
       * If @p _pairingRequestOrigin == BM_PAIRING_REQUEST_ORIGIN_REMOTE:
       *
       *    Set @p _pairingResult to BM_PAIRING_RESULT_ERR_NOT_PAIRABLE.
       *
       *    If @p _pairingType == BM_PAIRING_TYPE_LEGACY:
       *
       *       The pairing request is rejected by calling @a btsResult = BTStackIf::setPinCode(@p _bdAddress, "").
       *
       *    If @p _pairingType != BM_PAIRING_TYPE_LEGACY:
       *
       *       Get the appropriate SSP mode from the pairing type stored in @p _pairingType.
       *       The pairing request is rejected by calling
       *       @a btsResult = BTStackIf::confirmSecureSimplePairing(@p _bdAddress, sspMode, @p _sspPinCode, BTS_CONFIRM_SSP_REJECT).
       *
       * @return 0: success
       */
      Result handleNotPairable(IN const SwitchedOffReason reason);

      /**
       * @brief Cancels a pairing procedure initiated by the local BT device
       *
       * If @p _pairingRequestOrigin == BM_PAIRING_REQUEST_ORIGIN_LOCAL:
       *
       *    Set @p _pairingResult to BM_PAIRING_RESULT_ERR_PAIRING_CANCELED.
       *
       *    If @p _pairingType == BM_PAIRING_TYPE_UNKNOWN:
       *
       *       The pairing is canceled by calling @a btsResult = BTStackIf::cancelPairing(@p _bdAddress)
       *       and the trigger PairingSm::PAIRING_CANCELED(BM_PAIRING_RESULT_ERR_PAIRING_CANCELED) is sent.
       *
       *    If @p _pairingType == BM_PAIRING_TYPE_LEGACY:
       *
       *       The pairing is "canceled" by calling @a btsResult = BTStackIf::setPinCode(@p _bdAddress, "").
       *
       *    Else:
       *
       *       Get the appropriate SSP mode from the pairing type stored in @p _pairingType.
       *       The pairing request is "canceled" by calling
       *       @a btsResult = BTStackIf::confirmSecureSimplePairing(@p _bdAddress, sspMode, @p _sspPinCode, BTS_CONFIRM_SSP_REJECT).
       *
       * @return 0: success
       */
      Result cancelPairing();

      /**
       * @brief Check the attempt number of pairing
       *
       * The PairingStatus property is updated with the pairing result given in @a pairingResult,
       * the status is kept as PAIRING_ONGOING.
       * The PairingStatus property updated a second time (reset to default values, status is BM_PAIRING_STATE_IDLE again).
       *
       *
       *
       * If pairingResult != BM_PAIRING_RESULT_OK and attempt number of pairing is <= 3
       *
       *    Launch the pairing again  by calling
       *    PairingController::keepPairingSession().
       *
       * @param[in] pairingResult Pairing result
       *
       * @return 0: success
       */
      Result handleFinishedPairing(IN const BdAddressChrArr bdAddress, IN const PairingResult pairingResult);

      /**
       * @brief LAUNCH the pairing  and clears stored pairing information
       *
       *
       * The members @p _bdAddress, @p _pairingRequestOrigin, @p _pairingType, @p _sspPinCode
       * and @p _pairingResult are reset.
       *
       * The PairingStatus property updated a second time (reset to default values, status is BM_PAIRING_STATE_IDLE again).
       *
       * @return 0: success
       */
       Result keepPairingSession();

       /**
        * @brief Reports the pairing result and clears stored pairing information
        *
        * The PairingStatus property is updated with the pairing result given in @a pairingResult,
        * the status is kept as PAIRING_ONGOING.
        *
        * The members @p _bdAddress, @p _pairingRequestOrigin, @p _pairingType, @p _sspPinCode
        * and @p _pairingResult are reset.
        *
        * The PairingStatus property updated a second time (reset to default values, status is BM_PAIRING_STATE_IDLE again).
        *
        * The pairable mode is switched to the project specific value given in configuration
        * parameter DefaultLocalPairableMode by calling
        * BMController::doSwitchLocalPairableMode(DefaultLocalPairableMode, "", false).
        *
        * If pairingResult != BM_PAIRING_RESULT_OK:
        *
        *    The reserved Device Connection Controller is released by calling
        *    BMController::releaseDeviceConnectionController().
        *
        * @param[in] pairingResult Pairing result
        *
        * @return 0: success
        */
       Result stopPairingSession(IN const BdAddressChrArr bdAddress, IN const PairingResult pairingResult);

      // ************************************
      // * Overridden methods of ILocalSpm *
      // ************************************
   
      /**
       * @brief Creates the Pairing Controller instance
       *
       * Creates the Pairing state machine (including creation of message queue).
       * Informs LocalSpm that create is ready -> createDone(0).
       *
       * @attention: running in SPM thread context
       *
       * @return void
       */
      void create();
   
      /**
       * @brief Performs the initialization of the Pairing Controller instance, i.e.:
       * - initialization of member variables
       *
       * Synchronization: re-entrant
       *
       * Performance considerations:
       *
       * @return 0 on success and an error code otherwise.
       */
      Result init(InitReason reason);
   
      /**
       * @brief Starts the Pairing Controller instance, implies that all other
       * components are available
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none.
       *
       * @return 0 on success and an error code otherwise
       */
      Result run();
   
      /**
       * @brief Stops the Pairing Controller instance
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none
       *
       * @return 0 on success and an error code otherwise
       */
      Result stop();
   
      /**
       * @brief Cleans up the resources used by the Pairing Controller instance
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none
       *
       * @return 0 on success and an error code otherwise
       */
      Result done();
   
      bool isComponentSm() {return true;};

      /**
       * Returns the current state the state machine is in (for debugging of shutdown problems)
       *
       * @param[in,out] stateName buffer for storing the current state name
       * @param[in] size size of the stateName buffer
       *
       * @return pointer to stateName
       */
      char *getSmStateName(OUT tGeneralString stateName, IN size_t size);

      /**
       * @brief Returns statistic data as a string for the over all object statistics
       *
       * @param[out] stat string with statistics information to be printed
       *
       * @return 0 on success and an error code otherwise
       */
   //   int Statistics(OUT Statistics stat);

      Result checkPairingIsSuccessful(IN const BdAddressChrArr bdAddress, IN const PairingResult pairingResult);

      Result checkAndStartTimer();

      Result checkAllInfosAvailable(OUT bool& allInfosAvailable, IN DeviceId deviceId);

      Result stopTimerAndSendPairingResult();

      Result connectDeviceAsCPW();

      Result connectDeviceAsClassicBT();

      Result handleTimeout();

      Result updateInfo(IN DeviceId deviceId, IN DeviceInfoUpdateType deviceInfoUpdateType);

      Result handlePairingFinished();

      void checkPairingResult(OUT PairingResult& pairingResult);

      void handleSmTimerCallback(const char* message);

   private:
   
      /**
       * @brief Initializes members
       *
       * @return 0: success
       */
      Result initMembers();

      /**
       * @brief Stores the available pairing information of pairing requests in corresponding members
       *
       * The values of @a bdAddress, @a pairingRequestOrigin, @a pairingType and @a sspPinCode are
       * stored in the corresponding members and the PairingStatus property is updated accordingly
       * (in addition: status = PAIRING_ONGOING).
       *
       * @param[in] bdAddress Bluetooth device address of remote BT device
       * @param[in] pairingRequestOrigin Pairing type e.g. BM_PAIRING_REQUEST_ORIGIN_LOCAL (see PairingRequestOrigin)
       * @param[in] pairingType Pairing type e.g. BM_PAIRING_TYPE_LEGACY (see PairingType)
       * @param[in] sspPinCode Simple Secure Pairing PIN code
       *
       * @return 0: success
       */
      Result savePairingInfo(IN const BdAddressChrArr bdAddress, IN const DeviceNameChrArr deviceName, IN const PairingRequestOrigin pairingRequestOrigin, IN const PairingType pairingType, IN const SspPinCodeChrArr sspPinCode);

      unsigned char _attemptNumber;
      PairingResult _pairingResult;        /**< @todo */

      bool _sppCapabilitiesReceived;

      Timer      _waitfor_DID_SPP_InfoTimer;
      timer_t    _waitfor_DID_SPP_InfoTimerId;

      std::string _expiredTimerSmEvent;
   };
}

#endif //_PAIRING_CONTROLLER_H_

