#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BmCoreMainController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#endif
#endif

#include "BmCoreMainController.h"
#include "BmAllTypesInternal.h"
#include "BmCoreIfMessage.h"
#include "BmCoreIfMessages.h"
#include "BmCoreIfMessagesCreator.h"
#include "IBtStackIfConnectionRequest.h"
#include "BmUtils.h"
#include "LocalSpm.h"
#include "Utils.h"
#include "BmVarTrace.h"
#include <algorithm> // required for using find
#include "BmGlobalLock.h"
#include "Dispatcher.h"
#include "BmMeetsDeviceHandle.h"
#include "BmMeetsBdAddress.h"

namespace bmcore
{
   BmCoreMainController::BmCoreMainController(IN const ComponentId componentId) :
         ILocalSpm(componentId),
         _bmCoreIfMessageQueue("BmCoreIfMessageQueue"),
         _threadId(-1),
         _terminateThread(false),
         _expectedComponentState(COMPONENT_STATE_CREATE),
         _btStackIfConnectionRequestIfWrapper(),
         _btStackIfConnectionCallbackIf(*this),
         _btStackIfWblRequestIfWrapper(),
         _btStackIfWblCallbackIf(*this),
         _bmConfigDataPropHdl(*this),
         _btSystemStatePropHdl(*this),
         _btStatusSwitchPropHdl(*this),
         _multiHFPSupportSwitchPropHdl(*this),
         _localFriendlyNamePropHdl(*this),
         _localInfoPropHdl(*this),
         _localPairableModeSwitchPropHdl(*this),
         _localConnectableModeSwitchPropHdl(*this),
         _autoConnectionTypePropHdl(*this),
         _autoConnectionStatusPropHdl(*this),
         _discoveryStatusPropHdl(*this),
         _discoveredDeviceListPropHdl(*this),
         _pairingPinPropHdl(*this),
         _pairingStatusPropHdl(*this),
         _pairedDeviceListPropHdl(*this),
         _deviceConnectionStatusListPropHdl(*this),
         _linkQualityPropHdl(*this),
         _blockStatusPropHdl(*this),
         _btLimitationModePropHdl(*this),
         _testModePropHdl(*this),
         _userDecisionRequiredPropHdl(*this),
         _btProfileUsageInfoPropHdl(*this),
         _resetToDefaultStatusPropHdl(*this),
         _testModeLinkQualityPropHdl(*this),
         _bmServiceAvailabilityPropHdl(*this),
         _remoteConnectableController(*this),
         _serviceSearchController(*this),
         _remoteSupportedProtocols(),
         _remoteSupportedSppUuids(),
         _remoteDeviceIdentification(),
         _remoteDevicesSPPVersion(),
         _connectedDevicesWaitingForLinkQuality(),
         _intermediateLinkQuality(),
         _wblHealthinessIndicator(100u),
         _numberPendingUpdateLinkQualityRequests(0u),
         _currentFunctionalityRestrictionInfo(0u),
         _currentFunctionalityRestrictionInfoLock(),
         _restrictionBitStringMap(),
         _resetToFactorySettingsRequested(false),
         _delayBtOnRequestTimerId(0),
         _delayBtOnRequestTimer(),
         _lockDelayBtOnRequestTimer(),
         _waitingForEvoStackRestart(false)
   {
      ETG_TRACE_USR1(("BmCoreMainController: being created (0x%p)", (void*) this));

      _restrictionBitStringMap.insert(std::pair<RestrictionBit,std::string>(BM_RESTRICTION_BIT_LOCAL_SPM_STATE_OFF, "LSPM_OFF"));
      _restrictionBitStringMap.insert(std::pair<RestrictionBit,std::string>(BM_RESTRICTION_BIT_LOCAL_SPM_STATE_UNDERVOLTAGE, "LSPM_UV"));
      _restrictionBitStringMap.insert(std::pair<RestrictionBit,std::string>(BM_RESTRICTION_BIT_BT_SYSTEM_STATE_BLOCKED, "BT_SYS_BLOCKED"));
      _restrictionBitStringMap.insert(std::pair<RestrictionBit,std::string>(BM_RESTRICTION_BIT_BT_SYSTEM_STATE_OFF, "BT_SYS_OFF"));
      _restrictionBitStringMap.insert(std::pair<RestrictionBit,std::string>(BM_RESTRICTION_BIT_BT_STATUS_OFF, "BT_OFF"));
      _restrictionBitStringMap.insert(std::pair<RestrictionBit,std::string>(BM_RESTRICTION_BIT_OVERALL_DEVICE_BLOCK_STATE_BLOCKED, "ODBS_BLOCKED"));
      _restrictionBitStringMap.insert(std::pair<RestrictionBit,std::string>(BM_RESTRICTION_BIT_RESET_TO_DEFAULT_IN_PROGRESS, "RTD_ONGOING"));

      if (0 != _currentFunctionalityRestrictionInfoLock.lock())
      {
         ETG_TRACE_FATAL(("BmCoreMainController: acquiring lock _currentFunctionalityRestrictionInfoLock failed"));
      }

      _currentFunctionalityRestrictionInfo.set();

      _currentFunctionalityRestrictionInfoLock.unlock();
   }

   BmCoreMainController::~BmCoreMainController()
   {
      ETG_TRACE_USR1(("~BmCoreMainController: being destroyed (0x%p)", (void*) this));

      _pendingIssuesToBeResolvedList.clear();
   }

   void BmCoreMainController::notifyComponentState(IN const ComponentId componentId, IN const ComponentState componentState)
   {
      ETG_TRACE_USR1(("notifyComponentState: componentId = %d, componentState = %d, expected component state = %d",
            ETG_CENUM(ComponentId, componentId), ETG_CENUM(ComponentState, componentState),
            ETG_CENUM(ComponentState, _expectedComponentState)));

      // sub-components of this main controller (which are not managed by LocalSpm) have to call this method once they
      // finished switching to requested component state
      // up to now there are not yet any sub-components

/*
      if ((_expectedComponentState == <_subComponent_1.getComponentState())
         && (_expectedComponentState == <_subComponent_2.getComponentState()))
      {
*/
         ETG_TRACE_USR1(("notifyComponentState: all sub-components are in state %d",
               ETG_CENUM(ComponentState, _expectedComponentState)));

         if (COMPONENT_STATE_STOP == _expectedComponentState)
         {
            stopDone(0);
         }
         else if (COMPONENT_STATE_DONE == _expectedComponentState)
         {
            doneDone(0);
         }
/*
      }
      else
      {
         ETG_TRACE_USR1(("notifyComponentState: not yet all sub-components are in state %d",
               ETG_CENUM(ComponentState, _expectedComponentState)));
      }
*/
   }

   Result BmCoreMainController::initBtStackIfConnection(IN const BTSLocalStackConfiguration& configuration,
         IN const BTSBDAddressList& pairedDevices)
   {
      _btStackIfConnectionRequestIfWrapper.registerCallback(&_btStackIfConnectionCallbackIf);

      BTSErrorCode btsErrorCode = _btStackIfConnectionRequestIfWrapper.init(configuration, pairedDevices);

      if (BTS_OK != btsErrorCode)
      {
         ETG_TRACE_ERRMEM(("initBtStackIfConnection: initializing BtStackIf Connection failed (btsErrorCode = %d)",
               ETG_CENUM(BTSErrorCode, btsErrorCode)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmCoreMainController::initBtStackIfWbl(IN const BTSLocalWblConfiguration& configuration)
   {
      _btStackIfWblRequestIfWrapper.registerCallback(&_btStackIfWblCallbackIf);

      BTSErrorCode btsErrorCode = _btStackIfWblRequestIfWrapper.init(configuration);

      if (BTS_OK != btsErrorCode)
      {
         ETG_TRACE_ERRMEM(("initBtStackIfWbl: initializing BtStackIf WBL failed (btsErrorCode = %d)",
               ETG_CENUM(BTSErrorCode, btsErrorCode)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmCoreMainController::initBmConfigData(void)
   {
      ETG_TRACE_USR1(("initBmConfigData: initializing BmConfigData"));

      BmConfigData bmConfigData;
      bmConfigData._maxNumPairedDevices = LocalSpm::getDataProvider().getBmCoreConfiguration()._maxNumPairedDevices;

      _bmConfigDataPropHdl.set(bmConfigData);

      return CC_ERR_INT_NO_ERROR;
   }

   BmResult BmCoreMainController::getBmConfigData(OUT BmConfigData& bmConfigData, IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getBmConfigData: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getBmConfigData: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _bmConfigDataPropHdl.get(bmConfigData);

      VARTRACE(bmConfigData);

      return BM_RESULT_OK;
   }

   BmResult BmCoreMainController::getBmServiceAvailablity(OUT BmServiceAvailability& bmServiceAvailability, IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getBmServiceAvailablity: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getBmServiceAvailablity: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _bmServiceAvailabilityPropHdl.get(bmServiceAvailability);

      VARTRACE(bmServiceAvailability);

      return BM_RESULT_OK;
   }

   Result BmCoreMainController::initBtSystemState(void)
   {
      ETG_TRACE_USR1(("initBtSystemState: initializing BtSystemState"));

      BtSystemState btSystemState(BM_BT_SYSTEM_STATE_BLOCK);
      _btSystemStatePropHdl.set(btSystemState);

      BmCoreIfMessage_BtSystemState internalMsg(BM_BT_SYSTEM_STATE_BLOCK, 0u, BM_CORE_IF_MSG_ORIGIN_INTERNAL);

      this->handleBmCoreIfMessage_BtSystemStateChange(&internalMsg);

      return CC_ERR_INT_NO_ERROR;
   }

   BmResult BmCoreMainController::setBtSystemState(IN const BtSystemState& btSystemState)
   {
      ETG_TRACE_USR1(("setBtSystemState: btSystemState._bluetoothSystemState = %d",
            ETG_CENUM(BluetoothSystemState, btSystemState._bluetoothSystemState)));

      if (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01))
      {
         ETG_TRACE_ERR(("setBtSystemState: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      if (false == isValidBluetoothSystemState(btSystemState._bluetoothSystemState))
      {
         ETG_TRACE_ERR(("setBtSystemState: invalid parameter value for btSystemState._bluetoothSystemState"));
         return BM_RESULT_ERR_INVALID_PARAMETER;
      }

      _btSystemStatePropHdl.set(btSystemState);

      return BM_RESULT_OK;
   }

   BmResult BmCoreMainController::getBtSystemState(OUT BtSystemState& btSystemState, IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getBtSystemState: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getBtSystemState: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _btSystemStatePropHdl.get(btSystemState);

      VARTRACE(btSystemState);

      return BM_RESULT_OK;
   }

   Result BmCoreMainController::initLocalFriendlyName(void)
   {
      ETG_TRACE_USR1(("initLocalFriendlyName: initializing LocalFriendlyName"));

      Result result(CC_ERR_INT_NO_ERROR);
      LocalFriendlyName localFriendlyName;

      result = LocalSpm::getDbManager().getLocalBdName(localFriendlyName._localFriendlyName);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR1(("initLocalFriendlyName: BD name stored in DB is \"%50s\"", localFriendlyName._localFriendlyName.c_str()));

         if (false == isValidLocalBdName(localFriendlyName._localFriendlyName))
         {
            localFriendlyName._localFriendlyName = LocalSpm::getDataProvider().getBmCoreConfiguration()._btLocalFriendlyName;

            ETG_TRACE_USR1(("initLocalFriendlyName: BD name stored in DB is not valid (probably not yet set, using the configured default value \"%50s\")",
                  localFriendlyName._localFriendlyName.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("initLocalFriendlyName: getting local BD name from DB failed (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      _localFriendlyNamePropHdl.set(localFriendlyName);

      return result;
   }

   Result BmCoreMainController::initLocalInfo(void)
   {
      ETG_TRACE_USR1(("initLocalInfo: initializing LocalInfo"));

      Result result(CC_ERR_INT_NO_ERROR);
      Result tmpResult(CC_ERR_INT_NO_ERROR);

      LocalInfo localInfo;
      _localInfoPropHdl.get(localInfo);

      tmpResult = LocalSpm::getDbManager().getLocalBdAddress(localInfo._bdAddress);

      if (CC_ERR_INT_NO_ERROR == tmpResult)
      {
         ETG_TRACE_USR1(("initLocalInfo: BD address stored in DB is \"%50s\"", localInfo._bdAddress.c_str()));

         if (false == isValidBdAddress(localInfo._bdAddress))
         {
            ETG_TRACE_USR1(("initLocalInfo: BD address stored in DB is not valid (probably not yet set)"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("initLocalInfo: getting local BD address from DB failed (error = %d)", ETG_CENUM(CcErrorInternal, tmpResult)));

         result = tmpResult;
      }

      tmpResult = LocalSpm::getDbManager().getLocalBdName(localInfo._bdName);

      if (CC_ERR_INT_NO_ERROR == tmpResult)
      {
         ETG_TRACE_USR1(("initLocalInfo: BD name stored in DB is \"%50s\"", localInfo._bdName.c_str()));

         if (false == isValidLocalBdName(localInfo._bdName))
         {
            localInfo._bdName = LocalSpm::getDataProvider().getBmCoreConfiguration()._btLocalFriendlyName;

            ETG_TRACE_USR1(("initLocalInfo: BD name stored in DB is not valid (probably not yet set, using the configured default value \"%50s\")",
                  localInfo._bdName.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("initLocalInfo: getting local BD name from DB failed (error = %d)", ETG_CENUM(CcErrorInternal, tmpResult)));

         if (CC_ERR_INT_NO_ERROR == result)
         {
            result = tmpResult;
         }
      }

      tmpResult = LocalSpm::getDataProvider().getProtocolInfos(localInfo._protocolInfos, false);

      if (CC_ERR_INT_NO_ERROR != tmpResult)
      {
         ETG_TRACE_ERR(("initLocalInfo: getting local protocol infos from DataProvider failed (error = %d)",
               ETG_CENUM(CcErrorInternal, tmpResult)));

         if (CC_ERR_INT_NO_ERROR == result)
         {
            result = tmpResult;
         }
      }

      tmpResult = LocalSpm::getDataProvider().getVehicleIdentification(localInfo._vehicleIdentification);
      if (CC_ERR_INT_NO_ERROR != tmpResult)
      {
         ETG_TRACE_ERR(("initLocalInfo: getting Vehicle Identification Info from DataProvider failed (error = %d)",
               ETG_CENUM(CcErrorInternal, tmpResult)));
         if (CC_ERR_INT_NO_ERROR == result)
         {
            result = tmpResult;
         }
      }

      // update "property" LocalInfo
      _localInfoPropHdl.set(localInfo);

      return result;
   }

   Result BmCoreMainController::initBlockStatus(void)
   {
      ETG_TRACE_USR1(("initBlockStatus: initializing BlockStatus"));

      BlockStatus blockStatus;

      blockStatus._overallDeviceBlockStatus._blockState = BM_OVERALL_BLOCK_STATE_UNBLOCKED;
      blockStatus._overallDeviceBlockStatus._bdAddress = "";

      Result result = LocalSpm::getDbManager().getDeviceBlockStatusList(blockStatus._deviceBlockStatusList);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("initBlockStatus: could not get device block status list from DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
         return result;
      }

      _blockStatusPropHdl.set(blockStatus);

      return result;
   }

   Result BmCoreMainController::initMultiHFPSupport(void)
   {
      ETG_TRACE_USR1(("initMultiHFPSupport: initializing initMultiHFPSupportStatus"));

      Result result(CC_ERR_INT_NO_ERROR);

      TargetSwitchState multiHFPSupportTargetState;

      result = LocalSpm::getDbManager().getMultiHFPSupportStatus(multiHFPSupportTargetState);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR1(("initMultiHFPSupport: MultiHFPSupport status stored in DB is %d", ETG_CENUM(TargetSwitchState, multiHFPSupportTargetState)));

         if (false == isValidTargetSwitchState(multiHFPSupportTargetState))
         {
            multiHFPSupportTargetState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultmultiHFPSupport;

            ETG_TRACE_ERR(("initMultiHFPSupport:MultiHFPSupport status stored in DB is not valid (using configured default value = %d)", ETG_CENUM(TargetSwitchState, multiHFPSupportTargetState)));
         }
      }
      else
      {
         multiHFPSupportTargetState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultmultiHFPSupport;
         ETG_TRACE_ERR(("initMultiHFPSupport: getting multiHFPSupport status from DB failed (error = %d), using configured default value = %d", result, ETG_CENUM(TargetSwitchState, multiHFPSupportTargetState)));
      }

      _multiHFPSupportSwitchPropHdl.setTargetSwitchState(multiHFPSupportTargetState, BM_CORE_IF_MSG_ORIGIN_INTERNAL);

      return result;
   }

   BmResult BmCoreMainController::switchBtStatusInt(IN const TargetSwitchState& targetSwitchState,
         IN const bool suppressAutoConnectionAfterSwitchedOn, IN const bool updateLastMode)
   {
      ETG_TRACE_USR1(("switchBtStatusInt: targetSwitchState = %d, suppressAutoConnectionAfterSwitchedOn = %10s, updateLastMode = %10s",
            ETG_CENUM(TargetSwitchState, targetSwitchState),
            suppressAutoConnectionAfterSwitchedOn ? "true" : "false",
            updateLastMode ? "true" : "false"));

      BmResult bmResult(BM_RESULT_OK);
      SwitchStatus btStatus(SWITCH_STATE_SWITCHED_OFF, SWITCHED_OFF_REASON_CLIENT);
      bmResult = getBtStatus(btStatus);

      if (BM_RESULT_OK == bmResult)
      {
         if ((SWITCH_STATE_SWITCHED_ON == btStatus._switchState) && (suppressAutoConnectionAfterSwitchedOn == true))
         {
            ETG_TRACE_USR1(("switchBtStatusInt: suppressAutoConnectionAfterSwitchedOn make it false because BT status is already ON"));

            BmCoreIfMessage_SwitchBtStatusRequest internalMsg(targetSwitchState, false, false,
                  updateLastMode, 0u, BM_CORE_IF_MSG_ORIGIN_INTERNAL);

            return this->handleBmCoreIfMessage_SwitchBtStatusRequest(&internalMsg);
         }
         else
         {
            BmCoreIfMessage_SwitchBtStatusRequest internalMsg(targetSwitchState, suppressAutoConnectionAfterSwitchedOn,
                  false, updateLastMode, 0u, BM_CORE_IF_MSG_ORIGIN_INTERNAL);

            return this->handleBmCoreIfMessage_SwitchBtStatusRequest(&internalMsg);
         }
      }
      else
      {
         return bmResult;
      }
   }

   BmResult BmCoreMainController::setBtLocalFriendlyNameInt(IN const LocalFriendlyNameType& bdName)
   {
      ETG_TRACE_USR1(("setBtLocalFriendlyNameInt: bdName = \"%50s\"", bdName.c_str()));

      BmCoreIfMessage_SetBtLocalFriendlyNameRequest internalMsg(bdName, 0u, BM_CORE_IF_MSG_ORIGIN_INTERNAL);

      return this->handleBmCoreIfMessage_SetBtLocalFriendlyNameRequest(&internalMsg);
   }

   BmResult BmCoreMainController::setBtLocalFriendlyNameIntForced(IN const LocalFriendlyNameType& bdName)
   {
      ETG_TRACE_USR1(("setBtLocalFriendlyNameIntForced: bdName = \"%50s\"", bdName.c_str()));

      _localFriendlyNamePropHdl.set(bdName, true, true);

      return BM_RESULT_OK;
   }

   BmResult BmCoreMainController::switchLocalPairableModeInt(IN const TargetSwitchState& targetSwitchState,
         IN const BdAddress& bdAddress)
   {
      ETG_TRACE_USR1(("switchLocalPairableModeInt: targetSwitchState = %d, bdAddress = \"%50s\"",
            ETG_CENUM(TargetSwitchState, targetSwitchState), bdAddress.c_str()));

      BmCoreIfMessage_SwitchLocalPairableModeRequest internalMsg(targetSwitchState, bdAddress, 0u,
            BM_CORE_IF_MSG_ORIGIN_INTERNAL);

      return this->handleBmCoreIfMessage_SwitchLocalPairableRequest(&internalMsg);
   }

   Result BmCoreMainController::initLocalPairableMode(void)
   {
      ETG_TRACE_USR1(("initLocalPairableMode: initializing LocalPairableMode"));

      Result result(CC_ERR_INT_NO_ERROR);

      BmResult bmResult = this->switchLocalPairableModeInt(LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode, "");

      if (BM_RESULT_OK != bmResult)
      {
         result = CC_ERR_INT_GENERAL_ERROR;
      }

      return result;
   }

   BmResult BmCoreMainController::switchLocalConnectableModeInt(IN const TargetSwitchState& targetSwitchState,
         IN const BdAddress& bdAddress)
   {
      ETG_TRACE_USR1(("switchLocalConnectableModeInt: targetSwitchState = %d, bdAddress = \"%50s\"",
            ETG_CENUM(TargetSwitchState, targetSwitchState), bdAddress.c_str()));

      BmCoreIfMessage_SwitchLocalConnectableModeRequest internalMsg(targetSwitchState, bdAddress, 0u,
            BM_CORE_IF_MSG_ORIGIN_INTERNAL);

      return this->handleBmCoreIfMessage_SwitchLocalConnectableModeRequest(&internalMsg);
   }

   Result BmCoreMainController::initLocalConnectableMode(void)
   {
      ETG_TRACE_USR1(("initLocalConnectableMode: initializing LocalConnectableMode"));

      Result result(CC_ERR_INT_NO_ERROR);

      BmResult bmResult = this->switchLocalConnectableModeInt(LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode, "");

      if (BM_RESULT_OK != bmResult)
      {
         result = CC_ERR_INT_GENERAL_ERROR;
      }

      return result;
   }

   BmResult BmCoreMainController::deleteDeviceInt(IN const DeviceId deviceId, IN const DeviceHandleType deviceHandleType)
   {
      ETG_TRACE_USR1(("deleteDeviceInt: deviceId = %d, deviceHandleType = %d", deviceId,
            ETG_CENUM(DeviceHandleType, deviceHandleType)));

      BmResult bmResult(BM_RESULT_OK);
      DeviceId deviceHandle(deviceId);

      if (BM_DEVICE_HANDLE_TYPE_SINGLE == deviceHandleType)
      {
         // a particular device is requested to get deleted
         // get the device handle for given device ID from DB
         Result result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("deleteDeviceInt: could not get device handle for given device ID = %d from DB (error = %d)",
                  deviceId, ETG_CENUM(CcErrorInternal, result)));

            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }

      if (BM_RESULT_OK == bmResult)
      {
         BmCoreIfMessage_DeleteDeviceRequest internalMsg(deviceHandle, deviceHandleType, 0u,
               BM_CORE_IF_MSG_ORIGIN_INTERNAL);
         bmResult = this->handleBmCoreIfMessage_DeleteDeviceRequest(&internalMsg);
      }

      return bmResult;
   }

   BmResult BmCoreMainController::connectDeviceInt(IN const DeviceId deviceId, IN const bool delayRequestProcessing, IN const PageTimeout pageTimeout)
   {
      ETG_TRACE_USR1(("connectDeviceInt: deviceId = %d, delayRequestProcessing = %10s, PageTimeout = %u"  , deviceId,
            delayRequestProcessing ? "true" : "false", pageTimeout));

      BmResult bmResult(BM_RESULT_OK);
      DeviceId deviceHandle(deviceId);

      Result result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("connectDeviceInt: could not get device handle for given device ID = %d from DB (error = %d)",
               deviceId, ETG_CENUM(CcErrorInternal, result)));

         bmResult = BM_RESULT_ERR_GENERAL;
      }

      if (BM_RESULT_OK == bmResult)
      {
         BmCoreIfMessage_ConnectDeviceRequest internalMsg(deviceHandle, delayRequestProcessing, pageTimeout, 0u,
               BM_CORE_IF_MSG_ORIGIN_INTERNAL);
         bmResult = this->handleBmCoreIfMessage_ConnectDeviceRequest(&internalMsg);
      }

      return bmResult;
   }

   BmResult BmCoreMainController::disconnectDeviceInt(IN const DeviceId deviceId,
         IN const DeviceHandleType deviceHandleType, IN const DisconnectedReason disconnectedReason,
         IN const bool deleteDevice)
   {
      ETG_TRACE_USR1(("disconnectDeviceInt: deviceId = %d, deviceHandleType = %d, disconnectedReason = %d, deleteDevice = %10s",
            deviceId, ETG_CENUM(DeviceHandleType, deviceHandleType), ETG_CENUM(DisconnectedReason, disconnectedReason),
            deleteDevice ? "true" : "false"));

      BmResult bmResult(BM_RESULT_OK);
      DeviceId deviceHandle(deviceId);

      Result result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("disconnectDeviceInt: could not get device handle for given device ID = %d from DB (error = %d)",
               deviceId, ETG_CENUM(CcErrorInternal, result)));

         bmResult = BM_RESULT_ERR_GENERAL;
      }

      if (BM_RESULT_OK == bmResult)
      {
         BmCoreIfMessage_DisconnectDeviceRequest internalMsg(deviceHandle, deviceHandleType, disconnectedReason,
               deleteDevice, 0u, BM_CORE_IF_MSG_ORIGIN_INTERNAL);
         bmResult = this->handleBmCoreIfMessage_DisconnectDeviceRequest(&internalMsg);
      }

      return bmResult;
   }

   BmResult BmCoreMainController::connectProfilesInt(IN const DeviceId deviceId, IN const ProtocolList& protocolList, IN const PageTimeout pageTimeout)
   {
      ETG_TRACE_USR1(("connectProfilesInt: deviceId = %d, pageTimeout = %d", deviceId, pageTimeout));

      VARTRACE(protocolList);

      BmResult bmResult(BM_RESULT_OK);
      DeviceId deviceHandle(deviceId);

      Result result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("connectProfilesInt: could not get device handle for given device ID = %d from DB (error = %d)",
               deviceId, ETG_CENUM(CcErrorInternal, result)));

         bmResult = BM_RESULT_ERR_GENERAL;
      }

      if (BM_RESULT_OK == bmResult)
      {
         BmCoreIfMessage_ConnectProfilesRequest internalMsg(deviceHandle, protocolList, pageTimeout, 0u,
               BM_CORE_IF_MSG_ORIGIN_INTERNAL);
         bmResult = this->handleBmCoreIfMessage_ConnectProfilesRequest(&internalMsg);
      }

      return bmResult;
   }

   BmResult BmCoreMainController::disconnectProfilesInt(IN const DeviceId deviceId, IN const ProtocolList& protocolList)
   {
      ETG_TRACE_USR1(("disconnectProfilesInt: deviceId = %d", deviceId));

      VARTRACE(protocolList);

      BmResult bmResult(BM_RESULT_OK);
      DeviceId deviceHandle(deviceId);

      Result result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("disconnectProfilesInt: could not get device handle for given device ID = %d from DB (error = %d)",
               deviceId, ETG_CENUM(CcErrorInternal, result)));

         bmResult = BM_RESULT_ERR_GENERAL;
      }

      if (BM_RESULT_OK == bmResult)
      {
         BmCoreIfMessage_DisconnectProfilesRequest internalMsg(deviceHandle, protocolList, 0u,
               BM_CORE_IF_MSG_ORIGIN_INTERNAL);
         bmResult = this->handleBmCoreIfMessage_DisconnectProfilesRequest(&internalMsg);
      }

      return bmResult;
   }

   BmResult BmCoreMainController::blockProfilesInt(IN const DeviceId deviceId, IN const ProtocolList& protocolList, IN const bool ignoreStopAutoconnection)
   {
      ETG_TRACE_USR1(("blockProfilesInt: deviceId = %d, ignoreStopAutoconectionStatus = %10s", deviceId, ignoreStopAutoconnection ? "true" : "false"));

      VARTRACE(protocolList);

      BmResult bmResult(BM_RESULT_OK);
      DeviceId deviceHandle(deviceId);

      Result result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("blockProfilesInt: could not get device handle for given device ID = %d from DB (error = %d)",
               deviceId, ETG_CENUM(CcErrorInternal, result)));

         bmResult = BM_RESULT_ERR_GENERAL;
      }

      if (BM_RESULT_OK == bmResult)
      {
         BmCoreIfMessage_BlockProfilesRequest internalMsg(deviceHandle, protocolList, ignoreStopAutoconnection, 0u,
               BM_CORE_IF_MSG_ORIGIN_INTERNAL);
         bmResult = this->handleBmCoreIfMessage_BlockProfilesRequest(&internalMsg);
      }

      return bmResult;
   }

   BmResult BmCoreMainController::unblockProfilesInt(IN const DeviceId deviceId, IN const ProtocolList& protocolList)
   {
      ETG_TRACE_USR1(("unblockProfilesInt: deviceId = %d", deviceId));

      VARTRACE(protocolList);

      BmResult bmResult(BM_RESULT_OK);
      DeviceId deviceHandle(deviceId);

      Result result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("unblockProfilesInt: could not get device handle for given device ID = %d from DB (error = %d)",
               deviceId, ETG_CENUM(CcErrorInternal, result)));

         bmResult = BM_RESULT_ERR_GENERAL;
      }

      if (BM_RESULT_OK == bmResult)
      {
         BmCoreIfMessage_UnblockProfilesRequest internalMsg(deviceHandle, protocolList, 0u,
               BM_CORE_IF_MSG_ORIGIN_INTERNAL);
         bmResult = this->handleBmCoreIfMessage_UnblockProfilesRequest(&internalMsg);
      }

      return bmResult;
   }

   BmResult BmCoreMainController::switchMultiHFPSupportStatusInt(IN const TargetSwitchState& targetSwitchState)
   {
      ETG_TRACE_USR1(("switchMultiHFPSupportStatusInt: targetSwitchState = %d",
            ETG_CENUM(TargetSwitchState, targetSwitchState)));

      BmCoreIfMessage_SwitchMultiHFPSupportRequest internalMsg(targetSwitchState, 0u,
            BM_CORE_IF_MSG_ORIGIN_INTERNAL);

      return this->handleBmCoreIfMessage_SwitchMultiHFPSupportRequest(&internalMsg);
   }

   BmResult BmCoreMainController::startDeviceServiceSearch(IN const DeviceId deviceId)
   {
      ETG_TRACE_USR1(("startDeviceServiceSearch: deviceId = %d", deviceId));

      BmResult bmResult(BM_RESULT_OK);
      BdAddress bdAddress("");

      Result result = LocalSpm::getDbManager().getBdAddress(bdAddress, deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         // handle the request
         this->getBtStackIfConnectionRequestIfWrapper().startRemoteServiceSearch(bdAddress, BTS_SEARCH_SPP);
      }
      else
      {
         ETG_TRACE_ERR(("startDeviceServiceSearch: could not get device Address for given device Id = %d from DB (error = %d)",
               deviceId, ETG_CENUM(CcErrorInternal, result)));

         bmResult = BM_RESULT_ERR_GENERAL;
      }

      return bmResult;
   }

   BmResult BmCoreMainController::getDiscoveryStatus(OUT SwitchStatus& discoveryStatus,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getDiscoveryStatus: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getDiscoveryStatus: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _discoveryStatusPropHdl.get(discoveryStatus);

      VARTRACE(discoveryStatus);

      return BM_RESULT_OK;
   }

   Result BmCoreMainController::initDiscoveryStatus(void)
   {
      ETG_TRACE_USR1(("initDiscoveryStatus: initializing DiscoveryStatus"));

      BmResult bmResult = this->switchDiscoveryStatusInt(TARGET_SWITCH_STATE_SWITCHED_OFF);

      if (BM_RESULT_OK != bmResult)
      {
         return CC_ERR_INT_GENERAL_ERROR;
      }

      return CC_ERR_INT_NO_ERROR;
   }

   BmResult BmCoreMainController::switchDiscoveryStatusInt(IN const TargetSwitchState& targetSwitchState)
   {
      ETG_TRACE_USR1(("switchDiscoveryStatusInt: targetSwitchState = %d",
            ETG_CENUM(TargetSwitchState, targetSwitchState)));

      BmCoreIfMessage_SwitchDiscoveryStatusRequest internalMsg(targetSwitchState, 0u,
            BM_CORE_IF_MSG_ORIGIN_INTERNAL);

      return this->handleBmCoreIfMessage_SwitchDiscoveryStatusRequest(&internalMsg);
   }

   BmResult BmCoreMainController::getDiscoveredDeviceList(OUT DiscoveredDeviceList& discoveredDeviceList,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getDiscoveredDeviceList: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getDiscoveredDeviceList: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _discoveredDeviceListPropHdl.get(discoveredDeviceList);

      VARTRACE(discoveredDeviceList);

      return BM_RESULT_OK;
   }

   void BmCoreMainController::initDiscoveredDeviceList(void)
   {
      ETG_TRACE_USR1(("initDiscoveredDeviceList: initializing DiscoveredDeviceList"));

      DiscoveredDeviceList discoveredDeviceList;
      _discoveredDeviceListPropHdl.set(discoveredDeviceList);
   }

   Result BmCoreMainController::initAutoConnectionType(void)
   {
      ETG_TRACE_USR1(("initAutoConnectionType: initializing AutoConnectionType"));

      Result result(CC_ERR_INT_NO_ERROR);

      AutoConnectionType autoConnectionType;

      result = LocalSpm::getDbManager().getStandardAutoConnectionType(autoConnectionType._type);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR1(("initAutoConnectionType: standard auto connection type stored in DB is %d",
               ETG_CENUM(StandardAutoConnectionType, autoConnectionType._type)));

         if (false == isValidStandardAutoConnectionType(autoConnectionType._type))
         {
            autoConnectionType._type = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultStandardAutoConnectionType;

            ETG_TRACE_ERR(("initAutoConnectionType: standard auto connection type stored in DB is not valid (using configured default value = %d)",
                  ETG_CENUM(StandardAutoConnectionType, autoConnectionType._type)));

            result = LocalSpm::getDbManager().setStandardAutoConnectionType(autoConnectionType._type);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("initAutoConnectionType: storing standard auto connection type to DB failed (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }
      else
      {
         autoConnectionType._type = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultStandardAutoConnectionType;
         ETG_TRACE_ERR(("initAutoConnectionType: getting standard auto connection type from DB failed (error = %d), using configured default value = %d",
               ETG_CENUM(CcErrorInternal, result), ETG_CENUM(StandardAutoConnectionType, autoConnectionType._type)));
      }

      _autoConnectionTypePropHdl.set(autoConnectionType);

      return result;
   }

   BmResult BmCoreMainController::setAutoConnectionTypeInt(IN const AutoConnectionType& autoConnectionType)
   {
      ETG_TRACE_USR1(("setAutoConnectionTypeInt: autoConnectionType._type = %d",
            ETG_CENUM(StandardAutoConnectionType, autoConnectionType._type)));

      BmCoreIfMessage_SetAutoConnectionTypeRequest internalMsg(autoConnectionType, 0u,
            BM_CORE_IF_MSG_ORIGIN_INTERNAL);

      return this->handleBmCoreIfMessage_SetAutoConnectionTypeRequest(&internalMsg);
   }

   //TODO: following method has to be removed once BM Core IF setAutoConnectionType uses loopback mechanism
   BmResult BmCoreMainController::setAutoConnectionType(IN const AutoConnectionType& autoConnectionType,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("setAutoConnectionType: autoConnectionType._type = %d, act = %u",
            ETG_CENUM(StandardAutoConnectionType, autoConnectionType._type), act));

      BmCoreIfMessage_SetAutoConnectionTypeRequest externalMsg(autoConnectionType, act,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      return this->handleBmCoreIfMessage_SetAutoConnectionTypeRequest(&externalMsg);
   }

   BmResult BmCoreMainController::getAutoConnectionType(OUT AutoConnectionType& autoConnectionType,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getAutoConnectionType: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getAutoConnectionType: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _autoConnectionTypePropHdl.get(autoConnectionType);

      VARTRACE(autoConnectionType);

      return BM_RESULT_OK;
   }

   void BmCoreMainController::setAutoConnectionStatusInt(IN const AutoConnectionState autoConnectionState,
         IN const DeviceId deviceIdInProgress)
   {
      ETG_TRACE_USR1(("setAutoConnectionStatusInt: autoConnectionState = %d, deviceIdInProgress = %d",
            ETG_CENUM(AutoConnectionState, autoConnectionState), deviceIdInProgress));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId devHandle = 0u;

      // coverts the deviceId to deviceHandle for updating the property
      if(0u != deviceIdInProgress)
      {
         result = LocalSpm::getDbManager().getDeviceHandle(devHandle, deviceIdInProgress);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("setAutoConnectionStatusInt: getting device handle from DB is failed (error = %d)", ETG_CENUM(CcErrorInternal, result)));
         }
      }

      _autoConnectionStatusPropHdl.set(AutoConnectionStatus(autoConnectionState, devHandle));
   }

   BmResult BmCoreMainController::setBtLimitationModeInt(IN const BdAddress& bdAddress, IN const BdName& bdName, IN const LimitationMode& limitationMode,
         IN const LimitationAction limitationAction)
   {
      ETG_TRACE_USR1(("setBtLimitationModeInt: bdAddress = %50s, bdName = %50s, limitationMode.feature = %d, limitationMode.commIf = %d, limitationAction = %d",
                  bdAddress.c_str(), bdName.c_str(), ETG_CENUM(LimitationFeature, limitationMode._limitationFeature),
                  ETG_CENUM(LimitationCommunicationIf, limitationMode._limitationCommunicationIf),
                  ETG_CENUM(LimitationAction, limitationAction)));

      BmCoreIfMessage_SetBtLimitationModeRequest internalMsg(bdAddress, bdName, limitationMode, limitationAction , 0u,
            BM_CORE_IF_MSG_ORIGIN_INTERNAL);

      return this->handleBmCoreIfMessage_SetBtLimitationModeRequest(&internalMsg);
   }

   void BmCoreMainController::addOobPairedDeviceInt(IN const BdAddress& bdAddress, IN const BdName& bdName,
         IN const LinkKey& linkKey, IN const OobType oobType, IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("addOobPairedDeviceInt: bdAddress = \"%50s\", bdName = \"%50s\", linkKey = \"%50s\", oobType = %d, origin = %d",
            bdAddress.c_str(), bdName.c_str(), linkKey.c_str(), ETG_CENUM(OobType, oobType), ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      BmCoreIfMessage_AddOobPairedDeviceRequest internalMsg(bdAddress, bdName, linkKey, oobType, 0u,
            origin);

      this->handleBmCoreIfMessage_AddOobPairedDeviceRequest(&internalMsg);
   }

   void BmCoreMainController::pushBmCoreIfMessage(IN BmCoreIfMessage* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("pushBmCoreIfMessage: bmCoreIfMessage = 0x%p", (void*) bmCoreIfMessage));
//      void* msgSendBuffer = _bmCoreIfMessageQueue.CreateMessageBuffer(sizeof(bmCoreIfMessage));

//      memcpy(msgSendBuffer, &bmCoreIfMessage, sizeof(bmCoreIfMessage));

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("pushBmCoreIfMessage: bmCoreIfMessage is 0"));
         return;
      }

      bmCoreIfMessage->traceMessage();

      if (-1 != _bmCoreIfMessageQueue.Push(bmCoreIfMessage, sizeof(bmCoreIfMessage), 1))
      {
          ETG_TRACE_USR4(("pushBmCoreIfMessage: pushed message object = 0x%p to queue", (void*) bmCoreIfMessage));
      }
      else
      {
          ETG_TRACE_FATAL(("pushBmCoreIfMessage: could not push message object (0x%p)", (void*) bmCoreIfMessage));

          delete bmCoreIfMessage;

          FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   BmResult BmCoreMainController::getBtStatus(OUT SwitchStatus& btStatus,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getBtStatus: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getBtStatus: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _btStatusSwitchPropHdl.get(btStatus);

      VARTRACE(btStatus);

      return BM_RESULT_OK;
   }

   BmResult BmCoreMainController::getBtLocalFriendlyName(OUT LocalFriendlyName& localFriendlyName,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getBtLocalFriendlyName: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getBtLocalFriendlyName: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _localFriendlyNamePropHdl.get(localFriendlyName);

      VARTRACE(localFriendlyName);

      return BM_RESULT_OK;
   }

   BmResult BmCoreMainController::getLocalInfo(OUT LocalInfo& localInfo, IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getLocalInfo: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getLocalInfo: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _localInfoPropHdl.get(localInfo);

      VARTRACE(localInfo);

      return BM_RESULT_OK;
   }

   BmResult BmCoreMainController::getLocalPairableMode(OUT SwitchStatus& localPairableMode,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getLocalPairableMode: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getLocalPairableMode: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _localPairableModeSwitchPropHdl.get(localPairableMode);

      VARTRACE(localPairableMode);

      return BM_RESULT_OK;
   }

   BmResult BmCoreMainController::getLocalConnectableMode(OUT SwitchStatus& localConnectableMode,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getLocalConnectableMode: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getLocalConnectableMode: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _localConnectableModeSwitchPropHdl.get(localConnectableMode);

      VARTRACE(localConnectableMode);

      return BM_RESULT_OK;
   }

   BmResult BmCoreMainController::getPairingPin(OUT PairingPin& pairingPin,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getPairingPin: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getPairingPin: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _pairingPinPropHdl.get(pairingPin);

      VARTRACE(pairingPin);

      return BM_RESULT_OK;
   }

   const PairingPinType BmCoreMainController::getPairingPinCode()
   {
      ETG_TRACE_USR1(("getPairingPinCode"));

      PairingPin pairingPin;
      _pairingPinPropHdl.get(pairingPin);

      return pairingPin._pin;
   }

   Result BmCoreMainController::initPairingPin(void)
   {
      ETG_TRACE_USR1(("initPairingPin: initializing PairingPin"));

      Result result(CC_ERR_INT_NO_ERROR);
      PairingPin pairingPin;

      if (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._useFixedPinLegacyPairing)
      {
         result = LocalSpm::getDbManager().getLegacyPairingPin(pairingPin._pin);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR1(("initPairingPin: legacy pairing PIN code stored in DB is \"%50s\"", pairingPin._pin.c_str()));

            if (false == isValidLegacyPairingPin(pairingPin._pin))
            {
               ETG_TRACE_USR1(("initPairingPin: legacy pairing PIN code stored in DB is not valid (probably not yet set)"));

               pairingPin._pin = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultFixedPinLegacyPairing;
               ETG_TRACE_USR1(("initPairingPin: using the configured default value \"%50s\"", pairingPin._pin.c_str()));

               result = LocalSpm::getDbManager().setLegacyPairingPin(pairingPin._pin);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("initPairingPin: storing legacy pairing PIN code to DB failed (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }
         }
         else
         {
            ETG_TRACE_ERR(("initPairingPin: getting legacy pairing PIN code from DB failed (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         pairingPin._pin = this->generateLegacyPairingPin(LocalSpm::getDataProvider().getBmCoreConfiguration()._pinLengthLegacyPairing);
      }

      _pairingPinPropHdl.set(pairingPin);

      return result;
   }

   Result BmCoreMainController::updateLegacyPairingPin(void)
   {
      ETG_TRACE_USR1(("updateLegacyPairingPin"));

      Result result(CC_ERR_INT_NO_ERROR);

      if (false == LocalSpm::getDataProvider().getBmCoreConfiguration()._useFixedPinLegacyPairing)
      {
         ETG_TRACE_USR1(("updateLegacyPairingPin: generating a new legacy pairing PIN code"));

         PairingPin pairingPin;
         pairingPin._pin = this->generateLegacyPairingPin(LocalSpm::getDataProvider().getBmCoreConfiguration()._pinLengthLegacyPairing);

         _pairingPinPropHdl.set(pairingPin);
      }
      else
      {
         ETG_TRACE_USR1(("updateLegacyPairingPin: using the fixed legacy pairing PIN code"));
      }

      return result;
   }

   BmResult BmCoreMainController::setPairingPinInt(IN const PairingPin& pairingPin)
   {
      ETG_TRACE_USR1(("setPairingPinInt"));

      BmCoreIfMessage_SetPairingPinRequest internalMsg(pairingPin, 0u, BM_CORE_IF_MSG_ORIGIN_INTERNAL);

      return this->handleBmCoreIfMessage_SetPairingPinRequest(&internalMsg);;
   }

   BmResult BmCoreMainController::getPairingStatus(OUT PairingStatus& pairingStatus,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getPairingStatus: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getPairingStatus: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _pairingStatusPropHdl.get(pairingStatus);

      VARTRACE(pairingStatus);

      return BM_RESULT_OK;
   }

   void BmCoreMainController::setPairingStatusInt(IN const PairingStatus& pairingStatus)
   {
      ETG_TRACE_USR1(("setPairingStatusInt"));

      //!This logic is required to not allow remote connection requests
      //!for already paired devices when pairing for a new device is ongoing.
      PairingStatus currentPairingStatus;
      _pairingStatusPropHdl.get(currentPairingStatus);

      if(0 != currentPairingStatus._remoteBdAddress.compare(pairingStatus._remoteBdAddress))
      {
         RemoteConnectable remoteConnectable =
                  (false == pairingStatus._remoteBdAddress.empty()) ? (false) : ((TARGET_SWITCH_STATE_SWITCHED_ON == LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode) ? true : false);

         PairedDeviceList pairedDeviceList;
         _pairedDeviceListPropHdl.get(pairedDeviceList);

         if(0 != pairedDeviceList._deviceBaseInfoList.size())
         {
            DeviceBaseInfoList::iterator it;

            for(it = pairedDeviceList._deviceBaseInfoList.begin(); it != pairedDeviceList._deviceBaseInfoList.end(); it++)
            {
               //!Set the remoteconnectable flag as FALSE(remote connection request  will be rejected) for all other devices except the device for which pairing is ongoing
               if((0 != pairingStatus._remoteBdAddress.compare(it->_bdAddress)) &&
                     (0 != currentPairingStatus._remoteBdAddress.compare(it->_bdAddress)))
               {
                  it->_remoteConnectable = remoteConnectable;
               }
            }

            ETG_TRACE_USR4(("setPairingStatusInt: updating PairedDeviceList property"));
            _pairedDeviceListPropHdl.set(pairedDeviceList);
         }
      }

      _pairingStatusPropHdl.set(pairingStatus);
   }

   BmResult BmCoreMainController::getPairedDeviceList(OUT PairedDeviceList& pairedDeviceList,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getPairedDeviceList: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getPairedDeviceList: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _pairedDeviceListPropHdl.get(pairedDeviceList);

      VARTRACE(pairedDeviceList);

      return BM_RESULT_OK;
   }

   Result BmCoreMainController::initPairedDeviceList(void)
   {
      ETG_TRACE_USR1(("initPairedDeviceList: initializing PairedDeviceList"));

      PairedDeviceList pairedDeviceList;

      (void) LocalSpm::getDbManager().getAllDeviceBaseInfo(pairedDeviceList._deviceBaseInfoList);

      DeviceIdList pairedDeviceHandles;

      DeviceBaseInfoList::iterator it;

      for (it = pairedDeviceList._deviceBaseInfoList.begin(); it != pairedDeviceList._deviceBaseInfoList.end(); ++it)
      {
         // set the RemoteConnectable as true (allowing remote connection request) if defaultLocalConnectableMode is SWITCHED_ON
         it->_remoteConnectable = (TARGET_SWITCH_STATE_SWITCHED_ON == LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode) ? true : false;

         pairedDeviceHandles.push_back(it->_deviceHandle);
      }

      _pairedDeviceListPropHdl.set(pairedDeviceList);

      (void) _remoteConnectableController.initializeDeviceRemoteConnectableControlList(pairedDeviceHandles);

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmCoreMainController::resetRemoteConnections(void)
   {
      ETG_TRACE_USR1(("resetRemoteConnections: entered"));

      PairedDeviceList pairedDeviceList;

      (void) LocalSpm::getDbManager().getAllDeviceBaseInfo(pairedDeviceList._deviceBaseInfoList);

      DeviceBaseInfoList::iterator it;
      DeviceId deviceId(0u);
      Result result(CC_ERR_INT_NO_ERROR);

      for (it = pairedDeviceList._deviceBaseInfoList.begin(); it != pairedDeviceList._deviceBaseInfoList.end(); ++it)
      {
         ETG_TRACE_USR4(("resetRemoteConnections(device Handle = %d): connection status = %d and remoteConnectable = %10s",
               it->_deviceHandle, ETG_CENUM(ConnectionStatus, it->_connectionStatus), it->_remoteConnectable ? "true": "false"));

         if((BM_CONNECTION_STATUS_DISCONNECTED == it->_connectionStatus) && (false == it->_remoteConnectable))
         {
            result = LocalSpm::getDbManager().getDeviceId(deviceId, it->_deviceHandle);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("resetRemoteConnections: could not get device Id for given device handle from DB (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
            else
            {
               // reset the RemoteConnectable as true (allowing remote connection request)
               result = this->unblockRemoteConnections(deviceId);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("resetRemoteConnections: could not unblock remote connections for device with ID = %d (error = %d)",
                        deviceId, ETG_CENUM(CcErrorInternal, result)));
               }
            }
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   void BmCoreMainController::onDeviceBaseInfoChanged(void)
   {
      ETG_TRACE_USR1(("onDeviceBaseInfoChanged: updating PairedDeviceList property"));

      PairedDeviceList pairedDeviceList;
      Result result = LocalSpm::getDbManager().getAllDeviceBaseInfo(pairedDeviceList._deviceBaseInfoList);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         _pairedDeviceListPropHdl.set(pairedDeviceList);
      }
      else
      {
         ETG_TRACE_ERR(("onDeviceBaseInfoChanged: could get device base info for created device from DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void BmCoreMainController::onDeviceConnectionStatusChanged(IN const DeviceConnectionInfo& deviceConnectionInfo)
   {
      ETG_TRACE_USR1(("onDeviceConnectionStatusChanged: updating DeviceConnectionStatusList property"));

      DeviceConnectionStatusList deviceConnectionStatusList;
      _deviceConnectionStatusListPropHdl.get(deviceConnectionStatusList);

      DeviceConnectionInfoList::iterator itConn;

      itConn = std::find_if(deviceConnectionStatusList._deviceConnectionInfoList.begin(),
            deviceConnectionStatusList._deviceConnectionInfoList.end(),
            MeetsDeviceHandle<DeviceConnectionInfo>(deviceConnectionInfo._deviceHandle));

      if (itConn != deviceConnectionStatusList._deviceConnectionInfoList.end())
      {
         // found device connection info in device connection status list property for device handle
         // from given device connection info => update device connection info

         itConn->_connectionStatus = deviceConnectionInfo._connectionStatus;
         itConn->_disconnectedReason = deviceConnectionInfo._disconnectedReason;
         itConn->_connectionOrderInfo = deviceConnectionInfo._connectionOrderInfo;

         if (BM_CONNECTION_STATUS_DISCONNECTED == itConn->_connectionStatus)
         {
            _deviceConnectionStatusListPropHdl.set(deviceConnectionStatusList);

            deviceConnectionStatusList._deviceConnectionInfoList.erase(itConn);
         }

         _deviceConnectionStatusListPropHdl.set(deviceConnectionStatusList);
      }
      else
      {
         // device connection info not found in device connection status list property for device handle
         // from given device connection info

         if ((BM_CONNECTION_STATUS_CONNECTING == deviceConnectionInfo._connectionStatus)
               || (BM_CONNECTION_STATUS_DISCONNECTED == deviceConnectionInfo._connectionStatus))
         {
            if (BM_CONNECTION_STATUS_CONNECTING == deviceConnectionInfo._connectionStatus)
            {
               // add given device connection info to device connection status list property
               deviceConnectionStatusList._deviceConnectionInfoList.push_back(deviceConnectionInfo);
               _deviceConnectionStatusListPropHdl.set(deviceConnectionStatusList);
               //todo mohamed: ensure device connecting is sent and device is added before protocol connecting sent from dbmanager
            }
         }
         else
         {
            ETG_TRACE_ERR(("onDeviceConnectionStatusChanged: could not find device connection info in device connection status list property for given device handle = %d",
                  deviceConnectionInfo._deviceHandle));
         }
      }

      ETG_TRACE_USR1(("onDeviceConnectionStatusChanged: updating PairedDeviceList property"));

      PairedDeviceList pairedDeviceList;
      _pairedDeviceListPropHdl.get(pairedDeviceList);

      DeviceBaseInfoList::iterator itPaired = std::find_if(pairedDeviceList._deviceBaseInfoList.begin(),
            pairedDeviceList._deviceBaseInfoList.end(),
            MeetsDeviceHandle<DeviceBaseInfo>(deviceConnectionInfo._deviceHandle));

      if (itPaired != pairedDeviceList._deviceBaseInfoList.end())
      {
         // found device base info in paired device list property for device handle
         // from given device connection info => update device base info

         itPaired->_connectionStatus = deviceConnectionInfo._connectionStatus;
         itPaired->_disconnectedReason = deviceConnectionInfo._disconnectedReason;
         itPaired->_connectionOrderInfo = deviceConnectionInfo._connectionOrderInfo;

         _pairedDeviceListPropHdl.set(pairedDeviceList);

         DeviceId deviceId(0u);
         Result result = LocalSpm::getDbManager().getDeviceId(deviceId, deviceConnectionInfo._deviceHandle);

         if(CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("onDeviceConnectionStatusChanged: could not get device ID from DB (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
         else
         {
            LocalSpm::getBtLimitationController().onDeviceConnectionStatusChange(deviceId, deviceConnectionInfo._connectionStatus);
         }
      }
      else
      {
         // device base info not found in paired device list property for device handle
         // from given device connection info

         ETG_TRACE_ERR(("onDeviceConnectionStatusChanged: could not find device base info in paired device list property for given device handle = %d",
               deviceConnectionInfo._deviceHandle));
      }
   }

   void BmCoreMainController::onProtocolConnectionStatusChanged(IN const DeviceId deviceId, IN const Protocol& protocol,
         IN const ConnectionStatus connectionStatus, IN const DisconnectedReason disconnectedReason,
         IN const RfcommDevicePath& rfcommDevicePath)
   {
      ETG_TRACE_USR1(("onProtocolConnectionStatusChanged: deviceId = %d, protocol = (%d, \"%50s\"), connectionStatus = %d, disconnectedReason = %d, rfcommDevicePath = \"%50s\"",
            deviceId, ETG_CENUM(ProtocolId, protocol._protocolId), protocol._uuid.c_str(),
            ETG_CENUM(ConnectionStatus, connectionStatus), ETG_CENUM(DisconnectedReason, disconnectedReason),
            rfcommDevicePath.c_str()));

      ETG_TRACE_USR1(("onProtocolConnectionStatusChanged: updating DeviceConnectionStatusList property"));

      DeviceId deviceHandle;
      Result result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("onProtocolConnectionStatusChanged(deviceId = %d, protocol = (%d, \"%50s\")): could not get device handle from DB (error = %d)",
               deviceId, ETG_CENUM(ProtocolId, protocol._protocolId), protocol._uuid.c_str(),
               ETG_CENUM(CcErrorInternal, result)));
         return;
      }

      DeviceConnectionStatusList deviceConnectionStatusList;
      _deviceConnectionStatusListPropHdl.get(deviceConnectionStatusList);

      DeviceConnectionInfoList::iterator itConn = std::find_if(deviceConnectionStatusList._deviceConnectionInfoList.begin(),
            deviceConnectionStatusList._deviceConnectionInfoList.end(),
            MeetsDeviceHandle<DeviceConnectionInfo>(deviceHandle));

      if (itConn != deviceConnectionStatusList._deviceConnectionInfoList.end())
      {
         // found device connection info in device connection status list property for device handle
         // from given device connection info => update device connection info

         if (BM_PROTOCOL_ID_SPP != protocol._protocolId)
         {
            ProtocolConnectionInfoMap::iterator itProt = itConn->_protocolConnectionInfo.find(protocol._protocolId);

            if (itConn->_protocolConnectionInfo.end() != itProt)
            {
               itProt->second._connectionStatus = connectionStatus;
               itProt->second._disconnectedReason = disconnectedReason;
            }
            else
            {
               ETG_TRACE_ERR(("onProtocolConnectionStatusChanged(deviceId = %d, protocol = (%d, \"%50s\")): could not find protocol connection info in device connection status list property for device handle = %d",
                     deviceId, ETG_CENUM(ProtocolId, protocol._protocolId), protocol._uuid.c_str(), deviceHandle));
            }
         }
         else
         {
            SppConnectionInfoMap::iterator itSpp = itConn->_sppConnectionInfo.find(protocol._uuid);

            if (itConn->_sppConnectionInfo.end() != itSpp)
            {
               itSpp->second._connectionStatus = connectionStatus;
               itSpp->second._disconnectedReason = disconnectedReason;
               itSpp->second._rfcommDevicePath = rfcommDevicePath;
            }
            else
            {
               ETG_TRACE_ERR(("onProtocolConnectionStatusChanged(deviceId = %d, protocol = (%d, \"%50s\")): could not find SPP connection info in device connection status list property for device handle = %d",
                     deviceId, ETG_CENUM(ProtocolId, protocol._protocolId), protocol._uuid.c_str(), deviceHandle));
            }
         }

         if(BM_PROTOCOL_ID_HFP == protocol._protocolId)
         {
            getPrimaryStatus(deviceConnectionStatusList);
         }

         _deviceConnectionStatusListPropHdl.set(deviceConnectionStatusList);

         LocalSpm::getBtLimitationController().onProtocolConnectionStatusChange(deviceId, protocol, connectionStatus);
      }
      else
      {
         // device connection info not found in device connection status list property for device handle
         // from given device connection info

         ETG_TRACE_ERR(("onProtocolConnectionStatusChanged(deviceId = %d, protocol = (%d, \"%50s\")): could not find device connection info in device connection status list property for device handle = %d",
               deviceId, ETG_CENUM(ProtocolId, protocol._protocolId), protocol._uuid.c_str(), deviceHandle));
      }

      if (protocol._protocolId == BM_PROTOCOL_ID_HFP)
      {
         DeviceId multiHfpDeviceId = _multiHFPSupportSwitchPropHdl.getDeviceId();
         SwitchStatus multiHFPSupportStatus;
         _multiHFPSupportSwitchPropHdl.get(multiHFPSupportStatus);

         if((deviceId == multiHfpDeviceId) && (connectionStatus == BM_CONNECTION_STATUS_DISCONNECTED)
               && (multiHFPSupportStatus._switchState == SWITCH_STATE_SWITCHING_OFF))
         {
            _multiHFPSupportSwitchPropHdl.onInternalSwitchStateUpdate(SWITCH_STATE_SWITCHED_OFF);
         }
      }
   }

   BmResult BmCoreMainController::setDeviceRemoteConnectableInt(IN const DeviceId deviceId,
         IN const RemoteConnectable remoteConnectable)
   {
      ETG_TRACE_USR1(("setDeviceRemoteConnectableInt: deviceId = %d remoteConnectable = %10s", deviceId,
            remoteConnectable ? "true" : "false"));

      DeviceId deviceHandle(0u);

      if (0u != deviceId)
      {
         Result result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("setDeviceRemoteConnectableInt: could not get device handle for given device ID from DB (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));

            return BM_RESULT_ERR_GENERAL;
         }
      }

      BmCoreIfMessage_SetDeviceRemoteConnectable internalMsg(deviceHandle, remoteConnectable, 0u,
            BM_CORE_IF_MSG_ORIGIN_INTERNAL);

      return this->handleBmCoreIfMessage_SetDeviceRemoteConnectable(&internalMsg);
   }

   Result BmCoreMainController::getDeviceRemoteConnectable(OUT RemoteConnectable& remoteConnectable,
         IN const DeviceId deviceId)
   {
      ETG_TRACE_USR1(("getDeviceRemoteConnectable: deviceId = %d", deviceId));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceHandle(0u);

      result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         PairedDeviceList pairedDevicesProp;
         DeviceBaseInfoList::iterator it;

         _pairedDeviceListPropHdl.get(pairedDevicesProp);

         it = std::find_if(pairedDevicesProp._deviceBaseInfoList.begin(),
               pairedDevicesProp._deviceBaseInfoList.end(),
               MeetsDeviceHandle<DeviceBaseInfo>(deviceHandle));

         if (it != pairedDevicesProp._deviceBaseInfoList.end())
         {
            remoteConnectable = it->_remoteConnectable;

            ETG_TRACE_USR1(("getDeviceRemoteConnectable(deviceId = %d): remoteConnectable = %d", deviceId, remoteConnectable));
         }
         else
         {
            ETG_TRACE_ERR(("getDeviceRemoteConnectable:error: DeviceId(%d) was not found in PairedDeviceList Property", deviceHandle));
            result = CC_ERR_INT_DEVICE_NOT_FOUND;
         }
      }
      else
      {
         ETG_TRACE_ERR(("getDeviceRemoteConnectable(deviceId = %d): could not get device handle from DB (error = %d)", deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result BmCoreMainController::getDeviceRemoteConnectable(OUT RemoteConnectable& remoteConnectable,
         IN const BdAddress& bdAddress)
   {
      ETG_TRACE_USR1(("getDeviceRemoteConnectable: bdAddress = \"%50s\"", bdAddress.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);

      result = LocalSpm::getDbManager().getDeviceId(deviceId, bdAddress);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         result = getDeviceRemoteConnectable(remoteConnectable, deviceId);
      }
      else
      {
         ETG_TRACE_ERR(("getDeviceRemoteConnectable(bdAddress = \"%50s\"): could not get device ID from DB (error = %d)",
               bdAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result BmCoreMainController::setDeviceiAPoverBTSupported(IN const DeviceId deviceId, IN const bool iAPoverBTSupported)
   {
      ETG_TRACE_USR1(("setDeviceiAPoverBTSupported: deviceId = %d iAPoverBTSupported = %d", deviceId, iAPoverBTSupported));

      Result result(CC_ERR_INT_NO_ERROR);

      if(0u != deviceId)
      {
         DeviceId deviceHandle(0u);

         result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            PairedDeviceList pairedDevicesProp;
            DeviceBaseInfoList::iterator it;

            _pairedDeviceListPropHdl.get(pairedDevicesProp);

            it = std::find_if(pairedDevicesProp._deviceBaseInfoList.begin(),
                  pairedDevicesProp._deviceBaseInfoList.end(),
                  MeetsDeviceHandle<DeviceBaseInfo>(deviceHandle));

            if (it != pairedDevicesProp._deviceBaseInfoList.end())
            {
               it->_iAPoverBTSupported = iAPoverBTSupported;

               ETG_TRACE_USR1(("setDeviceiAPoverBTSupported: Update the pairedDeviceList property"));
               _pairedDeviceListPropHdl.set(pairedDevicesProp);
            }
            else
            {
               ETG_TRACE_ERR(("setDeviceiAPoverBTSupported:error: DeviceId(%d) was not found in PairedDeviceList Property", deviceHandle));
               result = CC_ERR_INT_DEVICE_NOT_FOUND;
            }
         }
         else
         {
            ETG_TRACE_ERR(("setDeviceiAPoverBTSupported(deviceId = %d): could not get device handle from DB (error = %d)", deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return result;
   }

   Result BmCoreMainController::getDeviceiAPoverBTSupported(OUT bool &iAPoverBTSupported, IN const DeviceId deviceId)
   {
      ETG_TRACE_USR1(("getDeviceiAPoverBTSupported: deviceId = %d", deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      DeviceId deviceHandle(0u);

      result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         PairedDeviceList pairedDevicesProp;
         DeviceBaseInfoList::iterator it;

         _pairedDeviceListPropHdl.get(pairedDevicesProp);

         it = std::find_if(pairedDevicesProp._deviceBaseInfoList.begin(),
               pairedDevicesProp._deviceBaseInfoList.end(),
               MeetsDeviceHandle<DeviceBaseInfo>(deviceHandle));

         if (it != pairedDevicesProp._deviceBaseInfoList.end())
         {
            iAPoverBTSupported = it->_iAPoverBTSupported;

            ETG_TRACE_USR1(("getDeviceiAPoverBTSupported: deviceId = %d is iAPoverBTSupported = %10s", deviceId, iAPoverBTSupported ? "true": "false"));
         }
         else
         {
            ETG_TRACE_ERR(("getDeviceiAPoverBTSupported:error: DeviceId(%d) was not found in PairedDeviceList Property", deviceHandle));
            result = CC_ERR_INT_DEVICE_NOT_FOUND;
         }
      }
      else
      {
         ETG_TRACE_ERR(("getDeviceiAPoverBTSupported(deviceId = %d): could not get device handle from DB (error = %d)", deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   BmResult BmCoreMainController::getRemoteDeviceInfo(OUT RemoteDeviceInfo& remoteDeviceInfo,
         IN const DeviceId deviceHandle, IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getRemoteDeviceInfo: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getRemoteDeviceInfo: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      BmResult bmResult(BM_RESULT_OK);
      Result result(CC_ERR_INT_NO_ERROR);

      PairedDeviceList pairedDeviceList;
      _pairedDeviceListPropHdl.get(pairedDeviceList);

      DeviceBaseInfoList::iterator it = std::find_if(pairedDeviceList._deviceBaseInfoList.begin(),
            pairedDeviceList._deviceBaseInfoList.end(),
            MeetsDeviceHandle<DeviceBaseInfo>(deviceHandle));

      if (it != pairedDeviceList._deviceBaseInfoList.end())
      {
         remoteDeviceInfo._deviceBaseInfo = (*it);

         DeviceId deviceId = 0;
         result = LocalSpm::getDbManager().getDeviceId(deviceId, deviceHandle);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("getRemoteDeviceInfo(deviceHandle = %d): could not get deviceId (error = %d)", deviceHandle,
                  ETG_CENUM(CcErrorInternal, result)));

            return BM_RESULT_ERR_INVALID_PARAMETER;
         }

         result = LocalSpm::getDbManager().getLinkKey(remoteDeviceInfo._linkKey, deviceId);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            DeviceConnectionStatusList deviceConnectionStatusList;
            _deviceConnectionStatusListPropHdl.get(deviceConnectionStatusList);

            DeviceConnectionInfoList::iterator it2 = std::find_if(deviceConnectionStatusList._deviceConnectionInfoList.begin(),
                  deviceConnectionStatusList._deviceConnectionInfoList.end(),
                  MeetsDeviceHandle<DeviceConnectionInfo>(deviceHandle));

            if (it2 != deviceConnectionStatusList._deviceConnectionInfoList.end())
            {
               remoteDeviceInfo._protocolConnectionInfo = it2->_protocolConnectionInfo;
               remoteDeviceInfo._sppConnectionInfo = it2->_sppConnectionInfo;
            }
            else
            {
               result = LocalSpm::getDbManager().getProtocolConnectionInfos(remoteDeviceInfo._protocolConnectionInfo,
                     remoteDeviceInfo._sppConnectionInfo, deviceId);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("getRemoteDeviceInfo(deviceHandle = %d): could not get protocol connection info from DB (error = %d)",
                        deviceHandle, ETG_CENUM(CcErrorInternal, result)));
                  bmResult = BM_RESULT_ERR_GENERAL;
               }
            }
         }
         else
         {
            ETG_TRACE_ERR(("getRemoteDeviceInfo(deviceHandle = %d): could not get link key from DB (error = %d)",
                  deviceHandle, ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         result = CC_ERR_INT_DEVICE_NOT_FOUND;
         ETG_TRACE_ERR(("getRemoteDeviceInfo(deviceHandle = %d): given device is not found in the paired device list (error = %d)",
               deviceHandle, ETG_CENUM(CcErrorInternal, result)));
         bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
      }

      VARTRACE(remoteDeviceInfo);

      return bmResult;
   }

   void BmCoreMainController::onDeviceDeleted(IN const DeviceId deviceHandle)
   {
      ETG_TRACE_USR1(("onDeviceDeleted: deviceHandle = %d", deviceHandle));

      ETG_TRACE_USR1(("onDeviceDeleted(deviceHandle = %d): updating PairedDeviceList, DeviceConnectionStatusList and BlockStatus properties",
            deviceHandle));

      PairedDeviceList pairedDeviceList;
      _pairedDeviceListPropHdl.get(pairedDeviceList);

      DeviceConnectionStatusList deviceConnectionStatusList;
      _deviceConnectionStatusListPropHdl.get(deviceConnectionStatusList);

      BlockStatus blockStatus;
      (void) this->getBlockStatus(blockStatus);

      if (0u == deviceHandle)
      {
         pairedDeviceList._deviceBaseInfoList.clear();
         deviceConnectionStatusList._deviceConnectionInfoList.clear();
         blockStatus._overallDeviceBlockStatus = OverallDeviceBlockStatus();
         blockStatus._deviceBlockStatusList.clear();
      }
      else
      {
         DeviceBaseInfoList::iterator itPaired = std::find_if(pairedDeviceList._deviceBaseInfoList.begin(),
               pairedDeviceList._deviceBaseInfoList.end(),
               MeetsDeviceHandle<DeviceBaseInfo>(deviceHandle));

         if (itPaired != pairedDeviceList._deviceBaseInfoList.end())
         {
            // found device base info in paired device list property for device handle
            // from given device connection info => delete device base info

            pairedDeviceList._deviceBaseInfoList.erase(itPaired);
         }
         else
         {
            // device base info not found in paired device list property for device handle
            // from given device connection info

            ETG_TRACE_ERR(("onDeviceDeleted(deviceHandle = %d): could not find device base info for given device handle in paired device list property",
                  deviceHandle));
         }

         DeviceConnectionInfoList::iterator itConn = std::find_if(deviceConnectionStatusList._deviceConnectionInfoList.begin(),
               deviceConnectionStatusList._deviceConnectionInfoList.end(),
               MeetsDeviceHandle<DeviceConnectionInfo>(deviceHandle));

         if (itConn != deviceConnectionStatusList._deviceConnectionInfoList.end())
         {
            // found device connection status info in device connection status list property for given device handle
            // => delete device base info

            ETG_TRACE_ERR(("onDeviceDeleted(deviceHandle = %d): could find device connection status info for given device handle in device connection status list property (device not yet disconnected?)",
                  deviceHandle));

            deviceConnectionStatusList._deviceConnectionInfoList.erase(itConn);
         }
         else
         {
            // as expected, device connection status info not found in device connection status list property for given device handle

            ETG_TRACE_USR1(("onDeviceDeleted(deviceHandle = %d): as expected, could not find device connection status info for given device handle in device connection status list property",
                  deviceHandle));
         }

         DeviceBlockStatusList::iterator itBlock = std::find_if(blockStatus._deviceBlockStatusList.begin(),
               blockStatus._deviceBlockStatusList.end(),
               MeetsDeviceHandle<DeviceBlockStatus>(deviceHandle));

         if (itBlock != blockStatus._deviceBlockStatusList.end())
         {
               blockStatus._deviceBlockStatusList.erase(itBlock);
         }
         else
         {
            ETG_TRACE_ERR(("onDeviceDeleted(deviceHandle = %d): could not find block status info for given device handle in block status property",
                  deviceHandle));
         }
      }

      _pairedDeviceListPropHdl.set(pairedDeviceList);

      (void) _remoteConnectableController.removeDeviceRemoteConnectableControl(deviceHandle);

      _deviceConnectionStatusListPropHdl.set(deviceConnectionStatusList);

      this->setBlockStatusInt(blockStatus);
   }

   BmResult BmCoreMainController::getDeviceConnectionStatusList(OUT DeviceConnectionStatusList& deviceConnectionStatusList,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getDeviceConnectionStatusList: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getDeviceConnectionStatusList: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _deviceConnectionStatusListPropHdl.get(deviceConnectionStatusList);

      VARTRACE(deviceConnectionStatusList);


      return BM_RESULT_OK;
   }

   BmResult BmCoreMainController::getLinkQuality(OUT LinkQuality& linkQuality, IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getLinkQuality: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getLinkQuality: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _linkQualityPropHdl.get(linkQuality);

      VARTRACE(linkQuality);

      return BM_RESULT_OK;
   }

   void BmCoreMainController::onLinkQuality(IN const LinkQuality& linkQuality)
   {
      ETG_TRACE_USR1(("onLinkQuality"));

      _linkQualityPropHdl.set(linkQuality);
   }

   BmResult BmCoreMainController::getBlockStatus(OUT BlockStatus& blockStatus,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getBlockStatus: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getBlockStatus: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _blockStatusPropHdl.get(blockStatus);

      VARTRACE(blockStatus);

      return BM_RESULT_OK;
   }

   void BmCoreMainController::setBlockStatusInt(IN const BlockStatus& blockStatus)
   {
      ETG_TRACE_USR1(("setBlockStatusInt"));

      _blockStatusPropHdl.set(blockStatus);
   }

   void BmCoreMainController::updateBlockStatus(IN const DeviceId devHandle, IN const BlockStatus& blockStatus,
         IN const BlockStatusUpdateType updateType)
   {
      ETG_TRACE_USR1(("updateBlockStatus: deviceHandle = %d, updateType = %d", devHandle,
            ETG_CENUM(BlockStatusUpdateType, updateType)));

      BlockStatus tmpblockStatus;
      (void) this->getBlockStatus(tmpblockStatus);

      if (BM_BLOCK_STATUS_UPDATE_TYPE_OVERALL == updateType)
      {
         tmpblockStatus._overallDeviceBlockStatus = blockStatus._overallDeviceBlockStatus;
         this->setBlockStatusInt(tmpblockStatus);
      }
      else
      {
         DeviceBlockStatusList::iterator it = std::find_if(tmpblockStatus._deviceBlockStatusList.begin(),
               tmpblockStatus._deviceBlockStatusList.end(),
               MeetsDeviceHandle<DeviceBlockStatus>(devHandle));

         if (it != tmpblockStatus._deviceBlockStatusList.end())
         {
            if (BM_BLOCK_STATUS_UPDATE_TYPE_DELETE_DEVICE == updateType)
            {
               tmpblockStatus._deviceBlockStatusList.erase(it);
               this->setBlockStatusInt(tmpblockStatus);
            }
            else if(BM_BLOCK_STATUS_UPDATE_TYPE_DEVICE == updateType)
            {
               if (!blockStatus._deviceBlockStatusList.empty())
               {
                  DeviceBlockStatus inputDevBlockStatus = *(blockStatus._deviceBlockStatusList.begin());
                  it->_blockState = inputDevBlockStatus._blockState;
                  this->setBlockStatusInt(tmpblockStatus);
               }
               else
               {
                  ETG_TRACE_ERR(("updateBlockStatus(deviceHandle = %d, updateType = %d): invalid Parameter[blockStatus._deviceBlockStatusList] couldn't find Device Data for update",
                        devHandle, ETG_CENUM(BlockStatusUpdateType, updateType)));
               }
            }
            else if(BM_BLOCK_STATUS_UPDATE_TYPE_DEVICE_PROTOCOL_BLOCK_STATE == updateType)
            {
               if (!blockStatus._deviceBlockStatusList.empty())
               {
                  DeviceBlockStatus inputDevBlockStatus = *(blockStatus._deviceBlockStatusList.begin());

                  if (!inputDevBlockStatus._protocolBlockStatusMap.empty())
                  {
                     Protocol protocol = inputDevBlockStatus._protocolBlockStatusMap.begin()->first;
                     BlockState protocolBlockState = inputDevBlockStatus._protocolBlockStatusMap.begin()->second;

                     ProtocolBlockStatusMap::iterator mapIt;
                     mapIt = it->_protocolBlockStatusMap.find(protocol);

                     if(mapIt != it->_protocolBlockStatusMap.end())
                     {
                        mapIt->second = protocolBlockState;
                        this->setBlockStatusInt(tmpblockStatus);
                     }
                     else
                     {
                        ETG_TRACE_ERR(("updateBlockStatus(deviceHandle = %d, updateType = %d): Couldn't find ProtocolId(%d), uuid(%50s) for DeviceHandle(%d) in BlockStatus Property",
                              devHandle, ETG_CENUM(BlockStatusUpdateType, updateType), ETG_CENUM(ProtocolId, protocol._protocolId), protocol._uuid.c_str(), devHandle));
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("updateBlockStatus(deviceHandle = %d, updateType = %d): Invalid Parameter[blockStatus._deviceBlockStatusList[0]._protocolBlockStatusMap] couldn't find Device Protocol Data for update",
                           devHandle, ETG_CENUM(BlockStatusUpdateType, updateType)));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("updateBlockStatus(deviceHandle = %d, updateType = %d): Invalid Parameter[blockStatus._deviceBlockStatusList] couldn't find Device Data for update",
                        devHandle, ETG_CENUM(BlockStatusUpdateType, updateType)));
               }
            }
            else if(BM_BLOCK_STATUS_UPDATE_TYPE_DEVICE_PROTOCOL_SUPPORT == updateType)
            {
               DeviceBlockStatus devBlockStatus;
               DeviceId devId;
               Result res(CC_ERR_INT_NO_ERROR);
               res = LocalSpm::getDbManager().getDeviceId(devId, devHandle);
               if(CC_ERR_INT_NO_ERROR == res)
               {
                  res = LocalSpm::getDbManager().getDeviceBlockStatus(devBlockStatus, devId);

                  if(CC_ERR_INT_NO_ERROR == res)
                  {
                     *it = devBlockStatus;
                     this->setBlockStatusInt(tmpblockStatus);
                  }
                  else
                  {
                     ETG_TRACE_ERR(("updateBlockStatus(deviceHandle = %d, updateType = %d): Couldn't get DeviceBlockStatus for DeviceId(%d)",
                           devHandle, ETG_CENUM(BlockStatusUpdateType, updateType), devId));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("updateBlockStatus(deviceHandle = %d, updateType = %d): Couldn't get DeviceId for DeviceHandle(%d)",
                        devHandle, ETG_CENUM(BlockStatusUpdateType, updateType), devHandle));
               }
            }
            else if(BM_BLOCK_STATUS_UPDATE_TYPE_INSERT_DEVICE == updateType)
            {
               ETG_TRACE_ERR(("updateBlockStatus(deviceHandle = %d, updateType = %d): DeviceHandle(%d) already exists!",
                     devHandle, ETG_CENUM(BlockStatusUpdateType, updateType), devHandle));
            }
            else
            {
               ETG_TRACE_ERR(("updateBlockStatus(deviceHandle = %d, updateType = %d): invalid BlockStatus update type",
                     devHandle, ETG_CENUM(BlockStatusUpdateType, updateType)));
            }
         }
         else
         {
            if (BM_BLOCK_STATUS_UPDATE_TYPE_INSERT_DEVICE == updateType)
            {
               DeviceBlockStatus devBlockStatus;
               DeviceId devId;
               Result res(CC_ERR_INT_NO_ERROR);
               res = LocalSpm::getDbManager().getDeviceId(devId, devHandle);
               if(CC_ERR_INT_NO_ERROR == res)
               {
                  res = LocalSpm::getDbManager().getDeviceBlockStatus(devBlockStatus, devId);

                  if(CC_ERR_INT_NO_ERROR == res)
                  {
                     tmpblockStatus._deviceBlockStatusList.push_back(devBlockStatus);
                     this->setBlockStatusInt(tmpblockStatus);
                  }
                  else
                  {
                     ETG_TRACE_ERR(("updateBlockStatus(deviceHandle = %d, updateType = %d): Couldn't get DeviceBlockStatus for DeviceId(%d)",
                           devHandle, ETG_CENUM(BlockStatusUpdateType, updateType), devId));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("updateBlockStatus(deviceHandle = %d, updateType = %d): Couldn't get DeviceId for DeviceHandle(%d)",
                        devHandle, ETG_CENUM(BlockStatusUpdateType, updateType), devHandle));
               }
            }
            else
            {
               ETG_TRACE_ERR(("updateBlockStatus(deviceHandle = %d, updateType = %d): Couldn't find DeviceHandle(%d) in BlockStatus Property",
                     devHandle, ETG_CENUM(BlockStatusUpdateType, updateType), devHandle));
            }
         }
      }
   }

   BmResult BmCoreMainController::getAutoConnectionStatus(OUT AutoConnectionStatus& autoConnectionStatus,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getAutoConnectionStatus: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getAutoConnectionStatus: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _autoConnectionStatusPropHdl.get(autoConnectionStatus);

      VARTRACE(autoConnectionStatus);

      return BM_RESULT_OK;
   }

   Result BmCoreMainController::getDeviceIdForBdName(OUT DeviceId& deviceId, IN const BdName& bdName)
   {
      ETG_TRACE_USR1(("getDeviceIdForBdName: bdName = \"%50s\"", bdName.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);
      deviceId = 0u;

      /*get the device ID from the Devices table for the given BD name*/
      PairedDeviceList pairedDeviceList;
      this->getPairedDeviceList(pairedDeviceList);

      DeviceBaseInfoList::iterator it;

      for (it = pairedDeviceList._deviceBaseInfoList.begin(); it != pairedDeviceList._deviceBaseInfoList.end(); ++it)
      {
         if(bdName == it->_bdName)
         {
            result = LocalSpm::getDbManager().getDeviceId(deviceId, it->_deviceHandle);
            break;
         }
      }

      if (0u == deviceId)
      {
         result = CC_ERR_INT_DEVICE_NOT_FOUND;
      }

      return result;
   }

Result BmCoreMainController::getDeviceHandleForBdName(OUT DeviceId& deviceHandle, IN const BdName& bdName)
{
   ETG_TRACE_USR1(("getDeviceHandleForBdName: bdName = \"%50s\"", bdName.c_str()));

   Result result(CC_ERR_INT_NO_ERROR);
   deviceHandle = 0u;

   /*get the device Handle for the given BD name*/
   PairedDeviceList pairedDeviceList;
   this->getPairedDeviceList(pairedDeviceList);

   DeviceBaseInfoList::iterator it;

   for (it = pairedDeviceList._deviceBaseInfoList.begin(); it != pairedDeviceList._deviceBaseInfoList.end(); ++it)
   {
      if(bdName == it->_bdName)
      {
         deviceHandle = it->_deviceHandle;
         break;
      }
   }

   if (0u == deviceHandle)
   {
      result = CC_ERR_INT_DEVICE_NOT_FOUND;
   }

   return result;
}
   BmResult BmCoreMainController::getBtLimitationMode(OUT BtLimitationMode& btLimitationMode,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getBtLimitationMode: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getBtLimitationMode: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _btLimitationModePropHdl.get(btLimitationMode);

      VARTRACE(btLimitationMode);

      return BM_RESULT_OK;
   }

   Result BmCoreMainController::getBtLimitationModeInfo(OUT BtLimitationModeInfo& btLimitationModeInfo,
         IN const BdAddress& bdAddress, IN BdName bdName)
   {
      ETG_TRACE_USR1(("getBtLimitationModeInfo"));

      Result result(CC_ERR_INT_ITEM_NOT_FOUND);

      BtLimitationMode btLimitationMode;
      _btLimitationModePropHdl.get(btLimitationMode);

      for (size_t idx = 0u; idx < btLimitationMode._btLimitationModeInfoList.size(); ++idx)
      {
         if ((bdAddress ==  btLimitationMode._btLimitationModeInfoList[idx]._bdAddress)
               && (bdName ==  btLimitationMode._btLimitationModeInfoList[idx]._bdName))
         {
            btLimitationModeInfo = btLimitationMode._btLimitationModeInfoList[idx];
            result = CC_ERR_INT_NO_ERROR;
            break;
         }
      }

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR1(("getBtLimitationModeInfo: found BT limitation mode for BD address = \"%50s\": BD name = \"%50s\": limitationMode = (%d, %d), limitationState = %d",
               btLimitationModeInfo._bdAddress.c_str(),
               btLimitationModeInfo._bdName.c_str(),
               ETG_CENUM(LimitationFeature, btLimitationModeInfo._limitationMode._limitationFeature),
               ETG_CENUM(LimitationCommunicationIf, btLimitationModeInfo._limitationMode._limitationCommunicationIf),
               ETG_CENUM(LimitationState, btLimitationModeInfo._limitationState)));
      }
      else
      {
         ETG_TRACE_USR1(("getBtLimitationModeInfo: no BT limitation mode found for BD address = \"%50s\"",
               bdAddress.c_str()));
      }

      return result;
   }

   Result BmCoreMainController::getBtLimitationModeInfo(OUT BtLimitationModeInfo& btLimitationModeInfo,
            IN const DeviceId deviceId)
   {
      ETG_TRACE_USR1(("getBtLimitationModeInfo : deviceId = %d", deviceId));
      Result result(CC_ERR_INT_NO_ERROR);

      BdAddress bdAddress;
      result = LocalSpm::getDbManager().getBdAddress(bdAddress, deviceId);

      if(CC_ERR_INT_NO_ERROR == result)
      {
         result = getBtLimitationModeInfo(btLimitationModeInfo, bdAddress);
      }

      return result;
   }

   Result BmCoreMainController::updateBtLimitationMode(IN const BdAddress& bdAddress, IN const BdName& bdName,
         IN const LimitationMode& limitationMode, IN const LimitationState limitationState)
   {
      ETG_TRACE_USR1(("updateBtLimitationMode: bdAddress = \"%50s\", bdName = \"%50s\", limitationMode = (%d, %d), limitationState = %d",
            bdAddress.c_str(), bdName.c_str(), ETG_CENUM(LimitationFeature, limitationMode._limitationFeature),
            ETG_CENUM(LimitationCommunicationIf, limitationMode._limitationCommunicationIf),
            ETG_CENUM(LimitationState, limitationState)));

      BtLimitationMode btLimitationMode;
      _btLimitationModePropHdl.get(btLimitationMode);

      BtLimitationModeInfoList::iterator it;

      for (it = btLimitationMode._btLimitationModeInfoList.begin(); it != btLimitationMode._btLimitationModeInfoList.end(); ++it)
      {
         if ((bdAddress == it->_bdAddress) && (bdName == it->_bdName))
         {
            break;
         }
      }

      if (it != btLimitationMode._btLimitationModeInfoList.end())
      {
         // found already existing BT limitation for given BD address, so update entry
         it->_limitationState = limitationState;

         //NCG3D-219484 : Update the proper Limitation communication and feature
         it->_limitationMode._limitationFeature = limitationMode._limitationFeature;
         it->_limitationMode._limitationCommunicationIf = limitationMode._limitationCommunicationIf;
      }
      else
      {
         // given BT limitation is new, so add entry
         btLimitationMode._btLimitationModeInfoList.push_back(BtLimitationModeInfo(bdAddress, bdName, limitationMode, limitationState));
      }

      _btLimitationModePropHdl.set(btLimitationMode);

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmCoreMainController::deleteBtLimitationModeItem(IN const BdAddress& bdAddress, IN const BdName& bdName, IN const LimitationMode& limitationMode)
   {
      ETG_TRACE_USR1(("deleteBtLimitationModeItem: bdAddress = \"%50s\",  bdName = \"%50s\", limitationMode = (%d, %d)",
            bdAddress.c_str(), bdName.c_str(), ETG_CENUM(LimitationFeature, limitationMode._limitationFeature),
            ETG_CENUM(LimitationCommunicationIf, limitationMode._limitationCommunicationIf)));

      BtLimitationMode btLimitationMode;
      _btLimitationModePropHdl.get(btLimitationMode);

      BtLimitationModeInfoList::iterator it;

      for (it = btLimitationMode._btLimitationModeInfoList.begin(); it != btLimitationMode._btLimitationModeInfoList.end(); ++it)
      {
         if ((bdAddress == it->_bdAddress) && (bdName == it->_bdName)
               && (limitationMode._limitationFeature == it->_limitationMode._limitationFeature)
               && (limitationMode._limitationCommunicationIf == it->_limitationMode._limitationCommunicationIf))
         {
            break;
         }
      }

      if (it != btLimitationMode._btLimitationModeInfoList.end())
      {
         // found existing BT limitation entry for given BD address, so remove entry
         btLimitationMode._btLimitationModeInfoList.erase(it);
      }
      else
      {
         // given BT limitation entry not found
      }

      _btLimitationModePropHdl.set(btLimitationMode);

      return CC_ERR_INT_NO_ERROR;
   }

   BmResult BmCoreMainController::getTestMode(OUT SwitchStatus& testMode, IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getTestMode: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getTestMode: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _testModePropHdl.get(testMode);

      VARTRACE(testMode);

      return BM_RESULT_OK;
   }

   BmResult BmCoreMainController::getMultiHFPSupportStatus(OUT SwitchStatus& multiHFPSupportStatus,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getMultiHFPSupportStatus: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getMultiHFPSupportStatus: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _multiHFPSupportSwitchPropHdl.get(multiHFPSupportStatus);

      VARTRACE(multiHFPSupportStatus);

      return BM_RESULT_OK;
   }

   BmResult BmCoreMainController::getDeviceUsagePreference(OUT UsagePreference& usagePreference,
         IN const DeviceId deviceHandle, IN const BmCoreIfMsgOrigin origin)
   {
      Result result(CC_ERR_INT_NO_ERROR);

      ETG_TRACE_USR1(("getDeviceUsagePreference: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getDeviceUsagePreference: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      PairedDeviceList pairedDevicesProp;
      DeviceBaseInfoList::iterator it;

      _pairedDeviceListPropHdl.get(pairedDevicesProp);

      it = std::find_if(pairedDevicesProp._deviceBaseInfoList.begin(),
            pairedDevicesProp._deviceBaseInfoList.end(),
            MeetsDeviceHandle<DeviceBaseInfo>(deviceHandle));

      if (it != pairedDevicesProp._deviceBaseInfoList.end())
      {
         usagePreference = it->_usagePreference;

         ETG_TRACE_USR1(("getDeviceUsagePreference(deviceHandle = %d): usagePreference = %d", deviceHandle,
               ETG_CENUM(UsagePreference, usagePreference)));
      }
      else
      {
         ETG_TRACE_ERR(("getDeviceUsagePreference:error: deviceHandle(%d) was not found in PairedDeviceList Property",
               deviceHandle));
         result = CC_ERR_INT_DEVICE_NOT_FOUND;
      }

      if (CC_ERR_INT_NO_ERROR != result)
      {
         return BM_RESULT_ERR_GENERAL;
      }

      VARTRACE(usagePreference);

      return BM_RESULT_OK;
   }

   Result BmCoreMainController::getBtProfileUsageInfoList(OUT ProfileUsageList& btProfileUsageInfoList)
   {
      ETG_TRACE_USR1(("getBtProfileUsageInfoList"));

      _btProfileUsageInfoPropHdl.get(btProfileUsageInfoList);

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmCoreMainController::updateBtProfileUsageInfoList(IN ProfileUsageList& btProfileUsageListUpdate)
   {
      ETG_TRACE_USR1(("updateBtProfileUsageInfoList entered"));

      Result result = CC_ERR_INT_NO_ERROR;
      ProfileUsageList btCurrProfileUsageList;
      _btProfileUsageInfoPropHdl.get(btCurrProfileUsageList);

      ETG_TRACE_USR4(("updateBtProfileUsageInfoList Current List is "));
      VARTRACE(btCurrProfileUsageList);

      ETG_TRACE_USR4(("updateBtProfileUsageInfoList Updated List is"));
      VARTRACE(btProfileUsageListUpdate);

      //TODO:If client sends profile usage with inUse flag set to False without setting it to true first - should be handled as an error case
      //!Current list is empty. Directly update it
      if(true == btCurrProfileUsageList._btProfileUsageInfoList.empty())
      {
          ETG_TRACE_USR4(("updateBtProfileUsageInfoList - Current list is empty. Update with new Usage list"));
          _btProfileUsageInfoPropHdl.set(btProfileUsageListUpdate);
      }//End of if(true == btCurrProfileUsageList._btProfileUsageInfoList.empty())

      //!Current list is not empty. Update the existing Profile Usage List
      else
      {
         ETG_TRACE_USR4(("updateBtProfileUsageInfoList - Current list is not empty. Updating existing Profile Usage List"));

         //!Iterate through the new incoming list
         for(ProfileUsageInfoListItr UpdateListItr = btProfileUsageListUpdate._btProfileUsageInfoList.begin();
               UpdateListItr != btProfileUsageListUpdate._btProfileUsageInfoList.end(); UpdateListItr++)
         {
            ProfileUsageInfoListItr CurrentListItr = btCurrProfileUsageList._btProfileUsageInfoList.begin();

            //!Iterate through the current existing list
            for (; CurrentListItr != btCurrProfileUsageList._btProfileUsageInfoList.end(); CurrentListItr++)
            {
               if((CurrentListItr->_deviceHandle == UpdateListItr->_deviceHandle) &&
                     (CurrentListItr->_protocol == UpdateListItr->_protocol)&&
                     (CurrentListItr->_ProfileUsageType == UpdateListItr->_ProfileUsageType) &&
                     (true == CurrentListItr->_InUse) && (false == UpdateListItr->_InUse))
               {
                  //!Update state of an existing conflict to SOLVED state as the profile is not in use
                  //!and resolved on its own before user decision
                  ETG_TRACE_USR4(("updateBtProfileUsageInfoList - Existing Profile usage is not in use now"));

                  IssueInfo conflictInfo(BM_CONFLICT_TYPE_PROFILE_IN_USE, UpdateListItr->_deviceHandle,
                        UpdateListItr->_protocol, UpdateListItr->_ProfileUsageType, BM_CONFLICT_STATE_SOLVED);
                  updateBtConflictsListItem(conflictInfo);

                  //!Profile is not in use anymore. Update the current profile list entry item.
                  CurrentListItr->_InUse = false;
                  _btProfileUsageInfoPropHdl.set(btCurrProfileUsageList);
                  break;
               }//End of if((CurrentListItr->_deviceHandle == UpdateListItr->_deviceHandle) &&...)
            }//End of for (; CurrentListItr != btCurrProfileUsageList._btProfileUsageInfoList.end() ...)

            //!Add to list only if the profile is in use
            if ((CurrentListItr == btCurrProfileUsageList._btProfileUsageInfoList.end()) && (true == UpdateListItr->_InUse))
            {
               ETG_TRACE_USR4(("updateBtProfileUsageInfoList - Extending the profile usage list"));
               //!New Entry. Add to the profile usage list
               btCurrProfileUsageList._btProfileUsageInfoList.push_back(*UpdateListItr);
               VARTRACE(btCurrProfileUsageList);
            }//End of if ((CurrentListItr == btCurrProfileUsageList._btProfileUsageInfoList.end())..)

            //!Erase from the list as the profile is not in use anymore
            else if(CurrentListItr != btCurrProfileUsageList._btProfileUsageInfoList.end())
            {
               btCurrProfileUsageList._btProfileUsageInfoList.erase(CurrentListItr);
            }//End of else if(CurrentListItr != btCurrProfileUsageList...)
         }//End of for(ProfileUsageInfoListConstItr UpdateListItr = btProfileUsageListUpdate._btProfileUsageInfoList.begin();..)

         ETG_TRACE_USR4(("updateBtProfileUsageInfoList - Updated Profile Usage List"));
         VARTRACE(btCurrProfileUsageList);
         _btProfileUsageInfoPropHdl.set(btCurrProfileUsageList);
      }//End of else

      return result;
   }

   Result BmCoreMainController::getBtConflictsDetectedList(OUT IssueInfoList& btConflictsDetectedList)
   {
      ETG_TRACE_USR1(("getBtConflictsDetectedList"));

      _userDecisionRequiredPropHdl.get(btConflictsDetectedList);

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmCoreMainController::updateBtConflictsDetectedList(IN const IssueInfoList& btConflictsDetectedList)
   {
      Result retVal = CC_ERR_INT_NO_ERROR;

      ETG_TRACE_USR4(("updateBtConflictsDetectedList - Conflicts detected List Update"));
      VARTRACE(btConflictsDetectedList);

      IssueInfoList btCurrConflictsList;
      _userDecisionRequiredPropHdl.get(btCurrConflictsList);

      //!No active conflict trigger
      if((BM_CONFLICT_TRIGGER_NONE == btCurrConflictsList._conflictTrigger)
            && (true == btCurrConflictsList._conflictInfoList.empty()))
      {
         btCurrConflictsList._conflictTrigger = btConflictsDetectedList._conflictTrigger;
         btCurrConflictsList._conflictInfoList = btConflictsDetectedList._conflictInfoList;

         ETG_TRACE_USR4(("updateBtConflictsDetectedList - Current Conflicts List"));
         VARTRACE(btCurrConflictsList);
         _userDecisionRequiredPropHdl.set(btCurrConflictsList);
      }//End of if((BM_CONFLICT_TRIGGER_NONE == btCurrConflictsList._conflictTrigger)...)
      //!Update for the current existing conflict trigger is received. Update the conflicts list
      else if(btCurrConflictsList._conflictTrigger == btConflictsDetectedList._conflictTrigger)
      {
         btCurrConflictsList._conflictInfoList = btConflictsDetectedList._conflictInfoList;

         ETG_TRACE_USR4(("updateBtConflictsDetectedList - Updating conflicts for an existing conflict trigger."));
         VARTRACE(btCurrConflictsList);
         _userDecisionRequiredPropHdl.set(btCurrConflictsList);
      }//End of else if(btCurrConflictsList._conflictTrigger == btConflictsDetectedList._conflictTrigger)
      else
      {
         ETG_TRACE_ERR(("updateBtConflictsDetectedList - Already one conflict is in the process to be resolved. "
               "Cannot handle new conflict trigger. Store conflicts list internally"));

         //!Store in a temporary list and update clients once the ongoing conflict is resolved
         _pendingIssuesToBeResolvedList.push_back(btConflictsDetectedList);

         retVal = CC_ERR_INT_NOT_ALLOWED;
      }//End of else

      return retVal;
   }

   Result BmCoreMainController::updateBtConflictsListItem(IN const IssueInfo& btConflictsListItem)
   {
      Result retVal = CC_ERR_INT_NO_ERROR;

      ETG_TRACE_USR4(("updateBtConflictsListItem - Conflict List Item Update"));
      VARTRACE(btConflictsListItem);

      IssueInfoList btCurrConflictsList;
      _userDecisionRequiredPropHdl.get(btCurrConflictsList);

      //!There is an active Conflict Trigger with list of conflicts.
      if((BM_CONFLICT_TRIGGER_NONE != btCurrConflictsList._conflictTrigger)
            && (false == btCurrConflictsList._conflictInfoList.empty()))
      {
         for(ConflictInfoListItr itr = btCurrConflictsList._conflictInfoList.begin();
               itr != btCurrConflictsList._conflictInfoList.end(); itr++)
         {
            //!Update the conflict state to Solved
            if((itr->_conflictType == btConflictsListItem._conflictType) &&
                  (itr->_deviceID == btConflictsListItem._deviceID) &&
                  (itr->_protocol == btConflictsListItem._protocol) &&
                  (itr->_profileUsageType == btConflictsListItem._profileUsageType) &&
                  (BM_CONFLICT_STATE_SOLVED > itr->_conflictState))
            {
               itr->_conflictState = BM_CONFLICT_STATE_SOLVED;
            }//End of if((itr->_conflictType == btConflictsListItem._conflictType) &&...)
            //!Update the conflict state to Unchanged for other conflicts
            else
            {
               itr->_conflictState = BM_CONFLICT_STATE_UNCHANGED;
            }//End of else
         }//End of for(ConflictInfoListItr itr = btCurrConflictsList._conflictInfoList.begin()...)

         //!Update the conflicts list to registered clients
         VARTRACE(btCurrConflictsList);
         _userDecisionRequiredPropHdl.set(btCurrConflictsList);
      }//End of if((BM_CONFLICT_TRIGGER_NONE != btCurrConflictsList._conflictTrigger) ...)

      return retVal;
   }

   Result BmCoreMainController::clearBtConflictsList(IN const ConflictTrigger conflictTrigger)
   {
      ETG_TRACE_USR1(("clearBtConflictsList Entered with conflictTrigger = %d", ETG_CENUM(ConflictTrigger, conflictTrigger)));

      Result retVal = CC_ERR_INT_NO_ERROR;

      IssueInfoList btCurrConflictsList;
      _userDecisionRequiredPropHdl.get(btCurrConflictsList);
      ETG_TRACE_USR4(("clearBtConflictsList - Current Conflicts List"));
      VARTRACE(btCurrConflictsList);

      if((btCurrConflictsList._conflictTrigger == conflictTrigger)
            && (false == btCurrConflictsList._conflictInfoList.empty()))
      {
         btCurrConflictsList._conflictTrigger = BM_CONFLICT_TRIGGER_NONE;
         btCurrConflictsList._conflictInfoList.clear();

         _userDecisionRequiredPropHdl.set(btCurrConflictsList);
      }//End of if((btCurrConflictsList._conflictTrigger == conflictTrigger)...)
      else
      {
         ETG_TRACE_ERR(("clearBtConflictsList - Invalid Conflict Trigger or the conflict list is empty"));
         retVal = CC_ERR_INT_GENERAL_ERROR;
      }//End of else

      return retVal;
   }

   Result BmCoreMainController::initResetToDefaultStatus(void)
   {
      ETG_TRACE_USR1(("initResetToDefaultStatus: initializing ResetToDefaultStatus"));

      ResetToDefaultStatus resetToDefaultStatus;
      resetToDefaultStatus._resetState = BM_RESET_STATE_IDLE;
      resetToDefaultStatus._resetResult = BM_RESET_RESULT_OK;

      _resetToDefaultStatusPropHdl.set(resetToDefaultStatus, false);
      (void) this->updateFunctionalityRestrictionInfo(BM_RESTRICTION_BIT_RESET_TO_DEFAULT_IN_PROGRESS, false);

      return CC_ERR_INT_NO_ERROR;
   }

   BmResult BmCoreMainController::getResetToDefaultStatus(OUT ResetToDefaultStatus& resetToDefaultStatus,
         IN const BmCoreIfMsgOrigin origin)
   {
      ETG_TRACE_USR1(("getResetToDefaultStatus: origin = %d", ETG_CENUM(BmCoreIfMsgOrigin, origin)));

      if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == origin)
            && (false == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)))
      {
         ETG_TRACE_ERR(("getResetToDefaultStatus: request is currently not allowed to be processed"));
         return BM_RESULT_ERR_NOT_ALLOWED;
      }

      _resetToDefaultStatusPropHdl.get(resetToDefaultStatus);

      VARTRACE(resetToDefaultStatus);

      return BM_RESULT_OK;
   }

   void BmCoreMainController::updateResetToDefaultStatus(IN const ResetToDefaultState resetToDefaultState,
         IN const ResetToDefaultResult resetToDefaultResult)
   {
      ETG_TRACE_USR1(("updateResetToDefaultStatus: resetToDefaultState = %d, resetToDefaultResult = %d",
            ETG_CENUM(ResetToDefaultState, resetToDefaultState), ETG_CENUM(ResetToDefaultResult, resetToDefaultResult)));

      ResetToDefaultStatus resetToDefaultStatus(resetToDefaultState,
            (BM_RESET_STATE_IDLE == resetToDefaultState) ? resetToDefaultResult : BM_RESET_RESULT_NOT_APPLICABLE);
      _resetToDefaultStatusPropHdl.set(resetToDefaultStatus);
   }

   void BmCoreMainController::onLocalSpmStateChanged(IN const SpmState spmState)
   {
      ETG_TRACE_USR1(("onLocalSpmStateChanged: spmState = %d", ETG_CENUM(SpmState, spmState)));

      Result result(CC_ERR_INT_NO_ERROR);

      this->checkAndUpdateBmServiceAvailability();

      if (SPM_STATE_OFF == spmState)
      {
         (void) this->updateFunctionalityRestrictionInfo(BM_RESTRICTION_BIT_LOCAL_SPM_STATE_OFF, true);

//         (void) this->stopAutoConnection();
         ETG_TRACE_USR4(("onLocalSpmStateChanged: sending event STOP_AUTO_CONNECTION to BmControllerOnOffSm"));
         result = LocalSpm::getBmController().SendEventByName("STOP_AUTO_CONNECTION", (char *) 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("onLocalSpmStateChanged: could not send event STOP_AUTO_CONNECTION (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else if (SPM_STATE_UNDERVOLTAGE == spmState)
      {
         (void) this->updateFunctionalityRestrictionInfo(BM_RESTRICTION_BIT_LOCAL_SPM_STATE_UNDERVOLTAGE, true);

//         (void) this->stopAutoConnection();
         ETG_TRACE_USR4(("onLocalSpmStateChanged: sending event STOP_AUTO_CONNECTION to BmControllerOnOffSm"));
         result = LocalSpm::getBmController().SendEventByName("STOP_AUTO_CONNECTION", (char *) 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("onLocalSpmStateChanged: could not send event STOP_AUTO_CONNECTION (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         (void) this->updateFunctionalityRestrictionInfo(BM_RESTRICTION_BIT_LOCAL_SPM_STATE_OFF, false);
         (void) this->updateFunctionalityRestrictionInfo(BM_RESTRICTION_BIT_LOCAL_SPM_STATE_UNDERVOLTAGE, false);

         //      (void) this->startAutoConnection(BM_AUTOCONNECTION_START_MODE_BOOK_AND_TRY_ONCE);

         char eventParams[20];
         result = LocalSpm::getBmController().ParameterSTART_AUTO_CONNECTION(eventParams, sizeof(eventParams), BM_AUTOCONNECTION_START_MODE_BOOK_AND_TRY_ONCE);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("onLocalSpmStateChanged: sending event START_AUTO_CONNECTION to BmControllerOnOffSm"));
            result = LocalSpm::getBmController().SendEventByName("START_AUTO_CONNECTION", eventParams);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("onLocalSpmStateChanged: could not send event START_AUTO_CONNECTION (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("onLocalSpmStateChanged: could not marshal event parameters for event START_AUTO_CONNECTION (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
   }

   Result BmCoreMainController::blockRemoteConnections(IN const DeviceId deviceId)
   {
      ETG_TRACE_USR1(("blockRemoteConnections: deviceId = %d", deviceId));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceHandle(0u);

      if (0u != deviceId)
      {
         result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("blockRemoteConnections: could not get device handle for given device ID from DB (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      if (CC_ERR_INT_NO_ERROR == result)
      {
         result = _remoteConnectableController.stopBlockRemoteConnectionsTimer(deviceHandle);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            BmResult bmResult = this->setDeviceRemoteConnectableInt(deviceId, false);

            if (BM_RESULT_OK != bmResult)
            {
               ETG_TRACE_ERR(("blockRemoteConnections: could not set device non remote connectable (bmResult = %d)",
                     ETG_CENUM(BmResult, bmResult)));

               result = CC_ERR_INT_GENERAL_ERROR;
            }
         }
         else
         {
            ETG_TRACE_ERR(("blockRemoteConnections: could not stop block remote connections timer (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return result;
   }

   Result BmCoreMainController::blockRemoteConnectionsTimeLimited(IN const DeviceId deviceId)
   {
      ETG_TRACE_USR1(("blockRemoteConnectionsTimeLimited: deviceId = %d", deviceId));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceHandle(0u);

      if (0u != deviceId)
      {
         result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("blockRemoteConnectionsTimeLimited: could not get device handle for given device ID from DB (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      if (CC_ERR_INT_NO_ERROR == result)
      {
         result = _remoteConnectableController.startBlockRemoteConnectionsTimer(deviceHandle);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("blockRemoteConnectionsTimeLimited: could not start block remote connections timer (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return result;
   }

   Result BmCoreMainController::unblockRemoteConnections(IN const DeviceId deviceId)
   {
      ETG_TRACE_USR1(("unblockRemoteConnections: deviceId = %d", deviceId));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceHandle(0u);

      if (0u != deviceId)
      {
         result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, deviceId);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("unblockRemoteConnections: could not get device handle for given device ID from DB (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      if (CC_ERR_INT_NO_ERROR == result)
      {
         result = _remoteConnectableController.stopBlockRemoteConnectionsTimer(deviceHandle);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("unblockRemoteConnections: could not stop block remote connections timer (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }

         RemoteConnectable remoteConnectable = (TARGET_SWITCH_STATE_SWITCHED_ON == LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode) ? true : false;

         BmResult bmResult = this->setDeviceRemoteConnectableInt(deviceId, remoteConnectable);

         if (BM_RESULT_OK != bmResult)
         {
            ETG_TRACE_ERR(("unblockRemoteConnections: could not set device remote connectable (bmResult = %d)",
                  ETG_CENUM(BmResult, bmResult)));

            result = CC_ERR_INT_GENERAL_ERROR;
         }
      }

      return result;
   }

   void BmCoreMainController::handlePropertyChange_BtStatus(IN const SwitchStatus btStatus)
   {
      ETG_TRACE_USR1(("handlePropertyChange_BtStatus"));

      Result result(CC_ERR_INT_NO_ERROR);

      if (SWITCH_STATE_SWITCHED_ON == btStatus._switchState)
      {
         this->updateFunctionalityRestrictionInfo(BM_RESTRICTION_BIT_BT_STATUS_OFF, false);

         ETG_TRACE_USR4(("handlePropertyChange_BtStatus: sending event BT_SWITCHED_ON to BmControllerOnOffSm"));
         result = LocalSpm::getBmController().SendEventByName("BT_SWITCHED_ON", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handlePropertyChange_BtStatus: could not send event BT_SWITCHED_ON (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         this->updateFunctionalityRestrictionInfo(BM_RESTRICTION_BIT_BT_STATUS_OFF, true);

         if (SWITCH_STATE_SWITCHED_OFF == btStatus._switchState)
         {
            ETG_TRACE_USR4(("handlePropertyChange_BtStatus: sending event BT_SWITCHED_OFF to BmControllerOnOffSm"));
            result = LocalSpm::getBmController().SendEventByName("BT_SWITCHED_OFF", 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handlePropertyChange_BtStatus: could not send event BT_SWITCHED_OFF (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         // we have to stop the auto connection controller SM in case of switching BT off TODO: to solve ticket NCG3D-18145; to be checked with Stefan
         else if (SWITCH_STATE_SWITCHING_OFF == btStatus._switchState)
         {
            ETG_TRACE_USR4(("handlePropertyChange_BtStatus: sending event STOP_AUTO_CONNECTION to AutoConnectionControllerSm"));
            result = LocalSpm::getAutoConnectionController().SendEventByName("STOP_AUTO_CONNECTION", 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handlePropertyChange_BtStatus: could not send event STOP_AUTO_CONNECTION (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }

      char eventParams[20];

      result = LocalSpm::getResetToDefaultController().ParameterBT_STATUS_CHANGED(OUT eventParams, IN sizeof(eventParams),
            IN btStatus._switchState);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR4(("handlePropertyChange_BtStatus: sending event BT_STATUS_CHANGED to ResetToDefaultControllerSm"));
         result = LocalSpm::getResetToDefaultController().SendEventByName("BT_STATUS_CHANGED", eventParams);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handlePropertyChange_BtStatus: could not send event BT_STATUS_CHANGED (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handlePropertyChange_BtStatus: could not marshal event parameters for event BT_STATUS_CHANGED (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      LocalSpm::getBtLimitationController().onBtStatusChanged(btStatus);
   }

   void BmCoreMainController::handlePropertyChange_LocalPairableMode(IN const SwitchStatus localPairableMode)
   {
      ETG_TRACE_USR1(("handlePropertyChange_LocalPairableMode"));

      LocalSpm::getBtLimitationController().onLocalPairableModeChanged(localPairableMode);

      Result result(CC_ERR_INT_NO_ERROR);

      if (SWITCH_STATE_SWITCHING_ON == localPairableMode._switchState)
      {
         if (TARGET_SWITCH_STATE_SWITCHED_OFF == LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode)
         {
            ETG_TRACE_USR4(("handlePropertyChange_LocalPairableMode: sending event START_REMOTE_PAIRING_SESSION to PairingControllerSm"));

            result = LocalSpm::getPairingController().SendEventByName("START_REMOTE_PAIRING_SESSION", 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handlePropertyChange_LocalPairableMode: could not send event START_REMOTE_PAIRING_SESSION (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }

            if (CC_ERR_INT_NO_ERROR != result)
            {
               (void) LocalSpm::getBmCoreMainController().switchLocalPairableModeInt(LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode, "");
            }
         }
      }
      else if (SWITCH_STATE_SWITCHED_OFF == localPairableMode._switchState)
      {
         char eventParams[20];

         result = LocalSpm::getPairingController().ParameterNOT_PAIRABLE(OUT eventParams,
               IN sizeof(eventParams), IN localPairableMode._switchedOffReason);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("handlePropertyChange_LocalPairableMode: sending event NOT_PAIRABLE to PairingControllerSm"));
            result = LocalSpm::getPairingController().SendEventByName("NOT_PAIRABLE", eventParams);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handlePropertyChange_LocalPairableMode: could not send event NOT_PAIRABLE (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("handlePropertyChange_LocalPairableMode: could not marshal event parameters for event NOT_PAIRABLE (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      char eventParams[20];

      result = LocalSpm::getResetToDefaultController().ParameterPAIRABLE_MODE_CHANGED(OUT eventParams,
            IN sizeof(eventParams), IN localPairableMode._switchState);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR4(("handlePropertyChange_LocalPairableMode: sending event PAIRABLE_MODE_CHANGED to ResetToDefaultControllerSm"));
         result = LocalSpm::getResetToDefaultController().SendEventByName("PAIRABLE_MODE_CHANGED", eventParams);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handlePropertyChange_LocalPairableMode: could not send event PAIRABLE_MODE_CHANGED (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handlePropertyChange_LocalPairableMode: could not marshal event parameters for event PAIRABLE_MODE_CHANGED (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void BmCoreMainController::handlePropertyChange_LocalConnectableMode(IN const SwitchStatus localConnectableMode)
   {
      ETG_TRACE_USR1(("handlePropertyChange_LocalConnectableMode"));

      LocalSpm::getBtLimitationController().onLocalConnectableModeChanged(localConnectableMode);

      char eventParams[20];

      Result result = LocalSpm::getResetToDefaultController().ParameterCONNECTABLE_MODE_CHANGED(OUT eventParams, IN sizeof(eventParams),
            IN localConnectableMode._switchState);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR4(("handlePropertyChange_LocalConnectableMode: sending event CONNECTABLE_MODE_CHANGED to ResetToDefaultControllerSm"));
         result = LocalSpm::getResetToDefaultController().SendEventByName("CONNECTABLE_MODE_CHANGED", eventParams);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handlePropertyChange_LocalConnectableMode: could not send event CONNECTABLE_MODE_CHANGED (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handlePropertyChange_LocalConnectableMode: could not marshal event parameters for event CONNECTABLE_MODE_CHANGED (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void BmCoreMainController::handlePropertyChange_DiscoveryStatus(IN const SwitchStatus& discoveryStatus)
   {
      ETG_TRACE_USR1(("handlePropertyChange_DiscoveryStatus"));

      if (SWITCH_STATE_SWITCHED_OFF == discoveryStatus._switchState)
      {
         //!Send event to BmControllerOnOff once discovery process is switched OFF
         ETG_TRACE_USR4(("handlePropertyChange_DiscoveryStatus: sending event DEVICE_DISCOVERY_STOPPED to BmControllerOnOffSm"));
         Result result = LocalSpm::getBmController().SendEventByName("DEVICE_DISCOVERY_STOPPED", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
          ETG_TRACE_ERR(("handlePropertyChange_DiscoveryStatus: could not send event DEVICE_DISCOVERY_STOPPED (error = %d)",
                ETG_CENUM(CcErrorInternal, result)));
         }

         // clear discovered devices list
         ETG_TRACE_USR1(("handlePropertyChange_DiscoveryStatus: clearing DiscoveredDeviceList"));

         DiscoveredDeviceList discoveredDeviceList;
         _discoveredDeviceListPropHdl.set(discoveredDeviceList);
      }
   }

   void BmCoreMainController::handlePropertyChange_AutoConnectionStatus(IN const AutoConnectionStatus& autoConnectionStatus)
   {
      ETG_TRACE_USR1(("handlePropertyChange_AutoConnectionStatus"));

      (void) autoConnectionStatus;
   }

   void BmCoreMainController::handlePropertyChange_PairedDeviceList(IN const PairedDeviceList& pairedDeviceList)
   {
      ETG_TRACE_USR1(("handlePropertyChange_PairedDeviceList"));

      if (true == pairedDeviceList._deviceBaseInfoList.empty())
      {
         ETG_TRACE_USR4(("handlePropertyChange_PairedDeviceList: sending event NO_PAIRED_DEVICES to ResetToDefaultControllerSm"));
         Result result = LocalSpm::getResetToDefaultController().SendEventByName("NO_PAIRED_DEVICES", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handlePropertyChange_PairedDeviceList: could not send event NO_PAIRED_DEVICES (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
   }

   void BmCoreMainController::handlePropertyChange_DeviceConnectionStatusList(IN const DeviceConnectionStatusList& deviceConnectionStatusList,
         IN const DeviceConnectionStatusList& previousDeviceConnectionStatusList)
   {
      ETG_TRACE_USR1(("handlePropertyChange_DeviceConnectionStatusList"));

      if (true == deviceConnectionStatusList._deviceConnectionInfoList.empty())
      {
         ETG_TRACE_USR4(("handlePropertyChange_DeviceConnectionStatusList: sending event ALL_DEVICES_DISCONNECTED to BmControllerOnOffSm"));
         Result result = LocalSpm::getBmController().SendEventByName("ALL_DEVICES_DISCONNECTED", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handlePropertyChange_DeviceConnectionStatusList: could not send event ALL_DEVICES_DISCONNECTED (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      DeviceIdList previousConnectedDevices;

      for (size_t idx = 0u; idx < previousDeviceConnectionStatusList._deviceConnectionInfoList.size(); idx++)
      {
         if (BM_CONNECTION_STATUS_CONNECTED == previousDeviceConnectionStatusList._deviceConnectionInfoList[idx]._connectionStatus)
         {
            previousConnectedDevices.push_back(previousDeviceConnectionStatusList._deviceConnectionInfoList[idx]._deviceHandle);
         }
      }

      DeviceIdList connectedDevices;

      for (size_t idx = 0u; idx < deviceConnectionStatusList._deviceConnectionInfoList.size(); idx++)
      {
         if (BM_CONNECTION_STATUS_CONNECTED == deviceConnectionStatusList._deviceConnectionInfoList[idx]._connectionStatus)
         {
            connectedDevices.push_back(deviceConnectionStatusList._deviceConnectionInfoList[idx]._deviceHandle);
         }
      }

      ETG_TRACE_USR1(("handlePropertyChange_DeviceConnectionStatusList: connected devices:"));
      VARTRACE(connectedDevices);

      ETG_TRACE_USR1(("handlePropertyChange_DeviceConnectionStatusList: previously connected devices:"));
      VARTRACE(previousConnectedDevices);

      // if periodical link quality requesting is used and no devices are connected anymore,
      // cancel the link quality request timer
      if ((0u != (LocalSpm::getDataProvider().getBmCoreConfiguration()._linkQualityRequestTimeOut))
            && (true == connectedDevices.empty()))
      {
         ETG_TRACE_USR1(("handlePropertyChange_DeviceConnectionStatusList: no connected devices - canceling link quality request timer"));

         LocalSpm::getBmController().cancelLinkQualityRequestTimer();
      }

      // determine the connected devices for which link quality has to be requested
      DeviceIdList connectedDevicesLinkQualityHasToBeRequested;

      for (size_t idx = 0u; idx < connectedDevices.size(); idx++)
      {
         if (previousConnectedDevices.end() == std::find(previousConnectedDevices.begin(),
               previousConnectedDevices.end(), connectedDevices[idx]))
         {
            connectedDevicesLinkQualityHasToBeRequested.push_back(connectedDevices[idx]);
         }
      }

      ETG_TRACE_USR4(("handlePropertyChange_DeviceConnectionStatusList: connected devices for which link quality has to be requested:"));
      VARTRACE(connectedDevicesLinkQualityHasToBeRequested);

      // determine devices which are not connected anymore
      DeviceIdList devicesNotConnectedAnymore;

      for (size_t idx = 0u; idx < previousConnectedDevices.size(); idx++)
      {
         if (connectedDevices.end() == std::find(connectedDevices.begin(), connectedDevices.end(),
               previousConnectedDevices[idx]))
         {
            devicesNotConnectedAnymore.push_back(previousConnectedDevices[idx]);
         }
      }

      ETG_TRACE_USR4(("handlePropertyChange_DeviceConnectionStatusList: devices which are not connected anymore:"));
      VARTRACE(devicesNotConnectedAnymore);

      // delete devices which are not connected anymore from intermediate link quality property
      for (size_t idx = 0u; idx < devicesNotConnectedAnymore.size(); idx++)
      {
         _intermediateLinkQuality._linkQualityInfoList.erase(devicesNotConnectedAnymore[idx]);

         DeviceIdList::iterator it = std::find(_connectedDevicesWaitingForLinkQuality.begin(),
               _connectedDevicesWaitingForLinkQuality.end(), devicesNotConnectedAnymore[idx]);

         if (_connectedDevicesWaitingForLinkQuality.end() != it)
         {
            _connectedDevicesWaitingForLinkQuality.erase(it);
         }
      }

      if (false == devicesNotConnectedAnymore.empty())
      {
         // at least one device has been removed from intermediate link quality property,
         // so update the link quality property

         ETG_TRACE_USR4(("handlePropertyChange_DeviceConnectionStatusList: at least one device has been removed from intermediate link quality property, update link quality property"));

         _linkQualityPropHdl.set(_intermediateLinkQuality);
      }

      // request link quality
      for (size_t idx = 0u; idx < connectedDevicesLinkQualityHasToBeRequested.size(); idx++)
      {
         BdAddress bdAddress("");
         Result result = LocalSpm::getDbManager().getBdAddressByDeviceHandle(bdAddress, connectedDevicesLinkQualityHasToBeRequested[idx]);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("handlePropertyChange_DeviceConnectionStatusList: requesting link quality at BtStackIf for device with BD address = \"%50s\"",
                  bdAddress.c_str()));

            this->getBtStackIfConnectionRequestIfWrapper().requestLinkQuality(bdAddress);

            // remember the new connected devices the link quality has been successfully requested for
            _connectedDevicesWaitingForLinkQuality.push_back(connectedDevicesLinkQualityHasToBeRequested[idx]);
         }
         else
         {
            ETG_TRACE_ERR(("handlePropertyChange_DeviceConnectionStatusList: could not get BD address for device handle = %d from DB (error = %d)",
                  connectedDevicesLinkQualityHasToBeRequested[idx], ETG_CENUM(CcErrorInternal, result)));
         }
      }

      // if periodical link quality requesting is used and there is a new connected device,
      // start the link quality request timer
      if ((0u != LocalSpm::getDataProvider().getBmCoreConfiguration()._linkQualityRequestTimeOut)
            && ((true == previousConnectedDevices.empty()))
            && (false == connectedDevicesLinkQualityHasToBeRequested.empty()))
      {
         LocalSpm::getBmController().startLinkQualityRequestTimer();
      }

      ETG_TRACE_USR4(("handlePropertyChange_DeviceConnectionStatusList: updated waiting list and intermediate link qualities:"));

      VARTRACE(_connectedDevicesWaitingForLinkQuality);
      VARTRACE(_intermediateLinkQuality);
   }

   void BmCoreMainController::handlePropertyChange_BlockStatus(IN const BlockStatus& blockStatus)
   {
      ETG_TRACE_USR1(("handlePropertyChange_BlockStatus"));

      Result result(CC_ERR_INT_NO_ERROR);

      if (BM_OVERALL_BLOCK_STATE_BLOCKED == blockStatus._overallDeviceBlockStatus._blockState)
      {
         (void) this->updateFunctionalityRestrictionInfo(BM_RESTRICTION_BIT_OVERALL_DEVICE_BLOCK_STATE_BLOCKED,
               true);

         ETG_TRACE_USR4(("handlePropertyChange_BlockStatus: sending event STOP_AUTO_CONNECTION to BmControllerOnOffSm"));

         result = LocalSpm::getBmController().SendEventByName("STOP_AUTO_CONNECTION", (char *) 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handlePropertyChange_BlockStatus: could not send event STOP_AUTO_CONNECTION (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         (void) this->updateFunctionalityRestrictionInfo(BM_RESTRICTION_BIT_OVERALL_DEVICE_BLOCK_STATE_BLOCKED,
               false);

         BlockStatus previousBlockStatus;
         _blockStatusPropHdl.getPrevious(previousBlockStatus);

         if (BM_OVERALL_BLOCK_STATE_BLOCKED == previousBlockStatus._overallDeviceBlockStatus._blockState)
         {
            char eventParams[20];
            result = LocalSpm::getBmController().ParameterSTART_AUTO_CONNECTION(eventParams, sizeof(eventParams),
                  BM_AUTOCONNECTION_START_MODE_TRY_ONCE_ONLY_IF_BOOKED_BEFORE);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               ETG_TRACE_USR4(("handlePropertyChange_BlockStatus: sending event START_AUTO_CONNECTION to BmControllerOnOffSm"));

               result = LocalSpm::getBmController().SendEventByName("START_AUTO_CONNECTION", eventParams);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("handlePropertyChange_BlockStatus: could not send event START_AUTO_CONNECTION (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               ETG_TRACE_ERR(("handlePropertyChange_BlockStatus: could not marshal event parameters for event START_AUTO_CONNECTION (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }
   }

   void BmCoreMainController::handlePropertyChange_BtLimitationMode(IN const BtLimitationMode& btLimitationMode)
   {
      ETG_TRACE_USR1(("handlePropertyChange_BtLimitationMode"));

      (void)btLimitationMode;
   }

   void BmCoreMainController::handlePropertyChange_MultiHFPSupportStatus(IN const SwitchStatus multiHFPSupportStatus)
   {
      ETG_TRACE_USR1(("handlePropertyChange_MultiHFPSupportStatus"));

      char eventParams[20];

      Result result = LocalSpm::getResetToDefaultController().ParameterMULTIHFPSUPPORT_STATUS_CHANGED(OUT eventParams, IN sizeof(eventParams),
            IN multiHFPSupportStatus._switchState);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR4(("handlePropertyChange_MultiHFPSupportStatus: sending event MULTIHFPSUPPORT_STATUS_CHANGED to ResetToDefaultControllerSm"));
         result = LocalSpm::getResetToDefaultController().SendEventByName("MULTIHFPSUPPORT_STATUS_CHANGED", eventParams);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handlePropertyChange_MultiHFPSupportStatus: could not send event MULTI_HFP_SUPPORT_CHANGED (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handlePropertyChange_MultiHFPSupportStatus: could not marshal event parameters for event MULTIHFPSUPPORT_STATUS_CHANGED (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void BmCoreMainController::handlePropertyChange_UserDecisionReqdInfo(IN const IssueInfoList& issueInfoList)
   {
      ETG_TRACE_USR1(("handlePropertyChange_UserDecisionReqdInfo"));
      (void)issueInfoList;
   }

   void BmCoreMainController::handlePropertyChange_ResetToDefaultStatus(IN const ResetToDefaultStatus& resetToDefaultStatus)
   {
      ETG_TRACE_USR1(("handlePropertyChange_ResetToDefaultStatus"));

      if (BM_RESET_STATE_IDLE == resetToDefaultStatus._resetState)
      {
         (void) this->updateFunctionalityRestrictionInfo(BM_RESTRICTION_BIT_RESET_TO_DEFAULT_IN_PROGRESS, false);
      }
      else
      {
         (void) this->updateFunctionalityRestrictionInfo(BM_RESTRICTION_BIT_RESET_TO_DEFAULT_IN_PROGRESS, true);
      }
   }

   Result BmCoreMainController::stopThread()
   {
      ETG_TRACE_USR1(("stopThread: terminating thread with ID = %d", _threadId));

      BmCoreIfMessage_TerminateMessageHandlerThread* bmCoreIfMessage = getNewBmCoreIfMessage_TerminateMessageHandlerThread();

      if (0 != bmCoreIfMessage)
      {
         ETG_TRACE_USR1(("stopThread: flushing message queue"));

         _bmCoreIfMessageQueue.Flush();

         this->pushBmCoreIfMessage(bmCoreIfMessage);
      }

      return CC_ERR_INT_NO_ERROR;
   }

   void BmCoreMainController::addConnectedDeviceToWaitingForLinkQuality(IN const DeviceId deviceHandle)
   {
      _connectedDevicesWaitingForLinkQuality.push_back(deviceHandle);
   }

   void BmCoreMainController::removeConnectedDeviceFromWaitingForLinkQuality(IN const DeviceId deviceHandle)
   {
      DeviceIdList::iterator it = find(_connectedDevicesWaitingForLinkQuality.begin(),
            _connectedDevicesWaitingForLinkQuality.end(), deviceHandle);

      if (_connectedDevicesWaitingForLinkQuality.end() != it)
      {
         _connectedDevicesWaitingForLinkQuality.erase(it);
      }
   }

   void BmCoreMainController::addLinkQualityToIntermediateLinkQuality(IN const DeviceId deviceHandle,
         IN const LinkQualityType linkQuality, IN const Rssi rssi)
   {
      _intermediateLinkQuality._linkQualityInfoList[deviceHandle]._linkQuality = linkQuality;
      _intermediateLinkQuality._linkQualityInfoList[deviceHandle]._rssi = rssi;
   }

   void BmCoreMainController::checkAndUpdateBmServiceAvailability()
   {
      ETG_TRACE_USR1(("checkAndUpdateBmServiceAvailability: entered"));

      BmServiceAvailability bmServiceAvailability;
      (void) this->getBmServiceAvailablity(bmServiceAvailability);

      SpmState spmState = LocalSpm::getInstance().getSpmState();

      // Set serviceAvailability as AVAILABLE if the SPMState is NORMAL and BtSystemState is NORMAL or NORMAL_SWITCH_OFF_BT
      // Set serviceAvailability as AVAILABLE_RESTRICTED if the SPMState is NORMAL and BtSystemState is BLOCK or OFF
      if(SPM_STATE_NORMAL == spmState)
      {
         BtSystemState btSystemState;
         (void) this->getBtSystemState(btSystemState);

         if((BM_BT_SYSTEM_STATE_NORMAL == btSystemState._bluetoothSystemState) ||
               (BM_BT_SYSTEM_STATE_NORMAL_SWITCH_OFF_BT == btSystemState._bluetoothSystemState))
         {
            bmServiceAvailability._serviceAvailability = BM_SERVICE_AVAILABLE;
         }
         else if((BM_BT_SYSTEM_STATE_BLOCK == btSystemState._bluetoothSystemState) ||
               (BM_BT_SYSTEM_STATE_OFF == btSystemState._bluetoothSystemState))
         {
            bmServiceAvailability._serviceAvailability = BM_SERVICE_AVAILABLE_RESTRICTED;
         }
      }
      // Set serviceAvailability as UNAVAILABLE if the SPMState is OFF/UNDERVOLTATE
      else
      {
         bmServiceAvailability._serviceAvailability = BM_SERVICE_UNAVAILABLE;
      }

      ETG_TRACE_USR1(("checkAndUpdateBmServiceAvailability: _serviceAvailability = %d",
            ETG_CENUM(ServiceAvailability, bmServiceAvailability._serviceAvailability)));

      _bmServiceAvailabilityPropHdl.set(bmServiceAvailability);
   }

   Result BmCoreMainController::updateFunctionalityRestrictionInfo(IN const RestrictionBit restrictionBit,
         IN const bool toBeSet)
   {
      ETG_TRACE_USR1(("updateFunctionalityRestrictionInfo: restrictionBit = %d, action = %s",
            ETG_CENUM(RestrictionBit, restrictionBit), toBeSet ? "set" : "clear"));

      Locker locker(&_currentFunctionalityRestrictionInfoLock);

      Result result(CC_ERR_INT_NO_ERROR);

      if ((BM_RESTRICTION_BIT_LOCAL_SPM_STATE_OFF <= restrictionBit) && (BM_RESTRICTION_BIT_MAX_EXCEEDED > restrictionBit))
      {
         _currentFunctionalityRestrictionInfo.set(static_cast<size_t>(restrictionBit), toBeSet);
      }
      else
      {
         ETG_TRACE_ERR(("updateFunctionalityRestrictionInfo: invalid parameter value for restrictionBit"));
      }

/*
      ETG_TRACE_USR1(("updateFunctionalityRestrictionInfo(restrictionBit = %d, toBeSet = %d): _currentFunctionalityRestrictionInfo = %50s",
            ETG_CENUM(RestrictionBit, restrictionBit), toBeSet, _currentFunctionalityRestrictionInfo.to_string().c_str()));
*/

      ETG_TRACE_USR1(("updateFunctionalityRestrictionInfo: active: %200s",
            getFunctionalityRestrictionInfoString(_currentFunctionalityRestrictionInfo).c_str()));

      return result;
   }

   bool BmCoreMainController::isFunctionalityPermitted(IN const FunctionalityRestrictionInfo restrictionInfo)
   {
//      ETG_TRACE_USR1(("isFunctionalityPermitted"));

      Locker locker(&_currentFunctionalityRestrictionInfoLock);

/*
      ETG_TRACE_USR1(("isFunctionalityPermitted: _currentFunctionalityRestrictionInfo = %50s",
            _currentFunctionalityRestrictionInfo.to_string().c_str()));
*/
      ETG_TRACE_USR1(("isFunctionalityPermitted: given : %200s",
            getFunctionalityRestrictionInfoString(restrictionInfo).c_str()));

      ETG_TRACE_USR1(("isFunctionalityPermitted: active: %200s",
            getFunctionalityRestrictionInfoString(_currentFunctionalityRestrictionInfo).c_str()));

      return ((_currentFunctionalityRestrictionInfo & restrictionInfo) == 0u);
   }

   std::string BmCoreMainController::getFunctionalityRestrictionInfoString(IN const FunctionalityRestrictionInfo restrictionInfo)
   {
      std::map<RestrictionBit, std::string>::iterator it;
      std::string restrictionInfoString("");

      for (size_t idx = 0u; idx < RestrictionBitfieldSize; ++idx)
      {
         it = _restrictionBitStringMap.find(static_cast<RestrictionBit>(idx));

         if (_restrictionBitStringMap.end() != it)
         {
            restrictionInfoString += (it->second + (restrictionInfo[idx] ? ": 1 | " : ": 0 | "));
         }
      }

      return restrictionInfoString;
   }

   Result BmCoreMainController:: getProfileConnectionPageTimeout( OUT PageTimeout& pausePagingTimeoutMilliSeconds, OUT PageTimeout& pageTimeout)
   {
      Result result(CC_ERR_INT_NO_ERROR);

      ETG_TRACE_USR1(("getProfileConnectionPageTimeout: _wifiHealthinessIndicatorInformation.size() = %u",
            LocalSpm::getDataProvider().getBmCoreConfiguration()._wifiHealthinessIndicatorInformation.size()));

      if(0u < LocalSpm::getDataProvider().getBmCoreConfiguration()._wifiHealthinessIndicatorInformation.size())
      {
         for (WifiHealthinessIndicatorInformation::const_iterator it = LocalSpm::getDataProvider().getBmCoreConfiguration()._wifiHealthinessIndicatorInformation.begin();
               it != LocalSpm::getDataProvider().getBmCoreConfiguration()._wifiHealthinessIndicatorInformation.end(); ++it)
         {
            ETG_TRACE_USR4(("getProfileConnectionPageTimeout: _wblHealthinessIndicator = %u, wifiHealthinessIndicatorThresholdValue = %u _connectionPageTimeoutMilliSeconds = %u",
                  _wblHealthinessIndicator, it->_wifiHealthinessIndicatorThresholdValue, it->_connectionPageTimeoutMilliSeconds));

            pageTimeout = it->_connectionPageTimeoutMilliSeconds;
            ETG_TRACE_USR4(("getProfileConnectionPageTimeout: _wblHealthinessIndicator set PageTimeout = %u", pageTimeout));

            // check the received wbl HealthinessIndicator value is greater than the threshold value. if the value is greater then send CONNECT_DEVICE event directly
            // else start the timer once the timer is expired then send the CONNECT_DEVICE event.

            // 100>= HealthIndicator >90: set page timeout to 10s
            // 90>= HealthIndicator >80: set page timeout to 7.5s
            // 80>= HealthIndicator >70: set page timeout to 5s
            // 70>= HealthIndicator: set page timeout to 2.5s
            if (_wblHealthinessIndicator > it->_wifiHealthinessIndicatorThresholdValue)
            {
               break;
            }
         }
      }
      else
      {
         pageTimeout = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultConnectionPageTimeoutMilliSeconds;
         ETG_TRACE_USR4(("getProfileConnectionPageTimeout: _wblHealthinessIndicator set PageTimeout default value = %u", pageTimeout));

      }

      pausePagingTimeoutMilliSeconds = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultConnectionPageTimeoutMilliSeconds - pageTimeout;
      ETG_TRACE_USR4(("getProfileConnectionPageTimeout: _wblHealthinessIndicator pausepagingtimeout = %u", pausePagingTimeoutMilliSeconds));

      return result;
   }

   void BmCoreMainController::create()
   {
      ETG_TRACE_USR1(("create"));

      _expectedComponentState = COMPONENT_STATE_CREATE;

//      _allDevicesDisconnectionController.create();

      createDone(0);
   }

   int BmCoreMainController::init(InitReason reason)
   {
      ETG_TRACE_USR1(("init: reason = %d", ETG_CENUM(InitReason, reason)));

      _expectedComponentState = COMPONENT_STATE_INIT;

      _resetToFactorySettingsRequested = false;

//      _allDevicesDisconnectionController.init();

      return initDone(0);
   }

   int BmCoreMainController::run()
   {
      ETG_TRACE_USR1(("run: starting thread"));

      _expectedComponentState = COMPONENT_STATE_RUNNING;

//      _allDevicesDisconnectionController.run();

      // check if thread is already running
      if (0 <= _threadId)
      {
         ETG_TRACE_USR1(("run: thread with ID = %d is already running", _threadId));
      }
      else
      {
      _threadId = LocalSpm::getThreadFactory().Do(this, 0, 0);
      }

      ETG_TRACE_USR1(("run: thread with ID = %d has been started", _threadId));

      return runDone(0);
   }

   int BmCoreMainController::stop()
   {
      ETG_TRACE_USR1(("stop"));

      _expectedComponentState = COMPONENT_STATE_STOP;

      // call stop method of every sub-component, sub-components then notifies the finished switching to requested
      // component state to main controller by calling notifyComponentState()
      // up to now there is not yet any sub-component, so notifyComponentState() is called here

//      _allDevicesDisconnectionController.stop();

      notifyComponentState(COMPONENT_ID_UNKNOWN, COMPONENT_STATE_STOP);

      return CC_ERR_INT_NO_ERROR;
   }

   int BmCoreMainController::done()
   {
      ETG_TRACE_USR1(("done"));

      _expectedComponentState = COMPONENT_STATE_DONE;

      // call stop method of every sub-component, sub-components then notifies the finished switching to requested
      // component state to main controller by calling notifyComponentState()
      // up to now there is not yet any sub-component, so notifyComponentState() is called here

//      _allDevicesDisconnectionController.done();

      notifyComponentState(COMPONENT_ID_UNKNOWN, COMPONENT_STATE_DONE);

      return CC_ERR_INT_NO_ERROR;
   }

   void BmCoreMainController::Do(int param, void* userPtr)
   {
      ETG_TRACE_USR1(("Do"));

      (void) param;
      (void) userPtr;

      LocalSpm::getThreadFactory().SetName(this->getComponentName());
      _terminateThread = false;

      size_t msgSize(0u);
      unsigned int msgCounter(0u);

      while (false == _terminateThread)
      {
         BmCoreIfMessage* bmCoreIfMessage = static_cast<BmCoreIfMessage*>(_bmCoreIfMessageQueue.WaitForMessage(&msgSize, 0u));

         if (0 != bmCoreIfMessage)
         {
            msgCounter++;
            //_bmCoreIfMessageQueue.DropMessage(msgReceivedBuffer);
            ETG_TRACE_USR1(("Do: popped BM Core IF message 0x%p, number of received BM Core IF messages increased to %u", (void *) bmCoreIfMessage, msgCounter));

            BM_GLOBAL_LOCKER_INTERNAL

            handleBmCoreIfMessage(bmCoreIfMessage);
         }
         else
         {
            ETG_TRACE_FATAL(("Do: popped BM Core IF message is 0"));

            FW_NORMAL_ASSERT_ALWAYS();
            break;
         }
      }

      ETG_TRACE_USR1(("Do: thread with ID = %d terminates", _threadId));

      _threadId = -1;

      ETG_TRACE_USR1(("Do: number of messages waiting in queue: 0x%08x%08x",
            (int)(_bmCoreIfMessageQueue.GetCurMessagesCount() >> 32),
            (int)(_bmCoreIfMessageQueue.GetCurMessagesCount()&0xffffffff)));

      ETG_TRACE_USR1(("Do: flushing message queue"));

      _bmCoreIfMessageQueue.Flush();
   }

   void BmCoreMainController::handleBmCoreIfMessage(BmCoreIfMessage* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      if (0 != bmCoreIfMessage)
      {
         bmCoreIfMessage->traceMessage();

         BmCoreIfMsgId msgId = bmCoreIfMessage->getMessageId();

         if ((BM_CORE_IF_MSG_ID_UNKNOWN < msgId) && (BM_CORE_IF_MSG_ID_LAST > msgId))
         {
            //ETG_TRACE_USR1(("handleBmCoreIfMessage(bmCoreIfMessage = 0x%p): msgId = %d", (void *) bmCoreIfMessage, ETG_CENUM(BmCoreIfMsgId, msgId)));

            switch (msgId)
            {
               case BM_CORE_IF_MSG_ID_SET_PROFILE_USER_AVAILABILITY_REQUEST:
                  (void) handleBmCoreIfMessage_SetProtocolUserAvailabilityRequest(static_cast<BmCoreIfMessage_SetProtocolUserAvailabilityRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SET_AUTO_CONNECTION_TYPE_REQUEST:
                  (void) handleBmCoreIfMessage_SetAutoConnectionTypeRequest(static_cast<BmCoreIfMessage_SetAutoConnectionTypeRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SWITCH_BT_STATUS_REQUEST:
                  (void) handleBmCoreIfMessage_SwitchBtStatusRequest(static_cast<BmCoreIfMessage_SwitchBtStatusRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SET_BT_LOCAL_FRIENDLY_NAME_REQUEST:
                  (void) handleBmCoreIfMessage_SetBtLocalFriendlyNameRequest(static_cast<BmCoreIfMessage_SetBtLocalFriendlyNameRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SWITCH_LOCAL_PAIRABLE_MODE_REQUEST:
                  (void) handleBmCoreIfMessage_SwitchLocalPairableRequest(static_cast<BmCoreIfMessage_SwitchLocalPairableModeRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SWITCH_LOCAL_CONNECTABLE_MODE_REQUEST:
                  (void) handleBmCoreIfMessage_SwitchLocalConnectableModeRequest(static_cast<BmCoreIfMessage_SwitchLocalConnectableModeRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SWITCH_DISCOVERY_STATUS_REQUEST:
                  (void) handleBmCoreIfMessage_SwitchDiscoveryStatusRequest(static_cast<BmCoreIfMessage_SwitchDiscoveryStatusRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_PAIRING_STATUS_UPDATE:
                  handleBmCoreIfMessage_PairingStatusUpdate(static_cast<BmCoreIfMessage_PairingStatusUpdate*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_START_PAIRING_REQUEST:
                  handleBmCoreIfMessage_StartPairingRequest(static_cast<BmCoreIfMessage_StartPairingRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_CANCEL_PAIRING_REQUEST:
                  handleBmCoreIfMessage_CancelPairingRequest(static_cast<BmCoreIfMessage_CancelPairingRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_ACCEPT_PAIRING_REQUEST_REQUEST:
                  handleBmCoreIfMessage_AcceptPairingRequestRequest(static_cast<BmCoreIfMessage_AcceptPairingRequestRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_REJECT_PAIRING_REQUEST_REQUEST:
                  handleBmCoreIfMessage_RejectPairingRequestRequest(static_cast<BmCoreIfMessage_RejectPairingRequestRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SET_PAIRING_PIN_REQUEST:
                  handleBmCoreIfMessage_SetPairingPinRequest(static_cast<BmCoreIfMessage_SetPairingPinRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_ADD_OOB_PAIRED_DEVICE_REQUEST:
                  handleBmCoreIfMessage_AddOobPairedDeviceRequest(static_cast<BmCoreIfMessage_AddOobPairedDeviceRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_DELETE_DEVICE_REQUEST:
                  (void) handleBmCoreIfMessage_DeleteDeviceRequest(static_cast<BmCoreIfMessage_DeleteDeviceRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_START_AUTO_CONNECTION_SEQUENCE_REQUEST:
                  (void) handleBmCoreIfMessage_StartAutoConnectionSequenceRequest(static_cast<BmCoreIfMessage_StartAutoConnectionSequenceRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_STOP_AUTO_CONNECTION_SEQUENCE_REQUEST:
                  (void) handleBmCoreIfMessage_StopAutoConnectionSequenceRequest(static_cast<BmCoreIfMessage_StopAutoConnectionSequenceRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_CONNECT_DEVICE_REQUEST:
                  (void) handleBmCoreIfMessage_ConnectDeviceRequest(static_cast<BmCoreIfMessage_ConnectDeviceRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_DISCONNECT_DEVICE_REQUEST:
                  (void) handleBmCoreIfMessage_DisconnectDeviceRequest(static_cast<BmCoreIfMessage_DisconnectDeviceRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_CONNECT_PROFILES_REQUEST:
                  (void) handleBmCoreIfMessage_ConnectProfilesRequest(static_cast<BmCoreIfMessage_ConnectProfilesRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_DISCONNECT_PROFILES_REQUEST:
                  (void) handleBmCoreIfMessage_DisconnectProfilesRequest(static_cast<BmCoreIfMessage_DisconnectProfilesRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BLOCK_DEVICE_REQUEST:
                  (void) handleBmCoreIfMessage_BlockDeviceRequest(static_cast<BmCoreIfMessage_BlockDeviceRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_UNBLOCK_DEVICE_REQUEST:
                  (void) handleBmCoreIfMessage_UnblockDeviceRequest(static_cast<BmCoreIfMessage_UnblockDeviceRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BLOCK_PROFILES_REQUEST:
                  (void) handleBmCoreIfMessage_BlockProfilesRequest(static_cast<BmCoreIfMessage_BlockProfilesRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_UNBLOCK_PROFILES_REQUEST:
                  (void) handleBmCoreIfMessage_UnblockProfilesRequest(static_cast<BmCoreIfMessage_UnblockProfilesRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_PAIRED_DEVICE_LIST_UPDATE:
                  handleBmCoreIfMessage_PairedDeviceListUpdate(static_cast<BmCoreIfMessage_PairedDeviceListUpdate*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_DEVICE_CONNECTION_STATUS_LIST_UPDATE:
                  handleBmCoreIfMessage_DeviceConnectionStatusListUpdate(static_cast<BmCoreIfMessage_DeviceConnectionStatusListUpdate*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SET_FAVORITE_REQUEST:
                  handleBmCoreIfMessage_SetFavoriteRequest(static_cast<BmCoreIfMessage_SetFavoriteRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_CLEAR_FAVORITE_REQUEST:
                  handleBmCoreIfMessage_ClearFavoriteRequest(static_cast<BmCoreIfMessage_ClearFavoriteRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SET_PRIMARY_HFP_DEVICE_REQUEST:
                  handleBmCoreIfMessage_SetPrimaryHfpDeviceRequest(static_cast<BmCoreIfMessage_SetPrimaryHfpDeviceRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_START_SERVICE_SEARCH_REQUEST:
                  handleBmCoreIfMessage_StartServiceSearchRequest(static_cast<BmCoreIfMessage_StartServiceSearchRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_STOP_SERVICE_SEARCH_REQUEST:
                  handleBmCoreIfMessage_StopServiceSearchRequest(static_cast<BmCoreIfMessage_StopServiceSearchRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_UPDATE_LINK_QUALITY_REQUEST:
                  handleBmCoreIfMessage_UpdateLinkQualityRequest(static_cast<BmCoreIfMessage_UpdateLinkQualityRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BLOCK_STATUS_UPDATE:
                  handleBmCoreIfMessage_BlockStatusUpdate(static_cast<BmCoreIfMessage_BlockStatusUpdate*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SET_BT_LIMITATION_MODE_REQUEST:
                  (void) handleBmCoreIfMessage_SetBtLimitationModeRequest(static_cast<BmCoreIfMessage_SetBtLimitationModeRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_REPLACE_BT_LIMITATION_MODE_REQUEST:
                  (void) handleBmCoreIfMessage_ReplaceBtLimitationModeRequest(static_cast<BmCoreIfMessage_ReplaceBtLimitationModeRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BT_LIMITATION_MODE_UPDATE:
                  handleBmCoreIfMessage_BtLimitationModeUpdate(static_cast<BmCoreIfMessage_BtLimitationModeUpdate*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_NOTIFY_USER_DECISION_REQUEST:
                  handleBmCoreIfMessage_NotifyUserDecisionRequest(static_cast<BmCoreIfMessage_NotifyUserDecisionRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SWITCH_TEST_MODE_REQUEST:
                  (void) handleBmCoreIfMessage_SwitchTestModeRequest(static_cast<BmCoreIfMessage_SwitchTestModeRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_RESET_TO_DEFAULT_REQUEST:
                  (void) handleBmCoreIfMessage_ResetToDefaultRequest(static_cast<BmCoreIfMessage_ResetToDefaultRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_FB_CONNECTION_INITIALIZED:
                  handleBmCoreIfMessage_BtsFbConnectionInitialized(static_cast<BmCoreIfMessage_BtsFbConnectionInitialized*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_LOCAL_VERSION_INFO:
                  handleBmCoreIfMessage_BtsLocalVersionInfo(static_cast<BmCoreIfMessage_BtsLocalVersionInfo*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_SWITCH_BLUETOOTH_ON_OFF_RESULT:
                  handleBmCoreIfMessage_BtsSwitchBluetoothOnOffResult(static_cast<BmCoreIfMessage_BtsSwitchBluetoothOnOffResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_CURRENT_BLUETOOTH_ON_OFF_STATUS:
                  handleBmCoreIfMessage_BtsCurrentBluetoothOnOffStatus(static_cast<BmCoreIfMessage_BtsCurrentBluetoothOnOffStatus*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_BLUETOOTH_HARDWARE_ON_OFF_RESULT:
                  handleBmCoreIfMessage_BtsBluetoothHardwareOnOffResult(static_cast<BmCoreIfMessage_BtsBluetoothHardwareOnOffResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_SET_LOCAL_BT_NAME_RESULT:
                  handleBmCoreIfMessage_BtsSetLocalBtNameResult(static_cast<BmCoreIfMessage_BtsSetLocalBtNameResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_CURRENT_LOCAL_BT_NAME:
                  handleBmCoreIfMessage_BtsCurrentLocalBtName(static_cast<BmCoreIfMessage_BtsCurrentLocalBtName*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_SET_LOCAL_ADAPTER_MODES_RESULT:
                  handleBmCoreIfMessage_BtsSetLocalAdapterModesResult(static_cast<BmCoreIfMessage_BtsSetLocalAdapterModesResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_CURRENT_DISCOVERABLE_MODE:
                  handleBmCoreIfMessage_BtsCurrentDiscoverableMode(static_cast<BmCoreIfMessage_BtsCurrentDiscoverableMode*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_CURRENT_CONNECTABLE_MODE:
                  handleBmCoreIfMessage_BtsCurrentConnectableMode(static_cast<BmCoreIfMessage_BtsCurrentConnectableMode*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_START_DISCOVERY_RESULT:
                  handleBmCoreIfMessage_BtsStartDiscoveryResult(static_cast<BmCoreIfMessage_BtsStartDiscoveryResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_STOP_DISCOVERY_RESULT:
                  handleBmCoreIfMessage_BtsStopDiscoveryResult(static_cast<BmCoreIfMessage_BtsStopDiscoveryResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_DISCOVERING_STATUS:
                  handleBmCoreIfMessage_BtsDiscoveringStatus(static_cast<BmCoreIfMessage_BtsDiscoveringStatus*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_DISCOVERED_DEVICE_FOUND:
                  handleBmCoreIfMessage_BtsDiscoveredDeviceFound(static_cast<BmCoreIfMessage_BtsDiscoveredDeviceFound*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_START_PAIRING_RESULT:
                  handleBmCoreIfMessage_BtsStartPairingResult(static_cast<BmCoreIfMessage_BtsStartPairingResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_PIN_CODE_REQUESTED:
                  handleBmCoreIfMessage_BtsPinCodeRequested(static_cast<BmCoreIfMessage_BtsPinCodeRequested*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_SECURE_SIMPLE_PAIRING_REQUESTED:
                  handleBmCoreIfMessage_BtsSecureSimplePairingRequested(static_cast<BmCoreIfMessage_BtsSecureSimplePairingRequested*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_PAIRING_FINISHED:
                  handleBmCoreIfMessage_BtsPairingFinished(static_cast<BmCoreIfMessage_BtsPairingFinished*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_LINK_KEY_REQUESTED:
                  handleBmCoreIfMessage_BtsLinkKeyRequested(static_cast<BmCoreIfMessage_BtsLinkKeyRequested*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_SUPPORTED_SERVICES:
                  handleBmCoreIfMessage_BtsSupportedServices(static_cast<BmCoreIfMessage_BtsSupportedServices*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_SPP_CAPABILITIES:
                  handleBmCoreIfMessage_BtsSppCapabilities(static_cast<BmCoreIfMessage_BtsSppCapabilities*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_REMOTE_DEVICE_ID_SERVICE_RECORDS:
                  handleBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords(static_cast<BmCoreIfMessage_BtsRemoteDeviceIdServiceRecords*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_DEVICE_CAPABILITIES:
                  handleBmCoreIfMessage_BtsDeviceCapabilities(static_cast<BmCoreIfMessage_BtsDeviceCapabilities*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_REMOTE_SERVICE_SEARCH_RESULT:
                  handleBmCoreIfMessage_BtsRemoteServiceSearchResult(static_cast<BmCoreIfMessage_BtsRemoteServiceSearchResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_REMOTE_NAME_RESULT:
                  handleBmCoreIfMessage_BtsRemoteNameResult(static_cast<BmCoreIfMessage_BtsRemoteNameResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_REMOTE_NAME_STATUS_UPDATE:
                  handleBmCoreIfMessage_BtsRemoteNameStatusUpdate(static_cast<BmCoreIfMessage_BtsRemoteNameStatusUpdate*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_CONNECT_PROTOCOL_RESULT:
                  handleBmCoreIfMessage_BtsConnectProtocolResult(static_cast<BmCoreIfMessage_BtsConnectProtocolResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_REMOTE_PROTOCOL_CONNECT_REQUEST:
                  handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest(static_cast<BmCoreIfMessage_BtsRemoteProtocolConnectRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_DISCONNECT_PROTOCOL_RESULT:
                  handleBmCoreIfMessage_BtsDisconnectProtocolResult(static_cast<BmCoreIfMessage_BtsDisconnectProtocolResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_DISCONNECT_DEVICE_RESULT:
                  handleBmCoreIfMessage_BtsDisconnectDeviceResult(static_cast<BmCoreIfMessage_BtsDisconnectDeviceResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_PROTOCOL_CONNECTION_STATUS:
                  handleBmCoreIfMessage_BtsProtocolConnectionStatus(static_cast<BmCoreIfMessage_BtsProtocolConnectionStatus*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_DEVICE_CONNECTION_STATUS:
                  handleBmCoreIfMessage_BtsDeviceConnectionStatus(static_cast<BmCoreIfMessage_BtsDeviceConnectionStatus*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_LINK_QUALITY_RESULT:
                  handleBmCoreIfMessage_BtsLinkQualityResult(static_cast<BmCoreIfMessage_BtsLinkQualityResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_START_TEST_MODE_RESULT:
                  handleBmCoreIfMessage_BtsStartTestModeResult(static_cast<BmCoreIfMessage_BtsStartTestModeResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_STOP_TEST_MODE_RESULT:
                  handleBmCoreIfMessage_BtsStopTestModeResult(static_cast<BmCoreIfMessage_BtsStopTestModeResult*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_TEST_MODE_STATUS:
                  handleBmCoreIfMessage_BtsTestModeStatus(static_cast<BmCoreIfMessage_BtsTestModeStatus*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_FB_WBL_INITIALIZED:
                  handleBmCoreIfMessage_BtsFbWblInitialized(static_cast<BmCoreIfMessage_BtsFbWblInitialized*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_WBL_SERVICE_AVAILABILITY:
                  handleBmCoreIfMessage_BtsWblServiceAvailability(static_cast<BmCoreIfMessage_BtsWblServiceAvailability*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_WBL_LAST_INTENDED_MODE:
                  handleBmCoreIfMessage_BtsWblLastIntendedMode(static_cast<BmCoreIfMessage_BtsWblLastIntendedMode*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_WBL_SUPPORTED_FREQUENCIES:
                  handleBmCoreIfMessage_BtsWblSupportedFrequencies(static_cast<BmCoreIfMessage_BtsWblSupportedFrequencies*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BTS_WBL_HEALTHINESS_INDICATOR:
                  handleBmCoreIfMessage_BtsWblHealthinessIndicator(static_cast<BmCoreIfMessage_BtsWblHealthinessIndicator*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_TERMINATE_MESSAGE_HANDLER_THREAD:
                  handleBmCoreIfMessage_TerminateThreadRequest(static_cast<BmCoreIfMessage_TerminateMessageHandlerThread*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SET_DEVICE_REMOTE_CONNECTABLE:
                  (void) handleBmCoreIfMessage_SetDeviceRemoteConnectable(static_cast<BmCoreIfMessage_SetDeviceRemoteConnectable*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SM_TIMEOUT:
                  handleBmCoreIfMessage_SmTimeout(static_cast<BmCoreIfMessage_SmTimeout*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SWITCH_MULTI_HFP_SUPPORT_REQUEST:
                  (void) handleBmCoreIfMessage_SwitchMultiHFPSupportRequest(static_cast<BmCoreIfMessage_SwitchMultiHFPSupportRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SET_BT_PROFILEUSAGE_INFO_REQUEST:
                  handleBmCoreIfMessage_SetProfileUsageInfoRequest(static_cast<BmCoreIfMessage_SetBtProfilesUsageRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_SET_DEVICE_USAGEPREFERENCE_REQUEST:
                  (void) handleBmCoreIfMessage_SetDeviceUsagePreferenceRequest(static_cast<BmCoreIfMessage_SetDeviceUsagePreferenceRequest*>(bmCoreIfMessage));
                  break;
               case BM_CORE_IF_MSG_ID_BT_SYSTEM_STATE_UPDATE:
                  (void) handleBmCoreIfMessage_BtSystemStateChange(static_cast<BmCoreIfMessage_BtSystemState*>(bmCoreIfMessage));
                  break;
               default:
                  ETG_TRACE_FATAL(("handleBmCoreIfMessage: bmCoreIfMessage = 0x%p with message ID = %d is not handled",
                        (void *) bmCoreIfMessage, ETG_CENUM(BmCoreIfMsgId, msgId)));
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
            }

            delete bmCoreIfMessage;
         }
         else
         {
            ETG_TRACE_FATAL(("handleBmCoreIfMessage(bmCoreIfMessage = 0x%p): invalid message ID = %i", (void *) bmCoreIfMessage, msgId));
            FW_NORMAL_ASSERT_ALWAYS();
            delete bmCoreIfMessage;
         }
      }
      else
      {
         ETG_TRACE_FATAL(("handleBmCoreIfMessage(bmCoreIfMessage = 0x%p): bmCoreIfMessage = 0", (void *) bmCoreIfMessage));
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_SetProtocolUserAvailabilityRequest(BmCoreIfMessage_SetProtocolUserAvailabilityRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SetProtocolUserAvailabilityRequest: bmCoreIfMessage = 0x%p",
            (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);


      if (BM_RESULT_OK == bmResult)
      {
         // specific request check e.g. parameter check
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_SetProtocolUserAvailabilityRequest: general request check failed (bmResult = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doSetProtocolUserAvailabilityResponse(bmResult, bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_SetAutoConnectionTypeRequest(BmCoreIfMessage_SetAutoConnectionTypeRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SetAutoConnectionTypeRequest: bmCoreIfMessage = 0x%p",
            (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if ((BM_CORE_IF_MSG_ORIGIN_INTERNAL != bmCoreIfMessage->getOrigin())
               && (false == isValidStandardAutoConnectionType(bmCoreIfMessage->getAutoConnectionType()._type)))
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SetAutoConnectionTypeRequest: invalid parameter value for autoConnectionType._type (= %d)",
                  bmCoreIfMessage->getAutoConnectionType()._type));
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         Result result = LocalSpm::getDbManager().setStandardAutoConnectionType(bmCoreIfMessage->getAutoConnectionType()._type);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SetAutoConnectionTypeRequest: could not set standard auto connection type in DB (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }

         _autoConnectionTypePropHdl.set(bmCoreIfMessage->getAutoConnectionType());
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doSetAutoConnectionTypeResponse(bmResult, bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_SwitchBtStatusRequest(BmCoreIfMessage_SwitchBtStatusRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SwitchBtStatusRequest: bmCoreIfMessage = 0x%p",
            (void *) bmCoreIfMessage));

      // general request check
      Result result(CC_ERR_INT_NO_ERROR);
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if (false == isValidTargetSwitchState(bmCoreIfMessage->getTargetSwitchState()))
         {
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchBtStatusRequest: invalid value for parameter targetSwitchState (bmResult = %d)",
                  ETG_CENUM(BmResult, bmResult)));
         }
         else if(false == LocalSpm::getDataProvider().getBmCoreConfiguration()._btOnOffSupport)
         {
            bmResult = BM_RESULT_ERR_NOT_ALLOWED;

            ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchBtStatusRequest: btOnOffSupport is OFF So not accepting the BT ON/OFF request(bmResult = %d)",
                  ETG_CENUM(BmResult, bmResult)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchBtStatusRequest: general request check failed (bmResult = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      if (BM_RESULT_OK != bmResult)
      {
         // BM Core will not handle the request
         if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
         {
            // respond to client's request
            LocalSpm::getBmCoreCallbackIfWrapper().doSwitchBtStatusResponse(bmResult, bmCoreIfMessage->getAct());
         }
      }
      else
      {
         bool delayBtOnRequest = bmCoreIfMessage->getDelayBtOnRequest();

         LocalInfo localInfo;
         this->getLocalInfo(localInfo);

         //set the delayBtOnRequest flag as false if the stack is intialized successful before.
         if((true == delayBtOnRequest) && (true == localInfo._stackInitialized))
         {
            delayBtOnRequest = false;
         }

         // handle the request, responding to a client's request is done in property handler
         LocalSpm::getBmController().setSuppressAutoConnection(bmCoreIfMessage->getSuppressAutoConnectionAfterSwitchedOn());

         if (true == bmCoreIfMessage->getUpdateLastMode())
         {
            result = LocalSpm::getDbManager().setBtStatusTargetState(bmCoreIfMessage->getTargetSwitchState());

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchBtStatusRequest: storing BT Status' target switch state to DB failed (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }

         _btStatusSwitchPropHdl.setTargetSwitchState(bmCoreIfMessage->getTargetSwitchState(),
               bmCoreIfMessage->getOrigin(), delayBtOnRequest, bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   bool BmCoreMainController::startBtOnDelayTimer()
   {
      bool timerStarted = false;

      _lockDelayBtOnRequestTimer.lock();

      //!Cancel the timer if it's active
      if(0 != _delayBtOnRequestTimerId)
      {
         _delayBtOnRequestTimer.CancelTimer(_delayBtOnRequestTimerId);
         _delayBtOnRequestTimerId = 0;
      }//End of if(0 != _delayBtOnRequestTimerId)

      //! Start timer and wait for the response from the stack
      if(true == _delayBtOnRequestTimer.StartTimer(_delayBtOnRequestTimerId, 1000L * static_cast<long>(DELAY_BT_ON_REQ_IN_SEC), 0, this,
            &BmCoreMainController::delayBtOnTimerCb, NULL))
      {
         char milliSecondsStr[200];

         snprintf(milliSecondsStr, 199, "%llu", (1000L * static_cast<long long unsigned>(DELAY_BT_ON_REQ_IN_SEC)));
         ETG_TRACE_USR4(("startBtOnDelayTimer: started timer for delaying BT On reqeust (timer ID = 0x%p, intervalMilliseconds = %50s)",
               _delayBtOnRequestTimerId, milliSecondsStr));

         timerStarted = true;
      }
      else
      {
         ETG_TRACE_ERR(("startBtOnDelayTimer: starting timer for delaying BT ON request is failed"));
      }

      _lockDelayBtOnRequestTimer.unlock();

      _waitingForEvoStackRestart = timerStarted;

      return timerStarted;
   }

   void BmCoreMainController::stopBtOnDelayTimerAndSendReq()
   {
      ETG_TRACE_USR1(("stopBtOnDelayTimerAndSendReq: entered"));

      _lockDelayBtOnRequestTimer.lock();

      // cancel the timer if active
      if(0 != _delayBtOnRequestTimerId)
      {
         _delayBtOnRequestTimer.CancelTimer(_delayBtOnRequestTimerId);
         _delayBtOnRequestTimerId = 0;
      }

      _lockDelayBtOnRequestTimer.unlock();

      if(true == _waitingForEvoStackRestart)
      {
         // sends the BT ON request to stack
         _waitingForEvoStackRestart = false;
         this->getBtStackIfConnectionRequestIfWrapper().switchBluetoothOnOff(BTS_BT_MODE_ON, BTS_ADAPTER_MODE_APP);
      }

      return;
   }

   bool BmCoreMainController::delayBtOnTimerCb(timer_t rTimerID, void *pvObject,
         const void *pvUserData)
   {
      (void)pvUserData;
      (void)rTimerID;
      ETG_TRACE_USR1(("delayBtOnTimerCb : Timeout occured for BT on request"));

      BmCoreMainController* mainController =
            static_cast<BmCoreMainController*>(pvObject);

      if(NULL != mainController)
      {
         // stop the BT ON delay timer and send the BT ON request to stack
         (void) mainController->stopBtOnDelayTimerAndSendReq();
      }//End of if(NULL != mainController)

      return true;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_SetBtLocalFriendlyNameRequest(BmCoreIfMessage_SetBtLocalFriendlyNameRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SetBtLocalFriendlyNameRequest: bmCoreIfMessage = 0x%p",
            (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
               && (false == isValidLocalBdName(bmCoreIfMessage->getLocalFriendlyName())))
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SetBtLocalFriendlyNameRequest: invalid value for parameter localFriendlyName._localFriendlyName"));
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request
         LocalFriendlyNameType localFriendlyNameType = "";

         if(BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
         {
            localFriendlyNameType = bmCoreIfMessage->getLocalFriendlyName();
         }

         ETG_TRACE_USR1(("handleBmCoreIfMessage_SetBtLocalFriendlyNameRequest: localFriendlyName : %s", localFriendlyNameType.c_str()));

         Result result = LocalSpm::getDbManager().setLocalBdName(localFriendlyNameType);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SetBtLocalFriendlyNameRequest: storing local BD name to the DB failed (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }

         _localFriendlyNamePropHdl.set(bmCoreIfMessage->getLocalFriendlyName(), !(_resetToFactorySettingsRequested));

         LocalInfo localInfo;
         _localInfoPropHdl.get(localInfo);

         localInfo._bdName = bmCoreIfMessage->getLocalFriendlyName();

         _localInfoPropHdl.set(localInfo);
      }

      return bmResult;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_SwitchLocalPairableRequest(BmCoreIfMessage_SwitchLocalPairableModeRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SwitchLocalPairableRequest: bmCoreIfMessage = 0x%p",
            (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if (false == isValidTargetSwitchState(bmCoreIfMessage->getTargetSwitchState()))
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchLocalPairableRequest: invalid value for parameter targetSwitchState"));
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
         else if (false == isValidBdAddress(bmCoreIfMessage->getBdAddress(), true))
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchLocalPairableRequest: invalid value for parameter bdAddress"));
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request, responding to a client's request is done in property handler

         if ((TARGET_SWITCH_STATE_SWITCHED_ON == bmCoreIfMessage->getTargetSwitchState()) &&
             (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin()))
         {
            defaultActionforPairableAndDiscoverable();
         }

         if (false == bmCoreIfMessage->getBdAddress().empty())
         {
            setDeviceBdAddrForRestrictedPairingConnecting(bmCoreIfMessage->getBdAddress());
         }

         _localPairableModeSwitchPropHdl.setTargetSwitchState(bmCoreIfMessage->getTargetSwitchState(),
               bmCoreIfMessage->getOrigin(), bmCoreIfMessage->getAct());
      }
      else
      {
         // BM Core will not handle the request
         if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
         {
            // respond to client's request
            LocalSpm::getBmCoreCallbackIfWrapper().doSwitchLocalPairableModeResponse(bmResult, bmCoreIfMessage->getAct());
         }
      }

      return bmResult;
   }

   void BmCoreMainController::setDeviceBdAddrForRestrictedPairingConnecting(IN BdAddress bdAddress)
   {
      ETG_TRACE_USR1(("setDeviceBdAddrForRestrictedPairingConnecting: BdAddress = \"%50s\"", bdAddress.c_str()));

      BdAddress lcBdAddress(bdAddress);

      if (false == bdAddress.empty())
      {
         convertBdAddress2LowerCase(lcBdAddress);
      }

      _btStackIfConnectionRequestIfWrapper.setDeviceForRestrictedPairingConnecting(lcBdAddress);
   }

   void BmCoreMainController::deactivateBtLimitationModeCPW(DeviceId deviceId)
   {
      ETG_TRACE_USR1(("deactivateBtLimitationModeCPW: deviceId - %d", deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      BdAddress bdAddress;
      result = LocalSpm::getDbManager().getBdAddress(bdAddress, deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         deactivateBtLimitationModeCPW(bdAddress);
      }
      else
      {
         ETG_TRACE_ERR(("deactivateBtLimitationModeCPW: could not get bdAddress from DB (error = %d)", ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void BmCoreMainController::deactivateBtLimitationModeCPW(BdAddress bdAddress)
   {
      ETG_TRACE_USR1(("deactivateBtLimitationModeCPW: bdAddress - %50s", bdAddress.c_str()));

      // Deactivate - CPW internally
      BdName bdName = "";
      LimitationMode limitationMode;
      LimitationAction limitationAction = BM_LIMITATION_ACTION_DEACTIVATE_INTERNAL;

      limitationMode._limitationCommunicationIf = BM_LIMITATION_COMMUNICATION_IF_WIFI;
      limitationMode._limitationFeature = BM_LIMITATION_FEATURE_CAR_PLAY;

      ETG_TRACE_USR4(("deactivateLimitationMode(): bdAddress = %50s, limitationMode.feature = %d, "
            "limitationMode.commIf = %d, limitationAction = %d",
            bdAddress.c_str(), ETG_CENUM(LimitationFeature, limitationMode._limitationFeature),
            ETG_CENUM(LimitationCommunicationIf, limitationMode._limitationCommunicationIf),
            ETG_CENUM(LimitationAction, limitationAction)));

      (void) setBtLimitationModeInt(bdAddress, bdName, limitationMode, limitationAction);
   }

   void BmCoreMainController::defaultActionforPairableAndDiscoverable(void)
   {
      Result result(CC_ERR_INT_NO_ERROR);

      DeviceConnectionControllerList dccInstances;
      DeviceId sendBtLimitationDeactivateToDeviceId(0u);

      result = LocalSpm::getBmController().getReleasableDeviceConnectionControllers(dccInstances, sendBtLimitationDeactivateToDeviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         //!Disconnect ongoing device connection due to autoconnection
         DeviceId deviceIdAutoConnection = LocalSpm::getAutoConnectionController().getDeviceIdConnectionInProgress();
         ProtocolList protocolList = LocalSpm::getAutoConnectionController().getDeviceProtocolListConnectionInProgress();

         ETG_TRACE_USR4(("defaultActionforPairableAndDiscoverable: sending event STOP_AUTO_CONNECTION to AutoConnectionControllerSm"));
         result = LocalSpm::getAutoConnectionController().SendUrgentEvent(LocalSpm::getAutoConnectionController().STOP_AUTO_CONNECTION, (char *) 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("defaultActionforPairableAndDiscoverable: could not send event STOP_AUTO_CONNECTION (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
         else
         {
            // This is due to connection order remains same in the HMI
            DeviceConnectionControllerList OrderedDccInstances;
            result = LocalSpm::getDbManager().getOrderedDeviceConnectionControllers(OrderedDccInstances, dccInstances);

            for (size_t dccIdx = 0u; dccIdx < OrderedDccInstances.size(); dccIdx++)
            {
               // disconnect the ongoing device which is handling lost device.
               if(true == OrderedDccInstances[dccIdx]->isHandlingLostDevice())
               {
                  ETG_TRACE_USR4(("defaultActionforPairableAndDiscoverable: disconnect the handling lost device - %d", OrderedDccInstances[dccIdx]->getDeviceId()));

                  (void) LocalSpm::getBmController().disconnectProtocols(OrderedDccInstances[dccIdx],  ProtocolList(),
                        true, BM_DISCONNECTED_REASON_AUTOMATIC);
               }
               // disconnect the ongoing device connection due to autoconnection.
               else if(OrderedDccInstances[dccIdx]->getDeviceId() == deviceIdAutoConnection)
               {
                  ETG_TRACE_USR4(("defaultActionforPairableAndDiscoverable: disconnect auto connection deviceId - %d", OrderedDccInstances[dccIdx]->getDeviceId()));

                  (void) LocalSpm::getBmController().disconnectProtocols(OrderedDccInstances[dccIdx],  protocolList,
                        false, BM_DISCONNECTED_REASON_AUTOMATIC);
               }
            }

            //Check any BTLimitationMode-CPW in preparation then deactivate internally while switching ON the Pairable/Discoverable
            if(0u != sendBtLimitationDeactivateToDeviceId)
            {
               ETG_TRACE_USR4(("defaultActionforPairableAndDiscoverable: deactivate BtLimitationMode for deviceId - %d", sendBtLimitationDeactivateToDeviceId));
               deactivateBtLimitationModeCPW(sendBtLimitationDeactivateToDeviceId);
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("defaultActionforPairableAndDiscoverable: could not get releasable DCC instances (error = %d)", result));
      }
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_SwitchLocalConnectableModeRequest(BmCoreIfMessage_SwitchLocalConnectableModeRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SwitchLocalConnectableModeRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check
         if ((false == isValidTargetSwitchState(bmCoreIfMessage->getTargetSwitchState()))
               || (false == isValidBdAddress(bmCoreIfMessage->getBdAddress(), true)))
         {
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchLocalConnectableModeRequest: invalid parameter value (bmResult = %d)",
                  ETG_CENUM(BmResult, bmResult)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchLocalConnectableModeRequest: general request check failed (bmResult = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      if (BM_RESULT_OK != bmResult)
      {
         // BM Core will not handle the request
         if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
         {
            // respond to client's request
            LocalSpm::getBmCoreCallbackIfWrapper().doSwitchLocalConnectableModeResponse(bmResult, bmCoreIfMessage->getAct());
         }
      }
      else
      {
         // handle the request, responding to a client's request is done in property handler

         if (false == bmCoreIfMessage->getBdAddress().empty())
         {
            setDeviceBdAddrForRestrictedPairingConnecting(bmCoreIfMessage->getBdAddress());
         }

         _localConnectableModeSwitchPropHdl.setTargetSwitchState(bmCoreIfMessage->getTargetSwitchState(),
               bmCoreIfMessage->getOrigin(), bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_SwitchDiscoveryStatusRequest(BmCoreIfMessage_SwitchDiscoveryStatusRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SwitchDiscoveryStatusRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check
         if (false == isValidTargetSwitchState(bmCoreIfMessage->getTargetSwitchState()))
         {
            result = CC_ERR_INT_INVALID_PARAMETER;
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchDiscoveryStatusRequest: invalid parameter value (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));

            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchDiscoveryStatusRequest: general request check failed (error = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      // respond to client's request if BM Core cannot handle it
      if (BM_RESULT_OK != bmResult)
      {
         // BM Core will not handle the request
         if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
         {
            // respond to client's request
            LocalSpm::getBmCoreCallbackIfWrapper().doSwitchDiscoveryStatusResponse(bmResult, bmCoreIfMessage->getAct());
         }
      }
      else
      {
         if(TARGET_SWITCH_STATE_SWITCHED_ON == bmCoreIfMessage->getTargetSwitchState())
         {
            defaultActionforPairableAndDiscoverable();
         }

         // handle the request, responding to a client's request is done in property handler
         _discoveryStatusPropHdl.setTargetSwitchState(bmCoreIfMessage->getTargetSwitchState(),
               bmCoreIfMessage->getOrigin(), bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   void BmCoreMainController::handleBmCoreIfMessage_PairingStatusUpdate(BmCoreIfMessage_PairingStatusUpdate* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_PairingStatusUpdate: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      //LocalSpm::getBtLimitationController().onPairingStatusChanged(bmCoreIfMessage->getPairingStatus());

      LocalSpm::getBmCoreCallbackIfWrapper().doOnPairingStatusChanged(bmCoreIfMessage->getPairingStatus());
   }

   void BmCoreMainController::handleBmCoreIfMessage_StartPairingRequest(BmCoreIfMessage_StartPairingRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_StartPairingRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if (false == isValidBdAddress(bmCoreIfMessage->getBdAddress(), false))
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_StartPairingRequest: invalid value for parameter bdAddress"));
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
         else
         {
            PairingStatus pairingStatus;
            bmResult = this->getPairingStatus(pairingStatus);

            if (BM_RESULT_OK == bmResult)
            {
               if ((BM_PAIRING_STATE_IDLE != pairingStatus._state)
                     && (BM_PAIRING_STATE_PAIRING_PREPARED != pairingStatus._state))
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_StartPairingRequest: a pairing is already in progress"));
                  bmResult = BM_RESULT_ERR_PAIRING_IN_PROGRESS;
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_StartPairingRequest: could not get pairing status"));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_StartPairingRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // send the disconnect Device request(because ACL link might be already established for this device) if device is available in db and also connection status of that device is DISCONNECTEDd
         ConnectionStatus connectionStatus = BM_CONNECTION_STATUS_UNKNOWN;
         DisconnectedReason disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;
         result = LocalSpm::getDbManager().getDeviceConnectionStatus(connectionStatus, disconnectedReason, bmCoreIfMessage->getBdAddress());

         if (CC_ERR_INT_NO_ERROR == result)
         {
            if(BM_CONNECTION_STATUS_DISCONNECTED == connectionStatus)
            {
               this->sendDisconnectDeviceRequest(bmCoreIfMessage->getBdAddress());
            }
            else
            {
               ETG_TRACE_USR4(("handleBmCoreIfMessage_StartPairingRequest: connectionStatus = %d", ETG_CENUM(ConnectionStatus, connectionStatus)));
            }
         }

         // handle the request
         BdAddress lcBdAddress(bmCoreIfMessage->getBdAddress());
         convertBdAddress2LowerCase(lcBdAddress);

         char eventParams[20];

         result = LocalSpm::getPairingController().ParameterPAIRING_REQUESTED(OUT eventParams, IN sizeof(eventParams), IN lcBdAddress.c_str());

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("handleBmCoreIfMessage_StartPairingRequest: sending event PAIRING_REQUESTED to PairingControllerSm"));
            result = LocalSpm::getPairingController().SendEventByName("PAIRING_REQUESTED", eventParams);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_StartPairingRequest: could not send event PAIRING_REQUESTED (error = %d) ",
                     ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_StartPairingRequest: could not marshal event parameters for event PAIRING_REQUESTED (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doStartPairingResponse(bmResult, bmCoreIfMessage->getAct());
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_CancelPairingRequest(BmCoreIfMessage_CancelPairingRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_CancelPairingRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         PairingStatus pairingStatus;
         bmResult = this->getPairingStatus(pairingStatus);

         if (BM_RESULT_OK == bmResult)
         {
            if (BM_PAIRING_STATE_IDLE == pairingStatus._state)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_CancelPairingRequest: no pairing in progress"));
               bmResult = BM_RESULT_ERR_NO_PAIRING_IN_PROGRESS;
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_CancelPairingRequest: could not get pairing status"));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_CancelPairingRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         ETG_TRACE_USR4(("handleBmCoreIfMessage_CancelPairingRequest: sending event CANCEL_PAIRING to PairingControllerSm"));
         result = LocalSpm::getPairingController().SendEventByName("CANCEL_PAIRING", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_CancelPairingRequest: could not send event CANCEL_PAIRING (error = %d) ",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doCancelPairingResponse(bmResult, bmCoreIfMessage->getAct());
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_AcceptPairingRequestRequest(BmCoreIfMessage_AcceptPairingRequestRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_AcceptPairingRequestRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         PairingStatus pairingStatus;
         bmResult = this->getPairingStatus(pairingStatus);

         if (BM_RESULT_OK == bmResult)
         {
            if (BM_PAIRING_STATE_IDLE == pairingStatus._state)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_AcceptPairingRequestRequest: no pairing in progress"));
               bmResult = BM_RESULT_ERR_NO_PAIRING_IN_PROGRESS;
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_AcceptPairingRequestRequest: could not get pairing status"));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_AcceptPairingRequestRequest: general request check failed"));
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doAcceptPairingRequestResponse(bmResult, bmCoreIfMessage->getAct());
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         ETG_TRACE_USR4(("handleBmCoreIfMessage_AcceptPairingRequestRequest: sending event ACCEPT_PAIRING to PairingControllerSm"));
         result = LocalSpm::getPairingController().SendEventByName("ACCEPT_PAIRING", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_AcceptPairingRequestRequest: could not send event ACCEPT_PAIRING (error = %d) ",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_RejectPairingRequestRequest(BmCoreIfMessage_RejectPairingRequestRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_RejectPairingRequestRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         PairingStatus pairingStatus;
         bmResult = this->getPairingStatus(pairingStatus);

         if (BM_RESULT_OK == bmResult)
         {
            if (BM_PAIRING_STATE_IDLE == pairingStatus._state)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_RejectPairingRequestRequest: no pairing in progress"));
               bmResult = BM_RESULT_ERR_NO_PAIRING_IN_PROGRESS;
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_RejectPairingRequestRequest: could not get pairing status"));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_RejectPairingRequestRequest: general request check failed"));
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doRejectPairingRequestResponse(bmResult, bmCoreIfMessage->getAct());
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         ETG_TRACE_USR4(("handleBmCoreIfMessage_RejectPairingRequestRequest: sending event REJECT_PAIRING to PairingControllerSm"));
         result = LocalSpm::getPairingController().SendEventByName("REJECT_PAIRING", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_RejectPairingRequestRequest: could not send event REJECT_PAIRING (error = %d) ",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_SetPairingPinRequest(BmCoreIfMessage_SetPairingPinRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SetPairingPinRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if ((BM_CORE_IF_MSG_ORIGIN_INTERNAL != bmCoreIfMessage->getOrigin())
               && (false == isValidLegacyPairingPin(bmCoreIfMessage->getPairingPin()._pin)))
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SetPairingPinRequest: invalid parameter value for pairingPin._pin (= \"%50s\")",
                  bmCoreIfMessage->getPairingPin()._pin.c_str()));
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_SetPairingPinRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         if (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._useFixedPinLegacyPairing)
         {
            Result result = LocalSpm::getDbManager().setLegacyPairingPin(bmCoreIfMessage->getPairingPin()._pin);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_SetPairingPinRequest: storing legacy pairing PIN code to DB failed (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }

         _pairingPinPropHdl.set(bmCoreIfMessage->getPairingPin(), false);
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doSetPairingPinResponse(bmResult, bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   void BmCoreMainController::handleBmCoreIfMessage_AddOobPairedDeviceRequest(BmCoreIfMessage_AddOobPairedDeviceRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_AddOobPairedDeviceRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);
      DeviceId deviceId(0u);
      bool update = true;

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if ((false == isValidBdAddress(bmCoreIfMessage->getBdAddress()))
               || (false == isValidOobType(bmCoreIfMessage->getOobType())))
         {
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;

            ETG_TRACE_ERR(("handleBmCoreIfMessage_AddOobPairedDeviceRequest: specific request check failed (bmResult = %d)",
                  ETG_CENUM(BmResult, bmResult)));
         }
         else if(false == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported)
         {
            bmResult = BM_RESULT_ERR_NOT_ALLOWED;

            ETG_TRACE_ERR(("handleBmCoreIfMessage_AddOobPairedDeviceRequest: Carplay Wireless is disabled So not accepting the OOB request(bmResult = %d)",
                  ETG_CENUM(BmResult, bmResult)));
         }
         else if(false == LocalSpm::getDataProvider().getBmCoreConfiguration()._btOnOffSupport)
         {
            bmResult = BM_RESULT_ERR_NOT_ALLOWED;

            ETG_TRACE_ERR(("handleBmCoreIfMessage_AddOobPairedDeviceRequest: btOnOffSupport is OFF So not accepting the OOB request(bmResult = %d)",
                  ETG_CENUM(BmResult, bmResult)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_AddOobPairedDeviceRequest: general request check failed (bmResult = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         bool deviceInDb(false);
         Result result = LocalSpm::getDbManager().isDeviceInDb(deviceInDb, bmCoreIfMessage->getBdAddress());

         if (true == deviceInDb)
         {
            // device is already paired, update data in DB

            result = LocalSpm::getDbManager().updateOobPairingInfo(bmCoreIfMessage->getBdAddress(), bmCoreIfMessage->getBdName(), bmCoreIfMessage->getLinkKey());

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_AddOobPairedDeviceRequest: could not update DB with OOB pairing information (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));

               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
         else
         {
            // device is a new device

            uint32_t numberOfPairedDevices(0u);
            result = LocalSpm::getDbManager().getNumberOfDevices(numberOfPairedDevices);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               // delete oldest connected device during OOBT pairing if Max. Paired devices are reached
               if (LocalSpm::getDataProvider().getBmCoreConfiguration()._maxNumPairedDevices <= numberOfPairedDevices)
               {
                  DeviceId deviceId = 0u;
                  ConnectionStatus connectionStatus;
                  DisconnectedReason disconnectedReason;

                  // get the oldest connected device from db
                  result =  LocalSpm::getDbManager().getOldestConnectedDevice(deviceId, connectionStatus, disconnectedReason);

                  // delete the device internally if the connection status of the device DISCONECTED
                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     //Store the OOBPairedDevice request if the connectionstatus of the device is not DISCONNECTED
                     //Once the device is disconnected, OOB request will be processed.
                     if(BM_CONNECTION_STATUS_DISCONNECTED != connectionStatus)
                     {
                        DeviceConnectionController* dccInstance(0);
                        result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, deviceId);

                        if (CC_ERR_INT_NO_ERROR == result)
                        {
                           ETG_TRACE_USR1(("handleBmCoreIfMessage_AddOobPairedDeviceRequest: device is not disconnected so update the OOB pairing info "
                                 "once device is deleted"));
                           dccInstance->setOOBPairedDeviceRequest(*bmCoreIfMessage);
                           update = false;
                        }
                     }

                     bmResult = deleteDeviceInt(deviceId, BM_DEVICE_HANDLE_TYPE_SINGLE);
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_AddOobPairedDeviceRequest: could not get oldest connected device (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));

                     bmResult = BM_RESULT_ERR_GENERAL;
                  }
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_AddOobPairedDeviceRequest: could not get number of devices from DB (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));

               bmResult = BM_RESULT_ERR_GENERAL;
            }

            if ((BM_RESULT_OK == bmResult) && (true == update))
            {
               // add new device to DB
               BTSLinkKeyType linkKeyType(BTS_LINK_KEY_TYPE_UNKNOWN);
               BTSDLinkKey dLinkKey("");
               BTSMajorServiceClass majorServiceClass;
               majorServiceClass.setData(0x0000u);
               BTSMajorDeviceClass majorDeviceClass(BTS_COD_MAJORDC_UNCATEGORIZED);
               BTSMinorDeviceClass minorDeviceClass(BTS_COD_MINORDC_UNCATEGORIZED);

               ProtocolIdVersionList supportedProtocols;
               supportedProtocols.push_back(std::make_pair(BM_PROTOCOL_ID_HFP, static_cast<ProtocolVersion>(0x0000u)));
               supportedProtocols.push_back(std::make_pair(BM_PROTOCOL_ID_AVP, static_cast<ProtocolVersion>(0x0000u)));

               UuidList supportedSppUuids;

               DeviceIdentification deviceIdentification;
               deviceIdentification._specificationId = 0x0103u;
               deviceIdentification._vendorId = 0xFFFFu;
               deviceIdentification._productId = 0xFFFFu;
               deviceIdentification._version = 0x0000u;
               deviceIdentification._vendorIdSource = 2u;

               if (BM_OOB_TYPE_APPLE == bmCoreIfMessage->getOobType())
               {
                  supportedProtocols.push_back(std::make_pair(BM_PROTOCOL_ID_SPP, static_cast<ProtocolVersion>(0xFFFFu)));

                  supportedSppUuids.push_back("00000000decafadedecadeafdecacafe");

                  deviceIdentification._vendorId = 0x004Cu;
                  deviceIdentification._vendorIdSource = 1u;
               }

               result = LocalSpm::getDbManager().createDevice(OUT deviceId, bmCoreIfMessage->getBdName(),
                     bmCoreIfMessage->getBdAddress(), linkKeyType, bmCoreIfMessage->getLinkKey(), dLinkKey,
                     majorServiceClass, majorDeviceClass, minorDeviceClass, supportedProtocols, supportedSppUuids,
                     deviceIdentification);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_AddOobPairedDeviceRequest: could not add new device to DB (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));

                  bmResult = BM_RESULT_ERR_GENERAL;
               }
            }
         }

         if ((BM_RESULT_OK == bmResult) && (true == update))
         {
            DeviceBaseInfo deviceBaseInfo;
            result = LocalSpm::getDbManager().getDeviceBaseInfo(deviceBaseInfo, bmCoreIfMessage->getBdAddress());

            if (CC_ERR_INT_NO_ERROR == result)
            {
               deviceBaseInfo._remoteConnectable = (TARGET_SWITCH_STATE_SWITCHED_ON == LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode) ? true : false;

               this->onDeviceAdded(deviceBaseInfo);

               // switch on BT with changing target state for BtStatus in DB
               (void) switchBtStatusInt(TARGET_SWITCH_STATE_SWITCHED_ON, true, true);

               //TODO: the following call will result in an additional update of paired device list (rework needed)
               (void) LocalSpm::getDbManager().setDeviceUsagePreference(deviceId, BM_UP_CPW);

               //Set the CPW flag for the device in the database. It will be used for Autoconnection to connect the CPW device first.
               (void) LocalSpm::getDbManager().setCPWDevice(deviceId);

               //!Update conflict to client to set up Wi-Fi AP for CPW
               LocalSpm::getConflictManager().resolveConflicts(BM_CONFLICT_TRIGGER_OOB_BT_PAIRING);
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_AddOobPairedDeviceRequest: could not get device base info for created device from DB (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }

      if(true == update)
      {
         if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
         {
            // respond to client's request
            LocalSpm::getBmCoreCallbackIfWrapper().doAddOobPairedDeviceResponse(bmResult, bmCoreIfMessage->getAct());
         }
      }
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_DeleteDeviceRequest(BmCoreIfMessage_DeleteDeviceRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_DeleteDeviceRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if (false == isValidDeviceHandleType(bmCoreIfMessage->getDeviceHandleType()))
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_DeleteDeviceRequest: invalid value for parameter deviceHandleType"));
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_DeleteDeviceRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         DeviceIdList deviceIdList;

         if (BM_DEVICE_HANDLE_TYPE_SINGLE == bmCoreIfMessage->getDeviceHandleType())
         {
            // a particular device is requested to get deleted
            // check whether device ID is available in DB for given device handle

            DeviceId deviceId(0u);
            result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_DeleteDeviceRequest: could not get device ID for given device handle = %d from DB (error = %d)",
                     bmCoreIfMessage->getDeviceHandle(), ETG_CENUM(CcErrorInternal, result)));

               bmResult = BM_RESULT_ERR_GENERAL;
            }
            else
            {
               deviceIdList.push_back(deviceId);
            }
         }
         else // i.e. BM_DEVICE_HANDLE_TYPE_ALL
         {
            result = LocalSpm::getDbManager().getAllDeviceIds(deviceIdList);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_DeleteDeviceRequest: getting all device IDs from DB failed (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }

         if (BM_RESULT_OK == bmResult)
         {
            DeviceConnectionControllerList dccInstances;
            result = LocalSpm::getBmController().getUsedDeviceConnectionControllers(dccInstances);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               bool initiatedDisconnection(false);
               BmResult intermediateBmResult(BM_RESULT_OK);

               for (size_t devIdIndex = 0u; devIdIndex < deviceIdList.size(); ++devIdIndex)
               {
                  for (size_t dccIndex = 0u; dccIndex < dccInstances.size(); ++dccIndex)
                  {
                     if (dccInstances[dccIndex]->getDeviceId() == deviceIdList[devIdIndex])
                     {
                        ETG_TRACE_USR1(("handleBmCoreIfMessage_DeleteDeviceRequest: initiating device disconnection for device with ID = %d",
                              deviceIdList[devIdIndex]));

                        initiatedDisconnection = true;

                        intermediateBmResult = LocalSpm::getBmController().disconnectProtocols(dccInstances[dccIndex],
                              ProtocolList(), true, BM_DISCONNECTED_REASON_AUTOMATIC, true);

                        if (BM_RESULT_OK != intermediateBmResult)
                        {
                           ETG_TRACE_ERR(("handleBmCoreIfMessage_DeleteDeviceRequest: initiating device disconnection failed (intermediateBmResult = %d)",
                                 ETG_CENUM(BmResult, intermediateBmResult)));
                        }
                     }
                  } // end of for (size_t dccIndex = 0u; dccIndex < dccInstances.size(); ++dccIndex)

                  if (false == initiatedDisconnection)
                  {
                     //NCG3D-104022 - send the disconnection request to the remote device for terminate the ACL link if any exists with that device
                     this->sendDisconnectDeviceRequest(deviceIdList[devIdIndex]);

                     result = LocalSpm::getDbManager().deleteDevice(deviceIdList[devIdIndex]);

                     if (CC_ERR_INT_NO_ERROR != result)
                     {
                        ETG_TRACE_ERR(("handleBmCoreIfMessage_DeleteDeviceRequest: could not delete device with device ID = %d from DB (error = %d)",
                              deviceIdList[devIdIndex], ETG_CENUM(CcErrorInternal, result)));
                        bmResult = BM_RESULT_ERR_GENERAL;
                     }
                  }

                  initiatedDisconnection = false;
               } // end of for (size_t devIdIndex = 0u; devIdIndex < deviceIdList.size(); ++devIdIndex)
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_DeleteDeviceRequest: could not get used DCC instances (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doDeleteDeviceResponse(bmCoreIfMessage->getDeviceHandle(), bmResult,
               bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   void BmCoreMainController::sendDisconnectDeviceRequest(DeviceId& deviceId)
   {
      ETG_TRACE_USR1(("sendDisconnectDeviceRequest : deviceId = %d", deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      BdAddress bdAddress("");
      result = LocalSpm::getDbManager().getBdAddress(bdAddress, deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR4(("sendDisconnectDeviceRequest(deviceId = %d): requesting disconnection of device with BD address = \"%50s\" from BtStackIf", deviceId, bdAddress.c_str()));
         this->sendDisconnectDeviceRequest(bdAddress);
      }
      else
      {
         ETG_TRACE_ERR(("sendDisconnectDeviceRequest(deviceId = %d): could not get BD address from DB (error = %d)", deviceId, ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void BmCoreMainController::sendDisconnectDeviceRequest(const BdAddress bdAddress)
   {
      ETG_TRACE_USR1(("sendDisconnectDeviceRequest : bdAddress = %50s", bdAddress.c_str()));

      this->getBtStackIfConnectionRequestIfWrapper().disconnectDevice(bdAddress);
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_StartAutoConnectionSequenceRequest(BmCoreIfMessage_StartAutoConnectionSequenceRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_StartAutoConnectionSequenceRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if (false == isValidTemporaryAutoConnectionType(bmCoreIfMessage->getTemporaryAutoConnectionType()))
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_StartAutoConnectionSequenceRequest: invalid parameter value for temporaryAutoConnectionType (= %d)",
                  bmCoreIfMessage->getTemporaryAutoConnectionType()));
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_StartAutoConnectionSequenceRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         BtLimitationMode btLimitationMode;
         (void) this->getBtLimitationMode(btLimitationMode);

         if (true == btLimitationMode._btLimitationModeInfoList.empty())
         {
            StandardAutoConnectionType standardAutoConnectionType(BM_STANDARD_AUTO_CONNECTION_UNKNOWN);

            if (BM_TEMPORARY_AUTO_CONNECTION_DEFAULT == bmCoreIfMessage->getTemporaryAutoConnectionType())
            {
               AutoConnectionType autoConnectionType;
               (void) this->getAutoConnectionType(autoConnectionType);

               standardAutoConnectionType = autoConnectionType._type;
            }
            else if (BM_TEMPORARY_AUTO_CONNECTION_LAST_CONNECTED_DEVICES == bmCoreIfMessage->getTemporaryAutoConnectionType())
            {
               standardAutoConnectionType = BM_STANDARD_AUTO_CONNECTION_LAST_CONNECTED_DEVICES;
            }

            char eventParams[20];

            Result result = LocalSpm::getAutoConnectionController().ParameterSTART_AUTO_CONNECTION(OUT eventParams,
                  IN sizeof(eventParams), IN standardAutoConnectionType);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               ETG_TRACE_USR4(("handleBmCoreIfMessage_StartAutoConnectionSequenceRequest: sending event START_AUTO_CONNECTION to AutoConnectionControllerSm"));
               result = LocalSpm::getAutoConnectionController().SendEventByName("START_AUTO_CONNECTION", eventParams);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_StartAutoConnectionSequenceRequest: could not send event START_AUTO_CONNECTION (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
                  bmResult = BM_RESULT_ERR_GENERAL;
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_StartAutoConnectionSequenceRequest: could not marshal event parameters for event START_AUTO_CONNECTION (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_StartAutoConnectionSequenceRequest: BT limitation mode is currently set, request is not allowed to be processed"));
            bmResult = BM_RESULT_ERR_NOT_ALLOWED;
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doStartAutoConnectSequenceResponse(bmResult, bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_StopAutoConnectionSequenceRequest(BmCoreIfMessage_StopAutoConnectionSequenceRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_StopAutoConnectionSequenceRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK != bmResult)
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_StopAutoConnectionSequenceRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         ETG_TRACE_USR4(("handleBmCoreIfMessage_StopAutoConnectionSequenceRequest: sending event STOP_AUTO_CONNECTION to AutoConnectionControllerSm"));
         Result result = LocalSpm::getAutoConnectionController().SendUrgentEvent(LocalSpm::getAutoConnectionController().STOP_AUTO_CONNECTION, 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_StopAutoConnectionSequenceRequest: could not send event STOP_AUTO_CONNECTION (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doStopAutoConnectSequenceResponse(bmResult, bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_ConnectDeviceRequest(BmCoreIfMessage_ConnectDeviceRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_ConnectDeviceRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);
      BdAddress bdAddress("");

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectDeviceRequest: could not get device ID for given device handle = %d from DB (error = %d)",
                  bmCoreIfMessage->getDeviceHandle(), ETG_CENUM(CcErrorInternal, result)));

            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectDeviceRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         BlockStatus blockStatus;
         (void) this->getBlockStatus(blockStatus);

         if(BM_OVERALL_BLOCK_STATE_BLOCKED != blockStatus._overallDeviceBlockStatus._blockState)
         {
            // handle the request

            BlockState deviceBlockState(BM_BLOCK_STATE_UNBLOCKED);
            result = LocalSpm::getDbManager().getDeviceBlockState(deviceBlockState, deviceId);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               if (BM_BLOCK_STATE_UNBLOCKED == deviceBlockState)
               {
                  ProtocolList protocolList;

                  if (true == LocalSpm::getBmController().isRequestQueueEmpty())
                  {
                     result = LocalSpm::getBmController().getProtocolsForConnectionRestoring(protocolList, deviceId, false);

                     if (CC_ERR_INT_NO_ERROR == result)
                     {
                        if (true == protocolList.empty())
                        {
                           ETG_TRACE_USR1(("handleBmCoreIfMessage_ConnectDeviceRequest: list of protocols expected to be connected for device with ID = %d is empty -> assuming all device protocols as to be connected",
                                 deviceId));

                           ProtocolIdList deviceConnectionProtocols;
                           result = LocalSpm::getDataProvider().getDeviceConnectionProtocols(deviceConnectionProtocols);

                           if (CC_ERR_INT_NO_ERROR == result)
                           {
                              for (size_t i = 0u; i < deviceConnectionProtocols.size(); i++)
                              {
                                 protocolList.push_back(Protocol(deviceConnectionProtocols[i], ""));
                              }
                           }
                           else
                           {
                              ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectDeviceRequest: could not get device connection protocols from DataProvider (error = %d) ",
                                    ETG_CENUM(CcErrorInternal, result)));
                              bmResult = BM_RESULT_ERR_GENERAL;
                           }
                        }

                        if (CC_ERR_INT_NO_ERROR == result)
                        {
                           BlockState protocolBlockState(BM_BLOCK_STATE_UNBLOCKED);
                           bool processRequest = false;

                           for (size_t idx = 0u; idx < protocolList.size(); ++idx)
                           {
                              // Process the request if the requested protocols are not blocked.
                              result = LocalSpm::getDbManager().getProtocolBlockState(protocolBlockState,
                                    deviceId, protocolList[idx]);

                              if (CC_ERR_INT_NO_ERROR == result)
                              {
                                 if(BM_BLOCK_STATE_UNBLOCKED == protocolBlockState)
                                 {
                                    processRequest = true;
                                 }
                              }
                           }

                           if(true == processRequest)
                           {
                              result = LocalSpm::getBmController().addConnectionRequestDataToQueue((bmCoreIfMessage->getOrigin() == BM_CORE_IF_MSG_ORIGIN_INTERNAL),
                                    BM_CONNECTION_REQUEST_ORIGIN_LOCAL, deviceId, protocolList, bmCoreIfMessage->getPageTimeout(),
                                    bmCoreIfMessage->getDelayRequestProcessing());

                              if (CC_ERR_INT_NO_ERROR == result)
                              {
                                 bmResult = LocalSpm::getBmController().handleRequest();
                              }
                              else
                              {
                                 ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectDeviceRequest: could not add connection request data to queue (error = %d)",
                                       ETG_CENUM(CcErrorInternal, result)));
                                 bmResult = BM_RESULT_ERR_GENERAL;
                              }
                           }
                           else
                           {
                              ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectDeviceRequest: requested protocols are blocked"));
                              bmResult = BM_RESULT_ERR_GENERAL;
                           }
                        }
                     }
                     else
                     {
                        ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectDeviceRequest: could not get list of protocols for connection restoring (error = %d)",
                              ETG_CENUM(CcErrorInternal, result)));
                        bmResult = BM_RESULT_ERR_GENERAL;
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectDeviceRequest: there is still a connection request waiting to be processed"));
                     bmResult = BM_RESULT_ERR_BUSY;
                  }
               }
               else
               {
                  result = CC_ERR_INT_DEVICE_BLOCKED;
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectDeviceRequest: device is currently blocked (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
                  bmResult = BM_RESULT_ERR_DEVICE_BLOCKED;
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectDeviceRequest: could not get device's block state (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
         else
         {
            result = CC_ERR_INT_DEVICE_BLOCKED;
            ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectDeviceRequest:overall devices are blocked(error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_DEVICE_BLOCKED;
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doConnectDeviceResponse(bmCoreIfMessage->getDeviceHandle(), bmResult,
               bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_DisconnectDeviceRequest(BmCoreIfMessage_DisconnectDeviceRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_DisconnectDeviceRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if (false == isValidDeviceHandleType(bmCoreIfMessage->getDeviceHandleType()))
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_DisconnectDeviceRequest: invalid value for parameter deviceHandleType"));
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
         else
         {
            if (BM_DEVICE_HANDLE_TYPE_SINGLE == bmCoreIfMessage->getDeviceHandleType())
            {
               result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_DisconnectDeviceRequest: could not get device ID for given device handle = %d from DB (error = %d)",
                        bmCoreIfMessage->getDeviceHandle(), ETG_CENUM(CcErrorInternal, result)));

                  bmResult = BM_RESULT_ERR_GENERAL;
               }
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_DisconnectDeviceRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         if (BM_DEVICE_HANDLE_TYPE_SINGLE == bmCoreIfMessage->getDeviceHandleType())
         {
            if ((false == LocalSpm::getBmController().isRequestQueueEmpty())
                  && (deviceId == LocalSpm::getBmController().getDeviceIdOfDeferredRequest()))
            {
               LocalSpm::getBmController().removeRequestDataFromQueue();
            }
            else
            {
               bmResult = LocalSpm::getBmController().disconnectProtocols(deviceId, ProtocolList(), true,
                     bmCoreIfMessage->getDisconnectedReason(), bmCoreIfMessage->getDeleteDevice());

               if((BM_RESULT_OK == bmResult) && (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported))
               {
                  //Check the device is used for CPW. If the CPW session is not Active then Deactivate the CPW internally
                  BtLimitationModeInfo btLimitationModeInfo;
                  result = this->getBtLimitationModeInfo(btLimitationModeInfo, deviceId);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     (void) checkAndDeactivateCPW(btLimitationModeInfo);
                  }
               }
            }
         }
         else // i.e. BM_DEVICE_HANDLE_TYPE_ALL
         {
            if (false == LocalSpm::getBmController().isRequestQueueEmpty())
            {
               LocalSpm::getBmController().removeRequestDataFromQueue();
            }

            DeviceConnectionControllerList dccInstances;
            result = LocalSpm::getBmController().getUsedDeviceConnectionControllers(dccInstances);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               // This is due to connection order remains same in the HMI
               DeviceConnectionControllerList orderedDccInstances;
               result = LocalSpm::getDbManager().getOrderedDeviceConnectionControllers(orderedDccInstances, dccInstances);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  for (size_t index = 0u; index < orderedDccInstances.size(); index++)
                  {
                     bmResult = LocalSpm::getBmController().disconnectProtocols(orderedDccInstances[index],
                           ProtocolList(), true, bmCoreIfMessage->getDisconnectedReason(),
                           bmCoreIfMessage->getDeleteDevice());
                  }
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_DisconnectDeviceRequest: could not get used DCC instances (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }

            //Check the device is used for CPW and disconnect request received from client. If the CPW session is not Active then Deactivate the CPW internally
            if(true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported)
            {
               BtLimitationMode btLimitationModeInfo;
               (void)getBtLimitationMode(btLimitationModeInfo);

               if(false == btLimitationModeInfo._btLimitationModeInfoList.empty())
               {
                  for(size_t idx = 0; idx < btLimitationModeInfo._btLimitationModeInfoList.size(); idx++)
                  {
                     (void) checkAndDeactivateCPW(btLimitationModeInfo._btLimitationModeInfoList[idx]);
                  }
               }
            }
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doDisconnectDeviceResponse(bmCoreIfMessage->getDeviceHandle(), bmResult,
               bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   Result BmCoreMainController::checkAndDeactivateCPW(BtLimitationModeInfo& btLimitationModeInfo)
   {
      ETG_TRACE_USR1(("checkAndDeactivateCPW: entered"));
      Result result(CC_ERR_INT_NO_ERROR);

      if((BM_LIMITATION_FEATURE_CAR_PLAY == btLimitationModeInfo._limitationMode._limitationFeature)
         && ((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == btLimitationModeInfo._limitationMode._limitationCommunicationIf)
               || (BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == btLimitationModeInfo._limitationMode._limitationCommunicationIf)))
      {
         if((BM_LIMITATION_STATE_ACTIVATING != btLimitationModeInfo._limitationState) &&
               (BM_LIMITATION_STATE_ACTIVE != btLimitationModeInfo._limitationState))
         {
            ETG_TRACE_USR4(("checkAndDeactivateCPW: sending event STOP_AUTO_CONNECTION to AutoConnectionControllerSm"));
            result = LocalSpm::getAutoConnectionController().SendUrgentEvent(LocalSpm::getAutoConnectionController().STOP_AUTO_CONNECTION, (char *) 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("checkAndDeactivateCPW: could not send event STOP_AUTO_CONNECTION (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }

            // Deactivate  CPW
            deactivateBtLimitationModeCPW(btLimitationModeInfo._bdAddress);
         }
         else
         {
            ETG_TRACE_USR4(("checkAndDeactivateCPW: CPW is active for bdAddress = %50s", btLimitationModeInfo._bdAddress.c_str()));
            result = CC_ERR_INT_ITEM_NOT_FOUND;
         }
      }
      else
      {
         ETG_TRACE_USR1(("checkAndDeactivateCPW: btLimitationMode - CPW is not set"));
      }

      return result;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_ConnectProfilesRequest(BmCoreIfMessage_ConnectProfilesRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_ConnectProfilesRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectProfilesRequest: could not get device ID for given device handle = %d from DB (error = %d)",
                  bmCoreIfMessage->getDeviceHandle(), ETG_CENUM(CcErrorInternal, result)));

            bmResult = BM_RESULT_ERR_GENERAL;
         }
         else if(false == LocalSpm::getDataProvider().getBmCoreConfiguration()._btOnOffSupport)
         {
            bmResult = BM_RESULT_ERR_NOT_ALLOWED;

            ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectProfilesRequest: btOnOffSupport is OFF So not accepting the connect request(bmResult = %d)",
                  ETG_CENUM(BmResult, bmResult)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectProfilesRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         if (false == bmCoreIfMessage->getProtocolList().empty())
         {
            BlockState deviceBlockState(BM_BLOCK_STATE_UNBLOCKED);
            result = LocalSpm::getDbManager().getDeviceBlockState(deviceBlockState, deviceId);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               if (BM_BLOCK_STATE_UNBLOCKED == deviceBlockState)
               {
                  if (true == LocalSpm::getBmController().isRequestQueueEmpty())
                  {
                     DeviceConnectionController* dccInstance(0);
                     result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, deviceId);

                     if (CC_ERR_INT_NO_ERROR == result)
                     {
                        // found a DCC already assigned to given device
                        ConnectionStatus connectionStatus;
                        DisconnectedReason disconnectedReason;

                        result = LocalSpm::getDbManager().getDeviceConnectionStatus(connectionStatus, disconnectedReason, deviceId);

                        if (CC_ERR_INT_NO_ERROR == result)
                        {
                           ETG_TRACE_USR1(("handleBmCoreIfMessage_ConnectProfilesRequest: found a DCC already assigned to device with given device ID = %d with device connection status = %d",
                                 deviceId, ETG_CENUM(ConnectionStatus, connectionStatus)));

                           if ((BM_CONNECTION_STATUS_CONNECTING != connectionStatus)
                                 && (BM_CONNECTION_STATUS_CONNECTED != connectionStatus))
                           {
                              bool isSlaveProtocolId(true);

                              for (size_t idx = 0u; idx < bmCoreIfMessage->getProtocolList().size(); ++idx)
                              {
                                 isSlaveProtocolId = true;

                                 if (CC_ERR_INT_NO_ERROR == LocalSpm::getDataProvider().isSlaveProtocolId(isSlaveProtocolId,
                                       bmCoreIfMessage->getProtocolList()[idx]._protocolId))
                                 {
                                    ETG_TRACE_USR1(("handleBmCoreIfMessage_ConnectProfilesRequest: given protocol = (%d, \"%50s\") is%10s a slave protocol",
                                          ETG_CENUM(ProtocolId, bmCoreIfMessage->getProtocolList()[idx]._protocolId),
                                          bmCoreIfMessage->getProtocolList()[idx]._uuid.c_str(), isSlaveProtocolId ? "" : " NOT"));

                                    if (false == isSlaveProtocolId)
                                    {
                                       break;
                                    }
                                    else
                                    {
                                      if(true == LocalSpm::getBmController().checkSupportedUUID(deviceId,
                                             bmCoreIfMessage->getProtocolList()[idx]._protocolId,
                                             bmCoreIfMessage->getProtocolList()[idx]._uuid, BM_CONNECTION_REQUEST_ORIGIN_LOCAL, true))
                                       {
                                          isSlaveProtocolId = false;
                                       }
                                    }
                                 }
                              }

                              if (true == isSlaveProtocolId)
                              {
                                 result = CC_ERR_INT_DCC_IN_DISCONNECTING;
                                 bmResult = BM_RESULT_ERR_DEVICE_DISCONNECTING;
                              }
                           }
                        }
                        else
                        {
                           ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectProfilesRequest: could not get device connection status for device with ID = %d from DB (error = %d)",
                                 deviceId, ETG_CENUM(CcErrorInternal, result)));
                           bmResult = BM_RESULT_ERR_GENERAL;
                        }
                     }
                     else if (CC_ERR_INT_DB_DCC_HANDLE_NOT_FOUND == result)
                     {
                        result = CC_ERR_INT_NO_ERROR;

                        // unblock the remote connection flag
                        if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
                        {
                           // reset the RemoteConnectable as true (allowing remote connection request)
                           if (false == LocalSpm::getDataProvider().getBmCoreConfiguration()._connectAfterPairing)
                           {
                              result = this->unblockRemoteConnections(deviceId);

                              if (CC_ERR_INT_NO_ERROR != result)
                              {
                                 ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectProfilesRequest: could not unblock remote connections for device with ID = %d (error = %d)",
                                       deviceId, ETG_CENUM(CcErrorInternal, result)));
                              }
                           }
                        }
                     }
                     else
                     {
                        bmResult = BM_RESULT_ERR_GENERAL;
                     }

                     if (CC_ERR_INT_NO_ERROR == result)
                     {
                        bool processRequest = false;
                        BlockState protocolBlockState(BM_BLOCK_STATE_UNBLOCKED);

                        ProtocolList protocolList = bmCoreIfMessage->getProtocolList();

                        for (size_t idx = 0u; idx < protocolList.size(); ++idx)
                        {
                           // Process the request if the requested protocols are not blocked.
                           result = LocalSpm::getDbManager().getProtocolBlockState(protocolBlockState,
                                 deviceId, protocolList[idx]);

                           if (CC_ERR_INT_NO_ERROR == result)
                           {
                              if(BM_BLOCK_STATE_UNBLOCKED == protocolBlockState)
                              {
                                 processRequest = true;
                              }
                           }
                        }

                        if(true == processRequest)
                        {
                           result = LocalSpm::getBmController().addConnectionRequestDataToQueue((BM_CORE_IF_MSG_ORIGIN_INTERNAL == bmCoreIfMessage->getOrigin()),
                                 BM_CONNECTION_REQUEST_ORIGIN_LOCAL, deviceId, protocolList, bmCoreIfMessage->getPageTimeout());

                           if (CC_ERR_INT_NO_ERROR == result)
                           {
                              bmResult = LocalSpm::getBmController().handleRequest();
                           }
                           else
                           {
                              ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectProfilesRequest: could not add connection request data to queue (error = %d)",
                                    ETG_CENUM(CcErrorInternal, result)));
                              bmResult = BM_RESULT_ERR_GENERAL;
                           }
                        }
                        else
                        {
                           ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectProfilesRequest: requested protocols are blocked"));
                           bmResult = BM_RESULT_ERR_GENERAL;
                        }
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectProfilesRequest: there is still a connection request waiting to be processed"));
                     bmResult = BM_RESULT_ERR_BUSY;
                  }
               }
               else
               {
                  result = CC_ERR_INT_DEVICE_BLOCKED;
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectProfilesRequest: device is currently blocked (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
                  bmResult = BM_RESULT_ERR_DEVICE_BLOCKED;
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectProfilesRequest: could not get device's block state for device with ID = %d (error = %d)",
                     deviceId, ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
         else
         {
            // protocol list is empty

            if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
            {
               // reset the RemoteConnectable as true (allowing remote connection request)
               if (false == LocalSpm::getDataProvider().getBmCoreConfiguration()._connectAfterPairing)
               {
                  result = this->unblockRemoteConnections(deviceId);

                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_ConnectProfilesRequest: could not unblock remote connections for device with ID = %d (error = %d)",
                           deviceId, ETG_CENUM(CcErrorInternal, result)));
                  }
               }
            }
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doConnectProfilesResponse(bmCoreIfMessage->getDeviceHandle(), bmResult,
               bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_DisconnectProfilesRequest(BmCoreIfMessage_DisconnectProfilesRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_DisconnectProfilesRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_DisconnectProfilesRequest: could not get device ID for given device handle = %d from DB (error = %d)",
                  bmCoreIfMessage->getDeviceHandle(), ETG_CENUM(CcErrorInternal, result)));

            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_DisconnectProfilesRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         if ((false == LocalSpm::getBmController().isRequestQueueEmpty())
               && (deviceId == LocalSpm::getBmController().getDeviceIdOfDeferredRequest()))
         {
            LocalSpm::getBmController().cleanUpDeferredConnectionRequest(bmCoreIfMessage->getProtocolList());
         }

         bmResult = LocalSpm::getBmController().disconnectProtocols(deviceId, bmCoreIfMessage->getProtocolList(),
               false, BM_DISCONNECTED_REASON_NORMAL_LOSS_LOCAL);
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doDisconnectProfilesResponse(bmCoreIfMessage->getDeviceHandle(), bmResult,
               bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_BlockDeviceRequest(BmCoreIfMessage_BlockDeviceRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BlockDeviceRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);
      DeviceIdList deviceIdList;

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if (false == isValidDeviceHandleType(bmCoreIfMessage->getDeviceHandleType(), true))
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockDeviceRequest: invalid value for parameter deviceHandleType"));
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
         else if ((BM_DEVICE_HANDLE_TYPE_SINGLE == bmCoreIfMessage->getDeviceHandleType())
               || (BM_DEVICE_HANDLE_TYPE_ALL_EXCEPT == bmCoreIfMessage->getDeviceHandleType()))
         {
            result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockDeviceRequest: could not get device ID for given device handle = %d from DB (error = %d)",
                     bmCoreIfMessage->getDeviceHandle(), ETG_CENUM(CcErrorInternal, result)));

               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockDeviceRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         if (BM_DEVICE_HANDLE_TYPE_SINGLE == bmCoreIfMessage->getDeviceHandleType())
         {
            deviceIdList.push_back(deviceId);

            result = LocalSpm::getDbManager().setDeviceBlockState(deviceId, BM_BLOCK_STATE_BLOCKED);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockDeviceRequest: could not set device's block state for device with ID = %d in DB (error = %d)",
                     deviceId, ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
         else
         {
            result = LocalSpm::getDbManager().getAllDeviceIds(deviceIdList);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               if (BM_DEVICE_HANDLE_TYPE_ALL == bmCoreIfMessage->getDeviceHandleType())
               {
                  OverallDeviceBlockStatus overallDeviceBlockStatus(BM_OVERALL_BLOCK_STATE_BLOCKED, "");
                  result = this->updateOverallDeviceBlockStatus(overallDeviceBlockStatus);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     this->setDeviceBdAddrForRestrictedPairingConnecting("");

                     (void) this->switchLocalPairableModeInt(LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode, "");

                     (void) this->switchLocalConnectableModeInt(LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode, "");
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockDeviceRequest: could not update overall device block status (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                     bmResult = BM_RESULT_ERR_GENERAL;
                  }
               }
               else // i.e. BM_DEVICE_HANDLE_TYPE_ALL_EXCEPT
               {
                  DeviceIdList::iterator it = std::find(deviceIdList.begin(), deviceIdList.end(), deviceId);

                  if (deviceIdList.end() != it)
                  {
                     deviceIdList.erase(it);

                     BdAddress bdAddress("");
                     result = LocalSpm::getDbManager().getBdAddress(bdAddress, deviceId);

                     if (CC_ERR_INT_NO_ERROR == result)
                     {
                        this->setDeviceBdAddrForRestrictedPairingConnecting(bdAddress);

                        OverallDeviceBlockStatus overallDeviceBlockStatus(BM_OVERALL_BLOCK_STATE_BLOCKED_EXCEPT, bdAddress);
                        result = this->updateOverallDeviceBlockStatus(overallDeviceBlockStatus);

                        if (CC_ERR_INT_NO_ERROR != result)
                        {
                           ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockDeviceRequest: could not update overall device block status (error = %d)",
                                 ETG_CENUM(CcErrorInternal, result)));
                           bmResult = BM_RESULT_ERR_GENERAL;
                        }
                     }
                     else
                     {
                        ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockDeviceRequest: could not get BD address for given device ID = %d from DB (error = %d)",
                              deviceId, ETG_CENUM(CcErrorInternal, result)));
                        bmResult = BM_RESULT_ERR_GENERAL;
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockDeviceRequest: could not find given device ID in list of all device IDs"));
                     bmResult = BM_RESULT_ERR_GENERAL;
                  }
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockDeviceRequest: could get all device IDs from DB (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }

         if (BM_RESULT_OK == bmResult)
         {
            DeviceConnectionControllerList dccInstances;
            result = LocalSpm::getBmController().getUsedDeviceConnectionControllers(dccInstances);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               // This is due to connection order remains same in the HMI
               DeviceConnectionControllerList orderedDccInstances;
               result = LocalSpm::getDbManager().getOrderedDeviceConnectionControllers(orderedDccInstances, dccInstances);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  for (size_t index = 0u; index < orderedDccInstances.size(); index++)
                  {
                     if (deviceIdList.end() != std::find(deviceIdList.begin(), deviceIdList.end(), orderedDccInstances[index]->getDeviceId()))
                     {
                        bmResult = LocalSpm::getBmController().disconnectProtocols(orderedDccInstances[index],
                              ProtocolList(), true, BM_DISCONNECTED_REASON_AUTOMATIC);
                     }
                  }
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockDeviceRequest: could not get used DCC instances (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doBlockDeviceResponse(bmCoreIfMessage->getDeviceHandle(), bmResult,
               bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_UnblockDeviceRequest(BmCoreIfMessage_UnblockDeviceRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_UnblockDeviceRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if (false == isValidDeviceHandleType(bmCoreIfMessage->getDeviceHandleType()))
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_UnblockDeviceRequest: invalid value for parameter deviceHandleType"));
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
         else if (BM_DEVICE_HANDLE_TYPE_SINGLE == bmCoreIfMessage->getDeviceHandleType())
         {
            result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockDeviceRequest: could not get device ID for given device handle = %d from DB (error = %d)",
                     bmCoreIfMessage->getDeviceHandle(), ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_UnblockDeviceRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         if (BM_DEVICE_HANDLE_TYPE_SINGLE == bmCoreIfMessage->getDeviceHandleType())
         {
            result = LocalSpm::getDbManager().setDeviceBlockState(deviceId, BM_BLOCK_STATE_UNBLOCKED);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_UnblockDeviceRequest: could not set device's block state for device with ID = %d in DB (error = %d)",
                     deviceId, ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
         else // i.e. BM_DEVICE_HANDLE_TYPE_ALL
         {
            this->setDeviceBdAddrForRestrictedPairingConnecting("");

            OverallDeviceBlockStatus overallDeviceBlockStatus(BM_OVERALL_BLOCK_STATE_UNBLOCKED, "");
            result = this->updateOverallDeviceBlockStatus(overallDeviceBlockStatus);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_UnblockDeviceRequest: could not update overall device block status (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doUnblockDeviceResponse(bmCoreIfMessage->getDeviceHandle(), bmResult,
               bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_BlockProfilesRequest(BmCoreIfMessage_BlockProfilesRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BlockProfilesRequest: bmCoreIfMessage = 0x%p, ignoreStopAutoconectionStatus = %10s", (void *) bmCoreIfMessage,
      bmCoreIfMessage->getIgnoreStopAutoconectionStatus() ? "true" : "false"));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockProfilesRequest: could not get device ID for given device handle = %d from DB (error = %d)",
                  bmCoreIfMessage->getDeviceHandle(), ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockProfilesRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         for (size_t index = 0u; index < bmCoreIfMessage->getProtocolList().size(); index++)
         {
            result = LocalSpm::getDbManager().setProtocolBlockState(deviceId,
                  bmCoreIfMessage->getProtocolList()[index]._protocolId,
                  bmCoreIfMessage->getProtocolList()[index]._uuid, BM_BLOCK_STATE_BLOCKED);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockProfilesRequest: could not set protocol block state for device with ID = %d in DB (error = %d)",
                     deviceId, ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }

            if((BM_PROTOCOL_ID_AVP == bmCoreIfMessage->getProtocolList()[index]._protocolId)
                  &&(BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin()))
            {
               BdAddress bdAddress;
               result = LocalSpm::getDbManager().getBdAddress(bdAddress, deviceId);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  ETG_TRACE_USR4(("handleBmCoreIfMessage_BlockProfilesRequest: Update AVP block state for device with bdAddress = \"%50s\"",
                        bdAddress.c_str()));

                  LocalSpm::getBtLimitationController().UpdateAvpBlockedStateForDevice(bdAddress, true);
               }
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockProfilesRequest: could not get BD address for given device ID = %d from DB (error = %d)",
                        deviceId, ETG_CENUM(CcErrorInternal, result)));
                  bmResult = BM_RESULT_ERR_GENERAL;
               }
            }//!End of if((BM_PROTOCOL_ID_AVP == bmCoreIfMessage->getProtocolList()[index]._protocolId)...)
         }//!End of for (size_t index = 0u; index < bmCoreIfMessage->getProtocolList().size(); index++)
      }//!End of if (BM_RESULT_OK == bmResult)

      if (BM_RESULT_OK == bmResult)
      {
         DeviceConnectionControllerList dccInstances;
         result = LocalSpm::getBmController().getUsedDeviceConnectionControllers(dccInstances);

         for (size_t index = 0u; index < dccInstances.size(); index++)
         {
            if (deviceId == dccInstances[index]->getDeviceId())
            {
               bmResult = LocalSpm::getBmController().disconnectProtocols(dccInstances[index],
                     bmCoreIfMessage->getProtocolList(), false, BM_DISCONNECTED_REASON_AUTOMATIC, false, bmCoreIfMessage->getIgnoreStopAutoconectionStatus());
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BlockProfilesRequest: could not get used DCC instances (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
         bmResult = BM_RESULT_ERR_GENERAL;
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doBlockProfilesResponse(bmCoreIfMessage->getDeviceHandle(), bmResult,
               bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_UnblockProfilesRequest(BmCoreIfMessage_UnblockProfilesRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_UnblockProfilesRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_UnblockProfilesRequest: could not get device ID for given device handle = %d from DB (error = %d)",
                  bmCoreIfMessage->getDeviceHandle(), ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_UnblockProfilesRequest: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         for (size_t index = 0u; index < bmCoreIfMessage->getProtocolList().size(); index++)
         {
            BlockState blockState = BM_BLOCK_STATE_UNBLOCKED;

            if((BM_PROTOCOL_ID_AVP == bmCoreIfMessage->getProtocolList()[index]._protocolId)
                  &&(BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin()))
            {
               BdAddress bdAddress;
               result = LocalSpm::getDbManager().getBdAddress(bdAddress, deviceId);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  ETG_TRACE_USR4(("handleBmCoreIfMessage_UnblockProfilesRequest: Update AVP block state for device with bdAddress = \"%50s\"",
                        bdAddress.c_str()));

                  LocalSpm::getBtLimitationController().UpdateAvpBlockedStateForDevice(bdAddress, false);

                  // Get the LimitationMode
                  BtLimitationModeInfo btLimitationModeInfo;
                  result = this->getBtLimitationModeInfo(btLimitationModeInfo, bdAddress);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     Protocol protocol(bmCoreIfMessage->getProtocolList()[index]._protocolId, bmCoreIfMessage->getProtocolList()[index]._uuid);

                     result = LocalSpm::getDbManager().getProtocolBlockState(blockState, deviceId, protocol);

                     if (CC_ERR_INT_NO_ERROR != result)
                     {
                        ETG_TRACE_ERR(("handleBmCoreIfMessage_UnblockProfilesRequest: could not get protocol block state for device with ID = %d in DB (error = %d)",
                              deviceId, ETG_CENUM(CcErrorInternal, result)));
                        bmResult = BM_RESULT_ERR_GENERAL;
                     }
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_UnblockProfilesRequest: could not get BD address for given device ID = %d from DB (error = %d)",
                        deviceId, ETG_CENUM(CcErrorInternal, result)));
                  bmResult = BM_RESULT_ERR_GENERAL;
               }
            }//!End of if((BM_PROTOCOL_ID_AVP == bmCoreIfMessage->getProtocolList()[index]._protocolId) ...)

            result = LocalSpm::getDbManager().setProtocolBlockState(deviceId,
                  bmCoreIfMessage->getProtocolList()[index]._protocolId,
                  bmCoreIfMessage->getProtocolList()[index]._uuid, blockState);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_UnblockProfilesRequest: could not set protocol block state for device with ID = %d in DB (error = %d)",
                     deviceId, ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }

         }//!End of for (size_t index = 0u; index < bmCoreIfMessage->getProtocolList().size(); index++)
      }//!End of if (BM_RESULT_OK == bmResult)

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doUnblockProfilesResponse(bmCoreIfMessage->getDeviceHandle(), bmResult,
               bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   void BmCoreMainController::handleBmCoreIfMessage_PairedDeviceListUpdate(BmCoreIfMessage_PairedDeviceListUpdate* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_PairedDeviceListUpdate: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      LocalSpm::getBmCoreCallbackIfWrapper().doOnPairedDeviceListChanged(bmCoreIfMessage->getPairedDeviceList());
   }

   void BmCoreMainController::handleBmCoreIfMessage_DeviceConnectionStatusListUpdate(BmCoreIfMessage_DeviceConnectionStatusListUpdate* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_DeviceConnectionStatusListUpdate: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      LocalSpm::getBmCoreCallbackIfWrapper().doOnDeviceConnectionStatusListChanged(bmCoreIfMessage->getDeviceConnectionStatusList());
   }

   void BmCoreMainController::handleBmCoreIfMessage_SetFavoriteRequest(BmCoreIfMessage_SetFavoriteRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SetFavoriteRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      DeviceId deviceId;
      Result result(CC_ERR_INT_NO_ERROR);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         // check if given deviceHandle is valid
         result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

         if (CC_ERR_INT_NO_ERROR != result)
         {
            if (CC_ERR_INT_DB_END_OF_LIST == result)
            {
               // no device with given device handle found in DB
               ETG_TRACE_ERR(("handleBmCoreIfMessage_SetFavoriteRequest: invalid value for parameter deviceHandle (error = %d)",
                     ETG_CENUM(CcErrorInternal, CC_ERR_INT_INVALID_PARAMETER)));

               bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_SetFavoriteRequest: could not get device ID from DB (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));

               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }

         if (BM_RESULT_OK == bmResult)
         {
            if (false == isValidFavoriteType(bmCoreIfMessage->getFavoriteType()))
            {
               // invalid value for given favoriteType
               ETG_TRACE_ERR(("handleBmCoreIfMessage_SetFavoriteRequest: invalid value for parameter favoriteType (error = %d)",
                     ETG_CENUM(CcErrorInternal, CC_ERR_INT_INVALID_PARAMETER)));

               bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_SetFavoriteRequest: basic check failed (error = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request
         result = LocalSpm::getDbManager().SetDeviceFavorite(deviceId, bmCoreIfMessage->getFavoriteType(),
               bmCoreIfMessage->getFavoriteIndex());

         if (CC_ERR_INT_NO_ERROR == result)
         {
            this->onDeviceBaseInfoChanged();
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SetFavoriteRequest: could not set device as favorite in DB (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));

            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doSetFavoriteResponse(bmResult, bmCoreIfMessage->getAct());
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_ClearFavoriteRequest(BmCoreIfMessage_ClearFavoriteRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_ClearFavoriteRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);
      DeviceId deviceId;
      Result result(CC_ERR_INT_NO_ERROR);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check
         result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

         if (CC_ERR_INT_NO_ERROR != result)
         {
            result = CC_ERR_INT_INVALID_PARAMETER;
            ETG_TRACE_ERR(("handleBmCoreIfMessage_ClearFavoriteRequest: invalid value for parameter deviceHandle (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));

            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_ClearFavoriteRequest: basic check failed (error = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request
         result = LocalSpm::getDbManager().SetDeviceFavorite(deviceId, bmCoreIfMessage->getFavoriteType(),0);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            this->onDeviceBaseInfoChanged();
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_ClearFavoriteRequest: could not clear favorite device in DB (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doClearFavoriteResponse(bmResult, bmCoreIfMessage->getAct());
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_SetPrimaryHfpDeviceRequest(BmCoreIfMessage_SetPrimaryHfpDeviceRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SetPrimaryHfpDeviceRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId;

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check
         result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

         if (CC_ERR_INT_NO_ERROR != result)
         {
            result = CC_ERR_INT_INVALID_PARAMETER;
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SetPrimaryHfpDeviceRequest: invalid value for parameter deviceHandle (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));

            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_SetPrimaryHfpDeviceRequest: basic check failed (error = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request
         result = LocalSpm::getDbManager().SetPrimaryHfpDevice(deviceId);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            DeviceConnectionStatusList deviceConnectionStatusList;
            _deviceConnectionStatusListPropHdl.get(deviceConnectionStatusList);

            getPrimaryStatus(deviceConnectionStatusList);

            _deviceConnectionStatusListPropHdl.set(deviceConnectionStatusList);
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SetPrimaryHfpDeviceRequest: could not set primary HFP device(error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         LocalSpm::getBmCoreCallbackIfWrapper().doSetPrimaryHfpDeviceResponse(bmResult);
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_StartServiceSearchRequest(BmCoreIfMessage_StartServiceSearchRequest* bmCoreIfMessage )
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_StartServiceSearch: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if(BM_CORE_IF_MSG_ORIGIN_INTERNAL == bmCoreIfMessage->getOrigin())
         {
            // Internal message of StartServiceSerch contains deviceId only
            deviceId = bmCoreIfMessage->getDeviceHandle();
         }
         else
         {
            result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_StartServiceSearch: could not get device ID for given device handle = %d from DB (error = %d)",
                     bmCoreIfMessage->getDeviceHandle(), ETG_CENUM(CcErrorInternal, result)));

               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_StartServiceSearch: general request check failed (bmResult = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request
         ConnectionStatus connectionStatus;
         DisconnectedReason disconnectedReason;

         result = LocalSpm::getDbManager().getDeviceConnectionStatus(connectionStatus, disconnectedReason, deviceId);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR1(("handleBmCoreIfMessage_StartServiceSearch: found a DCC already assigned to device with given device ID = %d with device connection status = %d",
                  deviceId, ETG_CENUM(ConnectionStatus, connectionStatus)));

            if ((BM_CONNECTION_STATUS_CONNECTING != connectionStatus)
                  && (BM_CONNECTION_STATUS_CONNECTED != connectionStatus))
            {
               bmResult = BM_RESULT_ERR_DEVICE_DISCONNECTING;
            }
            else
            {
               if (0u < LocalSpm::getDataProvider().getBmCoreConfiguration()._serviceSearchTimeoutSeconds)
               {
                  result = _serviceSearchController.addDeviceForServiceSearch(deviceId);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     bmResult = startDeviceServiceSearch(deviceId);
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_StartServiceSearch: could not add the device in service search list (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                     bmResult = BM_RESULT_ERR_GENERAL;
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_StartServiceSearch: _serviceSearchTimeoutSeconds is = %d",
                        LocalSpm::getDataProvider().getBmCoreConfiguration()._serviceSearchTimeoutSeconds));
                  bmResult = BM_RESULT_ERR_GENERAL;
               }
            }
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doStartServiceSearchResponse(bmResult, bmCoreIfMessage->getAct());
      }
   }


   void BmCoreMainController::handleBmCoreIfMessage_StopServiceSearchRequest(BmCoreIfMessage_StopServiceSearchRequest* bmCoreIfMessage )
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_StopServiceSearch: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_StopServiceSearch: could not get device ID for given device handle = %d from DB (error = %d)",
                  bmCoreIfMessage->getDeviceHandle(), ETG_CENUM(CcErrorInternal, result)));

            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_StopServiceSearch: general request check failed (bmResult = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // Stop the service search and remove the device in Service Search list
         result = stopDeviceServiceSearch(deviceId);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_StopServiceSearch: could not stop service search (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doStopServiceSearchResponse(bmResult, bmCoreIfMessage->getAct());
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_UpdateLinkQualityRequest(BmCoreIfMessage_UpdateLinkQualityRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_UpdateLinkQualityRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      DeviceIdList connectedDeviceHandles;
      this->getConnectedDevices(connectedDeviceHandles);

      DeviceIdList connectedDevicesWaitingForLinkQuality = this->getConnectedDevicesWaitingForLinkQuality();

      ETG_TRACE_USR4(("handleBmCoreIfMessage_UpdateLinkQualityRequest: current waiting list and intermediate link qualities:"));

      VARTRACE(connectedDevicesWaitingForLinkQuality);
      VARTRACE(this->getIntermediateLinkQuality());

      if (bmCoreIfMessage->getOrigin() == BM_CORE_IF_MSG_ORIGIN_CLIENT)
      {
         _numberPendingUpdateLinkQualityRequests++;

         ETG_TRACE_USR4(("handleBmCoreIfMessage_UpdateLinkQualityRequest: incremented number of pending updateLinkQuality requests to %d",
               _numberPendingUpdateLinkQualityRequests));
      }

      for (size_t idx = 0u; idx < connectedDeviceHandles.size(); idx++)
      {
         if (connectedDevicesWaitingForLinkQuality.end() == find(connectedDevicesWaitingForLinkQuality.begin(),
               connectedDevicesWaitingForLinkQuality.end(), connectedDeviceHandles[idx]))
         {
            // the connected device is not in the list of connected devices for which the link quality has already
            // been requested

            ETG_TRACE_USR4(("handleBmCoreIfMessage_UpdateLinkQualityRequest: found connected device with device handle = %d for which link quality has not yet been requested",
                  connectedDeviceHandles[idx]));

            BdAddress bdAddress("");
            Result result = LocalSpm::getDbManager().getBdAddressByDeviceHandle(bdAddress, connectedDeviceHandles[idx]);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               ETG_TRACE_USR4(("handleBmCoreIfMessage_UpdateLinkQualityRequest: requesting link quality from BtStackIf for device with BD address = \"%50s\"",
                     bdAddress.c_str()));

               LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().requestLinkQuality(static_cast<BTSBDAddress>(bdAddress));

               // remember the new connected device the link quality has been successfully requested for
               this->addConnectedDeviceToWaitingForLinkQuality(connectedDeviceHandles[idx]);
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_UpdateLinkQualityRequest: could not get BD address for device handle = %d from DB (error = %d)",
                     connectedDeviceHandles[idx], ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_UpdateLinkQualityRequest: already requested link quality for device with device handle = %d, still waiting for the response",
                  connectedDeviceHandles[idx]));
         }
      }

      ETG_TRACE_USR4(("handleBmCoreIfMessage_UpdateLinkQualityRequest: updated waiting list and intermediate link qualities:"));

      VARTRACE(connectedDevicesWaitingForLinkQuality);
      VARTRACE(this->getIntermediateLinkQuality());

      if (true == connectedDevicesWaitingForLinkQuality.empty())
      {
         ETG_TRACE_USR4(("handleBmCoreIfMessage_UpdateLinkQualityRequest: waiting list is empty, responding %d updateLinkQuality request(s)",
               _numberPendingUpdateLinkQualityRequests));

         for (unsigned int i = 0u; i < _numberPendingUpdateLinkQualityRequests; i++)
         {
            LocalSpm::getBmCoreCallbackIfWrapper().doUpdateLinkQualityResponse(BM_RESULT_OK);
         }

         _numberPendingUpdateLinkQualityRequests = 0u;
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BlockStatusUpdate(BmCoreIfMessage_BlockStatusUpdate* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BlockStatusUpdate: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      LocalSpm::getBmCoreCallbackIfWrapper().doOnBlockStatusChanged(bmCoreIfMessage->getBlockStatus());
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_SetBtLimitationModeRequest(BmCoreIfMessage_SetBtLimitationModeRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessageSetBtLimitationModeRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);

      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // Accept the request(user/stack) only if systemstate is normal(exception of DEACTIVATE & DEACTIVATE_SPM_STATE_OFF).
         BtSystemState btSystemState;
         (void) getBtSystemState(btSystemState);

         BlockStatus blockStatus;
         (void) this->getBlockStatus(blockStatus);

         BmCoreIfMsgOrigin bmCoreMsgOrigin = bmCoreIfMessage->getOrigin();
         LimitationAction bmCoreMsgLimiAction = bmCoreIfMessage->getLimitationAction();

         if ((BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreMsgOrigin) &&
            ((BM_BT_SYSTEM_STATE_BLOCK == btSystemState._bluetoothSystemState) ||
            (BM_BT_SYSTEM_STATE_OFF == btSystemState._bluetoothSystemState)))
         {
            if(BM_LIMITATION_ACTION_DEACTIVATE == bmCoreMsgLimiAction)
            {
               ETG_TRACE_USR4(("handleBmCoreIfMessageSetBtLimitationModeRequest: Accepting DEACTIVATE request when system state is not normal"));
               bmCoreIfMessage->setLimitationAction(BM_LIMITATION_ACTION_DEACTIVATE_INTERNAL);
            }
            else if(BM_LIMITATION_ACTION_DEACTIVATE_SPM_STATE_OFF == bmCoreMsgLimiAction)
            {
               ETG_TRACE_USR4(("handleBmCoreIfMessageSetBtLimitationModeRequest: Accepting DEACTIVATE_SPM_STATE_OFF request when system state is not normal"));
            }
            else
            {
               ETG_TRACE_USR4(("handleBmCoreIfMessageSetBtLimitationModeRequest: system state is not normal so not accept the request"));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
         // For suzuki, While E-Call is ongoing, it is possible user might plug the CP(USB)
         // In that above case, Request is not allowed
         else if((BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreMsgOrigin) && ((blockStatus._overallDeviceBlockStatus._blockState == BM_OVERALL_BLOCK_STATE_BLOCKED) &&
               (BM_LIMITATION_ACTION_ACTIVATE == bmCoreMsgLimiAction)))
         {
            ETG_TRACE_USR4(("handleBmCoreIfMessageSetBtLimitationModeRequest: overall device block state is blocked"));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }//!End of if (BM_RESULT_OK == bmResult)
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessageSetBtLimitationModeRequest: basic check failed (error = %d)", ETG_CENUM(BmResult, bmResult)));
      }//!End of else

      if(BM_RESULT_OK == bmResult)
      {
         bmResult = LocalSpm::getBtLimitationController().validateBTLimitationModeRequest(bmCoreIfMessage);
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         LocalSpm::getBmCoreCallbackIfWrapper().doSetBtLimitationModeResponse(bmResult, bmCoreIfMessage->getAct());
      }//!End of if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())

      if(BM_RESULT_OK == bmResult)
      {
         result = LocalSpm::getBtLimitationController().sendLimitationActionSmEvent("handleBmCoreIfMessageSetBtLimitationModeRequest",
               bmCoreIfMessage->getLimitationAction());

         if (CC_ERR_INT_NO_ERROR != result)
         {
            bmResult = BM_RESULT_ERR_GENERAL;
            FW_NORMAL_ASSERT_ALWAYS();
         }//!End of if (CC_ERR_INT_NO_ERROR != result)
      }//!End of if (BM_RESULT_OK == bmResult)

      return bmResult;
   }//!End of BmResult BmCoreMainController::handleBmCoreIfMessage_SetBtLimitationModeRequest(..)

   BmResult BmCoreMainController::handleBmCoreIfMessage_ReplaceBtLimitationModeRequest(BmCoreIfMessage_ReplaceBtLimitationModeRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_ReplaceBtLimitationModeRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         BtLimitationModeInfo btLimitationModeInfo;
         BtLimitationModeInfo btLimitationModeToBeReplaced = bmCoreIfMessage->getLimitationModeToBeReplaced();
         BdAddress bdAddress(btLimitationModeToBeReplaced._bdAddress);
         convertBdAddress2LowerCase(bdAddress);
         (void)getBtLimitationModeInfo(btLimitationModeInfo, bdAddress, btLimitationModeToBeReplaced._bdName);

         //!Check if the LimitationModeToBeReplaced is presently active else send an error. Cannot replace a limitation mode which is not active.
         if((btLimitationModeInfo._bdAddress == bdAddress)
               && (btLimitationModeInfo._bdName == btLimitationModeToBeReplaced._bdName)
               && (btLimitationModeInfo._limitationMode._limitationFeature == btLimitationModeToBeReplaced._limitationMode._limitationFeature)
               && (btLimitationModeInfo._limitationMode._limitationCommunicationIf == btLimitationModeToBeReplaced._limitationMode._limitationCommunicationIf))
         {
            //!End the current active Limitation Mode
            result = LocalSpm::getBtLimitationController().sendLimitationActionSmEvent("handleBmCoreIfMessageReplaceBtLimitationModeRequest",
                  BM_LIMITATION_ACTION_DEACTIVATE_WAIT);

              if (CC_ERR_INT_NO_ERROR != result)
              {
                 bmResult = BM_RESULT_ERR_GENERAL;
                 FW_NORMAL_ASSERT_ALWAYS();
              }//!End of if (CC_ERR_INT_NO_ERROR != result)

              //!Validate the new Limitation Mode
              bmResult = (BM_RESULT_OK == bmResult) ?
                    (LocalSpm::getBtLimitationController().validateReplaceBTLimitationModeRequest(bmCoreIfMessage)) : (bmResult);
              if(BM_RESULT_OK != bmResult)
              {
                 //!If validation fails end the Limitation Mode which will be in Waiting state. SM will go to IDLE state
                 result = LocalSpm::getBtLimitationController().sendLimitationActionSmEvent("handleBmCoreIfMessageReplaceBtLimitationModeRequest",
                       BM_LIMITATION_ACTION_DEACTIVATE);

                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     bmResult = BM_RESULT_ERR_GENERAL;
                     FW_NORMAL_ASSERT_ALWAYS();
                  }//!End of if (CC_ERR_INT_NO_ERROR != result)
              }//!End of if(BM_RESULT_OK != bmResult)
         }//!End of if((btLimitationModeInfo._bdAddress == btLimitationModeToBeReplaced._bdAddress) ...)
         else
         {
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
         }//!End of else if(BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      }//!End of if (BM_RESULT_OK == bmResult)
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_ReplaceBtLimitationModeRequest: basic check failed (error = %d)", ETG_CENUM(BmResult, bmResult)));
      }//!End of else

      if(BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         LocalSpm::getBmCoreCallbackIfWrapper().doReplaceBtLimitationModeResponse(bmResult, bmCoreIfMessage->getAct());
      }//!End of if(BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())

      if(BM_RESULT_OK == bmResult)
      {
          result = LocalSpm::getBtLimitationController().sendLimitationActionSmEvent("handleBmCoreIfMessageReplaceBtLimitationModeRequest",
                bmCoreIfMessage->getNewLimitationAction());

          if (CC_ERR_INT_NO_ERROR != result)
          {
             bmResult = BM_RESULT_ERR_GENERAL;
             FW_NORMAL_ASSERT_ALWAYS();
          }//!End of if (CC_ERR_INT_NO_ERROR != result)
      }//!End of if(BM_RESULT_OK == bmResult)

      return bmResult;
   }//!End of BmResult BmCoreMainController::handleBmCoreIfMessage_ReplaceBtLimitationModeRequest(...)

   void BmCoreMainController::handleBmCoreIfMessage_BtLimitationModeUpdate(BmCoreIfMessage_BtLimitationModeUpdate* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessageBtLimitationModeUpdate: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      LocalSpm::getBmCoreCallbackIfWrapper().doOnBtLimitationModeChanged(bmCoreIfMessage->getBtLimitationMode());
   }

   void BmCoreMainController::handleBmCoreIfMessage_NotifyUserDecisionRequest(BmCoreIfMessage_NotifyUserDecisionRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_NotifyUserDecisionRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         bmResult = LocalSpm::getConflictManager().processNotifyUserDecisionRequest(bmCoreIfMessage);
         LocalSpm::getBmCoreCallbackIfWrapper().doNotifyUserDecisionResponse(bmResult, bmCoreIfMessage->getAct());
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_NotifyUserDecisionRequest: basic check failed (error = %d)", ETG_CENUM(BmResult, bmResult)));
         LocalSpm::getBmCoreCallbackIfWrapper().doNotifyUserDecisionResponse(bmResult, bmCoreIfMessage->getAct());
      }

      //!Once one conflict is resolved, check if there are any other pending conflicts to be resolved meanwhile.
      //!This can be removed once BMApp Interface updates multiple conflicts concurrently
      if(BM_RESULT_OK == bmResult)
      {
         checkForPendingIssuesToBeResolved();
      }//!End of if(BM_RESULT_OK == bmResult)
   }

   void BmCoreMainController::checkForPendingIssuesToBeResolved()
   {
      size_t pendingIssueListSize = _pendingIssuesToBeResolvedList.size();
      ETG_TRACE_USR1(("checkForPendingIssuesToBeResolved: PendingIssueListSize = %d", pendingIssueListSize));

      if(0 < pendingIssueListSize)
      {
         IssueInfoList pendingIssueInfoList = *(_pendingIssuesToBeResolvedList.begin());
         (void)updateBtConflictsDetectedList(pendingIssueInfoList);
         _pendingIssuesToBeResolvedList.erase( _pendingIssuesToBeResolvedList.begin());
      }//!End of if(0 < pendingIssueListSize)
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_SwitchTestModeRequest(BmCoreIfMessage_SwitchTestModeRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SwitchTestModeRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check
         if ((false == isValidTargetSwitchState(bmCoreIfMessage->getTargetSwitchState()))
               || (false == isValidBdAddress(bmCoreIfMessage->getBdAddress())))
         {
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchTestModeRequest: invalid parameter value (bmResult = %d)",
                  ETG_CENUM(BmResult, bmResult)));
         }
         else if(false == LocalSpm::getDataProvider().getBmCoreConfiguration()._btOnOffSupport)
         {
            bmResult = BM_RESULT_ERR_NOT_ALLOWED;

            ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchTestModeRequest: btOnOffSupport is OFF So not accepting the test mode request(bmResult = %d)",
                  ETG_CENUM(BmResult, bmResult)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchTestModeRequest: general request check failed (bmResult = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      if (BM_RESULT_OK != bmResult)
      {
         // BM Core will not handle the request
         if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
         {
            // respond to client's request
            LocalSpm::getBmCoreCallbackIfWrapper().doSwitchTestModeResponse(bmResult, bmCoreIfMessage->getAct());
         }
      }
      else
      {
         // handle the request, responding to a client's request is done in property handler

         BdAddress lcBdAddress(bmCoreIfMessage->getBdAddress());
         convertBdAddress2LowerCase(lcBdAddress);

         _testModePropHdl.setTargetSwitchState(bmCoreIfMessage->getTargetSwitchState(), lcBdAddress);
      }

      return bmResult;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_ResetToDefaultRequest(BmCoreIfMessage_ResetToDefaultRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_ResetToDefaultRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         (void) this->updateFunctionalityRestrictionInfo(BM_RESTRICTION_BIT_RESET_TO_DEFAULT_IN_PROGRESS, true);

         Result result = LocalSpm::getResetToDefaultController().resetToDefault(bmCoreIfMessage->getFactorySettingsChanged());

         if (CC_ERR_INT_NO_ERROR == result)
         {
            _resetToFactorySettingsRequested = bmCoreIfMessage->getFactorySettingsChanged();
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_ResetToDefaultRequest: initiating the resetting to default failed (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_ResetToDefaultRequest: general request check failed (bmResult = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doResetToDefaultResponse(bmResult, bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsFbConnectionInitialized(BmCoreIfMessage_BtsFbConnectionInitialized* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsFbConnectionInitialized: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      std::string eventName("BTSTACKIF_INITIALIZED");
      bool stackInitialized = true;

      if (BTS_REQ_SUCCESS != bmCoreIfMessage->getRequestResult())
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsFbConnectionInitialized: BtStackIf initialization failed"));

         eventName = "BTSTACKIF_INITIALIZATION_ERROR";

         stackInitialized = false;
      }

      // update "property" LocalInfo
      LocalInfo localInfo;
      _localInfoPropHdl.get(localInfo);

      localInfo._stackInitialized = stackInitialized;
      _localInfoPropHdl.set(localInfo);

      // stop the BT ON delay timer and send the BT ON request to stack
      this->stopBtOnDelayTimerAndSendReq();

      ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsFbConnectionInitialized: sending event %50s to BmControllerOnOffSm",
            eventName.c_str()));

      Result result = LocalSpm::getBmController().SendEventByName(eventName.c_str(), (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsFbConnectionInitialized: could not send event %50s (error = %d)",
               eventName.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsLocalVersionInfo(BmCoreIfMessage_BtsLocalVersionInfo* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsLocalVersionInfo: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsSwitchBluetoothOnOffResult(BmCoreIfMessage_BtsSwitchBluetoothOnOffResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSwitchBluetoothOnOffResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      _btStatusSwitchPropHdl.onExternalSwitchingResponse(BTS_REQ_SUCCESS == bmCoreIfMessage->getRequestResult() ? CC_ERR_INT_NO_ERROR : CC_ERR_INT_BTS_ERROR);
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsCurrentBluetoothOnOffStatus(BmCoreIfMessage_BtsCurrentBluetoothOnOffStatus* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsCurrentBluetoothOnOffStatus: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);

      result = LocalSpm::getDbManager().setLocalBdAddress(static_cast<BdAddress>(bmCoreIfMessage->getLocalDeviceAddress()));

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsCurrentBluetoothOnOffStatus: storing local BD address to DB failed (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      LocalInfo localInfo;
      _localInfoPropHdl.get(localInfo);

      localInfo._bdAddress = static_cast<BdAddress>(bmCoreIfMessage->getLocalDeviceAddress());
      _localInfoPropHdl.set(localInfo);

      SwitchState switchState(SWITCH_STATE_SWITCHED_OFF);
      result = getBmSwitchStateFromBtsBluetoothMode(switchState, bmCoreIfMessage->getBluetoothMode());

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsCurrentBluetoothOnOffStatus: could not map given BtStackIf's BTSBluetoothMode = %d onto a corresponding BM Core's SwitchState (error = %d)",
               ETG_CENUM(btstackif::BTSBluetoothMode, bmCoreIfMessage->getBluetoothMode()),
               ETG_CENUM(CcErrorInternal, result)));
      }

      SwitchedOffReason switchedOffReason(SWITCHED_OFF_REASON_INTERNAL);
      result = getBmSwitchedOffReasonFromBtsBluetoothOffReason(switchedOffReason, bmCoreIfMessage->getBluetoothOffReason());

      // Ignore the switch state update if BMApp is waiting for stack initialization
      if(false == _waitingForEvoStackRestart)
      {
         _btStatusSwitchPropHdl.onExternalSwitchStateUpdate(switchState, switchedOffReason);
      }
      else
      {
         ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsCurrentBluetoothOnOffStatus: Waiting for stack to be initialized so ignore the update "));
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsBluetoothHardwareOnOffResult(BmCoreIfMessage_BtsBluetoothHardwareOnOffResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsBluetoothHardwareOnOffResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsSetLocalBtNameResult(BmCoreIfMessage_BtsSetLocalBtNameResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSetLocalBtNameResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsCurrentLocalBtName(BmCoreIfMessage_BtsCurrentLocalBtName* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsCurrentLocalBtName: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      LocalFriendlyName localFriendlyName;

      if (false == _resetToFactorySettingsRequested)
      {
         if (true == isValidLocalBdName(static_cast<BdName>(bmCoreIfMessage->getAdapterName())))
         {
            result = LocalSpm::getDbManager().getLocalBdName(localFriendlyName._localFriendlyName);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsCurrentLocalBtName: BD name stored in DB is \"%50s\"", localFriendlyName._localFriendlyName.c_str()));

               if (true == isValidLocalBdName(localFriendlyName._localFriendlyName))
               {
                  result = LocalSpm::getDbManager().setLocalBdName(static_cast<BdName>(bmCoreIfMessage->getAdapterName()));

                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsCurrentLocalBtName: storing local BD name to DB failed (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                  }
               }

               _localFriendlyNamePropHdl.get(localFriendlyName);

               localFriendlyName._localFriendlyName = static_cast<BdName>(bmCoreIfMessage->getAdapterName());
               _localFriendlyNamePropHdl.set(localFriendlyName, false);

               LocalInfo localInfo;
               _localInfoPropHdl.get(localInfo);

               localInfo._bdName = static_cast<BdName>(bmCoreIfMessage->getAdapterName());
               _localInfoPropHdl.set(localInfo, false);
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsCurrentLocalBtName: getting local BD name from DB failed (error = %d)", result));
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsCurrentLocalBtName: BD name is not valid"));
         }
      }
      else
      {
         ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsCurrentLocalBtName: reset to factory settings requested - ignoring local BD name update"));
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsSetLocalAdapterModesResult(BmCoreIfMessage_BtsSetLocalAdapterModesResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSetLocalAdapterModesResult: bmCoreIfMessage = 0x%p, ActType = %d", (void *) bmCoreIfMessage,
            bmCoreIfMessage->getAct()));

      if(BM_REQUEST_TYPE_PAIRABLE_MODE == bmCoreIfMessage->getAct())
      {
         _localPairableModeSwitchPropHdl.onExternalSwitchingResponse(BTS_REQ_SUCCESS == bmCoreIfMessage->getRequestResult() ? CC_ERR_INT_NO_ERROR : CC_ERR_INT_BTS_ERROR);
      }
      else if(BM_REQUEST_TYPE_CONNECTABLE_MODE == bmCoreIfMessage->getAct())
      {
         _localConnectableModeSwitchPropHdl.onExternalSwitchingResponse(BTS_REQ_SUCCESS == bmCoreIfMessage->getRequestResult() ? CC_ERR_INT_NO_ERROR : CC_ERR_INT_BTS_ERROR);
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSetLocalAdapterModesResult: Act type is not valid"));
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsCurrentConnectableMode(BmCoreIfMessage_BtsCurrentConnectableMode* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsCurrentConnectableMode: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      SwitchState switchState(SWITCH_STATE_SWITCHED_OFF);
      Result result = getBmSwitchStateFromBtsLocalMode(switchState, bmCoreIfMessage->getConnectableMode());

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsCurrentConnectableMode: could not map given BtStackIf's BTSLocalMode = %d onto a corresponding BM Core's SwitchState (error = %d)",
               ETG_CENUM(btstackif::BTSLocalMode, bmCoreIfMessage->getConnectableMode()),
               ETG_CENUM(CcErrorInternal, result)));
      }

      SwitchedOffReason switchedOffReason(SWITCHED_OFF_REASON_INTERNAL);
      result = getBmSwitchedOffReasonFromBtsLocalModeOffReason(switchedOffReason, bmCoreIfMessage->getOffReason());

      _localConnectableModeSwitchPropHdl.onExternalSwitchStateUpdate(switchState, switchedOffReason);
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsCurrentDiscoverableMode(BmCoreIfMessage_BtsCurrentDiscoverableMode* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsCurrentDiscoverableMode: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      SwitchState switchState(SWITCH_STATE_SWITCHED_OFF);
      Result result = getBmSwitchStateFromBtsLocalMode(switchState, bmCoreIfMessage->getDiscoverableMode());

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsCurrentDiscoverableMode: could not map given BtStackIf's BTSLocalMode = %d onto a corresponding BM Core's SwitchState (error = %d)",
               ETG_CENUM(btstackif::BTSLocalMode, bmCoreIfMessage->getDiscoverableMode()),
               ETG_CENUM(CcErrorInternal, result)));
      }

      SwitchedOffReason switchedOffReason(SWITCHED_OFF_REASON_INTERNAL);
      result = getBmSwitchedOffReasonFromBtsLocalModeOffReason(switchedOffReason, bmCoreIfMessage->getOffReason());

      _localPairableModeSwitchPropHdl.onExternalSwitchStateUpdate(switchState, switchedOffReason);
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsStartDiscoveryResult(BmCoreIfMessage_BtsStartDiscoveryResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsStartDiscoveryResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      _discoveryStatusPropHdl.onExternalSwitchingResponse((bmCoreIfMessage->getRequestResult() == BTS_REQ_SUCCESS) ? CC_ERR_INT_NO_ERROR : CC_ERR_INT_BTS_ERROR);
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsStopDiscoveryResult(BmCoreIfMessage_BtsStopDiscoveryResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsStopDiscoveryResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      _discoveryStatusPropHdl.onExternalSwitchingResponse((bmCoreIfMessage->getRequestResult() == BTS_REQ_SUCCESS) ? CC_ERR_INT_NO_ERROR : CC_ERR_INT_BTS_ERROR);
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsDiscoveringStatus(BmCoreIfMessage_BtsDiscoveringStatus* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsDiscoveringStatus: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      SwitchState discoverySwitchState(SWITCH_STATE_SWITCHED_OFF);
      Result result = getBmSwitchStateFromBtsDiscoveringStatus(discoverySwitchState, bmCoreIfMessage->getDiscoveringStatus());

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsDiscoveringStatus: updating DiscoveryStatus switchState to %d",
               ETG_CENUM(cc::SwitchState, discoverySwitchState)));

         _discoveryStatusPropHdl.onExternalSwitchStateUpdate(discoverySwitchState);
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsDiscoveringStatus: could not map BT Stack IF's discovering status to a corresponding BM core switch state (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsDiscoveredDeviceFound(BmCoreIfMessage_BtsDiscoveredDeviceFound* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsDiscoveredDeviceFound: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      DiscoveredDeviceInfo discoveredDeviceInfo;
      DeviceId deviceHandle(0u);

      Result result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, static_cast<BdAddress>(bmCoreIfMessage->getDiscoveredDevice().address));

      if (CC_ERR_INT_NO_ERROR == result)
      {
         // device is already in paired device list

         discoveredDeviceInfo._deviceHandle = deviceHandle;

         DisconnectedReason disconnectedReason(BM_DISCONNECTED_REASON_UNKNOWN);
         result = LocalSpm::getDbManager().getDeviceConnectionStatus(discoveredDeviceInfo._connectionStatus,
               disconnectedReason, static_cast<BdAddress>(bmCoreIfMessage->getDiscoveredDevice().address));

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsDiscoveredDeviceFound: could not get connection status for device with given BD address from DB (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));

            discoveredDeviceInfo._connectionStatus = BM_CONNECTION_STATUS_DISCONNECTED;
         }
      }
      else if (CC_ERR_INT_DB_END_OF_LIST == result)
      {
         // device is not in paired device list

         result = CC_ERR_INT_NO_ERROR;

         discoveredDeviceInfo._deviceHandle = 0u;
         discoveredDeviceInfo._connectionStatus = BM_CONNECTION_STATUS_DISCONNECTED;
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsDiscoveredDeviceFound: could not get device handle for given BD address from DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      if (CC_ERR_INT_NO_ERROR == result)
      {
         discoveredDeviceInfo._bdAddress = static_cast<BdAddress>(bmCoreIfMessage->getDiscoveredDevice().address);
         discoveredDeviceInfo._bdName = static_cast<BdName>(bmCoreIfMessage->getDiscoveredDevice().name);
         discoveredDeviceInfo._majorDeviceClass = bmCoreIfMessage->getDiscoveredDevice().majorDeviceClass;
         discoveredDeviceInfo._minorDeviceClass = bmCoreIfMessage->getDiscoveredDevice().minorDeviceClass;

         ProtocolInfoMap localSupportedProtocols;
         result = LocalSpm::getDataProvider().getProtocolInfos(localSupportedProtocols, false, false, true);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            BTSSupportedServicesBit btsSupportedServicesBitLocalProtocol;

            // read protocol support info from discovered remote device info

            ProtocolInfoMap::iterator itLocal;
            BTSServiceInfoList::const_iterator itRemote;

            for (itLocal = localSupportedProtocols.begin(); itLocal != localSupportedProtocols.end(); ++itLocal)
            {
               result = getBtsSupportedServicesBitFromBmProtocolId(btsSupportedServicesBitLocalProtocol, itLocal->first);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  discoveredDeviceInfo._protocolInfo[itLocal->first]._version = 0x0u;

                  if (true == bmCoreIfMessage->getDiscoveredDevice().services.getBit(btsSupportedServicesBitLocalProtocol))
                  {
                     // locally supported protocol is also supported by discovered remote device

                     discoveredDeviceInfo._protocolInfo[itLocal->first]._isSupported = true;
                  }
                  else
                  {
                     // locally supported protocol is not supported by discovered remote device

                     discoveredDeviceInfo._protocolInfo[itLocal->first]._isSupported = false;
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsDiscoveredDeviceFound: could not get BtStackIf's supported services bit from BM Core's protocol ID (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsDiscoveredDeviceFound: could not get locally supported protocols (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }

         DiscoveredDeviceList discoveredDeviceList;
         _discoveredDeviceListPropHdl.get(discoveredDeviceList);

         DiscoveredDeviceInfoList::iterator it = std::find_if(discoveredDeviceList._discoveredDeviceInfoList.begin(),
               discoveredDeviceList._discoveredDeviceInfoList.end(),
               MeetsBdAddress<DiscoveredDeviceInfo>(discoveredDeviceInfo._bdAddress));

         struct timespec monotonicTime;
         clock_gettime(CLOCK_MONOTONIC, &monotonicTime);

         discoveredDeviceInfo._lastDiscoveredTimestamp = monotonicTime.tv_sec;

         if (it != discoveredDeviceList._discoveredDeviceInfoList.end())
         {
            // device is already in discovered device list, update data

            it->_deviceHandle = discoveredDeviceInfo._deviceHandle;
            it->_bdName = discoveredDeviceInfo._bdName;
            it->_majorDeviceClass = discoveredDeviceInfo._majorDeviceClass;
            it->_minorDeviceClass = discoveredDeviceInfo._minorDeviceClass;
            it->_protocolInfo = discoveredDeviceInfo._protocolInfo;
            it->_lastDiscoveredTimestamp = discoveredDeviceInfo._lastDiscoveredTimestamp;
         }
         else
         {
            // device is not in discovered device list
            discoveredDeviceList._discoveredDeviceInfoList.push_back(discoveredDeviceInfo);
         }

         _discoveredDeviceListPropHdl.set(discoveredDeviceList);
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsStartPairingResult(BmCoreIfMessage_BtsStartPairingResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsStartPairingResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      if (BTS_REQ_SUCCESS != bmCoreIfMessage->getRequestResult())
      {
         char eventParams[40];

         Result result = LocalSpm::getPairingController().ParameterPAIRING_FINISHED(OUT eventParams,
               IN sizeof(eventParams), IN bmCoreIfMessage->getBdAddress().c_str(),
               IN BM_PAIRING_RESULT_ERR_START_PAIRING_FAILED);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsStartPairingResult: sending event PAIRING_FINISHED to PairingSm"));
            result = LocalSpm::getPairingController().SendEventByName("PAIRING_FINISHED", eventParams);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsStartPairingResult: could not send event PAIRING_FINISHED (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsStartPairingResult: could not marshal event parameters for event PAIRING_FINISHED (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsPinCodeRequested(BmCoreIfMessage_BtsPinCodeRequested* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsPinCodeRequested: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         if (BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE == bmCoreIfMessage->getPairingOriginator())
         {
            // pairing has been initiated by local device

            char eventParams[200];

            result = LocalSpm::getPairingController().ParameterPAIRING_CONFIRMATION_REQUESTED(OUT eventParams,
                  IN sizeof(eventParams), IN static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()).c_str(),
                  IN static_cast<BdName>(bmCoreIfMessage->getBdName()).c_str(), IN BM_PAIRING_TYPE_LEGACY, IN "");

            if (CC_ERR_INT_NO_ERROR == result)
            {
               ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsPinCodeRequested: sending event PAIRING_CONFIRMATION_REQUESTED to PairingControllerSm"));

               result = LocalSpm::getPairingController().SendEventByName("PAIRING_CONFIRMATION_REQUESTED", eventParams);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPinCodeRequested: could not send event PAIRING_CONFIRMATION_REQUESTED (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
                  bmResult = BM_RESULT_ERR_GENERAL;
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPinCodeRequested: could not marshal event parameters for event PAIRING_CONFIRMATION_REQUESTED (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
         else
         {
            // pairing initiated by remote device

            SwitchStatus localPairableMode;
            bmResult = this->getLocalPairableMode(localPairableMode);

            if (BM_RESULT_OK == bmResult)
            {
               if (SWITCH_STATE_SWITCHED_ON == localPairableMode._switchState)
               {
                  char eventParams[200];

                  result = LocalSpm::getPairingController().ParameterPAIRING_CONFIRMATION_REQUESTED(OUT eventParams,
                        IN sizeof(eventParams), IN (static_cast<BdAddress>(bmCoreIfMessage->getBdAddress())).c_str(),
                        IN (static_cast<BdName>(bmCoreIfMessage->getBdName())).c_str(), IN BM_PAIRING_TYPE_LEGACY,
                        IN "");

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsPinCodeRequested: sending event PAIRING_CONFIRMATION_REQUESTED to PairingControllerSm"));

                     result = LocalSpm::getPairingController().SendEventByName("PAIRING_CONFIRMATION_REQUESTED", eventParams);

                     if (CC_ERR_INT_NO_ERROR != result)
                     {
                        ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPinCodeRequested: could not send event PAIRING_CONFIRMATION_REQUESTED (error = %d)",
                              ETG_CENUM(CcErrorInternal, result)));
                        bmResult = BM_RESULT_ERR_GENERAL;
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPinCodeRequested: could not marshal event parameters for event PAIRING_CONFIRMATION_REQUESTED (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                     bmResult = BM_RESULT_ERR_GENERAL;
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPinCodeRequested: local pairable mode is not switched on"));
                  bmResult = BM_RESULT_ERR_NOT_PAIRABLE;
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPinCodeRequested: could not get local pairable mode"));
            }
         } // pairing initiated by remote device
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPinCodeRequested: general request check failed"));
      }

      if (BM_RESULT_OK != bmResult)
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPinCodeRequested: could not handle pairing request (bmResult = %d)",
               ETG_CENUM(BmResult, bmResult)));

         if (BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE == bmCoreIfMessage->getPairingOriginator())
         {
            ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsPinCodeRequested: canceling pairing initiated by local device"));
         }
         else
         {
            ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsPinCodeRequested: rejecting pairing request initiated by remote device"));
         }

         this->getBtStackIfConnectionRequestIfWrapper().setPinCode(bmCoreIfMessage->getBdAddress(), "");
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsSecureSimplePairingRequested(BmCoreIfMessage_BtsSecureSimplePairingRequested* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         if (BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE == bmCoreIfMessage->getPairingOriginator())
         {
            // pairing has been initiated by local device

            PairingType pairingType(BM_PAIRING_TYPE_UNKNOWN);
            result = getBmPairingTypeFromBtsSspMode(pairingType, bmCoreIfMessage->getSspMode());

            if (CC_ERR_INT_NO_ERROR == result)
            {
               char eventParams[200];

               result = LocalSpm::getPairingController().ParameterPAIRING_CONFIRMATION_REQUESTED(OUT eventParams,
                     IN sizeof(eventParams), IN static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()).c_str(),
                     IN static_cast<BdName>(bmCoreIfMessage->getBdName()).c_str(), IN pairingType,
                     IN static_cast<SspPinCode>(bmCoreIfMessage->getNumericValue()).c_str());

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: sending event PAIRING_CONFIRMATION_REQUESTED to PairingControllerSm"));

                  result = LocalSpm::getPairingController().SendEventByName("PAIRING_CONFIRMATION_REQUESTED", eventParams);

                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: could not send event PAIRING_CONFIRMATION_REQUESTED (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                     bmResult = BM_RESULT_ERR_GENERAL;
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: could not marshal event parameters for event PAIRING_CONFIRMATION_REQUESTED (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
                  bmResult = BM_RESULT_ERR_GENERAL;
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: could not get BM Core's pairing type from given BtStackIf's SSP mode (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         } // end of if (BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE == bmCoreIfMessage->getPairingOriginator())
         else
         {
            // pairing initiated by remote device

            SwitchStatus localPairableMode;
            bmResult = this->getLocalPairableMode(localPairableMode);

            if (BM_RESULT_OK == bmResult)
            {
               if (SWITCH_STATE_SWITCHED_ON == localPairableMode._switchState)
               {
                  PairingType pairingType(BM_PAIRING_TYPE_UNKNOWN);
                  result = getBmPairingTypeFromBtsSspMode(pairingType, bmCoreIfMessage->getSspMode());

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     char eventParams[200];

                     result = LocalSpm::getPairingController().ParameterPAIRING_CONFIRMATION_REQUESTED(OUT eventParams,
                           IN sizeof(eventParams), IN static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()).c_str(),
                           IN static_cast<BdName>(bmCoreIfMessage->getBdName()).c_str(),
                           IN pairingType, IN static_cast<SspPinCode>(bmCoreIfMessage->getNumericValue()).c_str());

                     if (CC_ERR_INT_NO_ERROR == result)
                     {
                        ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: sending event PAIRING_CONFIRMATION_REQUESTED to PairingControllerSm"));

                        result = LocalSpm::getPairingController().SendEventByName("PAIRING_CONFIRMATION_REQUESTED",
                              eventParams);

                        if (CC_ERR_INT_NO_ERROR != result)
                        {
                           ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: could not send event PAIRING_CONFIRMATION_REQUESTED (error = %d)",
                                 ETG_CENUM(CcErrorInternal, result)));
                           bmResult = BM_RESULT_ERR_GENERAL;
                        }
                     }
                     else
                     {
                        ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: could not marshal event parameters for event PAIRING_CONFIRMATION_REQUESTED (error = %d)",
                              ETG_CENUM(CcErrorInternal, result)));
                        bmResult = BM_RESULT_ERR_GENERAL;
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: could not get BM Core's pairing type from given BtStackIf's SSP mode (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                     bmResult = BM_RESULT_ERR_GENERAL;
                  }
               } // end of if (::cc::SWITCH_STATE_SWITCHED_ON == localPairableMode._switchState)
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: local pairable mode is not switched on"));
                  bmResult = BM_RESULT_ERR_NOT_PAIRABLE;
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: could not get local pairable mode"));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: general request check failed"));
      }

      if (BM_RESULT_OK != bmResult)
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: could not handle pairing request (bmResult = %d)",
               ETG_CENUM(BmResult, bmResult)));

         if (BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE == bmCoreIfMessage->getPairingOriginator())
         {
            ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: canceling pairing initiated by local device"));
         }
         else
         {
            ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSecureSimplePairingRequested: rejecting pairing request initiated by remote device"));
         }

         this->getBtStackIfConnectionRequestIfWrapper().confirmSecureSimplePairing(bmCoreIfMessage->getBdAddress(),
               bmCoreIfMessage->getSspMode(), bmCoreIfMessage->getNumericValue(), BTS_CONFIRM_REJECT);
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsPairingFinished(BmCoreIfMessage_BtsPairingFinished* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsPairingFinished: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      PairingStatus pairingStatus;
      PairingResult pairingResult(BM_PAIRING_RESULT_OK);

      //!Mainly for Suzuki case where pairing can be rejected after acceptance, required check for PairingStatus.
      //!If pairing is not in progress and BmCore receives pairing finished result, this should not be processed.
      (void)this->getPairingStatus(pairingStatus);

      if((BM_PAIRING_STATE_PAIRING_IN_PROGRESS == pairingStatus._state) &&
            (0 == pairingStatus._remoteBdAddress.compare(bmCoreIfMessage->getBtsBdAddress())))
      {
         if (BTS_REQ_SUCCESS == bmCoreIfMessage->getBtsResult())
         {
            // handle the request

            bool deviceInDb(false);
            result = LocalSpm::getDbManager().isDeviceInDb(deviceInDb, static_cast<BdAddress>(bmCoreIfMessage->getBtsBdAddress()));

            if (false == deviceInDb)
            {
               // device is a new device
               uint32_t numberOfPairedDevices(0u);
               result = LocalSpm::getDbManager().getNumberOfDevices(numberOfPairedDevices);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  if (LocalSpm::getDataProvider().getBmCoreConfiguration()._maxNumPairedDevices <= numberOfPairedDevices)
                  {
                     // delete oldest connected device during BT pairing if Max. Paired devices are reached
                     DeviceId deviceId = 0u;
                     ConnectionStatus connectionStatus;
                     DisconnectedReason disconnectedReason;

                     // get the oldest connected device from db
                     result =  LocalSpm::getDbManager().getOldestConnectedDevice(deviceId, connectionStatus, disconnectedReason);

                     // delete the device internally if the connection status of the device DISCONECTED
                     if (CC_ERR_INT_NO_ERROR == result)
                     {
                        if(BM_CONNECTION_STATUS_DISCONNECTED == connectionStatus)
                        {
                           BmResult bmResult = deleteDeviceInt(deviceId, BM_DEVICE_HANDLE_TYPE_SINGLE);

                           if(BM_RESULT_OK == bmResult)
                           {
                              ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsPairingFinished: device = %d is deleted successfully", deviceId));
                              result = CC_ERR_INT_NO_ERROR;
                           }
                           else
                           {
                              ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPairingFinished: not able to delete least connected device - %d", deviceId));
                              result = CC_ERR_INT_GENERAL_ERROR;
                           }
                        }
                        else
                        {
                           ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPairingFinished: least connected device is not in disconnected state(error = %d)",
                                 ETG_CENUM(CcErrorInternal, result)));
                           result = CC_ERR_INT_GENERAL_ERROR;
                        }
                     }
                     else
                     {
                        ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPairingFinished: could not get oldest connected devices from DB(error = %d)",
                              ETG_CENUM(CcErrorInternal, result)));
                     }
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPairingFinished: could not get number of devices from DB (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }

            if (CC_ERR_INT_NO_ERROR == result)
            {
               DeviceId deviceId(0u);

               result = LocalSpm::getDbManager().createDevice(INOUT deviceId, static_cast<BdName>(bmCoreIfMessage->getBtsDeviceName()),
                        static_cast<BdAddress>(bmCoreIfMessage->getBtsBdAddress()), bmCoreIfMessage->getBtsLinkKeyType(),
                        bmCoreIfMessage->getBtsLinkKey(), bmCoreIfMessage->getBtsDLinkKey(), bmCoreIfMessage->getBtsMajorServiceClass(),
                        bmCoreIfMessage->getBtsMajorDeviceClass(), bmCoreIfMessage->getBtsMinorDeviceClass(),
                        this->getRemoteSupportedProtocols(),
                        this->getRemoteSupportedSppUuids(),
                        this->getRemoteDeviceIdentification());

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  DeviceBaseInfo deviceBaseInfo;
                  result = LocalSpm::getDbManager().getDeviceBaseInfo(deviceBaseInfo, deviceId);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     //Set the _remoteConnectable as false if _connectAfterPairing flag is false except of AA/CL device
                     if (false == LocalSpm::getDataProvider().getBmCoreConfiguration()._connectAfterPairing)
                     {
                        deviceBaseInfo._remoteConnectable = false;

                        BtLimitationModeInfo btLimitationModeInfo;
                        result = this->getBtLimitationModeInfo(btLimitationModeInfo,  static_cast<BdAddress>(bmCoreIfMessage->getBtsBdAddress()));

                        if (CC_ERR_INT_NO_ERROR == result)
                        {
                           if((true == IS_LIMIMODE_FOR_HANDSFREE_PROFILE(btLimitationModeInfo._limitationMode._limitationFeature, btLimitationModeInfo._limitationMode._limitationCommunicationIf)) &&
                                 (TARGET_SWITCH_STATE_SWITCHED_ON == LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode))
                           {
                              deviceBaseInfo._remoteConnectable = true;
                           }
                        }
                        else
                        {
                           result = CC_ERR_INT_NO_ERROR;
                        }
                     }
                     else
                     {
                        if(TARGET_SWITCH_STATE_SWITCHED_OFF == LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode)
                        {
                           deviceBaseInfo._remoteConnectable = false;
                        }
                     }

                     ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsPairingFinished(deviceHandle = %d): _remoteConnectable = %d",
                           deviceBaseInfo._deviceHandle, deviceBaseInfo._remoteConnectable));

                     this->onDeviceAdded(deviceBaseInfo);
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPairingFinished: could get device base info for created device from DB (error = %d)",
                              ETG_CENUM(CcErrorInternal, result)));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPairingFinished: could not add new device to DB (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
               }
            }

            if (CC_ERR_INT_NO_ERROR != result)
            {
               pairingResult = BM_PAIRING_RESULT_ERR_GENERAL;
            }
         }//!End of if (BTS_REQ_SUCCESS == bmCoreIfMessage->getBtsResult())
         else
         {
            switch (bmCoreIfMessage->getBtsResult())
            {
               case BTS_REQ_PAIRING_ABORTED:
               case BTS_REQ_PAIRING_PAIRING_CANCELLED:
                  pairingResult = BM_PAIRING_RESULT_ERR_PAIRING_CANCELED;
                  break;
               case BTS_REQ_PAIRING_CONN_LOST:
                  pairingResult = BM_PAIRING_RESULT_ERR_CONNECTION_LOST;
                  break;
               case BTS_REQ_PAIRING_AUTHENTICATION_ERROR:
                  pairingResult = BM_PAIRING_RESULT_ERR_PAIRING_AUTHENTICATION_FAILED;
                  break;
               case BTS_REQ_PAIRING_MISSING_PIN:
                  pairingResult = BM_PAIRING_RESULT_ERR_MISSING_PIN;
                  break;
               case BTS_REQ_PAIRING_UNKNOWN:
               default:
                  pairingResult = BM_PAIRING_RESULT_ERR_GENERAL;
                  break;
            }
         }

         this->resetRemoteSupportedProtocols();
         this->resetRemoteSupportedSppUuids();
         this->resetRemoteDeviceIdentification();

         char pairingResultStr[40];

         result = LocalSpm::getPairingController().ParameterPAIRING_FINISHED(OUT pairingResultStr, IN sizeof(pairingResultStr),
                  IN bmCoreIfMessage->getBtsBdAddress().c_str(), IN pairingResult);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsPairingFinished(btsBdAddress = \"%50s\"): sending event PAIRING_FINISHED to PairingControllerSm",
                     bmCoreIfMessage->getBtsBdAddress().c_str()));
            result = LocalSpm::getPairingController().SendEventByName("PAIRING_FINISHED", pairingResultStr);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPairingFinished(btsBdAddress = \"%50s\"): could not send event PAIRING_FINISHED (error = %d)",
                        bmCoreIfMessage->getBtsBdAddress().c_str(), result));
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsPairingFinished(btsBdAddress = \"%50s\"): could not marshal event parameters for event PAIRING_FINISHED (error = %d)",
                     bmCoreIfMessage->getBtsBdAddress().c_str(), result));
         }
      }//!End of if(BM_PAIRING_STATE_PAIRING_IN_PROGRESS == pairingStatus._state)
      else
      {
         ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsPairingFinished: Pairing Not in Progress or device address is not matched so Ignoring this message from Bt Stack"));
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsSupportedServices(BmCoreIfMessage_BtsSupportedServices* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSupportedServices: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      this->resetRemoteSupportedProtocols();

      // Parital_Update is received from stack layer for updating the AVRCP version info to A2DP(AVRCP) device
      if ((BTS_REQ_SUCCESS == bmCoreIfMessage->getBtsResult()) || (BTS_REQ_PARTIAL_UPDATE == bmCoreIfMessage->getBtsResult()))
      {
         Result result(CC_ERR_INT_NO_ERROR);
         ProtocolInfoMap localSupportedProtocols;
         result = LocalSpm::getDataProvider().getProtocolInfos(localSupportedProtocols, true);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            BTSSupportedServicesBit btsSupportedServicesBitLocalProtocol;
            BTSSupportedServicesBit btsSupportedServicesBitLocalProtocolVersion;
            ProtocolVersion protocolVersion(0x0u);

            //read protocol support info from remote device info

            ProtocolInfoMap::iterator itLocal;
            BTSServiceInfoList::const_iterator itRemote;

            for (itLocal = localSupportedProtocols.begin(); itLocal != localSupportedProtocols.end(); ++itLocal)
            {
               result = getBtsSupportedServicesBitFromBmProtocolId(btsSupportedServicesBitLocalProtocol, itLocal->first);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  if (true == bmCoreIfMessage->getBtsSupportedServices().getBit(btsSupportedServicesBitLocalProtocol))
                  {
                     // locally supported protocol is supported by remote device,
                     // determine protocol version

                     if (BTS_SUPP_SRV_A2DP == btsSupportedServicesBitLocalProtocol)
                     {
                        btsSupportedServicesBitLocalProtocolVersion = BTS_SUPP_SRV_AVRCP;
                     }
                     else
                     {
                        btsSupportedServicesBitLocalProtocolVersion = btsSupportedServicesBitLocalProtocol;
                     }

                     protocolVersion = 0x0u;

                     for (itRemote = bmCoreIfMessage->getBtsServiceInfoList().begin(); itRemote != bmCoreIfMessage->getBtsServiceInfoList().end(); ++itRemote)
                     {
                        if (itRemote->service == btsSupportedServicesBitLocalProtocolVersion)
                        {
                           protocolVersion = itRemote->version;

                           if (BTS_SUPP_SRV_SPP == itRemote->service)
                           {
                              // cash spp version for this bdaddress for usage in remoteSppCapabilitiesCB
                              this->addRemoteDevicesSppVersion(bmCoreIfMessage->getBtsBdAddress(), protocolVersion);
                           }
                           break;
                        }
                     }

                     this->addRemoteSupportedProtocol(itLocal->first, protocolVersion);
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSupportedServices(btsBdAddress = \"%50s\"): could not get BtStackIf's supported services bit from BM Core's protocol ID (error = %d)",
                        bmCoreIfMessage->getBtsBdAddress().c_str(), ETG_CENUM(CcErrorInternal, result)));
               }
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSupportedServices(btsBdAddress = \"%50s\"): could not get locally supported protocols (error = %d)",
                  bmCoreIfMessage->getBtsBdAddress().c_str(), ETG_CENUM(CcErrorInternal, result)));
         }

         bool checkPairinginProgress = false;
         PairingStatus pairingStatus;

         (void) this->getPairingStatus(pairingStatus);

         if ((BM_PAIRING_STATE_PAIRING_IN_PROGRESS == pairingStatus._state) && (bmCoreIfMessage->getBtsBdAddress() == pairingStatus._remoteBdAddress))
         {
            PairingResult pairingResult = BM_PAIRING_RESULT_UNKNOWN;
            LocalSpm::getPairingController().checkPairingResult(pairingResult);

            if(BM_PAIRING_RESULT_OK != pairingResult)
            {
               checkPairinginProgress = true;

               ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSupportedServices(btsBdAddress = \"%50s\"): pairing with device with given BD address still in progress => caching supported protocols info for later storage in DB",
                     bmCoreIfMessage->getBtsBdAddress().c_str()));
            }
         }

         if(false == checkPairinginProgress)
         {
            ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSupportedServices(btsBdAddress = \"%50s\"): no pairing with device with given BD address in progress => updating supported protocols info in DB",
                  bmCoreIfMessage->getBtsBdAddress().c_str()));

            bool partialUpdate = (BTS_REQ_PARTIAL_UPDATE == bmCoreIfMessage->getBtsResult()) ? true : false;

            result = LocalSpm::getDbManager().updateProtocolSupport(bmCoreIfMessage->getBtsBdAddress(),
                  this->getRemoteSupportedProtocols(), partialUpdate);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               DeviceId deviceId(0u);
               result = LocalSpm::getDbManager().getDeviceId(OUT deviceId, IN bmCoreIfMessage->getBtsBdAddress());

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  DeviceId deviceHandle(0u);
                  result = LocalSpm::getDbManager().getDeviceHandle(OUT deviceHandle, IN deviceId);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     //ProtocolIdVersionList supportedProtocols;
                     ProtocolInfoMap protocolInfomap;
                     result = LocalSpm::getDbManager().getDeviceSupportedProtocols(OUT protocolInfomap, IN deviceId);

                     if (CC_ERR_INT_NO_ERROR == result)
                     {
                        this->onSupportedProtocolsChanged(deviceHandle, protocolInfomap);

                        ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsSupportedServices: sending event SUPPORTED_PROTOCOLS_CHANGED to PairingControllerSm"));

                        char eventParams[40];

                        result = LocalSpm::getPairingController().ParameterSUPPORTED_PROTOCOLS_CHANGED(OUT eventParams, IN sizeof(eventParams),
                              IN deviceId, IN BM_DEVICEINFO_UPDATE_SUPPORTED_PROTOCOLS);

                        if (CC_ERR_INT_NO_ERROR == result)
                        {
                           result = LocalSpm::getPairingController().SendEventByName("SUPPORTED_PROTOCOLS_CHANGED", eventParams);

                           if (CC_ERR_INT_NO_ERROR != result)
                           {
                              ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSupportedServices: could not send event SUPPORTED_PROTOCOLS_CHANGED (error = %d)",
                                    ETG_CENUM(CcErrorInternal, result)));
                           }
                        }
                        else
                        {
                           ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSupportedServices(deviceId = %d): could not marshal event parameters for event SUPPORTED_PROTOCOLS_CHANGED (error = %d)",
                                 deviceId , ETG_CENUM(CcErrorInternal, result)));
                        }
                     }
                     else
                     {
                        ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSupportedServices(bdAddress = \"%50s\"): could not get protocols supported by device from DB (error = %d)",
                              bmCoreIfMessage->getBtsBdAddress().c_str(), ETG_CENUM(CcErrorInternal, result)));
                     }

                     this->onSupportedSppUuidsChanged(deviceHandle);
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSupportedServices(bdAddress = \"%50s\"): could not get device handle from DB (error = %d)",
                           bmCoreIfMessage->getBtsBdAddress().c_str(), ETG_CENUM(CcErrorInternal, result)));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSupportedServices(bdAddress = \"%50s\"): could not get device ID from DB (error = %d)",
                        bmCoreIfMessage->getBtsBdAddress().c_str(), ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSupportedServices(bdAddress = \"%50s\"): could not update supported protocols in DB (error = %d)",
                     bmCoreIfMessage->getBtsBdAddress().c_str(), ETG_CENUM(CcErrorInternal, result)));
            }

            this->resetRemoteSupportedProtocols();
         }
      }
      else
      {
         ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSupportedServices(btsBdAddress = \"%50s\"): btsResult = %d not handled",
               bmCoreIfMessage->getBtsBdAddress().c_str(), ETG_CENUM(BTSRequestResult, bmCoreIfMessage->getBtsResult())));
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsSppCapabilities(BmCoreIfMessage_BtsSppCapabilities* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSppCapabilities: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      std::string eventName("");

      if (BTS_REQ_SUCCESS == bmCoreIfMessage->getBtsResult())
      {
         ProtocolInfoMap localSupportedProtocols;
         LocalSpm::getDataProvider().getProtocolInfos(localSupportedProtocols, true);

         if (localSupportedProtocols.end() != localSupportedProtocols.find(BM_PROTOCOL_ID_SPP))
         {
            this->resetRemoteSupportedSppUuids();

            for (size_t capIdx = 0u; capIdx < bmCoreIfMessage->getBtsSppCapabilities().size(); ++capIdx)
            {
               for (size_t uuidIdx = 0u; uuidIdx < bmCoreIfMessage->getBtsSppCapabilities()[capIdx].uuidList.size(); ++uuidIdx)
               {
                  if (true == isValidUuid(bmCoreIfMessage->getBtsSppCapabilities()[capIdx].uuidList[uuidIdx]))
                  {
                     this->addRemoteSupportedSppUuid(bmCoreIfMessage->getBtsSppCapabilities()[capIdx].uuidList[uuidIdx]);
                  }
                  else
                  {
                     ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSppCapabilities(btsBdAddress = \"%50s\"): UUID = \"%50s\" is ignored, since it is not a 128-bit SPP UUID",
                           bmCoreIfMessage->getBtsBdAddress().c_str(), bmCoreIfMessage->getBtsSppCapabilities()[capIdx].uuidList[uuidIdx].c_str()));
                  }
               }
            }

            bool checkPairinginProgress = false;

            PairingStatus pairingStatus;
            (void) this->getPairingStatus(pairingStatus);

            if ((BM_PAIRING_STATE_PAIRING_IN_PROGRESS == pairingStatus._state) && (bmCoreIfMessage->getBtsBdAddress() == pairingStatus._remoteBdAddress))
            {
               PairingResult pairingResult = BM_PAIRING_RESULT_UNKNOWN;
               LocalSpm::getPairingController().checkPairingResult(pairingResult);

               if(BM_PAIRING_RESULT_OK != pairingResult)
               {
                  checkPairinginProgress = true;

                  ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSppCapabilities(btsBdAddress = \"%50s\"): pairing with device with given BD address still in progress => caching supported SPP UUIDs for later storage in DB",
                        bmCoreIfMessage->getBtsBdAddress().c_str()));
               }
            }

            if (false == checkPairinginProgress)
            {
               ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsSppCapabilities(btsBdAddress = \"%50s\"): no pairing with device with given BD address in progress => updating supported SPP UUIDs in DB",
                     bmCoreIfMessage->getBtsBdAddress().c_str()));

               this->updateSupportedSppUuid(bmCoreIfMessage->getBtsBdAddress(), this->getRemoteSupportedSppUuids());

               this->resetRemoteSupportedSppUuids();

               eventName = "SERVICE_SEARCH_FINISHED";
            }
         }
      }
      else
      {
         eventName = "SERVICE_SEARCH_FINISHED";
      }

      if (false == eventName.empty())
      {
         DeviceId deviceId(0u);
         Result result = LocalSpm::getDbManager().getDeviceId(OUT deviceId, IN bmCoreIfMessage->getBtsBdAddress());

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsSppCapabilities: sending event UUID_INFO_UPDATED to PairingControllerSm"));

            char eventParams[40];

            Result result = LocalSpm::getPairingController().ParameterSUPPORTED_SPP_UUID_CHANGED(OUT eventParams, IN sizeof(eventParams),
                  IN deviceId, IN BM_DEVICEINFO_UPDATE_SPP_CAPABILITIES);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               result = LocalSpm::getPairingController().SendEventByName("SUPPORTED_SPP_UUID_CHANGED", eventParams);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSppCapabilities: could not send event SUPPORTED_SPP_UUID_CHANGED (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsSppCapabilities(deviceId = %d): could not marshal event parameters for event SUPPORTED_SPP_UUID_CHANGED (error = %d)",
                     deviceId , ETG_CENUM(CcErrorInternal, result)));
            }

            (void) _serviceSearchController.serviceSearchFinished(deviceId);
         }
      }
   }

   void BmCoreMainController::updateSupportedSppUuid(IN const BdAddress& bdAddress, IN const UuidList& supportedSppUuids)
   {

      ETG_TRACE_USR1(("updateSupportedSppUuid(bdAddress = \"%50s\"): entered", bdAddress.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);

      ProtocolVersion sppVersion = 0x0u;

      // get the cached SPP ProtocolVersion
      this->getRemoteDevicesSppVersion(bdAddress, sppVersion);

      // update supported SPP UUIDs in DB
      result = LocalSpm::getDbManager().updateSppUuidSupport(bdAddress, supportedSppUuids, sppVersion);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         // update properties
         DeviceId deviceHandle(0u);
         result = LocalSpm::getDbManager().getDeviceHandle(OUT deviceHandle, IN bdAddress);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            this->onSupportedSppUuidsChanged(deviceHandle);
         }
         else
         {
            ETG_TRACE_ERR(("updateSupportedSppUuid(btsBdAddress = \"%50s\"): could not get device handle from DB (error = %d)",
                  bdAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("updateSupportedSppUuid(btsBdAddress = \"%50s\"): could not update SPP UUID support in DB (error = %d)",
               bdAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords(BmCoreIfMessage_BtsRemoteDeviceIdServiceRecords* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      if (BTS_REQ_SUCCESS == bmCoreIfMessage->getResult())
      {
         DeviceIdentification deviceIdentification;

         for (size_t index = 0u; index < bmCoreIfMessage->getDeviceIdServiceRecordList().size(); ++index)
         {
            if (true == bmCoreIfMessage->getDeviceIdServiceRecordList()[index].primaryRecord)
            {
               deviceIdentification._specificationId = bmCoreIfMessage->getDeviceIdServiceRecordList()[index].specificationID;
               deviceIdentification._vendorId = bmCoreIfMessage->getDeviceIdServiceRecordList()[index].vendorID;
               deviceIdentification._productId = bmCoreIfMessage->getDeviceIdServiceRecordList()[index].productID;
               deviceIdentification._version = bmCoreIfMessage->getDeviceIdServiceRecordList()[index].version;
               deviceIdentification._vendorIdSource = bmCoreIfMessage->getDeviceIdServiceRecordList()[index].vendorIDSource;

               break;
            }
         }

         bool checkPairingisCompleted = false;

         PairingStatus pairingStatus;
         (void) this->getPairingStatus(pairingStatus);

         if ((BM_PAIRING_STATE_PAIRING_IN_PROGRESS == pairingStatus._state)
               && (bmCoreIfMessage->getBdAddress() == pairingStatus._remoteBdAddress))
         {
            PairingResult pairingResult;
            LocalSpm::getPairingController().checkPairingResult(pairingResult);

            if (BM_PAIRING_RESULT_OK != pairingResult)
            {
               checkPairingisCompleted = true;

               ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords: pairing with device with BD address = \"%50s\" still in progress => caching device ID info for later storage in DB",
                     bmCoreIfMessage->getBdAddress().c_str()));

               this->setRemoteDeviceIdentification(deviceIdentification);
            }
         }

         if (false == checkPairingisCompleted)
         {
            ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords: no pairing with device with BD address = \"%50s\" in progress => updating device ID info in DB",
                  bmCoreIfMessage->getBdAddress().c_str()));

            Result result = LocalSpm::getDbManager().updateDeviceIdentification(static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()),
                  deviceIdentification);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords: could not update device identification info for given BD address = \"%50s\" in DB (error = %d)",
                     static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()).c_str(),
                     ETG_CENUM(CcErrorInternal, result)));
            }
            else
            {
               DeviceId deviceId(0u);
               result = LocalSpm::getDbManager().getDeviceId(deviceId, static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()));

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords: sending event DID_INFO_CHANGED to PairingControllerSm"));

                  char eventParams[40];

                  result = LocalSpm::getPairingController().ParameterDID_INFO_CHANGED(OUT eventParams, IN sizeof(eventParams),
                        IN deviceId, IN BM_DEVICEINFO_UPDATE_DEVICE_IDENTIFICAION);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     result = LocalSpm::getPairingController().SendEventByName("DID_INFO_CHANGED", eventParams);

                     if (CC_ERR_INT_NO_ERROR != result)
                     {
                        ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords: could not send event DID_INFO_CHANGED (error = %d)",
                              ETG_CENUM(CcErrorInternal, result)));
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords(deviceId = %d): could not marshal event parameters for event DID_INFO_CHANGED (error = %d)",
                           deviceId , ETG_CENUM(CcErrorInternal, result)));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords: could not get DeviceId for given BD address = \"%50s\" in DB (error = %d)",
                        static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()).c_str(),
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }

            ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords: updating PairedDeviceList property"));

            PairedDeviceList pairedDeviceList;
            _pairedDeviceListPropHdl.get(pairedDeviceList);

            DeviceBaseInfoList::iterator itPaired = std::find_if(pairedDeviceList._deviceBaseInfoList.begin(),
                  pairedDeviceList._deviceBaseInfoList.end(),
                  MeetsBdAddress<DeviceBaseInfo>(static_cast<BdAddress>(bmCoreIfMessage->getBdAddress())));

            if (itPaired != pairedDeviceList._deviceBaseInfoList.end())
            {
               // found device base info in paired device list property for given BD address
               // => update device base info

               itPaired->_deviceIdentification = deviceIdentification;

               _pairedDeviceListPropHdl.set(pairedDeviceList);
            }
            else
            {
               // device base info not found in paired device list property for given BD address

               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords: could not find device base info in paired device list property for given BD address = \"%50s\"",
                     static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()).c_str()));
            }

         }
      }
      else
      {
         ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords: btsBdAddress = \"%50s\", btsResult = %d => not handled",
               bmCoreIfMessage->getBdAddress().c_str(), ETG_CENUM(BTSRequestResult, bmCoreIfMessage->getResult())));
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsDeviceCapabilities(BmCoreIfMessage_BtsDeviceCapabilities* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsDeviceCapabilities: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result = LocalSpm::getDbManager().setDeviceInbandRingtoneSupport(static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()),
            (true == bmCoreIfMessage->getDeviceCapabilities().inbandRingtoneSupported) ? BM_INBAND_RINGTONE_SUPPORT_AVAILABLE : BM_INBAND_RINGTONE_SUPPORT_NOT_AVAILABLE);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsDeviceCapabilities: could not set inband ringtone support in DB for device with BD address = \"%50s\" (error = %d)",
               static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()).c_str(), ETG_CENUM(CcErrorInternal, result)));
      }

      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsDeviceCapabilities: updating PairedDeviceList property"));

      PairedDeviceList pairedDeviceList;
      _pairedDeviceListPropHdl.get(pairedDeviceList);

      DeviceBaseInfoList::iterator itPaired = std::find_if(pairedDeviceList._deviceBaseInfoList.begin(),
            pairedDeviceList._deviceBaseInfoList.end(),
            MeetsBdAddress<DeviceBaseInfo>(static_cast<BdAddress>(bmCoreIfMessage->getBdAddress())));

      if (itPaired != pairedDeviceList._deviceBaseInfoList.end())
      {
         // found device base info in paired device list property for device handle
         // from given device connection info => update device base info

         itPaired->_inbandRingtoneSupport = (true == bmCoreIfMessage->getDeviceCapabilities().inbandRingtoneSupported) ? BM_INBAND_RINGTONE_SUPPORT_AVAILABLE : BM_INBAND_RINGTONE_SUPPORT_NOT_AVAILABLE;

         _pairedDeviceListPropHdl.set(pairedDeviceList);
      }
      else
      {
         // device base info not found in paired device list property for device handle
         // from given device connection info

         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsDeviceCapabilities: could not find device base info in paired device list property for given BD address = \"%50s\"",
               static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()).c_str()));
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsRemoteServiceSearchResult(BmCoreIfMessage_BtsRemoteServiceSearchResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsRemoteServiceSearchResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsRemoteNameResult(BmCoreIfMessage_BtsRemoteNameResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsRemoteNameResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsRemoteNameStatusUpdate(BmCoreIfMessage_BtsRemoteNameStatusUpdate* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsRemoteNameStatusUpdate: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      (void)LocalSpm::getDbManager().setBdDeviceName(bmCoreIfMessage->getBdAddress(), bmCoreIfMessage->getBdName());
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsLinkKeyRequested(BmCoreIfMessage_BtsLinkKeyRequested* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsLinkKeyRequested: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      BTSLinkKeyType linkKeyType(BTS_LINK_KEY_TYPE_UNKNOWN);
      BTSLinkKey linkKey("");
      BTSDLinkKey dLinkKey("");

      Result result = LocalSpm::getDbManager().getLinkKeyInformation(OUT linkKeyType, OUT linkKey, OUT dLinkKey,
            IN bmCoreIfMessage->getBdAddress());

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsLinkKeyRequested: could not get link key information from DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      this->getBtStackIfConnectionRequestIfWrapper().setLinkKey(bmCoreIfMessage->getBdAddress(), linkKeyType, linkKey,
            dLinkKey);
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsConnectProtocolResult(BmCoreIfMessage_BtsConnectProtocolResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsConnectProtocolResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      ProtocolId protocolId(BM_PROTOCOL_ID_UNKNOWN);

      result = getBmProtocolIdFromBtsProtocolId(protocolId, bmCoreIfMessage->getProtocolId());

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ProtocolConnectionController* pccInstance(0);

         result = LocalSpm::getDbManager().getProtocolConnectionController(&pccInstance,
               bmCoreIfMessage->getBdAddress(), protocolId, bmCoreIfMessage->getSppUuid());

         if (CC_ERR_INT_NO_ERROR == result)
         {
            pccInstance->updateNumPendingBtsConnDiscRequests(false);
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsConnectProtocolResult: could not get PCC instance (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsConnectProtocolResult: could not map BT Stack IF's protocol ID to a corresponding BM core protocol ID (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest(BmCoreIfMessage_BtsRemoteProtocolConnectRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      bool ignoreStopAutoconnection = false;

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         ProtocolId protocolId(BM_PROTOCOL_ID_UNKNOWN);

         result = getBmProtocolIdFromBtsProtocolId(protocolId, bmCoreIfMessage->getProtocolId());

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: BT Stack IF's protocol ID corresponds to BM core's protocol ID = %d",
                  ETG_CENUM(ProtocolId, protocolId)));

            result = checkAutoConnectionInProgress(protocolId, static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()), ignoreStopAutoconnection);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               RemoteConnectable remoteConnectable = false;
               result = LocalSpm::getBmCoreMainController().getDeviceRemoteConnectable(remoteConnectable,
                     static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()));

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  BlockState deviceBlockState(BM_BLOCK_STATE_UNBLOCKED);
                  result = LocalSpm::getDbManager().getDeviceBlockState(deviceBlockState,
                        static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()));

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     if (BM_BLOCK_STATE_UNBLOCKED == deviceBlockState)
                     {
                        if (true == LocalSpm::getBmController().isRequestQueueEmpty())
                        {
                           Uuid uuid = static_cast<Uuid>(bmCoreIfMessage->getSppUuid());

                           if(false == uuid.empty())
                           {
                              result = checkAndUpdateSupportedUuid(uuid, static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()));
                           }

                           if (CC_ERR_INT_NO_ERROR == result)
                           {
                              DeviceConnectionController* dccInstance(0);
                              result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance,
                                    static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()));

                              if (CC_ERR_INT_NO_ERROR == result)
                              {
                                 // found a DCC already assigned to given device
                                 ConnectionStatus connectionStatus;
                                 DisconnectedReason disconnectedReason;

                                 result = LocalSpm::getDbManager().getDeviceConnectionStatus(connectionStatus,
                                       disconnectedReason, static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()));

                                 if (CC_ERR_INT_NO_ERROR == result)
                                 {
                                    ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: found a DCC already assigned to device with given BD address with device connection status = %d",
                                          ETG_CENUM(ConnectionStatus, connectionStatus)));

                                    if ((BM_CONNECTION_STATUS_CONNECTING != connectionStatus)
                                          && (BM_CONNECTION_STATUS_CONNECTED != connectionStatus))
                                    {
                                       result = CC_ERR_INT_DCC_IN_DISCONNECTING;
                                       ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: Requested device is not connecting/Connected state (error = %d)",
                                             ETG_CENUM(CcErrorInternal, result)));
                                    }
                                    else
                                    {
                                       // check the RemoteConnectionrequest to be allowed even if the remote connectable flag is false
                                       if((false == remoteConnectable) && (BM_PROTOCOL_ID_SPP != protocolId))
                                       {
                                          result = checkRemoteConnectionrequestTobeAllowed(protocolId, static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()), uuid);
                                       }
                                    }
                                 }
                              }
                              else
                              {
                                 // check the RemoteConnectionrequest to be allowed even if the remote connectable flag is false
                                 if(false == remoteConnectable)
                                 {
                                    //accepting the remote request(HFP) if BtlimitionMode - AA(USB) is active for the requested bdAddress
                                    BtLimitationModeInfo btLimitationModeInfo;
                                    result = this->getBtLimitationModeInfo(btLimitationModeInfo,  static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()));

                                    if (CC_ERR_INT_NO_ERROR == result)
                                    {
                                       if(true == IS_LIMIMODE_FOR_HANDSFREE_PROFILE(btLimitationModeInfo._limitationMode._limitationFeature, btLimitationModeInfo._limitationMode._limitationCommunicationIf))
                                       {
                                          ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: BtLimitionMode - AA/CL/ONCAR is in progress so accept the remote request"));
                                       }
                                       else
                                       {
                                          result = CC_ERR_INT_NOT_ALLOWED;
                                          ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: remote connection requests are currently blocked for the given device (DeviceRemoteConnectable == false) (error = %d)",
                                                ETG_CENUM(CcErrorInternal, result)));
                                       }
                                    }
                                    else
                                    {
                                       result = CC_ERR_INT_NOT_ALLOWED;
                                       ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: remote connection requests are currently blocked for the given device (DeviceRemoteConnectable == false) (error = %d)",
                                             ETG_CENUM(CcErrorInternal, result)));
                                    }
                                 }
                                 else
                                 {
                                    result = CC_ERR_INT_NO_ERROR;
                                 }
                              }

                              if (CC_ERR_INT_NO_ERROR == result)
                              {
                                 ProtocolList protocolList;
                                 protocolList.push_back(Protocol(protocolId, uuid));

                                 result = LocalSpm::getBmController().addConnectionRequestDataToQueue(false,
                                       BM_CONNECTION_REQUEST_ORIGIN_REMOTE, static_cast<BdAddress>(bmCoreIfMessage->getBdAddress()),
                                       protocolList, LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultConnectionPageTimeoutMilliSeconds);

                                 if (CC_ERR_INT_NO_ERROR == result)
                                 {
                                    bmResult = LocalSpm::getBmController().handleRequest(false, ignoreStopAutoconnection);
                                 }
                                 else
                                 {
                                    ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: could not add connection request data to queue (error = %d)",
                                          ETG_CENUM(CcErrorInternal, result)));
                                 }
                              }
                           }
                           else
                           {
                              ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: requested remote uuid is not supported (error = %d)",
                                    ETG_CENUM(CcErrorInternal, result)));
                           }
                        }
                        else
                        {
                           result = CC_ERR_INT_BUSY;
                           ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: there is still a connection request waiting to be processed (error = %d)",
                                 ETG_CENUM(CcErrorInternal, result)));
                        }
                     }
                     else
                     {
                        result = CC_ERR_INT_DEVICE_BLOCKED;
                        ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: device is currently blocked (error = %d)",
                              ETG_CENUM(CcErrorInternal, result)));
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: could not get device's block state (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: could not get device's remote connectable flag (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: autoconnection is in progress (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: could not map BtStackIf's protocol ID to BM core's protocol ID (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: general request check failed"));
      }

      if ((CC_ERR_INT_NO_ERROR != result) || (BM_RESULT_OK != bmResult))
      {
         ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: rejecting protocol connection request at BtStackIf"));

         this->getBtStackIfConnectionRequestIfWrapper().rejectRemoteProtocolConnect(bmCoreIfMessage->getBdAddress(), bmCoreIfMessage->getProtocolId(), bmCoreIfMessage->getSppUuid());
      }
   }

   Result BmCoreMainController::checkAndUpdateSupportedUuid(Uuid uuid, BdAddress bdAddress)
   {
      Result result(CC_ERR_INT_NO_ERROR);
      bool uuidAvailable = false;
      bool uuidSupported = false;

      // Check the SPP profile which are supporting from HU side
      for (SppServiceInformation::const_iterator it = LocalSpm::getDataProvider().getBmCoreConfiguration()._sppServiceInformation.begin();
            it != LocalSpm::getDataProvider().getBmCoreConfiguration()._sppServiceInformation.end(); ++it)
      {
         // check the received UUID is matched with HU supported UUID's
         if(0 == (strcmp(it->_remoteUuid.c_str(), uuid.c_str())))
         {
            ETG_TRACE_USR4(("checkAndUpdateSupportedUuid(bdAddress = \"%50s\"): %50s uuid is supported", bdAddress.c_str(), it->_serviceName.c_str()));
            uuidSupported = true;
            break;
         }
         else if(0 == (strcmp(it->_localUuid.c_str(), uuid.c_str())))
         {
            ETG_TRACE_USR4(("checkAndUpdateSupportedUuid(bdAddress = \"%50s\"): %50s uuid is supported", bdAddress.c_str(), it->_serviceName.c_str()));
            uuidSupported = true;
            break;
         }
      }

      // if the received UUID is matched with supported UUID then add the UUID in DB
      if(true == uuidSupported)
      {
         UuidList supportedSppUuids;
         result = LocalSpm::getDbManager().getDeviceSppUuidSupport(supportedSppUuids, bdAddress);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            for (UuidList::iterator itUuid = supportedSppUuids.begin(); itUuid != supportedSppUuids.end(); ++itUuid)
            {
               Uuid uuidlocal = *itUuid;

               // check the received UUID is matched with IAP2BT UUID
               if(0 == (strcmp(uuid.c_str(), uuidlocal.c_str())))
               {
                  uuidAvailable = true;
                  break;
               }
            }

            ETG_TRACE_USR4(("checkAndUpdateSupportedUuid: uuid is available in db = %d", uuidAvailable));

            if(false == uuidAvailable)
            {
               supportedSppUuids.push_back(uuid);

               this->updateSupportedSppUuid(bdAddress, supportedSppUuids);
            }
         }
      }
      else
      {
         result = CC_ERR_INT_UUID_NOT_SUPPORTED;
      }

      return result;
   }

   Result BmCoreMainController::checkRemoteConnectionrequestTobeAllowed(ProtocolId protocolId, BdAddress bdAddress, Uuid uuid)
   {
      Result result(CC_ERR_INT_NO_ERROR);

      ETG_TRACE_USR4(("checkRemoteConnectionrequestTobeAllowed: entered"));

      ConnectionStatus connectionStatus;
      DisconnectedReason disconnectedReason;

      // Get the LimitationMode
      BtLimitationModeInfo btLimitationModeInfo;
      result = this->getBtLimitationModeInfo(btLimitationModeInfo, bdAddress);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         //accepting the remote request(HFP) if BtlimitionMode - AA(USB) is active for the requested bdAddress
         if(true == IS_LIMIMODE_FOR_HANDSFREE_PROFILE(btLimitationModeInfo._limitationMode._limitationFeature, btLimitationModeInfo._limitationMode._limitationCommunicationIf))
         {
            ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: BtLimitionMode - AA/CL/ONCAR is in progress so accept the remote request"));
         }
         else
         {
            result = CC_ERR_INT_ITEM_NOT_FOUND;
         }
      }

      if (CC_ERR_INT_ITEM_NOT_FOUND == result)
      {
         // Accept the connection request if BMCore already send a connnection request and waiting for response
         result = LocalSpm::getDbManager().getProtocolConnectionStatus(connectionStatus,
               disconnectedReason, bdAddress, protocolId, uuid);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            if ((BM_CONNECTION_STATUS_CONNECTING != connectionStatus)
                  && (BM_CONNECTION_STATUS_CONNECTED != connectionStatus))
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: Requested protocol state is = %d ",
                     ETG_CENUM(ConnectionStatus, connectionStatus)));

               ProtocolConnectionController* pccInstance(0);

               result = LocalSpm::getDbManager().getProtocolConnectionController(&pccInstance,
                     bdAddress, protocolId, uuid);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsRemoteProtocolConnectRequest: PCC 0x%p state = %50s",
                        (void *) pccInstance, pccInstance->GetCurrentState()));

                  if (0 != strncmp(pccInstance->GetCurrentState(), "DelayConnecting", strlen(pccInstance->GetCurrentState())))
                  {
                     result = CC_ERR_INT_DCC_IN_DISCONNECTING;
                  }
               }
            }
         }
      }

      ETG_TRACE_USR4(("checkRemoteConnectionrequestTobeAllowed: result = %d", ETG_CENUM(CcErrorInternal, result)));

      return result;
   }

   Result BmCoreMainController::checkAutoConnectionInProgress(ProtocolId protocolId, BdAddress bdAddress, bool& ignoreStopAutoconnection)
   {
      Result result(CC_ERR_INT_NO_ERROR);

      // get the Autoconnection status
      AutoConnectionStatus autoConnectionStatus;
      _autoConnectionStatusPropHdl.get(autoConnectionStatus);

      // Check Autoconnection is running or not.
      if (BM_AUTO_CONNECTION_STATE_IDLE != autoConnectionStatus._autoConnectionState)
      {
         ETG_TRACE_USR4(("checkAutoConnectionInProgress: auto connection is in progress"));

         DeviceConnectionController* dccInstance(0);
         result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, bdAddress);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            // found a DCC already assigned to given device
            ETG_TRACE_USR4(("checkAutoConnectionInProgress: found a DCC already assigned to device with given BD address"));

            if(BM_PROTOCOL_ID_SPP != protocolId)
            {
               DeviceId deviceId = 0u;
               result = LocalSpm::getDbManager().getDeviceHandle(deviceId, bdAddress);

               // Check the Remote protocol requested device Handle is matched with autoconnect-device Handle
               if(deviceId == autoConnectionStatus._deviceIdInProgress)
               {
                  bool available = false;
                  ProtocolList protocolList;

                  // get the ProtocolList to be connected for autoconnect-deviceId
                  // Check the ProtocolList contains the Remote protocol request.
                  // If available then check the connectionstatus of the protocol.
                  // If the protocol connection status is CONNECTING/CONNECTED then accept the remote protocol request
                  // and auto connection to be continued else reject the request.
                  protocolList = LocalSpm::getAutoConnectionController().getDeviceProtocolListConnectionInProgress();

                  if(!protocolList.empty())
                  {
                     for (size_t i = 0u; i < protocolList.size(); i++)
                     {
                        if(protocolList[i]._protocolId == protocolId)
                        {
                           ETG_TRACE_USR4(("checkAutoConnectionInProgress: requested protocol is matched with ProtocolListConnectionInProgress"));
                           available = true;
                           break;
                        }
                     }
                  }

                  if(false == available)
                  {
                     ETG_TRACE_ERR(("checkAutoConnectionInProgress: requested protocol is not matched with ProtocolListConnectionInProgress"));
                     result = CC_ERR_INT_AUTO_CONNECTION_IN_PROGRESS;
                  }
                  else
                  {
                     ConnectionStatus connectionStatus;
                     DisconnectedReason disconnectedReason;

                     result = LocalSpm::getDbManager().getProtocolConnectionStatus(connectionStatus,
                           disconnectedReason, bdAddress, protocolId, "");

                     if (CC_ERR_INT_NO_ERROR == result)
                     {
                        ETG_TRACE_USR4(("checkAutoConnectionInProgress: Requested protocol connectionStatus - %d", ETG_CENUM(ConnectionStatus, connectionStatus)));

                        if ((BM_CONNECTION_STATUS_CONNECTING != connectionStatus)
                              && (BM_CONNECTION_STATUS_CONNECTED != connectionStatus))
                        {
                           result = CC_ERR_INT_DCC_IN_DISCONNECTING;
                           ETG_TRACE_ERR(("checkAutoConnectionInProgress: Requested protocol is not connecting/Connected state (error = %d)",
                                 ETG_CENUM(CcErrorInternal, result)));
                        }
                        else
                        {
                           // Accept the connection request and auto connection will be continued.
                           ignoreStopAutoconnection = true;
                        }
                     }
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("checkAutoConnectionInProgress: Autoconection is running on different device - %d", deviceId));
                  result = CC_ERR_INT_AUTO_CONNECTION_IN_PROGRESS;
               }
            }
            else
            {
               // Accept the connection request and auto connection will be continued.
               ignoreStopAutoconnection = true;
               ETG_TRACE_USR4(("checkAutoConnectionInProgress: Accept the SPP Connection request"));
            }
         }
      }
      else
      {
         ETG_TRACE_USR4(("checkAutoConnectionInProgress: auto connection is NOT in progress"));
      }

      ETG_TRACE_USR4(("checkAutoConnectionInProgress: ignoreStopAutoconnection - %10s", ignoreStopAutoconnection ? "true": "false"));

      return result;
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsDisconnectProtocolResult(BmCoreIfMessage_BtsDisconnectProtocolResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsDisconnectProtocolResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      ProtocolId protocolId(BM_PROTOCOL_ID_UNKNOWN);

      result = getBmProtocolIdFromBtsProtocolId(protocolId, bmCoreIfMessage->getProtocolId());

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ProtocolConnectionController* pccInstance(0);

         result = LocalSpm::getDbManager().getProtocolConnectionController(&pccInstance,
               bmCoreIfMessage->getBdAddress(), protocolId, bmCoreIfMessage->getSppUuid());

         if (CC_ERR_INT_NO_ERROR == result)
         {
            pccInstance->updateNumPendingBtsConnDiscRequests(false);
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsConnectProtocolResult: could not get PCC instance (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsConnectProtocolResult: could not map BT Stack IF's protocol ID to a corresponding BM core protocol ID (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsDisconnectDeviceResult(BmCoreIfMessage_BtsDisconnectDeviceResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsDisconnectDeviceResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      if (BTS_REQ_SUCCESS != bmCoreIfMessage->getRequestResult())
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsDisconnectDeviceResult: device disconnection failed"));

         DeviceConnectionController* dccInstance(0);

         Result result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, bmCoreIfMessage->getBdAddress());

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsDisconnectDeviceResult: sending event DISCONNECTION_FAILED to DCC instance 0x%p",
                  (void *) dccInstance));

            result = dccInstance->SendEventByName("DISCONNECTION_FAILED", 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsDisconnectDeviceResult: could not send event DISCONNECTION_FAILED (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsDisconnectDeviceResult: could not get DCC instance assigned to given device (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsProtocolConnectionStatus(BmCoreIfMessage_BtsProtocolConnectionStatus* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsProtocolConnectionStatus: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      ProtocolId protocolId(BM_PROTOCOL_ID_UNKNOWN);

      Result result = getBmProtocolIdFromBtsProtocolId(protocolId, bmCoreIfMessage->getProtocolId());

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ProtocolConnectionController* pccInstance(0);

         result = LocalSpm::getDbManager().getProtocolConnectionController(&pccInstance, bmCoreIfMessage->getBdAddress(),
               protocolId, bmCoreIfMessage->getSppUuid());

         if (CC_ERR_INT_NO_ERROR == result)
         {
            if (BTS_CONN_CONNECTED == bmCoreIfMessage->getConnectionStatus())
            {
               pccInstance->updateProtocolConnectionStatus(BM_CONNECTION_STATUS_CONNECTED, bmCoreIfMessage->getRfCommDevice());
            }
            else if (BTS_CONN_DISCONNECTED == bmCoreIfMessage->getConnectionStatus())
            {
               DisconnectedReason disconnectedReason(BM_DISCONNECTED_REASON_UNKNOWN);

               result = getBmDisconnectedReasonFromBtsDisconnectReason(disconnectedReason,
                     bmCoreIfMessage->getDisconnectReason());

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsProtocolConnectionStatus: could not map BT Stack IF's disconnect reason = %d to a corresponding BM core disconnected reason (error = %d)",
                        ETG_CENUM(BTSDisconnectReason, bmCoreIfMessage->getDisconnectReason()),
                        ETG_CENUM(CcErrorInternal, result)));
               }

               pccInstance->updateProtocolConnectionStatus(BM_CONNECTION_STATUS_DISCONNECTED,
                     bmCoreIfMessage->getRfCommDevice(), disconnectedReason);
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsProtocolConnectionStatus: unhandled BT Stack IF connection status = %d",
                     ETG_CENUM(BTSConnectionStatus, bmCoreIfMessage->getConnectionStatus())));
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsProtocolConnectionStatus: could not get PCC instance (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));

            if (BTS_CONN_CONNECTED == bmCoreIfMessage->getConnectionStatus())
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsProtocolConnectionStatus: Protocol Connection Status mismatch between BM Core and BtStackIf!!!"));

               //this->getBtStackIfConnectionRequestIfWrapper().disconnectProtocol(bmCoreIfMessage->getBdAddress(),
               //      bmCoreIfMessage->getProtocolId(), bmCoreIfMessage->getSppUuid(), "");
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsProtocolConnectionStatus: could not map BT Stack IF's protocol ID to a corresponding BM core protocol ID (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsDeviceConnectionStatus(BmCoreIfMessage_BtsDeviceConnectionStatus* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsDeviceConnectionStatus: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      if (BTS_CONN_DISCONNECTED == bmCoreIfMessage->getConnectionStatus())
      {
         Result result(CC_ERR_INT_NO_ERROR);
         DeviceConnectionController* dccInstance(0);

         result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, bmCoreIfMessage->getBdAddress());

         if(CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsDeviceConnectionStatus: sending event DISCONNECTED to DCC instance 0x%p",
                  (void *) dccInstance));

            result = dccInstance->SendEventByName("DISCONNECTED", 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsDeviceConnectionStatus: could not send event DISCONNECTED (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsDeviceConnectionStatus: could not get DCC instance assigned to given device (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsLinkQualityResult(BmCoreIfMessage_BtsLinkQualityResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsLinkQualityResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsLinkQualityResult(btsBdAddress = \"%50s\"): current waiting list and intermediate link qualities:",
            bmCoreIfMessage->getBdAddress().c_str()));

      DeviceIdList connectedDevicesWaitingForLinkQuality(this->getConnectedDevicesWaitingForLinkQuality());

      VARTRACE(connectedDevicesWaitingForLinkQuality);
      VARTRACE(this->getIntermediateLinkQuality());

      DeviceId deviceHandle(0u);
      Result result = LocalSpm::getDbManager().getDeviceHandle(deviceHandle, bmCoreIfMessage->getBdAddress());

      if (CC_ERR_INT_NO_ERROR == result)
      {
         DeviceIdList::iterator it = find(connectedDevicesWaitingForLinkQuality.begin(),
               connectedDevicesWaitingForLinkQuality.end(), deviceHandle);

         if (connectedDevicesWaitingForLinkQuality.end() != it)
         {
            if (BTS_REQ_SUCCESS == bmCoreIfMessage->getResult())
            {
               this->addLinkQualityToIntermediateLinkQuality(deviceHandle, bmCoreIfMessage->getLinkQuality(),
                     bmCoreIfMessage->getRssi());
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsLinkQualityResult(btsBdAddress = \"%50s\"): requesting link quality failed at BtStackIf (btsResult = %d), using default values for link quality for device with device handle = %d",
                     bmCoreIfMessage->getBdAddress().c_str(), ETG_CENUM(BTSRequestResult, bmCoreIfMessage->getResult()),
                     deviceHandle));

               this->addLinkQualityToIntermediateLinkQuality(deviceHandle, 0xFFu, 127);
            }

            this->removeConnectedDeviceFromWaitingForLinkQuality(deviceHandle);

            connectedDevicesWaitingForLinkQuality.erase(it);

            if (true == connectedDevicesWaitingForLinkQuality.empty())
            {
               ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsLinkQualityResult(btsBdAddress = \"%50s\"): waiting list is empty, updating link quality property and responding %d updateLinkQuality request(s)",
                     bmCoreIfMessage->getBdAddress().c_str(), _numberPendingUpdateLinkQualityRequests));

               this->onLinkQuality(this->getIntermediateLinkQuality());

               for (unsigned int i = 0u; i < _numberPendingUpdateLinkQualityRequests; i++)
               {
                  LocalSpm::getBmCoreCallbackIfWrapper().doUpdateLinkQualityResponse(BM_RESULT_OK);
               }

               _numberPendingUpdateLinkQualityRequests = 0u;
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsLinkQualityResult(btsBdAddress = \"%50s\"): could not find device handle = %d in _connectedDevicesWaitingForLinkQuality",
                  bmCoreIfMessage->getBdAddress().c_str(), deviceHandle));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_BtsLinkQualityResult(btsBdAddress = \"%50s\"): could not get device handle for given BD address from DB (error = %d)",
               bmCoreIfMessage->getBdAddress().c_str(), ETG_CENUM(CcErrorInternal, result)));
      }

      ETG_TRACE_USR4(("handleBmCoreIfMessage_BtsLinkQualityResult(btsBdAddress = \"%50s\"): updated waiting list and intermediate link qualities:",
            bmCoreIfMessage->getBdAddress().c_str()));

      VARTRACE(this->getConnectedDevicesWaitingForLinkQuality());
      VARTRACE(this->getIntermediateLinkQuality());
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsStartTestModeResult(BmCoreIfMessage_BtsStartTestModeResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsStartTestModeResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      _testModePropHdl.onExternalSwitchingResponse((BTS_REQ_SUCCESS == bmCoreIfMessage->getRequestResult()) ? CC_ERR_INT_NO_ERROR : CC_ERR_INT_BTS_ERROR);
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsStopTestModeResult(BmCoreIfMessage_BtsStopTestModeResult* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsStopTestModeResult: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      _testModePropHdl.onExternalSwitchingResponse((BTS_REQ_SUCCESS == bmCoreIfMessage->getRequestResult()) ? CC_ERR_INT_NO_ERROR : CC_ERR_INT_BTS_ERROR);
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsTestModeStatus(BmCoreIfMessage_BtsTestModeStatus* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsTestModeStatus: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      SwitchState switchState(SWITCH_STATE_SWITCHED_OFF);

      if (BTS_CONN_CONNECTED == bmCoreIfMessage->getConnectionStatus())
      {
         switchState = SWITCH_STATE_SWITCHED_ON;
      }

      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsTestModeStatus: setting TestMode's switchState to %d", switchState));

      _testModePropHdl.onExternalSwitchStateUpdate(switchState);
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsFbWblInitialized(BmCoreIfMessage_BtsFbWblInitialized* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsFbWblInitialized: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsWblServiceAvailability(BmCoreIfMessage_BtsWblServiceAvailability* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsWblServiceAvailability: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      LocalSpm::getAutoConnectionController().updateWblServiceAvailability(bmCoreIfMessage->getWblServiceAvailable());
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsWblLastIntendedMode(BmCoreIfMessage_BtsWblLastIntendedMode* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsWblLastIntendedMode: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      LocalSpm::getAutoConnectionController().updateWblLastIntendedMode(bmCoreIfMessage->getAccessPointStatusList());
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsWblSupportedFrequencies(BmCoreIfMessage_BtsWblSupportedFrequencies* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsWblSupportedFrequencies: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      BTSWblSupportedRestrictionEntryList wblSupportedFrequenciesList = bmCoreIfMessage->getSupportedFrequenciesList();

      LocalSpm::getBtLimitationController().updateWblSupportedFrequencies(wblSupportedFrequenciesList);
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtsWblHealthinessIndicator(BmCoreIfMessage_BtsWblHealthinessIndicator* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtsWblHealthinessIndicator: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      this->updateWblHealthinessIndicator(bmCoreIfMessage->getWblHealthinessIndicator());
   }

   void BmCoreMainController::updateWblHealthinessIndicator(const BTSWblHealthinessIndicator wblHealthinessIndicator)
   {
      ETG_TRACE_USR1(("updateWblHealthinessIndicator: wblHealthinessIndicator - %u", wblHealthinessIndicator));

      _wblHealthinessIndicator = wblHealthinessIndicator;
   }

   void BmCoreMainController::handleBmCoreIfMessage_TerminateThreadRequest(BmCoreIfMessage_TerminateMessageHandlerThread* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessageTerminateThreadRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      // do not stop thread in order to handle multiple NORMAL->OFF->NORMAL state changes
      // _terminateThread = true;
   }

   Result BmCoreMainController::stopDeviceServiceSearch(IN const DeviceId deviceId)
   {
      ETG_TRACE_USR1(("stopDeviceServiceSearch: deviceId = %d", deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      result = _serviceSearchController.stopServiceSearchTimer(deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         (void) _serviceSearchController.removeDeviceFromServiceSearch(deviceId);
      }
      else
      {
         ETG_TRACE_ERR(("stopDeviceServiceSearch: could not stop service search timer (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_SetDeviceRemoteConnectable(BmCoreIfMessage_SetDeviceRemoteConnectable* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SetDeviceRemoteConnectable: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);
      DeviceIdList deviceIdList;

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if (0u != bmCoreIfMessage->getDeviceHandle())
         {
            result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_SetDeviceRemoteConnectable: could not get device ID for given device handle = %d from DB (error = %d)",
                     bmCoreIfMessage->getDeviceHandle(), ETG_CENUM(CcErrorInternal, result)));

               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_SetDeviceRemoteConnectable: general request check failed"));
      }

      if (BM_RESULT_OK == bmResult)
      {
         // handle the request

         PairedDeviceList pairedDeviceList;
         _pairedDeviceListPropHdl.get(pairedDeviceList);

         DeviceBaseInfoList::iterator it;

         if (0u != deviceId)
         {
            it = std::find_if(pairedDeviceList._deviceBaseInfoList.begin(),
                  pairedDeviceList._deviceBaseInfoList.end(), MeetsDeviceHandle<DeviceBaseInfo>(bmCoreIfMessage->getDeviceHandle()));

            if (it != pairedDeviceList._deviceBaseInfoList.end())
            {
               it->_remoteConnectable = bmCoreIfMessage->getRemoteConnectable();

               ETG_TRACE_USR1(("handleBmCoreIfMessage_SetDeviceRemoteConnectable: updating PairedDeviceList property"));
               _pairedDeviceListPropHdl.set(pairedDeviceList);
            }
            else
            {
               result = CC_ERR_INT_DEVICE_NOT_FOUND;

               ETG_TRACE_ERR(("handleBmCoreIfMessage_SetDeviceRemoteConnectable: element with given device handle = %d not found in PairedDeviceList property (error = %d)",
                     bmCoreIfMessage->getDeviceHandle(), ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            if (0u < pairedDeviceList._deviceBaseInfoList.size())
            {
               for (it = pairedDeviceList._deviceBaseInfoList.begin(); it != pairedDeviceList._deviceBaseInfoList.end(); ++it)
               {
                  it->_remoteConnectable = bmCoreIfMessage->getRemoteConnectable();
               }

               ETG_TRACE_USR1(("handleBmCoreIfMessage_SetDeviceRemoteConnectable: updating PairedDeviceList property"));
               _pairedDeviceListPropHdl.set(pairedDeviceList);
            }
         }
      }

      return bmResult;
   }

   void BmCoreMainController::handleBmCoreIfMessage_SmTimeout(BmCoreIfMessage_SmTimeout* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SmTimeout: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      FW_NORMAL_ASSERT(0 == Dispatcher::GetInstance().SendMessage(bmCoreIfMessage->getMessage()));
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_SwitchMultiHFPSupportRequest(BmCoreIfMessage_SwitchMultiHFPSupportRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SwitchMultiHFPSupportRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         if (false == isValidTargetSwitchState(bmCoreIfMessage->getTargetSwitchState()))
         {
            bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchMultiHFPSupportRequest: invalid value for parameter targetSwitchState (bmResult = %d)",
                  ETG_CENUM(BmResult, bmResult)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_SwitchMultiHFPSupportRequest: general request check failed (bmResult = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      if (BM_RESULT_OK == bmResult)
      {
         if(BM_CORE_IF_MSG_ORIGIN_INTERNAL == bmCoreIfMessage->getOrigin())
         {
            // Set the default value
            int iMultiHFSupport = 3;
            Result result = LocalSpm::getDbManager().setMultiHFPSupportStatus(iMultiHFSupport);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("setTargetSwitchState: storing MultiHFPSupport Status' target switch state to DB failed (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }

         _multiHFPSupportSwitchPropHdl.setTargetSwitchState(bmCoreIfMessage->getTargetSwitchState(),
               bmCoreIfMessage->getOrigin(), bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   void BmCoreMainController::handleBmCoreIfMessage_SetProfileUsageInfoRequest(BmCoreIfMessage_SetBtProfilesUsageRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SetProfileUsageInfoRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if (BM_RESULT_OK == bmResult)
      {
         bmResult = LocalSpm::getConflictManager().processSetBtProfileUsageRequest(bmCoreIfMessage);
         LocalSpm::getBmCoreCallbackIfWrapper().doOnSetBtProfileUsageInfoResponse(bmResult, bmCoreIfMessage->getAct());
      }//End of if (BM_RESULT_OK == bmResult)
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_SetProfileUsageInfoRequest: basic check failed (error = %d)", ETG_CENUM(BmResult, bmResult)));
         LocalSpm::getBmCoreCallbackIfWrapper().doOnSetBtProfileUsageInfoResponse(bmResult, bmCoreIfMessage->getAct());
      }//!End of else
   }

   BmResult BmCoreMainController::handleBmCoreIfMessage_SetDeviceUsagePreferenceRequest(BmCoreIfMessage_SetDeviceUsagePreferenceRequest* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_SetDeviceUsagePreferenceRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      DeviceId deviceId;
      Result result(CC_ERR_INT_NO_ERROR);

      if (BM_RESULT_OK == bmResult)
      {
         // specific request check

         // check if given deviceHandle is valid
         result = LocalSpm::getDbManager().getDeviceId(deviceId, bmCoreIfMessage->getDeviceHandle());

         if (CC_ERR_INT_NO_ERROR != result)
         {
            if (CC_ERR_INT_DB_END_OF_LIST == result)
            {
               // no device with given device handle found in DB
               ETG_TRACE_ERR(("handleBmCoreIfMessage_SetDeviceUsagePreferenceRequest: invalid value for parameter deviceHandle (error = %d)",
                        ETG_CENUM(CcErrorInternal, CC_ERR_INT_INVALID_PARAMETER)));

               bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
            }
            else
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_SetDeviceUsagePreferenceRequest: could not get device ID from DB (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));

               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }

         if (BM_RESULT_OK == bmResult)
         {
            if (false == isValidUsagePreference(bmCoreIfMessage->getUsagePreference()))
            {
               bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
               ETG_TRACE_ERR(("handleBmCoreIfMessage_SetDeviceUsagePreferenceRequest: invalid value for parameter targetSwitchState (bmResult = %d)",
                        ETG_CENUM(BmResult, bmResult)));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("handleBmCoreIfMessage_SetDeviceUsagePreferenceRequest: general request check failed (bmResult = %d)",
                  ETG_CENUM(BmResult, bmResult)));
      }

      if (BM_RESULT_OK == bmResult)
      {
         result = LocalSpm::getDbManager().setDeviceUsagePreference(deviceId, bmCoreIfMessage->getUsagePreference());

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("handleBmCoreIfMessage_SetDeviceUsagePreferenceRequest: set DeviceUsagePreference to DB failed (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }

      if (BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
      {
         // respond to client's request
         LocalSpm::getBmCoreCallbackIfWrapper().doSetDeviceUsagePreferenceResponse(bmResult, bmCoreIfMessage->getAct());
      }

      return bmResult;
   }

   void BmCoreMainController::handleBmCoreIfMessage_BtSystemStateChange(BmCoreIfMessage_BtSystemState* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handleBmCoreIfMessage_BtSystemStateChange: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      Result result(CC_ERR_INT_NO_ERROR);

      // general request check
      BmResult bmResult = checkBmCoreIfRequest(bmCoreIfMessage);

      if(BM_RESULT_OK == bmResult)
      {
         BtSystemState btSystemState = bmCoreIfMessage->getBtSystemState();

         if ((BM_BT_SYSTEM_STATE_OFF == btSystemState._bluetoothSystemState)
                  || (BM_BT_SYSTEM_STATE_BLOCK == btSystemState._bluetoothSystemState)
                  || (BM_BT_SYSTEM_STATE_NORMAL_SWITCH_OFF_BT == btSystemState._bluetoothSystemState))
         {
            //!Block state is also set by default from initBtSystemState().
            //!Remote connections should not be reset if Bt system state is set internally.
            if(BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
            {
               (void) resetRemoteConnections();
            }//!End of if(BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())

            processBtSystemStateOffOrBlock(btSystemState);
         }//!End of if((BM_BT_SYSTEM_STATE_OFF == btSystemState._bluetoothSystemState) ||..)
         else
         {
            // switch BtStatus to target state stored in DB
            ETG_TRACE_USR4(("handleBmCoreIfMessage_BtSystemStateChange: sending event RESTORE_BT_STATUS to BmControllerOnOffSm"));
            result = LocalSpm::getBmController().SendEventByName("RESTORE_BT_STATUS", 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleBmCoreIfMessage_BtSystemStateChange: could not send event RESTORE_BT_STATUS (error = %d)", result));
            }

            if (BM_BT_SYSTEM_STATE_DISCONNECT == btSystemState._bluetoothSystemState)
            {
               // disconnect all devices
               ETG_TRACE_USR4(("handleBmCoreIfMessage_BtSystemStateChange: sending event DISCONNECT_ALL_DEVICES to BmControllerOnOffSm"));
               result = LocalSpm::getBmController().SendEventByName("DISCONNECT_ALL_DEVICES", 0);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtSystemStateChange: could not send event DISCONNECT_ALL_DEVICES (error = %d)", result));
               }
            }
            else if (BM_BT_SYSTEM_STATE_NORMAL == btSystemState._bluetoothSystemState)
            {
               char eventParams[20];
               result = LocalSpm::getBmController().ParameterSTART_AUTO_CONNECTION(eventParams, sizeof(eventParams),
                        BM_AUTOCONNECTION_START_MODE_BOOK_AND_TRY_ONCE);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  ETG_TRACE_USR4(("handleBmCoreIfMessage_BtSystemStateChange: sending event START_AUTO_CONNECTION to BmControllerOnOffSm"));
                  result = LocalSpm::getBmController().SendEventByName("START_AUTO_CONNECTION", eventParams);

                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_ERR(("handleBmCoreIfMessage_BtSystemStateChange: could not send event START_AUTO_CONNECTION (error = %d)", result));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleBmCoreIfMessage_BtSystemStateChange: could not marshal event parameters for event START_AUTO_CONNECTION "
                           "(error = %d)", result));
               }
            }
            else if (BM_BT_SYSTEM_STATE_NO_ACTION == btSystemState._bluetoothSystemState)
            {
               //do nothing
            }//!End of else if (BM_BT_SYSTEM_STATE_NO_ACTION == btSystemState._bluetoothSystemState)
         }//!End of else
      }//!End of if(BM_RESULT_OK == bmResult)
   }

   void BmCoreMainController::processBtSystemStateOffOrBlock(IN const BtSystemState btSystemState)
   {
      Result result(CC_ERR_INT_NO_ERROR);
      ETG_TRACE_USR1(("processBtSystemStateOffOrBlock: BtSystemStste = %d",
               ETG_CENUM(BluetoothSystemState, btSystemState._bluetoothSystemState)));

      ETG_TRACE_USR4(("processBtSystemStateOffOrBlock: sending event STOP_AUTO_CONNECTION to BmControllerOnOffSm"));
      result = LocalSpm::getBmController().SendEventByName("STOP_AUTO_CONNECTION", (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("processBtSystemStateOffOrBlock: could not send event STOP_AUTO_CONNECTION (error = %d)", result));
      }//!End of if (CC_ERR_INT_NO_ERROR != result)

      // Changes added for Addnewdevice is pressed and off is received
      PairingStatus pairingStatus;
      (void) getPairingStatus(pairingStatus);

      //check the pairing state
      if(BM_PAIRING_STATE_IDLE != pairingStatus._state)
      {
         ETG_TRACE_USR4(("processBtSystemStateOffOrBlock: sending event REJECT_PAIRING to PairingController"));
         result = LocalSpm::getPairingController().SendEventByName("REJECT_PAIRING", (char *) 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("processBtSystemStateOffOrBlock: could not send event REJECT_PAIRING (error = %d)", result));
         }//!End of if (CC_ERR_INT_NO_ERROR != result)
      }//!End of if(BM_PAIRING_STATE_IDLE != pairingStatus._state)

      // Changes added for while scan is going on and off is received
      SwitchStatus discoveryStatus;
      (void) getDiscoveryStatus(discoveryStatus);

      if(SWITCH_STATE_SWITCHED_ON == discoveryStatus._switchState)
      {
         (void) switchDiscoveryStatusInt(TARGET_SWITCH_STATE_SWITCHED_OFF);
      }//!End of if(SWITCH_STATE_SWITCHED_ON == discoveryStatus._switchState)

      if((BM_BT_SYSTEM_STATE_OFF == btSystemState._bluetoothSystemState) ||
               (BM_BT_SYSTEM_STATE_NORMAL_SWITCH_OFF_BT == btSystemState._bluetoothSystemState))
      {
         // switch off BT without changing target state for BtStatus in DB
         (void) switchBtStatusInt(TARGET_SWITCH_STATE_SWITCHED_OFF, false, false);
      }//!End of if(BM_BT_SYSTEM_STATE_OFF == btSystemState._bluetoothSystemState)
      else
      {
         // disconnect all devices (BlockStatus is not changed, BT functionality is blocked since functionality restriction bit is set)
         ETG_TRACE_USR4(("processBtSystemStateOffOrBlock: sending event DISCONNECT_ALL_DEVICES to BmControllerOnOffSm"));
         result = LocalSpm::getBmController().SendEventByName("DISCONNECT_ALL_DEVICES", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("processBtSystemStateOffOrBlock: could not send event DISCONNECT_ALL_DEVICES (error = %d)", result));
         }//!End of if (CC_ERR_INT_NO_ERROR != result)
      }//!End of else

      // stop the BT ON delay timer and send the BT ON request to stack
      this->stopBtOnDelayTimerAndSendReq();
   }

   BmResult BmCoreMainController::checkBmCoreIfRequest(BmCoreIfMessage* bmCoreIfMessage)
   {
      ETG_TRACE_USR1(("checkBmCoreIfRequest: bmCoreIfMessage = 0x%p", (void *) bmCoreIfMessage));

      BmResult bmResult(BM_RESULT_OK);

      if (0 != bmCoreIfMessage)
      {
         if (((BM_CORE_IF_MSG_ORIGIN_CLIENT == bmCoreIfMessage->getOrigin())
               || (BM_CORE_IF_MSG_ORIGIN_BT_STACK_IF == bmCoreIfMessage->getOrigin()))
             && (true != this->isFunctionalityPermitted(bmCoreIfMessage->getRestrictionInfo())))
         {
            bmResult = BM_RESULT_ERR_NOT_ALLOWED;
         }
      }
      else
      {
         bmResult = BM_RESULT_ERR_GENERAL;
         FW_NORMAL_ASSERT_ALWAYS();
      }

      return bmResult;
   }

   void BmCoreMainController::addRemoteSupportedProtocol(IN const ProtocolId protocolId, IN const ProtocolVersion protocolVersion)
   {
      ETG_TRACE_USR1(("addRemoteSupportedProtocol"));

      _remoteSupportedProtocols.push_back(std::make_pair(protocolId, protocolVersion));
   }

   const ProtocolIdVersionList BmCoreMainController::getRemoteSupportedProtocols(void) const
   {
      ETG_TRACE_USR1(("getRemoteSupportedProtocols"));

      return _remoteSupportedProtocols;
   }

   void BmCoreMainController::resetRemoteSupportedProtocols(void)
   {
      ETG_TRACE_USR1(("resetRemoteSupportedProtocols"));

      _remoteSupportedProtocols.clear();
   }

   void BmCoreMainController::addRemoteSupportedSppUuid(IN const Uuid& uuid)
   {
      ETG_TRACE_USR1(("addRemoteSupportedSppUuid"));

      _remoteSupportedSppUuids.push_back(uuid);
   }

   const UuidList BmCoreMainController::getRemoteSupportedSppUuids(void) const
   {
      ETG_TRACE_USR1(("getRemoteSupportedSppUuids"));

      return _remoteSupportedSppUuids;
   }

   void BmCoreMainController::resetRemoteSupportedSppUuids(void)
   {
      ETG_TRACE_USR1(("resetRemoteSupportedSppUuids"));

      _remoteSupportedSppUuids.clear();
   }

   void BmCoreMainController::setRemoteDeviceIdentification(IN const DeviceIdentification& deviceIdentification)
   {
      ETG_TRACE_USR1(("setRemoteDeviceIdentification"));

      _remoteDeviceIdentification = deviceIdentification;
   }

   const DeviceIdentification BmCoreMainController::getRemoteDeviceIdentification(void) const
   {
      ETG_TRACE_USR1(("getRemoteDeviceIdentification"));

      return _remoteDeviceIdentification;
   }

   void BmCoreMainController::resetRemoteDeviceIdentification(void)
   {
      ETG_TRACE_USR1(("resetRemoteDeviceIdentification"));

      _remoteDeviceIdentification.reset();
   }

   void BmCoreMainController::addRemoteDevicesSppVersion(IN const BTSBDAddress& btsBdAddress, IN const ProtocolVersion sppVersion)
   {
      ETG_TRACE_USR1(("addRemoteDevicesSppVersion"));

      _remoteDevicesSPPVersion[btsBdAddress] = sppVersion;
   }

   void BmCoreMainController::getRemoteDevicesSppVersion(IN const BdAddress& bdAddress, ProtocolVersion& sppVersion)
   {
      ETG_TRACE_USR1(("getRemoteDevicesSppVersion: bdAddress = \"%50s\"", bdAddress.c_str()));

      sppVersion = 0x0u;

      std::map<BTSBDAddress, ProtocolVersion>::iterator it = _remoteDevicesSPPVersion.find(bdAddress);

      if (it != _remoteDevicesSPPVersion.end())
      {
         sppVersion = it->second;
      }
      else
      {
         ETG_TRACE_ERR(("getRemoteDevicesSppVersion(btsBdAddress = \"%50s\"): could not find a cached SPP version for given BD address",
               bdAddress.c_str()));
      }
   }

   void BmCoreMainController::onDeviceAdded(IN const DeviceBaseInfo& deviceBaseInfo)
   {
      ETG_TRACE_USR1(("onDeviceAdded: deviceHandle = %d", deviceBaseInfo._deviceHandle));

      ETG_TRACE_USR1(("onDeviceAdded(deviceHandle = %d): updating PairedDeviceList property",
            deviceBaseInfo._deviceHandle));

      bool newDevice(true);

      PairedDeviceList pairedDeviceList;
      _pairedDeviceListPropHdl.get(pairedDeviceList);

      DeviceBaseInfoList::iterator itPaired = std::find_if(pairedDeviceList._deviceBaseInfoList.begin(),
            pairedDeviceList._deviceBaseInfoList.end(),
            MeetsDeviceHandle<DeviceBaseInfo>(deviceBaseInfo._deviceHandle));

      if (itPaired != pairedDeviceList._deviceBaseInfoList.end())
      {
         ETG_TRACE_USR1(("onDeviceAdded(deviceHandle = %d): device base info for given device handle already exists in paired device list property => updating device base info",
               deviceBaseInfo._deviceHandle));

         newDevice = false;
         pairedDeviceList._deviceBaseInfoList.erase(itPaired);
      }

      pairedDeviceList._deviceBaseInfoList.push_back(deviceBaseInfo);
      _pairedDeviceListPropHdl.set(pairedDeviceList);

      if (true == newDevice)
      {
         (void) _remoteConnectableController.addDeviceRemoteConnectableControl(deviceBaseInfo._deviceHandle);
      }

      ETG_TRACE_USR1(("onDeviceAdded(deviceHandle = %d): updating BlockStatus property",
            deviceBaseInfo._deviceHandle));

      BlockStatus blockStatus;
      (void) this->getBlockStatus(blockStatus);

      DeviceBlockStatusList::iterator itBlock = std::find_if(blockStatus._deviceBlockStatusList.begin(),
            blockStatus._deviceBlockStatusList.end(),
            MeetsDeviceHandle<DeviceBlockStatus>(deviceBaseInfo._deviceHandle));

      if (itBlock != blockStatus._deviceBlockStatusList.end())
      {
         ETG_TRACE_ERR(("onDeviceAdded(deviceHandle = %d): block status for given device handle already exists in block status property",
               deviceBaseInfo._deviceHandle));
      }
      else
      {
         DeviceId deviceId(0u);
         Result result = LocalSpm::getDbManager().getDeviceId(deviceId, deviceBaseInfo._deviceHandle);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            DeviceBlockStatus deviceBlockStatus;
            result = LocalSpm::getDbManager().getDeviceBlockStatus(deviceBlockStatus, deviceId);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               blockStatus._deviceBlockStatusList.push_back(deviceBlockStatus);
               LocalSpm::getBmCoreMainController().setBlockStatusInt(blockStatus);
            }
            else
            {
               ETG_TRACE_ERR(("onDeviceAdded(deviceHandle = %d): could not get device block status from DB (error = %d)",
                     deviceBaseInfo._deviceHandle, ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("onDeviceAdded(deviceHandle = %d): could not get device ID from DB (error = %d)",
                  deviceBaseInfo._deviceHandle, ETG_CENUM(CcErrorInternal, result)));
         }
      }
   }

   LegacyPinCode BmCoreMainController::generateLegacyPairingPin(IN const uint32_t numDigits)
   {
      ETG_TRACE_USR1(("generateLegacyPairingPin: numDigits = %d", numDigits));

      LegacyPinCodeChrArr legacyPairingPinCodeChrArr;

      // minimum number of digits of a legacy pairing pin is 1
      // maximum number of digits of a legacy pairing pin is BM_LEGACY_PIN_CODE_STRING_SIZE - 1,
      // currently only up to 9 digits are supported due to type ranges

      const uint32_t maxNumDigits(9u);
      uint32_t correctedNumDigits(numDigits);

      if (!((0u < numDigits) && (maxNumDigits >= numDigits)))
      {
         ETG_TRACE_ERR(("generateLegacyPairingPin: given number of legacy pin digits is out of range [1, %d]",
               maxNumDigits));

         correctedNumDigits = std::max(1u, std::min(numDigits, maxNumDigits));
      }

      // correctedNumDigits is in range [1, maxNumDigits]
      // if maxNumDigits is 9 following type cast is allowed
      unsigned long numBins = static_cast<unsigned long>(pow(static_cast<double>(10.0), static_cast<double>(correctedNumDigits)));

      if ((static_cast<unsigned long>(RAND_MAX) + 1u) < numBins)
      {
         ETG_TRACE_ERR(("generateLegacyPairingPin: maximum random number is too small for generation of legacy pin with %d digits - prepending zeros",
               correctedNumDigits));

         //maximum of number of bins is RAND_MAX + 1
         numBins = static_cast<unsigned long>(RAND_MAX) + 1u;
      }

      unsigned long binSizeQuotient = (static_cast<unsigned long>(RAND_MAX) + 1u) / numBins;
      unsigned long binSizeRemainder = (static_cast<unsigned long>(RAND_MAX) + 1u) % numBins;
      unsigned long randNum(0u);

      do
      {
         randNum = static_cast<unsigned long>(rand());
      }
      while (randNum >= (static_cast<unsigned long>(RAND_MAX) + 1u - binSizeRemainder));

      sprintf(legacyPairingPinCodeChrArr, "%0*lu", correctedNumDigits, randNum / binSizeQuotient);

      ETG_TRACE_USR1(("generateLegacyPairingPin: generated PIN is \"%50s\"", legacyPairingPinCodeChrArr));

      return legacyPairingPinCodeChrArr;
   }

   void BmCoreMainController::onSupportedProtocolsChanged(IN const DeviceId deviceHandle, IN const ProtocolInfoMap& protocolInfomap)
   {
      ETG_TRACE_USR1(("onSupportedProtocolsChanged(deviceHandle = %d): updating PairedDeviceList property",
            deviceHandle));

      PairedDeviceList pairedDeviceList;
      _pairedDeviceListPropHdl.get(pairedDeviceList);

      DeviceBaseInfoList::iterator itPair = std::find_if(pairedDeviceList._deviceBaseInfoList.begin(),
            pairedDeviceList._deviceBaseInfoList.end(),
            MeetsDeviceHandle<DeviceBaseInfo>(deviceHandle));

      if (itPair != pairedDeviceList._deviceBaseInfoList.end())
      {
         ProtocolInfoMap::const_iterator itProtVersConst;

         for (ProtocolInfoMap::iterator mapIter = itPair->_protocolInfo.begin(); mapIter != itPair->_protocolInfo.end(); ++mapIter)
         {
            mapIter->second._isSupported = false;
            mapIter->second._version = 0u;

            for (itProtVersConst = protocolInfomap.begin(); itProtVersConst != protocolInfomap.end(); ++itProtVersConst)
            {
               if (itProtVersConst->first == mapIter->first)
               {
                  mapIter->second._isSupported = true;
                  mapIter->second._version = itProtVersConst->second._version;
                  break;
               }
            }
         }

         for (ProtocolInfoMap::iterator mapIter = itPair->_protocolInfo.begin(); mapIter != itPair->_protocolInfo.end(); ++mapIter)
         {
            if(mapIter->second._isSupported == false)
            {
               mapIter->second._favoriteIndex = 0u;
            }
         }

         _pairedDeviceListPropHdl.set(pairedDeviceList);
      }
      else
      {
         ETG_TRACE_ERR(("onSupportedProtocolsChanged(deviceHandle = %d): no device base info found for given device handle in paired device list property",
               deviceHandle));
      }
   }

   void BmCoreMainController::onSupportedSppUuidsChanged(IN const DeviceId deviceHandle)
   {
      ETG_TRACE_USR1(("onSupportedSppUuidsChanged: deviceHandle = %d", deviceHandle));

      ETG_TRACE_USR4(("onSupportedSppUuidsChanged(deviceHandle = %d): updating DeviceConnectionStatusList property",
            deviceHandle));

      DeviceId deviceId(0u);
      Result result = LocalSpm::getDbManager().getDeviceId(deviceId, deviceHandle);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("onSupportedSppUuidsChanged(deviceHandle = %d): could not get device ID from DB (error = %d)",
               deviceHandle, ETG_CENUM(CcErrorInternal, result)));

         return;
      }

      UuidList supportedUuids;
      result = LocalSpm::getDbManager().getDeviceSppUuidSupport(supportedUuids, deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         PairedDeviceList pairedDeviceList;
         _pairedDeviceListPropHdl.get(pairedDeviceList);

         DeviceBaseInfoList::iterator itPaired = std::find_if(pairedDeviceList._deviceBaseInfoList.begin(),
               pairedDeviceList._deviceBaseInfoList.end(),
               MeetsDeviceHandle<DeviceBaseInfo>(deviceHandle));

         if (itPaired != pairedDeviceList._deviceBaseInfoList.end())
         {
            bool uuidSupported = false;

            for (UuidList::iterator itUuid = supportedUuids.begin(); itUuid != supportedUuids.end(); ++itUuid)
            {
               Uuid uuid = *itUuid;

               // check the received UUID is matched with IAP2BT UUID
               if(0 == (strcmp((const char *)IAP2BT_SPP_UUID, uuid.c_str())))
               {
                  uuidSupported = true;
                  break;
               }
            }

            // Read the iAPoverBTSupported flag from db.
            bool iAPoverBTSupported = false;
            result = LocalSpm::getBmCoreMainController().getDeviceiAPoverBTSupported(iAPoverBTSupported, deviceId);

            if(CC_ERR_INT_NO_ERROR == result)
            {
               // set the iAPoverBTSupported flag based on the uuidSupported(either IAPoverBT is supported or not supported) update.
               if(uuidSupported != iAPoverBTSupported)
               {
                  (void) LocalSpm::getBmCoreMainController().setDeviceiAPoverBTSupported(deviceId, uuidSupported);
               }
            }
         }

         DeviceConnectionStatusList deviceConnectionStatusList;
         _deviceConnectionStatusListPropHdl.get(deviceConnectionStatusList);

         DeviceConnectionInfoList::iterator it = std::find_if(deviceConnectionStatusList._deviceConnectionInfoList.begin(),
               deviceConnectionStatusList._deviceConnectionInfoList.end(),
               MeetsDeviceHandle<DeviceConnectionInfo>(deviceHandle));

         if (it != deviceConnectionStatusList._deviceConnectionInfoList.end())
         {
            SppConnectionInfoMap::iterator itSppConn;
            UuidList::iterator itUuid;
            ConnectionStatus connectionStatus;
            DisconnectedReason disconnectedReason;

            //compare the new(parameter) and old(map) list of supported protocols
            //1.remove the protocols that doesn't exist in the new list from the map
            //2.keep procotols the exist in both old and new list unchanged
            //3.add the new protocols that doesn't exist in the old list but exists in the new list to the map
            for (itSppConn = it->_sppConnectionInfo.begin(); itSppConn != it->_sppConnectionInfo.end();)
            {
               itUuid = std::find(supportedUuids.begin(), supportedUuids.end(), itSppConn->first);

               if (itUuid != supportedUuids.end())
               {
                  // Disconnected Reason of an existing entry in property SPPServiceConnection is updated to NONE
                  // if service is reported to be supported and its Connection Status as DISCONNECTED.
                  ProtocolConnectionInfo protocolConnectionInfo = itSppConn->second;

                  result = LocalSpm::getDbManager().getProtocolConnectionStatus(connectionStatus, disconnectedReason, deviceId, BM_PROTOCOL_ID_SPP, *itUuid);

                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_ERR(("updateSppUuidSupport(deviceHandle = %d): read protocol connection status is failed (error = %d)",
                           deviceHandle, ETG_CENUM(CcErrorInternal, result)));
                  }
                  else
                  {
                     if(disconnectedReason != protocolConnectionInfo._disconnectedReason)
                     {
                        protocolConnectionInfo._disconnectedReason = disconnectedReason;

                        it->_sppConnectionInfo[itSppConn->first] = protocolConnectionInfo;
                     }
                  }

                  supportedUuids.erase(itUuid);
                  ++itSppConn;
               }
               else
               {
                  it->_sppConnectionInfo.erase(itSppConn++);
               }
            }

            //new entry is added to property SPPServiceConnection with Connection Status as DISCONNECTED and Disconnected Reason as NONE
            ProtocolConnectionInfo defaultProtConnInfo;
            defaultProtConnInfo._disconnectedReason = BM_DISCONNECTED_REASON_NONE;

            for (itUuid = supportedUuids.begin(); itUuid != supportedUuids.end(); ++itUuid)
            {
               it->_sppConnectionInfo.insert(std::make_pair(*itUuid, defaultProtConnInfo));
            }

            _deviceConnectionStatusListPropHdl.set(deviceConnectionStatusList);
         }
         else
         {
            ETG_TRACE_ERR(("onSupportedSppUuidsChanged(deviceHandle = %d): no device connection info found for given device handle in device connection status list property",
                  deviceHandle));
         }
      }
      else
      {
         ETG_TRACE_ERR(("onSupportedSppUuidsChanged(deviceHandle = %d): could not get supported SPP UUIDs from DB (error = %d)",
               deviceHandle, ETG_CENUM(CcErrorInternal, result)));
      }

      ETG_TRACE_USR4(("onSupportedSppUuidsChanged(deviceHandle = %d): updating BlockStatus property",
            deviceHandle));

      BlockStatus blockStatus;
      (void) this->getBlockStatus(blockStatus);

      DeviceBlockStatusList::iterator itBlock = std::find_if(blockStatus._deviceBlockStatusList.begin(),
            blockStatus._deviceBlockStatusList.end(),
            MeetsDeviceHandle<DeviceBlockStatus>(deviceHandle));

      if (itBlock != blockStatus._deviceBlockStatusList.end())
      {
         DeviceBlockStatus deviceBlockStatus;
         result = LocalSpm::getDbManager().getDeviceBlockStatus(deviceBlockStatus, deviceId);

         if(CC_ERR_INT_NO_ERROR == result)
         {
            *itBlock = deviceBlockStatus;
            this->setBlockStatusInt(blockStatus);
         }
         else
         {
            ETG_TRACE_ERR(("onSupportedSppUuidsChanged(deviceHandle = %d): could not get device block status from DB (error = %d)",
                  deviceHandle, ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("onSupportedSppUuidsChanged(deviceHandle = %d): no device connection info found for given device handle in block status list property",
               deviceHandle));
      }
   }

   void BmCoreMainController::getConnectedDevices(OUT DeviceIdList& connectedDeviceHandles)
   {
      ETG_TRACE_USR1(("getConnectedDevices"));

      connectedDeviceHandles.clear();

      DeviceConnectionStatusList devConnList;
      DeviceConnectionInfoList::iterator it;
      _deviceConnectionStatusListPropHdl.get(devConnList);

      for (it = devConnList._deviceConnectionInfoList.begin(); it != devConnList._deviceConnectionInfoList.end(); it++)
      {
         if(BM_CONNECTION_STATUS_CONNECTED == it->_connectionStatus)
         {
            connectedDeviceHandles.push_back(it->_deviceHandle);
         }
      }
   }

   Result BmCoreMainController::updateOverallDeviceBlockStatus(IN const OverallDeviceBlockStatus& overallDeviceBlockStatus)
   {
      ETG_TRACE_USR1(("updateOverallDeviceBlockStatus: blockState = %d, bdAddress = \"%50s\"",
            ETG_CENUM(OverallBlockState, overallDeviceBlockStatus._blockState),
            overallDeviceBlockStatus._bdAddress.c_str()));

      BlockStatus blockStatus;
      _blockStatusPropHdl.get(blockStatus);

      blockStatus._overallDeviceBlockStatus._blockState = overallDeviceBlockStatus._blockState;
      blockStatus._overallDeviceBlockStatus._bdAddress = overallDeviceBlockStatus._bdAddress;

      this->updateBlockStatus(0u, blockStatus, BM_BLOCK_STATUS_UPDATE_TYPE_OVERALL);

      return CC_ERR_INT_NO_ERROR;
   }

   void BmCoreMainController::getPrimaryStatus(INOUT DeviceConnectionStatusList& deviceConnectionStatusList)
   {
      ETG_TRACE_USR1(("getPrimaryStatus"));

      Protocol protocol;
      protocol._protocolId = BM_PROTOCOL_ID_HFP;

      for (size_t count = 0; count < deviceConnectionStatusList._deviceConnectionInfoList.size(); count++)
      {
         ProtocolConnectionInfoMap::iterator itProt = deviceConnectionStatusList._deviceConnectionInfoList[count]._protocolConnectionInfo.find(protocol._protocolId);

         if (deviceConnectionStatusList._deviceConnectionInfoList[count]._protocolConnectionInfo.end() != itProt)
         {
            LocalSpm::getDbManager().getPrimaryStatus(deviceConnectionStatusList._deviceConnectionInfoList[count]._deviceHandle, itProt->second._primary);
         }
      }
   }

   BmResult BmCoreMainController::doGetLinkQualityTestMode(OUT TestModeLinkQuality& linkQuality)
   {
      ETG_TRACE_USR1(("doGetLinkQualityTestMode"));

      BmResult bmResult(BM_RESULT_OK);

      if ((true == this->isFunctionalityPermitted(BM_RESTRICTION_GROUP_01)) &&
            (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._btOnOffSupport))
      {
         _testModeLinkQualityPropHdl.get(linkQuality);

         VARTRACE(linkQuality);
      }
      else
      {
         ETG_TRACE_ERR(("doGetLinkQualityTestMode: request is currently not allowed to be processed"));
         bmResult = BM_RESULT_ERR_NOT_ALLOWED;
      }

      return bmResult;
   }
}
