/**
 * @defgroup LocalSpm LocalSpm
 * @ingroup BtConnectionMgr
 * @author Dinesh D
 *
 * Public interface to Start and Shutdown the Media-Player.
 * Internally it manages the start and stop of all the internal components of
 * the BtConnection Manager
 * and also provides public interface to all other global components like Dispatcher ,
 * DataProvider , ThreadFactory etc...
 * @{
 */

#ifndef _LOCAL_SPM_H_
#define _LOCAL_SPM_H_

/*LocalSpm generated statemachine's headers*/
#include "LocalSpmStartSm.h"
#include "LocalSpmStopSm.h"

#include "ThreadFactory.h"
#include "DbManager.h"
#include "DataProvider.h"
#include "AutoConnectionController.h"
#include "ResetToDefaultController.h"
#include "PairingController.h"
#include "BtLimitationController.h"
#include "BmCoreMainController.h"
#include "BmController.h"
#include "IBmCoreCallbackIf.h"
#include "BmCoreCallbackIfWrapper.h"
#include "ConflictManager.h"

namespace bmcore
{
   /**
    * Class LocalSpm implements the component LocalSpm which manages Start and Shutdown of
    * all other internal components and provides interface to common global components
    *
    */
   class LocalSpm : public LocalSpmStartSm, public LocalSpmStopSm {

   public:
      /**
       * Creates singleton for Local SPM
       *
       * @return pointer to global static instance of LocalSpm object
       */
      static LocalSpm& getInstance(void);

      /**
       * interface to get the Thread Factory instance
       *
       * return pointer to global thread factory object
       */
      static ThreadFactory& getThreadFactory(void);

      /**
       * interface to get the DataProvider instance
       *
       * return pointer to global data provider object
       */
      static DataProvider& getDataProvider(void);

      /**
       * interface to get the Database manager instance
       *
       * return pointer to global database manager object
       */
      static DbManager& getDbManager(void);

      static AutoConnectionController& getAutoConnectionController(void);

      static ResetToDefaultController& getResetToDefaultController(void);

      static BtLimitationController& getBtLimitationController(void);

      static PairingController& getPairingController(void);

      static BmCoreMainController& getBmCoreMainController(void);

      static BmController& getBmController(void);

      static BmCoreCallbackIfWrapper& getBmCoreCallbackIfWrapper(void);

      static ConflictManager& getConflictManager(void);

      /**
       * SPM Interface for local components to register themselves with the LocalSpm to
       * handle their Start and Stop
       *
       * @param[in] component component object which implements the ILocalSpm interface
       *
       * @return int = 0: ok, !=0: failed
       */
      Result Register(ILocalSpm* component);

      /**
       * Interface to register custom output wrapper component with Generic BtConnectionMgr
       *
       * @param[in] outputWrapper pointer to custom output wrapper object which derives from the OutWrapper class
       *
       * @return int = 0: ok, !=0: failed
       */
      Result Register(IBmCoreCallbackIf* bmCoreCallbackIf);

      /**
       * interface to create Mediaplayer components
       *
       * @return int = 0: ok, !=0: failed
       */
      Result createBtConnectionMgr(void);

      /**
       * interface to request Mediaplayer transition to NORMAL state
       *
       * @return int = 0: ok, !=0: failed
       */
      Result stateChangeNormal(void);
      //Result StartUp(void);

      /**
       * interface to request Mediaplayer transition to OFF state
       *
       * @return int = 0: ok, !=0: failed
       */
      Result stateChangeOff(void);

      /**
       * interface to request Mediaplayer transition to UNDERVOLTAGE state or back to NORMAL
       * Roadmap 13024
       *
       * if (undervoltage == TRUE)
       * - Send all components to OFF (StopAll and DoneAll)
       * else
       * - Send all components to NORMAL (InitAll(IR_UNDERVOLTAGE) and RunAll)
       *
       * @param[in] undervoltage undervoltage flag
       *
       * @return int = 0: ok, !=0: failed
       */
      Result stateChangeUndervoltage(const Undervoltage undervoltage); //Roadmap 13024: 100%

      /**
       * returns the current state of the Local SPM
       *
       * @return SPMState current state
       */
      SpmState getSpmState(void) const;

      void printStatistics();

   private:

      LocalSpm(const LocalSpm& other);
      LocalSpm& operator=(const LocalSpm& other);

      vector <ILocalSpm*>         _components; /**< list of all local components which implements ILocalSpm interface */

      ThreadFactory               _threadFactory;
      DbManager                   _dbManager;
      DataProvider                _dataProvider;
      AutoConnectionController    _autoConnectionController;
      ResetToDefaultController    _resetToDefaultController;
      PairingController           _pairingController;
      BtLimitationController      _btLimitationController;
      BmCoreMainController        _bmCoreMainController;
      BmController                _bmController;
      ConflictManager             _conflictManager;
      BmCoreCallbackIfWrapper*    _bmCoreCallbackIfWrapper;

      pthread_cond_t  _waitCondition; /**< Condition used to wait for Start and Stop of Mediaplayer */
      pthread_mutex_t _waitMutex;		/**< Mutex lock used to wait for Start and Stop of Mediaplayer */

      Timer           _timer;
      timer_t         _timerID;

      Timer           _statisticsTimer;
      timer_t         _statisticsTimerID;

      SpmState    _spmState;

      enum
      {
         LOCALSPM_START_SM_THREAD,
         LOCALSPM_STOP_SM_THREAD
      };

      /**
       * private Constructor
       *
       */
      LocalSpm();

      /**
       * Destructor
       *
       */
      virtual ~LocalSpm();

      /**
       * Implements StartStaeMachine action InitAll to Init all the registered -
       * ILocaSPM based components
       * In case of leaving UNDERVOLTAGE state initReason is IR_UNDERVOLTAGE instaed of IR_BOOT //Roadmap 13024
       *
       *@return int = 0: ok, != 0: failed
       */
      int initAll(void); //Roadmap 13024: 100%

      /**
       * Implements StartStaeMachine action RunAll to run all the registered -
       * ILocaSPM based components
       *
       * @return int = 0: ok, != 0: failed
       */
      int runAll();

      /**
       * Implements StartStaeMachine action StopAll to stop all the registered -
       * ILocaSPM based components
       *
       * @return int = 0: ok, != 0: failed
       */
      int stopAll(void);

      /**
       * Implements StartStateMachine action DoneAll to DeInit all the registered -
       * ILocaSPM based SM components
       *
       * @return int = 0: ok, != 0: failed
       */
      int doneAllSm(void);

      /**
       * Implements StartStateMachine action DoneAll to DeInit all the registered -
       * ILocaSPM based non SM components
       *
       * @return int = 0: ok, != 0: failed
       */
      int doneAll(void);

      /**
       * Function  used by Statemachine to check whether all the registered components are
       * already initialised successfully
       *
       * @return int = 0: ok, != 0: failed
       */
      int isAllInit();

      /**
       * Function  used by Statemachine to check whether all the registered components have
       * reached the running state successfully
       *
       * @return int = 0: ok, != 0: failed
       */
      int isAllRun() ;

      /**
       * Function  used by Statemachine to check whether all the registered components have
       * been stopped .
       *
       * @return int = 0: ok, != 0: failed
       */
      int isAllInStop();

      /**
       * Function used by Statemachine to check whether all the registered SM components
       * (like PlayerManager) have been deinitialised
       *
       * @return int = 0: ok, != 0: failed
       */
      int isAllSmInDone();

      /**
       * Function used by Statemachine to check whether all the registered non SM components
       * (like DbManager) have been deinitialised
       *
       * @return int = 0: ok, != 0: failed
       */
      int isAllInDone();

      int checkComponentsState(ComponentState state, ComponentGroup group = COMPONENT_GROUP_ALL);

      int setState(ComponentId componentID, ComponentState componentState);
      int setInitState(ComponentId componentID);
      int setRunState(ComponentId componentID);
      int setStopState(ComponentId componentID);
      int setDoneState(ComponentId componentID);

      int startupTimedout(void);
      int shutdownTimedout(void);
      void logComponentStates(void);

      static bool statisticsTimer(timer_t timerID , void* pObject, const void* userData);
   };
}

#endif // _LOCAL_SPM_H_

/** @} */
