/**
 * @file BtStackIfConnectionRequestIfWrapper.cpp
 *
 * @swcomponent BluetoothConnectionManagerCore
 *
 * @brief This file contains the definition of the class BtStackIfConnectionRequestIfWrapper
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup BmControllerModule
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BtStackIfConnectionRequestIfWrapper.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#endif
#endif


#include "BtStackIfConnectionRequestIfWrapper.h"
#include "BtStackIfFactory.h"
#include "IBtStackIfConnectionRequest.h"
#include "FunctionTracer.h"
#include "FwAssert.h"

namespace bmcore
{
   BtStackIfConnectionRequestIfWrapper::BtStackIfConnectionRequestIfWrapper() : _btStackIfConnectionRequestIf(0)
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("BtStackIfConnectionRequestIfWrapper: is being created"));

      if (0 != _btStackIfConnectionRequestIf)
      {
         // already created
      }
      else
      {
         _btStackIfConnectionRequestIf = BtStackIfFactory::getInstance().getConnectionRequestIf();

         if (0 == _btStackIfConnectionRequestIf)
         {
            ETG_TRACE_ERRMEM(("BtStackIfConnectionRequestIfWrapper: _btStackIfConnectionRequestIf == 0"));
            // we should never reach this point here TODO: maybe FATAL ASSERT
            FW_NORMAL_ASSERT_ALWAYS();
         }
      }
   }

   BtStackIfConnectionRequestIfWrapper::~BtStackIfConnectionRequestIfWrapper()
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("~BtStackIfConnectionRequestIfWrapper"));

      ETG_TRACE_USR1(("~BtStackIfConnectionRequestIfWrapper: terminating BT Stack IF"));

      BTSErrorCode btsErrorCode = this->deInit();

      if (BTS_OK != btsErrorCode)
      {
         ETG_TRACE_ERRMEM(("~BtStackIfConnectionRequestIfWrapper: terminating BtStackIf failed (btsErrorCode = %d)",
               ETG_CENUM(BTSErrorCode, btsErrorCode)));
      }

      FW_NORMAL_ASSERT(BTS_OK == btsErrorCode);

      BtStackIfFactory::getInstance().destroyConnectionRequestIf(_btStackIfConnectionRequestIf);

      _btStackIfConnectionRequestIf = 0;
   }

   void BtStackIfConnectionRequestIfWrapper::registerCallback(IBtStackIfConnectionCallback* callback)
   {
      ETG_TRACE_USR1(("registerCallback"));

      _btStackIfConnectionRequestIf->registerCallback(callback);
   }

   BTSSessionHandle BtStackIfConnectionRequestIfWrapper::getSessionHandle(void) const
   {
      ETG_TRACE_USR1(("getSessionHandle"));

      return _btStackIfConnectionRequestIf->getSessionHandle();
   }

   BTSErrorCode BtStackIfConnectionRequestIfWrapper::init(IN const BTSLocalStackConfiguration& configuration,
         IN const BTSBDAddressList& pairedDevices) const
   {
      ETG_TRACE_USR1(("init"));

      return _btStackIfConnectionRequestIf->init(configuration, pairedDevices);
   }

   BTSErrorCode BtStackIfConnectionRequestIfWrapper::deInit(void) const
   {
      ETG_TRACE_USR1(("deInit"));

      return _btStackIfConnectionRequestIf->deInit();
   }

   void BtStackIfConnectionRequestIfWrapper::switchBluetoothOnOff(IN const BTSBluetoothMode bluetoothMode,
         IN const BTSAdapterMode adapterMode)
   {
      ETG_TRACE_USR1(("switchBluetoothOnOff: bluetoothMode = %d, adapterMode = %d",
            ETG_CENUM(BTSBluetoothMode, bluetoothMode), ETG_CENUM(BTSAdapterMode, adapterMode)));

      _btStackIfConnectionRequestIf->switchBluetoothOnOff(bluetoothMode, adapterMode);
   }

   void BtStackIfConnectionRequestIfWrapper::getBluetoothOnOffStatus(void)
   {
      ETG_TRACE_USR1(("getBluetoothOnOffStatus"));

      _btStackIfConnectionRequestIf->getBluetoothOnOffStatus();
   }

   void BtStackIfConnectionRequestIfWrapper::switchBluetoothHardwareOnOff(IN const BTSBluetoothMode bluetoothMode)
   {
      ETG_TRACE_USR1(("switchBluetoothHardwareOnOff: bluetoothMode = %d", ETG_CENUM(BTSBluetoothMode, bluetoothMode)));

      _btStackIfConnectionRequestIf->switchBluetoothHardwareOnOff(bluetoothMode);
   }

   void BtStackIfConnectionRequestIfWrapper::setLocalBtName(IN const BTSDeviceName& adapterName)
   {
      ETG_TRACE_USR1(("setLocalBtName: adapterName = \"%50s\"", adapterName.c_str()));

      _btStackIfConnectionRequestIf->setLocalBtName(adapterName);
   }

   void BtStackIfConnectionRequestIfWrapper::getLocalBtName(void)
   {
      ETG_TRACE_USR1(("getLocalBtName"));

      _btStackIfConnectionRequestIf->getLocalBtName();
   }

   void BtStackIfConnectionRequestIfWrapper::setDeviceIdServiceRecord(IN const BTSDeviceIdServiceRecord& serviceRecord)
   {
      ETG_TRACE_USR1(("setDeviceIdServiceRecord"));

      _btStackIfConnectionRequestIf->setDeviceIdServiceRecord(serviceRecord);
   }

   void BtStackIfConnectionRequestIfWrapper::setDeviceForRestrictedPairingConnecting(IN const BTSBDAddress& expectedDeviceAddress,
         IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("setDeviceForRestrictedPairingConnecting: expectedDeviceAddress = \"%50s\", sessionHandle = %u",
            expectedDeviceAddress.c_str(), sessionHandle));

      _btStackIfConnectionRequestIf->setDeviceForRestrictedPairingConnecting(expectedDeviceAddress, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::setLocalAdapterModes(IN const BTSLocalMode discoverableMode,
         IN const BTSDiscoverableTimeout discoverableTimeout, IN const BTSLocalMode connectableMode,
         IN const BTSConnectableTimeout connectableTimeout, IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("setLocalAdapterModes: discoverableMode = %d, discoverableTimeout = %u, connectableMode = %d, connectableTimeout = %u, sessionHandle = %u",
            ETG_CENUM(BTSLocalMode, discoverableMode), discoverableTimeout, ETG_CENUM(BTSLocalMode, connectableMode),
            connectableTimeout, sessionHandle));

      _btStackIfConnectionRequestIf->setLocalAdapterModes(discoverableMode, discoverableTimeout, connectableMode,
            connectableTimeout, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::getLocalAdapterModes(IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("getLocalAdapterModes: sessionHandle = %u", sessionHandle));

      _btStackIfConnectionRequestIf->getLocalAdapterModes(sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::startDiscovery(IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("startDiscovery: sessionHandle = %u", sessionHandle));

      _btStackIfConnectionRequestIf->startDiscovery(sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::stopDiscovery(IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("stopDiscovery: sessionHandle = %u", sessionHandle));

      _btStackIfConnectionRequestIf->stopDiscovery(sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::getDiscoveringStatus(IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("getDiscoveringStatus: sessionHandle = %u", sessionHandle));

      _btStackIfConnectionRequestIf->getDiscoveringStatus(sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::startPairing(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("startPairing: deviceAddress = \"%50s\", sessionHandle = %u", deviceAddress.c_str(), sessionHandle));

      _btStackIfConnectionRequestIf->startPairing(deviceAddress, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::cancelPairing(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("cancelPairing: deviceAddress = \"%50s\", sessionHandle = %u", deviceAddress.c_str(), sessionHandle));

      _btStackIfConnectionRequestIf->cancelPairing(deviceAddress, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::setPinCode(IN const BTSBDAddress& deviceAddress, IN const BTSPinCode& pinCode,
         IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("setPinCode: deviceAddress = \"%50s\", pinCode = \"%50s\", sessionHandle = %u",
            deviceAddress.c_str(), pinCode.c_str(), sessionHandle));

      _btStackIfConnectionRequestIf->setPinCode(deviceAddress, pinCode, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::confirmSecureSimplePairing(IN const BTSBDAddress& deviceAddress,
         IN const BTSSspMode sspMode, IN const BTSNumericValue& numericValue, IN const BTSConfirmationMode confirmationMode,
         IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("confirmSecureSimplePairing: deviceAddress = \"%50s\", sspMode = %d, numericValue = \"%50s\", confirmationMode = %d, sessionHandle = %u",
            deviceAddress.c_str(), ETG_CENUM(BTSSspMode, sspMode), numericValue.c_str(),
            ETG_CENUM(BTSConfirmationMode, confirmationMode), sessionHandle));

      _btStackIfConnectionRequestIf->confirmSecureSimplePairing(deviceAddress, sspMode, numericValue, confirmationMode, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::setLinkKey(IN const BTSBDAddress& deviceAddress, IN const BTSLinkKeyType linkKeyType,
         IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey, IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("setLinkKey: deviceAddress = \"%50s\", linkKeyType = %d, linkKey = \"%50s\", dLinkKey = \"%50s\", sessionHandle = %u",
            deviceAddress.c_str(), linkKeyType, linkKey.c_str(), dLinkKey.c_str(), sessionHandle));

      _btStackIfConnectionRequestIf->setLinkKey(deviceAddress, linkKeyType, linkKey, dLinkKey, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::connectProtocol(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId,
         IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const BTSTimeValue pageTimeout, IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("connectProtocol: deviceAddress = \"%50s\", protocolId = %d, sppUuid = \"%50s\", masInstance= \"%50s\ pageTimeout = %u, sessionHandle = %u",
            deviceAddress.c_str(), ETG_CENUM(BTSProtocolId, protocolId), sppUuid.c_str(), masInstance.c_str(), pageTimeout, sessionHandle));

      _btStackIfConnectionRequestIf->connectProtocol(deviceAddress, protocolId, sppUuid, masInstance, pageTimeout, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::acceptRemoteProtocolConnect(IN const BTSBDAddress& deviceAddress,
         IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("acceptRemoteProtocolConnect: deviceAddress = \"%50s\", protocolId = %d, sppUuid = \"%50s\", sessionHandle = %u",
            deviceAddress.c_str(), ETG_CENUM(BTSProtocolId, protocolId), sppUuid.c_str(), sessionHandle));

      _btStackIfConnectionRequestIf->acceptRemoteProtocolConnect(deviceAddress, protocolId, sppUuid, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::rejectRemoteProtocolConnect(IN const BTSBDAddress& deviceAddress,
         IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("rejectRemoteProtocolConnect: deviceAddress = \"%50s\", protocolId = %d, sppUuid = \"%50s\", sessionHandle = %u",
            deviceAddress.c_str(), ETG_CENUM(BTSProtocolId, protocolId), sppUuid.c_str(), sessionHandle));

      _btStackIfConnectionRequestIf->rejectRemoteProtocolConnect(deviceAddress, protocolId, sppUuid, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::disconnectProtocol(IN const BTSBDAddress& deviceAddress,
         IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance,
         IN const bool pauseBtStreaming, IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("disconnectProtocol: deviceAddress = \"%50s\", protocolId = %d, sppUuid = \"%50s\", masInstance= \"%50s\", pauseBtStreaming = \"%10s\", sessionHandle = %u",
            deviceAddress.c_str(), ETG_CENUM(BTSProtocolId, protocolId), sppUuid.c_str(), masInstance.c_str(), pauseBtStreaming ? "true" : "false", sessionHandle));

      _btStackIfConnectionRequestIf->disconnectProtocol(deviceAddress, protocolId, sppUuid, masInstance, pauseBtStreaming, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::disconnectDevice(IN const BTSBDAddress& deviceAddress, IN const bool pauseBtStreaming,
         IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("disconnectDevice: deviceAddress = \"%50s\", pauseBtStreaming = \"%10s\", sessionHandle = %u", deviceAddress.c_str(),
            pauseBtStreaming ? "true" : "false" , sessionHandle));

      _btStackIfConnectionRequestIf->disconnectDevice(deviceAddress, pauseBtStreaming, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::getProtocolConnectionStatus(IN const BTSBDAddress& deviceAddress,
         IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance,
         IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("getProtocolConnectionStatus: deviceAddress = \"%50s\", protocolId = %d, sppUuid = \"%50s\", masInstance= \"%50s\", sessionHandle = %u",
            deviceAddress.c_str(), ETG_CENUM(BTSProtocolId, protocolId), sppUuid.c_str(), masInstance.c_str(), sessionHandle));

      _btStackIfConnectionRequestIf->getProtocolConnectionStatus(deviceAddress, protocolId, sppUuid, masInstance, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::getDeviceConnectionStatus(IN const BTSBDAddress& deviceAddress,
         IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("getDeviceConnectionStatus: deviceAddress = \"%50s\", sessionHandle = %u", deviceAddress.c_str(),
            sessionHandle));

      _btStackIfConnectionRequestIf->getDeviceConnectionStatus(deviceAddress, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::startRemoteServiceSearch(IN const BTSBDAddress& deviceAddress,
         IN const BTSSearchType searchType, IN const bool rejectIfNotConnected, IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("startRemoteServiceSearch: deviceAddress = \"%50s\", searchType = %d, rejectIfNotConnected = %d, sessionHandle = %u",
            deviceAddress.c_str(), ETG_CENUM(BTSSearchType, searchType), rejectIfNotConnected, sessionHandle));

      _btStackIfConnectionRequestIf->startRemoteServiceSearch(deviceAddress, searchType, rejectIfNotConnected, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::cancelRemoteServiceSearch(IN const BTSBDAddress& deviceAddress,
         IN const BTSSearchType searchType, IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("cancelRemoteServiceSearch: deviceAddress = \"%50s\", searchType = %d, sessionHandle = %u",
            deviceAddress.c_str(), ETG_CENUM(BTSSearchType, searchType), sessionHandle));

      _btStackIfConnectionRequestIf->cancelRemoteServiceSearch(deviceAddress, searchType, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::queryRemoteName(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("queryRemoteName: deviceAddress = \"%50s\", sessionHandle = %d", deviceAddress.c_str(), sessionHandle));

      _btStackIfConnectionRequestIf->queryRemoteName(deviceAddress, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::requestLinkQuality(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("requestLinkQuality: deviceAddress = \"%50s\", sessionHandle = %d", deviceAddress.c_str(), sessionHandle));

      _btStackIfConnectionRequestIf->requestLinkQuality(deviceAddress, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::startTestMode(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("startTestMode: deviceAddress = \"%50s\", sessionHandle = %d", deviceAddress.c_str(), sessionHandle));

      _btStackIfConnectionRequestIf->startTestMode(deviceAddress, sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::stopTestMode(IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("stopTestMode: sessionHandle = %u", sessionHandle));

      _btStackIfConnectionRequestIf->stopTestMode(sessionHandle);
   }

   void BtStackIfConnectionRequestIfWrapper::requestTestModeLinkQuality(IN const BTSSessionHandle sessionHandle)
   {
      ETG_TRACE_USR1(("requestTestModeLinkQuality: sessionHandle = %u", sessionHandle));

      _btStackIfConnectionRequestIf->requestTestModeLinkQuality(sessionHandle);
   }
}
