#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_INTERFACE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BmCoreRequestIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_INTERFACE
#endif
#endif

#include "BmCoreRequestIf.h"
#include "BmCoreIfMessagesCreator.h"
#include "LocalSpm.h"
#include "TimeTrace.h"

#include "BmVarTrace.h"
#include "FunctionTracer.h"
#include "BmGlobalLock.h"

namespace bmcore
{
   #define LOCK Locker locker(&_interfaceLock);

   BmInterface::BmInterface() : _isInitialized(false)
   {
      ENTRY
   }

   BmInterface::~BmInterface()
   {
      ENTRY
   }

   BmResult BmInterface::registerBmCoreCallbackIf(IBmCoreCallbackIf* bmCoreCallbackIf)
   {
      ETG_TRACE_USR1(("registerBmCoreCallbackIf: bmCoreCallbackIf = 0x%p", (void *) bmCoreCallbackIf));

      BmResult bmResult(BM_RESULT_OK);

      Result result = LocalSpm::getInstance().Register(bmCoreCallbackIf);

      if (0 != result)
      {
         bmResult = BM_RESULT_ERR_GENERAL;
      }

      ETG_TRACE_USR1(("registerBmCoreCallbackIf: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::initializeBmCore(void)
   {
      ETG_TRACE_USR1(("initializeBmCore"));

      LOCK

      if (true == _isInitialized)
      {
         ETG_TRACE_FATAL(("initializeBmCore: %s has already been called - BM Core is already initialized", __PRETTY_FUNCTION__));
         return BM_RESULT_ERR_GENERAL;
      }

      BmResult bmResult(BM_RESULT_OK);
      TimeTrace ticks(__PRETTY_FUNCTION__);

      Result result = LocalSpm::getInstance().createBtConnectionMgr();

      _isInitialized = true;

      ticks.elapsed();

      if (0 != result)
      {
         bmResult = BM_RESULT_ERR_GENERAL;
      }

      ETG_TRACE_USR1(("initializeBmCore: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::stateChangeNormal(IN const ActType act)
   {
      ETG_TRACE_USR1(("stateChangeNormal: act = %d", act));

      // no global lock because this is a blocking call in LocalSpm::stateChangeNormal()

      LOCK

      BmResult bmResult(BM_RESULT_OK);
      TimeTrace ticks(__PRETTY_FUNCTION__);

      if (true == _isInitialized)
      {
         Result result = LocalSpm::getInstance().stateChangeNormal();

         if (CC_ERR_INT_NO_ERROR != result)
         {
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         ETG_TRACE_FATAL(("stateChangeNormal: %s has been called before BtConnectionManager was created", __PRETTY_FUNCTION__));
         bmResult = BM_RESULT_ERR_GENERAL;
      }

      ticks.elapsed();

      LocalSpm::getBmCoreCallbackIfWrapper().doStateChangeNormalResponse(bmResult);

      ETG_TRACE_USR1(("stateChangeNormal: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::stateChangeOff(IN const ActType act)
   {
      ETG_TRACE_USR1(("stateChangeOff: act = %u", act));

      // no global lock because this is a blocking call in LocalSpm::stateChangeOff()

      LOCK

      BmResult bmResult(BM_RESULT_OK);
      TimeTrace ticks(__PRETTY_FUNCTION__);

      if (true == _isInitialized)
      {
         Result result = LocalSpm::getInstance().stateChangeOff();

         if (CC_ERR_INT_NO_ERROR != result)
         {
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         ETG_TRACE_FATAL(("stateChangeOff: %s has been called before BtConnectionManager was created", __PRETTY_FUNCTION__));
         bmResult = BM_RESULT_ERR_GENERAL;
      }

      ticks.elapsed();

      LocalSpm::getBmCoreCallbackIfWrapper().doStateChangeOffResponse(bmResult);

      ETG_TRACE_USR1(("stateChangeOff: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::setBtSystemState(IN const BtSystemState& btSystemState, IN const ActType act)
   {
      ETG_TRACE_USR1(("setBtSystemState: btSystemState = %d, act = %u",
            ETG_CENUM(BluetoothSystemState, btSystemState._bluetoothSystemState), act));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().setBtSystemState(btSystemState);

      if(BM_RESULT_OK == bmResult)
      {
         BmCoreIfMessage_BtSystemState* bmCoreIfMessage = getNewBmCoreIfMessage_BtSystemState(btSystemState,
               act, BM_CORE_IF_MSG_ORIGIN_CLIENT);

         if(0 != bmCoreIfMessage)
         {
            LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);
         }
         else
         {
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }

      ETG_TRACE_USR4(("setBtSystemState: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::getBtSystemState(OUT BtSystemState& btSystemState)
   {
      ETG_TRACE_USR1(("getBtSystemState"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getBtSystemState(btSystemState,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getBtSystemState: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::setProtocolUserAvailability(IN const Protocol& protocol, IN const bool available, IN const ActType act)
   {
      ETG_TRACE_USR1(("setProtocolUserAvailability: protocol = (%d, \"%50s\"), available = %10s, act = %u",
            ETG_CENUM(ProtocolId, protocol._protocolId), protocol._uuid.c_str(), available ? "true" : "false", act));

      // no global lock because internal message is created

      BmCoreIfMessage_SetProtocolUserAvailabilityRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SetProtocolUserAvailabilityRequest(protocol,
            available, act, BM_CORE_IF_MSG_ORIGIN_CLIENT);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("setProtocolUserAvailability: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::switchBtStatus(const TargetSwitchState targetSwitchState,
         IN const bool suppressAutoConnectionAfterSwitchedOn, IN const bool delayBtOnRequest, IN const ActType act)
   {
      ETG_TRACE_USR1(("switchBtStatus: targetSwitchState = %d, suppressAutoConnectionAfterSwitchedOn = %10s, delayRequest = %10s, act = %u",
            ETG_CENUM(TargetSwitchState, targetSwitchState), suppressAutoConnectionAfterSwitchedOn ? "true" : "false",
                  delayBtOnRequest ? "true": "false", act));

      // no global lock because internal message is created

      BmCoreIfMessage_SwitchBtStatusRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SwitchBtStatusRequest(targetSwitchState,
            suppressAutoConnectionAfterSwitchedOn, delayBtOnRequest, true, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("switchBtStatus: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::getBtStatus(OUT SwitchStatus& bluetoothStatus)
   {
      ETG_TRACE_USR1(("getBtStatus"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getBtStatus(bluetoothStatus, BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getBtStatus: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::switchLocalPairableMode(IN const TargetSwitchState targetSwitchState,
         IN const BdAddress& bdAddress, IN const ActType act)
   {
      ETG_TRACE_USR1(("switchLocalPairableMode: targetSwitchState = %d, bdAddress = \"%50s\", act = %u",
            ETG_CENUM(TargetSwitchState, targetSwitchState), bdAddress.c_str(), act));

      // no global lock because internal message is created

      BmCoreIfMessage_SwitchLocalPairableModeRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SwitchLocalPairableModeRequest(targetSwitchState,
            bdAddress, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("switchLocalPairableMode: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::getLocalPairableMode(OUT SwitchStatus& localPairableMode)
   {
      ETG_TRACE_USR1(("getLocalPairableMode"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getLocalPairableMode(localPairableMode,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getLocalPairableMode: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::switchLocalConnectableMode(IN const TargetSwitchState targetSwitchState,
         IN const BdAddress& bdAddress, IN const ActType act)
   {
      ETG_TRACE_USR1(("switchLocalConnectableMode: targetSwitchState = %d, bdAddress = \"%50s\", act = %u",
            ETG_CENUM(TargetSwitchState, targetSwitchState), bdAddress.c_str(), act));

      // no global lock because internal message is created

      BmCoreIfMessage_SwitchLocalConnectableModeRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SwitchLocalConnectableModeRequest(targetSwitchState,
            bdAddress, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("switchLocalConnectableMode: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::getLocalConnectableMode(OUT SwitchStatus& localConnectableMode)
   {
      ETG_TRACE_USR1(("getLocalConnectableMode"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getLocalConnectableMode(localConnectableMode,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getLocalConnectableMode: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::switchDiscoveryStatus(IN const TargetSwitchState targetSwitchState,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("switchDiscoveryStatus: targetSwitchState = %d, act = %u",
            ETG_CENUM(TargetSwitchState, targetSwitchState), act));

      // no global lock because internal message is created

      BmCoreIfMessage_SwitchDiscoveryStatusRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SwitchDiscoveryStatusRequest(targetSwitchState,
            act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("doSwitchDiscoveryStatus: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::getDiscoveryStatus(OUT SwitchStatus& discoveryStatus)
   {
      ETG_TRACE_USR1(("getDiscoveryStatus"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getDiscoveryStatus(discoveryStatus,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getDiscoveryStatus: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::getDiscoveredDeviceList(OUT DiscoveredDeviceList& discoveredDeviceList)
   {
      ETG_TRACE_USR1(("getDiscoveredDeviceList"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getDiscoveredDeviceList(discoveredDeviceList,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getDiscoveredDeviceList: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::setBtLocalFriendlyName(IN const LocalFriendlyName& localFriendlyName,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("setBtLocalFriendlyName: localFriendlyName = \"%100s\", act = %u",
            localFriendlyName._localFriendlyName.c_str(), act));

      // no global lock because internal message is created

      BmCoreIfMessage_SetBtLocalFriendlyNameRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SetBtLocalFriendlyNameRequest(localFriendlyName._localFriendlyName,
            act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("setBtLocalFriendlyName: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::getBtLocalFriendlyName(OUT LocalFriendlyName& localFriendlyName)
   {
      ETG_TRACE_USR1(("getBtLocalFriendlyName"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getBtLocalFriendlyName(localFriendlyName,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getBtLocalFriendlyName: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::getBtLocalInfo(OUT LocalInfo& localInfo)
   {
      ETG_TRACE_USR1(("getBtLocalInfo"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getLocalInfo(localInfo, BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getBtLocalInfo: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::startPairing(IN const BdAddress& bdAddress, IN const ActType act)
   {
      ETG_TRACE_USR1(("startPairing: bdAddress = \"%50s\", act = %u", bdAddress.c_str(), act));

      // no global lock because internal message is created

      BmCoreIfMessage_StartPairingRequest* bmCoreIfMessage = getNewBmCoreIfMessage_StartPairingRequest(bdAddress, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("startPairing: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::cancelPairing(IN const ActType act)
   {
      ETG_TRACE_USR1(("cancelPairing: act = %u", act));

      // no global lock because internal message is created

      BmCoreIfMessage_CancelPairingRequest* bmCoreIfMessage = getNewBmCoreIfMessage_CancelPairingRequest(act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("cancelPairing: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::acceptPairingRequest(IN const ActType act)
   {
      ETG_TRACE_USR1(("acceptPairingRequest: act = %u", act));

      // no global lock because internal message is created

      BmCoreIfMessage_AcceptPairingRequestRequest* bmCoreIfMessage = getNewBmCoreIfMessage_AcceptPairingRequestRequest(act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("acceptPairingRequest: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::rejectPairingRequest(IN const ActType act)
   {
      ETG_TRACE_USR1(("rejectPairingRequest: act = %u", act));

      // no global lock because internal message is created

      BmCoreIfMessage_RejectPairingRequestRequest* bmCoreIfMessage = getNewBmCoreIfMessage_RejectPairingRequestRequest(act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("rejectPairingRequest: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::getPairingStatus(OUT PairingStatus& pairingStatus)
   {
      ETG_TRACE_USR1(("getPairingStatus"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus, BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getPairingStatus: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::setPairingPin(IN const PairingPin& pairingPin, IN const ActType act)
   {
      ETG_TRACE_USR1(("setPairingPin: pairingPin = \"%50s\", act = %u", pairingPin._pin.c_str(), act));

      // no global lock because internal message is created

      BmCoreIfMessage_SetPairingPinRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SetPairingPinRequest(pairingPin, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("setPairingPin: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::getPairingPin(OUT PairingPin& pairingPin)
   {
      ETG_TRACE_USR1(("getPairingPin"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getPairingPin(pairingPin, BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getPairingPin: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::getPairedDeviceList(OUT PairedDeviceList& pairedDeviceList)
   {
      ETG_TRACE_USR1(("getPairedDeviceList"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getPairedDeviceList(pairedDeviceList,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getPairedDeviceList: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::addOobPairedDevice(IN const BdAddress& bdAddress, IN const BdName& bdName,
         IN const LinkKey& linkKey, IN const OobType oobType, IN const ActType act)
   {
      ETG_TRACE_USR1(("addOobPairedDevice: bdAddress = \"%50s\", bdName = \"%50s\", linkKey = \"%50s\", oobType = %d, act = %u",
            bdAddress.c_str(), bdName.c_str(), linkKey.c_str(), ETG_CENUM(OobType, oobType), act));

      // no global lock because internal message is created

      BmCoreIfMessage_AddOobPairedDeviceRequest* bmCoreIfMessage = getNewBmCoreIfMessage_AddOobPairedDeviceRequest(bdAddress,
            bdName, linkKey, oobType, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("addOobPairedDevice: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::deleteDevice(IN const DeviceId deviceHandle, IN const DeviceHandleType deviceHandleType,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("deleteDevice: deviceHandle = %d, deviceHandleType = %d, act = %u",
            deviceHandle, ETG_CENUM(DeviceHandleType, deviceHandleType), act));

      // no global lock because internal message is created

      BmCoreIfMessage_DeleteDeviceRequest* bmCoreIfMessage = getNewBmCoreIfMessage_DeleteDeviceRequest(deviceHandle,
            deviceHandleType, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("deleteDevice: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::connectDevice(IN const DeviceId deviceHandle, IN const ActType act)
   {
      ETG_TRACE_USR1(("connectDevice: deviceHandle = %d, act = %u", deviceHandle, act));

      // no global lock because internal message is created

      BmCoreIfMessage_ConnectDeviceRequest* bmCoreIfMessage = getNewBmCoreIfMessage_ConnectDeviceRequest(deviceHandle,
            false, LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultConnectionPageTimeoutMilliSeconds, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("connectDevice: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::disconnectDevice(IN const DeviceId deviceHandle, IN const DeviceHandleType deviceHandleType,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("disconnectDevice: deviceHandle = %d, deviceHandleType = %d, act = %u",
            deviceHandle, ETG_CENUM(DeviceHandleType, deviceHandleType), act));

      // no global lock because internal message is created

      BmCoreIfMessage_DisconnectDeviceRequest* bmCoreIfMessage = getNewBmCoreIfMessage_DisconnectDeviceRequest(deviceHandle,
            deviceHandleType, BM_DISCONNECTED_REASON_UNKNOWN, false, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("disconnectDevice: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::connectProfiles(IN const DeviceId deviceHandle, IN const ProtocolList& protocolList,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("connectProfiles: deviceHandle = %d, act = %u", deviceHandle, act));

#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
      // TODO: @Stefan: compiler error: 'VARTRACE_USR2' was not declared in this scope
#else
      VARTRACE_USR2(protocolList);
#endif

      // no global lock because internal message is created

      BmCoreIfMessage_ConnectProfilesRequest* bmCoreIfMessage = getNewBmCoreIfMessage_ConnectProfilesRequest(deviceHandle,
            protocolList, LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultConnectionPageTimeoutMilliSeconds, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("connectProfiles: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::disconnectProfiles(IN const DeviceId deviceHandle, IN const ProtocolList& protocolList,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("disconnectProfiles: deviceHandle = %d, act = %u", deviceHandle, act));

#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
      // TODO: @Stefan: compiler error: 'VARTRACE_USR2' was not declared in this scope
#else
      VARTRACE_USR2(protocolList);
#endif

      // no global lock because internal message is created

      BmCoreIfMessage_DisconnectProfilesRequest* bmCoreIfMessage = getNewBmCoreIfMessage_DisconnectProfilesRequest(deviceHandle,
            protocolList, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("disconnectProfiles: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::storeDeviceRingtoneIdInDb(IN const DeviceId deviceHandle, IN const RingtoneId ringtoneId,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("storeDeviceRingtoneIdInDb: deviceHandle = %d, ringtoneId = %d, act = %u",
            deviceHandle, ringtoneId, act));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult(BM_RESULT_OK);
      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(deviceHandle);

      if (0u != deviceHandle)
      {
         result = LocalSpm::getDbManager().getDeviceId(deviceId, deviceHandle);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("storeDeviceRingtoneIdInDb(deviceHandle = %d): could not get device ID from DB (error = %d)",
                  deviceHandle, ETG_CENUM(CcErrorInternal, result)));

            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }

      if (BM_RESULT_OK == bmResult)
      {
         if (true == LocalSpm::getBmCoreMainController().isFunctionalityPermitted(BM_RESTRICTION_GROUP_03))
         {
            result = LocalSpm::getDbManager().setDeviceRingtoneId(deviceId, ringtoneId);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("storeDeviceRingtoneIdInDb(deviceHandle = %d): could not store ringtone ID in DB (error = %d)",
                     deviceHandle, ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
         else
         {
            ETG_TRACE_ERR(("storeDeviceRingtoneIdInDb(deviceHandle = %d): request is currently not allowed to be processed",
                  deviceHandle));
            bmResult = BM_RESULT_ERR_NOT_ALLOWED;
         }
      }

      LocalSpm::getBmCoreCallbackIfWrapper().doStoreDeviceRingtoneIdInDbResponse(deviceHandle, bmResult);

      ETG_TRACE_USR1(("storeDeviceRingtoneIdInDb: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::readDeviceRingtoneIdFromDb(OUT RingtoneId& ringtoneId, IN const DeviceId deviceHandle)
   {
      ETG_TRACE_USR1(("readDeviceRingtoneIdFromDb: deviceHandle = %d", deviceHandle));

      BM_GLOBAL_LOCKER_APP_LAYER

      Result result(CC_ERR_INT_NO_ERROR);
      BmResult bmResult(BM_RESULT_OK);
      DeviceId deviceId(deviceHandle);

      result = LocalSpm::getDbManager().getDeviceId(deviceId, deviceHandle);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("readDeviceRingtoneIdFromDb(deviceHandle = %d): could not get deviceId from DB (error = %d)",
               deviceHandle, ETG_CENUM(CcErrorInternal, result)));
         bmResult = BM_RESULT_ERR_GENERAL;
      }

      if (BM_RESULT_OK == bmResult)
      {
         if (true == LocalSpm::getBmCoreMainController().isFunctionalityPermitted(BM_RESTRICTION_GROUP_01))
         {
            result = LocalSpm::getDbManager().getDeviceRingtoneId(ringtoneId, deviceId);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("readDeviceRingtoneIdFromDb(deviceHandle = %d): could not get ringtone ID from DB (error = %d)",
                     deviceHandle, ETG_CENUM(CcErrorInternal, result)));
               bmResult = BM_RESULT_ERR_GENERAL;
            }
         }
         else
         {
            ETG_TRACE_ERR(("readDeviceRingtoneIdFromDb(deviceHandle = %d): request is currently not allowed to be processed",
                  deviceHandle));
            bmResult = BM_RESULT_ERR_NOT_ALLOWED;
         }
      }

      LocalSpm::getBmCoreCallbackIfWrapper().doReadDeviceRingtoneIdFromDbResponse(deviceHandle, bmResult);

      ETG_TRACE_USR1(("readDeviceRingtoneIdFromDb: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::storeSystemWideRingtoneIdInDb(IN const RingtoneId ringtoneId, IN const ActType act)
   {
      ETG_TRACE_USR1(("storeSystemWideRingtoneIdInDb: ringtoneId = %d, act = %u", ringtoneId, act));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult(BM_RESULT_OK);

      if (true == LocalSpm::getBmCoreMainController().isFunctionalityPermitted(BM_RESTRICTION_GROUP_01))
      {
         Result result = LocalSpm::getDbManager().setSystemWideRingtoneId(ringtoneId);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("storeSystemWideRingtoneIdInDb: could not store system wide ringtone ID in DB (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         ETG_TRACE_ERR(("storeSystemWideRingtoneIdInDb: request is currently not allowed to be processed"));
         bmResult = BM_RESULT_ERR_NOT_ALLOWED;
      }

      LocalSpm::getBmCoreCallbackIfWrapper().doStoreSystemWideRingtoneIdInDbResponse(bmResult);

      ETG_TRACE_USR1(("storeSystemWideRingtoneIdInDb: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::readSystemWideRingtoneIdFromDb(OUT RingtoneId& ringtoneId)
   {
      ETG_TRACE_USR1(("readSystemWideRingtoneIdFromDb"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult(BM_RESULT_OK);

      if (true == LocalSpm::getBmCoreMainController().isFunctionalityPermitted(BM_RESTRICTION_GROUP_01))
      {
         Result result = LocalSpm::getDbManager().getSystemWideRingtoneId(ringtoneId);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("readSystemWideRingtoneIdFromDb: could not get system wide ringtone ID from DB (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
            bmResult = BM_RESULT_ERR_GENERAL;
         }
      }
      else
      {
         ETG_TRACE_ERR(("readSystemWideRingtoneIdFromDb: request is currently not allowed to be processed"));
         bmResult = BM_RESULT_ERR_NOT_ALLOWED;
      }

      LocalSpm::getBmCoreCallbackIfWrapper().doReadSystemWideRingtoneIdFromDbResponse(bmResult);

      ETG_TRACE_USR1(("readSystemWideRingtoneIdFromDb: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::getRemoteDeviceInfo(OUT RemoteDeviceInfo& remoteDeviceInfo, IN const DeviceId deviceHandle)
   {
      ETG_TRACE_USR1(("getRemoteDeviceInfo: deviceHandle = %d", deviceHandle));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getRemoteDeviceInfo(remoteDeviceInfo, deviceHandle,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getRemoteDeviceInfo: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::getDeviceConnectionStatusList(OUT DeviceConnectionStatusList& deviceConnectionStatusList)
   {
      ETG_TRACE_USR1(("getDeviceConnectionStatusList"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getDeviceConnectionStatusList(deviceConnectionStatusList,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getDeviceConnectionStatusList: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::blockDevice(IN const DeviceId deviceHandle, IN const DeviceHandleType deviceHandleType,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("blockDevice: deviceHandle = %d, deviceHandleType = %d, act = %u",
            deviceHandle, ETG_CENUM(DeviceHandleType, deviceHandleType), act));

      // no global lock because internal message is created

      BmCoreIfMessage_BlockDeviceRequest* bmCoreIfMessage = getNewBmCoreIfMessage_BlockDeviceRequest(deviceHandle,
            deviceHandleType, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("blockDevice: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::unblockDevice(IN const DeviceId deviceHandle, IN const DeviceHandleType deviceHandleType,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("unblockDevice: deviceHandle = %d, deviceHandleType = %d, act = %u",
            deviceHandle, ETG_CENUM(DeviceHandleType, deviceHandleType), act));

      // no global lock because internal message is created

      BmCoreIfMessage_UnblockDeviceRequest* bmCoreIfMessage = getNewBmCoreIfMessage_UnblockDeviceRequest(deviceHandle,
            deviceHandleType, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("unblockDevice: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::blockProfiles(IN const DeviceId deviceHandle, IN const ProtocolList& protocolList,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("blockProfiles: deviceHandle = %d, act = %u", deviceHandle, act));

#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
      // TODO: @Stefan: compiler error: 'VARTRACE_USR2' was not declared in this scope
#else
      VARTRACE_USR2(protocolList);
#endif

      // no global lock because internal message is created

      BmCoreIfMessage_BlockProfilesRequest* bmCoreIfMessage = getNewBmCoreIfMessage_BlockProfilesRequest(deviceHandle,
            protocolList, false, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("blockProfiles: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::unblockProfiles(IN const DeviceId deviceHandle, IN const ProtocolList& protocolList,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("unblockProfiles: deviceHandle = %d, act = %u", deviceHandle, act));

#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
      // TODO: @Stefan: compiler error: 'VARTRACE_USR2' was not declared in this scope
#else
      VARTRACE_USR2(protocolList);
#endif

      // no global lock because internal message is created

      BmCoreIfMessage_UnblockProfilesRequest* bmCoreIfMessage = getNewBmCoreIfMessage_UnblockProfilesRequest(deviceHandle,
            protocolList, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("unblockProfiles: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::getBlockStatus(OUT BlockStatus& blockStatus)
   {
      ETG_TRACE_USR1(("getBlockStatus"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getBlockStatus(blockStatus, BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getBlockStatus: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::setFavorite(IN const DeviceId deviceHandle, IN const FavoriteType favoriteType,
         IN const FavoriteIndex favoriteIndex, IN const ActType act)
   {
      ETG_TRACE_USR1(("setFavorite: deviceHandle = %d, favoriteType = %d, favoriteIndex = %u, act = %u",
            deviceHandle, ETG_CENUM(FavoriteType, favoriteType), favoriteIndex, act));

      // no global lock because internal message is created

      BmCoreIfMessage_SetFavoriteRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SetFavoriteRequest(deviceHandle,
            favoriteType, favoriteIndex, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("setFavorite: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::clearFavorite(IN const DeviceId deviceHandle, IN const FavoriteType favoriteType,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("clearFavorite: deviceHandle = %d, favoriteType = %d, act = %u",
            deviceHandle, ETG_CENUM(FavoriteType, favoriteType), act));

      // no global lock because internal message is created

      BmCoreIfMessage_ClearFavoriteRequest* bmCoreIfMessage = getNewBmCoreIfMessage_ClearFavoriteRequest(deviceHandle,
            favoriteType, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("clearFavorite: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::startServiceSearch (IN const DeviceId deviceHandle, IN const ActType act )
   {
      ETG_TRACE_USR1(("startServiceSearch: deviceHandle = %d,  act = %u",
            deviceHandle, act));

      // no global lock because internal message is created
      BmCoreIfMessage_StartServiceSearchRequest* bmCoreIfMessage = getNewBmCoreIfMessage_StartServiceSearchRequest(deviceHandle,act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("startServiceSearch: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::stopServiceSearch (IN const DeviceId deviceHandle, IN const ActType act)
   {
      ETG_TRACE_USR1(("stopServiceSearch: deviceHandle = %d,  act = %u",
            deviceHandle, act));

      // no global lock because internal message is created
      BmCoreIfMessage_StopServiceSearchRequest* bmCoreIfMessage = getNewBmCoreIfMessage_StopServiceSearchRequest(deviceHandle,act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("stopServiceSearch: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;

   }
   BmResult BmInterface::setAutoConnectionType(IN const AutoConnectionType autoConnectionType,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("setAutoConnectionType: autoConnectionType = %d, act = %u", autoConnectionType._type, act));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().setAutoConnectionType(autoConnectionType, act);

      ETG_TRACE_USR1(("setAutoConnectionType: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::getAutoConnectionType(OUT AutoConnectionType& autoConnectionType)
   {
      ETG_TRACE_USR1(("getAutoConnectionType"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getAutoConnectionType(autoConnectionType,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getAutoConnectionType: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::startAutoConnectSequence(IN const TemporaryAutoConnectionType autoConnectionType,
         IN const ActType act)
   {
      ETG_TRACE_USR1(("startAutoConnectSequence: autoConnectionType = %d, act = %u",
            ETG_CENUM(TemporaryAutoConnectionType, autoConnectionType), act));

      // no global lock because internal message is created

      BmCoreIfMessage_StartAutoConnectionSequenceRequest* bmCoreIfMessage = getNewBmCoreIfMessage_StartAutoConnectionSequenceRequest(autoConnectionType,
            act, BM_CORE_IF_MSG_ORIGIN_CLIENT);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("startAutoConnectSequence: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::stopAutoConnectSequence(IN const ActType act)
   {
      ETG_TRACE_USR1(("stopAutoConnectSequence: act = %u", act));

      // no global lock because internal message is created

      BmCoreIfMessage_StopAutoConnectionSequenceRequest* bmCoreIfMessage = getNewBmCoreIfMessage_StopAutoConnectionSequenceRequest(act,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("stopAutoConnectSequence: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::getBmConfigData(OUT BmConfigData& bmConfigData)
   {
      ETG_TRACE_USR1(("getBmConfigData"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getBmConfigData(bmConfigData,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getBmConfigData: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   void BmInterface::updateLinkQuality(IN const ActType act)
   {
      ETG_TRACE_USR1(("updateLinkQuality: act = %u", act));

      // no global lock because internal message is created

      BmCoreIfMessage_UpdateLinkQualityRequest* bmCoreIfMessage = getNewBmCoreIfMessage_UpdateLinkQualityRequest(act,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("updateLinkQuality: bmCoreIfMessage is 0"));
         return;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);
   }

   BmResult BmInterface::getLinkQuality(OUT LinkQuality& linkQuality)
   {
      ETG_TRACE_USR1(("getLinkQuality"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getLinkQuality(linkQuality,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getLinkQuality: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   void BmInterface::setBtLimitationMode(IN const BdAddress& bdAddress, IN const BdName& bdName, IN const LimitationMode& limitationMode,
         IN const LimitationAction limitationAction, IN const ActType act)
   {
      ETG_TRACE_USR1(("setBtLimitationMode: bdAddress = %50s, bdName = %50s, limitationMode.feature = %d, limitationMode.commIf = %d, limitationAction = %d, act = %u",
            bdAddress.c_str(), bdName.c_str(), ETG_CENUM(LimitationFeature, limitationMode._limitationFeature),
            ETG_CENUM(LimitationCommunicationIf, limitationMode._limitationCommunicationIf),
            ETG_CENUM(LimitationAction, limitationAction), act));

      // no global lock because internal message is created

      BmCoreIfMessage_SetBtLimitationModeRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SetBtLimitationModeRequest(act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("setBtLimitationMode: bmCoreIfMessage is 0"));
      }//!End of if (0 == bmCoreIfMessage)
      else
      {
         bmCoreIfMessage->setBdAddress(bdAddress);
         bmCoreIfMessage->setBdName(bdName);
         bmCoreIfMessage->setLimitationMode(limitationMode);
         bmCoreIfMessage->setLimitationAction(limitationAction);

         LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);
      }//!End of else
   }//!End of void BmInterface::setBtLimitationMode(IN const BdAddress& bdAddress,...)

   BmResult BmInterface::getBtLimitationMode(OUT BtLimitationMode& btLimitationMode)
   {
      ETG_TRACE_USR1(("getBtLimitationMode"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getBtLimitationMode(btLimitationMode,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getBtLimitationMode: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   void BmInterface::replaceBtLimitationMode(IN const BtLimitationModeInfo& limitationModeToBeReplaced,
         IN const BdAddress& newBdAddress, IN const BdName& newBdName, IN const LimitationMode& newLimitationMode,
         IN const LimitationAction newLimitationAction, IN const ActType act)
   {
      ETG_TRACE_USR1(("replaceBtLimitationMode: limitationModeToBeReplaced: bdAddress = %50s, bdName = %50s, limitationFeature = %d, limitationCommunicationIf = %d"
            "limitationState = %d, act = %u", limitationModeToBeReplaced._bdAddress.c_str(), limitationModeToBeReplaced._bdName.c_str(),
            ETG_CENUM(LimitationFeature, limitationModeToBeReplaced._limitationMode._limitationFeature),
            ETG_CENUM(LimitationCommunicationIf, limitationModeToBeReplaced._limitationMode._limitationCommunicationIf),
            ETG_CENUM(LimitationState, limitationModeToBeReplaced._limitationState), act));

      ETG_TRACE_USR1(("replaceBtLimitationMode: newBdAddress = %50s, newBdName = %50s, newLimitationFeature = %d, newLimitationCommunicationIf = %d, "
            "newLimitationAction = %d", newBdAddress.c_str(), newBdName.c_str(), ETG_CENUM(LimitationFeature, newLimitationMode._limitationFeature),
            ETG_CENUM(LimitationCommunicationIf, newLimitationMode._limitationCommunicationIf),
            ETG_CENUM(LimitationAction, newLimitationAction)));

      // no global lock because internal message is created

      BmCoreIfMessage_ReplaceBtLimitationModeRequest* bmCoreIfMessage = getNewBmCoreIfMessage_ReplaceBtLimitationModeRequest(act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("replaceBtLimitationMode: bmCoreIfMessage is 0"));
      }//!End of if (0 == bmCoreIfMessage)
      else
      {
         bmCoreIfMessage->setLimitationModeToBeReplaced(limitationModeToBeReplaced);
         bmCoreIfMessage->setNewBdAddress(newBdAddress);
         bmCoreIfMessage->setNewBdName(newBdName);
         bmCoreIfMessage->setNewLimitationMode(newLimitationMode);
         bmCoreIfMessage->setNewLimitationAction(newLimitationAction);

          LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);
      }//!End of else
   }//!End of void BmInterface::replaceBtLimitationMode(IN const BtLimitationModeInfo& limitationModeToBeReplaced, ...)

   void BmInterface::notifyUserDecision(IN const UserDecisionInfo userDecisionInfo, IN const ActType act)
   {
      ETG_TRACE_USR1(("notifyUserDecision: userDecisionInfo = (conflictTrigger = %d, userDecision = %d), act = %u",
            ETG_CENUM(ConflictTrigger, userDecisionInfo._conflictTrigger),
            ETG_CENUM(UserDecision, userDecisionInfo._userDecision), act));

      // no global lock because internal message is created

      BmCoreIfMessage_NotifyUserDecisionRequest* bmCoreIfMessage = getNewBmCoreIfMessage_NotifyUserDecisionRequest(act);

      if (0 != bmCoreIfMessage)
      {
          bmCoreIfMessage->setUserDecisionInfo(userDecisionInfo);

          LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_FATAL(("notifyUserDecision: bmCoreIfMessage is NULL"));
      }
   }

   void BmInterface::setBtProfileUsageInfo(IN const ProfileUsageList& profileUsageInfoList, IN const ActType act)
   {
      ETG_TRACE_USR1(("BmInterface::setBtProfileUsageInfo Entered"));

      // no global lock because internal message is created

      VARTRACE(profileUsageInfoList);

      BmCoreIfMessage_SetBtProfilesUsageRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SetBtProfilesUsageRequest(act);

      if (0 != bmCoreIfMessage)
      {
          bmCoreIfMessage->setProfileUsageInfoList(profileUsageInfoList);

          LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_FATAL(("setBtProfileUsageInfo: bmCoreIfMessage is NULL"));
      }
   }

   BmResult BmInterface::resetToDefault(IN const bool factorySettingsChanged, IN const ActType act)
   {
      ETG_TRACE_USR1(("resetToDefault: factorySettingsChanged = %10s, act = %u",
            factorySettingsChanged ? "true" : "false", act));

      // no global lock because internal message is created

      BmCoreIfMessage_ResetToDefaultRequest* bmCoreIfMessage = getNewBmCoreIfMessage_ResetToDefaultRequest(factorySettingsChanged,
            act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("resetToDefault: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::getResetToDefaultStatus(OUT ResetToDefaultStatus& resetToDefaultStatus)
   {
      ETG_TRACE_USR1(("getResetToDefaultStatus"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getResetToDefaultStatus(resetToDefaultStatus,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getResetToDefaultStatus: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::switchTestMode(IN const TargetSwitchState& targetSwitchState,
         IN const BdAddress& bdAddress, IN const ActType act)
   {
      ETG_TRACE_USR1(("switchTestMode: targetSwitchState = %d, bdAddress = \"%50s\", act = %u",
            ETG_CENUM(TargetSwitchState, targetSwitchState), bdAddress.c_str(), act));

      // no global lock because internal message is created

      BmCoreIfMessage_SwitchTestModeRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SwitchTestModeRequest(targetSwitchState,
            bdAddress, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("switchTestMode: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::getTestMode(OUT SwitchStatus& testMode)
   {
      ETG_TRACE_USR1(("getTestMode"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getTestMode(testMode,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getTestMode: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::requestTestModeLinkQuality(IN const ActType act)
   {
      ETG_TRACE_USR1(("requestTestModeLinkQuality: act = %u", act));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmController().doRequestTestModeLinkQuality();

      ETG_TRACE_USR1(("requestTestModeLinkQuality: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::getLinkQualityTestMode(OUT TestModeLinkQuality& linkQuality)
   {
      ETG_TRACE_USR1(("getLinkQualityTestMode"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().doGetLinkQualityTestMode(linkQuality);

      ETG_TRACE_USR1(("getLinkQualityTestMode: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;
   }

   BmResult BmInterface::setPrimaryHfpDevice(IN const DeviceId deviceHandle)
   {
      ETG_TRACE_USR1(("setPrimaryHfpDevice: deviceHandle = %d", deviceHandle));

      // no global lock because internal message is created

      BmCoreIfMessage_SetPrimaryHfpDeviceRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SetPrimaryHfpDeviceRequest(deviceHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("setPrimaryHfpDevice: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::switchMultiHFPSupport(IN const TargetSwitchState targetSwitchState, IN const ActType act)
   {
      ETG_TRACE_USR1(("switchMultiHFPSupport: targetSwitchState = %d, act = %u", ETG_CENUM(TargetSwitchState, targetSwitchState), act));

      // no global lock because internal message is created

      BmCoreIfMessage_SwitchMultiHFPSupportRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SwitchMultiHFPSupportRequest(targetSwitchState, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("switchMultiHFPSupport: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::getMultiHFPSupport(OUT SwitchStatus& multiHFPSupport)
   {
      ETG_TRACE_USR1(("getMultiHFPSupport"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getMultiHFPSupportStatus(multiHFPSupport,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getMultiHFPSupport: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;

   }

   BmResult BmInterface::setDeviceUsagePreference(IN const DeviceId deviceHandle, IN const UsagePreference usagePreference, IN const ActType act)
   {
      ETG_TRACE_USR1(("setDeviceUsagePreference: devicehandle = %d, usagePreference = %d, act = %u", deviceHandle,
            ETG_CENUM(UsagePreference, usagePreference), act));

      // no global lock because internal message is created

      BmCoreIfMessage_SetDeviceUsagePreferenceRequest* bmCoreIfMessage = getNewBmCoreIfMessage_SetDeviceUsagePreferenceRequest(deviceHandle, usagePreference, act);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("setDeviceUsagePreference: bmCoreIfMessage is 0"));
         return BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

      return BM_RESULT_OK;
   }

   BmResult BmInterface::getDeviceUsagePreference(OUT UsagePreference& usagePreference, IN const DeviceId deviceHandle)
   {
      ETG_TRACE_USR1(("getDeviceUsagePreference"));

      BM_GLOBAL_LOCKER_APP_LAYER

      BmResult bmResult = LocalSpm::getBmCoreMainController().getDeviceUsagePreference(usagePreference, deviceHandle,
            BM_CORE_IF_MSG_ORIGIN_CLIENT);

      ETG_TRACE_USR1(("getDeviceUsagePreference: bmResult = %d", ETG_CENUM(BmResult, bmResult)));

      return bmResult;

   }
}
