/**
 * @file BtStackIfConnectionRequestIfWrapper.h
 * 
 * @swcomponent BluetoothConnectionManagerCore
 *
 * @brief This file contains the declaration of the class BtStackIfConnectionRequestIfWrapper
 * 
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup BmControllerModule
 *
 */

#ifndef _BT_STACK_IF_CONNECTION_REQUEST_IF_WRAPPER_H_
#define _BT_STACK_IF_CONNECTION_REQUEST_IF_WRAPPER_H_

#include "BtStackIfTypes.h"

using namespace ::btstackif;

// class forward declaration
namespace btstackif
{
   class IBtStackIfConnectionRequest;
   class IBtStackIfConnectionCallback;
} // btstackif

namespace bmcore
{
   /**
    * @class BtStackIfConnectionRequestIfWrapper
    * @ingroup BmControllerModule
    * 
    * @brief
    * 
    */
   class BtStackIfConnectionRequestIfWrapper
   {
   public:

       /**
        * @brief Constructor of class BtStackIfConnectionRequestIfWrapper which creates an instance
        *
        * @return void
        */
       explicit BtStackIfConnectionRequestIfWrapper();

       /**
        * @brief Destructor of class BtStackIfConnectionRequestIfWrapper
        *
        * @return void
        */
       virtual ~BtStackIfConnectionRequestIfWrapper();

      /**************************************************************************
       General interfaces
       **************************************************************************/
      /**
       * Register callback instance.
       *
       * @param[in] callback: callback instance
       */
      void registerCallback(IBtStackIfConnectionCallback* callback);

      /**
       * Requests a session handle to be used for communication between BtStackIf user and BtStackIf library itself.
       * The session handle provides the possibility to distinguish different BtStackIf users while accessing same BtStackIf request interface.
       * The session handle 0 indicates a common interface call without session relation.
       *
       * @return = session handle, always > 0
       */
      BTSSessionHandle getSessionHandle(void) const;

      /**
       * Initialize the BtStackIf library.
       * - opens/prepares DBUS communication
       * - starts internal threads
       *
       * @param[in] configuration: local stack configuration data
       * @param[in] pairedDevices: list of paired Bluetooth devices
       *
       * @return != BTS_OK: error,
       * @return  = BTS_OK: OK
       */
      BTSErrorCode init(IN const BTSLocalStackConfiguration& configuration, IN const BTSBDAddressList& pairedDevices) const;

      /**
       * Deinitialize the BtStackIf library.
       * - stops internal threads
       * - closes DBUS communication
       *
       * @return != BTS_OK: error,
       * @return  = BTS_OK: OK
       */
      BTSErrorCode deInit(void) const;

      /**************************************************************************
       Connection management related interfaces
       **************************************************************************/
      void switchBluetoothOnOff(IN const BTSBluetoothMode bluetoothMode, IN const BTSAdapterMode adapterMode);

      void getBluetoothOnOffStatus(void);

      void switchBluetoothHardwareOnOff(IN const BTSBluetoothMode bluetoothMode);

      void setLocalBtName(IN const BTSDeviceName& adapterName);

      void getLocalBtName(void);

      void setDeviceIdServiceRecord(IN const BTSDeviceIdServiceRecord& serviceRecord);

      void setDeviceForRestrictedPairingConnecting(IN const BTSBDAddress& expectedDeviceAddress, IN const BTSSessionHandle sessionHandle = 0);

      void setLocalAdapterModes(IN const BTSLocalMode discoverableMode, IN const BTSDiscoverableTimeout discoverableTimeout, IN const BTSLocalMode connectableMode, IN const BTSConnectableTimeout connectableTimeout, IN const BTSSessionHandle sessionHandle = 0);

      void getLocalAdapterModes(IN const BTSSessionHandle sessionHandle = 0);

      void startDiscovery(IN const BTSSessionHandle sessionHandle = 0);

      void stopDiscovery(IN const BTSSessionHandle sessionHandle = 0);

      void getDiscoveringStatus(IN const BTSSessionHandle sessionHandle = 0);

      void startPairing(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0);

      void cancelPairing(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0);

      void setPinCode(IN const BTSBDAddress& deviceAddress, IN const BTSPinCode& pinCode, IN const BTSSessionHandle sessionHandle = 0);

      void confirmSecureSimplePairing(IN const BTSBDAddress& deviceAddress, IN const BTSSspMode sspMode, IN const BTSNumericValue& numericValue, IN const BTSConfirmationMode confirmationMode, IN const BTSSessionHandle sessionHandle = 0);

      void setLinkKey(IN const BTSBDAddress& deviceAddress, IN const BTSLinkKeyType linkKeyType, IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey, IN const BTSSessionHandle sessionHandle = 0);

      void connectProtocol(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const BTSTimeValue pageTimeout = 10240, IN const BTSSessionHandle sessionHandle = 0);

      void acceptRemoteProtocolConnect(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSSessionHandle sessionHandle = 0);

      void rejectRemoteProtocolConnect(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSSessionHandle sessionHandle = 0);

      void disconnectProtocol(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool pauseBtStreaming = true, IN const BTSSessionHandle sessionHandle = 0);

      void disconnectDevice(IN const BTSBDAddress& deviceAddress, IN const bool pauseBtStreaming = true, IN const BTSSessionHandle sessionHandle = 0);

      void getProtocolConnectionStatus(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const BTSSessionHandle sessionHandle = 0);

      void getDeviceConnectionStatus(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0);

      void startRemoteServiceSearch(IN const BTSBDAddress& deviceAddress, IN const BTSSearchType searchType, IN const bool rejectIfNotConnected = true, IN const BTSSessionHandle sessionHandle = 0);

      void cancelRemoteServiceSearch(IN const BTSBDAddress& deviceAddress, IN const BTSSearchType searchType, IN const BTSSessionHandle sessionHandle = 0);

      void queryRemoteName(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0);

      void requestLinkQuality(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0);

      void startTestMode(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0);

      void stopTestMode(IN const BTSSessionHandle sessionHandle = 0);

      void requestTestModeLinkQuality(IN const BTSSessionHandle sessionHandle = 0);

   private:

      BtStackIfConnectionRequestIfWrapper& operator=(const BtStackIfConnectionRequestIfWrapper &other);

      BtStackIfConnectionRequestIfWrapper(const BtStackIfConnectionRequestIfWrapper &other);

      IBtStackIfConnectionRequest* _btStackIfConnectionRequestIf;
   };
}

#endif // _BT_STACK_IF_CONNECTION_REQUEST_IF_WRAPPER_H_

