#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_DCCONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DeviceConnectionController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_DCCONTROLLER
#endif
#endif

#include "LocalSpm.h"
#include "FunctionTracer.h"
#include "Dispatcher.h"
#include "DeviceConnectionController.h"
#include "RequestResponseSM.h"
#include "ProtocolConnectionController.h"
#include "BmVarTrace.h"
#include "BmUtils.h"
#include <algorithm>
#include "IBtStackIfConnectionRequest.h"
#include "BmGlobalLock.h"
#include "BmCoreIfMessagesCreator.h"

namespace bmcore
{
   DeviceConnectionController::DeviceConnectionController() :
         _deviceId(0u), _deviceToBeDeleted(false), _reservedForCpw(false), _delayLocalProtocolConnections(false),
         _disconnectionLocallyInitiated(false), _deviceLost(false), _allowDisconnectedReasonUpdate(true),
         _allProtocolsDisconnected(true), _deviceDisconnected(true),
         _disconnectedReason(BM_DISCONNECTED_REASON_UNKNOWN), _origin(BM_CONNECTION_REQUEST_ORIGIN_UNKNOWN), _delayConnectionRetryTimer(),
         _delayConnectionRetryTimerId(), _delayDisconnectionRetryTimer(), _delayDisconnectionRetryTimerId(),
         _expiredTimerSmEvent(this->GetSMNameFull()), _protocolsPendingToConnect(),
         _localProtocolTargetConnectionStatusMap(), _protocolConnectionsLock(), _oobPairedDeviceRequest(), _oobPairedDeviceRequestReceived(false), _pageTimeout(10240u)
   {
      ETG_TRACE_USR1(("DeviceConnectionController(this = 0x%p, deviceId = %d): being created",
            (void *) this, _deviceId));

      // threadless SM
      disableAllLocks(true);
   }

   DeviceConnectionController::~DeviceConnectionController()
   {
      ETG_TRACE_USR1(("~DeviceConnectionController(this = 0x%p, deviceId = %d): being destroyed",
            (void *) this, _deviceId));

      _delayConnectionRetryTimerId = 0;
      _delayDisconnectionRetryTimerId = 0;
   }

   Result DeviceConnectionController::assign(const DeviceId deviceId)
   {
      ENTRY

      ETG_TRACE_USR1(("assign(this = 0x%p, deviceId = %d): deviceId = %d", (void *) this, _deviceId, deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      if (0u == _deviceId)
      {
         _deviceId = deviceId;

         ETG_TRACE_USR1(("assign(this = 0x%p, deviceId = %d): assigning DCC instance to given device", (void *) this, _deviceId));

         result = LocalSpm::getDbManager().setDeviceConnectionController(_deviceId, this);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("assign(this = 0x%p, deviceId = %d): setting DCC instance assignment in DB failed (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         result = CC_ERR_INT_DCC_ALREADY_ASSIGNED;
         ETG_TRACE_ERR(("assign(this = 0x%p, deviceId = %d): DCC instance is already assigned to a device (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result DeviceConnectionController::release()
   {
      ENTRY

      ETG_TRACE_USR1(("release(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      if (0u != _deviceId)
      {
         if(true == _reservedForCpw)
         {
            BtLimitationModeInfo btLimitationModeInfo;
            result = LocalSpm::getBmCoreMainController().getBtLimitationModeInfo(btLimitationModeInfo, _deviceId);

            if(CC_ERR_INT_NO_ERROR == result)
            {
               if((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == btLimitationModeInfo._limitationMode._limitationCommunicationIf)
                     ||(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == btLimitationModeInfo._limitationMode._limitationCommunicationIf))
               {
                  if((BM_LIMITATION_FEATURE_CAR_PLAY == btLimitationModeInfo._limitationMode._limitationFeature) &&
                        (BM_LIMITATION_STATE_IDLE == btLimitationModeInfo._limitationState))

                  {
                     _reservedForCpw = false;
                  }
               }
            }
            else if(CC_ERR_INT_ITEM_NOT_FOUND == result)
            {
               _reservedForCpw = false;
            }
         }

         if (false == _reservedForCpw)
         {
            ETG_TRACE_USR1(("release(this = 0x%p, deviceId = %d): releasing DCC instance", (void *) this, _deviceId));

            result = LocalSpm::getDbManager().setDeviceConnectionController(_deviceId, static_cast<DeviceConnectionController*>(0));

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("release(this = 0x%p, deviceId = %d): setting DCC instance's release in DB failed (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
            }

            (void) initMembers();

            LocalSpm::getBmController().onDeviceConnectionControllerReleased(this);
         }
         else
         {
            ETG_TRACE_USR1(("release(this = 0x%p, deviceId = %d): DCC instance is still reserved for handling CPW",
                  (void *) this, _deviceId));

            if(true == _deviceToBeDeleted)
            {
               // DeActivate the BtLimitationMode(CPW)
               LocalSpm::getBmCoreMainController().deactivateBtLimitationModeCPW(_deviceId);
            }
         }
      }
      else
      {
         result = CC_ERR_INT_DCC_NOT_ASSIGNED;
         ETG_TRACE_ERR(("release(this = 0x%p, deviceId = %d): DCC instance is currently not assigned to a device (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result DeviceConnectionController::checkAndReleasePccForSlaveProtocols(bool disconnectOnRequest, ProtocolId protocolId)
   {
      ENTRY

      ETG_TRACE_USR1(("checkAndReleasePccForSlaveProtocols(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      if (0u != _deviceId)
      {
         ProtocolConnectionControllerList pccInstances;

         // Get all the PCC controllers assigned to this device
         result = LocalSpm::getDbManager().getProtocolConnectionControllers(pccInstances, _deviceId);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR1(("checkAndReleasePccForSlaveProtocols(this = 0x%p, deviceId = %d): number of PCC instances still assigned to device: %d", (void *) this, _deviceId, pccInstances.size()));

            if (true == pccInstances.empty())
            {
               ETG_TRACE_USR4(("checkAndReleasePccForSlaveProtocols(this = 0x%p, deviceId = %d): sending event ALL_PROTOCOLS_DISCONNECTED to myself", (void *) this, _deviceId));
               result = this->SendEventByName("ALL_PROTOCOLS_DISCONNECTED", 0);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("checkAndReleasePccForSlaveProtocols(this = 0x%p, deviceId = %d): could not send event ALL_PROTOCOLS_DISCONNECTED (error = %d)",
                        (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               VARTRACE(pccInstances);

               ProtocolConnectionControllerList pccInstancesSlaves;
               ProtocolIdList protocolIdsSlave;

               (void) LocalSpm::getDataProvider().getSlaveProtocolIds(protocolIdsSlave);
               VARTRACE(protocolIdsSlave);

               // Push all the slave protocols PCC instance into a list
               for (size_t pccIdx = 0u; pccIdx < pccInstances.size(); ++pccIdx)
               {
                  if (std::find(protocolIdsSlave.begin(), protocolIdsSlave.end(), pccInstances[pccIdx]->getProtocolId())
                        != protocolIdsSlave.end())
                  {
                     pccInstancesSlaves.push_back(pccInstances[pccIdx]);
                  }
               }

               VARTRACE(pccInstancesSlaves);

               ProtocolIdList protocolIdsMaster;
               bool disconnectSlaveProtocol(true);
               ProtocolConnectionControllerList pccInstancesToBeDisconnected;

               for (size_t pccSlaveIdx = 0u; pccSlaveIdx < pccInstancesSlaves.size(); ++pccSlaveIdx)
               {
                  disconnectSlaveProtocol = true;

                  // Get the master protocolsId for the slave protocols
                  if (CC_ERR_INT_NO_ERROR == LocalSpm::getDataProvider().getMasterProtocolIds(protocolIdsMaster, pccInstancesSlaves[pccSlaveIdx]->getProtocolId()))
                  {
                     VARTRACE(protocolIdsMaster);

                     for (size_t pccIdx = 0u; pccIdx < pccInstances.size(); ++pccIdx)
                     {
                        // Check the request is received from protocol disconnection.
                        if(false == disconnectOnRequest)
                        {
                           // Set disconnectSlaveProtocol flag as false if the Master ProtocolId PCC instacne is available in the PCC controllers list
                           if (std::find(protocolIdsMaster.begin(), protocolIdsMaster.end(), pccInstances[pccIdx]->getProtocolId())
                                 != protocolIdsMaster.end())
                           {
                              //do not sent disconnect to pccInstancesSlaves[pccSlaveIdx]
                              disconnectSlaveProtocol = false;
                              break;
                           }
                        }
                        else
                        {
                           // Set disconnectSlaveProtocol flag as false if the Master ProtocolId is not matched with the requested protocolId
                           // exception of SPP protocol
                           if(BM_PROTOCOL_ID_SPP != pccInstancesSlaves[pccSlaveIdx]->getProtocolId())
                           {
                              if (std::find(protocolIdsMaster.begin(), protocolIdsMaster.end(), protocolId)
                              == protocolIdsMaster.end())
                              {
                                 //do not sent disconnect to pccInstancesSlaves[pccSlaveIdx]
                                 disconnectSlaveProtocol = false;
                                 break;
                              }
                           }
                           else
                           {
                              disconnectSlaveProtocol = false;
                              break;
                           }
                        }
                     }

                     if(true == disconnectSlaveProtocol)
                     {
                        if(true == (LocalSpm::getBmController().checkSupportedUUID(_deviceId,
                              pccInstancesSlaves[pccSlaveIdx]->getProtocolId(), pccInstancesSlaves[pccSlaveIdx]->getUuid(), BM_CONNECTION_REQUEST_ORIGIN_LOCAL, true)))
                        {
                           disconnectSlaveProtocol = false;
                        }
                     }
                  }
                  else
                  {
                     //error: do not sent disconnect to pccInstancesSlaves[pccSlaveIdx]
                     disconnectSlaveProtocol = false;
                  }

                  if (true == disconnectSlaveProtocol)
                  {
                     pccInstancesToBeDisconnected.push_back(pccInstancesSlaves[pccSlaveIdx]);
                  }
               }

               if (false == pccInstancesToBeDisconnected.empty())
               {
                  ETG_TRACE_USR1(("checkAndReleasePccForSlaveProtocols(this = 0x%p, deviceId = %d): initiating disconnection of slave protocols:",
                        (void *) this, _deviceId));
                  VARTRACE(pccInstancesToBeDisconnected);
/*
                  if (pccInstancesToBeDisconnected.size() == pccInstances.size())
                  {
                     // the DISCONNECT event will be sent to all currently assigned PCC instances,
                     // the DCC SM has to be put in DISCONNECTING state

                     ETG_TRACE_USR4(("checkAndReleasePccForSlaveProtocols(this = 0x%p, deviceId = %d): sending event DISCONNECT to myself",
                           (void *) this, _deviceId));
                     result = this->SendEventByName("DISCONNECT", 0);

                     if (CC_ERR_INT_NO_ERROR != result)
                     {
                        ETG_TRACE_ERR(("checkAndReleasePccForSlaveProtocols(this = 0x%p, deviceId = %d): could not send event DISCONNECT (error = %d)",
                              (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
                     }
                  }
*/
                  (void) this->initiateProtocolDisconnections(pccInstancesToBeDisconnected, BM_DISCONNECTED_REASON_NORMAL_LOSS_LOCAL);
               }
               else
               {
                  ETG_TRACE_USR1(("checkAndReleasePccForSlaveProtocols(this = 0x%p, deviceId = %d): no slave protocols to be disconnected",
                        (void *) this, _deviceId));
               }
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkAndReleasePccForSlaveProtocols(this = 0x%p, deviceId = %d): getting all PCC instances assigned to the device failed (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         result = CC_ERR_INT_DCC_NOT_ASSIGNED;
         ETG_TRACE_ERR(("checkAndReleasePccForSlaveProtocols(this = 0x%p, deviceId = %d): DCC instance is currently not assigned to a device (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   void DeviceConnectionController::create()
   {
      ENTRY

      ETG_TRACE_USR1(("create(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      /* Create the state machine */
      this->Create();
   }

   Result DeviceConnectionController::init()
   {
      ENTRY

      ETG_TRACE_USR1(("init(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      /* Init the state machine */
      this->Init();
      SetAnswerTimeout(2000);

      /* Register state machine with dispatcher */
      Dispatcher::GetInstance().Register(IN this);

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::run()
   {
      ENTRY

      ETG_TRACE_USR1(("run(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::stop()
   {
      ENTRY

      ETG_TRACE_USR1(("stop(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      ETG_TRACE_USR4(("stop(this = 0x%p, deviceId = %d): sending event STOP_SM to myself",
            (void *) this, _deviceId));

      Result result = this->SendForceEvent(STOP_SM, (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("stop(this = 0x%p, deviceId = %d): could not send event STOP_SM to myself (error = %d)",
               (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result DeviceConnectionController::done()
   {
      ENTRY

      ETG_TRACE_USR1(("done(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      /* Deregister state machine with dispatcher */
      Dispatcher::GetInstance().DeRegister(IN this);

      /* Send DONE message to own SM */
      result = SendForceEvent(DeviceConnectionController::DONE, (char *) NULL);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("done(this = 0x%p, deviceId = %d): could not send event DONE (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result DeviceConnectionController::requestProtocolConnections(IN const ProtocolList& protocolList,
         IN const PageTimeout pageTimeout, IN const ConnectionRequestOrigin connectionRequestOrigin)
   {
      ENTRY

      Locker locker(&_protocolConnectionsLock);

      ETG_TRACE_USR1(("requestProtocolConnections(this = 0x%p, deviceId = %d): pageTimeout = %d, connectionRequestOrigin = %d",
            (void *) this, _deviceId, pageTimeout, ETG_CENUM(ConnectionRequestOrigin, connectionRequestOrigin)));
      VARTRACE(protocolList);

      Result result(CC_ERR_INT_NO_ERROR);

      if (0u != _deviceId)
      {
         ProtocolList supportedProtocols;

         if ((BM_CONNECTION_REQUEST_ORIGIN_LOCAL != connectionRequestOrigin)
               && (BM_CONNECTION_REQUEST_ORIGIN_REMOTE != connectionRequestOrigin))
         {
            ETG_TRACE_ERR(("requestProtocolConnections(this = 0x%p, deviceId = %d): connection request originator is unknown", (void *) this, _deviceId));
            return CC_ERR_INT_INVALID_PARAMETER;
         }

         //!Not allowed connection requests for a device when deletedevice is ongoing.
         if(true == _deviceToBeDeleted)
         {
            ETG_TRACE_ERR(("requestProtocolConnections(this = 0x%p, deviceId = %d): delete request is in progress", (void *) this, _deviceId));
            return CC_ERR_INT_BUSY;
         }

         ProtocolInfoMap localSupportedProtocols;
         (void) LocalSpm::getDataProvider().getProtocolInfos(localSupportedProtocols, true, false, false);

         ProtocolInfoMap::iterator it;

         for(it = localSupportedProtocols.begin(); it != localSupportedProtocols.end(); ++it)
         {
            supportedProtocols.push_back(Protocol(it->first, ""));
         }

         ProtocolConnectionController* pccInstance(0);
         ProtocolConnectionControllerList pccInstances;

         for (size_t i = 0u; i < protocolList.size(); i++)
         {
            // check if the given protocol is locally supported

            bool protocolSupported(false);

            for (size_t supProtIdx = 0u; supProtIdx < supportedProtocols.size(); supProtIdx++)
            {
               if (protocolList[i]._protocolId == supportedProtocols[supProtIdx]._protocolId)
               {
                  protocolSupported = true;
                  break;
               }
            }

            if (true == protocolSupported)
            {
               // given protocol is locally supported

               if (protocolList[i]._protocolId == BM_PROTOCOL_ID_SPP)
               {
                  if (false == isValidUuid(protocolList[i]._uuid))
                  {
                     result = CC_ERR_INT_INVALID_PARAMETER;
                     ETG_TRACE_ERR(("requestProtocolConnections(this = 0x%p, deviceId = %d): given UUID (= \"%50s\") is not valid (error = %d)",
                           (void *) this, _deviceId, protocolList[i]._uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
                  }
               }

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  BlockState protocolBlockState(BM_BLOCK_STATE_UNBLOCKED);

                  if (BM_PROTOCOL_ID_SPP != protocolList[i]._protocolId)
                  {
                     // consider protocol block state only for non-SPP protocols
                     result = LocalSpm::getDbManager().getProtocolBlockState(protocolBlockState,
                           _deviceId, protocolList[i]);

                     if (CC_ERR_INT_NO_ERROR != result)
                     {
                        ETG_TRACE_ERR(("requestProtocolConnections(this = 0x%p, deviceId = %d): could not get protocol's block state (error = %d), assuming protocol being unblocked",
                              (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
                     }
                  }

                  if (BM_BLOCK_STATE_UNBLOCKED == protocolBlockState)
                  {
                     result = this->assignProtocolConnectionController(&pccInstance, protocolList[i]._protocolId,
                           protocolList[i]._uuid, pageTimeout);

                     if (CC_ERR_INT_NO_ERROR == result)
                     {
                        pccInstances.push_back(pccInstance);
                     }
                     else
                     {
                        ETG_TRACE_ERR(("requestProtocolConnections(this = 0x%p, deviceId = %d): could not assign a PCC instance for protocolId = %d (error = %d)",
                              (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolList[i]._protocolId),
                              ETG_CENUM(CcErrorInternal, result)));
                     }
                  }
                  else
                  {
                     result = CC_ERR_INT_PROTOCOL_BLOCKED;
                     ETG_TRACE_ERR(("requestProtocolConnections(this = 0x%p, deviceId = %d): protocol (protocolId = %d, uuid = \"%50s\") is currently blocked (error = %d)",
                           (void *) this, _deviceId, protocolList[i]._protocolId, protocolList[i]._uuid.c_str(),
                           ETG_CENUM(CcErrorInternal, result)));
                  }
               }
            }
            else
            {
               // given protocol is not locally supported
               result = CC_ERR_INT_PROTOCOL_NOT_SUPPORTED;
               ETG_TRACE_ERR(("requestProtocolConnections(this = 0x%p, deviceId = %d): given protocol (protocol ID = %d, uuid = \"%50s\") is not supported by %50s device (error = %d)",
                     (void *) this, _deviceId, protocolList[i]._protocolId, protocolList[i]._uuid.c_str(),
                     (BM_CONNECTION_REQUEST_ORIGIN_LOCAL == connectionRequestOrigin) ? "remote" : "local",
                     ETG_CENUM(CcErrorInternal, result)));
            }

            result = CC_ERR_INT_NO_ERROR;
         } // for (size_t i = 0u; i < protocolList.size(); i++)

         if (false == pccInstances.empty())
         {
            result = initiateProtocolConnections(pccInstances, connectionRequestOrigin);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               ETG_TRACE_USR1(("requestProtocolConnections(this = 0x%p, deviceId = %d): all connection requests could be successfully processed", (void *) this, _deviceId));
            }
            else if (CC_ERR_INT_DCC_ONLY_PARTIALLY_HANDLED == result)
            {
               ETG_TRACE_USR1(("requestProtocolConnections(this = 0x%p, deviceId = %d): some connection requests could not be successfully processed (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
               result = CC_ERR_INT_NO_ERROR;
            }
            else
            {
               ETG_TRACE_ERR(("requestProtocolConnections(this = 0x%p, deviceId = %d): no connection request could be successfully processed (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            result = CC_ERR_INT_PCC_NO_RESOURCES;
            ETG_TRACE_ERR(("requestProtocolConnections(this = 0x%p, deviceId = %d): could not assign any PCC instance (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         result = CC_ERR_INT_DCC_NOT_ASSIGNED;
         ETG_TRACE_ERR(("requestProtocolConnections(this = 0x%p, deviceId = %d): DCC instance is currently not assigned to a device (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      (void) checkAndReleasePccForSlaveProtocols();

      return result;
   }

   Result DeviceConnectionController::requestProtocolDisconnections(IN const ProtocolList& protocolList, IN const DisconnectedReason disconnectedReason)
   {
      ENTRY

      Locker locker(&_protocolConnectionsLock);

      ETG_TRACE_USR1(("requestProtocolDisconnections(this = 0x%p, deviceId = %d): disconnectedReason = %d",
            (void *) this, _deviceId, ETG_CENUM(DisconnectedReason, disconnectedReason)));

      Result result(CC_ERR_INT_NO_ERROR);

      if (0u != _deviceId)
      {
         ProtocolConnectionController* pccInstance(0);
         ProtocolConnectionControllerList pccInstances;

         for (size_t i = 0u; i < protocolList.size(); i++)
         {
            result = LocalSpm::getDbManager().getProtocolConnectionController(&pccInstance, _deviceId, protocolList[i]._protocolId, protocolList[i]._uuid);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               pccInstances.push_back(pccInstance);
            }
            else
            {
               ETG_TRACE_ERR(("requestProtocolDisconnections(this = 0x%p, deviceId = %d): no PCC instance for protocol ID = %d and uuid = \"%50s\" is assigned to device (error = %d)",
                     (void *) this, _deviceId, protocolList[i]._protocolId, protocolList[i]._uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
            }
         }

         if (true != pccInstances.empty())
         {
            result = initiateProtocolDisconnections(pccInstances, disconnectedReason);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               ETG_TRACE_USR1(("requestProtocolDisconnections(this = 0x%p, deviceId = %d): all disconnection requests could be successfully processed (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));

               //Usecase: HFP is connected and MAP/PBAP connection is ongoing and user is not accepting the MSG/PBAP popup in Phone side.
               // In between user disconnects the HFP profile.
               //Disconnecting the slave protocols(like PBAP, MAP, etc.,) if user is requesting to disconnect Master(HFP, AVP, etc.,) protocols.
               ProtocolList hmiUserSelectableProtocols;
               result = LocalSpm::getDataProvider().getHmiUserSelectableProtocols(hmiUserSelectableProtocols);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  for (size_t i = 0u; i < pccInstances.size(); i++)
                  {
                     Protocol protocolToBeDisconnected(pccInstances[i]->getProtocolId(), pccInstances[i]->getUuid());

                     if (hmiUserSelectableProtocols.end() != std::find(hmiUserSelectableProtocols.begin(),
                           hmiUserSelectableProtocols.end(), protocolToBeDisconnected))
                     {
                        (void) checkAndReleasePccForSlaveProtocols(true, pccInstances[i]->getProtocolId());
                     }
                  }
               }
            }
            else if (CC_ERR_INT_DCC_ONLY_PARTIALLY_HANDLED == result)
            {
               ETG_TRACE_USR1(("requestProtocolDisconnections(this = 0x%p, deviceId = %d): some disconnection requests could not be successfully processed (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
               result = CC_ERR_INT_NO_ERROR;
            }
            else
            {
               ETG_TRACE_ERR(("requestProtocolDisconnections(this = 0x%p, deviceId = %d): no disconnection request could be successfully processed (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            result = CC_ERR_INT_PCC_NONE_ASSIGNED;
            ETG_TRACE_ERR(("requestProtocolDisconnections(this = 0x%p, deviceId = %d): no PCC instances are assigned for given protocols (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         result = CC_ERR_INT_DCC_NOT_ASSIGNED;
         ETG_TRACE_ERR(("requestProtocolDisconnections(this = 0x%p, deviceId = %d): DCC instance is currently not assigned to a device (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result DeviceConnectionController::requestDeviceDisconnection(IN const DisconnectedReason disconnectedReason, IN const bool deviceToBeDeleted)
   {
      ENTRY

      ETG_TRACE_USR1(("requestDeviceDisconnection(this = 0x%p, deviceId = %d): disconnectedReason = %d, deviceToBeDeleted = %d",
            (void *) this, _deviceId, ETG_CENUM(DisconnectedReason, disconnectedReason), deviceToBeDeleted));

      Result result(CC_ERR_INT_NO_ERROR);

      if (BM_DISCONNECTED_REASON_UNKNOWN != disconnectedReason)
      {
         ETG_TRACE_USR1(("requestDeviceDisconnection(this = 0x%p, deviceId = %d): updating intermediate device disconnected reason with given disconnected reason = %d",
               (void *) this, _deviceId, ETG_CENUM(DisconnectedReason, disconnectedReason)));

         _disconnectedReason = disconnectedReason;
         _allowDisconnectedReasonUpdate = false;
      }

      if (false == _deviceToBeDeleted)
      {
         _deviceToBeDeleted = deviceToBeDeleted;
      }

      ETG_TRACE_USR4(("requestDeviceDisconnection(this = 0x%p, deviceId = %d): sending event DISCONNECT to myself", (void *) this, _deviceId));
      result = this->SendEventByName("DISCONNECT", 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("requestDeviceDisconnection(this = 0x%p, deviceId = %d): could not send event DISCONNECT (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      if(0u != _deviceId)
      {
         // Use case:  HFP & AVP is connected and device is moved out of range. So BMCore retry to connect device(last expected connection status(HFP & AVP)).
         // Stack response is received for HFP as DISCONNECTED. During this time, BT off request is received from HMI. So BMCore disconnect all the connected devices.
         // In this case, BMCore should not reset the last expected connection status otherwise last expected connection status of HFP will be reset here.
         if(true == _deviceLost)
         {
            (void) LocalSpm::getDbManager().resetLastExpectedProtocolConnectionStatus(_deviceId);
         }

         ProtocolConnectionControllerList pccInstances;

         result = LocalSpm::getDbManager().getProtocolConnectionControllers(pccInstances, _deviceId);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            for (size_t idx = 0u; idx < pccInstances.size(); idx++)
            {
               ETG_TRACE_USR4(("requestDeviceDisconnection(this = 0x%p, deviceId = %d): sending event CANCEL_WAITING_CONNECT_LOCAL to PCC SM instance (instance = 0x%p, protocol ID = %d, uuid = \"%50s\")",
                     (void *) this, _deviceId, (void *) pccInstances[idx], ETG_CENUM(ProtocolId, pccInstances[idx]->getProtocolId()), pccInstances[idx]->getUuid().c_str()));
               result = pccInstances[idx]->SendEventByName("CANCEL_WAITING_CONNECT_LOCAL", 0);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("requestDeviceDisconnection(this = 0x%p, deviceId = %d): could not send event CANCEL_WAITING_CONNECT_LOCAL (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
               }
   /*
               ETG_TRACE_USR4(("requestDeviceDisconnection(this = 0x%p, deviceId = %d): sending event DISCONNECT to PCC instance 0x%p",
                     (void *) this, _deviceId, (void *) pccInstances[idx] ));
               result = pccInstances[idx]->SendEventByName("DISCONNECT", 0);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("requestDeviceDisconnection(this = 0x%p, deviceId = %d): could not send event DISCONNECT (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
               }
   */
            }
         }
         else
         {
            ETG_TRACE_ERR(("requestDeviceDisconnection(this = 0x%p, deviceId = %d): could not get PCC instances from DB (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return result;
   }

   Result DeviceConnectionController::releaseProtocolConnectionController(IN const ProtocolId protocolId, IN const Uuid uuid, IN const DisconnectedReason disconnectedReason)
   {
      ENTRY

      Locker locker(&_protocolConnectionsLock);

      ETG_TRACE_USR1(("releaseProtocolConnectionController(this = 0x%p, deviceId = %d): protocolId = %d, uuid = \"%50s\", disconnectedReason = %d", (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid.c_str(), ETG_CENUM(DisconnectedReason, disconnectedReason)));

      Result result(CC_ERR_INT_NO_ERROR);
      ProtocolConnectionController* pccInstance(0);

      bool isSlaveProtocol(false);
      result = LocalSpm::getDataProvider().isSlaveProtocolId(isSlaveProtocol, protocolId);

      if ((true == _allowDisconnectedReasonUpdate) && (false == isSlaveProtocol) && (BM_DISCONNECTED_REASON_MISSING_LINK_KEY != _disconnectedReason))
      {
         ETG_TRACE_USR1(("releaseProtocolConnectionController(this = 0x%p, deviceId = %d): updating intermediate device disconnected reason with protocol disconnected reason = %d",
               (void *) this, _deviceId, ETG_CENUM(DisconnectedReason, disconnectedReason)));

         _disconnectedReason = disconnectedReason;
      }

      if (BM_DISCONNECTED_REASON_ABNORMAL_LOSS == disconnectedReason)
      {
         _deviceLost = true;
      }

      bool sendFinishedEvent = false;
      ConnectionStatus deviceConnectionStatus = BM_CONNECTION_STATUS_UNKNOWN;
      DisconnectedReason localdisconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;

      result = LocalSpm::getDbManager().getDeviceConnectionStatus(deviceConnectionStatus, localdisconnectedReason, _deviceId);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("enterConnected(this = 0x%p, deviceId = %d): could not set device connection status in DB (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }
      else
      {
         if(BM_CONNECTION_STATUS_CONNECTING != deviceConnectionStatus)
         {
            sendFinishedEvent = true;
         }
      }

      // reset the RemoteConnectable as true (allowing remote connection request)
      ProtocolList protocolList;
      Protocol disconnectedProtocol(protocolId, uuid);

      ProtocolTargetConnectionStatusMap::iterator it = _localProtocolTargetConnectionStatusMap.find(disconnectedProtocol);

      // NCG3D-120734 - [UPF61] Only HFP is connected after pairing not A2DP
      if (it != _localProtocolTargetConnectionStatusMap.end())
      {
         TargetConnectionStatus targetConnectionStatus = it->second;

         if((BM_TARGET_CONNECTION_STATUS_CONNECTED == targetConnectionStatus) && (BM_CONNECTION_STATUS_CONNECTED == deviceConnectionStatus))
         {
            // Get all the protocols for the device which is in CONNECTED/CONNECTING
            result = LocalSpm::getDbManager().getDeviceProtocolsConnectedStatus(protocolList, _deviceId);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               // Check the major device protocols which is in CONNECTING/CONNECTED.
               // If major protocol is available then reset the RemoteConnectable as true
               for (size_t idx = 0u; idx < protocolList.size(); idx++)
               {
                  if((BM_PROTOCOL_ID_HFP == protocolList[idx]._protocolId) || (BM_PROTOCOL_ID_AVP == protocolList[idx]._protocolId)
                        || (BM_PROTOCOL_ID_PAN == protocolList[idx]._protocolId))
                  {
                     ETG_TRACE_USR4(("releaseProtocolConnectionController(this = 0x%p, deviceId = %d): reset the RemoteConnectable as true (allowing remote connection request)",
                           (void *) this, _deviceId));

                     result = LocalSpm::getBmCoreMainController().unblockRemoteConnections(_deviceId);

                     if (CC_ERR_INT_NO_ERROR != result)
                     {
                        ETG_TRACE_ERR(("releaseProtocolConnectionController(this = 0x%p, deviceId = %d): could not unblock remote connections for device (error = %d)",
                              (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
                     }

                     break;
                  }//!End of if((BM_PROTOCOL_ID_HFP == protocolList[idx]._protocolId) || ...)
               }//!End of for (size_t idx = 0u; idx < protocolList.size(); idx++)
            }//End of if(CC_ERR_INT_NO_ERROR == result)
         }//End of if(BM_TARGET_CONNECTION_STATUS_CONNECTED == targetConnectionStatus) && ...)
      }//End of if (it != _localProtocolTargetConnectionStatusMap.end())

      result = removeProtocolPendingToConnect(protocolId, uuid, sendFinishedEvent);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("releaseProtocolConnectionController(this = 0x%p, deviceId = %d): removing protocol ID/UUID entry from list failed (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      result = LocalSpm::getDbManager().getProtocolConnectionController(&pccInstance, _deviceId, protocolId, uuid);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         result = LocalSpm::getDbManager().releaseProtocolConnectionController(pccInstance, disconnectedReason);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            (void) pccInstance->release();

            result = LocalSpm::getDbManager().setProtocolConnectionStatus(_deviceId, protocolId, uuid, BM_CONNECTION_STATUS_DISCONNECTED, disconnectedReason, "");

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("releaseProtocolConnectionController(this = 0x%p, deviceId = %d): could not set protocol connection status (error = %d)",
                     (void *) this, _deviceId, ETG_CENUM(cc::CcErrorInternal, result)));
            }

            result = checkAndReleasePccForSlaveProtocols();

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("releaseProtocolConnectionController(this = 0x%p, deviceId = %d): checking PCC instances assignments failed (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("releaseProtocolConnectionController(this = 0x%p, deviceId = %d): releasing PCC instance in DB failed (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("releaseProtocolConnectionController(this = 0x%p, deviceId = %d): could not get PCC instance from DB (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result DeviceConnectionController::initSm()
   {
      ENTRY

      ETG_TRACE_USR1(("initSm(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      initMembers();

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::handleStopSm()
   {
      ENTRY

      ETG_TRACE_USR1(("handleStopSm(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      ETG_TRACE_USR4(("handleStopSm(this = 0x%p, deviceId = %d): sending event CONN_CTRL_STOPPED to BmControllerOnOffSm",
            (void *) this, _deviceId));

      Result result = LocalSpm::getBmController().SendEventByName("CONN_CTRL_STOPPED", (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleStopSm(this = 0x%p, deviceId = %d): could not send event CONN_CTRL_STOPPED to BmControllerOnOffSm (error = %d)",
               (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::handleDone()
   {
      ENTRY

      ETG_TRACE_USR1(("handleDone(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      ETG_TRACE_USR4(("handleDone(this = 0x%p, deviceId = %d): sending event CONN_CTRL_TERMINATED to BmControllerOnOffSm",
            (void *) this, _deviceId));

      Result result = LocalSpm::getBmController().SendEventByName("CONN_CTRL_TERMINATED", (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleDone(this = 0x%p, deviceId = %d): could not send event CONN_CTRL_TERMINATED to BmControllerOnOffSm (error = %d)",
               (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::enterIdle()
   {
      ENTRY

      ETG_TRACE_USR1(("enterIdle(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      if (0u != _deviceId)
      {
         //copy deviceid before it get cleared by release
         DeviceId deviceId(_deviceId);
         DisconnectedReason disconnectedReason(_disconnectedReason);
         bool disconnectionLocallyInitiated(_disconnectionLocallyInitiated);
         bool deviceToBeDeleted(_deviceToBeDeleted);
         bool oobPairedDeviceRequestReceived(_oobPairedDeviceRequestReceived);

         result = this->release();

         if (CC_ERR_INT_NO_ERROR == result)
         {
            if (true == disconnectionLocallyInitiated)
            {
               if ((BM_DISCONNECTED_REASON_AUTOMATIC != disconnectedReason) && (BM_DISCONNECTED_REASON_TEMPORARILY != disconnectedReason))
               {
                  ETG_TRACE_USR1(("enterIdle(this = 0x%p, deviceId = %d): changing intermediate device disconnected reason from %d to %d",
                        (void *) this, _deviceId, ETG_CENUM(DisconnectedReason, disconnectedReason), ETG_CENUM(DisconnectedReason, BM_DISCONNECTED_REASON_NORMAL_LOSS_LOCAL)));

                  disconnectedReason = BM_DISCONNECTED_REASON_NORMAL_LOSS_LOCAL;
               }
            }
            else
            {
               if (BM_DISCONNECTED_REASON_CONNECTION_FAILED_UUID_NOT_SUPPORTED == disconnectedReason)
               {
                  ETG_TRACE_USR1(("enterIdle(this = 0x%p, deviceId = %d): changing intermediate device disconnected reason from %d to %d",
                        (void *) this, _deviceId, ETG_CENUM(DisconnectedReason, disconnectedReason), ETG_CENUM(DisconnectedReason, BM_DISCONNECTED_REASON_CONNECTION_FAILED)));

                  disconnectedReason = BM_DISCONNECTED_REASON_CONNECTION_FAILED;
               }
            }

            result = LocalSpm::getDbManager().setDeviceConnectionStatus(deviceId, BM_CONNECTION_STATUS_DISCONNECTED, disconnectedReason);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("enterIdle(this = 0x%p, deviceId = %d): could not set device connection status in DB (error = %d)",
                     (void *) this, deviceId, ETG_CENUM(CcErrorInternal, result)));
            }

            if (true == deviceToBeDeleted)
            {
               //NCG3D-104022 - send the disconnection request to the remote device for terminating the ACL link if any exists with that device
               LocalSpm::getBmCoreMainController().sendDisconnectDeviceRequest(deviceId);

               result = LocalSpm::getDbManager().deleteDevice(deviceId);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("enterIdle(this = 0x%p, deviceId = %d): could not delete device from DB (error = %d)",
                        (void *) this, deviceId, ETG_CENUM(CcErrorInternal, result)));
               }
               else
               {
                  //internally add the device into paired device list if OOB request is available
                  if(true == oobPairedDeviceRequestReceived)
                  {
                     _oobPairedDeviceRequestReceived = false;

                     LocalSpm::getBmCoreMainController().addOobPairedDeviceInt(_oobPairedDeviceRequest.getBdAddress(),_oobPairedDeviceRequest.getBdName(),
                           _oobPairedDeviceRequest.getLinkKey(), _oobPairedDeviceRequest.getOobType(), _oobPairedDeviceRequest.getOrigin());
                  }
               }
            }

            if (false == _reservedForCpw)
            {
               char eventParams[20];

               result = LocalSpm::getAutoConnectionController().ParameterDEVICE_CONNECTION_FINISHED(eventParams, sizeof(eventParams), deviceId);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  ETG_TRACE_USR4(("enterIdle(this = 0x%p, deviceId = %d): sending event DEVICE_CONNECTION_FINISHED to AutoConnectionController SM",
                        (void *) this, deviceId));
                  result = LocalSpm::getAutoConnectionController().SendEventByName("DEVICE_CONNECTION_FINISHED", eventParams);

                  if(CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_ERR(("enterIdle(this = 0x%p, deviceId = %d): could not send event DEVICE_CONNECTION_FINISHED (error = %d)",
                           (void *) this, deviceId, ETG_CENUM(CcErrorInternal, result)));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("enterIdle(this = 0x%p, deviceId = %d): could not marshal event parameters for event DEVICE_CONNECTION_FINISHED (error = %d)",
                        (void *) this, deviceId, ETG_CENUM(CcErrorInternal, result)));
               }
            }

            if (0u < LocalSpm::getDataProvider().getBmCoreConfiguration()._blockDeviceRemoteConnectionsTimeoutSeconds)
            {
               // allow remote connections for the device currently handled by DCC
               result = LocalSpm::getBmCoreMainController().unblockRemoteConnections(deviceId);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("enterIdle(this = 0x%p, deviceId = %d): could not unblock remote connection requests (error = %d)",
                        (void *) this, deviceId, ETG_CENUM(CcErrorInternal, result)));
               }
            }

            if (0u < LocalSpm::getDataProvider().getBmCoreConfiguration()._serviceSearchTimeoutSeconds)
            {
               // stop the service search for the device currently handled by DCC
               result = LocalSpm::getBmCoreMainController().stopDeviceServiceSearch(deviceId);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("enterIdle(this = 0x%p, deviceId = %d): could not stop device service search (error = %d)",
                        (void *) this, deviceId, ETG_CENUM(CcErrorInternal, result)));
               }
            }
         }
         else
         {
            ETG_TRACE_ERR(("enterIdle(this = 0x%p, deviceId = %d): releasing DCC instance failed (error = %d)",
                  (void *) this, deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   void DeviceConnectionController::setOOBPairedDeviceRequest(IN const BmCoreIfMessage_AddOobPairedDeviceRequest OobPairedDeviceRequest)
   {
      _oobPairedDeviceRequest = OobPairedDeviceRequest;
      _oobPairedDeviceRequestReceived = true;

      ETG_TRACE_USR1(("setOOBPairedDeviceRequest: bdAddress = \"%50s\", bdName = \"%50s\", linkKey = \"%50s\", oobType = %d, Origin = %d",
            _oobPairedDeviceRequest.getBdAddress().c_str(), _oobPairedDeviceRequest.getBdName().c_str(),  _oobPairedDeviceRequest.getLinkKey().c_str(),
            ETG_CENUM(OobType, _oobPairedDeviceRequest.getOobType()), ETG_CENUM(BmCoreIfMsgOrigin, _oobPairedDeviceRequest.getOrigin())));
   }

   void DeviceConnectionController::getOOBPairedDeviceRequest(OUT BmCoreIfMessage_AddOobPairedDeviceRequest& OobPairedDeviceRequest)
   {
      ETG_TRACE_USR1(("getOOBPairedDeviceRequest(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      OobPairedDeviceRequest = _oobPairedDeviceRequest;
   }

   Result DeviceConnectionController::enterInUse()
   {
      ENTRY

      ETG_TRACE_USR1(("enterInUse(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::reserveDcc()
   {
      ETG_TRACE_USR1(("reserveDcc(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      _reservedForCpw = true;

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::unreserveDcc()
   {
      ETG_TRACE_USR1(("unreserveDcc(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      _reservedForCpw = false;

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::connectProtocolLocal(IN const ProtocolId protocolId, IN const UuidChrArr uuid)
   {
      ENTRY

      ETG_TRACE_USR1(("connectProtocolLocal(this = 0x%p, deviceId = %d): protocolId = %d, uuid = \"%50s\"", (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid));

      _disconnectionLocallyInitiated = false;

      (void) connectProtocol(protocolId, uuid, BM_CONNECTION_REQUEST_ORIGIN_LOCAL);

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::connectProtocolRemote(IN const ProtocolId protocolId, IN const UuidChrArr uuid)
   {
      ENTRY

      ETG_TRACE_USR1(("connectProtocolRemote(this = 0x%p, deviceId = %d): protocolId = %d , uuid = \"%50s\"", (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid));

      (void) connectProtocol(protocolId, uuid, BM_CONNECTION_REQUEST_ORIGIN_REMOTE);

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::enterConnecting()
   {
      ENTRY

      ETG_TRACE_USR1(("enterConnecting(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      _allowDisconnectedReasonUpdate = true;
      _allProtocolsDisconnected = false;
      _deviceDisconnected = false;

      ETG_TRACE_USR1(("enterConnecting(this = 0x%p, deviceId = %d): changing intermediate device disconnected reason from %d to %d",
            (void *) this, _deviceId, ETG_CENUM(DisconnectedReason, _disconnectedReason), ETG_CENUM(DisconnectedReason, BM_DISCONNECTED_REASON_CONNECTION_FAILED)));

      _disconnectedReason = BM_DISCONNECTED_REASON_CONNECTION_FAILED;

      result = LocalSpm::getDbManager().setDeviceConnectionStatus(_deviceId, BM_CONNECTION_STATUS_CONNECTING, BM_DISCONNECTED_REASON_NOT_APPLICABLE);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("enterConnecting(this = 0x%p, deviceId = %d): could not set device connection status in DB (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::releaseDcc()
   {
      ETG_TRACE_USR1(("releaseDcc(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(_deviceId);
      bool deviceToBeDeleted(_deviceToBeDeleted);

      _reservedForCpw = false;

      this->release();

      if (true == deviceToBeDeleted)
      {
         //NCG3D-104022 - send the disconnection request to the remote device for terminating the ACL link if any exists with that device
         LocalSpm::getBmCoreMainController().sendDisconnectDeviceRequest(deviceId);

         result = LocalSpm::getDbManager().deleteDevice(deviceId);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("releaseDcc(this = 0x%p, deviceId = %d): could not delete device from DB (error = %d)", (void *) this, deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }

      char eventParams[20];

      result = LocalSpm::getAutoConnectionController().ParameterDEVICE_CONNECTION_FINISHED(eventParams, sizeof(eventParams), deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR4(("releaseDcc(this = 0x%p, deviceId = %d): sending event DEVICE_CONNECTION_FINISHED to AutoConnectionController SM", (void *) this, deviceId));
         result = LocalSpm::getAutoConnectionController().SendEventByName("DEVICE_CONNECTION_FINISHED", eventParams);

         if(CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("releaseDcc(this = 0x%p, deviceId = %d): could not send event DEVICE_CONNECTION_FINISHED (error = %d)",
                  (void *) this, deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("releaseDcc(this = 0x%p, deviceId = %d): could not marshal event parameters for event DEVICE_CONNECTION_FINISHED (error = %d)",
               (void *) this, deviceId, ETG_CENUM(CcErrorInternal, result)));
      }
      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::isDeviceLost()
   {
      ENTRY

      ETG_TRACE_USR1(("isDeviceLost(this = 0x%p, deviceId = %d): device is %10s", (void *) this, _deviceId, (_deviceLost ? "lost" : "not lost")));

      _allProtocolsDisconnected = true;

      if(BM_DISCONNECTED_REASON_MISSING_LINK_KEY == _disconnectedReason)
      {
         ETG_TRACE_USR4(("isDeviceLost(this = 0x%p, deviceId = %d): Pairing information is deleted in Phone side hence stopping the retry of lost device",
               (void *) this, _deviceId));
         return 0;
      }
      else if (true == _deviceLost)
      {
         return 1;
      }

      return 0;
   }

   Result DeviceConnectionController::handleLostDevice()
   {
      ETG_TRACE_USR1(("handleLostDevice(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      _deviceDisconnected = true;

      result = LocalSpm::getDbManager().setDeviceConnectionStatus(_deviceId, BM_CONNECTION_STATUS_DISCONNECTED, BM_DISCONNECTED_REASON_ABNORMAL_LOSS);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleLostDevice(this = 0x%p, deviceId = %d): could not set device connection status in DB (error = %d)",
               (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      (void) this->reconnectDevice();

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::enterLost()
   {
      ENTRY

      ETG_TRACE_USR1(("enterLost(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      Result result(CC_ERR_INT_NO_ERROR);
      PageTimeout pausePagingTimeoutMilliSeconds = 0u;
      PageTimeout pageTimeout = 0u;
      _deviceDisconnected = true;

      result = LocalSpm::getDbManager().setDeviceConnectionStatus(_deviceId, BM_CONNECTION_STATUS_DISCONNECTED, BM_DISCONNECTED_REASON_ABNORMAL_LOSS);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("enterLost(this = 0x%p, deviceId = %d): could not set device connection status in DB (error = %d)",
               (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      result = LocalSpm::getBmCoreMainController().getProfileConnectionPageTimeout(pausePagingTimeoutMilliSeconds, pageTimeout);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         _pageTimeout = pageTimeout;

         // set the pause pageTimeout as 300 ms if the disconnected reason is not OUT_OF_RANGE
         if ((0u == pausePagingTimeoutMilliSeconds) && (BM_DISCONNECTED_REASON_OUT_OF_RANGE != _disconnectedReason))
         {
            pausePagingTimeoutMilliSeconds = LocalSpm::getDataProvider().getBmCoreConfiguration()._lostDeviceConnectionRetryDelayMilliSeconds;
         }

         if(0 < pausePagingTimeoutMilliSeconds)
         {
            _expiredTimerSmEvent = this->GetSMNameFull();
            _expiredTimerSmEvent += "::CONNECTION_RETRY";

            if (true == _delayConnectionRetryTimer.StartSMTimer(_delayConnectionRetryTimerId,
                  static_cast<long>(pausePagingTimeoutMilliSeconds),
                  0L, _expiredTimerSmEvent.c_str(), this))
            {
               char milliSecondsStr[200];
               snprintf(milliSecondsStr, 199, "%llu", static_cast<long long unsigned>(pausePagingTimeoutMilliSeconds));
               ETG_TRACE_USR4(("enterLost(this = 0x%p, deviceId = %d): started delay connection retry timer (timeder().getBmCorer ID = 0x%p, intervalMilliseconds = %50s)",
                     (void *) this, _deviceId, _delayConnectionRetryTimerId, milliSecondsStr));
            }
            else
            {
               ETG_TRACE_ERR(("enterLost(this = 0x%p, deviceId = %d): starting delay disconnection retry timer failed so sending event CONNECTION_RETRY to myself",
                     (void *) this, _deviceId));

               result = this->SendEventByName("CONNECTION_RETRY", 0);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("enterLost(this = 0x%p, deviceId = %d): could not send event CONNECTION_RETRY (error = %d)",
                        (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
               }
            }
         }
         else
         {
            ETG_TRACE_USR4(("enterLost(this = 0x%p, deviceId = %d): sending event CONNECTION_RETRY to myself",
                  (void *) this, _deviceId));
            result = this->SendEventByName("CONNECTION_RETRY", 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("enterLost(this = 0x%p, deviceId = %d): could not send event CONNECTION_RETRY (error = %d)",
                     (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("enterLost:getProfileConnectionPageTimeout(this = 0x%p, deviceId = %d): Could not get ProfileConnectionPageTimeout from MainController (error = %d)",
               (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::exitLost()
   {
      ENTRY

      ETG_TRACE_USR1(("exitLost(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      _delayConnectionRetryTimer.CancelTimer(_delayConnectionRetryTimerId);

      _expiredTimerSmEvent = this->GetSMNameFull();

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::reconnectDevice()
   {
      ENTRY

      ETG_TRACE_USR1(("reconnectDevice(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      BmResult bmResult = LocalSpm::getBmCoreMainController().connectDeviceInt(_deviceId, false, _pageTimeout);

      if (BM_RESULT_OK != bmResult)
      {
         ETG_TRACE_ERR(("reconnectDevice(this = 0x%p, deviceId = %d): device re-connection failed (bmResult = %d)", (void *) this, _deviceId, bmResult));

         ETG_TRACE_USR4(("reconnectDevice(this = 0x%p, deviceId = %d): sending event ALL_PROTOCOLS_DISCONNECTED to myself", (void *) this, _deviceId));
         Result result = this->SendEventByName("ALL_PROTOCOLS_DISCONNECTED", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("reconnectDevice(this = 0x%p, deviceId = %d): could not send event ALL_PROTOCOLS_DISCONNECTED (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::handleProtocolConnected(IN const ProtocolId protocolId, IN const UuidChrArr uuid)
   {
      ENTRY

      ETG_TRACE_USR1(("handleProtocolConnected(this = 0x%p, deviceId = %d): protocolId = %d, uuid = \"%50s\"", (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid));

      Result result(CC_ERR_INT_NO_ERROR);

      // This changes due to StateMachine informed the Current state of DCC is CONNECTED before going to the Connected state itself.
      //Autoconnectionfinished event should be send only device status is changed to CONNECTED.
      bool sendFinishedEvent = false;
      ConnectionStatus connectionStatus = BM_CONNECTION_STATUS_UNKNOWN;
      DisconnectedReason localdisconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;

      result = LocalSpm::getDbManager().getDeviceConnectionStatus(connectionStatus, localdisconnectedReason, _deviceId);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("enterConnected(this = 0x%p, deviceId = %d): could not set device connection status in DB (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }
      else
      {
         if(BM_CONNECTION_STATUS_CONNECTING != connectionStatus)
         {
            sendFinishedEvent = true;
         }
      }

      result = removeProtocolPendingToConnect(protocolId, uuid, sendFinishedEvent);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleProtocolConnected(this = 0x%p, deviceId = %d): removing protocol ID/UUID entry from list of protocols waiting for getting connected failed (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::enterConnected()
   {
      ENTRY

      ETG_TRACE_USR1(("enterConnected(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      ETG_TRACE_USR1(("enterConnected(this = 0x%p, deviceId = %d): changing intermediate device disconnected reason from %d to %d",
            (void *) this, _deviceId, ETG_CENUM(DisconnectedReason, _disconnectedReason), ETG_CENUM(DisconnectedReason, BM_DISCONNECTED_REASON_UNKNOWN)));

      _disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;
      _deviceLost = false;

      result = LocalSpm::getDbManager().setDeviceConnectionStatus(_deviceId, BM_CONNECTION_STATUS_CONNECTED, BM_DISCONNECTED_REASON_NOT_APPLICABLE);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("enterConnected(this = 0x%p, deviceId = %d): could not set device connection status in DB (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      // If connection request is Remote then check the usage preference
      if((BM_CONNECTION_REQUEST_ORIGIN_REMOTE == _origin) && (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported))
      {
         UsagePreference usagePreference = BM_UP_UNDECIDED;

         result = LocalSpm::getDbManager().getDeviceUsagePreference(usagePreference, _deviceId);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            // if the usage preference is TEMP_CLASSIC_BT or CPW and also  Wifi AP mode is already set
            // then only internally need to start the CPW
            if((BM_UP_TEMP_CLASSIC_BT == usagePreference) || (BM_UP_CPW == usagePreference))
            {
               BtLimitationMode btLimitationMode;
               //!Fix for CID 124460: Inferred misuse of enum (MIXED_ENUMS)mixed_enums:
               //!Solution - Not using the return value as it is not required in this case
               //result = LocalSpm::getBmCoreMainController().getBtLimitationMode(btLimitationMode);
               (void)LocalSpm::getBmCoreMainController().getBtLimitationMode(btLimitationMode);

               if(true == btLimitationMode._btLimitationModeInfoList.empty())
               {
                  BdAddress bdAddress;
                  result = LocalSpm::getDbManager().getBdAddress(bdAddress, _deviceId);

                  // If no Limitation mode is not available for the device then send the request
                  if(CC_ERR_INT_NO_ERROR == result)
                  {
                     BdName bdName = "";
                     LimitationMode limitationMode;
                     LimitationAction limitationAction = BM_LIMITATION_ACTION_PREPARE;

                     limitationMode._limitationCommunicationIf = BM_LIMITATION_COMMUNICATION_IF_WIFI;
                     limitationMode._limitationFeature = BM_LIMITATION_FEATURE_CAR_PLAY;

                     ETG_TRACE_USR1(("enterConnected(): bdAddress = %50s, limitationMode.feature = %d, limitationMode.commIf = %d, limitationAction = %d",
                           bdAddress.c_str(), ETG_CENUM(LimitationFeature, limitationMode._limitationFeature),
                           ETG_CENUM(LimitationCommunicationIf, limitationMode._limitationCommunicationIf),
                           ETG_CENUM(LimitationAction, limitationAction)));

                     // setBtLimitationMode for CPW
                     (void)LocalSpm::getBmCoreMainController().setBtLimitationModeInt(bdAddress, bdName, limitationMode, limitationAction);
                  }
                  else
                  {
                     ETG_TRACE_USR1(("enterConnected(): could not get bdAddress from DB (error = %d)", ETG_CENUM(CcErrorInternal, result)));
                  }
               }
               else
               {
                  ETG_TRACE_USR1(("enterConnected(): btLimitationMode is not empty"));
               }
            }
         }
      }

      //set to Default
      setConnectionRequestOrigin(BM_CONNECTION_REQUEST_ORIGIN_UNKNOWN);

      // Set delayprotocolConnection as false once device is getting connected
      _delayLocalProtocolConnections = false;

      // Send the AutoconnnectedFinished event if only one protocol is requested to be connected for this device
      checkAndSendAutoConnectedFinished();

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::disconnectProtocol(IN const ProtocolId protocolId, IN const UuidChrArr uuid)
   {
      ENTRY

      ETG_TRACE_USR1(("disconnectProtocol(this = 0x%p, deviceId = %d): protocolId = %d, uuid = \"%50s\"", (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid));

      Result result(CC_ERR_INT_NO_ERROR);
      ProtocolConnectionController* pccInstance(0);

      result = LocalSpm::getDbManager().getProtocolConnectionController(&pccInstance, _deviceId, protocolId, uuid);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR4(("disconnectProtocol(this = 0x%p, deviceId = %d): sending event DISCONNECT to PCC SM instance (instance = 0x%p, protocol ID = %d, uuid = \"%50s\")",
               (void *) this, _deviceId, (void *) pccInstance, ETG_CENUM(ProtocolId, pccInstance->getProtocolId()), pccInstance->getUuid().c_str()));

         char eventParams[20];

         result = pccInstance->ParameterDISCONNECT(eventParams, sizeof(eventParams), _disconnectedReason);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            result = pccInstance->SendEventByName("DISCONNECT", eventParams);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("disconnectProtocol(this = 0x%p, deviceId = %d): could not send event DISCONNECT (error = %d)",
                     (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
            }
            else
            {
               Protocol protocolToBeDisconnected(protocolId, uuid);

               ProtocolList hmiUserSelectableProtocols;
               result = LocalSpm::getDataProvider().getHmiUserSelectableProtocols(hmiUserSelectableProtocols);

               if (hmiUserSelectableProtocols.end() != std::find(hmiUserSelectableProtocols.begin(),
                     hmiUserSelectableProtocols.end(), protocolToBeDisconnected))
               {
                  this->setLocalProtocolTargetConnectionStatusInMap(protocolToBeDisconnected, BM_TARGET_CONNECTION_STATUS_DISCONNECTED);

                  if (0u < LocalSpm::getDataProvider().getBmCoreConfiguration()._blockDeviceRemoteConnectionsTimeoutSeconds)
                  {
                     // block remote connections for the device currently handled by DCC
                     result = LocalSpm::getBmCoreMainController().blockRemoteConnections(_deviceId);

                     if (CC_ERR_INT_NO_ERROR != result)
                     {
                        ETG_TRACE_ERR(("disconnectProtocol(this = 0x%p, deviceId = %d): could not activate blocking of remote connection requests (error = %d)",
                              (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
                     }
                  }
               }
            }
         }
         else
         {
            ETG_TRACE_ERR(("disconnectProtocol(this = 0x%p, deviceId = %d): could not marshal event parameters for event DISCONNECTED (error = %d)",
                  (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("disconnectProtocol(this = 0x%p, deviceId = %d): could not get PCC instance from DB (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::handleDisconnect(void)
   {
      ENTRY

      ETG_TRACE_USR1(("handleDisconnect(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      _disconnectionLocallyInitiated = true;

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::enterDisconnecting()
   {
      ENTRY

      ETG_TRACE_USR1(("enterDisconnecting(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      result = LocalSpm::getDbManager().setDeviceConnectionStatus(_deviceId, BM_CONNECTION_STATUS_DISCONNECTING, BM_DISCONNECTED_REASON_NOT_APPLICABLE);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("enterDisconnecting(this = 0x%p, deviceId = %d): could not set device connection status in DB (error = %d)",
               (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      ProtocolConnectionControllerList pccInstances;

      result = LocalSpm::getDbManager().getProtocolConnectionControllers(pccInstances, _deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         if (false == pccInstances.empty())
         {
            ProtocolList hmiUserSelectableProtocols;
            (void) LocalSpm::getDataProvider().getHmiUserSelectableProtocols(hmiUserSelectableProtocols);

            for (size_t idx = 0u; idx < pccInstances.size(); idx++)
            {
               ETG_TRACE_USR4(("enterDisconnecting(this = 0x%p, deviceId = %d): sending event DISCONNECT to PCC SM instance (instance = 0x%p, protocol ID = %d, uuid = \"%50s\")",
                     (void *) this, _deviceId, (void *) pccInstances[idx], ETG_CENUM(ProtocolId, pccInstances[idx]->getProtocolId()), pccInstances[idx]->getUuid().c_str()));

               char eventParams[20];

               result = pccInstances[idx]->ParameterDISCONNECT(eventParams, sizeof(eventParams), _disconnectedReason);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  result = pccInstances[idx]->SendEventByName("DISCONNECT", eventParams);

                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_ERR(("enterDisconnecting(this = 0x%p, deviceId = %d): could not send event DISCONNECT (error = %d)",
                           (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
                  }
                  else
                  {
                     if (true == _disconnectionLocallyInitiated)
                     {
                        Protocol protocolToBeDisconnected(pccInstances[idx]->getProtocolId(),
                              pccInstances[idx]->getUuid());

                        if (hmiUserSelectableProtocols.end() != std::find(hmiUserSelectableProtocols.begin(),
                              hmiUserSelectableProtocols.end(), protocolToBeDisconnected))
                        {
                           this->setLocalProtocolTargetConnectionStatusInMap(protocolToBeDisconnected, BM_TARGET_CONNECTION_STATUS_DISCONNECTED);

                           if (0u < LocalSpm::getDataProvider().getBmCoreConfiguration()._blockDeviceRemoteConnectionsTimeoutSeconds)
                           {
                              // block remote connections for the device currently handled by DCC
                              result = LocalSpm::getBmCoreMainController().blockRemoteConnections(_deviceId);

                              if (CC_ERR_INT_NO_ERROR != result)
                              {
                                 ETG_TRACE_ERR(("enterDisconnecting(this = 0x%p, deviceId = %d): could not activate blocking of remote connection requests (error = %d)",
                                       (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
                              }
                           }
                        }
                     }
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("enterDisconnecting(this = 0x%p, deviceId = %d): could not marshal event parameters for event DISCONNECTED (error = %d)",
                        (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
               }
            }
         }
         else
         {
            ETG_TRACE_USR4(("enterDisconnecting(this = 0x%p, deviceId = %d): sending event ALL_PROTOCOLS_DISCONNECTED to myself",
                  (void *) this, _deviceId));

            result = this->SendEventByName("ALL_PROTOCOLS_DISCONNECTED", 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("enterDisconnecting(this = 0x%p, deviceId = %d): could not send event ALL_PROTOCOLS_DISCONNECTED (error = %d)",
                     (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("enterDisconnecting(this = 0x%p, deviceId = %d): could not get PCC instances from DB (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      if (0u < LocalSpm::getDataProvider().getBmCoreConfiguration()._serviceSearchTimeoutSeconds)
      {
         // stop the service search for the device currently handled by DCC
         result = LocalSpm::getBmCoreMainController().stopDeviceServiceSearch(_deviceId);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("enterDisconnecting(this = 0x%p, deviceId = %d): could not stop the device service Search (error = %d)",
                  (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }

//      (void) this->disconnectDevice();

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::handleAllProtocolsDisconnected()
   {
      ENTRY

      ETG_TRACE_USR1(("handleAllProtocolsDisconnected(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      _allProtocolsDisconnected = true;

//      (void) this->isDisconnectionFinished();
      (void) this->disconnectDevice();

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::handleDeviceDisconnected()
   {
      ENTRY

      ETG_TRACE_USR1(("handleDeviceDisconnected(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      _deviceDisconnected = true;

      (void) this->isDisconnectionFinished();

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::rejectProtocolConnectionRemote(IN const ProtocolId protocolId, IN const UuidChrArr uuid)
   {
      ENTRY

      ETG_TRACE_USR1(("rejectProtocolConnectionRemote(this = 0x%p, deviceId = %d): protocolId = %d, uuid = \"%50s\"", (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid));

      Result result(CC_ERR_INT_NO_ERROR);

      BTSProtocolId btsProtocolId(BTS_PROTO_LAST);
      result = getBtsProtocolIdFromBmProtocolId(btsProtocolId, protocolId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR1(("rejectProtocolConnectionRemote(this = 0x%p, deviceId = %d): BM core's protocol ID = %d corresponds to BT Stack IF's protocol ID = %d", (void *) this, _deviceId, protocolId, btsProtocolId));

         BdAddress bdAddress("");
         result = LocalSpm::getDbManager().getBdAddress(bdAddress, _deviceId);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR1(("rejectProtocolConnectionRemote(this = 0x%p, deviceId = %d): rejecting protocol connection request for device with BD address = \"%50s\" and protocol with ID = %d", (void *) this, _deviceId, bdAddress.c_str(), protocolId));

            LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().rejectRemoteProtocolConnect(bdAddress, btsProtocolId, uuid);
         }
         else
         {
            ETG_TRACE_ERR(("enterDisconnecting(this = 0x%p, deviceId = %d): could not get BD address from DB (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("rejectProtocolConnectionRemote(this = 0x%p, deviceId = %d): could not map BM core's protocol ID to a corresponding BT Stack IF protocol ID (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::enterDisconnectionFailed()
   {
      ENTRY

      ETG_TRACE_USR1(("enterDisconnectionFailed(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      _expiredTimerSmEvent += "::DISCONNECTION_RETRY";

      if (true == _delayDisconnectionRetryTimer.StartSMTimer(_delayDisconnectionRetryTimerId, 200L, 0L,
            _expiredTimerSmEvent.c_str(), this))
      {
         ETG_TRACE_USR1(("enterDisconnectionFailed(this = 0x%p, deviceId = %d): started delay disconnection retry timer (timer ID = 0x%p, intervalMilliseconds = 200)", (void *) this, _deviceId, _delayDisconnectionRetryTimerId));
      }
      else
      {
         ETG_TRACE_ERR(("enterDisconnectionFailed(this = 0x%p, deviceId = %d): starting delay disconnection retry timer failed", (void *) this, _deviceId));

         ETG_TRACE_USR4(("enterDisconnectionFailed(this = 0x%p, deviceId = %d): sending event DISCONNECTION_RETRY to myself", (void *) this, _deviceId));
         result = this->SendEventByName("DISCONNECTION_RETRY", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("enterDisconnectionFailed(this = 0x%p, deviceId = %d): could not send event DISCONNECTION_RETRY (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::exitDisconnectionFailed()
   {
      ENTRY

      ETG_TRACE_USR1(("exitDisconnectionFailed(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      _delayDisconnectionRetryTimer.CancelTimer(_delayDisconnectionRetryTimerId);

      _expiredTimerSmEvent = this->GetSMNameFull();

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::disconnectDevice()
   {
      ENTRY

      ETG_TRACE_USR1(("disconnectDevice(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      LocalSpm::getBmCoreMainController().sendDisconnectDeviceRequest(_deviceId);

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::disconnectInIdle()
   {
      ENTRY

      ETG_TRACE_USR1(("disconnectInIdle(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      // Internally trigger deactivate if Dcc is used for CPW
      if(true == _reservedForCpw)
      {
         LocalSpm::getBmCoreMainController().deactivateBtLimitationModeCPW(_deviceId);
      }

      return CC_ERR_INT_NO_ERROR;
   }

   void DeviceConnectionController::handleSmTimerCallback(const char* message)
   {
      if(0 != message)
      {
         LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(getNewBmCoreIfMessage_BmCoreIfMessage_SmTimeout(message));
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   void DeviceConnectionController::traceProtocolTargetConnectionStatusMap()
   {
      if (false == _localProtocolTargetConnectionStatusMap.empty())
      {
         ETG_TRACE_USR1(("traceProtocolTargetConnectionStatusMap(this = 0x%p, deviceId = %d): list of pending locally initiated protocol connections/disconnections:",
               (void *) this, _deviceId));

      for (ProtocolTargetConnectionStatusMap::iterator it = _localProtocolTargetConnectionStatusMap.begin(); it != _localProtocolTargetConnectionStatusMap.end(); ++it)
      {
         ETG_TRACE_USR1(("traceProtocolTargetConnectionStatusMap(this = 0x%p, deviceId = %d): protocol = (%d, \"%50s\"), target connection status = %d",
               (void *) this, _deviceId, ETG_CENUM(ProtocolId, it->first._protocolId), it->first._uuid.c_str(),
               ETG_CENUM(TargetConnectionStatus, it->second)));
      }
      }
      else
      {
         ETG_TRACE_USR1(("traceProtocolTargetConnectionStatusMap(this = 0x%p, deviceId = %d): list of pending locally initiated protocol connections/disconnections is empty",
               (void *) this, _deviceId));
      }
   }

   Result DeviceConnectionController::initMembers()
   {
      ENTRY

      ETG_TRACE_USR1(("initMembers(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      /* Initialize member variables */

      _deviceId = 0u;
      _deviceToBeDeleted = false;
      _reservedForCpw = false;
      _delayLocalProtocolConnections = false;
      _disconnectionLocallyInitiated = false;
      _deviceLost = false;
      _allowDisconnectedReasonUpdate = true;
      _allProtocolsDisconnected = true;
      _deviceDisconnected = true;
      _disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;
      _origin = BM_CONNECTION_REQUEST_ORIGIN_UNKNOWN;
      _expiredTimerSmEvent = this->GetSMNameFull();
      _protocolsPendingToConnect.clear();
      _localProtocolTargetConnectionStatusMap.clear();
      _oobPairedDeviceRequestReceived = false;
      _pageTimeout = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultConnectionPageTimeoutMilliSeconds;

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::assignProtocolConnectionController(OUT ProtocolConnectionController** pccInstance, IN const ProtocolId protocolId,
         IN const Uuid& uuid, IN const PageTimeout pageTimeout)
   {
      ENTRY

      ETG_TRACE_USR1(("assignProtocolConnectionController(this = 0x%p, deviceId = %d): protocolId = %d, uuid = \"%50s\" pageTimeout = %d", (void *) this,
            _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid.c_str(), pageTimeout));

      Result result(CC_ERR_INT_NO_ERROR);

      *pccInstance = 0;

      result = LocalSpm::getDbManager().assignProtocolConnectionController(pccInstance, _deviceId, protocolId, uuid);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         result = (*pccInstance)->assign(_deviceId, uuid, pageTimeout);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("assignProtocolConnectionController(this = 0x%p, deviceId = %d, protocolId = %d, uuid = \"%50s\"): assigning device ID and UUID to PCC instance failed", (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid.c_str()));

            Result releaseResult = LocalSpm::getDbManager().releaseProtocolConnectionController(*pccInstance, BM_DISCONNECTED_REASON_UNKNOWN);

            if (CC_ERR_INT_NO_ERROR != releaseResult)
            {
               ETG_TRACE_ERR(("assignProtocolConnectionController(this = 0x%p, deviceId = %d, protocolId = %d, uuid = \"%50s\"): releasing PCC instance in DB failed (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid.c_str(), releaseResult));
            }

            *pccInstance = 0;
         }
      }
      else if (CC_ERR_INT_DB_END_OF_LIST == result)
      {
         ETG_TRACE_USR1(("assignProtocolConnectionController(this = 0x%p, deviceId = %d, protocolId = %d, uuid = \"%50s\"): no PCC instance available (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
         result = CC_ERR_INT_PCC_NO_RESOURCES;
      }
      else
      {
         ETG_TRACE_ERR(("assignProtocolConnectionController(this = 0x%p, deviceId = %d, protocolId = %d, uuid = \"%50s\"): assigning PCC instance in DB failed (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result DeviceConnectionController::initiateProtocolConnections(IN const ProtocolConnectionControllerList& pccInstances, IN const ConnectionRequestOrigin connectionRequestOrigin)
   {
      ENTRY

      ETG_TRACE_USR1(("initiateProtocolConnections(this = 0x%p, deviceId = %d): connectionRequestOrigin = %d", (void *) this, _deviceId, connectionRequestOrigin));

      Result result(CC_ERR_INT_NO_ERROR);
      Result finalResult(CC_ERR_INT_NO_ERROR);
      std::string eventName("");
      char eventParams[60];
      uint8_t successfullyHandledCnt(0u);

      if (BM_CONNECTION_REQUEST_ORIGIN_LOCAL == connectionRequestOrigin)
      {
         eventName = "CONNECT_LOCAL";
      }
      else if (BM_CONNECTION_REQUEST_ORIGIN_REMOTE == connectionRequestOrigin)
      {
         eventName = "CONNECT_REMOTE";
      }
      else
      {
         ETG_TRACE_ERR(("initiateProtocolConnections(this = 0x%p, deviceId = %d): connection request originator is unknown", (void *) this, _deviceId));
         return CC_ERR_INT_INVALID_PARAMETER;
      }

      for (size_t i = 0; i < pccInstances.size(); i++)
      {
         if (0 != pccInstances[i])
         {
            if (BM_CONNECTION_REQUEST_ORIGIN_LOCAL == connectionRequestOrigin)
            {
               result = this->ParameterCONNECT_LOCAL(eventParams, sizeof(eventParams), pccInstances[i]->getProtocolId(), pccInstances[i]->getUuid().c_str());
            }
            else
            {
               result = this->ParameterCONNECT_REMOTE(eventParams, sizeof(eventParams), pccInstances[i]->getProtocolId(), pccInstances[i]->getUuid().c_str());
            }

            if (CC_ERR_INT_NO_ERROR == result)
            {
               //add protocolId to protocols pending to be connected
               ProtocolList::iterator it;
               Protocol protocolToConnect(pccInstances[i]->getProtocolId(), pccInstances[i]->getUuid());

               it = std::find(_protocolsPendingToConnect.begin(), _protocolsPendingToConnect.end(), protocolToConnect);

               if (it == _protocolsPendingToConnect.end())
               {
                  _protocolsPendingToConnect.push_back(protocolToConnect);
               }

               VARTRACE(_protocolsPendingToConnect);

               ETG_TRACE_USR4(("initiateProtocolConnections(this = 0x%p, deviceId = %d): sending event %50s to myself",
                     (void *) this, _deviceId, eventName.c_str()));
               result = this->SendEventByName(eventName.c_str(), eventParams);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  successfullyHandledCnt++;
               }
               else
               {
                  ETG_TRACE_ERR(("initiateProtocolConnections(this = 0x%p, deviceId = %d): could not send event %50s (error = %d)",
                        (void *) this, _deviceId, eventName.c_str(), ETG_CENUM(CcErrorInternal, result)));
                  finalResult = result;
               }
            }
            else
            {
               ETG_TRACE_ERR(("initiateProtocolConnections(this = 0x%p, deviceId = %d): could not marshal event parameters for event %50s (error = %d)",
                     (void *) this, _deviceId, eventName.c_str(), ETG_CENUM(CcErrorInternal, result)));
               finalResult = result;
            }
         }
         else
         {
            result = CC_ERR_INT_NULL_POINTER;
            ETG_TRACE_ERR(("initiateProtocolConnections(this = 0x%p, deviceId = %d): pointer to PCC instance is 0 (error = %d)",
                  (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
            finalResult = result;
         }
      }

      ETG_TRACE_USR1(("initiateProtocolConnections(this = 0x%p, deviceId = %d): %d of %d connection request have been handled successfully",
            (void *) this, _deviceId, successfullyHandledCnt, pccInstances.size()));

      if ((0u < successfullyHandledCnt) && (pccInstances.size() > successfullyHandledCnt))
      {
         finalResult = CC_ERR_INT_DCC_ONLY_PARTIALLY_HANDLED;
      }

      return finalResult;
   }

   Result DeviceConnectionController::initiateProtocolDisconnections(IN const ProtocolConnectionControllerList& pccInstances,
         IN const DisconnectedReason disconnectedReason)
   {
      ENTRY

      ETG_TRACE_USR1(("initiateProtocolDisconnections(this = 0x%p, deviceId = %d, disconnectedReason = %d)",
            (void *) this, _deviceId, ETG_CENUM(DisconnectedReason, disconnectedReason)));

      Result result(CC_ERR_INT_NO_ERROR);
      Result finalResult(CC_ERR_INT_NO_ERROR);
      char eventParams[60];
      uint8_t successfullyHandledCnt(0u);

      for (size_t i = 0u; i < pccInstances.size(); i++)
      {
         if (0 != pccInstances[i])
         {
            if ((BM_DISCONNECTED_REASON_NORMAL_LOSS == disconnectedReason) || (BM_DISCONNECTED_REASON_NORMAL_LOSS_LOCAL == disconnectedReason))
            {
               result = LocalSpm::getDbManager().setLastExpectedProtocolConnectionStatus(_deviceId, pccInstances[i]->getProtocolId(), pccInstances[i]->getUuid(), BM_LAST_EXPECTED_CONNECTION_STATUS_DISCONNECTED);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("initiateProtocolDisconnections(this = 0x%p, deviceId = %d): could not set last expected protocol connection status in DB (instance = 0x%p, protocol ID = %d, uuid = \"%50s\") (error = %d)", (void *) this, _deviceId, pccInstances[i], pccInstances[i]->getProtocolId(), pccInstances[i]->getUuid().c_str(), ETG_CENUM(CcErrorInternal, result)));
               }

               bool isSlaveProtocol(false);
               (void)LocalSpm::getDataProvider().isSlaveProtocolId(isSlaveProtocol, pccInstances[i]->getProtocolId());

               //!Fix for NCG3D-108093
               if((false == isSlaveProtocol) && (_disconnectedReason != disconnectedReason))
               {
                  ETG_TRACE_USR4(("initiateProtocolDisconnections(this = 0x%p, deviceId = %d): changing device disconnected reason from %d to %d",
                           (void *) this, _deviceId, ETG_CENUM(DisconnectedReason, _disconnectedReason), ETG_CENUM(DisconnectedReason, disconnectedReason)));

                  //! Update the DisconnectReason variable
                  _disconnectedReason = disconnectedReason;
               }
            }

            result = this->ParameterDISCONNECT_PROTOCOL(eventParams, sizeof(eventParams), pccInstances[i]->getProtocolId(), pccInstances[i]->getUuid().c_str());

            if (CC_ERR_INT_NO_ERROR == result)
            {
               ETG_TRACE_USR4(("initiateProtocolDisconnections(this = 0x%p, deviceId = %d): sending event DISCONNECT_PROTOCOL to myself", (void *) this, _deviceId));
               result = this->SendEventByName("DISCONNECT_PROTOCOL", eventParams);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  successfullyHandledCnt++;
               }
               else
               {
                  ETG_TRACE_ERR(("initiateProtocolDisconnections(this = 0x%p, deviceId = %d): could not send event DISCONNECT_PROTOCOL to PCC instance (instance = 0x%p, protocol ID = %d, uuid = \"%50s\") (error = %d)", (void *) this, _deviceId, pccInstances[i], pccInstances[i]->getProtocolId(), pccInstances[i]->getUuid().c_str(), ETG_CENUM(CcErrorInternal, result)));
                  finalResult = result;
               }
            }
            else
            {
               ETG_TRACE_ERR(("initiateProtocolDisconnections(this = 0x%p, deviceId = %d): could not marshal event parameters for event DISCONNECT_PROTOCOL for PCC instance (instance = 0x%p, protocol ID = %d, uuid = \"%50s\") (error = %d)", (void *) this, _deviceId, pccInstances[i], pccInstances[i]->getProtocolId(), pccInstances[i]->getUuid().c_str(), ETG_CENUM(CcErrorInternal, result)));
               finalResult = result;
            }
         }
         else
         {
            result = CC_ERR_INT_NULL_POINTER;
            ETG_TRACE_ERR(("initiateProtocolDisconnections(this = 0x%p, deviceId = %d): pointer to PCC instance is 0 (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
            finalResult = result;
         }
      }

      ETG_TRACE_USR1(("initiateProtocolDisconnections(this = 0x%p, deviceId = %d): %d of %d disconnection request have been handled successfully", (void *) this, _deviceId, successfullyHandledCnt, pccInstances.size()));

      if ((0u < successfullyHandledCnt) && (pccInstances.size() > successfullyHandledCnt))
      {
         finalResult = CC_ERR_INT_DCC_ONLY_PARTIALLY_HANDLED;
      }

      return finalResult;
   }

   Result DeviceConnectionController::connectProtocol(IN const ProtocolId protocolId, IN const Uuid& uuid, IN const ConnectionRequestOrigin connectionRequestOrigin)
   {
      ENTRY

      ETG_TRACE_USR1(("connectProtocol(this = 0x%p, deviceId = %d): protocolId = %d, uuid = \"%50s\", connectionRequestOrigin = %d", (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid.c_str(), ETG_CENUM(ConnectionRequestOrigin, connectionRequestOrigin)));

      Result result(CC_ERR_INT_NO_ERROR);
      ProtocolConnectionController* pccInstance(0);
      std::string eventName("");
      char eventParams[60] = {0};

      if (BM_CONNECTION_REQUEST_ORIGIN_LOCAL == connectionRequestOrigin)
      {
         eventName = "CONNECT_LOCAL";
      }
      else if (BM_CONNECTION_REQUEST_ORIGIN_REMOTE == connectionRequestOrigin)
      {
         eventName = "CONNECT_REMOTE";
      }
      else
      {
         ETG_TRACE_ERR(("connectProtocol(this = 0x%p, deviceId = %d): connection request originator is unknown", (void *) this, _deviceId));
         return CC_ERR_INT_INVALID_PARAMETER;
      }

      result = LocalSpm::getDbManager().getProtocolConnectionController(&pccInstance, _deviceId, protocolId, uuid);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR4(("connectProtocol(this = 0x%p, deviceId = %d): sending event %50s to PCC SM instance (instance = 0x%p, protocol ID = %d, uuid = \"%50s\")",
               (void *) this, _deviceId, eventName.c_str(), (void *) pccInstance, protocolId, uuid.c_str()));

         result = pccInstance->SendEventByName(eventName.c_str(), eventParams);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("connectProtocol(this = 0x%p, deviceId = %d): could not send event %50s to PCC SM instance (instance = 0x%p, protocol ID = %d, uuid = \"%50s\") (error = %d)",
                  (void *) this, _deviceId, eventName.c_str(), (void *) pccInstance, protocolId, uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("connectProtocol(this = 0x%p, deviceId = %d): could not get PCC instance for protocol ID = %d, uuid = \"%50s\" from DB (error = %d)",
               (void *) this, _deviceId, protocolId, uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceConnectionController::doConnectProtocols(IN const ProtocolConnectionControllerList& pccInstances, IN const ConnectionRequestOrigin connectionRequestOrigin)
   {
      ENTRY

      ETG_TRACE_USR1(("doConnectProtocols(this = 0x%p, deviceId = %d): connectionRequestOrigin = %d", (void *) this, _deviceId, connectionRequestOrigin));

      Result result(CC_ERR_INT_NO_ERROR);
      Result finalResult(CC_ERR_INT_NO_ERROR);
      std::string eventName("");

      if (BM_CONNECTION_REQUEST_ORIGIN_LOCAL == connectionRequestOrigin)
      {
         eventName = "CONNECT_LOCAL";
      }
      else if (BM_CONNECTION_REQUEST_ORIGIN_REMOTE == connectionRequestOrigin)
      {
         eventName = "CONNECT_REMOTE";
      }
      else
      {
         ETG_TRACE_ERR(("doConnectProtocols(this = 0x%p, deviceId = %d): connection request originator is unknown", (void *) this, _deviceId));
         return CC_ERR_INT_INVALID_PARAMETER;
      }

      for (size_t i = 0; i < pccInstances.size(); i++)
      {
         if (0 != pccInstances[i])
         {
            ETG_TRACE_USR4(("doConnectProtocols(this = 0x%p, deviceId = %d): sending event %50s to PCC SM instance (instance = 0x%p, protocol ID = %d, uuid = \"%50s\")",
                  (void *) this, _deviceId, eventName.c_str(), (void *) pccInstances[i], pccInstances[i]->getProtocolId(), pccInstances[i]->getUuid().c_str()));

            result = pccInstances[i]->SendEventByName(eventName.c_str(), 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("doConnectProtocols(this = 0x%p, deviceId = %d): could not send event %50s to PCC instance (instance = 0x%p, protocol ID = %d, uuid = \"%50s\") (error = %d)",
                     (void *) this, _deviceId, eventName.c_str(), (void *) pccInstances[i], pccInstances[i]->getProtocolId(), pccInstances[i]->getUuid().c_str(), ETG_CENUM(CcErrorInternal, result)));
               finalResult = result;
            }
         }
         else
         {
            result = CC_ERR_INT_NULL_POINTER;
            ETG_TRACE_ERR(("doConnectProtocols(this = 0x%p, deviceId = %d): pointer to PCC instance is 0 (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
            finalResult = result;
         }
      }

      return finalResult;
   }

   Result DeviceConnectionController::removeProtocolPendingToConnect(IN const ProtocolId protocolId, IN const Uuid& uuid, IN const bool sendFinishedEvent)
   {
      ENTRY

      ETG_TRACE_USR1(("removeProtocolPendingToConnect(this = 0x%p, deviceId = %d): protocolId = %d, uuid = \"%50s\" sendFinishedEvent - %10s",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid.c_str(), sendFinishedEvent ? "true" : "false"));

      Result result(CC_ERR_INT_NO_ERROR);
      ProtocolList::iterator it;

      it = std::find(_protocolsPendingToConnect.begin(), _protocolsPendingToConnect.end(), Protocol(protocolId, uuid));

      if (it != _protocolsPendingToConnect.end())
      {
         _protocolsPendingToConnect.erase(it);

         if(true == sendFinishedEvent)
         {
            checkAndSendAutoConnectedFinished();
         }
      }
      else
      {
         result = CC_ERR_INT_GENERAL_ERROR;
         ETG_TRACE_ERR(("removeProtocolPendingToConnect(this = 0x%p, deviceId = %d): could not find given protocol ID/UUID in list of protocols waiting for getting connected (error = %d)",
               (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      ProtocolList hmiUserSelectableProtocols;
      (void) LocalSpm::getDataProvider().getHmiUserSelectableProtocols(hmiUserSelectableProtocols);

      Protocol disconnectedProtocol(protocolId, uuid);

      if (hmiUserSelectableProtocols.end() != std::find(hmiUserSelectableProtocols.begin(),
            hmiUserSelectableProtocols.end(), disconnectedProtocol))
      {
         size_t eraseResult = _localProtocolTargetConnectionStatusMap.erase(disconnectedProtocol);

         if (0 != eraseResult)
         {
            this->traceProtocolTargetConnectionStatusMap();

            if (true == _localProtocolTargetConnectionStatusMap.empty())
            {
               ETG_TRACE_USR1(("removeProtocolPendingToConnect(this = 0x%p, deviceId = %d): list of pending locally initiated protocol connections/disconnections is empty -> activating time-limited blocking of remote connection requests",
                     (void *) this, _deviceId));
            }

            if (0u < LocalSpm::getDataProvider().getBmCoreConfiguration()._blockDeviceRemoteConnectionsTimeoutSeconds)
            {
               // temporarily block remote connections for the device currently handled by DCC
               result = LocalSpm::getBmCoreMainController().blockRemoteConnectionsTimeLimited(_deviceId);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("removeProtocolPendingToConnect(this = 0x%p, deviceId = %d): could not activate time-limited blocking of remote connection requests (error = %d)",
                        (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
               }
            }
         }
         else
         {
            // element has not been erased, since it was not in the map
            ETG_TRACE_ERR(("removeProtocolPendingToConnect(this = 0x%p, deviceId = %d): could not remove given protocol = (%d, \"%50s\") from list of pending locally initiated protocol connections/disconnections (not found)",
                  (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocolId), uuid.c_str()));
         }
      }

      return result;
   }

   void DeviceConnectionController::checkAndSendAutoConnectedFinished()
   {
      ENTRY

      ETG_TRACE_USR1(("checkAndSendAutoConnectedFinished(this = 0x%p, deviceId = %d): entered", (void *) this, _deviceId));
      Result result(CC_ERR_INT_NO_ERROR);

      VARTRACE(_protocolsPendingToConnect);

      if (_protocolsPendingToConnect.empty())
      {
         _delayLocalProtocolConnections = false;

         bool sendEvent = true;

         if (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported)
         {
            // check autoconnection is running for the device.
            DeviceId deviceIdConnectionInProgress = LocalSpm::getAutoConnectionController().getDeviceIdConnectionInProgress();

            //If only SPP profile is getting connected due to auto connection for CPW device so need to wait for the update from SPI/HMI
            // So auto-connection finished event should not be send to autoconnectioncontroller.
            if(deviceIdConnectionInProgress == _deviceId)
            {
               BtLimitationModeInfo btLimitationModeInfo;
               result = LocalSpm::getBmCoreMainController().getBtLimitationModeInfo(btLimitationModeInfo, _deviceId);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  if ((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == btLimitationModeInfo._limitationMode._limitationCommunicationIf)
                        ||(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == btLimitationModeInfo._limitationMode._limitationCommunicationIf))
                  {
                     if ((BM_LIMITATION_FEATURE_CAR_PLAY == btLimitationModeInfo._limitationMode._limitationFeature) &&
                           (BM_LIMITATION_STATE_PREPARED == btLimitationModeInfo._limitationState))
                     {
                        sendEvent = false;
                        ETG_TRACE_USR4(("checkAndSendAutoConnectedFinished(this = 0x%p, deviceId = %d): Not sending event DEVICE_CONNECTION_FINISHED beacuse BTLimitation is going on for this device",
                              (void *) this, _deviceId));
                     }
                  }
               }
            }
         }

         if(true == sendEvent)
         {
            char eventParams[20];

            result = LocalSpm::getAutoConnectionController().ParameterDEVICE_CONNECTION_FINISHED(eventParams, sizeof(eventParams), _deviceId);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               ETG_TRACE_USR4(("checkAndSendAutoConnectedFinished(this = 0x%p, deviceId = %d): sending event DEVICE_CONNECTION_FINISHED to AutoConnectionController SM",
                     (void *) this, _deviceId));

               result = LocalSpm::getAutoConnectionController().SendEventByName("DEVICE_CONNECTION_FINISHED", eventParams);

               if(CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("checkAndSendAutoConnectedFinished(this = 0x%p, deviceId = %d): could not send event DEVICE_CONNECTION_FINISHED (error = %d)",
                        (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               ETG_TRACE_ERR(("checkAndSendAutoConnectedFinished(this = 0x%p, deviceId = %d): could not marshal event parameters for event DEVICE_CONNECTION_FINISHED (error = %d)",
                     (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }
   }

   Result DeviceConnectionController::isDisconnectionFinished()
   {
      ENTRY

      ETG_TRACE_USR1(("isDisconnectionFinished(this = 0x%p, deviceId = %d)", (void *) this, _deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      if ((true == _allProtocolsDisconnected) && (true == _deviceDisconnected))
      {
         ETG_TRACE_USR4(("isDisconnectionFinished(this = 0x%p, deviceId = %d): sending event DISCONNECTION_FINISHED to myself", (void *) this, _deviceId));
         result = this->SendEventByName("DISCONNECTION_FINISHED", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("enterDisconnectionFailed(this = 0x%p, deviceId = %d): could not send event DISCONNECTION_FINISHED (error = %d)", (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   void DeviceConnectionController::setConnectionRequestOrigin(IN const ConnectionRequestOrigin origin)
   {
      ETG_TRACE_USR1(("setConnectionRequestOrigin(this = 0x%p, origin = %d)", (void *) this, ETG_CENUM(ConnectionRequestOrigin, origin)));

      _origin = origin;
   }

   void DeviceConnectionController::setLocalProtocolTargetConnectionStatusInMap(IN Protocol protocol, IN TargetConnectionStatus targetConnectionStatus)
   {
      ETG_TRACE_USR1(("setLocalProtocolTargetConnectionStatusInMap(this = 0x%p, deviceId = %d): protocolId = %d, uuid = \"%50s\" targetConnectionStatus = %d",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, protocol._protocolId), protocol._uuid.c_str() ,ETG_CENUM(TargetConnectionStatus, targetConnectionStatus)));

      std::pair<ProtocolTargetConnectionStatusMap::iterator, bool> insertResult =
            _localProtocolTargetConnectionStatusMap.insert(std::pair<Protocol, TargetConnectionStatus>(protocol,
                  targetConnectionStatus));

      if (false == insertResult.second)
      {
         // element has not been inserted, since it is already in map => update value of that element
         insertResult.first->second = targetConnectionStatus;
      }

      this->traceProtocolTargetConnectionStatusMap();
   }
}
