/**
 * @file BmMeetsDeviceHandleInst.cpp
 *
 * @par SW-Component
 * Bluetooth Connection Manager Core
 *
 * @brief This file contains the definition of the class BlockRemoteConnectionsController
 *
 * @copyright (c) 2017 Robert Bosch GmbH
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup BmCoreModule
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BlockRemoteConnectionsController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#endif
#endif

#include "BlockRemoteConnectionsController.h"
#include "BmMeetsDeviceHandle.h"
#include "BmVarTrace.h"
#include <algorithm> // required for using find_if

namespace bmcore
{
   BlockRemoteConnectionsController::BlockRemoteConnectionsController(BmCoreMainController& bmCoreMainController) :
         _bmCoreMainController(bmCoreMainController), _deviceBlockRemoteConnectionsControlList()
   {
   }

   BlockRemoteConnectionsController::~BlockRemoteConnectionsController()
   {
   }

   Result BlockRemoteConnectionsController::initializeDeviceRemoteConnectableControlList(const DeviceIdList& pairedDeviceHandles)
   {
      ETG_TRACE_USR1(("initializeDeviceRemoteConnectableControlList: pairedDeviceHandles ="));
      VARTRACE(pairedDeviceHandles);

      for (size_t idx = 0u; idx < _deviceBlockRemoteConnectionsControlList.size(); ++idx)
      {
         _deviceBlockRemoteConnectionsControlList[idx]->stopBlockRemoteConnectionsTimer();
      }

      _deviceBlockRemoteConnectionsControlList.clear();

      for (size_t idx = 0u; idx < pairedDeviceHandles.size(); ++idx)
      {
         (void) this->addDeviceRemoteConnectableControl(pairedDeviceHandles[idx]);
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result BlockRemoteConnectionsController::addDeviceRemoteConnectableControl(const DeviceId deviceHandle)
   {
      bool deviceIsAvailable = false;
      ETG_TRACE_USR1(("addDeviceRemoteConnectableControl: deviceHandle = %d", deviceHandle));

      // Check whether device is already available in the list
      for (size_t idx = 0u; idx < _deviceBlockRemoteConnectionsControlList.size(); ++idx)
      {
         if(_deviceBlockRemoteConnectionsControlList[idx]->getDeviceHandle() == deviceHandle)
         {
            deviceIsAvailable = true;
            break;
         }
      }

      if(false == deviceIsAvailable)
      {
         DeviceBlockRemoteConnectionsControl *deviceBlockRemoteConnectionsControl = new DeviceBlockRemoteConnectionsControl(deviceHandle);

         if(deviceBlockRemoteConnectionsControl)
         {
            ETG_TRACE_USR1(("addDeviceRemoteConnectableControl: added in the deviceBlockRemoteConnectionsControlList"));
            _deviceBlockRemoteConnectionsControlList.push_back(deviceBlockRemoteConnectionsControl);
         }
         else
         {
            ETG_TRACE_ERR(("addDeviceRemoteConnectableControl: not able to create memory for deviceBlockRemoteConnectionsControl"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("addDeviceRemoteConnectableControl: device block remote connections control with given device handle is already in device block remote connections control list"));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result BlockRemoteConnectionsController::removeDeviceRemoteConnectableControl(const DeviceId deviceHandle)
   {
      ETG_TRACE_USR1(("removeDeviceRemoteConnectableControl: deviceHandle = %d", deviceHandle));

      for (size_t idx = 0u; idx < _deviceBlockRemoteConnectionsControlList.size(); ++idx)
      {
         if(_deviceBlockRemoteConnectionsControlList[idx]->getDeviceHandle() == deviceHandle)
         {
            DeviceBlockRemoteConnectionsControl *deviceBlockRemoteConnectionsControl =
                  static_cast<DeviceBlockRemoteConnectionsControl*>(_deviceBlockRemoteConnectionsControlList[idx]);

            _deviceBlockRemoteConnectionsControlList.erase(_deviceBlockRemoteConnectionsControlList.begin() + idx);

            ETG_TRACE_USR1(("removeDeviceRemoteConnectableControl: erased the device from list"));

            delete deviceBlockRemoteConnectionsControl;

            break;
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result BlockRemoteConnectionsController::startBlockRemoteConnectionsTimer(const DeviceId deviceHandle)
   {
      ETG_TRACE_USR1(("startBlockRemoteConnectionsTimer: deviceHandle = %d", deviceHandle));

      if (0u == deviceHandle)
      {
         ETG_TRACE_USR1(("startBlockRemoteConnectionsTimer: starting block remote connections timer for ALL paired devices"));

         for (size_t idx = 0u; idx < _deviceBlockRemoteConnectionsControlList.size(); ++idx)
         {
            (void) _deviceBlockRemoteConnectionsControlList[idx]->startBlockRemoteConnectionsTimer();
         }
      }
      else
      {
         for (size_t idx = 0u; idx < _deviceBlockRemoteConnectionsControlList.size(); ++idx)
         {
            if(_deviceBlockRemoteConnectionsControlList[idx]->getDeviceHandle() == deviceHandle)
            {
               ETG_TRACE_USR1(("startBlockRemoteConnectionsTimer: starting block remote connections timer for device with given device handle"));
               (void) _deviceBlockRemoteConnectionsControlList[idx]->startBlockRemoteConnectionsTimer();
               break;
            }
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result BlockRemoteConnectionsController::stopBlockRemoteConnectionsTimer(const DeviceId deviceHandle)
   {
      ETG_TRACE_USR1(("stopBlockRemoteConnectionsTimer: deviceHandle = %d", deviceHandle));

      if (0u == deviceHandle)
      {
         ETG_TRACE_USR1(("stopBlockRemoteConnectionsTimer: stopping block remote connections timer for ALL paired devices"));

         for (size_t idx = 0u; idx < _deviceBlockRemoteConnectionsControlList.size(); ++idx)
         {
            (void) _deviceBlockRemoteConnectionsControlList[idx]->stopBlockRemoteConnectionsTimer();
         }
      }
      else
      {
         for (size_t idx = 0u; idx < _deviceBlockRemoteConnectionsControlList.size(); ++idx)
         {
            if(_deviceBlockRemoteConnectionsControlList[idx]->getDeviceHandle() == deviceHandle)
            {
               ETG_TRACE_USR1(("stopBlockRemoteConnectionsTimer: stopping block remote connections timer for device with given device handle"));
               (void) _deviceBlockRemoteConnectionsControlList[idx]->stopBlockRemoteConnectionsTimer();
               break;
            }
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }
}
