#ifndef _QUERY_H_
#define _QUERY_H_

/* includes */
#include <map>
#include "Database.h"
#include "Lock.h"

namespace bmcore
{
   /* the list of all queries */
   typedef struct {
       const char *query;
   } tSQLStatement;

   /* transaction lock global to all data base connections */
   static Lock gTransactionLock;

   class Query
   {
   public:
      Query();
      ~Query();

      Result Select(const ListType listType, const DeviceId deviceID, const char *format, ...);
      Result Select(const ListType listType, const DeviceId deviceID, const Index startIndex, const Index noOfRows, const char *format, ...);
      Result Select(const ListType listType, const DeviceId deviceID, const Index startIndex, const Index noOfRows, const char *format, va_list vl);
      Result Explain(char *planResult, size_t size);
      Result SetRow(RowNumber row);
      Result ResetRow();
      Result GetRow(RowNumber &row) const;
      Result SetLimit(const Index limit);
      Result SetLimitAndRow(const Index limit, const RowNumber row);
      Result Get(const char *format, ...);
      Result Get(const int doStep, const char *format, ...);
      Result Save();
      Result Restore();
      Result End();

      Result PositionSelect(const Query *parent, const DeviceId deviceID, const char *format, ...);
      Result PositionGet(Position &position, const ObjectId objectID);

      Result Insert(const ListType listType);
      Result Put(const char *format, ...);

      Result Update(const ListType listType);
      Result Change(const char *format, ...);

      Result Delete(const ListType listType);
      Result Remove(const char *format, ...);

      Result BeginTransaction(const ListType listType);
      Result EndTransaction();

      static Result Finalize(sqlite3 *db);

      Result DoLock();
      Result UnLock();

      static int GetQueryCacheCount();

   protected:
       Result GetInternal(const int DoStep, const char *format, va_list vl);
      Result Execute();
      Result Peek(const char *format, ...);

       Database *mDB;
      sqlite3_stmt *mStatement;
      ListType mListType;
      class Lock mLock; // to lock against multi threaded

      /* for select only */
      char *mSelectString;
      int mRow;
      int mPrevRow;
      int mStartIndex;
      int mLimit;
      int mEOF;
      int mSavedRow;
      int mSavedLimit;
      int mBindingForOffset;
      int mBindingForLimit;
      static void SqlTrace(void *context, const char *string);
      static void SqlProfile(void *context, const char *string, sqlite3_uint64 nanosecs);
      Result Prepare(sqlite3_stmt **statement, sqlite3 *db, const char *query);
      Result Finalize(sqlite3_stmt *statement) const;
      void CheckFatalError(sqlite3 *db, int sqlite3ResultCode);

      /* for others */
      int mLastSqlError;
   };
}

#endif // _QUERY_H_
