#ifndef _CONFLICTMANAGER_H_
#define _CONFLICTMANAGER_H_

#include "ILocalSpm.h"
#include "BmAllTypesInternal.h"
#include "BmCoreIfMessages.h"
#include <map>

namespace bmcore
{
using namespace std;

   class ConflictManager : public ILocalSpm
   {
   public:

      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  ConflictManager::ConflictManager(IN const ComponentId componentID)
       ***************************************************************************/
      /*!
       * \fn      ConflictManager()
       * \brief   Default Constructor
       * \param   [IN] componentID: Component ID of Conflict Manager
       ***************************************************************************/
      ConflictManager(IN const ComponentId componentID);

      /***************************************************************************
       ** FUNCTION:  ConflictManager::~ConflictManager()
       ***************************************************************************/
      /*!
       * \fn      ~ConflictManager()
       * \brief   Virtual Destructor
       ***************************************************************************/
      virtual ~ConflictManager();

      /***************************************************************************
       ** FUNCTION:  ConflictManager::processSetBtProfileUsageRequest(...)
       ***************************************************************************/
      /*!
       * \fn      processSetBtProfileUsageRequest(IN const BmCoreIfMessage_SetBtProfilesUsageRequest* bmCoreIfMessage)
       * \brief   Method to process incoming SetBtProfileUsageRequest message.
       * \param   [IN] bmCoreIfMessage: Pointer to class instance BmCoreIfMessage_SetBtProfilesUsageRequest
       * \retval  BmResult: BM_RESULT_OK if success else Non-zero error code
       ***************************************************************************/
      BmResult processSetBtProfileUsageRequest(IN const BmCoreIfMessage_SetBtProfilesUsageRequest* bmCoreIfMessage);

      /***************************************************************************
       ** FUNCTION:  ConflictManager::processNotifyUserDecisionRequest(...)
       ***************************************************************************/
      /*!
       * \fn      processNotifyUserDecisionRequest(IN const BmCoreIfMessage_NotifyUserDecisionRequest* bmCoreIfMessage,
       *				 IN const bool isInternallyTriggered = false)
       * \brief   Method to process incoming NotifyUserDecisionRequest message.
       * \param   [IN] bmCoreIfMessage: Pointer to class instance BmCoreIfMessage_NotifyUserDecisionRequest
       * \param   [IN] isInternallyTriggered: Boolean flag to be set/reset based on the message trigger
       * \retval  BmResult: BM_RESULT_OK if success else Non-zero error code
       ***************************************************************************/
      BmResult processNotifyUserDecisionRequest(IN const BmCoreIfMessage_NotifyUserDecisionRequest* bmCoreIfMessage,
            IN const bool isInternallyTriggered = false);

      /***************************************************************************
       ** FUNCTION:  ConflictManager::processNotifyUserDecisionRequest(...)
       ***************************************************************************/
      /*!
       * \fn      processNotifyUserDecisionRequest(IN const UserDecisionInfo& userDecision)
       * \brief   Method to process incoming conflict resolution internally.
       * \param   [IN] userDecision: Reference to class instance UserDecisionInfo
       * \retval  void
       ***************************************************************************/
      void processNotifyUserDecisionRequest(IN const UserDecisionInfo& userDecision);

      /***************************************************************************
       ** FUNCTION:  ConflictManager::resolveConflicts(...)
       ***************************************************************************/
      /*!
       * \fn      resolveConflicts(IN const ConflictTrigger conflictTrigger,
       * 			IN const DeviceId deviceID = 0u)
       * \brief   Method to be called when requester wants to resolve any conflicts.
       * \param   [IN] conflictTrigger: Conflict Trigger enum value of type ConflictTrigger
       * \param   [IN] deviceID: Unique Device ID for which the conflict has to be resolved
       * \retval  None
       ***************************************************************************/
      void resolveConflicts(IN const ConflictTrigger conflictTrigger, IN const DeviceId deviceID = 0u);

#if 0
      //Currently LocalSpm API's are not in use as Conflict Manager does not have a state machine.
      //Will use the default implementation of the virtual functions
      // ************************************
      // * Overridden methods of ILocalSpm *
      // ************************************

      /**
       * @brief Creates the BtLimitationController instance
       *
       * Creates the Pairing state machine (including creation of message queue).
       * Informs LocalSpm that create is ready -> createDone(0).
       *
       * @attention: running in SPM thread context
       *
       * @return void
       */
      void create();

      /**
       * @brief Performs the initialization of the Pairing Controller instance, i.e.:
       * - initialization of member variables
       *
       * Synchronization: re-entrant
       *
       * Performance considerations:
       *
       * @return 0 on success and an error code otherwise.
       */
      Result init(InitReason reason);

      /**
       * @brief Starts the Pairing Controller instance, implies that all other
       * components are available
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none.
       *
       * @return 0 on success and an error code otherwise
       */
      Result run();

      /**
       * @brief Stops the Pairing Controller instance
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none
       *
       * @return 0 on success and an error code otherwise
       */
      Result stop();

      /**
       * @brief Cleans up the resources used by the Pairing Controller instance
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none
       *
       * @return 0 on success and an error code otherwise
       */
      Result done();
#endif

   private:

      /***************************************************************************
       ** FUNCTION:  ConflictManager::updateConflictsDetectedList(...)
       ***************************************************************************/
      /*!
       * \fn      updateConflictsDetectedList(IN IssueInfoList& btConflictsDetectedList)
       * \brief   Method called to update conflicts list once the conflict is resolved.
       * \param   [IN] btConflictsDetectedList: List of updated conflicts.
       * \retval  None
       ***************************************************************************/
      void updateConflictsDetectedList(IN IssueInfoList& btConflictsDetectedList);

      map<ConflictTrigger, ConflictInfoList> conflictsOfInterest;
   };
}//End of namespace bmcore
#endif // _CONFLICTMANAGER_H_
