/**
 * @file BmVarTrace.cpp
 *
 * @par SW-Component
 * Bluetooth Connection Manager Core
 *
 * @brief This file contains the definition of variable tracing functions used by Bluetooth Connection Manager Core
 *
 * @copyright (c) 2016 Robert Bosch GmbH
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup BmCoreModule
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_VARIABLE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BmVarTrace.cpp.trc.h"
#endif
#endif

#include <cstring>
#include "BmVarTrace.h"

namespace bmcore
{
   const char* removePath(const char *path)
   {
      // search last '/'. if found return string after last '/' else return complete path
      char *filename = strrchr(const_cast<char*>(path), '/');
      if (filename)
         return filename+1;
      else
         return path;
   }

   void VarTrace(tClassType cls, ComponentState var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "ComponentState(%100s): %d [(%d)%s]", name, ETG_CENUM(ComponentState,var),line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "ComponentState(%100s): %d [(%d)%s]", name, ETG_CENUM(ComponentState,var),line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "ComponentState(%100s): %d [(%d)%s]", name, ETG_CENUM(ComponentState,var),line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "ComponentState(%100s): %d [(%d)%s]", name, ETG_CENUM(ComponentState,var),line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "ComponentState(%100s): %d", name, ETG_CENUM(ComponentState,var)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "ComponentState(%100s): %d", name, ETG_CENUM(ComponentState,var)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "ComponentState(%100s): %d", name, ETG_CENUM(ComponentState,var)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "ComponentState(%100s): %d", name, ETG_CENUM(ComponentState,var)));
          }
       }
   }

   void VarTrace(tClassType cls, CcErrorInternal var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "CcErrorInternal(%100s): %d [(%d)%s]", name, ETG_CENUM(CcErrorInternal,var),line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "CcErrorInternal(%100s): %d [(%d)%s]", name, ETG_CENUM(CcErrorInternal,var),line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "CcErrorInternal(%100s): %d [(%d)%s]", name, ETG_CENUM(CcErrorInternal,var),line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "CcErrorInternal(%100s): %d [(%d)%s]", name, ETG_CENUM(CcErrorInternal,var),line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "CcErrorInternal(%100s): %d", name, ETG_CENUM(CcErrorInternal,var)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "CcErrorInternal(%100s): %d", name, ETG_CENUM(CcErrorInternal,var)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "CcErrorInternal(%100s): %d", name, ETG_CENUM(CcErrorInternal,var)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "CcErrorInternal(%100s): %d", name, ETG_CENUM(CcErrorInternal,var)));
          }
       }
   }

   void VarTrace(tClassType cls, const char* var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "char*(%100s): %100s [(%d)%s]", name, CHECK_NULL(var),line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "char*(%100s): %100s [(%d)%s]", name, CHECK_NULL(var),line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "char*(%100s): %100s [(%d)%s]", name, CHECK_NULL(var),line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "char*(%100s): %100s [(%d)%s]", name, CHECK_NULL(var),line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "char*(%100s): %s", name, CHECK_NULL(var)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "char*(%100s): %s", name, CHECK_NULL(var)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "char*(%100s): %s", name, CHECK_NULL(var)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "char*(%100s): %s", name, CHECK_NULL(var)));
          }
       }
   }

   void VarTrace(tClassType cls, unsigned char* var, const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned char*(%100s): %50s [(%d)%s]", name, CHECK_NULL((char*) var), line, removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned char*(%100s): %50s [(%d)%s]", name, CHECK_NULL((char*) var), line, removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned char*(%100s): %50s [(%d)%s]", name, CHECK_NULL((char*) var), line, removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned char*(%100s): %50s [(%d)%s]", name, CHECK_NULL((char*) var), line, removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned char*(%100s): %s", name, CHECK_NULL((char*) var)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned char*(%100s): %s", name, CHECK_NULL((char*) var)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned char*(%100s): %s", name, CHECK_NULL((char*) var)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned char*(%100s): %s", name, CHECK_NULL((char*) var)));
          }
       }
   }

   void VarTrace(tClassType cls, bool var,const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "bool(%100s): %10s [(%d)%s]", name,(var ? "true" : "false"),line,removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "bool(%100s): %10s [(%d)%s]", name,(var ? "true" : "false"),line,removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "bool(%100s): %10s [(%d)%s]", name,(var ? "true" : "false"),line,removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "bool(%100s): %10s [(%d)%s]", name,(var ? "true" : "false"),line,removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "bool(%100s): %10s", name,var ? "true" : "false"));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "bool(%100s): %10s", name,var ? "true" : "false"));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "bool(%100s): %10s", name,var ? "true" : "false"));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "bool(%100s): %10s", name,var ? "true" : "false"));
         }
      }
   }

   void VarTrace(tClassType cls, int var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "int(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "int(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "int(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "int(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "int(%100s): %d", name, var));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "int(%100s): %d", name, var));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "int(%100s): %d", name, var));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "int(%100s): %d", name, var));
          }
       }
   }

   void VarTrace(tClassType cls, unsigned int var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned int(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned int(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned int(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned int(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned int(%100s): %d", name, var));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned int(%100s): %d", name, var));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned int(%100s): %d", name, var));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned int(%100s): %d", name, var));
          }
       }
   }
   #ifdef __x86_64
   void VarTrace(tClassType cls, size_t var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned int(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned int(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned int(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned int(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned int(%100s): %d", name, var));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned int(%100s): %d", name, var));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned int(%100s): %d", name, var));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned int(%100s): %d", name, var));
          }
       }
   }
   #endif
   void VarTrace(tClassType cls, float var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned int(%100s): %f [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned int(%100s): %f [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned int(%100s): %f [(%d)%s]", name, var,line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned int(%100s): %f [(%d)%s]", name, var,line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned int(%100s): %f", name, var));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned int(%100s): %f", name, var));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned int(%100s): %f", name, var));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned int(%100s): %f", name, var));
          }
       }
   }

   void VarTrace(tClassType cls, char var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "char(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "char(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "char(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "char(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "char(%100s): %d", name, var));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "char(%100s): %d", name, var));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "char(%100s): %d", name, var));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "char(%100s): %d", name, var));
          }
       }
   }

   void VarTrace(tClassType cls, unsigned char var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned char(%100s): %u [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned char(%100s): %u [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned char(%100s): %u [(%d)%s]", name, var,line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned char(%100s): %u [(%d)%s]", name, var,line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned char(%100s): %u", name, var));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned char(%100s): %u", name, var));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned char(%100s): %u", name, var));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned char(%100s): %u", name, var));
          }
       }
   }


   void VarTrace(tClassType cls, short var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "short(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "short(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "short(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "short(%100s): %d [(%d)%s]", name, var,line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "short(%100s): %d", name, var));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "short(%100s): %d", name, var));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "short(%100s): %d", name, var));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "short(%100s): %d", name, var));
          }
       }
   }

   void VarTrace(tClassType cls, unsigned short var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned short(%100s): %u [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned short(%100s): %u [(%d)%s]", name, var,line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned short(%100s): %u [(%d)%s]", name, var,line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned short(%100s): %u [(%d)%s]", name, var,line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned short(%100s): %u", name, var));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned short(%100s): %u", name, var));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned short(%100s): %u", name, var));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned short(%100s): %u", name, var));
          }
       }
   }

   void VarTrace(tClassType cls, long long var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "long long(%100s): 0x%08x%08x [(%d)%s]", name, (int)((unsigned long long)var>>32), (int)(var&0xffffffff),line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "long long(%100s): 0x%08x%08x [(%d)%s]", name, (int)((unsigned long long)var>>32), (int)(var&0xffffffff),line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "long long(%100s): 0x%08x%08x [(%d)%s]", name, (int)((unsigned long long)var>>32), (int)(var&0xffffffff),line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "long long(%100s): 0x%08x%08x [(%d)%s]", name, (int)((unsigned long long)var>>32), (int)(var&0xffffffff),line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "long long(%100s): 0x%08x%08x ", name, (int)((unsigned long long)var>>32), (int)(var&0xffffffff)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "long long(%100s): 0x%08x%08x ", name, (int)((unsigned long long)var>>32), (int)(var&0xffffffff)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "long long(%100s): 0x%08x%08x ", name, (int)((unsigned long long)var>>32), (int)(var&0xffffffff)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "long long(%100s): 0x%08x%08x ", name, (int)((unsigned long long)var>>32), (int)(var&0xffffffff)));
          }
       }
   }

   void VarTrace(tClassType cls, unsigned long long var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned long long(%100s): 0x%08x%08x [(%d)%s]", name, (int)(var>>32), (int)(var&0xffffffff),line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned long long(%100s): 0x%08x%08x [(%d)%s]", name, (int)(var>>32), (int)(var&0xffffffff),line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned long long(%100s): 0x%08x%08x [(%d)%s]", name, (int)(var>>32), (int)(var&0xffffffff),line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned long long(%100s): 0x%08x%08x [(%d)%s]", name, (int)(var>>32), (int)(var&0xffffffff),line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "unsigned long long(%100s):  0x%08x%08x ", name, (int)(var>>32), (int)(var&0xffffffff)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "unsigned long long(%100s):  0x%08x%08x ", name, (int)(var>>32), (int)(var&0xffffffff)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "unsigned long long(%100s):  0x%08x%08x ", name, (int)(var>>32), (int)(var&0xffffffff)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "unsigned long long(%100s):  0x%08x%08x ", name, (int)(var>>32), (int)(var&0xffffffff)));
          }
       }
   }

//   void VarTrace(tClassType cls, std::vector <tUInt> var,const char* name, const unsigned char level, const char *file, const int line)
//   {
//      if (true == etg_bIsTraceActive(cls, TR_LEVEL_USER_3))
//      {
//           if (file != NULL && line != -1)
//           {
//             ETG_TRACE_USR3_DCL((cls, "vector<tUInt>(%100s): [(%d)%s]", name,line,removePath(file)));
//           }
//           else
//           {
//             ETG_TRACE_USR3_DCL((cls, "vector<tUInt>(%s):", name));
//           }
//         std::vector<tUInt>::iterator it;
//         int i;
//         char elementName[MAXNAME];
//         for(it = var.begin(), i=0; it < var.end(); it++, i++)
//         {
//            snprintf(elementName,MAXNAME-1,"%s[%d]",name,i);
//            VarTrace(cls, *it, elementName, level);
//         }
//      }
//   }

   void VarTrace(tClassType cls, std::string var,const char* name, const unsigned char level, const char *file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "string(%100s): %50s [(%d)%s]", name, var.c_str(),line,removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "string(%100s): %50s [(%d)%s]", name, var.c_str(),line,removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "string(%100s): %50s [(%d)%s]", name, var.c_str(),line,removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "string(%100s): %50s [(%d)%s]", name, var.c_str(),line,removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "string(%100s): %50s", name, var.c_str()));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "string(%100s): %50s", name, var.c_str()));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "string(%100s): %50s", name, var.c_str()));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "string(%100s): %50s", name, var.c_str()));
          }
       }
   }

   void VarTrace(tClassType cls, SwitchStatus var, const char* name, const unsigned char level, const char* file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "SwitchStatus(%100s): switchState: %d, switchedOffReason: %d [(%d)%s]",
                  name, ETG_CENUM(SwitchState, var._switchState), ETG_CENUM(SwitchedOffReason, var._switchedOffReason),
                  line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "SwitchStatus(%100s): switchState: %d, switchedOffReason: %d [(%d)%s]",
                  name, ETG_CENUM(SwitchState, var._switchState), ETG_CENUM(SwitchedOffReason, var._switchedOffReason),
                  line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "SwitchStatus(%100s): switchState: %d, switchedOffReason: %d [(%d)%s]",
                  name, ETG_CENUM(SwitchState, var._switchState), ETG_CENUM(SwitchedOffReason, var._switchedOffReason),
                  line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "SwitchStatus(%100s): switchState: %d, switchedOffReason: %d [(%d)%s]",
                  name, ETG_CENUM(SwitchState, var._switchState), ETG_CENUM(SwitchedOffReason, var._switchedOffReason),
                  line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "SwitchStatus(%100s): switchState: %d switchedOffReason: %d",
                  name, ETG_CENUM(SwitchState, var._switchState), ETG_CENUM(SwitchedOffReason, var._switchedOffReason)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "SwitchStatus(%100s): switchState: %d switchedOffReason: %d",
                  name, ETG_CENUM(SwitchState, var._switchState), ETG_CENUM(SwitchedOffReason, var._switchedOffReason)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "SwitchStatus(%100s): switchState: %d switchedOffReason: %d",
                  name, ETG_CENUM(SwitchState, var._switchState), ETG_CENUM(SwitchedOffReason, var._switchedOffReason)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "SwitchStatus(%100s): switchState: %d switchedOffReason: %d",
                  name, ETG_CENUM(SwitchState, var._switchState), ETG_CENUM(SwitchedOffReason, var._switchedOffReason)));
         }
      }
   }

   void VarTrace(tClassType cls, BtSystemState var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BtSystemState(%100s): _btSystemState = %d [(%d)%s]", name, ETG_CENUM(BluetoothSystemState, var._bluetoothSystemState), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BtSystemState(%100s): _btSystemState = %d [(%d)%s]", name, ETG_CENUM(BluetoothSystemState, var._bluetoothSystemState), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BtSystemState(%100s): _btSystemState = %d [(%d)%s]", name, ETG_CENUM(BluetoothSystemState, var._bluetoothSystemState), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BtSystemState(%100s): _btSystemState = %d [(%d)%s]", name, ETG_CENUM(BluetoothSystemState, var._bluetoothSystemState), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BtSystemState(%100s): _btSystemState = %d", name, ETG_CENUM(BluetoothSystemState, var._bluetoothSystemState)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BtSystemState(%100s): _btSystemState = %d", name, ETG_CENUM(BluetoothSystemState, var._bluetoothSystemState)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BtSystemState(%100s): _btSystemState = %d", name, ETG_CENUM(BluetoothSystemState, var._bluetoothSystemState)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BtSystemState(%100s): _btSystemState = %d", name, ETG_CENUM(BluetoothSystemState, var._bluetoothSystemState)));
         }
      }
   }

   void VarTrace(tClassType cls, LocalFriendlyName var, const char* name, const unsigned char level, const char* file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "BtLocalFriendlyName(%100s): btLocalFriendlyName: %50s [(%d)%s]", name, var._localFriendlyName.c_str(), line, removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "BtLocalFriendlyName(%100s): btLocalFriendlyName: %50s [(%d)%s]", name, var._localFriendlyName.c_str(), line, removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "BtLocalFriendlyName(%100s): btLocalFriendlyName: %50s [(%d)%s]", name, var._localFriendlyName.c_str(), line, removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "BtLocalFriendlyName(%100s): btLocalFriendlyName: %50s [(%d)%s]", name, var._localFriendlyName.c_str(), line, removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "BtLocalFriendlyName(%100s): btLocalFriendlyName: %50s", name, var._localFriendlyName.c_str()));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "BtLocalFriendlyName(%100s): btLocalFriendlyName: %50s", name, var._localFriendlyName.c_str()));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "BtLocalFriendlyName(%100s): btLocalFriendlyName: %50s", name, var._localFriendlyName.c_str()));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "BtLocalFriendlyName(%100s): btLocalFriendlyName: %50s", name, var._localFriendlyName.c_str()));
          }
       }
   }

   void VarTrace(tClassType cls,const LocalInfo& var, const char* name, const unsigned char level, const char* file, const int line)
   {
      std::string parameters("module name = \"");
      parameters += var._moduleName;
      parameters += "\", BD address = \"";
      parameters += var._bdAddress;
      parameters += "\", BD friendly name = \"";
      parameters += var._bdName;
      parameters += "\", stackInitialized = \"";
      parameters += (var._stackInitialized ? "true": "false");
      parameters += "\"";

       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "LocalInfo(%100s): %200s [(%d)%s]", name, parameters.c_str(), line, removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "LocalInfo(%100s): %200s [(%d)%s]", name, parameters.c_str(), line, removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "LocalInfo(%100s): %200s [(%d)%s]", name, parameters.c_str(), line, removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "LocalInfo(%100s): %200s [(%d)%s]", name, parameters.c_str(), line, removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "LocalInfo(%100s): module name = \"%50s\", BD address = \"%50s\", BD friendly name = \"%50s\", _stackInitialized = \"%10s\"",
                   name, var._moduleName.c_str(), var._bdAddress.c_str(), var._bdName.c_str(), var._stackInitialized ? "true" : "false"));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "LocalInfo(%100s): module name = \"%50s\", BD address = \"%50s\", BD friendly name = \"%50s\", _stackInitialized = \"%10s\"",
                   name, var._moduleName.c_str(), var._bdAddress.c_str(), var._bdName.c_str(), var._stackInitialized ? "true" : "false"));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "LocalInfo(%100s): module name = \"%50s\", BD address = \"%50s\", BD friendly name = \"%50s\", _stackInitialized = \"%10s\"",
                   name, var._moduleName.c_str(), var._bdAddress.c_str(), var._bdName.c_str(), var._stackInitialized ? "true" : "false"));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "LocalInfo(%100s): module name = \"%50s\", BD address = \"%50s\", BD friendly name = \"%50s\", _stackInitialized = \"%10s\"",
                   name, var._moduleName.c_str(), var._bdAddress.c_str(), var._bdName.c_str(), var._stackInitialized ? "true" : "false"));
          }
       }

       std::string elementName(name);
       VarTrace(cls, var._protocolInfos, (elementName + ".protocolInfoMap").c_str(), level);
       VarTrace(cls, var._vehicleIdentification, (elementName + ".deviceIdentification").c_str(), level);
   }

   void VarTrace(tClassType cls, ProtocolInfo var, const char* name, const unsigned char level, const char* file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "ProtocolInfo(%100s): name = \"%50s\", version = 0x%04X, isSupported = %u, favoriteIndex = %u [(%d)%s]",
                   name, var._name.c_str(), var._version, var._isSupported, var._favoriteIndex, line, removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "ProtocolInfo(%100s): name = \"%50s\", version = 0x%04X, isSupported = %u, favoriteIndex = %u [(%d)%s]",
                   name, var._name.c_str(), var._version, var._isSupported, var._favoriteIndex, line, removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "ProtocolInfo(%100s): name = \"%50s\", version = 0x%04X, isSupported = %u,  favoriteIndex = %u [(%d)%s]",
                   name, var._name.c_str(), var._version, var._isSupported, var._favoriteIndex, line, removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "ProtocolInfo(%100s): name = \"%50s\", version = 0x%04X, isSupported = %u, favoriteIndex = %u [(%d)%s]",
                   name, var._name.c_str(), var._version, var._isSupported, var._favoriteIndex, line, removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "ProtocolInfo(%100s): name = \"%50s\", version = 0x%04X, isSupported = %u, favoriteIndex = %u",
                   name, var._name.c_str(), var._version, var._isSupported, var._favoriteIndex));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "ProtocolInfo(%100s): name = \"%50s\", version = 0x%04X, isSupported = %u, favoriteIndex = %u",
                   name, var._name.c_str(), var._version, var._isSupported, var._favoriteIndex));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "ProtocolInfo(%100s): name = \"%50s\", version = 0x%04X, isSupported = %u, favoriteIndex = %u",
                   name, var._name.c_str(), var._version, var._isSupported, var._favoriteIndex));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "ProtocolInfo(%100s): name = \"%50s\", version = 0x%04X, isSupported = %u, favoriteIndex = %u",
                   name, var._name.c_str(), var._version, var._isSupported, var._favoriteIndex));
          }
       }
   }

   void VarTrace(tClassType cls, ProtocolInfoMap var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "ProtocolInfoMap(%100s): [(%d)%s]", name, line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "ProtocolInfoMap(%100s): [(%d)%s]", name, line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "ProtocolInfoMap(%100s): [(%d)%s]", name, line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "ProtocolInfoMap(%100s): [(%d)%s]", name, line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "ProtocolInfoMap(%100s):", name));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "ProtocolInfoMap(%100s):", name));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "ProtocolInfoMap(%100s):", name));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "ProtocolInfoMap(%100s):", name));
         }
      }

      ProtocolInfoMap::iterator it;
      char keyName[MAXNAME];

      for (it = var.begin(); it != var.end(); it++)
      {
         snprintf(keyName, MAXNAME - 1, "%s[%d]", name, ETG_CENUM(ProtocolId, it->first));
         VarTrace(cls, it->second, keyName, level);
      }
   }

   void VarTrace(tClassType cls, UsagePreference var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "UsagePreference(%100s): %d [(%d)%s]",
                  name, ETG_CENUM(UsagePreference, var), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "UsagePreference(%100s): %d [(%d)%s]",
                  name, ETG_CENUM(UsagePreference, var), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "UsagePreference(%100s): %d [(%d)%s]",
                  name, ETG_CENUM(UsagePreference, var), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "UsagePreference(%100s): %d [(%d)%s]",
                  name, ETG_CENUM(UsagePreference, var), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "UsagePreference(%100s): %d", name, ETG_CENUM(UsagePreference, var)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "UsagePreference(%100s): %d", name, ETG_CENUM(UsagePreference, var)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "UsagePreference(%100s): %d", name, ETG_CENUM(UsagePreference, var)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "UsagePreference(%100s): %d", name, ETG_CENUM(UsagePreference, var)));
         }
      }
   }

   void VarTrace(tClassType cls, InbandRingtoneSupport var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "InbandRingtoneSupport(%100s): %d [(%d)%s]",
                  name, ETG_CENUM(InbandRingtoneSupport, var), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "InbandRingtoneSupport(%100s): %d [(%d)%s]",
                  name, ETG_CENUM(InbandRingtoneSupport, var), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "InbandRingtoneSupport(%100s): %d [(%d)%s]",
                  name, ETG_CENUM(InbandRingtoneSupport, var), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "InbandRingtoneSupport(%100s): %d [(%d)%s]",
                  name, ETG_CENUM(InbandRingtoneSupport, var), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "InbandRingtoneSupport(%100s): %d", name, ETG_CENUM(InbandRingtoneSupport, var)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "InbandRingtoneSupport(%100s): %d", name, ETG_CENUM(InbandRingtoneSupport, var)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "InbandRingtoneSupport(%100s): %d", name, ETG_CENUM(InbandRingtoneSupport, var)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "InbandRingtoneSupport(%100s): %d", name, ETG_CENUM(InbandRingtoneSupport, var)));
         }
      }
   }

   void VarTrace(tClassType cls, AbsoluteVolumeSupport var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "AbsoluteVolumeSupport(%100s): %d [(%d)%s]",
                  name, ETG_CENUM(AbsoluteVolumeSupport, var), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "AbsoluteVolumeSupport(%100s): %d [(%d)%s]",
                  name, ETG_CENUM(AbsoluteVolumeSupport, var), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "AbsoluteVolumeSupport(%100s): %d [(%d)%s]",
                  name, ETG_CENUM(AbsoluteVolumeSupport, var), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "AbsoluteVolumeSupport(%100s): %d [(%d)%s]",
                  name, ETG_CENUM(AbsoluteVolumeSupport, var), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "AbsoluteVolumeSupport(%100s): %d", name, ETG_CENUM(AbsoluteVolumeSupport, var)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "AbsoluteVolumeSupport(%100s): %d", name, ETG_CENUM(AbsoluteVolumeSupport, var)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "AbsoluteVolumeSupport(%100s): %d", name, ETG_CENUM(AbsoluteVolumeSupport, var)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "AbsoluteVolumeSupport(%100s): %d", name, ETG_CENUM(AbsoluteVolumeSupport, var)));
         }
      }
   }

   void VarTrace(tClassType cls, PairingStatus var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "PairingStatus(%100s): state = %d, remoteBdAddress = \"%50s\", remoteBdName = \"%50s\", [(%d)%s]",
                  name, ETG_CENUM(PairingState, var._state), var._remoteBdAddress.c_str(), var._remoteBdName.c_str(), line,
                  removePath(file)));
            ETG_TRACE_USR1_DCL((cls, "PairingStatus(%100s): pairingType = %d, pin = \"%50s\", origin = %d, userConfirmationRequired = %d, result = %d",
                  name, ETG_CENUM(PairingType,var._pairingType),var._pin.c_str(), ETG_CENUM(PairingRequestOrigin, var._origin),
                  ETG_CENUM(UserConfirmationRequired, var._userConfirmationRequired), ETG_CENUM(PairingResult, var._result)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "PairingStatus(%100s): state = %d, remoteBdAddress = \"%50s\", remoteBdName = \"%50s\", [(%d)%s]",
                  name, ETG_CENUM(PairingState, var._state), var._remoteBdAddress.c_str(), var._remoteBdName.c_str(), line,
                  removePath(file)));
            ETG_TRACE_USR2_DCL((cls, "PairingStatus(%100s): pairingType = %d, pin = \"%50s\", origin = %d, userConfirmationRequired = %d, result = %d",
                  name, ETG_CENUM(PairingType,var._pairingType),var._pin.c_str(), ETG_CENUM(PairingRequestOrigin, var._origin),
                  ETG_CENUM(UserConfirmationRequired, var._userConfirmationRequired), ETG_CENUM(PairingResult, var._result)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "PairingStatus(%100s): state = %d, remoteBdAddress = \"%50s\", remoteBdName = \"%50s\", [(%d)%s]",
                  name, ETG_CENUM(PairingState, var._state), var._remoteBdAddress.c_str(), var._remoteBdName.c_str(), line,
                  removePath(file)));
            ETG_TRACE_USR3_DCL((cls, "PairingStatus(%100s): pairingType = %d, pin = \"%50s\", origin = %d, userConfirmationRequired = %d, result = %d",
                  name, ETG_CENUM(PairingType,var._pairingType),var._pin.c_str(), ETG_CENUM(PairingRequestOrigin, var._origin),
                  ETG_CENUM(UserConfirmationRequired, var._userConfirmationRequired), ETG_CENUM(PairingResult, var._result)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "PairingStatus(%100s): state = %d, remoteBdAddress = \"%50s\", remoteBdName = \"%50s\", [(%d)%s]",
                  name, ETG_CENUM(PairingState, var._state), var._remoteBdAddress.c_str(), var._remoteBdName.c_str(), line,
                  removePath(file)));
            ETG_TRACE_USR4_DCL((cls, "PairingStatus(%100s): pairingType = %d, pin = \"%50s\", origin = %d, userConfirmationRequired = %d, result = %d",
                  name, ETG_CENUM(PairingType,var._pairingType),var._pin.c_str(), ETG_CENUM(PairingRequestOrigin, var._origin),
                  ETG_CENUM(UserConfirmationRequired, var._userConfirmationRequired), ETG_CENUM(PairingResult, var._result)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "PairingStatus(%100s): state = %d, remoteBdAddress = \"%50s\", remoteBdName = \"%50s\"",
                  name, ETG_CENUM(PairingState, var._state), var._remoteBdAddress.c_str(), var._remoteBdName.c_str()));
            ETG_TRACE_USR1_DCL((cls, "PairingStatus(%100s): pairingType = %d, pin = \"%50s\", origin = %d, userConfirmationRequired = %d, result = %d",
                  name, ETG_CENUM(PairingType,var._pairingType), var._pin.c_str(), ETG_CENUM(PairingRequestOrigin, var._origin),
                  ETG_CENUM(UserConfirmationRequired, var._userConfirmationRequired), ETG_CENUM(PairingResult, var._result)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "PairingStatus(%100s): state = %d, remoteBdAddress = \"%50s\", remoteBdName = \"%50s\"",
                  name, ETG_CENUM(PairingState, var._state), var._remoteBdAddress.c_str(), var._remoteBdName.c_str()));
            ETG_TRACE_USR2_DCL((cls, "PairingStatus(%100s): pairingType = %d, pin = \"%50s\", origin = %d, userConfirmationRequired = %d, result = %d",
                  name, ETG_CENUM(PairingType,var._pairingType), var._pin.c_str(), ETG_CENUM(PairingRequestOrigin, var._origin),
                  ETG_CENUM(UserConfirmationRequired, var._userConfirmationRequired), ETG_CENUM(PairingResult, var._result)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "PairingStatus(%100s): state = %d, remoteBdAddress = \"%50s\", remoteBdName = \"%50s\"",
                  name, ETG_CENUM(PairingState, var._state), var._remoteBdAddress.c_str(), var._remoteBdName.c_str()));
            ETG_TRACE_USR3_DCL((cls, "PairingStatus(%100s): pairingType = %d, pin = \"%50s\", origin = %d, userConfirmationRequired = %d, result = %d",
                  name, ETG_CENUM(PairingType,var._pairingType), var._pin.c_str(), ETG_CENUM(PairingRequestOrigin, var._origin),
                  ETG_CENUM(UserConfirmationRequired, var._userConfirmationRequired), ETG_CENUM(PairingResult, var._result)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "PairingStatus(%100s): state = %d, remoteBdAddress = \"%50s\", remoteBdName = \"%50s\"",
                  name, ETG_CENUM(PairingState, var._state), var._remoteBdAddress.c_str(), var._remoteBdName.c_str()));
            ETG_TRACE_USR4_DCL((cls, "PairingStatus(%100s): pairingType = %d, pin = \"%50s\", origin = %d, userConfirmationRequired = %d, result = %d",
                  name, ETG_CENUM(PairingType,var._pairingType), var._pin.c_str(), ETG_CENUM(PairingRequestOrigin, var._origin),
                  ETG_CENUM(UserConfirmationRequired, var._userConfirmationRequired), ETG_CENUM(PairingResult, var._result)));
         }
      }
   }

   void VarTrace(tClassType cls, PairingPin var,const char* name, const unsigned char level, const char *file, const int line)
    {
       if(file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "PairingPin(%100s): pin = \"%50s\" [(%d)%s]", name, var._pin.c_str(), line, removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "PairingPin(%100s): pin = \"%50s\" [(%d)%s]", name, var._pin.c_str(), line, removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "PairingPin(%100s): pin = \"%50s\" [(%d)%s]", name, var._pin.c_str(), line, removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "PairingPin(%100s): pin = \"%50s\" [(%d)%s]", name, var._pin.c_str(), line, removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "PairingPin(%100s): pin = \"%50s\"", name, var._pin.c_str()));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "PairingPin(%100s): pin = \"%50s\"", name, var._pin.c_str()));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "PairingPin(%100s): pin = \"%50s\"", name, var._pin.c_str()));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "PairingPin(%100s): pin = \"%50s\"", name, var._pin.c_str()));
          }
       }
    }

   //!Fix for CID 29997: Big parameter passed by value (PASS_BY_VALUE)
   void VarTrace(tClassType cls, const DeviceBaseInfo& var, const char* name, const unsigned char level, const char* file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "DeviceBaseInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\" [(%d)%s]",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str(), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "DeviceBaseInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\" [(%d)%s]",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str(), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "DeviceBaseInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\" [(%d)%s]",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str(), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "DeviceBaseInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\" [(%d)%s]",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str(), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "DeviceBaseInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\"",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str()));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "DeviceBaseInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\"",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str()));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "DeviceBaseInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\"",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str()));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "DeviceBaseInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\"",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str()));
         }
      }

      std::string elementName(name);

      VarTrace(cls, var._majorDeviceClass, (elementName + ".BTSMajorDeviceClass").c_str(), level);
      VarTrace(cls, var._deviceIdentification, (elementName + ".DeviceIdentification").c_str(), level);
      VarTrace(cls, var._favoriteIndex, (elementName + ".FavoriteIndex").c_str(), level);
      VarTrace(cls, var._remoteConnectable, (elementName + "._remoteConnectable").c_str(), level);
      VarTrace(cls, var._usagePreference, (elementName + "._usagePreference").c_str(), level);
      VarTrace(cls, var._protocolInfo, (elementName + ".ProtocolInfoMap").c_str(), level);
      VarTrace(cls, var._inbandRingtoneSupport, (elementName + ".InbandRingtoneSupport").c_str(), level);
      VarTrace(cls, var._absoluteVolumeSupport, (elementName + ".AbsoluteVolumeSupport").c_str(), level);
      VarTrace(cls, var._connectionStatus, (elementName + ".ConnectionStatus").c_str(), level);
      VarTrace(cls, var._disconnectedReason, (elementName + ".DisconnectedReason").c_str(), level);
      VarTrace(cls, var._connectionOrderInfo, (elementName + ".ConnectionOrderInfo").c_str(), level);
      VarTrace(cls, var._pairingOrderInfo, (elementName + ".PairingOrderInfo").c_str(), level);
      VarTrace(cls, var._iAPoverBTSupported, (elementName + "._iAPoverBTSupported").c_str(), level);
   }

   void VarTrace(tClassType cls, DeviceBaseInfoList var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceBaseInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceBaseInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceBaseInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceBaseInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceBaseInfoList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceBaseInfoList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceBaseInfoList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceBaseInfoList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceBaseInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceBaseInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceBaseInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceBaseInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceBaseInfoList(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceBaseInfoList(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceBaseInfoList(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceBaseInfoList(%100s):", name));
            }
         }

         DeviceBaseInfoList::iterator it;
         char elementName[MAXNAME];
         unsigned int i(0u);

         for (it = var.begin(); it != var.end(); ++it, ++i)
         {
            snprintf(elementName, MAXNAME - 1, "%s[%d]", name, i);
            VarTrace(cls, *it, elementName, level);
         }
      }
   }

   void VarTrace(tClassType cls, PairedDeviceList var, const char* name, const unsigned char level, const char *file, const int line)
   {

      if (true == var._deviceBaseInfoList.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "PairedDeviceList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "PairedDeviceList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "PairedDeviceList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "PairedDeviceList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "PairedDeviceList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "PairedDeviceList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "PairedDeviceList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "PairedDeviceList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "PairedDeviceList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "PairedDeviceList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "PairedDeviceList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "PairedDeviceList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "PairedDeviceList(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "PairedDeviceList(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "PairedDeviceList(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "PairedDeviceList(%100s):", name));
            }
         }

         std::string elementName(name);
         VarTrace(cls, var._deviceBaseInfoList, (elementName + "._deviceBaseInfoList").c_str(), level);
      }
   }

   void VarTrace(tClassType cls, const DiscoveredDeviceInfo& var, const char* name, const unsigned char level, const char* file, const int line)
   {
      char lastDiscoveredTimestampStr[MAXNAME];
      snprintf(lastDiscoveredTimestampStr, MAXNAME - 1, "%llu", var._lastDiscoveredTimestamp);

      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "DiscoveredDeviceInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\" [(%d)%s]",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str(), line, removePath(file)));
            ETG_TRACE_USR1_DCL((cls, "DiscoveredDeviceInfo(%100s): majorDeviceClass = %d, minorDeviceClass = %d, connectionStatus = %d, lastDiscoveredTimestamp = %50s [(%d)%s]",
                  name, ETG_CENUM(BTSMajorDeviceClass, var._majorDeviceClass),
                  ETG_CENUM(BTSMinorDeviceClass, var._minorDeviceClass),
                  ETG_CENUM(ConnectionStatus, var._connectionStatus), lastDiscoveredTimestampStr, line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "DiscoveredDeviceInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\" [(%d)%s]",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str(), line, removePath(file)));
            ETG_TRACE_USR2_DCL((cls, "DiscoveredDeviceInfo(%100s): majorDeviceClass = %d, minorDeviceClass = %d, connectionStatus = %d, lastDiscoveredTimestamp = %50s [(%d)%s]",
                  name, ETG_CENUM(BTSMajorDeviceClass, var._majorDeviceClass),
                  ETG_CENUM(BTSMinorDeviceClass, var._minorDeviceClass),
                  ETG_CENUM(ConnectionStatus, var._connectionStatus), lastDiscoveredTimestampStr, line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "DiscoveredDeviceInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\" [(%d)%s]",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str(), line, removePath(file)));
            ETG_TRACE_USR3_DCL((cls, "DiscoveredDeviceInfo(%100s): majorDeviceClass = %d, minorDeviceClass = %d, connectionStatus = %d, lastDiscoveredTimestamp = %50s [(%d)%s]",
                  name, ETG_CENUM(BTSMajorDeviceClass, var._majorDeviceClass),
                  ETG_CENUM(BTSMinorDeviceClass, var._minorDeviceClass),
                  ETG_CENUM(ConnectionStatus, var._connectionStatus), lastDiscoveredTimestampStr, line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "DiscoveredDeviceInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\" [(%d)%s]",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str(), line, removePath(file)));
            ETG_TRACE_USR4_DCL((cls, "DiscoveredDeviceInfo(%100s): majorDeviceClass = %d, minorDeviceClass = %d, connectionStatus = %d, lastDiscoveredTimestamp = %50s [(%d)%s]",
                  name, ETG_CENUM(BTSMajorDeviceClass, var._majorDeviceClass),
                  ETG_CENUM(BTSMinorDeviceClass, var._minorDeviceClass),
                  ETG_CENUM(ConnectionStatus, var._connectionStatus), lastDiscoveredTimestampStr, line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "DiscoveredDeviceInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\"",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str()));
            ETG_TRACE_USR1_DCL((cls, "DiscoveredDeviceInfo(%100s): majorDeviceClass = %d, minorDeviceClass = %d, connectionStatus = %d, lastDiscoveredTimestamp = %50s",
                  name, ETG_CENUM(BTSMajorDeviceClass, var._majorDeviceClass),
                  ETG_CENUM(BTSMinorDeviceClass, var._minorDeviceClass),
                  ETG_CENUM(ConnectionStatus, var._connectionStatus), lastDiscoveredTimestampStr));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "DiscoveredDeviceInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\"",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str()));
            ETG_TRACE_USR2_DCL((cls, "DiscoveredDeviceInfo(%100s): majorDeviceClass = %d, minorDeviceClass = %d, connectionStatus = %d, lastDiscoveredTimestamp = %50s",
                  name, ETG_CENUM(BTSMajorDeviceClass, var._majorDeviceClass),
                  ETG_CENUM(BTSMinorDeviceClass, var._minorDeviceClass),
                  ETG_CENUM(ConnectionStatus, var._connectionStatus), lastDiscoveredTimestampStr));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "DiscoveredDeviceInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\"",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str()));
            ETG_TRACE_USR3_DCL((cls, "DiscoveredDeviceInfo(%100s): majorDeviceClass = %d, minorDeviceClass = %d, connectionStatus = %d, lastDiscoveredTimestamp = %50s",
                  name, ETG_CENUM(BTSMajorDeviceClass, var._majorDeviceClass),
                  ETG_CENUM(BTSMinorDeviceClass, var._minorDeviceClass),
                  ETG_CENUM(ConnectionStatus, var._connectionStatus), lastDiscoveredTimestampStr));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "DiscoveredDeviceInfo(%100s): deviceHandle = %d, bdAddress = \"%50s\", bdName = \"%50s\"",
                  name, var._deviceHandle, var._bdAddress.c_str(), var._bdName.c_str()));
            ETG_TRACE_USR4_DCL((cls, "DiscoveredDeviceInfo(%100s): majorDeviceClass = %d, minorDeviceClass = %d, connectionStatus = %d, lastDiscoveredTimestamp = %50s",
                  name, ETG_CENUM(BTSMajorDeviceClass, var._majorDeviceClass),
                  ETG_CENUM(BTSMinorDeviceClass, var._minorDeviceClass),
                  ETG_CENUM(ConnectionStatus, var._connectionStatus), lastDiscoveredTimestampStr));
         }
      }

      std::string elementName(name);
      VarTrace(cls, var._protocolInfo, (elementName + ".ProtocolInfoMap").c_str(), level);
   }

   void VarTrace(tClassType cls, DiscoveredDeviceInfoList var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DiscoveredDeviceInfoList(%100s): <empty> [(%d)%s]",
                     name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DiscoveredDeviceInfoList(%100s): <empty> [(%d)%s]",
                     name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DiscoveredDeviceInfoList(%100s): <empty> [(%d)%s]",
                     name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DiscoveredDeviceInfoList(%100s): <empty> [(%d)%s]",
                     name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DiscoveredDeviceInfoList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DiscoveredDeviceInfoList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DiscoveredDeviceInfoList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DiscoveredDeviceInfoList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DiscoveredDeviceInfoList(%100s): [(%d)%s]",
                     name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DiscoveredDeviceInfoList(%100s): [(%d)%s]",
                     name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DiscoveredDeviceInfoList(%100s): [(%d)%s]",
                     name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DiscoveredDeviceInfoList(%100s): [(%d)%s]",
                     name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DiscoveredDeviceInfoList(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DiscoveredDeviceInfoList(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DiscoveredDeviceInfoList(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DiscoveredDeviceInfoList(%100s):", name));
            }
         }

         DiscoveredDeviceInfoList::iterator it;
         char elementName[MAXNAME];
         unsigned int i(0u);

         for (it = var.begin(); it != var.end(); ++it, ++i)
         {
            snprintf(elementName, MAXNAME - 1, "%s[%d]", name, i);
            VarTrace(cls, *it, elementName, level);
         }
      }
   }

   void VarTrace(tClassType cls, DiscoveredDeviceList var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var._discoveredDeviceInfoList.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DiscoveredDeviceList(%100s): <empty> [(%d)%s]",
                     name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DiscoveredDeviceList(%100s): <empty> [(%d)%s]",
                     name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DiscoveredDeviceList(%100s): <empty> [(%d)%s]",
                     name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DiscoveredDeviceList(%100s): <empty> [(%d)%s]",
                     name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DiscoveredDeviceList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DiscoveredDeviceList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DiscoveredDeviceList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DiscoveredDeviceList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DiscoveredDeviceList(%100s): [(%d)%s]",
                     name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DiscoveredDeviceList(%100s): [(%d)%s]",
                     name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DiscoveredDeviceList(%100s): [(%d)%s]",
                     name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DiscoveredDeviceList(%100s): [(%d)%s]",
                     name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DiscoveredDeviceList(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DiscoveredDeviceList(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DiscoveredDeviceList(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceBaseInfoList(%100s):", name));
            }
         }

         std::string elementName(name);
         VarTrace(cls, var._discoveredDeviceInfoList, (elementName + "._discoveredDeviceInfoList").c_str(), level);
      }
   }

   void VarTrace(tClassType cls, DeviceIdentification var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "DeviceIdentification(%100s): specification ID = 0x%04X, vendor ID = 0x%04X, vendor ID source = 0x%04X, product ID = 0x%04X, version = 0x%04X [(%d)%s]",
                  name, var._specificationId, var._vendorId, var._vendorIdSource, var._productId, var._version, line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "DeviceIdentification(%100s): specification ID = 0x%04X, vendor ID = 0x%04X, vendor ID source = 0x%04X, product ID = 0x%04X, version = 0x%04X [(%d)%s]",
                  name, var._specificationId, var._vendorId, var._vendorIdSource, var._productId, var._version, line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "DeviceIdentification(%100s): specification ID = 0x%04X, vendor ID = 0x%04X, vendor ID source = 0x%04X, product ID = 0x%04X, version = 0x%04X [(%d)%s]",
                  name, var._specificationId, var._vendorId, var._vendorIdSource, var._productId, var._version, line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "DeviceIdentification(%100s): specification ID = 0x%04X, vendor ID = 0x%04X, vendor ID source = 0x%04X, product ID = 0x%04X, version = 0x%04X [(%d)%s]",
                  name, var._specificationId, var._vendorId, var._vendorIdSource, var._productId, var._version, line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "DeviceIdentification(%100s): specification ID = 0x%04X, vendor ID = 0x%04X, vendor ID source = 0x%04X, product ID = 0x%04X, version = 0x%04X",
                  name, var._specificationId, var._vendorId, var._vendorIdSource, var._productId, var._version));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "DeviceIdentification(%100s): specification ID = 0x%04X, vendor ID = 0x%04X, vendor ID source = 0x%04X, product ID = 0x%04X, version = 0x%04X",
                  name, var._specificationId, var._vendorId, var._vendorIdSource, var._productId, var._version));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "DeviceIdentification(%100s): specification ID = 0x%04X, vendor ID = 0x%04X, vendor ID source = 0x%04X, product ID = 0x%04X, version = 0x%04X",
                  name, var._specificationId, var._vendorId, var._vendorIdSource, var._productId, var._version));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "DeviceIdentification(%100s): specification ID = 0x%04X, vendor ID = 0x%04X, vendor ID source = 0x%04X, product ID = 0x%04X, version = 0x%04X",
                  name, var._specificationId, var._vendorId, var._vendorIdSource, var._productId, var._version));
         }
      }
   }

   void VarTrace(tClassType cls, ConnectionOrderInfo var,const char* name, const unsigned char level, const char *file, const int line)
   {
      char secondsStr[MAXNAME];
      snprintf(secondsStr, MAXNAME - 1, "%llu", var._seconds);

      char nanosecondsStr[MAXNAME];
      snprintf(nanosecondsStr, MAXNAME - 1, "%ld", var._nanoseconds);

      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "ConnectionOrderInfo(%100s): index = %u, seconds = %25s, nanoseconds = %25s [(%d)%s]",
                  name, var._index, secondsStr, nanosecondsStr, line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "ConnectionOrderInfo(%100s): index = %u, seconds = %25s, nanoseconds = %25s [(%d)%s]",
                  name, var._index, secondsStr, nanosecondsStr, line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "ConnectionOrderInfo(%100s): index = %u, seconds = %25s, nanoseconds = %25s [(%d)%s]",
                  name, var._index, secondsStr, nanosecondsStr, line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "ConnectionOrderInfo(%100s): index = %u, seconds = %25s, nanoseconds = %25s [(%d)%s]",
                  name, var._index, secondsStr, nanosecondsStr, line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "ConnectionOrderInfo(%100s): index = %u, seconds = %25s, nanoseconds = %25s",
                  name, var._index, secondsStr, nanosecondsStr));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "ConnectionOrderInfo(%100s): index = %u, seconds = %25s, nanoseconds = %25s",
                  name, var._index, secondsStr, nanosecondsStr));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "ConnectionOrderInfo(%100s): index = %u, seconds = %25s, nanoseconds = %25s",
                  name, var._index, secondsStr, nanosecondsStr));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "ConnectionOrderInfo(%100s): index = %u, seconds = %25s, nanoseconds = %25s",
                  name, var._index, secondsStr, nanosecondsStr));
         }
      }
   }

   void VarTrace(tClassType cls, BTSMajorDeviceClass var,const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BTSMajorDeviceClass(%100s): Major Device Class = %d [(%d)%s]",
                  name, ETG_CENUM(BTSMajorDeviceClass, var), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BTSMajorDeviceClass(%100s): Major Device Class = %d [(%d)%s]",
                  name, ETG_CENUM(BTSMajorDeviceClass, var), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BTSMajorDeviceClass(%100s): Major Device Class = %d [(%d)%s]",
                  name, ETG_CENUM(BTSMajorDeviceClass, var), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BTSMajorDeviceClass(%100s): Major Device Class = %d [(%d)%s]",
                  name, ETG_CENUM(BTSMajorDeviceClass, var), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BTSMajorDeviceClass(%100s): Major Device Class = %d",
                  name, ETG_CENUM(BTSMajorDeviceClass, var)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BTSMajorDeviceClass(%100s): Major Device Class = %d",
                  name, ETG_CENUM(BTSMajorDeviceClass, var)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BTSMajorDeviceClass(%100s): Major Device Class = %d",
                  name, ETG_CENUM(BTSMajorDeviceClass, var)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BTSMajorDeviceClass(%100s): Major Device Class = %d",
                  name, ETG_CENUM(BTSMajorDeviceClass, var)));
         }
      }
   }

   void VarTrace(tClassType cls, ConnectionStatus var,const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "ConnectionStatus(%100s): connection status = %d [(%d)%s]",
                  name, ETG_CENUM(ConnectionStatus, var), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "ConnectionStatus(%100s): connection status = %d [(%d)%s]",
                  name, ETG_CENUM(ConnectionStatus, var), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "ConnectionStatus(%100s): connection status = %d [(%d)%s]",
                  name, ETG_CENUM(ConnectionStatus, var), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "ConnectionStatus(%100s): connection status = %d [(%d)%s]",
                  name, ETG_CENUM(ConnectionStatus, var), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "ConnectionStatus(%100s): connection status = %d",
                  name, ETG_CENUM(ConnectionStatus, var)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "ConnectionStatus(%100s): connection status = %d",
                  name, ETG_CENUM(ConnectionStatus, var)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "ConnectionStatus(%100s): connection status = %d",
                  name, ETG_CENUM(ConnectionStatus, var)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "ConnectionStatus(%100s): connection status = %d",
                  name, ETG_CENUM(ConnectionStatus, var)));
         }
      }
   }

   void VarTrace(tClassType cls, DisconnectedReason var,const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "DisconnectedReason(%100s): disconnected reason = %d [(%d)%s]",
                  name, ETG_CENUM(DisconnectedReason, var), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "DisconnectedReason(%100s): disconnected reason = %d [(%d)%s]",
                  name, ETG_CENUM(DisconnectedReason, var), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "DisconnectedReason(%100s): disconnected reason = %d [(%d)%s]",
                  name, ETG_CENUM(DisconnectedReason, var), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "DisconnectedReason(%100s): disconnected reason = %d [(%d)%s]",
                  name, ETG_CENUM(DisconnectedReason, var), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "DisconnectedReason(%100s): disconnected reason = %d",
                  name, ETG_CENUM(DisconnectedReason, var)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "DisconnectedReason(%100s): disconnected reason = %d",
                  name, ETG_CENUM(DisconnectedReason, var)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "DisconnectedReason(%100s): disconnected reason = %d",
                  name, ETG_CENUM(DisconnectedReason, var)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "DisconnectedReason(%100s): disconnected reason = %d",
                  name, ETG_CENUM(DisconnectedReason, var)));
         }
      }
   }

   void VarTrace(tClassType cls, ProtocolConnectionInfo var, const char* name, const unsigned char level, const char* file, const int line)
   {
       if (file != NULL && line != -1)
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "ProtocolConnectionInfo(%150s): connection status = %d, disconnected reason = %d, rfcommDevicePath = \"%50s\", _primary = %u [(%d)%s]",
                   name, ETG_CENUM(ConnectionStatus, var._connectionStatus), ETG_CENUM(DisconnectedReason, var._disconnectedReason),
                   var._rfcommDevicePath.c_str(), var._primary, line, removePath(file)));
          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "ProtocolConnectionInfo(%150s): connection status = %d, disconnected reason = %d, rfcommDevicePath = \"%50s\", _primary = %u [(%d)%s]",
                   name, ETG_CENUM(ConnectionStatus, var._connectionStatus), ETG_CENUM(DisconnectedReason, var._disconnectedReason),
                   var._rfcommDevicePath.c_str(), var._primary, line, removePath(file)));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "ProtocolConnectionInfo(%150s): connection status = %d, disconnected reason = %d, rfcommDevicePath = \"%50s\", _primary = %u [(%d)%s]",
                   name, ETG_CENUM(ConnectionStatus, var._connectionStatus), ETG_CENUM(DisconnectedReason, var._disconnectedReason),
                   var._rfcommDevicePath.c_str(), var._primary, line, removePath(file)));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "ProtocolConnectionInfo(%150s): connection status = %d, disconnected reason = %d, rfcommDevicePath = \"%50s\", _primary = %u [(%d)%s]",
                   name, ETG_CENUM(ConnectionStatus, var._connectionStatus), ETG_CENUM(DisconnectedReason, var._disconnectedReason),
                   var._rfcommDevicePath.c_str(), var._primary, line, removePath(file)));
          }
       }
       else
       {
          if (1u == level)
          {
             ETG_TRACE_USR1_DCL((cls, "ProtocolConnectionInfo(%150s): connection status = %d, disconnected reason = %d, rfcommDevicePath = \"%50s\", _primary = %u",
                   name, ETG_CENUM(ConnectionStatus, var._connectionStatus), ETG_CENUM(DisconnectedReason, var._disconnectedReason),
                   var._rfcommDevicePath.c_str(), var._primary));

          }
          else if (2u == level)
          {
             ETG_TRACE_USR2_DCL((cls, "ProtocolConnectionInfo(%150s): connection status = %d, disconnected reason = %d, rfcommDevicePath = \"%50s\", _primary = %u",
                   name, ETG_CENUM(ConnectionStatus, var._connectionStatus), ETG_CENUM(DisconnectedReason, var._disconnectedReason),
                   var._rfcommDevicePath.c_str(), var._primary));
          }
          else if (3u == level)
          {
             ETG_TRACE_USR3_DCL((cls, "ProtocolConnectionInfo(%150s): connection status = %d, disconnected reason = %d, rfcommDevicePath = \"%50s\", _primary = %u",
                   name, ETG_CENUM(ConnectionStatus, var._connectionStatus), ETG_CENUM(DisconnectedReason, var._disconnectedReason),
                   var._rfcommDevicePath.c_str(), var._primary));
          }
          else
          {
             ETG_TRACE_USR4_DCL((cls, "ProtocolConnectionInfo(%150s): connection status = %d, disconnected reason = %d, rfcommDevicePath = \"%50s\", _primary = %u",
                   name, ETG_CENUM(ConnectionStatus, var._connectionStatus), ETG_CENUM(DisconnectedReason, var._disconnectedReason),
                   var._rfcommDevicePath.c_str(), var._primary));
          }
       }
   }

   void VarTrace(tClassType cls, ProtocolConnectionInfoMap var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolConnectionInfoMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolConnectionInfoMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolConnectionInfoMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolConnectionInfoMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolConnectionInfoMap(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolConnectionInfoMap(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolConnectionInfoMap(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolConnectionInfoMap(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolConnectionInfoMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolConnectionInfoMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolConnectionInfoMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolConnectionInfoMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolConnectionInfoMap(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolConnectionInfoMap(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolConnectionInfoMap(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolConnectionInfoMap(%100s):", name));
            }
         }

         ProtocolConnectionInfoMap::iterator it;
         char keyName[MAXNAME];

         for (it = var.begin(); it != var.end(); it++)
         {
            snprintf(keyName, MAXNAME - 1, "%s[%d]", name, ETG_CENUM(ProtocolId, it->first));
            VarTrace(cls, it->second, keyName, level);
         }
      }
   }

   void VarTrace(tClassType cls, SppConnectionInfoMap var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "SppConnectionInfoMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "SppConnectionInfoMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "SppConnectionInfoMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "SppConnectionInfoMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "SppConnectionInfoMap(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "SppConnectionInfoMap(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "SppConnectionInfoMap(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "SppConnectionInfoMap(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "SppConnectionInfoMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "SppConnectionInfoMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "SppConnectionInfoMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "SppConnectionInfoMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "SppConnectionInfoMap(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "SppConnectionInfoMap(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "SppConnectionInfoMap(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "SppConnectionInfoMap(%100s):", name));
            }
         }

         SppConnectionInfoMap::iterator it;
         char keyName[MAXNAME];

         for (it = var.begin(); it != var.end(); it++)
         {
            snprintf(keyName, MAXNAME - 1, "%s[\"%s\"]", name, it->first.c_str());
            VarTrace(cls, it->second, keyName, level);
         }
      }
   }

   void VarTrace(tClassType cls, ProtocolIdList var,const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolIdList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolIdList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolIdList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolIdList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolIdList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolIdList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolIdList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolIdList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         std::string elements("");
         char element[10];

         for (size_t i = 0u; i < var.size(); i++)
         {
            if (0u < i)
            {
               elements = elements + ", ";
            }

            sprintf(element, "%d", ETG_CENUM(ProtocolId, var[i]));
            elements += element;
         }

         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolIdList(%100s): [%100s] [(%d)%s]", name, elements.c_str(),
                     line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolIdList(%100s): [%100s] [(%d)%s]", name, elements.c_str(),
                     line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolIdList(%100s): [%100s] [(%d)%s]", name, elements.c_str(),
                     line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolIdList(%100s): [%100s] [(%d)%s]", name, elements.c_str(),
                     line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolIdList(%100s): [%100s]", name, elements.c_str()));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolIdList(%100s): [%100s]", name, elements.c_str()));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolIdList(%100s): [%100s]", name, elements.c_str()));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolIdList(%100s): [%100s]", name, elements.c_str()));
            }
         }
      }
   }

   void VarTrace(tClassType cls, DeviceIdList var,const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceIdList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceIdList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceIdList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceIdList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceIdList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceIdList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceIdList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceIdList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         std::string elements("");
         char element[10];

         for (size_t i = 0u; i < var.size(); i++)
         {
            if (0u < i)
            {
               elements = elements + ", ";
            }

            sprintf(element, "%d", var[i]);
            elements += element;
         }

         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceIdList(%100s): [%100s] [(%d)%s]", name, elements.c_str(),
                     line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceIdList(%100s): [%100s] [(%d)%s]", name, elements.c_str(),
                     line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceIdList(%100s): [%100s] [(%d)%s]", name, elements.c_str(),
                     line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceIdList(%100s): [%100s] [(%d)%s]", name, elements.c_str(),
                     line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceIdList(%100s): [%100s]", name, elements.c_str()));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceIdList(%100s): [%100s]", name, elements.c_str()));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceIdList(%100s): [%100s]", name, elements.c_str()));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceIdList(%100s): [%100s]", name, elements.c_str()));
            }
         }
      }
   }

   void VarTrace(tClassType cls, UuidList var,const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "UuidList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "UuidList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "UuidList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "UuidList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "UuidList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "UuidList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "UuidList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "UuidList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         for (size_t i = 0; i < var.size(); i++)
         {
            if (file != NULL && line != -1)
            {
               if (1u == level)
               {
                  ETG_TRACE_USR1_DCL((cls, "UuidList(%100s)[%d]: \"%50s\" [(%d)%s]", name, i, var[i].c_str(),
                        line, removePath(file)));
               }
               else if (2u == level)
               {
                  ETG_TRACE_USR2_DCL((cls, "UuidList(%100s)[%d]: \"%50s\" [(%d)%s]", name, i, var[i].c_str(),
                        line, removePath(file)));
               }
               else if (3u == level)
               {
                  ETG_TRACE_USR3_DCL((cls, "UuidList(%100s)[%d]: \"%50s\" [(%d)%s]", name, i, var[i].c_str(),
                        line, removePath(file)));
               }
               else
               {
                  ETG_TRACE_USR4_DCL((cls, "UuidList(%100s)[%d]: \"%50s\" [(%d)%s]", name, i, var[i].c_str(),
                        line, removePath(file)));
               }
            }
            else
            {
               if (1u == level)
               {
                  ETG_TRACE_USR1_DCL((cls, "UuidList(%100s)[%d]: \"%50s\"", name, i, var[i].c_str()));
               }
               else if (2u == level)
               {
                  ETG_TRACE_USR2_DCL((cls, "UuidList(%100s)[%d]: \"%50s\"", name, i, var[i].c_str()));
               }
               else if (3u == level)
               {
                  ETG_TRACE_USR3_DCL((cls, "UuidList(%100s)[%d]: \"%50s\"", name, i, var[i].c_str()));
               }
               else
               {
                  ETG_TRACE_USR4_DCL((cls, "UuidList(%100s)[%d]: \"%50s\"", name, i, var[i].c_str()));
               }
            }
         }
      }
   }

   void VarTrace(tClassType cls, ProtocolList var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         for (size_t i = 0u; i < var.size(); i++)
         {
            if (file != NULL && line != -1)
            {
               if (1u == level)
               {
                  ETG_TRACE_USR1_DCL((cls, "ProtocolList(%100s)[%d]: protocolId = %d, uuid = \"%50s\" [(%d)%s]", name, i,
                        var[i]._protocolId, var[i]._uuid.c_str(), line, removePath(file)));
               }
               else if (2u == level)
               {
                  ETG_TRACE_USR2_DCL((cls, "ProtocolList(%100s)[%d]: protocolId = %d, uuid = \"%50s\" [(%d)%s]", name, i,
                        var[i]._protocolId, var[i]._uuid.c_str(), line, removePath(file)));
               }
               else if (3u == level)
               {
                  ETG_TRACE_USR3_DCL((cls, "ProtocolList(%100s)[%d]: protocolId = %d, uuid = \"%50s\" [(%d)%s]", name, i,
                        var[i]._protocolId, var[i]._uuid.c_str(), line, removePath(file)));
               }
               else
               {
                  ETG_TRACE_USR4_DCL((cls, "ProtocolList(%100s)[%d]: protocolId = %d, uuid = \"%50s\" [(%d)%s]", name, i,
                        var[i]._protocolId, var[i]._uuid.c_str(), line, removePath(file)));
               }
            }
            else
            {
               if (1u == level)
               {
                  ETG_TRACE_USR1_DCL((cls, "ProtocolList(%100s)[%d]: protocolId = %d, uuid = \"%50s\"", name, i,
                        var[i]._protocolId, var[i]._uuid.c_str()));
               }
               else if (2u == level)
               {
                  ETG_TRACE_USR2_DCL((cls, "ProtocolList(%100s)[%d]: protocolId = %d, uuid = \"%50s\"", name, i,
                        var[i]._protocolId, var[i]._uuid.c_str()));
               }
               else if (3u == level)
               {
                  ETG_TRACE_USR3_DCL((cls, "ProtocolList(%100s)[%d]: protocolId = %d, uuid = \"%50s\"", name, i,
                        var[i]._protocolId, var[i]._uuid.c_str()));
               }
               else
               {
                  ETG_TRACE_USR4_DCL((cls, "ProtocolList(%100s)[%d]: protocolId = %d, uuid = \"%50s\"", name, i,
                        var[i]._protocolId, var[i]._uuid.c_str()));
               }
            }
         }
      }
   }

   void VarTrace(tClassType cls, const DeviceConnectionInfo& var,const char* name, const unsigned char level, const char *file , const int line )
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "DeviceConnectionInfo(%100s): deviceHandle = %d, connectionStatus = %d, disconnectedReason = %d [(%d)%s]",
                  name, var._deviceHandle, ETG_CENUM(ConnectionStatus, var._connectionStatus),
                  ETG_CENUM(DisconnectedReason, var._disconnectedReason), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "DeviceConnectionInfo(%100s): deviceHandle = %d, connectionStatus = %d, disconnectedReason = %d [(%d)%s]",
                  name, var._deviceHandle, ETG_CENUM(ConnectionStatus, var._connectionStatus),
                  ETG_CENUM(DisconnectedReason, var._disconnectedReason), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "DeviceConnectionInfo(%100s): deviceHandle = %d, connectionStatus = %d, disconnectedReason = %d [(%d)%s]",
                  name, var._deviceHandle, ETG_CENUM(ConnectionStatus, var._connectionStatus),
                  ETG_CENUM(DisconnectedReason, var._disconnectedReason), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "DeviceConnectionInfo(%100s): deviceHandle = %d, connectionStatus = %d, disconnectedReason = %d [(%d)%s]",
                  name, var._deviceHandle, ETG_CENUM(ConnectionStatus, var._connectionStatus),
                  ETG_CENUM(DisconnectedReason, var._disconnectedReason), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "DeviceConnectionInfo(%100s): deviceHandle = %d, connectionStatus = %d, disconnectedReason = %d",
                  name, var._deviceHandle, ETG_CENUM(ConnectionStatus, var._connectionStatus),
                  ETG_CENUM(DisconnectedReason, var._disconnectedReason)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "DeviceConnectionInfo(%100s): deviceHandle = %d, connectionStatus = %d, disconnectedReason = %d",
                  name, var._deviceHandle, ETG_CENUM(ConnectionStatus, var._connectionStatus),
                  ETG_CENUM(DisconnectedReason, var._disconnectedReason)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "DeviceConnectionInfo(%100s): deviceHandle = %d, connectionStatus = %d, disconnectedReason = %d",
                  name, var._deviceHandle, ETG_CENUM(ConnectionStatus, var._connectionStatus),
                  ETG_CENUM(DisconnectedReason, var._disconnectedReason)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "DeviceConnectionInfo(%100s): deviceHandle = %d, connectionStatus = %d, disconnectedReason = %d",
                  name, var._deviceHandle, ETG_CENUM(ConnectionStatus, var._connectionStatus),
                  ETG_CENUM(DisconnectedReason, var._disconnectedReason)));
         }
      }

      std::string elementName(name);

      VarTrace(cls, var._connectionOrderInfo, (elementName + "._connectionOrderInfo").c_str(), level);
      VarTrace(cls, var._protocolConnectionInfo, (elementName + "._protocolConnectionInfo").c_str(), level);
      VarTrace(cls, var._sppConnectionInfo, (elementName + "._sppConnectionInfo").c_str(), level);
   }

   void VarTrace(tClassType cls, DeviceConnectionInfoList var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceConnectionInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceConnectionInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceConnectionInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceConnectionInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceConnectionInfoList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceConnectionInfoList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceConnectionInfoList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceConnectionInfoList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceConnectionInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceConnectionInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceConnectionInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceConnectionInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceConnectionInfoList(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceConnectionInfoList(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceConnectionInfoList(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceConnectionInfoList(%100s):", name));
            }
         }

         DeviceConnectionInfoList::iterator it;
         char elementName[MAXNAME];
         unsigned int i(0u);

         for (it = var.begin(); it != var.end(); it++)
         {
            snprintf(elementName, MAXNAME - 1, "%s[%d]", name, i);
            VarTrace(cls, *it, elementName, level);
         }
      }
   }

   void VarTrace(tClassType cls, DeviceConnectionStatusList var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var._deviceConnectionInfoList.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceConnectionStatusList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceConnectionStatusList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceConnectionStatusList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceConnectionStatusList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceConnectionStatusList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceConnectionStatusList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceConnectionStatusList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceConnectionStatusList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceConnectionStatusList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceConnectionStatusList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceConnectionStatusList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceConnectionStatusList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceConnectionStatusList(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceConnectionStatusList(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceConnectionStatusList(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceConnectionStatusList(%100s):", name));
            }
         }

         std::string elementName(name);

         VarTrace(cls, var._deviceConnectionInfoList, (elementName + "._deviceConnectionInfoList").c_str(), level);
      }
   }

   //!Fix for CID 22856: Big parameter passed by value (PASS_BY_VALUE)
   void VarTrace(tClassType cls, const RemoteDeviceInfo& var, const char* name, const unsigned char level, const char* file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "RemoteDeviceInfo(%100s): [(%d)%s]", name, line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "RemoteDeviceInfo(%100s): [(%d)%s]", name, line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "RemoteDeviceInfo(%100s): [(%d)%s]", name, line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "RemoteDeviceInfo(%100s): [(%d)%s]", name, line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "RemoteDeviceInfo(%100s):", name));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "RemoteDeviceInfo(%100s):", name));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "RemoteDeviceInfo(%100s):", name));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "RemoteDeviceInfo(%100s):", name));
         }
      }

      std::string elementName(name);

      VarTrace(cls, var._deviceBaseInfo, (elementName + "._deviceBaseInfo").c_str(), level);
      VarTrace(cls, var._linkKey, (elementName + "._linkKey").c_str(), level);
      VarTrace(cls, var._protocolConnectionInfo, (elementName + "._protocolConnectionInfo").c_str(), level);
      VarTrace(cls, var._sppConnectionInfo, (elementName + "._sppConnectionInfo").c_str(), level);
   }

   void VarTrace(tClassType cls, AutoConnectionType var, const char* name, const unsigned char level, const char* file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "AutoConnectionType(%100s): type = %d [(%d)%s]", name,
                  ETG_CENUM(StandardAutoConnectionType, var._type), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "AutoConnectionType(%100s): type = %d [(%d)%s]", name,
                  ETG_CENUM(StandardAutoConnectionType, var._type), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "AutoConnectionType(%100s): type = %d [(%d)%s]", name,
                  ETG_CENUM(StandardAutoConnectionType, var._type), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "AutoConnectionType(%100s): type = %d [(%d)%s]", name,
                  ETG_CENUM(StandardAutoConnectionType, var._type), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "AutoConnectionType(%100s): type = %d", name,
                  ETG_CENUM(StandardAutoConnectionType, var._type)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "AutoConnectionType(%100s): type = %d", name,
                  ETG_CENUM(StandardAutoConnectionType, var._type)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "AutoConnectionType(%100s): type = %d", name,
                  ETG_CENUM(StandardAutoConnectionType, var._type)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "AutoConnectionType(%100s): type = %d", name,
                  ETG_CENUM(StandardAutoConnectionType, var._type)));
         }
      }
   }

   void VarTrace(tClassType cls, AutoConnectionStatus var, const char* name, const unsigned char level,
         const char* file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "AutoConnectionStatus(%100s): autoConnectionState = %d, deviceIdInProgress = %d [(%d)%s]", name,
                  ETG_CENUM(AutoConnectionState, var._autoConnectionState), var._deviceIdInProgress, line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "AutoConnectionStatus(%100s): autoConnectionState = %d, deviceIdInProgress = %d [(%d)%s]", name,
                  ETG_CENUM(AutoConnectionState, var._autoConnectionState), var._deviceIdInProgress, line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "AutoConnectionStatus(%100s): autoConnectionState = %d, deviceIdInProgress = %d [(%d)%s]", name,
                  ETG_CENUM(AutoConnectionState, var._autoConnectionState), var._deviceIdInProgress, line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "AutoConnectionStatus(%100s): autoConnectionState = %d, deviceIdInProgress = %d [(%d)%s]", name,
                  ETG_CENUM(AutoConnectionState, var._autoConnectionState), var._deviceIdInProgress, line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "AutoConnectionStatus(%100s): autoConnectionState = %d, deviceIdInProgress = %d", name,
                  ETG_CENUM(AutoConnectionState, var._autoConnectionState), var._deviceIdInProgress));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "AutoConnectionStatus(%100s): autoConnectionState = %d, deviceIdInProgress = %d", name,
                  ETG_CENUM(AutoConnectionState, var._autoConnectionState), var._deviceIdInProgress));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "AutoConnectionStatus(%100s): autoConnectionState = %d, deviceIdInProgress = %d", name,
                  ETG_CENUM(AutoConnectionState, var._autoConnectionState), var._deviceIdInProgress));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "AutoConnectionStatus(%100s): autoConnectionState = %d, deviceIdInProgress = %d", name,
                  ETG_CENUM(AutoConnectionState, var._autoConnectionState), var._deviceIdInProgress));
         }
      }
   }

   void VarTrace(tClassType cls, OverallDeviceBlockStatus var, const char* name, const unsigned char level, const char* file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "OverallDeviceBlockStatus(%100s): blockState = %d, bdAddress = \"%50s\" [(%d)%s]",
                  name, ETG_CENUM(OverallBlockState, var._blockState), var._bdAddress.c_str(), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "OverallDeviceBlockStatus(%100s): blockState = %d, bdAddress = \"%50s\" [(%d)%s]",
                  name, ETG_CENUM(OverallBlockState, var._blockState), var._bdAddress.c_str(), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "OverallDeviceBlockStatus(%100s): blockState = %d, bdAddress = \"%50s\" [(%d)%s]",
                  name, ETG_CENUM(OverallBlockState, var._blockState), var._bdAddress.c_str(), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "OverallDeviceBlockStatus(%100s): blockState = %d, bdAddress = \"%50s\" [(%d)%s]",
                  name, ETG_CENUM(OverallBlockState, var._blockState), var._bdAddress.c_str(), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "OverallDeviceBlockStatus(%100s): blockState = %d, bdAddress = \"%50s\"", name,
                  ETG_CENUM(OverallBlockState, var._blockState), var._bdAddress.c_str()));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "OverallDeviceBlockStatus(%100s): blockState = %d, bdAddress = \"%50s\"", name,
                  ETG_CENUM(OverallBlockState, var._blockState), var._bdAddress.c_str()));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "OverallDeviceBlockStatus(%100s): blockState = %d, bdAddress = \"%50s\"", name,
                  ETG_CENUM(OverallBlockState, var._blockState), var._bdAddress.c_str()));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "OverallDeviceBlockStatus(%100s): blockState = %d, bdAddress = \"%50s\"", name,
                  ETG_CENUM(OverallBlockState, var._blockState), var._bdAddress.c_str()));
         }
      }
   }

   void VarTrace(tClassType cls, BlockState var,const char* name, const unsigned char level, const char *file , const int line )
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BlockState(%100s): %d [(%d)%s]", name, ETG_CENUM(BlockState, var),
                  line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BlockState(%100s): %d [(%d)%s]", name, ETG_CENUM(BlockState, var),
                  line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BlockState(%100s): %d [(%d)%s]", name, ETG_CENUM(BlockState, var),
                  line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BlockState(%100s): %d [(%d)%s]", name, ETG_CENUM(BlockState, var),
                  line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BlockState(%100s): %d", name, ETG_CENUM(BlockState, var)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BlockState(%100s): %d", name, ETG_CENUM(BlockState, var)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BlockState(%100s): %d", name, ETG_CENUM(BlockState, var)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BlockState(%100s): %d", name, ETG_CENUM(BlockState, var)));
         }
      }
   }

   void VarTrace(tClassType cls, ProtocolBlockStatusMap var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolBlockStatusMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolBlockStatusMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolBlockStatusMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolBlockStatusMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolBlockStatusMap(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolBlockStatusMap(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolBlockStatusMap(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolBlockStatusMap(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolBlockStatusMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolBlockStatusMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolBlockStatusMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolBlockStatusMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolBlockStatusMap(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolBlockStatusMap(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolBlockStatusMap(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolBlockStatusMap(%100s):", name));
            }
         }

         ProtocolBlockStatusMap::iterator it;

         for (it = var.begin(); it != var.end(); it++)
         {
            if (file != NULL && line != -1)
            {
               if (1u == level)
               {
                  ETG_TRACE_USR1_DCL((cls, "ProtocolBlockStatusMap(%100s)[%d, \"%50s\"]: blockState = %d [(%d)%s]",
                        name, ETG_CENUM(ProtocolId, it->first._protocolId), it->first._uuid.c_str(),
                        ETG_CENUM(BlockState, it->second), line, removePath(file)));
               }
               else if (2u == level)
               {
                  ETG_TRACE_USR2_DCL((cls, "ProtocolBlockStatusMap(%100s)[%d, \"%50s\"]: blockState = %d [(%d)%s]",
                        name, ETG_CENUM(ProtocolId, it->first._protocolId), it->first._uuid.c_str(),
                        ETG_CENUM(BlockState, it->second), line, removePath(file)));
               }
               else if (3u == level)
               {
                  ETG_TRACE_USR3_DCL((cls, "ProtocolBlockStatusMap(%100s)[%d, \"%50s\"]: blockState = %d [(%d)%s]",
                        name, ETG_CENUM(ProtocolId, it->first._protocolId), it->first._uuid.c_str(),
                        ETG_CENUM(BlockState, it->second), line, removePath(file)));
               }
               else
               {
                  ETG_TRACE_USR4_DCL((cls, "ProtocolBlockStatusMap(%100s)[%d, \"%50s\"]: blockState = %d [(%d)%s]",
                        name, ETG_CENUM(ProtocolId, it->first._protocolId), it->first._uuid.c_str(),
                        ETG_CENUM(BlockState, it->second), line, removePath(file)));
               }
            }
            else
            {
               if (1u == level)
               {
                  ETG_TRACE_USR1_DCL((cls, "ProtocolBlockStatusMap(%100s)[%d, \"%50s\"]: blockState = %d",
                        name, ETG_CENUM(ProtocolId, it->first._protocolId), it->first._uuid.c_str(),
                        ETG_CENUM(BlockState, it->second)));
               }
               else if (2u == level)
               {
                  ETG_TRACE_USR2_DCL((cls, "ProtocolBlockStatusMap(%100s)[%d, \"%50s\"]: blockState = %d",
                        name, ETG_CENUM(ProtocolId, it->first._protocolId), it->first._uuid.c_str(),
                        ETG_CENUM(BlockState, it->second)));
               }
               else if (3u == level)
               {
                  ETG_TRACE_USR3_DCL((cls, "ProtocolBlockStatusMap(%100s)[%d, \"%50s\"]: blockState = %d",
                        name, ETG_CENUM(ProtocolId, it->first._protocolId), it->first._uuid.c_str(),
                        ETG_CENUM(BlockState, it->second)));
               }
               else
               {
                  ETG_TRACE_USR4_DCL((cls, "ProtocolBlockStatusMap(%100s)[%d, \"%50s\"]: blockState = %d",
                        name, ETG_CENUM(ProtocolId, it->first._protocolId), it->first._uuid.c_str(),
                        ETG_CENUM(BlockState, it->second)));
               }
            }
         }
      }
   }

   void VarTrace(tClassType cls, DeviceBlockStatus var,const char* name, const unsigned char level, const char *file , const int line )
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "DeviceBlockStatus(%100s): deviceHandle = %d, blockState = %d [(%d)%s]",
                  name, var._deviceHandle, ETG_CENUM(BlockState, var._blockState), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "DeviceBlockStatus(%100s): deviceHandle = %d, blockState = %d [(%d)%s]",
                  name, var._deviceHandle, ETG_CENUM(BlockState, var._blockState), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "DeviceBlockStatus(%100s): deviceHandle = %d, blockState = %d [(%d)%s]",
                  name, var._deviceHandle, ETG_CENUM(BlockState, var._blockState), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "DeviceBlockStatus(%100s): deviceHandle = %d, blockState = %d [(%d)%s]",
                  name, var._deviceHandle, ETG_CENUM(BlockState, var._blockState), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "DeviceBlockStatus(%100s): deviceHandle = %d, blockState = %d",
                  name, var._deviceHandle, ETG_CENUM(BlockState, var._blockState)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "DeviceBlockStatus(%100s): deviceHandle = %d, blockState = %d",
                  name, var._deviceHandle, ETG_CENUM(BlockState, var._blockState)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "DeviceBlockStatus(%100s): deviceHandle = %d, blockState = %d",
                  name, var._deviceHandle, ETG_CENUM(BlockState, var._blockState)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "DeviceBlockStatus(%100s): deviceHandle = %d, blockState = %d",
                  name, var._deviceHandle, ETG_CENUM(BlockState, var._blockState)));
         }
      }

      std::string elementName(name);

      VarTrace(cls, var._protocolBlockStatusMap, (elementName + "._protocolBlockStatusMap").c_str(), level);
   }

   void VarTrace(tClassType cls, DeviceBlockStatusList var, const char* name, const unsigned char level, const char* file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceBlockStatusList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceBlockStatusList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceBlockStatusList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceBlockStatusList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceBlockStatusList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceBlockStatusList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceBlockStatusList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceBlockStatusList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceBlockStatusList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceBlockStatusList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceBlockStatusList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceBlockStatusList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceBlockStatusList(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceBlockStatusList(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceBlockStatusList(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceBlockStatusList(%100s):", name));
            }
         }

         DeviceBlockStatusList::iterator it;
         char elementName[MAXNAME];
         unsigned int i(0u);

         for (it = var.begin(); it != var.end(); it++, i++)
         {
            snprintf(elementName, MAXNAME - 1, "%s[%d]", name, i);
            VarTrace(cls, *it, elementName, level);
         }
      }
   }

   void VarTrace(tClassType cls, BlockStatus var, const char* name, const unsigned char level, const char* file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BlockStatus(%100s): [(%d)%s]", name, line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BlockStatus(%100s): [(%d)%s]", name, line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BlockStatus(%100s): [(%d)%s]", name, line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BlockStatus(%100s): [(%d)%s]", name, line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BlockStatus(%100s):", name));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BlockStatus(%100s):", name));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BlockStatus(%100s):", name));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BlockStatus(%100s):", name));
         }
      }

      std::string elementName(name);

      VarTrace(cls, var._overallDeviceBlockStatus, (elementName + "._overallDeviceBlockStatus").c_str(), level);
      VarTrace(cls, var._deviceBlockStatusList, (elementName + "._deviceBlockStatusList").c_str(), level);
   }

   void VarTrace(tClassType cls, BmConfigData var, const char* name, const unsigned char level, const char* file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BmConfigData(%100s): [(%d)%s]", name, line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BmConfigData(%100s): [(%d)%s]", name, line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BmConfigData(%100s): [(%d)%s]", name, line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BmConfigData(%100s): [(%d)%s]", name, line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BmConfigData(%100s):", name));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BmConfigData(%100s):", name));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BmConfigData(%100s):", name));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BmConfigData(%100s):", name));
         }
      }

      std::string elementName(name);

      VarTrace(cls, var._maxNumPairedDevices, (elementName + "._maxNumPairedDevices").c_str(), level);
   }

   void VarTrace(tClassType cls, BmServiceAvailability var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BmServiceAvailability(%100s): _bmServiceAvailability = %d [(%d)%s]", name, ETG_CENUM(ServiceAvailability, var._serviceAvailability), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BmServiceAvailability(%100s): _bmServiceAvailability = %d [(%d)%s]", name, ETG_CENUM(ServiceAvailability, var._serviceAvailability), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BmServiceAvailability(%100s): _bmServiceAvailability = %d [(%d)%s]", name, ETG_CENUM(ServiceAvailability, var._serviceAvailability), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BmServiceAvailability(%100s): _bmServiceAvailability = %d [(%d)%s]", name, ETG_CENUM(ServiceAvailability, var._serviceAvailability), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BmServiceAvailability(%100s): _bmServiceAvailability = %d", name, ETG_CENUM(ServiceAvailability, var._serviceAvailability)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BmServiceAvailability(%100s): _bmServiceAvailability = %d", name, ETG_CENUM(ServiceAvailability, var._serviceAvailability)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BmServiceAvailability(%100s): _bmServiceAvailability = %d", name, ETG_CENUM(ServiceAvailability, var._serviceAvailability)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BmServiceAvailability(%100s): _bmServiceAvailability = %d", name, ETG_CENUM(ServiceAvailability, var._serviceAvailability)));
         }
      }
   }

   void VarTrace(tClassType cls, BtLimitationModeInfo var,const char* name, const unsigned char level, const char *file , const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BtLimitationModeInfo(%100s): bdAddress = \"%50s\", limitationFeature = %d, limitationCommunicationIf = %d, limitationState = %d [(%d)%s]",
                  name, var._bdAddress.c_str(), ETG_CENUM(LimitationFeature, var._limitationMode._limitationFeature),
                  ETG_CENUM(LimitationCommunicationIf, var._limitationMode._limitationCommunicationIf),
                  ETG_CENUM(LimitationState, var._limitationState), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BtLimitationModeInfo(%100s): bdAddress = \"%50s\", limitationFeature = %d, limitationCommunicationIf = %d, limitationState = %d [(%d)%s]",
                  name, var._bdAddress.c_str(), ETG_CENUM(LimitationFeature, var._limitationMode._limitationFeature),
                  ETG_CENUM(LimitationCommunicationIf, var._limitationMode._limitationCommunicationIf),
                  ETG_CENUM(LimitationState, var._limitationState), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BtLimitationModeInfo(%100s): bdAddress = \"%50s\", limitationFeature = %d, limitationCommunicationIf = %d, limitationState = %d [(%d)%s]",
                  name, var._bdAddress.c_str(), ETG_CENUM(LimitationFeature, var._limitationMode._limitationFeature),
                  ETG_CENUM(LimitationCommunicationIf, var._limitationMode._limitationCommunicationIf),
                  ETG_CENUM(LimitationState, var._limitationState), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BtLimitationModeInfo(%100s): bdAddress = \"%50s\", limitationFeature = %d, limitationCommunicationIf = %d, limitationState = %d [(%d)%s]",
                  name, var._bdAddress.c_str(), ETG_CENUM(LimitationFeature, var._limitationMode._limitationFeature),
                  ETG_CENUM(LimitationCommunicationIf, var._limitationMode._limitationCommunicationIf),
                  ETG_CENUM(LimitationState, var._limitationState), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "BtLimitationModeInfo(%100s): bdAddress = \"%50s\", limitationFeature = %d, limitationCommunicationIf = %d, limitationState = %d",
                  name, var._bdAddress.c_str(), ETG_CENUM(LimitationFeature, var._limitationMode._limitationFeature),
                  ETG_CENUM(LimitationCommunicationIf, var._limitationMode._limitationCommunicationIf),
                  ETG_CENUM(LimitationState, var._limitationState)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "BtLimitationModeInfo(%100s): bdAddress = \"%50s\", limitationFeature = %d, limitationCommunicationIf = %d, limitationState = %d",
                  name, var._bdAddress.c_str(), ETG_CENUM(LimitationFeature, var._limitationMode._limitationFeature),
                  ETG_CENUM(LimitationCommunicationIf, var._limitationMode._limitationCommunicationIf),
                  ETG_CENUM(LimitationState, var._limitationState)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "BtLimitationModeInfo(%100s): bdAddress = \"%50s\", limitationFeature = %d, limitationCommunicationIf = %d, limitationState = %d",
                  name, var._bdAddress.c_str(), ETG_CENUM(LimitationFeature, var._limitationMode._limitationFeature),
                  ETG_CENUM(LimitationCommunicationIf, var._limitationMode._limitationCommunicationIf),
                  ETG_CENUM(LimitationState, var._limitationState)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "BtLimitationModeInfo(%100s): bdAddress = \"%50s\", limitationFeature = %d, limitationCommunicationIf = %d, limitationState = %d",
                  name, var._bdAddress.c_str(), ETG_CENUM(LimitationFeature, var._limitationMode._limitationFeature),
                  ETG_CENUM(LimitationCommunicationIf, var._limitationMode._limitationCommunicationIf),
                  ETG_CENUM(LimitationState, var._limitationState)));
         }
      }
   }

   void VarTrace(tClassType cls, BtLimitationModeInfoList var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "BtLimitationModeInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "BtLimitationModeInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "BtLimitationModeInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "BtLimitationModeInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "BtLimitationModeInfoList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "BtLimitationModeInfoList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "BtLimitationModeInfoList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "BtLimitationModeInfoList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "BtLimitationModeInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "BtLimitationModeInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "BtLimitationModeInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "BtLimitationModeInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "BtLimitationModeInfoList(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "BtLimitationModeInfoList(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "BtLimitationModeInfoList(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "BtLimitationModeInfoList(%100s):", name));
            }
         }

         BtLimitationModeInfoList::iterator it;
         char elementName[MAXNAME];
         unsigned int i(0u);

         for (it = var.begin(); it != var.end(); it++)
         {
            snprintf(elementName, MAXNAME - 1, "%s[%d]", name, i);
            VarTrace(cls, *it, elementName, level);
         }
      }
   }

   void VarTrace(tClassType cls, BtLimitationMode var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var._btLimitationModeInfoList.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "BtLimitationMode(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "BtLimitationMode(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "BtLimitationMode(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "BtLimitationMode(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "BtLimitationMode(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "BtLimitationMode(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "BtLimitationMode(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "BtLimitationMode(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "BtLimitationMode(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "BtLimitationMode(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "BtLimitationMode(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "BtLimitationMode(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "BtLimitationMode(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "BtLimitationMode(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "BtLimitationMode(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "BtLimitationMode(%100s):", name));
            }
         }

         std::string elementName(name);

         VarTrace(cls, var._btLimitationModeInfoList, (elementName + "._btLimitationModeInfoList").c_str(), level);
      }
   }

   void VarTrace(tClassType cls, ProfileUsageInfo var,const char* name, const unsigned char level, const char *file , const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "ProfileUsageInfo(%100s): DeviceID = %d, protocolID = %d, UUid = \"%50s\" [(%d)%s]",
                  name, var._deviceHandle, ETG_CENUM(ProtocolId, var._protocol._protocolId) , var._protocol._uuid.c_str(), line, removePath(file)));

            ETG_TRACE_USR1_DCL((cls, "ProfileUsageInfo(%100s): BTProfileUsageType = %d, InUseFlag = %d [(%d)%s]",
                  name, ETG_CENUM(BTProfileUsageType, var._ProfileUsageType), var._InUse, line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "ProfileUsageInfo(%100s): DeviceID = %d, protocolID = %d, UUid = \"%50s\" [(%d)%s]",
                  name, var._deviceHandle, ETG_CENUM(ProtocolId, var._protocol._protocolId) , var._protocol._uuid.c_str(), line, removePath(file)));

            ETG_TRACE_USR2_DCL((cls, "ProfileUsageInfo(%100s): BTProfileUsageType = %d, InUseFlag = %d [(%d)%s]",
                  name, ETG_CENUM(BTProfileUsageType, var._ProfileUsageType), var._InUse, line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "ProfileUsageInfo(%100s): DeviceID = %d, protocolID = %d, UUid = \"%50s\" [(%d)%s]",
                  name, var._deviceHandle, ETG_CENUM(ProtocolId, var._protocol._protocolId) , var._protocol._uuid.c_str(), line, removePath(file)));

            ETG_TRACE_USR3_DCL((cls, "ProfileUsageInfo(%100s): BTProfileUsageType = %d, InUseFlag = %d [(%d)%s]",
                  name, ETG_CENUM(BTProfileUsageType, var._ProfileUsageType), var._InUse, line, removePath(file)));
         }
         else
         {
        	ETG_TRACE_USR4_DCL((cls, "ProfileUsageInfo(%100s): DeviceID = %d, protocolID = %d, UUid = \"%50s\" [(%d)%s]",
                  name, var._deviceHandle, ETG_CENUM(ProtocolId, var._protocol._protocolId) , var._protocol._uuid.c_str(), line, removePath(file)));

            ETG_TRACE_USR4_DCL((cls, "ProfileUsageInfo(%100s): BTProfileUsageType = %d, InUseFlag = %d [(%d)%s]",
                  name, ETG_CENUM(BTProfileUsageType, var._ProfileUsageType), var._InUse, line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "ProfileUsageInfo(%100s): DeviceID = %d, protocolID = %d, UUid = \"%50s\"", name,
            		var._deviceHandle, ETG_CENUM(ProtocolId, var._protocol._protocolId) , var._protocol._uuid.c_str()));

            ETG_TRACE_USR1_DCL((cls, "ProfileUsageInfo(%100s): BTProfileUsageType = %d, InUseFlag = %d", name,
            		ETG_CENUM(BTProfileUsageType, var._ProfileUsageType), var._InUse));
         }
         else if (2u == level)
         {
        	ETG_TRACE_USR2_DCL((cls, "ProfileUsageInfo(%100s): DeviceID = %d, protocolID = %d, UUid = \"%50s\"", name,
            		var._deviceHandle, ETG_CENUM(ProtocolId, var._protocol._protocolId) , var._protocol._uuid.c_str()));

            ETG_TRACE_USR2_DCL((cls, "ProfileUsageInfo(%100s): BTProfileUsageType = %d, InUseFlag = %d", name,
            		ETG_CENUM(BTProfileUsageType, var._ProfileUsageType), var._InUse));
         }
         else if (3u == level)
         {
        	ETG_TRACE_USR3_DCL((cls, "ProfileUsageInfo(%100s): DeviceID = %d, protocolID = %d, UUid = \"%50s\"", name,
            		var._deviceHandle, ETG_CENUM(ProtocolId, var._protocol._protocolId) , var._protocol._uuid.c_str()));

            ETG_TRACE_USR3_DCL((cls, "ProfileUsageInfo(%100s): BTProfileUsageType = %d, InUseFlag = %d", name,
            		ETG_CENUM(BTProfileUsageType, var._ProfileUsageType), var._InUse));
         }
         else
         {
        	ETG_TRACE_USR4_DCL((cls, "ProfileUsageInfo(%100s): DeviceID = %d, protocolID = %d, UUid = \"%50s\"", name,
            		var._deviceHandle, ETG_CENUM(ProtocolId, var._protocol._protocolId) , var._protocol._uuid.c_str()));

            ETG_TRACE_USR4_DCL((cls, "ProfileUsageInfo(%100s): BTProfileUsageType = %d, InUseFlag = %d", name,
            		ETG_CENUM(BTProfileUsageType, var._ProfileUsageType), var._InUse));
         }
      }
   }

   void VarTrace(tClassType cls, ProfileUsageInfoList var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProfileUsageInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProfileUsageInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProfileUsageInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProfileUsageInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProfileUsageInfoList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProfileUsageInfoList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProfileUsageInfoList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProfileUsageInfoList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProfileUsageInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProfileUsageInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProfileUsageInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProfileUsageInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProfileUsageInfoList(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProfileUsageInfoList(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProfileUsageInfoList(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProfileUsageInfoList(%100s):", name));
            }
         }

         ProfileUsageInfoListItr it;
         char elementName[MAXNAME];
         unsigned int i(0u);

         for (it = var.begin(); it != var.end(); it++)
         {
            snprintf(elementName, MAXNAME - 1, "%s[%d]", name, i);
            VarTrace(cls, *it, elementName, level);
         }
      }
   }

   void VarTrace(tClassType cls, ProfileUsageList var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var._btProfileUsageInfoList.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProfileUsageList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProfileUsageList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProfileUsageList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProfileUsageList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProfileUsageList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProfileUsageList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProfileUsageList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProfileUsageList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProfileUsageList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProfileUsageList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProfileUsageList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProfileUsageList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProfileUsageList(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProfileUsageList(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProfileUsageList(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProfileUsageList(%100s):", name));
            }
         }

         std::string elementName(name);

         VarTrace(cls, var._btProfileUsageInfoList, (elementName + "._btProfileUsageInfoList").c_str(), level);
      }
   }

   void VarTrace(tClassType cls, IssueInfo var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "IssueInfo(%100s): conflictType = %d, deviceID = %d [(%d)%s]", name,
            		ETG_CENUM(ConflictType, var._conflictType), var._deviceID, line, removePath(file)));

            ETG_TRACE_USR1_DCL((cls, "IssueInfo(%100s): ProtocolID =  %d, UUID = \"%50s\", "
            		"ProfileUsageType = %d, ConflictState = %d [(%d)%s]", name, ETG_CENUM(ProtocolId, var._protocol._protocolId),
            		var._protocol._uuid.c_str(), ETG_CENUM(BTProfileUsageType, var._profileUsageType),
            		ETG_CENUM(ConflictState, var._conflictState), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "IssueInfo(%100s): conflictType = %d, deviceID = %d [(%d)%s]", name,
            		ETG_CENUM(ConflictType, var._conflictType), var._deviceID, line, removePath(file)));

            ETG_TRACE_USR2_DCL((cls, "IssueInfo(%100s): ProtocolID =  %d, UUID = \"%50s\", "
            		"ProfileUsageType = %d, ConflictState = %d [(%d)%s]", name, ETG_CENUM(ProtocolId, var._protocol._protocolId),
            		var._protocol._uuid.c_str(), ETG_CENUM(BTProfileUsageType, var._profileUsageType),
            		ETG_CENUM(ConflictState, var._conflictState), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "IssueInfo(%100s): conflictType = %d, deviceID = %d [(%d)%s]", name,
            		ETG_CENUM(ConflictType, var._conflictType), var._deviceID, line, removePath(file)));

            ETG_TRACE_USR3_DCL((cls, "IssueInfo(%100s): ProtocolID =  %d, UUID = \"%50s\", "
            		"ProfileUsageType = %d, ConflictState = %d [(%d)%s]", name, ETG_CENUM(ProtocolId, var._protocol._protocolId),
            		var._protocol._uuid.c_str(), ETG_CENUM(BTProfileUsageType, var._profileUsageType),
            		ETG_CENUM(ConflictState, var._conflictState), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "IssueInfo(%100s): conflictType = %d, deviceID = %d [(%d)%s]", name,
            		ETG_CENUM(ConflictType, var._conflictType), var._deviceID, line, removePath(file)));

            ETG_TRACE_USR4_DCL((cls, "IssueInfo(%100s): ProtocolID =  %d, UUID = \"%50s\", "
            		"ProfileUsageType = %d, ConflictState = %d [(%d)%s]", name, ETG_CENUM(ProtocolId, var._protocol._protocolId),
            		var._protocol._uuid.c_str(), ETG_CENUM(BTProfileUsageType, var._profileUsageType),
            		ETG_CENUM(ConflictState, var._conflictState), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "IssueInfo(%100s): conflictType = %d, deviceID = %d ", name,
            		ETG_CENUM(ConflictType, var._conflictType), var._deviceID));

            ETG_TRACE_USR1_DCL((cls, "IssueInfo(%100s): ProtocolID =  %d, UUID = \"%50s\", "
            		"ProfileUsageType = %d, ConflictState = %d", name, ETG_CENUM(ProtocolId, var._protocol._protocolId),
            		var._protocol._uuid.c_str(), ETG_CENUM(BTProfileUsageType, var._profileUsageType),
            		ETG_CENUM(ConflictState, var._conflictState)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "IssueInfo(%100s): conflictType = %d, deviceID = %d ", name,
            		ETG_CENUM(ConflictType, var._conflictType), var._deviceID));

            ETG_TRACE_USR2_DCL((cls, "IssueInfo(%100s): ProtocolID =  %d, UUID = \"%50s\", "
            		"ProfileUsageType = %d, ConflictState = %d", name, ETG_CENUM(ProtocolId, var._protocol._protocolId),
            		var._protocol._uuid.c_str(), ETG_CENUM(BTProfileUsageType, var._profileUsageType),
            		ETG_CENUM(ConflictState, var._conflictState)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "IssueInfo(%100s): conflictType = %d, deviceID = %d ", name,
            		ETG_CENUM(ConflictType, var._conflictType), var._deviceID));

            ETG_TRACE_USR3_DCL((cls, "IssueInfo(%100s): ProtocolID =  %d, UUID = \"%50s\", "
            		"ProfileUsageType = %d, ConflictState = %d", name, ETG_CENUM(ProtocolId, var._protocol._protocolId),
            		var._protocol._uuid.c_str(), ETG_CENUM(BTProfileUsageType, var._profileUsageType),
            		ETG_CENUM(ConflictState, var._conflictState)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "IssueInfo(%100s): conflictType = %d, deviceID = %d ", name,
            		ETG_CENUM(ConflictType, var._conflictType), var._deviceID));

            ETG_TRACE_USR4_DCL((cls, "IssueInfo(%100s): ProtocolID =  %d, UUID = \"%50s\", "
            		"ProfileUsageType = %d, ConflictState = %d", name, ETG_CENUM(ProtocolId, var._protocol._protocolId),
            		var._protocol._uuid.c_str(), ETG_CENUM(BTProfileUsageType, var._profileUsageType),
            		ETG_CENUM(ConflictState, var._conflictState)));
         }
      }
   }

   void VarTrace(tClassType cls, ConflictInfoList var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ConflictInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ConflictInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ConflictInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ConflictInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ConflictInfoList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ConflictInfoList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ConflictInfoList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ConflictInfoList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ConflictInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ConflictInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ConflictInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ConflictInfoList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ConflictInfoList(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ConflictInfoList(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ConflictInfoList(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ConflictInfoList(%100s):", name));
            }
         }

         ConflictInfoListItr it;
         char elementName[MAXNAME];
         unsigned int i(0u);

         for (it = var.begin(); it != var.end(); it++)
         {
            snprintf(elementName, MAXNAME - 1, "%s[%d]", name, i);
            VarTrace(cls, *it, elementName, level);
         }
      }
   }

   void VarTrace(tClassType cls, IssueInfoList var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var._conflictInfoList.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "IssueInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "IssueInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "IssueInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "IssueInfoList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "IssueInfoList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "IssueInfoList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "IssueInfoList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "IssueInfoList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "IssueInfoList(%100s): conflictTrigger = %d [(%d)%s]", name,
            		   ETG_CENUM(ConflictTrigger, var._conflictTrigger), line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "IssueInfoList(%100s): conflictTrigger = %d [(%d)%s]", name,
            		   ETG_CENUM(ConflictTrigger, var._conflictTrigger), line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "IssueInfoList(%100s): conflictTrigger = %d [(%d)%s]", name,
            		   ETG_CENUM(ConflictTrigger, var._conflictTrigger), line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "IssueInfoList(%100s): conflictTrigger = %d [(%d)%s]", name,
            		   ETG_CENUM(ConflictTrigger, var._conflictTrigger), line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "IssueInfoList(%100s): conflictTrigger = %d", name,
            		   ETG_CENUM(ConflictTrigger, var._conflictTrigger)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "IssueInfoList(%100s): conflictTrigger = %d", name,
            		   ETG_CENUM(ConflictTrigger, var._conflictTrigger)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "IssueInfoList(%100s): conflictTrigger = %d", name,
            		   ETG_CENUM(ConflictTrigger, var._conflictTrigger)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "IssueInfoList(%100s): conflictTrigger = %d", name,
            		   ETG_CENUM(ConflictTrigger, var._conflictTrigger)));
            }
         }

         std::string elementName(name);

         VarTrace(cls, var._conflictInfoList, (elementName + "._conflictInfoList").c_str(), level);
      }
   }

   void VarTrace(tClassType cls, ResetToDefaultStatus var, const char* name, const unsigned char level, const char* file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "ResetToDefaultStatus(%100s): resetState = %d, resetResult = %d [(%d)%s]",
                  name, ETG_CENUM(ResetToDefaultState, var._resetState), ETG_CENUM(ResetToDefaultResult, var._resetResult),
                  line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "ResetToDefaultStatus(%100s): resetState = %d, resetResult = %d [(%d)%s]",
                  name, ETG_CENUM(ResetToDefaultState, var._resetState), ETG_CENUM(ResetToDefaultResult, var._resetResult),
                  line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "ResetToDefaultStatus(%100s): resetState = %d, resetResult = %d [(%d)%s]",
                  name, ETG_CENUM(ResetToDefaultState, var._resetState), ETG_CENUM(ResetToDefaultResult, var._resetResult),
                  line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "ResetToDefaultStatus(%100s): resetState = %d, resetResult = %d [(%d)%s]",
                  name, ETG_CENUM(ResetToDefaultState, var._resetState), ETG_CENUM(ResetToDefaultResult, var._resetResult),
                  line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "ResetToDefaultStatus(%100s): resetState = %d, resetResult = %d",
                  name, ETG_CENUM(ResetToDefaultState, var._resetState), ETG_CENUM(ResetToDefaultResult, var._resetResult)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "ResetToDefaultStatus(%100s): resetState = %d, resetResult = %d",
                  name, ETG_CENUM(ResetToDefaultState, var._resetState), ETG_CENUM(ResetToDefaultResult, var._resetResult)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "ResetToDefaultStatus(%100s): resetState = %d, resetResult = %d",
                  name, ETG_CENUM(ResetToDefaultState, var._resetState), ETG_CENUM(ResetToDefaultResult, var._resetResult)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "ResetToDefaultStatus(%100s): resetState = %d, resetResult = %d",
                  name, ETG_CENUM(ResetToDefaultState, var._resetState), ETG_CENUM(ResetToDefaultResult, var._resetResult)));
         }
      }
   }

   void VarTrace(tClassType cls, LinkQuality var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var._linkQualityInfoList.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "LinkQuality(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "LinkQuality(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "LinkQuality(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "LinkQuality(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "LinkQuality(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "LinkQuality(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "LinkQuality(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "LinkQuality(%100s): <empty>", name));
            }
         }
      }
      else
      {
         LinkQualityInfoList::iterator it;

         for (it = var._linkQualityInfoList.begin(); it != var._linkQualityInfoList.end(); ++it)
         {
            if (file != NULL && line != -1)
            {
               if (1u == level)
               {
                  ETG_TRACE_USR1_DCL((cls, "LinkQuality(%50s._linkQualityInfoList)[deviceHandle = %d]: RSSI = %d, LinkQuality = %d [(%d)%s]",
                        name, it->first, it->second._rssi, it->second._linkQuality, line, removePath(file)));
               }
               else if (2u == level)
               {
                  ETG_TRACE_USR2_DCL((cls, "LinkQuality(%50s._linkQualityInfoList)[deviceHandle = %d]: RSSI = %d, LinkQuality = %d [(%d)%s]",
                        name, it->first, it->second._rssi, it->second._linkQuality, line, removePath(file)));
               }
               else if (3u == level)
               {
                  ETG_TRACE_USR3_DCL((cls, "LinkQuality(%50s._linkQualityInfoList)[deviceHandle = %d]: RSSI = %d, LinkQuality = %d [(%d)%s]",
                        name, it->first, it->second._rssi, it->second._linkQuality, line, removePath(file)));
               }
               else
               {
                  ETG_TRACE_USR4_DCL((cls, "LinkQuality(%50s._linkQualityInfoList)[deviceHandle = %d]: RSSI = %d, LinkQuality = %d [(%d)%s]",
                        name, it->first, it->second._rssi, it->second._linkQuality, line, removePath(file)));
               }
            }
            else
            {
               if (1u == level)
               {
                  ETG_TRACE_USR1_DCL((cls, "LinkQuality(%50s._linkQualityInfoList)[deviceHandle = %d]: RSSI = %d, LinkQuality = %d", name, it->first, it->second._rssi, it->second._linkQuality));
               }
               else if (2u == level)
               {
                  ETG_TRACE_USR2_DCL((cls, "LinkQuality(%50s._linkQualityInfoList)[deviceHandle = %d]: RSSI = %d, LinkQuality = %d", name, it->first, it->second._rssi, it->second._linkQuality));
               }
               else if (3u == level)
               {
                  ETG_TRACE_USR3_DCL((cls, "LinkQuality(%50s._linkQualityInfoList)[deviceHandle = %d]: RSSI = %d, LinkQuality = %d", name, it->first, it->second._rssi, it->second._linkQuality));
               }
               else
               {
                  ETG_TRACE_USR4_DCL((cls, "LinkQuality(%50s._linkQualityInfoList)[deviceHandle = %d]: RSSI = %d, LinkQuality = %d", name, it->first, it->second._rssi, it->second._linkQuality));
               }
            }
         }
      }
   }

   void VarTrace(tClassType cls, TestModeLinkQuality var, const char* name, const unsigned char level, const char* file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "TestModeLinkQuality(%50s._linkQualityInfo): RSSI = %d, LinkQuality = %d [(%d)%s]", name, var._linkQualityInfo._rssi, var._linkQualityInfo._linkQuality, line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "TestModeLinkQuality(%50s._linkQualityInfo): RSSI = %d, LinkQuality = %d [(%d)%s]", name, var._linkQualityInfo._rssi, var._linkQualityInfo._linkQuality, line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "TestModeLinkQuality(%50s._linkQualityInfo): RSSI = %d, LinkQuality = %d [(%d)%s]", name, var._linkQualityInfo._rssi, var._linkQualityInfo._linkQuality, line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "TestModeLinkQuality(%50s._linkQualityInfo): RSSI = %d, LinkQuality = %d [(%d)%s]", name, var._linkQualityInfo._rssi, var._linkQualityInfo._linkQuality, line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "TestModeLinkQuality(%50s._linkQualityInfo): RSSI = %d, LinkQuality = %d", name, var._linkQualityInfo._rssi, var._linkQualityInfo._linkQuality));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "TestModeLinkQuality(%50s._linkQualityInfo): RSSI = %d, LinkQuality = %d", name, var._linkQualityInfo._rssi, var._linkQualityInfo._linkQuality));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "TestModeLinkQuality(%50s._linkQualityInfo): RSSI = %d, LinkQuality = %d", name, var._linkQualityInfo._rssi, var._linkQualityInfo._linkQuality));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "TestModeLinkQuality(%50s._linkQualityInfo): RSSI = %d, LinkQuality = %d", name, var._linkQualityInfo._rssi, var._linkQualityInfo._linkQuality));
         }
      }
   }

   void VarTrace(tClassType cls, ProtocolConnectionController* var,const char* name, const unsigned char level, const char *file, const int line)
   {
      if (file != NULL && line != -1)
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "ProtocolConnectionController(%100s): instance = 0x%p, protocolId = %d, uuid = \"%50s\" [(%d)%s]",
                  name, (void *) var, ETG_CENUM(ProtocolId, var->getProtocolId()),
                  var->getUuid().c_str(), line, removePath(file)));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "ProtocolConnectionController(%100s): instance = 0x%p, protocolId = %d, uuid = \"%50s\" [(%d)%s]",
                  name, (void *) var, ETG_CENUM(ProtocolId, var->getProtocolId()),
                  var->getUuid().c_str(), line, removePath(file)));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "ProtocolConnectionController(%100s): instance = 0x%p, protocolId = %d, uuid = \"%50s\" [(%d)%s]",
                  name, (void *) var, ETG_CENUM(ProtocolId, var->getProtocolId()),
                  var->getUuid().c_str(), line, removePath(file)));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "ProtocolConnectionController(%100s): instance = 0x%p, protocolId = %d, uuid = \"%50s\" [(%d)%s]",
                  name, (void *) var, ETG_CENUM(ProtocolId, var->getProtocolId()),
                  var->getUuid().c_str(), line, removePath(file)));
         }
      }
      else
      {
         if (1u == level)
         {
            ETG_TRACE_USR1_DCL((cls, "ProtocolConnectionController(%100s): instance = 0x%p, protocolId = %d, uuid = \"%50s\"",
                  name, (void *) var, ETG_CENUM(ProtocolId, var->getProtocolId()),
                  var->getUuid().c_str()));
         }
         else if (2u == level)
         {
            ETG_TRACE_USR2_DCL((cls, "ProtocolConnectionController(%100s): instance = 0x%p, protocolId = %d, uuid = \"%50s\"",
                  name, (void *) var, ETG_CENUM(ProtocolId, var->getProtocolId()),
                  var->getUuid().c_str()));
         }
         else if (3u == level)
         {
            ETG_TRACE_USR3_DCL((cls, "ProtocolConnectionController(%100s): instance = 0x%p, protocolId = %d, uuid = \"%50s\"",
                  name, (void *) var, ETG_CENUM(ProtocolId, var->getProtocolId()),
                  var->getUuid().c_str()));
         }
         else
         {
            ETG_TRACE_USR4_DCL((cls, "ProtocolConnectionController(%100s): instance = 0x%p, protocolId = %d, uuid = \"%50s\"",
                  name, (void *) var, ETG_CENUM(ProtocolId, var->getProtocolId()),
                  var->getUuid().c_str()));
         }
      }
   }

   void VarTrace(tClassType cls, ProtocolConnectionControllerList& var, const char *name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolConnectionControllerList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolConnectionControllerList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolConnectionControllerList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolConnectionControllerList(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolConnectionControllerList(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolConnectionControllerList(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolConnectionControllerList(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolConnectionControllerList(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolConnectionControllerList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolConnectionControllerList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolConnectionControllerList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolConnectionControllerList(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "ProtocolConnectionControllerList(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "ProtocolConnectionControllerList(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "ProtocolConnectionControllerList(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "ProtocolConnectionControllerList(%100s):", name));
            }
         }

         ProtocolConnectionControllerList::iterator it;
         char elementName[MAXNAME];
         unsigned int i(0u);

         for (it = var.begin(); it != var.end(); it++)
         {
            snprintf(elementName, MAXNAME - 1, "%s[%d]", name, i);
            VarTrace(cls, *it, elementName, level);
         }
      }
   }

   void VarTrace(tClassType cls, DeviceProtocolListMap var, const char* name, const unsigned char level, const char *file, const int line)
   {
      if (true == var.empty())
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceProtocolListMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceProtocolListMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceProtocolListMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceProtocolListMap(%100s): <empty> [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceProtocolListMap(%100s): <empty>", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceProtocolListMap(%100s): <empty>", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceProtocolListMap(%100s): <empty>", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceProtocolListMap(%100s): <empty>", name));
            }
         }
      }
      else
      {
         if (file != NULL && line != -1)
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceProtocolListMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceProtocolListMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceProtocolListMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceProtocolListMap(%100s): [(%d)%s]", name, line, removePath(file)));
            }
         }
         else
         {
            if (1u == level)
            {
               ETG_TRACE_USR1_DCL((cls, "DeviceProtocolListMap(%100s):", name));
            }
            else if (2u == level)
            {
               ETG_TRACE_USR2_DCL((cls, "DeviceProtocolListMap(%100s):", name));
            }
            else if (3u == level)
            {
               ETG_TRACE_USR3_DCL((cls, "DeviceProtocolListMap(%100s):", name));
            }
            else
            {
               ETG_TRACE_USR4_DCL((cls, "DeviceProtocolListMap(%100s):", name));
            }
         }

         DeviceProtocolListMap::iterator it;

         for (it = var.begin(); it != var.end(); it++)
         {
            if (file != NULL && line != -1)
            {
               if (1u == level)
               {
                  ETG_TRACE_USR1_DCL((cls, "DeviceProtocolListMap(%100s): deviceHandle = %d [(%d)%s]",
                        name, it->first, line, removePath(file)));
               }
               else if (2u == level)
               {
                  ETG_TRACE_USR2_DCL((cls, "DeviceProtocolListMap(%100s): deviceHandle = %d [(%d)%s]",
                        name, it->first, line, removePath(file)));
               }
               else if (3u == level)
               {
                  ETG_TRACE_USR3_DCL((cls, "DeviceProtocolListMap(%100s): deviceHandle = %d [(%d)%s]",
                        name, it->first, line, removePath(file)));
               }
               else
               {
                  ETG_TRACE_USR4_DCL((cls, "DeviceProtocolListMap(%100s): deviceHandle = %d [(%d)%s]",
                        name, it->first, line, removePath(file)));
               }
            }
            else
            {
               if (1u == level)
               {
                  ETG_TRACE_USR1_DCL((cls, "DeviceProtocolListMap(%100s): deviceHandle = %d", name, it->first));
               }
               else if (2u == level)
               {
                  ETG_TRACE_USR2_DCL((cls, "DeviceProtocolListMap(%100s): deviceHandle = %d", name, it->first));
               }
               else if (3u == level)
               {
                  ETG_TRACE_USR3_DCL((cls, "DeviceProtocolListMap(%100s): deviceHandle = %d", name, it->first));
               }
               else
               {
                  ETG_TRACE_USR4_DCL((cls, "DeviceProtocolListMap(%100s): deviceHandle = %d", name, it->first));
               }
            }

            std::string elementName(name);

            VarTrace(cls, it->second, (elementName + "._protocolList").c_str(), level);
         }
      }
   }
}
