#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BT_LIMITATION_CONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BtLimitationController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BT_LIMITATION_CONTROLLER
#endif
#endif

#include "BmAllTypes.h"
#include "LocalSpm.h"
#include "FunctionTracer.h"
#include "Dispatcher.h"
#include "BmVarTrace.h"
#include "BtLimitationController.h"
#include "BmCoreIfMessages.h"
#include "BmUtils.h"
#include "BmGlobalLock.h"
#include <algorithm>

namespace bmcore
{
#define EVENT_PARAMETERS_SIZE 20
#define NUM_OF_MSEC_PER_SEC 1000

static LimitationState selimitationErrorType = BM_LIMITATION_STATE_ERROR;

/*************************************************************************
 ** FUNCTION:  BtLimitationController::BtLimitationController(...)
 **************************************************************************/
BtLimitationController::BtLimitationController(const ComponentId ComponentId):
      ILocalSpm(ComponentId), _ProjectionDeviceBDAddress(""), _ProjectionDeviceBDName(""),
      _CurrentLimitationMode(), _CurrentLimitationAction(BM_LIMITATION_ACTION_LAST),
      _ProjectionDeviceBTDeviceId(0u), _ProtocolsToBeBlockedForProjectionDevice(),
      _ProtocolsToBeBlockedForNonProjectionDevices(), _bdAddressOfAvpBlockedDevice(""),
      _deviceProtocolsWaitingToGetBlocked(), _CPWReconnectionTimerID(0),
      _CPWReconnectionTimer(), _LockCPWReconnectionTimer(), _deviceIdWaitingForBeingDisconnected(0u),
      _deviceLost(false), _wblOperatingFreq(BTS_WBL_WIFI_FREQUENCY_LAST)
{
   ETG_TRACE_USR1(("BtLimitationController: constructor entered"));

   // threadless SM
   disableAllLocks(true);
}//! end of BtLimitationController()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::~BtLimitationController(...)
 **************************************************************************/
BtLimitationController::~BtLimitationController()
{
   ENTRY_INTERNAL
   ETG_TRACE_USR1(("~BtLimitationController: destructor entered"));

   _ProjectionDeviceBDAddress = "";
   _ProjectionDeviceBDName = "";
   _CurrentLimitationAction = BM_LIMITATION_ACTION_LAST;
   _ProjectionDeviceBTDeviceId = 0;
   _bdAddressOfAvpBlockedDevice = "";
   _CPWReconnectionTimerID = 0;
   _deviceLost = false;
   _wblOperatingFreq = BTS_WBL_WIFI_FREQUENCY_LAST;
}//! end of ~BtLimitationController()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::validateBTLimitationModeRequest(...)
 **************************************************************************/
BmResult BtLimitationController::validateBTLimitationModeRequest(BmCoreIfMessage_SetBtLimitationModeRequest* bmCoreIfMessage)
{
   ETG_TRACE_USR1(("validateBTLimitationModeRequest: entered"));

   BmResult bmResult(BM_RESULT_OK);
   Result result(CC_ERR_INT_NO_ERROR);

   if(NULL != bmCoreIfMessage)
   {
      LimitationAction LimiAction = bmCoreIfMessage->getLimitationAction();
      LimitationMode LimiMode = bmCoreIfMessage->getLimitationMode();
      BdAddress bdAddress(bmCoreIfMessage->getBdAddress());
      BdName bdName(bmCoreIfMessage->getBdName());

      if (((false == isValidBdAddress(bdAddress, false)) && (false == isValidLocalBdName(bdName))) || (false == isValidLimitationMode(LimiMode))
            || (false == isValidLimitationAction(LimiAction)))
      {
         ETG_TRACE_ERR(("validateBTLimitationModeRequest(bdAddress = \"%50s\", bdName = \"%50s\"): invalid parameter value(s) found in requested BT limitation mode",
               bdAddress.c_str(), bdName.c_str()));
         bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
      }//End of if ((false == isValidBdAddress(bmCoreIfMessage->getBdAddress(), false))..)
      else
      {
         //! Actions PREPARE and PREACTIVATE are not allowed for USB CommIntf
         if (((BM_LIMITATION_ACTION_PREPARE == LimiAction) || (BM_LIMITATION_ACTION_PREACTIVATE == LimiAction))
               && (BM_LIMITATION_COMMUNICATION_IF_USB == LimiMode._limitationCommunicationIf))
         {
            ETG_TRACE_ERR(("validateBTLimitationModeRequest(bdAddress = \"%50s\", bdName = \"%50s\"): requested BT limitation action is not allowed "
                  "for requested communication interface", bdAddress.c_str(), bdName.c_str()));
            bmResult = BM_RESULT_ERR_BT_LIMITATION_ACTION_NOT_ALLOWED;
         }//End of if(((BM_LIMITATION_ACTION_PREPARE == LimiAction) ||...)
         else if((true == bdName.empty()) && ((BM_LIMITATION_FEATURE_MY_SPIN_IOS == LimiMode._limitationFeature)
               || (BM_LIMITATION_FEATURE_CAR_LIFE_IOS == LimiMode._limitationFeature)))
         {
            ETG_TRACE_ERR(("validateBTLimitationModeRequest(bdAddress = \"%50s\", bdName = \"%50s\"): requested BT limitation feature is not allowed "
                  "for requested communication interface", bdAddress.c_str(), bdName.c_str()));
            bmResult = BM_RESULT_ERR_BT_LIMITATION_ACTION_NOT_ALLOWED;
         }
         else
         {
            //! convert BD address given in request to lower case
            convertBdAddress2LowerCase(bdAddress);

            //! Get Limitation Mode Info for the BT address of incoming request
            BtLimitationModeInfo btLimitationModeInfo;
            result = LocalSpm::getBmCoreMainController().getBtLimitationModeInfo(btLimitationModeInfo, bdAddress, bdName);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               //! BT address of requested Limitation Mode matches the current BT Address of the Active Limitation Mode,
               //! i.e. a BT limitation mode change for a currently handled projection device is requested
               if ((bdAddress == _ProjectionDeviceBDAddress) && (bdName == _ProjectionDeviceBDName))
               {
                  bmResult = IsLimitationActionRequestAllowed(btLimitationModeInfo._limitationState, LimiAction);
               }//End of if (bdAddress == _ProjectionDeviceBTAddress)
               else
               {
                  //! BT address of requested Limitation Mode does not match the current BT Address of the Active Limitation Mode - must not occur
                  ETG_TRACE_ERR(("validateBTLimitationModeRequest(bdAddress = \"%50s\", bdName = \"%50s\"): BD address/BD Name given in request does not match the "
                        "current BT Address of the Active Limitation Mode", bdAddress.c_str(), bdName.c_str()));
                  bmResult = BM_RESULT_ERR_GENERAL;
               }//End of else
            }//End of if (CC_ERR_INT_NO_ERROR == result)

            //! Deactivate_Wait->Activate, BT address/BT name will change for new limitation technology
            else if ((false == _ProjectionDeviceBDAddress.empty()) || (false == _ProjectionDeviceBDName.empty()))
            {
               ETG_TRACE_USR4(("validateBTLimitationModeRequest(bdAddress = \"%50s\", bdName = \"%50s\"): no BT limitation mode found for BD address given in request",
                     bdAddress.c_str(), bdName.c_str()));

               //! Get Limitation Mode Info for the existing BT address for which Limitation Mode is already set
               BtLimitationModeInfo btLimitationModeInfo;
               result = LocalSpm::getBmCoreMainController().getBtLimitationModeInfo(btLimitationModeInfo, _ProjectionDeviceBDAddress, _ProjectionDeviceBDName);
               if (CC_ERR_INT_NO_ERROR == result)
               {
                  //!When SM is in WAITING state, only Activate request will be processed with a different BT address or BT name
                  //!No Limitation Action for different BT address is allowed when current state is not WAITING
                  //!Note: If multiple SPI sessions are supported in future, this check has to be modified.
                  if (((bdAddress != _ProjectionDeviceBDAddress) || (bdName != _ProjectionDeviceBDName)) && (((BM_LIMITATION_STATE_WAITING == btLimitationModeInfo._limitationState)
                        && (BM_LIMITATION_ACTION_ACTIVATE != LimiAction)) || (BM_LIMITATION_STATE_WAITING != btLimitationModeInfo._limitationState)))
                  {
                     ETG_TRACE_ERR(("validateBTLimitationModeRequest: requested BT limitation action for different BD address "
                           "is not allowed in current BT limitation state"));
                     bmResult = BM_RESULT_ERR_BT_LIMITATION_ACTION_NOT_ALLOWED;
                  }//End of if ((bdAddress != _ProjectionDeviceBTAddress) && ...)

                  //!When SM is in WAITING state, only Deactivate and Deactivate_SpmOff request will be processed with same BT address
                  else if((bdAddress == _ProjectionDeviceBDAddress) && (bdName == _ProjectionDeviceBDName) && ((BM_LIMITATION_STATE_WAITING == btLimitationModeInfo._limitationState)
                        && (BM_LIMITATION_ACTION_DEACTIVATE != LimiAction) && (BM_LIMITATION_ACTION_DEACTIVATE_SPM_STATE_OFF != LimiAction)))
                  {
                     ETG_TRACE_ERR(("validateBTLimitationModeRequest: requested BT limitation action for same BD address "
                           "is not allowed in current BT limitation state"));
                     bmResult = BM_RESULT_ERR_BT_LIMITATION_ACTION_NOT_ALLOWED;
                  }//End of else if((bdAddress == _ProjectionDeviceBTAddress) && ...)

                  //!Delete the Deactivate_Waiting entry from limitationInfo list when a new activate request is received which is valid.
                  else if (((bdAddress != _ProjectionDeviceBDAddress) || (bdName != _ProjectionDeviceBDName)) && ((BM_LIMITATION_STATE_WAITING == btLimitationModeInfo._limitationState)
                        && (BM_LIMITATION_ACTION_ACTIVATE == LimiAction)))
                  {
                     //!Delete the entry from the LimitationMode List updated to the clients
                     LocalSpm::getBmCoreMainController().deleteBtLimitationModeItem(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName,
                           _CurrentLimitationMode);
                  }//End of else if ((bdAddress != _ProjectionDeviceBTAddress) &&...)
               }//End of if (CC_ERR_INT_NO_ERROR == result)
               else
               {
                  ETG_TRACE_ERR(("validateBTLimitationModeRequest(bdAddress = \"%50s\", bdName = \"%50s\"): no BT limitation mode found for BD address "
                        "of projection device already handled by BT limitation mode controller",
                        _ProjectionDeviceBDAddress.c_str(), bdName.c_str()));
                  bmResult = BM_RESULT_ERR_GENERAL;
               }//End of else
            }//End of else if (false == _ProjectionDeviceBTAddress.empty())

            //!Initially _ProjectionDeviceBTAddress and _ProjectionDeviceBDName will be empty for the first Limitation Request
            //!and will not be present in the LimitationModeInfo list which is not an error case
            else if ((true == _ProjectionDeviceBDAddress.empty()) && (true == _ProjectionDeviceBDName.empty()))
            {
               // _ProjectionDeviceBDAddress is empty, no BT limitation mode set for BD address given in request
               // => SM is in IDLE state

               ETG_TRACE_USR1(("validateBTLimitationModeRequest(bdAddress = \"%50s\", bdName = \"%50s\"): no BT limitation mode found for BD address given in request "
                     "and BT limitation mode controller is not handling any projection device", bdAddress.c_str(), bdName.c_str()));

               //! Action PREACTIVATE is not allowed
               if ((BM_LIMITATION_ACTION_PREPARE != LimiAction) && (BM_LIMITATION_ACTION_ACTIVATE != LimiAction))
               {
                  ETG_TRACE_ERR(("validateBTLimitationModeRequest(bdAddress = \"%50s\", bdName = \"%50s\"): requested BT limitation action is not allowed",
                        bdAddress.c_str(), bdName.c_str()));
                  bmResult = BM_RESULT_ERR_BT_LIMITATION_ACTION_NOT_ALLOWED;
               }//End of if ((BM_LIMITATION_ACTION_PREPARE != LimiAction) && (BM_LIMITATION_ACTION_ACTIVATE != LimiAction))

               //Action ACTIVATE is allowed for Wifi CommIntf
               else if ((BM_LIMITATION_ACTION_ACTIVATE == LimiAction) &&
                     ((BM_LIMITATION_COMMUNICATION_IF_WIFI == LimiMode._limitationCommunicationIf)))
               {
                  //!Update the Limitation Mode comm Intf based on WBL operating frequency
                  LimiMode._limitationCommunicationIf = (BTS_WBL_WIFI_FREQUENCY_5_GHZ == _wblOperatingFreq) ?
                        (BM_LIMITATION_COMMUNICATION_IF_WIFI_5) : (BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4);

                  ETG_TRACE_USR4(("validateBTLimitationModeRequest(bdAddress = \"%50s\", bdName = \"%50s\"): requested BT limitation action is allowed and "
                        "updated the communication interface - %d", bdAddress.c_str(), bdName.c_str(), ETG_CENUM(LimitationCommunicationIf, LimiMode._limitationCommunicationIf)))
               }//End of else if ((BM_LIMITATION_ACTION_ACTIVATE == LimiAction) && ...)
            }//End of else if ((true == _ProjectionDeviceBDAddress.empty()))
         }//End of if (((BM_LIMITATION_ACTION_PREPARE == LimiAction) || (BM_LIMITATION_ACTION_PREACTIVATE == LimiAction))...
      }//End of if ((false == isValidBdAddress(bdAddress, false)) || (false == isValidLimitationMode(LimiMode))...

      if(BM_RESULT_OK == bmResult)
      {
         //! Update the variables with the incoming request parameters
         _ProjectionDeviceBDAddress = bdAddress;
         _ProjectionDeviceBDName = bdName;

         _CurrentLimitationMode._limitationFeature = LimiMode._limitationFeature;
         _CurrentLimitationMode._limitationCommunicationIf =
               ((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
                     || (BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf)) ?
                           (_CurrentLimitationMode._limitationCommunicationIf) : (LimiMode._limitationCommunicationIf);

         _CurrentLimitationAction = LimiAction;

         getBTDeviceIdForProjectionDevice();
      }//End of if(BM_RESULT_OK == bmResult)
   }//End of if(NULL != bmCoreIfMessage)
   else
   {
      ETG_TRACE_ERR(("validateBTLimitationModeRequest: invalid parameter BM Core message"));
      bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
   }//End of else

   return bmResult;

}//! End of BtLimitationController::validateBTLimitationModeRequest(...)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::validateReplaceBTLimitationModeRequest(...)
 **************************************************************************/
BmResult BtLimitationController::validateReplaceBTLimitationModeRequest(BmCoreIfMessage_ReplaceBtLimitationModeRequest* bmCoreIfMessage)
{
   ETG_TRACE_USR1(("validateReplaceBTLimitationModeRequest: entered"));

   BmResult bmResult(BM_RESULT_OK);
   Result result(CC_ERR_INT_NO_ERROR);

   if(NULL != bmCoreIfMessage)
   {
      LimitationAction LimiAction = bmCoreIfMessage->getNewLimitationAction();
      LimitationMode LimiMode = bmCoreIfMessage->getNewLimitationMode();
      BdAddress bdAddress(bmCoreIfMessage->getNewBdAddress());
      convertBdAddress2LowerCase(bdAddress);
      BdName bdName(bmCoreIfMessage->getNewBdName());

      if (((false == isValidBdAddress(bdAddress, false)) && (false == isValidLocalBdName(bdName))) || (false == isValidLimitationMode(LimiMode))
            || (false == isValidLimitationAction(LimiAction)))
      {
         ETG_TRACE_ERR(("validateReplaceBTLimitationModeRequest(bdAddress = \"%50s\", bdName = \"%50s\"): invalid parameter value(s) "
               "found in requested BT limitation mode", bdAddress.c_str(), bdName.c_str()));
         bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
      }//End of if ((false == isValidBdAddress(bmCoreIfMessage->getBdAddress(), false))..)
      else
      {
         //! Get Limitation Mode Info for current active BT address
         BtLimitationModeInfo btLimitationModeInfo;
         result = LocalSpm::getBmCoreMainController().getBtLimitationModeInfo(btLimitationModeInfo, _ProjectionDeviceBDAddress, _ProjectionDeviceBDName);
         if (CC_ERR_INT_NO_ERROR == result)
         {
            //!If current limitation state is WAITING and incoming request is not ACTIVATE or PREPARE, result into an error.
            if ((BM_LIMITATION_STATE_WAITING == btLimitationModeInfo._limitationState) &&
                  (BM_LIMITATION_ACTION_ACTIVATE != LimiAction) && (BM_LIMITATION_ACTION_PREPARE != LimiAction))
            {
               ETG_TRACE_ERR(("validateReplaceBTLimitationModeRequest: requested BT limitation action"
                     "is not allowed in current BT limitation state"));
               bmResult = BM_RESULT_ERR_BT_LIMITATION_ACTION_NOT_ALLOWED;
            }//End of if ((BM_LIMITATION_STATE_WAITING == btLimitationModeInfo._limitationState) && ...)
         }//!End of if (CC_ERR_INT_NO_ERROR == result)
         else
         {
            ETG_TRACE_ERR(("validateReplaceBTLimitationModeRequest(bdAddress = \"%50s\", bdName = \"%50s\"): no BT limitation mode found for BD address/BD name "
                  "of projection device already handled by BT limitation mode controller", _ProjectionDeviceBDAddress.c_str(), _ProjectionDeviceBDName.c_str()));
            bmResult = BM_RESULT_ERR_GENERAL;
         }//End of else
      }//!End of else

      if(BM_RESULT_OK == bmResult)
      {
         if((BM_LIMITATION_ACTION_ACTIVATE == LimiAction) && (BM_LIMITATION_COMMUNICATION_IF_WIFI == LimiMode._limitationCommunicationIf))
         {
            //!Update the Limitation Mode comm Intf based on WBL operating frequency
            LimiMode._limitationCommunicationIf = (BTS_WBL_WIFI_FREQUENCY_5_GHZ == _wblOperatingFreq) ?
                  (BM_LIMITATION_COMMUNICATION_IF_WIFI_5) : (BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4);

            ETG_TRACE_USR4(("validateBTLimitationModeRequest(bdAddress = \"%50s\", bdName = \"%50s\"): requested BT limitation action is allowed and "
                  "updated the communication interface - %d", bdAddress.c_str(), bdName.c_str(),
                  ETG_CENUM(LimitationCommunicationIf, LimiMode._limitationCommunicationIf)));
         }//!End of if((BM_LIMITATION_ACTION_ACTIVATE == LimiAction) && ...)

         //!Delete the entry from the LimitationMode List updated to the clients
         LocalSpm::getBmCoreMainController().deleteBtLimitationModeItem(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName, _CurrentLimitationMode);

         //!Reset the current limitation mode
         _CurrentLimitationMode._limitationFeature = BM_LIMITATION_FEATURE_LAST;
         _CurrentLimitationMode._limitationCommunicationIf = BM_LIMITATION_COMMUNICATION_IF_LAST;
         _CurrentLimitationAction = BM_LIMITATION_ACTION_LAST;

         //! Update the variables with the incoming request parameters
         _ProjectionDeviceBDAddress = bdAddress;
         _ProjectionDeviceBDName = bdName;

         _CurrentLimitationMode._limitationFeature = LimiMode._limitationFeature;
         _CurrentLimitationMode._limitationCommunicationIf =
               ((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
                     || (BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf)) ?
                           (_CurrentLimitationMode._limitationCommunicationIf) : (LimiMode._limitationCommunicationIf);

         _CurrentLimitationAction = LimiAction;

         getBTDeviceIdForProjectionDevice();
      }//!End of if(BM_RESULT_OK == bmResult)
   }//!End of if(NULL != bmCoreIfMessage)
   else
   {
      ETG_TRACE_ERR(("validateReplaceBTLimitationModeRequest: invalid parameter BM Core message"));
      bmResult = BM_RESULT_ERR_INVALID_PARAMETER;
   }//End of else

   return bmResult;
}//! End of BtLimitationController::validateReplaceBTLimitationModeRequest(...)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::sendLimitationActionSmEvent(...)
 **************************************************************************/
Result BtLimitationController::sendLimitationActionSmEvent(IN const string& callingMethodName,
      const LimitationAction& limitationAction)
{
   string smEventName = getSmEventNameByLimitationAction(limitationAction);
   Result result(CC_ERR_INT_NO_ERROR);

   result = (false == smEventName.empty()) ?
         (sendSmEvent(callingMethodName, smEventName)) : (CC_ERR_INT_GENERAL_ERROR);

   return result;
}//! End of BtLimitationController::sendLimitationActionSmEvent(...)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::onBtStatusChanged(...)
 **************************************************************************/
void BtLimitationController::onBtStatusChanged(IN const SwitchStatus& bluetoothStatus)
{
   ETG_TRACE_USR1(("onBtStatusChanged: SwitchState = %d, SwitchedOffReason = %d",
         ETG_CENUM(SwitchState, bluetoothStatus._switchState),
         ETG_CENUM(SwitchedOffReason, bluetoothStatus._switchedOffReason)));

   if (SWITCH_STATE_SWITCHED_ON == bluetoothStatus._switchState)
   {
      (void) sendSmEvent("onBtStatusChanged", "BT_ON");
   }//End of if (SWITCH_STATE_SWITCHED_ON == bluetoothStatus._switchState)
   else if (SWITCH_STATE_SWITCHED_OFF == bluetoothStatus._switchState)
   {
      (void) sendSmEvent("onBtStatusChanged", "FAILED");
   }//End of else if (SWITCH_STATE_SWITCHED_OFF == bluetoothStatus._switchState)
}//! End of BtLimitationController::onBtStatusChanged()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::onLocalPairableModeChanged(...)
 **************************************************************************/
void BtLimitationController::onLocalPairableModeChanged(IN const SwitchStatus& localPairableMode)
{
   ETG_TRACE_USR1(("onLocalPairableModeChanged: localPairableMode._switchState = %d, "
         "localPairableMode._switchedOffReason = %d",
         ETG_CENUM(SwitchState, localPairableMode._switchState),
         ETG_CENUM(SwitchedOffReason, localPairableMode._switchedOffReason)));

   //TODO:Check with Stefan .Write comment why even on switched OFF state Pairable is sent and not failed
   if ((SWITCH_STATE_SWITCHED_ON == localPairableMode._switchState)
         || (SWITCH_STATE_SWITCHED_OFF == localPairableMode._switchState))
   {
      (void) sendSmEvent("onLocalPairableModeChanged", "PAIRABLE");
   }//End of if ((SWITCH_STATE_SWITCHED_ON == localPairableMode._switchState)...)
}//!End of BtLimitationController::onLocalPairableModeChanged(..)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::onLocalConnectableModeChanged(...)
 **************************************************************************/
void BtLimitationController::onLocalConnectableModeChanged(IN const SwitchStatus& localConnectableMode)
{
   ETG_TRACE_USR1(("onLocalConnectableModeChanged: localConnectableMode._switchState = %d, "
         "localConnectableMode._switchedOffReason = %d",
         ETG_CENUM(SwitchState, localConnectableMode._switchState),
         ETG_CENUM(SwitchedOffReason, localConnectableMode._switchedOffReason)));

   if ((SWITCH_STATE_SWITCHED_ON == localConnectableMode._switchState)
         || (SWITCH_STATE_SWITCHED_OFF == localConnectableMode._switchState))
   {
      (void) sendSmEvent("onLocalConnectableModeChanged", "CONNECTABLE");
   }//End of if ((SWITCH_STATE_SWITCHED_ON == localConnectableMode._switchState)...)
}//!End of BtLimitationController::onLocalConnectableModeChanged(..)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::onProtocolConnectionStatusChange(...)
 **************************************************************************/
void BtLimitationController::onProtocolConnectionStatusChange(IN const DeviceId deviceId, IN const Protocol& protocol,
      IN const ConnectionStatus connectionStatus)
{
   ETG_TRACE_USR1(("onProtocolConnectionStatusChange: deviceId = %d, protocol = (%d, \"%50s\"), connectionStatus = %d",
         deviceId, ETG_CENUM(ProtocolId, protocol._protocolId), protocol._uuid.c_str(),
         ETG_CENUM(ConnectionStatus, connectionStatus)));

   //! Check Profile connection status of iAP2 via SPP for CPW device.
   if((deviceId == _ProjectionDeviceBTDeviceId) && (BM_PROTOCOL_ID_SPP == protocol._protocolId)
         && (IAP2BT_SPP_UUID == protocol._uuid))
   {
      //!If iAP2 via SPP connects successfully
      if(BM_CONNECTION_STATUS_CONNECTED == connectionStatus)
      {
         _LockCPWReconnectionTimer.lock();

         //! Stop and Cancel timer if timer is active once iAP2 over SPP profile is connected for CPW device.
         if(0 != _CPWReconnectionTimerID)
         {
            _CPWReconnectionTimer.CancelTimer(_CPWReconnectionTimerID);
            _CPWReconnectionTimerID = 0;
         }//End of if(0 != _CPWReconnectionTimerID)

         _LockCPWReconnectionTimer.unlock();

         (void) sendSmEvent("onProtocolConnectionStatusChange", "PREPARED");
      }//End of if(BM_CONNECTION_STATUS_CONNECTED == connectionStatus)

      //!If iAP2 via SPP connects is not successful, retry until timer expires.
      else if (BM_CONNECTION_STATUS_DISCONNECTED == connectionStatus)
         /*TODO: Also extend to check the reason of PAGE_TIMEOUT from BT stack*/
      {
         (void) sendSmEvent("onProtocolConnectionStatusChange", "SPP_IAP_DISCONNECTED");
      }//End of else if ((0 != _CPWReconnectionTimerID) &&...)
   }//End of if((deviceId == _ProjectionDeviceBTDeviceId) && (BM_PROTOCOL_ID_SPP == protocol._protocolId)...)

   //! Check Profile disconnection status for devices for which BlockProtocols is triggered
   DeviceProtocol deviceProtocol(deviceId, protocol);
   if (BM_CONNECTION_STATUS_DISCONNECTED == connectionStatus)
   {
      DeviceProtocolList::iterator it;
      it = find(_deviceProtocolsWaitingToGetBlocked.begin(), _deviceProtocolsWaitingToGetBlocked.end(), deviceProtocol);
      if (_deviceProtocolsWaitingToGetBlocked.end() != it)
      {
         _deviceProtocolsWaitingToGetBlocked.erase(it);

         if ((true == _deviceProtocolsWaitingToGetBlocked.empty()) && (0u == _deviceIdWaitingForBeingDisconnected))
         {
            ETG_TRACE_USR1(("onProtocolConnectionStatusChange: All blocked protocols are disconnected"));

            (void) sendSmEvent("onProtocolConnectionStatusChange", "PROFILES_BLOCKED");
         }//End of if (true == _deviceProtocolsWaitingToGetBlocked.empty())
      }//End of if (_deviceProtocolsWaitingToGetBlocked.end() != it)
   }//End of if (BM_CONNECTION_STATUS_DISCONNECTED == connectionStatus)

   //!Switch OFF the local pairable mode once the projection device is HFP connected.
   //!This is to avoid to keep the pairable mode ON until timeout(5mins) once device is HFP connected.
   if ((deviceId == _ProjectionDeviceBTDeviceId) && (BM_PROTOCOL_ID_HFP == protocol._protocolId)
         && (BM_CONNECTION_STATUS_CONNECTED == connectionStatus) &&
         (true == IS_LIMIMODE_FOR_HANDSFREE_PROFILE(_CurrentLimitationMode._limitationFeature, _CurrentLimitationMode._limitationCommunicationIf)))
   {
      ETG_TRACE_USR4(("onProtocolConnectionStatusChange: switching off pairable mode as HFP profile "
            "is connected for the projection device"));

      //Remove restriction of pairable/connectable mode
      LocalSpm::getBmCoreMainController().setDeviceBdAddrForRestrictedPairingConnecting("");

      (void) LocalSpm::getBmCoreMainController().switchLocalPairableModeInt(
            LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode, "");

      (void) LocalSpm::getBmCoreMainController().switchLocalConnectableModeInt(
            LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode, "");

   }//End of if ((deviceId == _deviceId) && (BM_PROTOCOL_ID_HFP == protocol._protocolId)...)
}//!End of void BtLimitationController::onProtocolConnectionStatusChange(IN const DeviceId deviceId,...)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::onDeviceConnectionStatusChange(...)
 **************************************************************************/
void BtLimitationController::onDeviceConnectionStatusChange(IN const DeviceId deviceId,
      IN const ConnectionStatus connectionStatus)
{
   ETG_TRACE_USR1(("onDeviceConnectionStatusChange: deviceId = %d, connectionStatus = %d",
         deviceId, ETG_CENUM(ConnectionStatus, connectionStatus)));

   if ((0 != _deviceIdWaitingForBeingDisconnected) && (BM_CONNECTION_STATUS_DISCONNECTED == connectionStatus))
   {
      ETG_TRACE_USR1(("onDeviceConnectionStatusChange: device with ID = %d got disconnected (DCC got released)",
            deviceId));

      _deviceIdWaitingForBeingDisconnected = 0u;

      if (true == _deviceProtocolsWaitingToGetBlocked.empty())
      {
         (void) sendSmEvent("onDeviceConnectionStatusChange", "PROFILES_BLOCKED");
      }//End of if(true == _deviceProtocolsWaitingToGetBlocked.empty())
   }//End of if ((deviceId == _deviceIdWaitingForBeingDisconnected) &&...
}//End of void BtLimitationController::onDeviceConnectionStatusChange(IN const DeviceId deviceId,...)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::UpdateAvpBlockedStateForDevice()
 **************************************************************************/
void BtLimitationController::UpdateAvpBlockedStateForDevice(IN const BdAddress& bdAddress,
      IN bool AvpBlockedState)
{
   ETG_TRACE_USR1(("UpdateAvpBlockedStateForDevice with BDAddress = \"%50s\" is AvpBlockedState = %s",
         bdAddress.c_str(), AvpBlockedState ? "true" : "false"));

   _bdAddressOfAvpBlockedDevice = (true == AvpBlockedState) ? (bdAddress) : ("");
}//!End of void BtLimitationController::UpdateAvpBlockedStateForDevice(..)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::onConflictsResolved()
 **************************************************************************/
void BtLimitationController::onConflictsResolved(IN const UserDecisionInfo& userDecision)
{
   ETG_TRACE_USR1(("onConflictsResolved: UserDecision = %d", ETG_CENUM(UserDecision, userDecision._userDecision)));

   Result result(CC_ERR_INT_NO_ERROR);
   char eventParams[EVENT_PARAMETERS_SIZE] = {};

   result = ParameterCONFLICTS_RESOLVED(eventParams, sizeof(eventParams), userDecision._userDecision);

   if (CC_ERR_INT_NO_ERROR == result)
   {
      ETG_TRACE_USR4(("onConflictsResolved: Sending event CONFLICTS_RESOLVED to self SM"));

      result = SendEventByName("CONFLICTS_RESOLVED", eventParams);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("onConflictsResolved: Could not send event CONFLICTS_RESOLVED (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));

         result = sendSmEvent("ResolveConflicts", "FAILED");
      }//End of if (CC_ERR_INT_NO_ERROR != result)
   }//End of if (CC_ERR_INT_NO_ERROR == result)
   else
   {
      ETG_TRACE_ERR(("onConflictsResolved: Could not marshal event parameters for event CONFLICTS_RESOLVED (error = %d)",
            ETG_CENUM(CcErrorInternal, result)));

      result = sendSmEvent("ResolveConflicts", "FAILED");
   }//End of else
}//!End of void BtLimitationController::onConflictsResolved(IN const UserDecisionInfo& userDecision)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::updateWblSupportedFrequencies()
 **************************************************************************/
void BtLimitationController::updateWblSupportedFrequencies(IN BTSWblSupportedRestrictionEntryList& supportedFrequencies)
{
   ETG_TRACE_USR1(("updateWblSupportedFrequencies: supportedRestrictionEntryList Size = %d", supportedFrequencies.size()));

   _wblOperatingFreq = BTS_WBL_WIFI_FREQUENCY_LAST;
   BTSWblSupportedRestrictionEntryList::iterator suppResEntryListItr;
   BTSWblSupportedRestrictionList::iterator suppResListItr;
   BTSWblSupportedTechnologyList::iterator suppTechListItr;

   for(suppResEntryListItr = supportedFrequencies.begin(); suppResEntryListItr != supportedFrequencies.end();
         suppResEntryListItr++)
   {
      BTSWblSupportedRestrictionList supportedRestrictionList = suppResEntryListItr->supportedRestrictions;

      ETG_TRACE_USR2(("updateWblSupportedFrequencies: supportedRestrictionList Size = %d", supportedRestrictionList.size()));

      for(suppResListItr = supportedRestrictionList.begin(); suppResListItr != supportedRestrictionList.end();
            suppResListItr++)
      {
         BTSWblSupportedTechnologyList supportedTechnologyList = suppResListItr->technologies;

         ETG_TRACE_USR2(("updateWblSupportedFrequencies: supportedTechnologyList Size = %d", supportedTechnologyList.size()));
         ETG_TRACE_USR2(("updateWblSupportedFrequencies: supportedFreq = %d", ETG_CENUM(BTSWblWifiFrequency, suppResListItr->frequency)));

         //!Iterate through the technology list
         for(suppTechListItr = supportedTechnologyList.begin(); suppTechListItr != supportedTechnologyList.end();
               suppTechListItr++)
         {
            ETG_TRACE_USR2(("updateWblSupportedFrequencies: APType = %d, RestrictionSetting = %d",
                  ETG_CENUM(BTSWblAccessPointConfigType, suppTechListItr->type),
                  ETG_CENUM(BTSWblSupportedSetting, suppTechListItr->setting)));

            if((BTS_WBL_WIFI_FREQUENCY_5_GHZ == suppResListItr->frequency) &&
                  (BTS_WBL_AP_CONFIG_CPW == suppTechListItr->type) &&
                  (BTS_WBL_SUPPORTED_SETTING_ALLOWED == suppTechListItr->setting))
            {
               _wblOperatingFreq = BTS_WBL_WIFI_FREQUENCY_5_GHZ;
               break;
            }//!End of if((BTS_WBL_WIFI_FREQUENCY_5_GHZ == suppResListItr->frequency) &&
            else if((BTS_WBL_WIFI_FREQUENCY_2_4_GHZ == suppResListItr->frequency) &&
                  (BTS_WBL_AP_CONFIG_CPW == suppTechListItr->type) &&
                  (BTS_WBL_SUPPORTED_SETTING_ALLOWED == suppTechListItr->setting))
            {
               _wblOperatingFreq = BTS_WBL_WIFI_FREQUENCY_2_4_GHZ;
            }//!End of else if((BTS_WBL_WIFI_FREQUENCY_2_4_GHZ == suppResListItr->frequency) &&
         }//End of for(suppTechListItr = supportedTechnologyList.begin(); suppTechListItr != supportedTechnologyList.end();..)

         //!Break from 2nd for loop as we are not interested in further iterations if 5GHz is supported
         if(BTS_WBL_WIFI_FREQUENCY_5_GHZ == _wblOperatingFreq)
         {
            break;
         }//!End of if(BTS_WBL_WIFI_FREQUENCY_5_GHZ == _wblOperatingFreq)
      }//!End of (suppResListItr = supportedRestrictionList.begin(); ...)

      //!Break from 1st for loop as we are not interested in further iterations if 5GHz is supported
      if(BTS_WBL_WIFI_FREQUENCY_5_GHZ == _wblOperatingFreq)
      {
         break;
      }//!End of if(BTS_WBL_WIFI_FREQUENCY_5_GHZ == _wblOperatingFreq)
   }//!End of  for(suppResEntryListItr = supportedFrequencies.begin(); ...)

   //!If valid data not available from WBL, assign default of 2.4GHz
   _wblOperatingFreq = (BTS_WBL_WIFI_FREQUENCY_LAST == _wblOperatingFreq) ? (BTS_WBL_WIFI_FREQUENCY_2_4_GHZ) : (_wblOperatingFreq);

   ETG_TRACE_USR4(("updateWblSupportedFrequencies: wblOperatingFreq = %d", ETG_CENUM(BTSWblWifiFrequency, _wblOperatingFreq)));
}//!End of void BtLimitationController::updateWblSupportedFrequencies(IN BTSWblSupportedRestrictionEntryList& supportedFrequencies)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::messageNotConsumed()
 **************************************************************************/
Result BtLimitationController::messageNotConsumed()
{
   ENTRY

   ETG_TRACE_USR1(("messageNotConsumed: entered"));

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::messageNotConsumed()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::initSm()
 **************************************************************************/
Result BtLimitationController::initSm()
{
   ENTRY

   ETG_TRACE_USR1(("initSm"));

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::initSm()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::handleStopSm()
 **************************************************************************/
Result BtLimitationController::handleStopSm()
{
   ENTRY

   ETG_TRACE_USR1(("handleStopSm"));

   /* Send stop done to SPM in the transition to final state in state machine */
   return stopDone(0);
}//!End of Result BtLimitationController::handleStopSm()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::handleDoneSm()
 **************************************************************************/
Result BtLimitationController::handleDoneSm()
{
   ENTRY

   ETG_TRACE_USR1(("handleDoneSm: entered"));

   /* Send done done to SPM in the transition to final state in state machine */
   return doneDone(0);
}//!End of Result BtLimitationController::handleDoneSm()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterActivating(...)
 **************************************************************************/
Result BtLimitationController::enterActivating()
{
   ETG_TRACE_USR1(("enterActivating: entered"));

   //!Update BT limitation mode status to registered clients.
   LocalSpm::getBmCoreMainController().updateBtLimitationMode(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName,
         _CurrentLimitationMode, BM_LIMITATION_STATE_ACTIVATING);

   switch(_CurrentLimitationMode._limitationFeature)
   {
      case BM_LIMITATION_FEATURE_CAR_PLAY:
      {
         if((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
               ||(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf))
         {
            (void) sendSmEvent("enterActivating", "ACTIVATECPW");
         }
         else
         {
            (void) sendSmEvent("enterActivating", "ACTIVATECP");
         }
      }
      break;
      case BM_LIMITATION_FEATURE_ANDROID_AUTO:
      {
         (void) sendSmEvent("enterActivating", "ACTIVATEHANDSFREEPROFILE");
      }
      break;
      case BM_LIMITATION_FEATURE_CAR_LIFE:
      case BM_LIMITATION_FEATURE_ON_CAR:
      {
         if(BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)
         {
            (void) sendSmEvent("enterActivating", "ACTIVATEHANDSFREEPROFILE");
         }
      }
      break;
      case BM_LIMITATION_FEATURE_MIRROR_LINK:
      case BM_LIMITATION_FEATURE_MY_SPIN_IOS:
      case BM_LIMITATION_FEATURE_CAR_LIFE_IOS:
      {
         if(BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)
         {
            (void) sendSmEvent("enterActivating", "ACTIVATEMANUALHFP");
         }
      }
      break;
      default:
      {
         ETG_TRACE_USR4(("enterActivating: Default case. Invalid Limitation Feature = %d ",
               ETG_CENUM(LimitationFeature, _CurrentLimitationMode._limitationFeature)));
      }
   }
   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterActivating()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterActive(...)
 **************************************************************************/
Result BtLimitationController::enterActive()
{
   ETG_TRACE_USR1(("enterActive: entered"));

   Result result(CC_ERR_INT_NO_ERROR);

   //!Update BT limitation mode status to registered clients.
   LocalSpm::getBmCoreMainController().updateBtLimitationMode(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName,
         _CurrentLimitationMode, BM_LIMITATION_STATE_ACTIVE);

   if(((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
         ||(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf)) &&
         (BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature))
   {
      //Once the CPW session is active, send DEVICE_CONNECTION_FINISHED signal to AutoconnectionController
      //to proceed further for connect the other device in the list
      ETG_TRACE_USR4(("enterActive(deviceId = %d): sending event DEVICE_CONNECTION_FINISHED to AutoConnectionController SM", _ProjectionDeviceBTDeviceId));

      char eventParams[20];
      result = LocalSpm::getAutoConnectionController().ParameterDEVICE_CONNECTION_FINISHED(eventParams, sizeof(eventParams), _ProjectionDeviceBTDeviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         result = LocalSpm::getAutoConnectionController().SendEventByName("DEVICE_CONNECTION_FINISHED", eventParams);

         if(CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("enterActive(deviceId = %d): could not send event DEVICE_CONNECTION_FINISHED (error = %d)",
                  _ProjectionDeviceBTDeviceId, ETG_CENUM(CcErrorInternal, result)));
         }//!End of if(CC_ERR_INT_NO_ERROR != result)
      }
      else
      {
         ETG_TRACE_ERR(("enterActive(deviceId = %d): could not marshal event parameters for event DEVICE_CONNECTION_FINISHED (error = %d)",
               _ProjectionDeviceBTDeviceId, ETG_CENUM(CcErrorInternal, result)));
      }//!End of if(CC_ERR_INT_NO_ERROR != result)

      //Set the CPW flag for the projection device in the database. It will be used for Autoconnection to connect the CPW device first.
      (void) LocalSpm::getDbManager().setCPWDevice(_ProjectionDeviceBTDeviceId);
   }//!End of if((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf) ...)

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterActive()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterDeactivateWaiting(...)
 **************************************************************************/
Result BtLimitationController::enterDeactivateWaiting()
{
   ETG_TRACE_USR1(("enterDeactivateWaiting: entered"));

   _LockCPWReconnectionTimer.lock();

   //! Stop and Cancel timer if timer is active.
   if(0 != _CPWReconnectionTimerID)
   {
      _CPWReconnectionTimer.CancelTimer(_CPWReconnectionTimerID);
      _CPWReconnectionTimerID = 0;
   }//End of if(0 != _CPWReconnectionTimerID)

   _LockCPWReconnectionTimer.unlock();

   if((BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature) &&
         ((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
               ||(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf)))
   {
      //Reset the CPW flag in the database for the device once it is deactivated by User.
      bool resetCPWDevice = true;
      (void) LocalSpm::getDbManager().setCPWDevice(_ProjectionDeviceBTDeviceId, resetCPWDevice);

      defaultActionforDeactivateCPW(_ProjectionDeviceBTDeviceId, _ProjectionDeviceBDAddress);
   }//!End of if((BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature) && ...)

   //!Update BT limitation mode status to registered clients.
   LocalSpm::getBmCoreMainController().updateBtLimitationMode(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName,
         _CurrentLimitationMode, BM_LIMITATION_STATE_WAITING);


   //!Resolve conflicts internally if conflict resolution is in progress
   ConflictTrigger conflictTrigger = getConflictTriggerFromCurrLimiMode();
   UserDecisionInfo userDecision(conflictTrigger, BM_USER_DECISION_PROCEED);
   (void)LocalSpm::getConflictManager().processNotifyUserDecisionRequest(userDecision);

   UnblockProfiles();

   // Reset the members
   _ProtocolsToBeBlockedForProjectionDevice.clear();
   _ProtocolsToBeBlockedForNonProjectionDevices.clear();
   _deviceProtocolsWaitingToGetBlocked.clear();
   _deviceIdWaitingForBeingDisconnected = 0u;
   _deviceLost = false;

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterDeactivateWaiting()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::defaultActionforDeactivateCPW(...)
 **************************************************************************/
void BtLimitationController::defaultActionforDeactivateCPW(const DeviceId& projectionDeviceBTDeviceId, const BdAddress& projectionDeviceBDAddress)
{
   ETG_TRACE_USR1(("defaultActionforDeactivateCPW: entered"));

   Result result(CC_ERR_INT_NO_ERROR);
   ProtocolList protocolList;

   //!Step1: Get all the protocols for the projection device which is in CONNECTED/CONNECTING and disconnect SPP profile of projection device
   result = LocalSpm::getDbManager().getDeviceProtocolsConnectedStatus(protocolList, projectionDeviceBTDeviceId);

   if (CC_ERR_INT_NO_ERROR == result)
   {
      // Check the major device protocols which is in CONNECTING/CONNECTED.
      // If no major protocol is available then disconnect IAP over SPP profile internally
      for (size_t idx = 0u; idx < protocolList.size(); idx++)
      {
         if((BM_PROTOCOL_ID_HFP == protocolList[idx]._protocolId) || (BM_PROTOCOL_ID_AVP == protocolList[idx]._protocolId)
               || (BM_PROTOCOL_ID_PAN == protocolList[idx]._protocolId))
         {
            break;
         }//!End of if((BM_PROTOCOL_ID_HFP == protocolList[idx]._protocolId) || ...)
         else if((BM_PROTOCOL_ID_SPP == protocolList[idx]._protocolId) && (IAP2BT_SPP_UUID == protocolList[idx]._uuid))
         {
            ProtocolList protocolList;
            protocolList.push_back(Protocol(BM_PROTOCOL_ID_SPP, IAP2BT_SPP_UUID));

            (void) LocalSpm::getBmCoreMainController().disconnectProfilesInt(projectionDeviceBTDeviceId, protocolList);
         }//! End of else if((BM_PROTOCOL_ID_SPP == protocolList[idx]._protocolId) && ...)
      }//!End of for (size_t idx = 0u; idx < protocolList.size(); idx++)
   }//End of if(CC_ERR_INT_NO_ERROR == result)

   //!Step2:Get the UsagePreference of the projection device
   //!and set the usagePreference as TEMP_CLASSIC_BT if UsagePreference is UNDECIDED
   UsagePreference usagePreference;
   result = LocalSpm::getDbManager().getDeviceUsagePreference(usagePreference, projectionDeviceBTDeviceId);

   if(CC_ERR_INT_NO_ERROR == result)
   {
      if(BM_UP_UNDECIDED == usagePreference)
      {
         (void) LocalSpm::getDbManager().setDeviceUsagePreference(projectionDeviceBTDeviceId, BM_UP_TEMP_CLASSIC_BT);
      }//!End of if(BM_UP_UNDECIDED == usagePreference)
      else
      {
         ETG_TRACE_USR4(("defaultActionforDeactivateCPW: usagepreference = %d is already set for device - %d",
               ETG_CENUM(UsagePreference, usagePreference), projectionDeviceBTDeviceId));
      }//!End of else
   }//End of if(CC_ERR_INT_NO_ERROR == result)

   //!Step3: Send CPW_ENDED event to assigned DCC to unreserve the DCC is use for CPW
   DeviceConnectionController* dccInstance(0);
   result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, projectionDeviceBDAddress);

   if (CC_ERR_INT_NO_ERROR == result)
   {
      ETG_TRACE_USR4(("defaultActionforDeactivateCPW(bdAddress = \"%50s\"): sending event CPW_ENDED to DCC instance 0x%p",
            projectionDeviceBDAddress.c_str(), (void *) dccInstance));
      result = dccInstance->SendEventByName("CPW_ENDED", 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("defaultActionforDeactivateCPW(bdAddress = \"%50s\"): could not send event CPW_ENDED (error = %d)",
               projectionDeviceBDAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }//!End of if (CC_ERR_INT_NO_ERROR != result)
   }//!End of if (CC_ERR_INT_NO_ERROR == result)
   else
   {
      ETG_TRACE_ERR(("defaultActionforDeactivateCPW(bdAddress = \"%50s\"): could not get DCC instance assigned to the projection device (error = %d)",
            projectionDeviceBDAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));
   }//!End of else

   //!Step4: Set the HU connectable mode to default
   (void)LocalSpm::getBmCoreMainController().switchLocalConnectableModeInt
         (LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode, "");
}//!End of void BtLimitationController::defaultActionforDeactivateCPW()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterDeactivating(...)
 **************************************************************************/
Result BtLimitationController::enterDeactivating()
{
   ETG_TRACE_USR1(("enterDeactivating: entered"));

   //!Update BT limitation mode status to registered clients.
   LocalSpm::getBmCoreMainController().updateBtLimitationMode(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName,
         _CurrentLimitationMode, BM_LIMITATION_STATE_DEACTIVATING);

   switch(_CurrentLimitationMode._limitationFeature)
   {
      case BM_LIMITATION_FEATURE_CAR_PLAY:
      {
         if((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
               ||(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf))
         {
            (void) sendSmEvent("enterDeactivating", "DEACTIVATECPW");
         }
         else
         {
            (void) sendSmEvent("enterDeactivating", "DEACTIVATECP");
         }
      }
      break;
      case BM_LIMITATION_FEATURE_ANDROID_AUTO:
      {
         (void) sendSmEvent("enterDeactivating", "DEACTIVATEHANDSFREEPROFILE");
      }
      break;
      case BM_LIMITATION_FEATURE_CAR_LIFE:
      case BM_LIMITATION_FEATURE_ON_CAR:
      {
         if(BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)
         {
            (void) sendSmEvent("enterDeactivating", "DEACTIVATEHANDSFREEPROFILE");
         }
      }
      break;
      case BM_LIMITATION_FEATURE_MIRROR_LINK:
      case BM_LIMITATION_FEATURE_MY_SPIN_IOS:
      case BM_LIMITATION_FEATURE_CAR_LIFE_IOS:
      {
         if(BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)
         {
            (void) sendSmEvent("enterDeactivating", "DEACTIVATEMANUALHFP");
         }
      }
      break;
      default:
      {
         ETG_TRACE_USR4(("enterDeactivating: Default case. Invalid Limitation Feature = %d ",
               ETG_CENUM(LimitationFeature, _CurrentLimitationMode._limitationFeature)));
      }
   }
   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterDeactivating()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterDeactivatingHandsFreeProfile(...)
 **************************************************************************/
Result BtLimitationController::enterDeactivatingHandsFreeProfile()
{
   ETG_TRACE_USR1(("enterDeactivatingHandsFreeProfile: entered"));

   //Remove restriction of pairable/connectable mode
   LocalSpm::getBmCoreMainController().setDeviceBdAddrForRestrictedPairingConnecting("");

   //!Rollback if de-activation is triggered while AA is in the process of activation.
   (void)LocalSpm::getBmCoreMainController().switchLocalPairableModeInt(
         LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode, "");

   (void)LocalSpm::getBmCoreMainController().switchLocalConnectableModeInt(
         LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode, "");

   (void) UnblockProfiles();

   (void)sendSmEvent("enterDeactivatingHandsFreeProfile", "DEACTIVATED");

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterDeactivatingHandsFreeProfile()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterDeactivatingManualHFP(...)
 **************************************************************************/
Result BtLimitationController::enterDeactivatingManualHFP()
{
   ETG_TRACE_USR1(("enterDeactivatingManualHFP: entered"));

   (void) UnblockProfiles();

   (void)sendSmEvent("enterDeactivatingManualHFP", "DEACTIVATED");

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterDeactivatingManualHFP()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterDeactivatingCP(...)
 **************************************************************************/
Result BtLimitationController::enterDeactivatingCP()
{
   ETG_TRACE_USR1(("enterDeactivatingCP: entered"));

   (void) UnblockProfiles();

   (void) sendSmEvent("enterDeactivatingCP", "DEACTIVATED");

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterDeactivatingCP()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterDeactivatingCPW(...)
 **************************************************************************/
Result BtLimitationController::enterDeactivatingCPW()
{
   ETG_TRACE_USR1(("enterDeactivatingCPW: entered"));

   (void)UnblockProfiles();

   //Reset the CPW flag in the database for the device once it is deactivated by User.
   bool resetCPWDevice = true;
   (void) LocalSpm::getDbManager().setCPWDevice(_ProjectionDeviceBTDeviceId, resetCPWDevice);

   // check autoconnection is running for the projection device.
   DeviceId deviceIdConnectionInProgress = LocalSpm::getAutoConnectionController().getDeviceIdConnectionInProgress();

   if(deviceIdConnectionInProgress == _ProjectionDeviceBTDeviceId)
   {
      // Irrespective of the usage preference trigger connectProfiles for the projection Device if autoconnection
      // is running for this device
      BmResult bmResult = LocalSpm::getBmCoreMainController().connectProfilesInt(_ProjectionDeviceBTDeviceId,
            LocalSpm::getAutoConnectionController().getDeviceProtocolListConnectionInProgress(), LocalSpm::getAutoConnectionController().getPageTimeout());

      if (BM_RESULT_OK != bmResult)
      {
         ETG_TRACE_ERR(("enterDeactivatingCPW: could not initiate connect profiles (bmResult = %d) for deviceId = %d", bmResult, _ProjectionDeviceBTDeviceId));
      }
   }
   else
   {
      ETG_TRACE_USR4(("enterDeactivatingCPW(deviceId = %d): Autoconection is processing different device - %d",
            _ProjectionDeviceBTDeviceId, deviceIdConnectionInProgress));
   }

   (void) sendSmEvent("enterDeactivatingCPW", "DEACTIVATED");

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterDeactivatingCPW()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterDeactivatingInternal(...)
 **************************************************************************/
Result BtLimitationController::enterDeactivatingInternal()
{
   ETG_TRACE_USR1(("enterDeactivatingInternal: entered"));

   //!Update BT limitation mode status to registered clients.
   LocalSpm::getBmCoreMainController().updateBtLimitationMode(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName,
         _CurrentLimitationMode, BM_LIMITATION_STATE_DEACTIVATING);

   (void) UnblockProfiles();

   (void) sendSmEvent("enterDeactivatingInternal", "DEACTIVATED");

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterDeactivatingInternal()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterDeactivatingSpmStateOff(...)
 **************************************************************************/
Result BtLimitationController::enterDeactivatingSpmStateOff()
{
   ETG_TRACE_USR1(("enterDeactivatingSpmStateOff: entered"));

   //TODO: Prioritize projection device for auto connection.

   //!Update BT limitation mode status to registered clients.
   LocalSpm::getBmCoreMainController().updateBtLimitationMode(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName,
         _CurrentLimitationMode, BM_LIMITATION_STATE_DEACTIVATING);

   (void)UnblockProfiles();

   if((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
         ||(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf))
   {
      if(BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature)
      {
         //TODO:Should we reconnect iAP via SPP
         //Do we start autoconnection during short ignition cycle.?

      }//End of if(BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature)
   }//End of if((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 ==...)

   else if (BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)
   {
      if(BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature)
      {
         //Should Autoconnection be performed here?
      }//End of if(BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature)
      else if(IS_LIMIFEATURE_FOR_HANDSFREE_PROFILE(_CurrentLimitationMode._limitationFeature))
      {
         //Remove restriction of pairable/connectable mode
         LocalSpm::getBmCoreMainController().setDeviceBdAddrForRestrictedPairingConnecting("");

         //!Rollback if de-activation Spmoff is triggered while AA is in the process of activation.
         (void)LocalSpm::getBmCoreMainController().switchLocalPairableModeInt(
               LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode, "");

         (void)LocalSpm::getBmCoreMainController().switchLocalConnectableModeInt(
               LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode, "");
      }//End of else if(BM_LIMITATION_FEATURE_ANDROID_AUTO == _CurrentLimitationMode._limitationFeature) ...)
   }//End of else if (BM_LIMITATION_COMMUNICATION_IF_USB == ...)

   (void) sendSmEvent("enterDeactivatingSpmStateOff", "DEACTIVATED");

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterDeactivatingSpmStateOff()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterError(...)
 **************************************************************************/
Result BtLimitationController::enterError()
{
   ETG_TRACE_USR1(("enterError: entered"));

   //!Update BT limitation mode status to registered clients.
   (BM_LIMITATION_STATE_ERROR == selimitationErrorType) ?
         (LocalSpm::getBmCoreMainController().updateBtLimitationMode(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName,
               _CurrentLimitationMode, BM_LIMITATION_STATE_ERROR)):
               (LocalSpm::getBmCoreMainController().updateBtLimitationMode(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName,
                     _CurrentLimitationMode, BM_LIMITATION_STATE_ERROR_USER_DENIED));

   selimitationErrorType = BM_LIMITATION_STATE_ERROR;

   (void) UnblockProfiles();

   (void) sendSmEvent("enterError", "DEACTIVATED");

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterError()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterIdle(...)
 **************************************************************************/
Result BtLimitationController::enterIdle()
{
   ETG_TRACE_USR1(("enterIdle(bdAddress = \"%50s\", bdName = \"%50s\"): entered", _ProjectionDeviceBDAddress.c_str(), _ProjectionDeviceBDName.c_str()));

   _LockCPWReconnectionTimer.lock();
   //! Stop and Cancel timer if timer is active.
   if(0 != _CPWReconnectionTimerID)
   {
      _CPWReconnectionTimer.CancelTimer(_CPWReconnectionTimerID);
      _CPWReconnectionTimerID = 0;
   }//End of if(0 != _CPWReconnectionTimerID)
   _LockCPWReconnectionTimer.unlock();

   //!Resolve conflicts internally if conflict resolution is in progress
   ConflictTrigger conflictTrigger = getConflictTriggerFromCurrLimiMode();
   UserDecisionInfo userDecision(conflictTrigger, BM_USER_DECISION_PROCEED);
   (void)LocalSpm::getConflictManager().processNotifyUserDecisionRequest(userDecision);

   DeviceId projectionDeviceBTDeviceId = _ProjectionDeviceBTDeviceId;
   BdAddress projectionDeviceBDAddress = _ProjectionDeviceBDAddress;
   BdAddress projectionDeviceBDName = _ProjectionDeviceBDName;
   LimitationMode currentLimitationMode = _CurrentLimitationMode;

   //! Reset all member variables
   _ProjectionDeviceBDAddress = "";
   _ProjectionDeviceBDName = "";
   _CurrentLimitationMode._limitationFeature = BM_LIMITATION_FEATURE_LAST;
   _CurrentLimitationMode._limitationCommunicationIf = BM_LIMITATION_COMMUNICATION_IF_LAST;
   _CurrentLimitationAction = BM_LIMITATION_ACTION_LAST;
   _ProjectionDeviceBTDeviceId = 0u;
   //!No need to reset _wblOperatingFreq as we might not get update from WBL if no change in the Wi-Fi mode

   _ProtocolsToBeBlockedForProjectionDevice.clear();
   _ProtocolsToBeBlockedForNonProjectionDevices.clear();
   _deviceProtocolsWaitingToGetBlocked.clear();
   _deviceIdWaitingForBeingDisconnected = 0u;
   _deviceLost = false;

   //!Perform default action if limitation mode for CPW was present.
   if ((BM_LIMITATION_FEATURE_CAR_PLAY == currentLimitationMode._limitationFeature) &&
         ((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == currentLimitationMode._limitationCommunicationIf)
               || (BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == currentLimitationMode._limitationCommunicationIf)))
   {
      defaultActionforDeactivateCPW(projectionDeviceBTDeviceId, projectionDeviceBDAddress);
   } //!End of if ((BM_LIMITATION_FEATURE_CAR_PLAY == limitationMode._limitationFeature) ...)

   if ((false == projectionDeviceBDAddress.empty()) || (false == projectionDeviceBDName.empty()))
   {
      //!Update BT limitation mode status to registered clients.
      LocalSpm::getBmCoreMainController().updateBtLimitationMode(projectionDeviceBDAddress, projectionDeviceBDName,
            currentLimitationMode, BM_LIMITATION_STATE_IDLE);

      //Delete the entry from the LimitationMode List updated to the clients
      LocalSpm::getBmCoreMainController().deleteBtLimitationModeItem(projectionDeviceBDAddress, projectionDeviceBDName, currentLimitationMode);

      bool isLimitationModeActive = (((BM_LIMITATION_FEATURE_CAR_PLAY == currentLimitationMode._limitationFeature) ||
            (BM_LIMITATION_FEATURE_ANDROID_AUTO == currentLimitationMode._limitationFeature) ||
            (BM_LIMITATION_FEATURE_CAR_LIFE == currentLimitationMode._limitationFeature) ||
            (BM_LIMITATION_FEATURE_CAR_LIFE_IOS == currentLimitationMode._limitationFeature) ||
            (BM_LIMITATION_FEATURE_ON_CAR == currentLimitationMode._limitationFeature) ||
            (BM_LIMITATION_FEATURE_MIRROR_LINK == currentLimitationMode._limitationFeature) ||
            (BM_LIMITATION_FEATURE_MY_SPIN_IOS == currentLimitationMode._limitationFeature)) &&
            (BM_LIMITATION_COMMUNICATION_IF_USB == currentLimitationMode._limitationCommunicationIf));

      if ((true == isLimitationModeActive) && (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._enableDefaultActionAtEndOfBtLimitation)
            && (true == LocalSpm::getBmCoreMainController().isFunctionalityPermitted(BM_RESTRICTION_GROUP_06)))
      {
         // After ending the BtLimitationMode, BMCore starts the autoconnection and connect device as CLASSIC_BT(ignore the usage preference of that device)
         ETG_TRACE_USR4(("enterIdle: sending event AUTO_CONNECTION_LITE to AutoConnectionControllerSm"));
         Result result = LocalSpm::getAutoConnectionController().SendEventByName("AUTO_CONNECTION_LITE", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("enterIdle: could not send event AUTO_CONNECTION_LITE (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }

         (void) InitiateAutoConnection();
      }//!End of if ((true == isLimitationModeActive) ...)

   }//!End of if (false == _ProjectionDeviceBDAddress.empty() ...)
   // else, it was the initial entering of the IDLE state and BT limitation mode property must not be updated

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterIdle()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::checkRequestedLimitationFeature(...)
 **************************************************************************/
Result BtLimitationController::checkLimitationFeatureToBePrepared()
{
   ETG_TRACE_USR1(("checkLimitationFeatureToBePrepared: entered"));

   //!Update the Limitation Mode comm Intf based on WBL operating frequency
   _CurrentLimitationMode._limitationCommunicationIf = (BTS_WBL_WIFI_FREQUENCY_5_GHZ == _wblOperatingFreq) ?
         (BM_LIMITATION_COMMUNICATION_IF_WIFI_5) : (BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4);

   if((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
         ||(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf))
   {
      if(BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature)
      {
         (void) sendSmEvent("checkLimitationFeatureToBePrepared", "PREPARECPW");
      }//End of if(BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature)

      //! For future Use
      /*else if(BM_LIMITATION_FEATURE_ANDROID_AUTO == _CurrentLimitationMode._limitationFeature)
      {
         (void) sendSmEvent("checkLimitationFeatureToBePrepared", "PREPAREAAW");
      }*/
   }//End of if((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 ==...)

   return CC_ERR_INT_NO_ERROR;
}

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterPreparing(...)
 **************************************************************************/
Result BtLimitationController::enterPreparing()
{
   ETG_TRACE_USR1(("enterPreparing: entered"));

   //!Update BT limitation mode status to registered clients.
   (void) LocalSpm::getBmCoreMainController().updateBtLimitationMode(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName,
         _CurrentLimitationMode, BM_LIMITATION_STATE_PREPARING);

   return CC_ERR_INT_NO_ERROR;
}//! End of Result BtLimitationController::enterPreparing()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterPrepared(...)
 **************************************************************************/
Result BtLimitationController::enterPrepared()
{
   ETG_TRACE_USR1(("enterPrepared: entered"));

   //!Update BT limitation mode status to registered clients.
   LocalSpm::getBmCoreMainController().updateBtLimitationMode(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName,
         _CurrentLimitationMode, BM_LIMITATION_STATE_PREPARED);

   //!Reset the flag Once IAP over SPP is getting connected
   _deviceLost = false;

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterPrepared()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterPreactivating(...)
 **************************************************************************/
Result BtLimitationController::enterPreactivating()
{
   ETG_TRACE_USR1(("enterPreactivating: entered"));

   //!Update BT limitation mode status to registered clients.
   LocalSpm::getBmCoreMainController().updateBtLimitationMode(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName,
         _CurrentLimitationMode, BM_LIMITATION_STATE_PREACTIVATING);

   if((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
         ||(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf))
   {
      if(BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature)
      {
         (void) sendSmEvent("enterPreactivating", "PREACTIVATECPW");
      }//End of if(BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature)

      //! For future Use
      /*else if(BM_LIMITATION_FEATURE_ANDROID_AUTO == _CurrentLimitationMode._limitationFeature)
      {
         (void) sendSmEvent("enterPreactivating", "PREACTIVATEAAW");
      }*/
   }//End of if((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 ==...)

   return CC_ERR_INT_NO_ERROR;
}//End of Result BtLimitationController::enterPreactivating()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::enterPreactivated(...)
 **************************************************************************/
Result BtLimitationController::enterPreactivated()
{
   ETG_TRACE_USR1(("enterPreactivated: entered"));

   //!Update BT limitation mode status to registered clients.
   LocalSpm::getBmCoreMainController().updateBtLimitationMode(_ProjectionDeviceBDAddress, _ProjectionDeviceBDName,
         _CurrentLimitationMode, BM_LIMITATION_STATE_PREACTIVATED);

   return CC_ERR_INT_NO_ERROR;
}//!End of Result BtLimitationController::enterPreactivated()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::ConnectProfilesCPW(...)
 **************************************************************************/
Result BtLimitationController::ConnectProfilesCPW()
{
   ETG_TRACE_USR1(("ConnectProfilesCPW: entered"));

   Result result(CC_ERR_INT_NO_ERROR);
   ConnectionStatus connectionStatus = BM_CONNECTION_STATUS_UNKNOWN;
   DisconnectedReason disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;

   //!If projection device is paired,Check if SPP for iAP2 is already connected else request for connection
   if(0 != _ProjectionDeviceBTDeviceId)
   {
      result = LocalSpm::getDbManager().getProtocolConnectionStatus(connectionStatus, disconnectedReason,
            _ProjectionDeviceBTDeviceId, BM_PROTOCOL_ID_SPP, IAP2BT_SPP_UUID);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         if(BM_CONNECTION_STATUS_CONNECTED == connectionStatus)
         {
            //reservce the Dcc
            reserveDccforCPW();
            result = sendSmEvent("ConnectProfilesCPW", "PREPARED");
         }//End of if(BM_CONNECTION_STATUS_CONNECTED == connectionStatus)

         else if ((BM_CONNECTION_STATUS_CONNECTED != connectionStatus)
               && (BM_CONNECTION_STATUS_CONNECTING != connectionStatus))
         {
            Protocol BTProtocol(BM_PROTOCOL_ID_SPP, IAP2BT_SPP_UUID);
            ProtocolList ProtocolsToBeConnected;
            BmResult bmResult(BM_RESULT_OK);
            ProtocolsToBeConnected.push_back(BTProtocol);

            bmResult = LocalSpm::getBmCoreMainController().connectProfilesInt(_ProjectionDeviceBTDeviceId, ProtocolsToBeConnected,
                  LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultConnectionPageTimeoutMilliSeconds);

            if(BM_RESULT_OK != bmResult)
            {
               result = CC_ERR_INT_GENERAL_ERROR;
               ETG_TRACE_ERR(("ConnectProfilesCPW(bdAddress = \"%50s\"): could not send Connect Profile request (error = %d)",
                     _ProjectionDeviceBDAddress.c_str(), ETG_CENUM(CcErrorInternal, bmResult)));

               result = sendSmEvent("ConnectProfilesCPW", "FAILED");
            }//End of if(CC_ERR_INT_NO_ERROR == result)
            else
            {
               //reservce the Dcc
               reserveDccforCPW();
            }
         }//End of else if ((BM_CONNECTION_STATUS_CONNECTED != connectionStatus)...)
      }//End of if (CC_ERR_INT_NO_ERROR == result)
      else
      {
         ETG_TRACE_ERR(("ConnectProfilesCPW(bdAddress = \"%50s\"): could not get protocol connection status from DB (error = %d)",
               _ProjectionDeviceBDAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));

         result = sendSmEvent("ConnectProfilesCPW", "FAILED");
      }//End of Else
   }//End of if(0 != _ProjectionDeviceBTDeviceId)
   else
   {
      ETG_TRACE_ERR(("ConnectProfilesCPW(bdAddress = \"%50s\"): device is not found in paired device list",
            _ProjectionDeviceBDAddress.c_str()));

      result = sendSmEvent("ConnectProfilesCPW", "FAILED");
   }//End of Else

   return CC_ERR_INT_NO_ERROR;
}//!End of BtLimitationController::ConnectProfilesCPW()


/*************************************************************************
 ** FUNCTION:  BtLimitationController::reserveDccforCPW(...)
 **************************************************************************/
void BtLimitationController::reserveDccforCPW()
{
   DeviceConnectionController* dccInstance(0);
   Result result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, _ProjectionDeviceBDAddress);

   if (CC_ERR_INT_NO_ERROR == result)
   {
      // check the DCC is already reserved for CPW. If not then send CPW_STARTED event to reserve Dcc
      if(false == dccInstance->isReservedforCPW())
      {
         ETG_TRACE_USR4(("reserveDccforCPW(bdAddress = \"%50s\"): sending event CPW_STARTED to DCC instance 0x%p",
               _ProjectionDeviceBDAddress.c_str(), (void *) dccInstance));
         result = dccInstance->SendEventByName("CPW_STARTED", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("reserveDccforCPW(bdAddress = \"%50s\"): could not send event CPW_STARTED (error = %d)",
                  _ProjectionDeviceBDAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_USR4(("reserveDccforCPW(bdAddress = \"%50s\"):  DCC instance 0x%p is already reserved",
               _ProjectionDeviceBDAddress.c_str(), (void *) dccInstance));
      }
   }
   else
   {
      ETG_TRACE_ERR(("reserveDccforCPW(bdAddress = \"%50s\"): could not get DCC instance assigned to the projection device (error = %d)",
            _ProjectionDeviceBDAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));
   }
}

/*************************************************************************
 ** FUNCTION:  BtLimitationController::DisconnectAndBlockProfilesCPW(...)
 **************************************************************************/
Result BtLimitationController::DisconnectAndBlockProfilesCPW()
{
   ETG_TRACE_USR1(("DisconnectAndBlockProfilesCPW: entered"));

   //checkAvpBlockStateOfProjectionDevice();

   //!Set profiles to be blocked for projection and nonprojection devices and call blockprotocols
   SetProtocolsToBeBlockedForCarPlay();

   Result result = BlockProfiles();
   if(CC_ERR_INT_NO_ERROR != result)
   {
      result = sendSmEvent("DisconnectAndBlockProfilesCPW", "FAILED");
   }//End of if(CC_ERR_INT_NO_ERROR != result)

   //!Even though remote connections are blocked since we have blocked required profiles for devices,
   //!ACL link can still be connected from Remote device. For this, Set the HU BT connectable state to false
   //!after blocking and disconnecting. Switch it back to ON in IDLE state
   (void) LocalSpm::getBmCoreMainController().switchLocalConnectableModeInt(TARGET_SWITCH_STATE_SWITCHED_OFF, "");

   return CC_ERR_INT_NO_ERROR;
}//!End of BtLimitationController::DisconnectAndBlockProfilesCPW()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::DisconnectAndBlockProfilesOnActivatingHandsFreeProfile(...)
 **************************************************************************/
Result BtLimitationController::DisconnectAndBlockProfilesOnActivatingHandsFreeProfile()
{
   ETG_TRACE_USR1(("DisconnectAndBlockProfilesOnActivatingHandsFreeProfile: entered"));

   //checkAvpBlockStateOfProjectionDevice();

   //!Set profiles to be blocked for projection and nonprojection devices and call blockprotocols
   SetProtocolsToBeBlockedForHandsFreeProfile();

   Result result = BlockProfiles();
   if(CC_ERR_INT_NO_ERROR != result)
   {
      result = sendSmEvent("DisconnectAndBlockProfilesOnActivatingHandsFreeProfile", "FAILED");
   }//End of if(CC_ERR_INT_NO_ERROR != result)

   return CC_ERR_INT_NO_ERROR;
}//!End of BtLimitationController::DisconnectAndBlockProfilesOnActivatingAAP()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::DisconnectAndBlockProfilesOnActivatingAAP(...)
 **************************************************************************/
Result BtLimitationController::DisconnectAndBlockProfilesOnActivatingManualHFP()
{
   ETG_TRACE_USR1(("DisconnectAndBlockProfilesOnActivatingManualHFP: entered"));

   //checkAvpBlockStateOfProjectionDevice();

   //!Set profiles to be blocked for projection and nonprojection devices and call blockprotocols
   SetProtocolsToBeBlockedForManualHFP();

   Result result = BlockProfiles();
   if(CC_ERR_INT_NO_ERROR != result)
   {
      result = sendSmEvent("DisconnectAndBlockProfilesOnActivatingManualHFP", "FAILED");
   }//End of if(CC_ERR_INT_NO_ERROR != result)

   return CC_ERR_INT_NO_ERROR;
}//!End of BtLimitationController::DisconnectAndBlockProfilesOnActivatingManualHFP()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::DisconnectAndBlockProfilesOnActivatingCP(...)
 **************************************************************************/
Result BtLimitationController::DisconnectAndBlockProfilesOnActivatingCP()
{
   ETG_TRACE_USR1(("DisconnectAndBlockProfilesOnActivatingCP: entered"));

   //Set profiles to be blocked for projection and nonprojection devices and call blockprotocols
   SetProtocolsToBeBlockedForCarPlayOnActivating();

   Result result = BlockProfiles();
   if(CC_ERR_INT_NO_ERROR != result)
   {
      result = sendSmEvent("DisconnectAndBlockProfilesOnActivatingCP", "FAILED");
   }//End of if(CC_ERR_INT_NO_ERROR != result)

   return CC_ERR_INT_NO_ERROR;
}//!End of BtLimitationController::DisconnectAndBlockProfilesOnActivatingCP()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::disconnectSppIap(...)
 **************************************************************************/
Result BtLimitationController::disconnectSppIap()
{
   ETG_TRACE_USR1(("disconnectSpp: entered"));

   ProtocolList protocolList;
   protocolList.push_back(Protocol(BM_PROTOCOL_ID_SPP, IAP2BT_SPP_UUID));

   (void) LocalSpm::getBmCoreMainController().disconnectProfilesInt(_ProjectionDeviceBTDeviceId, protocolList);

   return CC_ERR_INT_NO_ERROR;
}//!End of BtLimitationController::disconnectSppIap()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::unblockSppIap(...)
 **************************************************************************/
Result BtLimitationController::unblockSppIap()
{
   ETG_TRACE_USR1(("unblockSppIap: entered"));

   ProtocolList protocolList;
   protocolList.push_back(Protocol(BM_PROTOCOL_ID_SPP, IAP2BT_SPP_UUID));

   (void) LocalSpm::getBmCoreMainController().unblockProfilesInt(_ProjectionDeviceBTDeviceId, protocolList);

   return CC_ERR_INT_NO_ERROR;
}//!End of BtLimitationController::unblockSppIap()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::IsConflictResAndProfileBlockReqdOnPreparingCPW(...)
 **************************************************************************/
Result BtLimitationController::IsConflictResAndProfileBlockReqdOnPreparingCPW()
{
   ETG_TRACE_USR1(("IsConflictResAndProfileBlockReqdOnPreparingCPW: entered"));

   // Fix for Bug 912588
   return false;
}//!End of BtLimitationController::IsConflictResAndProfileBlockReqdOnPreparingCPW()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::IsConflictResAndProfileBlockReqdOnPreactivatingCPW(...)
 **************************************************************************/
Result BtLimitationController::IsConflictResAndProfileBlockReqdOnPreactivatingCPW()
{
   ETG_TRACE_USR1(("IsConflictResAndProfileBlockReqdOnPreactivatingCPW"));

   //!Check if conflict resolution and disconnect profiles is required based on commintf
   Result RetVal = (BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf) ?
         (true) : (false);

   return RetVal;
}//!End of BtLimitationController::IsConflictResAndProfileBlockReqdOnPreactivatingCPW()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::IsConflictResReqdOnActivatingCPW(...)
 **************************************************************************/
Result BtLimitationController::IsConflictResReqdOnActivatingCPW()
{
   ETG_TRACE_USR1(("IsConflictResReqdOnActivatingCPW"));

   // CARPLAY session is started over WIFI even though iPhone is connected via USB.
   // BMApp accepts the CARPLAY WIFI(ACTIVATE) request without PREPARE. So Check if conflict resolution
   // and disconnect profiles is required irrespective of commintf
   return true;
}//!End of BtLimitationController::IsConflictResReqdOnActivatingCPW()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::HandlePrepareEvent(...)
 **************************************************************************/
Result BtLimitationController::HandlePrepareEvent()
{
   //!Resolve conflicts internally if conflict resolution is in progress and CPW session ends due to Link Loss
   ETG_TRACE_USR1(("HandlePrepareEvent: entered"));

   UserDecisionInfo userDecision(BM_CONFLICT_TRIGGER_CP_WIRELESS, BM_USER_DECISION_PROCEED);

   (void)LocalSpm::getConflictManager().processNotifyUserDecisionRequest(userDecision);

   (void) sendSmEvent("HandlePrepareEvent", "PREPARE");

   return CC_ERR_INT_NO_ERROR;
}//!End of BtLimitationController::HandlePrepareEvent()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::ReconnectCPW(...)
 **************************************************************************/
Result BtLimitationController::ReconnectCPW()
{
   uint32_t CPWReconnectTimeOutInMilliSecs =
         (LocalSpm::getDataProvider().getBmCoreConfiguration()._CPWReconnectTimeoutInSecs * NUM_OF_MSEC_PER_SEC);

   ETG_TRACE_USR1(("ReconnectCPW entered: CPWReconnectTimeOutInMilliSecs - %d", CPWReconnectTimeOutInMilliSecs));

   //Start a timer to check if device comes in range and try reconnecting until timeout
   if (0 != CPWReconnectTimeOutInMilliSecs)
   {
      timer_t rTimerID = 0;
      //! Start timer and wait for the response from the component
      _CPWReconnectionTimer.StartTimer(rTimerID, CPWReconnectTimeOutInMilliSecs, 0, this,
            &BtLimitationController::CPWReconnectTimerCb, NULL);

      _LockCPWReconnectionTimer.lock();
      _CPWReconnectionTimerID = rTimerID;
      _LockCPWReconnectionTimer.unlock();
   }//End of if (0 != CPWReconnectTimeOutInMilliSecs)

   //!Set the flag Once Wifi is getting lost
   _deviceLost = true;

   this->ConnectProfilesCPW();

   return CC_ERR_INT_NO_ERROR;
}//!End of BtLimitationController::ReconnectCPW()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::ResolveConflicts(...)
 **************************************************************************/
Result BtLimitationController::ResolveConflicts()
{
   //Call conflict manager to resolve conflicts and set the conflict trigger based on the commchannl and tech
   //Currently triggering CONFLICTS_RESOLVED without invoking Conflict Manager
   ETG_TRACE_USR1(("BtLimitationController ResolveConflicts: entered"));

   ConflictTrigger conflictTrigger = getConflictTriggerFromCurrLimiMode();
   DeviceId deviceHandle = 0;

   if(0u != _ProjectionDeviceBTDeviceId)
   {
      (void)LocalSpm::getDbManager().getDeviceHandle(deviceHandle, _ProjectionDeviceBTDeviceId);
   }
   else
   {
      if(false == _ProjectionDeviceBDAddress.empty())
      {
         (void)LocalSpm::getDbManager().getDeviceHandle(deviceHandle, _ProjectionDeviceBDAddress);
      }
      else if(false == _ProjectionDeviceBDName.empty())
      {
         (void)LocalSpm::getBmCoreMainController().getDeviceHandleForBdName(deviceHandle, _ProjectionDeviceBDName);
      }
   }

   LocalSpm::getConflictManager().resolveConflicts(conflictTrigger, deviceHandle);

   return CC_ERR_INT_NO_ERROR;
}//!End of BtLimitationController::ResolveConflicts()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::getConflictTriggerFromCurrLimiMode(...)
 **************************************************************************/
ConflictTrigger BtLimitationController::getConflictTriggerFromCurrLimiMode()
{
   ETG_TRACE_USR1(("getConflictTriggerFromCurrLimiMode entered"));

   ConflictTrigger conflictTrigger = BM_CONFLICT_TRIGGER_NONE;

   switch(_CurrentLimitationMode._limitationFeature)
   {
      case BM_LIMITATION_FEATURE_CAR_PLAY:
      {
         conflictTrigger = (BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)
               ? (BM_CONFLICT_TRIGGER_CP_USB) : (BM_CONFLICT_TRIGGER_CP_WIRELESS);
      }
      break;
      case BM_LIMITATION_FEATURE_ANDROID_AUTO:
      {
         conflictTrigger = (BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)
               ? (BM_CONFLICT_TRIGGER_AAP_USB) : (BM_CONFLICT_TRIGGER_AAP_WIRELESS);
      }
      break;
      case BM_LIMITATION_FEATURE_CAR_LIFE:
      {
         conflictTrigger = BM_CONFLICT_TRIGGER_CARLIFE_USB;
      }
      break;
      case BM_LIMITATION_FEATURE_CAR_LIFE_IOS:
      {
         conflictTrigger = BM_CONFLICT_TRIGGER_CARLIFE_IOS_USB;
      }
      break;
      case BM_LIMITATION_FEATURE_ON_CAR:
      {
         conflictTrigger = BM_CONFLICT_TRIGGER_ONCAR_USB;
      }
      break;
      case BM_LIMITATION_FEATURE_MIRROR_LINK:
      {
         conflictTrigger = BM_CONFLICT_TRIGGER_MIRRORLINK_USB;
      }
      break;
      case BM_LIMITATION_FEATURE_MY_SPIN:
      {
         conflictTrigger = BM_CONFLICT_TRIGGER_MYSPIN_USB;
      }
      break;
      case BM_LIMITATION_FEATURE_MY_SPIN_IOS:
      {
         conflictTrigger = BM_CONFLICT_TRIGGER_MYSPIN_IOS_USB;
      }
      break;
      default:
      {
         break;
      }
   }

   ETG_TRACE_USR4(("getConflictTriggerFromCurrLimiMode : ConflictTrigger = %d ",
         ETG_CENUM(ConflictTrigger, conflictTrigger)));

   return conflictTrigger;
}//!End of ConflictTrigger BtLimitationController::getConflictTriggerFromCurrLimiMode()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::SwitchLocalPairableMode(void)
 **************************************************************************/
Result BtLimitationController::SwitchLocalPairableMode(void)
{
   ETG_TRACE_USR1(("SwitchLocalPairableMode: entered"));

   Result result(CC_ERR_INT_NO_ERROR);

   //! Projection device is already paired, check if it is HFP connected
   ConnectionStatus connectionStatus = BM_CONNECTION_STATUS_UNKNOWN;
   DisconnectedReason disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;
   result = LocalSpm::getDbManager().getProtocolConnectionStatus(connectionStatus, disconnectedReason,
         _ProjectionDeviceBTDeviceId, BM_PROTOCOL_ID_HFP, "");

   if (CC_ERR_INT_NO_ERROR == result)
   {
      if (BM_CONNECTION_STATUS_CONNECTED == connectionStatus)
      {
         result = sendSmEvent("SwitchLocalPairableMode", "PAIRABLE");
      }//End of if (BM_CONNECTION_STATUS_CONNECTED == connectionStatus)
      else if (BM_CONNECTION_STATUS_CONNECTED != connectionStatus)
      {
         result = SwitchPairableMode();
      }//End of else if (BM_CONNECTION_STATUS_CONNECTED != connectionStatus)
   }//End of if (CC_ERR_INT_NO_ERROR == result)

   //!Always switch  HU pairable mode as the pairing info on the remote device is not known to HU.
   //!Irrespective of the device is known or not for the HU.
   else
   {
      ETG_TRACE_ERR(("SwitchLocalPairableMode(bdAddress = \"%50s\"): could not get protocol connection "
            "status from DB (error = %d)", _ProjectionDeviceBDAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));

      result = SwitchPairableMode();
   }//End of else if else if((CC_ERR_INT_NO_ERROR != result) && ...))

   return CC_ERR_INT_NO_ERROR;
}//! End of Result BtLimitationController::SwitchLocalPairableMode(void)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::SwitchLocalConnectableMode(void)
 **************************************************************************/
Result BtLimitationController::SwitchLocalConnectableMode(void)
{
   ETG_TRACE_USR1(("SwitchLocalConnectableMode: entered"));

   Result result(CC_ERR_INT_NO_ERROR);

   //! Projection device is already paired, check if it is HFP connected
   ConnectionStatus connectionStatus = BM_CONNECTION_STATUS_UNKNOWN;
   DisconnectedReason disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;
   result = LocalSpm::getDbManager().getProtocolConnectionStatus(connectionStatus, disconnectedReason,
         _ProjectionDeviceBTDeviceId, BM_PROTOCOL_ID_HFP, "");

   if (CC_ERR_INT_NO_ERROR == result)
   {
      if (BM_CONNECTION_STATUS_CONNECTED == connectionStatus)
      {
         result = sendSmEvent("SwitchLocalConnectableMode", "CONNECTABLE");
      }//End of if (BM_CONNECTION_STATUS_CONNECTED == connectionStatus)
      else if (BM_CONNECTION_STATUS_CONNECTED != connectionStatus)
      {
         result = SwitchConnectableMode();
      }//End of else if (BM_CONNECTION_STATUS_CONNECTED != connectionStatus)
   }//End of if (CC_ERR_INT_NO_ERROR == result)

   //!Always switch HU connectable mode as the pairing info on the remote device is not known to HU.
   //!Irrespective of the device is known or not for the HU.
   else
   {
      ETG_TRACE_ERR(("SwitchLocalConnectableMode(bdAddress = \"%50s\"): could not get protocol connection "
            "status from DB (error = %d)", _ProjectionDeviceBDAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));

      result = SwitchConnectableMode();
   }//End of else if else if((CC_ERR_INT_NO_ERROR != result) && ...))

   return CC_ERR_INT_NO_ERROR;
}//! End of Result BtLimitationController::SwitchLocalConnectableMode(void)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::SwitchOnBT(void)
 **************************************************************************/
Result BtLimitationController::SwitchOnBT(void)
{
   ETG_TRACE_USR1(("SwitchOnBT: entered"));

   BmResult bmResult(BM_RESULT_OK);

   //Get the current BT switch status
   SwitchStatus btStatus(SWITCH_STATE_SWITCHED_OFF, SWITCHED_OFF_REASON_CLIENT);
   bmResult = LocalSpm::getBmCoreMainController().getBtStatus(btStatus);
   if (BM_RESULT_OK == bmResult)
   {
      //If BT status is already ON, trigger BT_ON to self SM
      if (SWITCH_STATE_SWITCHED_ON == btStatus._switchState)
      {
         (void) sendSmEvent("SwitchOnBT", "BT_ON");
      }//End of if (SWITCH_STATE_SWITCHED_ON == btStatus._switchState)
      else if(SWITCH_STATE_SWITCHING_ON != btStatus._switchState) /*If BT status is not ON/SWITCHING_ON, request to Switch ON*/
      {
         bmResult = LocalSpm::getBmCoreMainController().switchBtStatusInt(TARGET_SWITCH_STATE_SWITCHED_ON, true, false);
         if (BM_RESULT_OK != bmResult)
         {
            ETG_TRACE_ERR(("SwitchOnBT: could not send message to switch ON BT state (error = %d)",
                  ETG_CENUM(BmResult, bmResult)));
            (void) sendSmEvent("SwitchOnBT", "FAILED");
         }//End of if (BM_RESULT_OK != bmResult)
      }//end of else
   }//End of if (BM_RESULT_OK == bmResult)
   else
   {
      ETG_TRACE_ERR(("SwitchOnBT: could not get BT status from BM Main Controller (error = %d)",
            ETG_CENUM(BmResult, bmResult)));
      (void) sendSmEvent("SwitchOnBT", "FAILED");
   }//End of else

   return CC_ERR_INT_NO_ERROR;
}//! End of Result BtLimitationController::SwitchOnBT(void)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::ValidateConflictsStatus(...)
 **************************************************************************/
Result BtLimitationController::ValidateConflictsStatus(const UserDecision ConflictResolvedStatus)
{
   ETG_TRACE_USR1(("ValidateConflictsStatus: entered - ConflictResolvedStatus = %d",
         ETG_CENUM(UserDecision, ConflictResolvedStatus)));

   //Check the user decision and send true or false
   Result RetVal = (BM_USER_DECISION_PROCEED == ConflictResolvedStatus)? true : false;
   selimitationErrorType = (BM_USER_DECISION_PROCEED == ConflictResolvedStatus) ?
         selimitationErrorType : BM_LIMITATION_STATE_ERROR_USER_DENIED;

   return RetVal;
}//!End of BtLimitationController::ValidateConflictsStatus()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::create()
 **************************************************************************/
void BtLimitationController::create()
{
   ENTRY

   ETG_TRACE_USR1(("create: entered"));

   /* Create the state machine */
   BtLimitationSm::Create();

   createDone(0);
}//!End of void BtLimitationController::create()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::init()
 **************************************************************************/
Result BtLimitationController::init(InitReason reason)
{
   ENTRY

   ETG_TRACE_USR1(("init: entered - reason = %d", reason));

   (void) reason;

   /* Init the state machine */
   BtLimitationSm::Init();
   SetAnswerTimeout(2000);

   /* Register state machine with dispatcher */
   Dispatcher::GetInstance().Register(IN this);

   return initDone(0);
}//!End of Result BtLimitationController::init(InitReason reason)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::run()
 **************************************************************************/
Result BtLimitationController::run()
{
   ENTRY

   ETG_TRACE_USR1(("run: entered"));

   return runDone(0);
}//!End of Result BtLimitationController::run()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::stop()
 **************************************************************************/
Result BtLimitationController::stop()
{
   ENTRY
   ETG_TRACE_USR1(("stop: entered"));

   Result result(CC_ERR_INT_NO_ERROR);

   /* Send STOP message to own SM */
   result = SendForceEvent(STOP_SM, (char *) 0);
   FW_NORMAL_ASSERT(CC_ERR_INT_NO_ERROR == result);

   if (CC_ERR_INT_NO_ERROR != result)
   {
      ETG_TRACE_ERR(("stop: could not send event STOP_SM (error = %d)", result));
   }//End of if (CC_ERR_INT_NO_ERROR != result)

   return result;
}

/*************************************************************************
 ** FUNCTION:  BtLimitationController::done()
 **************************************************************************/
Result BtLimitationController::done()
{
   ENTRY

   ETG_TRACE_USR1(("done: entered"));

   Result result(CC_ERR_INT_NO_ERROR);

   /* Deregister state machine with dispatcher */
   Dispatcher::GetInstance().DeRegister(IN this);

   /* Send DONE message to own SM */
   result = SendForceEvent(DONE_SM, (char *) 0);
   FW_NORMAL_ASSERT(CC_ERR_INT_NO_ERROR == result);

   if (CC_ERR_INT_NO_ERROR != result)
   {
      ETG_TRACE_ERR(("done: could not send event DONE (error = %d)", result));
   }//End of if (CC_ERR_INT_NO_ERROR != result)

   return result;
}//!End of Result BtLimitationController::done()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::getSmStateName(OUT tGeneralString stateName, ...)
 **************************************************************************/
char* BtLimitationController::getSmStateName(OUT tGeneralString stateName, size_t size)
{
   ENTRY

   ETG_TRACE_USR1(("getSmStateName: entered"));

   GetCurrentState((char *) stateName, size);

   ETG_TRACE_USR4(("getSmStateName: state name = \"%s\"", stateName));

   return stateName;
}//!End of char* BtLimitationController::getSmStateName(OUT tGeneralString stateName, ...)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::getBTDeviceIdForProjectionDevice(void)
 **************************************************************************/
void BtLimitationController::getBTDeviceIdForProjectionDevice(void)
{
   ETG_TRACE_USR1(("getBTDeviceIdForProjectionDevice: entered with BD address = \"%50s\" and BD name = \"%50s\"",
         _ProjectionDeviceBDAddress.c_str(), _ProjectionDeviceBDName.c_str()));

   DeviceId deviceId(0u);
   Result result(CC_ERR_INT_NO_ERROR);

   if(false == _ProjectionDeviceBDAddress.empty())
   {
      result = LocalSpm::getDbManager().getDeviceId(deviceId, _ProjectionDeviceBDAddress);
   }
   else if(false == _ProjectionDeviceBDName.empty())
   {
      result = LocalSpm::getBmCoreMainController().getDeviceIdForBdName(deviceId, _ProjectionDeviceBDName);
   }

   if (CC_ERR_INT_NO_ERROR == result)
   {
      ETG_TRACE_USR2(("getBTDeviceIdForProjectionDevice: device with BD address = \"%50s\" or  BD name = \"%50s\" found "
            "in paired device list: device ID = %d", _ProjectionDeviceBDAddress.c_str(), _ProjectionDeviceBDName.c_str(), deviceId));
   }//End of if (CC_ERR_INT_NO_ERROR == result)
   else if (CC_ERR_INT_DB_END_OF_LIST == result)
   {
      ETG_TRACE_USR2(("getBTDeviceIdForProjectionDevice: device with BD address = \"%50s\" or  BD name = \"%50s\" not found in paired device list",
            _ProjectionDeviceBDAddress.c_str(), _ProjectionDeviceBDName.c_str()));
   }//End of else if (CC_ERR_INT_DB_END_OF_LIST == result)
   else
   {
      ETG_TRACE_ERR(("getBTDeviceIdForProjectionDevice: could not get device ID from DB (error = %d)",
            ETG_CENUM(CcErrorInternal, result)));
   }//End of else

   _ProjectionDeviceBTDeviceId = deviceId;
}//!End of  BtLimitationController::getBTDeviceIdForProjectionDevice(IN const bool updateCurrentDeviceId)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::sendSmEvent(...)
 **************************************************************************/
Result BtLimitationController::sendSmEvent(IN const string& callingMethodName, IN const string& eventName)
{
   ETG_TRACE_USR4(("%50s: sending event %50s to BtLimitationSm", callingMethodName.c_str(), eventName.c_str()));
   Result result = SendEventByName(eventName.c_str(), 0);

   if (CC_ERR_INT_NO_ERROR != result)
   {
      ETG_TRACE_ERR(("%50s: could not send event %50s to BtLimitationSm (error = %d)", callingMethodName.c_str(),
            eventName.c_str(), ETG_CENUM(CcErrorInternal, result)));
      FW_NORMAL_ASSERT(CC_ERR_INT_NO_ERROR == result);
   }//End of if (CC_ERR_INT_NO_ERROR != result)

   return result;
}//!End of const Result BtLimitationController::sendSmEvent(...)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::getSmEventNameByLimitationAction(...)
 **************************************************************************/
const string BtLimitationController::getSmEventNameByLimitationAction(const LimitationAction limitationAction) const
{
   ETG_TRACE_USR1(("getSmEventNameByLimitationAction Entered"));
   string smEventName("");

   switch (limitationAction)
   {
      case BM_LIMITATION_ACTION_PREPARE:
      {
         smEventName = "PREPARE";
      }
      break;
      case BM_LIMITATION_ACTION_PREACTIVATE:
      {
         smEventName = "PREACTIVATE";
      }
      break;
      case BM_LIMITATION_ACTION_ACTIVATE:
      {
         smEventName = "ACTIVATE";
      }
      break;
      case BM_LIMITATION_ACTION_DEACTIVATE:
      {
         smEventName = "DEACTIVATE";
      }
      break;
      case BM_LIMITATION_ACTION_DEACTIVATE_WAIT:
      {
         smEventName = "DEACTIVATE_WAIT";
      }
      break;
      case BM_LIMITATION_ACTION_DEACTIVATE_INTERNAL:
      {
         smEventName = "DEACTIVATE_INTERNAL";
      }
      break;
      case BM_LIMITATION_ACTION_DEACTIVATE_SPM_STATE_OFF:
      {
         smEventName = "DEACTIVATE_SPM_STATE_OFF";
      }
      break;
      default:
      {
         //smEventName = "UNKNOWN";
      }
      break;
   }
   return smEventName;
}//! End of BtLimitationController::getSmEventNameByLimitationAction()


/*************************************************************************
 ** FUNCTION:  BtLimitationController::getListOfProtocolsToBeBlocked(...)
 **************************************************************************/
void BtLimitationController::getListOfProtocolsToBeBlocked(OUT DeviceProtocolListMap& protocolsToBeBlockedForDevices)
{
   ETG_TRACE_USR1(("getListOfProtocolsToBeBlocked: entered (bdAddress = \"%50s\")", _ProjectionDeviceBDAddress.c_str()));

   Result result(CC_ERR_INT_NO_ERROR);
   bool ignoreSppIap(false);
   DeviceIdList allPairedDeviceIds;
   ProtocolList protocolsToBeBlocked;
   ProtocolInfoMap protocolInfoMap;
   UuidList uuidList;

   result = LocalSpm::getDbManager().getAllDeviceIds(allPairedDeviceIds);

   for (size_t devIdx = 0u; devIdx < allPairedDeviceIds.size(); ++devIdx)
   {
      //!Add list of protocols to be blocked for Projection devices
      if (allPairedDeviceIds[devIdx] == _ProjectionDeviceBTDeviceId)
      {
         for (size_t protIdx = 0u; protIdx < _ProtocolsToBeBlockedForProjectionDevice.size(); ++protIdx)
         {
            if (BM_PROTOCOL_ID_SPP != _ProtocolsToBeBlockedForProjectionDevice[protIdx]._protocolId)
            {
               result = LocalSpm::getDbManager().getDeviceSupportedProtocols(protocolInfoMap, allPairedDeviceIds[devIdx]);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  if (protocolInfoMap.end() != protocolInfoMap.find(_ProtocolsToBeBlockedForProjectionDevice[protIdx]._protocolId))
                  {
                     protocolsToBeBlocked.push_back(_ProtocolsToBeBlockedForProjectionDevice[protIdx]);
                  }//!End of if (protocolInfoMap.end() != protocolInfoMap.find(_ProtocolsToBeBlockedForProjectionDevice[protIdx]._protocolId))
               }//!End of if (CC_ERR_INT_NO_ERROR == result)
               else
               {
                  ETG_TRACE_ERR(("getListOfProtocolsToBeBlocked(bdAddress = \"%50s\"): could not get supported protocols for device ID = %d "
                        "from DB (error = %d) => ignoring this error", _ProjectionDeviceBDAddress.c_str(), allPairedDeviceIds[devIdx],
                        ETG_CENUM(CcErrorInternal, result)));

                  result = CC_ERR_INT_NO_ERROR;
               }//!End of else
            }//!End of if (BM_PROTOCOL_ID_SPP != _ProtocolsToBeBlockedForProjectionDevice[protIdx]._protocolId)
            else
            {
               ignoreSppIap = false;

               bool isWifiCommActive = (((BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature)
                     && (BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf))
                     || ((BM_LIMITATION_FEATURE_ANDROID_AUTO == _CurrentLimitationMode._limitationFeature)
                     && ((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
                     || (BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf))));

               if ((true == _ProtocolsToBeBlockedForProjectionDevice[protIdx]._uuid.empty()) && (true == isWifiCommActive))
               {
                  ignoreSppIap = true;
               }//!End of if ((true == _ProtocolsToBeBlockedForProjectionDevice[protIdx]._uuid.empty()) ...)

               result = LocalSpm::getDbManager().getDeviceSppUuidSupport(uuidList, allPairedDeviceIds[devIdx]);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  if (false == uuidList.empty())
                  {
                     for (size_t uuidIdx = 0u; uuidIdx < uuidList.size(); ++uuidIdx)
                     {
                        if ((false == ignoreSppIap) || ((IAP2BT_SPP_UUID != uuidList[uuidIdx])
                              && (AAW_SPP_UUID != uuidList[uuidIdx])))
                        {
                           protocolsToBeBlocked.push_back(Protocol(_ProtocolsToBeBlockedForProjectionDevice[protIdx]._protocolId,
                                 uuidList[uuidIdx]));
                        }//!End of if ((false == ignoreSppIap) || (IAP2BT_SPP_UUID != uuidList[uuidIdx]))
                     }//!End of for (size_t uuidIdx = 0u; uuidIdx < uuidList.size(); ++uuidIdx)

                     uuidList.clear();
                  }//!End of if (false == uuidList.empty())
               }//!End of  if (CC_ERR_INT_NO_ERROR == result)
               else
               {
                  ETG_TRACE_ERR(("getListOfProtocolsToBeBlocked(bdAddress = \"%50s\"): could not get supported SPP UUIDs "
                        "for device ID = %d from DB (error = %d)  => ignoring this error", _ProjectionDeviceBDAddress.c_str(),
                        allPairedDeviceIds[devIdx], ETG_CENUM(CcErrorInternal, result)));

                  result = CC_ERR_INT_NO_ERROR;
               }//!End of else
            }//!End of else

            protocolInfoMap.clear();
         }//!End of for (size_t protIdx = 0u; protIdx < _ProtocolsToBeBlockedForProjectionDevice.size(); ++protIdx)
      }//!End of if (allPairedDeviceIds[devIdx] == _ProjectionDeviceBTDeviceId)

      //!Add list of protocols to be blocked for Non-Projection devices
      else
      {
         for (size_t protIdx = 0u; protIdx < _ProtocolsToBeBlockedForNonProjectionDevices.size(); ++protIdx)
         {
            if (BM_PROTOCOL_ID_SPP != _ProtocolsToBeBlockedForNonProjectionDevices[protIdx]._protocolId)
            {
               result = LocalSpm::getDbManager().getDeviceSupportedProtocols(protocolInfoMap, allPairedDeviceIds[devIdx]);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  if (protocolInfoMap.end() != protocolInfoMap.find(_ProtocolsToBeBlockedForNonProjectionDevices[protIdx]._protocolId))
                  {
                     protocolsToBeBlocked.push_back(_ProtocolsToBeBlockedForNonProjectionDevices[protIdx]);
                  }//!End of if (protocolInfoMap.end() != protocolInfoMap.find(_ProtocolsToBeBlockedForNonProjectionDevices[protIdx]._protocolId))
               }//!End of if (CC_ERR_INT_NO_ERROR == result)
               else
               {
                  ETG_TRACE_ERR(("getListOfProtocolsToBeBlocked(bdAddress = \"%50s\"): could not get supported protocols "
                        "for device ID = %d from DB (error = %d) => ignoring this error", _ProjectionDeviceBDAddress.c_str(),
                        allPairedDeviceIds[devIdx], ETG_CENUM(CcErrorInternal, result)));

                  result = CC_ERR_INT_NO_ERROR;
               }//!End of else
            }//!End of if (BM_PROTOCOL_ID_SPP != _ProtocolsToBeBlockedForNonProjectionDevices[protIdx]._protocolId)
            else
            {
               result = LocalSpm::getDbManager().getDeviceSppUuidSupport(uuidList, allPairedDeviceIds[devIdx]);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  if (false == uuidList.empty())
                  {
                     for (size_t uuidIdx = 0u; uuidIdx < uuidList.size(); ++uuidIdx)
                     {
                        protocolsToBeBlocked.push_back(Protocol(_ProtocolsToBeBlockedForNonProjectionDevices[protIdx]._protocolId,
                              uuidList[uuidIdx]));
                     }//!End of for (size_t uuidIdx = 0u; uuidIdx < uuidList.size(); ++uuidIdx)

                     uuidList.clear();
                  }//!End of if (false == uuidList.empty())
               }//!End of if (CC_ERR_INT_NO_ERROR == result)
               else
               {
                  ETG_TRACE_ERR(("getListOfProtocolsToBeBlocked(bdAddress = \"%50s\"): could not get supported SPP UUIDs "
                        "for device ID = %d from DB (error = %d)  => ignoring this error", _ProjectionDeviceBDAddress.c_str(),
                        allPairedDeviceIds[devIdx], ETG_CENUM(CcErrorInternal, result)));

                  result = CC_ERR_INT_NO_ERROR;
               }//!End of else
            }//!End of else

            protocolInfoMap.clear();
         }//!End of for (size_t protIdx = 0u; protIdx < _ProtocolsToBeBlockedForNonProjectionDevices.size() ..)
      }//!End of else

      if (false == protocolsToBeBlocked.empty())
      {
         protocolsToBeBlockedForDevices[allPairedDeviceIds[devIdx]] = protocolsToBeBlocked;

         protocolsToBeBlocked.clear();
      }//!End of if (false == protocolsToBeBlocked.empty())
   }//!End of for (size_t devIdx = 0u; devIdx < allPairedDeviceIds.size(); ++devIdx)
}//!End of void BtLimitationController::getListOfProtocolsToBeBlocked(OUT DeviceProtocolListMap& protocolsToBeBlockedForDevices)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::BlockProfiles(void)
 **************************************************************************/
Result BtLimitationController::BlockProfiles(void)
{
   ETG_TRACE_USR1(("BlockProfiles: entered (bdAddress = \"%50s\")", _ProjectionDeviceBDAddress.c_str()));

   Result result(CC_ERR_INT_NO_ERROR);
   BmResult bmResult(BM_RESULT_OK);
   DeviceProtocolListMap deviceProtocolsToBeBlocked;

   //!For AA ,CarLife & CPW, first check if DCC is available for HFP/SPP connection else remember to wait until
   //!DCC gets released due to subsequent protocol blocking
   result = CheckIfDccIsAvailableForConnection();

   //!Get list of protocols to be blocked for both Projection & Non-Projection devices.
   getListOfProtocolsToBeBlocked(deviceProtocolsToBeBlocked);

   ConnectionStatus connectionStatus = BM_CONNECTION_STATUS_UNKNOWN;
   DisconnectedReason disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;

   _deviceProtocolsWaitingToGetBlocked.clear();

   for (DeviceProtocolListMap::iterator it = deviceProtocolsToBeBlocked.begin();
         it != deviceProtocolsToBeBlocked.end(); ++it)
   {
      for (size_t protIdx = 0u; protIdx < it->second.size(); ++protIdx)
      {
         result = LocalSpm::getDbManager().getProtocolConnectionStatus(connectionStatus, disconnectedReason,
               it->first, it->second[protIdx]._protocolId, it->second[protIdx]._uuid);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            if (BM_CONNECTION_STATUS_DISCONNECTED != connectionStatus)
            {
               _deviceProtocolsWaitingToGetBlocked.push_back(DeviceProtocol(it->first, it->second[protIdx]));
            }//End of if (BM_CONNECTION_STATUS_DISCONNECTED != connectionStatus)
         }//End of if (CC_ERR_INT_NO_ERROR == result)
         else if (CC_ERR_INT_DB_END_OF_LIST == result)
         {
            ETG_TRACE_ERR(("BlockProfiles(bdAddress = \"%50s\"): could not find protocol = (%d, \"%50s\") for device ID = %d in DB "
                  "(error = %d) => not treated as an error", _ProjectionDeviceBDAddress.c_str(),
                  ETG_CENUM(ProtocolId, it->second[protIdx]._protocolId),
                  it->second[protIdx]._uuid.c_str(),
                  it->first,
                  ETG_CENUM(CcErrorInternal, result)));

            result = CC_ERR_INT_NO_ERROR;
         }//!End of else if (CC_ERR_INT_DB_END_OF_LIST == result)
         else
         {
            ETG_TRACE_ERR(("BlockProfiles(bdAddress = \"%50s\"): could not get protocol connection status from DB (error = %d) => ignoring this error",
                  _ProjectionDeviceBDAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));

            result = CC_ERR_INT_NO_ERROR;
         }//End of else
      }//end of for (size_t protIdx = 0u; protIdx < it->second.size(); ++protIdx)
   }//end of for (DeviceProtocolListMap::iterator it = deviceProtocolsToBeBlocked.begin(); it != deviceProtocolsToBeBlocked.end(); ++it)

   traceDeviceProtocolList("BlockProfiles", _deviceProtocolsWaitingToGetBlocked);

   //!Get the ordered list from the list of protocols to be blocked
   //!Disconnection of protocols should be in the reverse order as shown to user.
   //!E.g . First DevA is HFP connected and then DevB is AVP connected. DevA is listed first and then DevB and disconnection
   //!of protocols should be in reverse order (First DevB and then DevA).
   //!This will help to autoconnect/reconnect the devices next time based on the connection index and same list will be seen by the user.
   //!Next time Devices will be autoconnected based on the disconnection index(Dev A first and then DevB)
   DeviceProtocolListMap ordereddeviceProtocolsToBeBlocked;
   result = LocalSpm::getDbManager().getOrderedDeviceProtocolListMap(ordereddeviceProtocolsToBeBlocked, deviceProtocolsToBeBlocked);

   for (DeviceProtocolListMap::iterator it = ordereddeviceProtocolsToBeBlocked.begin();
         it != ordereddeviceProtocolsToBeBlocked.end(); ++it)
   {
      bmResult = LocalSpm::getBmCoreMainController().blockProfilesInt(it->first, it->second, true);

      if (BM_RESULT_OK != bmResult)
      {
         result = CC_ERR_INT_GENERAL_ERROR;

         ETG_TRACE_ERR(("BlockProfiles(bdAddress = \"%50s\"): blocking protocols failed for device ID = %d (error = %d) => ignoring this error",
               _ProjectionDeviceBDAddress.c_str(), it->first, ETG_CENUM(CcErrorInternal, result)));

         result = CC_ERR_INT_NO_ERROR;
      }//End of if (BM_RESULT_OK != bmResult)
   }//!End of for (DeviceProtocolListMap::iterator it = ordereddeviceProtocolsToBeBlocked.begin(); ...)

   if ((true == _deviceProtocolsWaitingToGetBlocked.empty()) && (0u == _deviceIdWaitingForBeingDisconnected)
         && (CC_ERR_INT_NO_ERROR == result))
   {
      result = sendSmEvent("BlockProfiles", "PROFILES_BLOCKED");
   }//End of if ((true == _deviceProtocolsWaitingToGetBlocked.empty()) && ...)

   return result;
}//!End of Result BtLimitationController::BlockProfiles(void)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::UnblockProfiles(...)
 **************************************************************************/
Result BtLimitationController::UnblockProfiles(void)
{
   ETG_TRACE_USR1(("UnblockProfiles(bdAddress = \"%50s\", bdName = \"%50s\")", _ProjectionDeviceBDAddress.c_str(), _ProjectionDeviceBDName.c_str()));

   Result result(CC_ERR_INT_NO_ERROR);
   BmResult bmResult(BM_RESULT_OK);

   if (0u == _ProjectionDeviceBTDeviceId)
   {
      ETG_TRACE_USR1(("UnblockProfiles(bdAddress = \"%50s\", bdName = \"%50s\"): projection device ID is still 0, trying to update it",
            _ProjectionDeviceBDAddress.c_str(), _ProjectionDeviceBDName.c_str()));

      (void) this->getBTDeviceIdForProjectionDevice();

      if (0u == _ProjectionDeviceBTDeviceId)
      {
         ETG_TRACE_USR1(("UnblockProfiles(bdAddress = \"%50s\", bdName = \"%50s\"): projection device ID is still 0, no profiles will be unblocked for projection device",
               _ProjectionDeviceBDAddress.c_str(), _ProjectionDeviceBDName.c_str()));
      }
   }

   DeviceIdList allPairedDeviceIds;
   result = LocalSpm::getDbManager().getAllDeviceIds(allPairedDeviceIds);

   ProtocolInfoMap protocolInfoMap;
   ProtocolList protocolsToBeUnblocked;
   UuidList uuidList;
   DeviceProtocolListMap deviceProtocolsToBeUnblocked;

   for (size_t devIdx = 0u; devIdx < allPairedDeviceIds.size(); ++devIdx)
   {
      result = LocalSpm::getDbManager().getDeviceSupportedProtocols(protocolInfoMap,
            allPairedDeviceIds[devIdx]);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         for (ProtocolInfoMap::iterator it = protocolInfoMap.begin(); it != protocolInfoMap.end(); ++it)
         {
            if (BM_PROTOCOL_ID_AVP == it->first)
            {
               //! If Apple device is used for Audio streaming via iAP2, AVP profile is switched OFF by BM.
               //! Therefore we keep track of AVP Blocked status before and during Limitation mode activation
               //! and keep it blocked even after the limitation mode is deactivated.
               BdAddress bdAddress = "";
               (void) LocalSpm::getDbManager().getBdAddress(bdAddress, allPairedDeviceIds[devIdx]);

               if (bdAddress == _bdAddressOfAvpBlockedDevice)
               {
                  // Keep AVP protocol blocked for projection device as it was blocked before Limitation Mode was activated or
                  // during the Limitation mode was Active.
                  ETG_TRACE_USR1(("UnblockProfiles(bdAddress = \"%50s\", bdName = \"%50s\"): keeping AVP protocol blocked for projection device",
                        _ProjectionDeviceBDAddress.c_str(), _ProjectionDeviceBDName.c_str()));
               }
               else
               {
                  protocolsToBeUnblocked.push_back(Protocol(it->first, ""));
               }
            }
            else if (BM_PROTOCOL_ID_SPP == it->first)
            {
               result = LocalSpm::getDbManager().getDeviceSppUuidSupport(uuidList, allPairedDeviceIds[devIdx]);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  if (false == uuidList.empty())
                  {
                     for (size_t uuidIdx = 0u; uuidIdx < uuidList.size(); ++uuidIdx)
                     {
                        protocolsToBeUnblocked.push_back(Protocol(it->first, uuidList[uuidIdx]));
                     }

                     uuidList.clear();
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("UnblockProfiles(bdAddress = \"%50s\", bdName = \"%50s\"): could not get supported SPP UUIDs for device ID = %d from DB (error = %d)  => ignoring this error",
                        _ProjectionDeviceBDAddress.c_str(), _ProjectionDeviceBDName.c_str(), allPairedDeviceIds[devIdx],
                        ETG_CENUM(CcErrorInternal, result)));

                  result = CC_ERR_INT_NO_ERROR;
               }
            }
            else
            {
               protocolsToBeUnblocked.push_back(Protocol(it->first, ""));
            }
         } // end of for (ProtocolInfoMap::iterator it = protocolInfoMap.begin(); it != protocolInfoMap.end(); ++it)

         protocolInfoMap.clear();
      }
      else
      {
         ETG_TRACE_ERR(("UnblockProfiles(bdAddress = \"%50s\", bdName = \"%50s\"): could not get supported protocols for device ID = %d from DB (error = %d) => ignoring this error",
               _ProjectionDeviceBDAddress.c_str(), _ProjectionDeviceBDName.c_str(), allPairedDeviceIds[devIdx],
               ETG_CENUM(CcErrorInternal, result)));

         result = CC_ERR_INT_NO_ERROR;
      }

      if (false == protocolsToBeUnblocked.empty())
      {
         deviceProtocolsToBeUnblocked[allPairedDeviceIds[devIdx]] = protocolsToBeUnblocked;

         protocolsToBeUnblocked.clear();
      }
   } // end of for (size_t devIdx = 0u; devIdx < allPairedDeviceIds.size(); ++devIdx)

   if(CC_ERR_INT_NO_ERROR == result)
   {
      for (DeviceProtocolListMap::iterator it = deviceProtocolsToBeUnblocked.begin();
            it != deviceProtocolsToBeUnblocked.end(); ++it)
      {
         bmResult = LocalSpm::getBmCoreMainController().unblockProfilesInt(it->first, it->second);

         if (BM_RESULT_OK != bmResult)
         {
            result = CC_ERR_INT_GENERAL_ERROR;

            ETG_TRACE_ERR(("UnblockProfiles(bdAddress = \"%50s\", bdName = \"%50s\"): unblocking protocols failed for device ID = %d (error = %d) => ignoring this error",
                  _ProjectionDeviceBDAddress.c_str(), _ProjectionDeviceBDName.c_str(), it->first, ETG_CENUM(CcErrorInternal, result)));

            result = CC_ERR_INT_NO_ERROR;
         }//End of if (BM_RESULT_OK != bmResult)
      }
   }

   return result;
}//!End of Result BtLimitationController::unblockProtocols(void)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::SwitchPairableMode(void)
 **************************************************************************/
Result BtLimitationController::SwitchPairableMode(void)
{
   ETG_TRACE_USR4(("SwitchPairableMode: BD address = %s", _ProjectionDeviceBDAddress.c_str()));

   Result result(CC_ERR_INT_NO_ERROR);
   BmResult bmResult(BM_RESULT_OK);

   //Get the current local pairable mode from BM Main Controller
   SwitchStatus localPairableMode(SWITCH_STATE_SWITCHED_OFF, SWITCHED_OFF_REASON_CLIENT);
   bmResult = LocalSpm::getBmCoreMainController().getLocalPairableMode(localPairableMode);

   //If the mode is already switched ON send Pairable Event.
   if (BM_RESULT_OK == bmResult)
   {
      if (SWITCH_STATE_SWITCHED_ON == localPairableMode._switchState)
      {
         LocalSpm::getBmCoreMainController().setDeviceBdAddrForRestrictedPairingConnecting(_ProjectionDeviceBDAddress);
         result = sendSmEvent("SwitchPairableMode", "PAIRABLE");
      }//End of if (SWITCH_STATE_SWITCHED_ON == localPairableMode._switchState)

      //!Local Pairable Mode is not SWITCHING_ON, Switch it ON.
      //!If the status is SWITCHING_ON, appropriate events will be triggered on callback onLocalPairableModeChanged
      else if (SWITCH_STATE_SWITCHING_ON != localPairableMode._switchState)
      {
         bmResult = LocalSpm::getBmCoreMainController().switchLocalPairableModeInt(TARGET_SWITCH_STATE_SWITCHED_ON,
               _ProjectionDeviceBDAddress);
         if (BM_RESULT_OK != bmResult)
         {
            //Todo: Conflicts need to be triggered for MAXIMUM_NUMBER_PAIRED_DEVICES_REACHED
            //!In case max devices is reached and not able to switch on LocalPairableMode, still we transit to Active state.
            //!On activating limitation Mode, HMI shows pop-up(Incase of Renault) to delete any device from paired list
            //!and make place for AA device.
            //!In case of Nissan place will be already made for AA device when plugged in as we have common device list.
            if(BM_RESULT_ERR_MAXIMUM_NUMBER_PAIRED_DEVICES_REACHED == bmResult)
            {
               ETG_TRACE_ERR(("SwitchPairableMode: Max number of Paired device is reached (error = %d)",
                     ETG_CENUM(BmResult, bmResult)));

               result = sendSmEvent("SwitchPairableMode", "PAIRABLE");
            }
            //!If any other error received other than BM_RESULT_ERR_MAXIMUM_NUMBER_PAIRED_DEVICES_REACHED
            else
            {
               ETG_TRACE_ERR(("SwitchPairableMode: could not send message to switch ON Local Pairable Mode (error = %d)",
                     ETG_CENUM(BmResult, bmResult)));
               result = sendSmEvent("SwitchPairableMode", "FAILED");
            }
         }//End of if (BM_RESULT_OK != bmResult)
      }//End of else if (SWITCH_STATE_SWITCHING_ON != ...)
   }//End of if (BM_RESULT_OK == bmResult)
   else
   {
      ETG_TRACE_ERR(("SwitchPairableMode: could not get Local Pairable status from BM Main Controller (error = %d)",
            ETG_CENUM(BmResult, bmResult)));
      result = sendSmEvent("SwitchPairableMode", "FAILED");
   }//End of else

   return result;
}//!End of Result BtLimitationController::SwitchPairableMode(void)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::SwitchConnectableMode(void)
 **************************************************************************/
Result BtLimitationController::SwitchConnectableMode(void)
{
   ETG_TRACE_USR4(("SwitchConnectableMode: BD address = %s", _ProjectionDeviceBDAddress.c_str()));

   Result result(CC_ERR_INT_NO_ERROR);
   BmResult bmResult(BM_RESULT_OK);

   //Get the current local ConnectableMode mode from BM Main Controller
   SwitchStatus localConnectableMode(SWITCH_STATE_SWITCHED_OFF, SWITCHED_OFF_REASON_CLIENT);
   bmResult = LocalSpm::getBmCoreMainController().getLocalConnectableMode(localConnectableMode);

   //If the mode is already switched ON send Connectable Event.
   if (BM_RESULT_OK == bmResult)
   {
      if (SWITCH_STATE_SWITCHED_ON == localConnectableMode._switchState)
      {
         LocalSpm::getBmCoreMainController().setDeviceBdAddrForRestrictedPairingConnecting(_ProjectionDeviceBDAddress);
         result = sendSmEvent("SwitchConnectableMode", "CONNECTABLE");
      }//End of if (SWITCH_STATE_SWITCHED_ON == localConnectableMode._switchState)

      //!Local Connectable Mode is not SWITCHING_ON, Switch it ON.
      //!If the status is SWITCHING_ON, appropriate events will be triggered on callback onLocalConnectableModeChanged
      else if (SWITCH_STATE_SWITCHING_ON != localConnectableMode._switchState)
      {
         bmResult = LocalSpm::getBmCoreMainController().switchLocalConnectableModeInt(TARGET_SWITCH_STATE_SWITCHED_ON,
               _ProjectionDeviceBDAddress);
         if (BM_RESULT_OK != bmResult)
         {
            ETG_TRACE_ERR(("SwitchConnectableMode: could not send message to switch ON Local Connectable Mode (error = %d)",
                  ETG_CENUM(BmResult, bmResult)));
            result = sendSmEvent("SwitchConnectableMode", "FAILED");
         }//End of if (BM_RESULT_OK != bmResult)
      }//End of else if (SWITCH_STATE_SWITCHING_ON != ...)
   }//End of if (BM_RESULT_OK == bmResult)
   else
   {
      ETG_TRACE_ERR(("SwitchConnectableMode: could not get Local Connectable status from BM Main Controller (error = %d)",
            ETG_CENUM(BmResult, bmResult)));
      result = sendSmEvent("SwitchConnectableMode", "FAILED");
   }//End of else

   return result;
}//!End of Result BtLimitationController::SwitchConnectableMode(void)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::InitiateAutoConnection(void)
 **************************************************************************/
Result BtLimitationController::InitiateAutoConnection(void)
{
   Result result(CC_ERR_INT_NO_ERROR);

   char eventParams[EVENT_PARAMETERS_SIZE] = {};
   result = LocalSpm::getBmController().ParameterSTART_AUTO_CONNECTION(eventParams, sizeof(eventParams),
         BM_AUTOCONNECTION_START_MODE_BOOK_AND_TRY_ONCE);

   if (CC_ERR_INT_NO_ERROR == result)
   {
      ETG_TRACE_USR4(("InitiateAutoConnection(bdAddress = \"%50s\"): sending event START_AUTO_CONNECTION to BmControllerOnOffSm",
            _ProjectionDeviceBDAddress.c_str()));

      result = LocalSpm::getBmController().SendEventByName("START_AUTO_CONNECTION", eventParams);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("InitiateAutoConnection(bdAddress = \"%50s\"): could not send event START_AUTO_CONNECTION (error = %d)",
               _ProjectionDeviceBDAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }//End of if (CC_ERR_INT_NO_ERROR != result)
   }//End of if (CC_ERR_INT_NO_ERROR == result)
   else
   {
      ETG_TRACE_ERR(("InitiateAutoConnection(bdAddress = \"%50s\"): could not marshal event parameters for event START_AUTO_CONNECTION (error = %d)",
            _ProjectionDeviceBDAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));
   }//End of else

   FW_NORMAL_ASSERT(CC_ERR_INT_NO_ERROR == result);

   return result;
}//End of Result BtLimitationController::InitiateAutoConnection(void)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::SetProtocolsToBeBlockedForHandsFreeProfile(void)
 **************************************************************************/
void BtLimitationController::SetProtocolsToBeBlockedForHandsFreeProfile(void)
{
   ETG_TRACE_USR1(("SetProtocolsToBeBlockedForHandsFreeProfile: entered"));

   _ProtocolsToBeBlockedForProjectionDevice.clear();
   _ProtocolsToBeBlockedForNonProjectionDevices.clear();

   //!Set profiles to be blocked for projection and nonprojection devices and call blockprotocols
   if((BM_LIMITATION_FEATURE_ANDROID_AUTO == _CurrentLimitationMode._limitationFeature) &&
         ((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
         ||(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf)))
   {
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_PAN, ""));

      // SPP with empty UUID has a special meaning for projection device:
      // block SPP protocol for all UUIDs except of AAW UUID
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_SPP, ""));

      //! Set Protocols to be blocked for Non-Projection device
      if (BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
      {
         _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_HFP, ""));
         _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));
         _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_PBDL, ""));
         _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_MSG, ""));
         _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_PAN, ""));
         _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_SPP, ""));
      }
      else if(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf)
      {
         _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_HFP, ""));
         _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_PBDL, ""));
         _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_MSG, ""));

         //! For Wi-Fi 5 GHz, block AVP for Non-Projection devices based on project configuration
         if (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._blockAllProtocolsForNonProjectionDevices)
         {
            _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));
         }
      }
   }
   else
   {
      //! Set Protocols to be blocked for Projection device
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));

      //! Set HFP Protocol to be blocked for Non-Projection device
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_HFP, ""));
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_PBDL, ""));
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_MSG, ""));

      //! Blocking of AVP profile for Non-Projection devices is project specific configuration
      //! For USB, block AVP based on project configuration.
      if((BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)
            && (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._blockAllProtocolsForNonProjectionDevices))
      {
         _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));
      }//End of if((BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)
   }

   ETG_TRACE_USR4(("SetProtocolsToBeBlockedForHandsFreeProfile: Protocols to be blocked for Projection Device are"));
   VARTRACE(_ProtocolsToBeBlockedForProjectionDevice);

   ETG_TRACE_USR4(("SetProtocolsToBeBlockedForHandsFreeProfile: Protocols to be blocked for Non Projection Devices are"));
   VARTRACE(_ProtocolsToBeBlockedForNonProjectionDevices);
}//!End of BtLimitationController::SetProtocolsToBeBlockedForHandsFreeProfile()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::SetProtocolsToBeBlockedForManualHFP(void)
 **************************************************************************/
void BtLimitationController::SetProtocolsToBeBlockedForManualHFP(void)
{
   ETG_TRACE_USR1(("SetProtocolsToBeBlockedForManualHFP: entered"));

   _ProtocolsToBeBlockedForProjectionDevice.clear();
   _ProtocolsToBeBlockedForNonProjectionDevices.clear();

   //! Set Protocols to be blocked for Projection device
   _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));

   //! Set HFP Protocol to be blocked for Non-Projection device
   _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_HFP, ""));
   _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_PBDL, ""));
   _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_MSG, ""));

   //! Blocking of AVP profile for Non-Projection devices is project specific configuration
   //! For USB, block AVP based on project configuration.
   if((BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)
         && (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._blockAllProtocolsForNonProjectionDevices))
   {
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));
   }//End of if((BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)

   ETG_TRACE_USR4(("SetProtocolsToBeBlockedForManualHFP: Protocols to be blocked for Projection Device are"));
   VARTRACE(_ProtocolsToBeBlockedForProjectionDevice);

   ETG_TRACE_USR4(("SetProtocolsToBeBlockedForManualHFP: Protocols to be blocked for Non Projection Devices are"));
   VARTRACE(_ProtocolsToBeBlockedForNonProjectionDevices);
}//!End of BtLimitationController::SetProtocolsToBeBlockedForManualHFP()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::SetProtocolsToBeBlockedForCarPlay(void)
 **************************************************************************/
void BtLimitationController::SetProtocolsToBeBlockedForCarPlay(void)
{
   ETG_TRACE_USR1(("SetProtocolsToBeBlockedForCarPlay: entered"));

   _ProtocolsToBeBlockedForProjectionDevice.clear();
   _ProtocolsToBeBlockedForNonProjectionDevices.clear();

   //! Set Protocols to be blocked for Projection device for commIntf(Wi-Fi 2.4 GHz)
   if (BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
   {
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_HFP, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_PBDL, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_MSG, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_PAN, ""));

      // SPP with empty UUID has a special meaning for projection device:
      // block SPP protocol for all UUIDs except of iAP UUID
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_SPP, ""));
   }

   //! Set HFP Protocol to be blocked for Non-Projection device for commIntf(Wi-Fi 2.4 GHz)
   if (BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
   {
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_HFP, ""));
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_PBDL, ""));
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_MSG, ""));
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_PAN, ""));
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_SPP, ""));
   }

   ETG_TRACE_USR4(("SetProtocolsToBeBlockedForCarPlay: Protocols to be blocked for Projection Device are"));
   VARTRACE(_ProtocolsToBeBlockedForProjectionDevice);

   ETG_TRACE_USR4(("SetProtocolsToBeBlockedForCarPlay: Protocols to be blocked for Non Projection Devices are"));
   VARTRACE(_ProtocolsToBeBlockedForNonProjectionDevices);

}//!End of BtLimitationController::SetProtocolsToBeBlockedForCarPlay()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::SetProtocolsToBeBlockedForCarPlayOnActivating(void)
 **************************************************************************/
void BtLimitationController::SetProtocolsToBeBlockedForCarPlayOnActivating(void)
{
   ETG_TRACE_USR1(("SetProtocolsToBeBlockedForCarPlayOnActivating: entered"));

   _ProtocolsToBeBlockedForProjectionDevice.clear();
   _ProtocolsToBeBlockedForNonProjectionDevices.clear();

   //! Set Protocols to be blocked for Projection device
   if (BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
   {
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_HFP, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_PBDL, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_MSG, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_PAN, ""));

      // SPP with empty UUID has a special meaning for projection device:
      // block SPP protocol for all UUIDs except of iAP UUID
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_SPP, ""));

      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_SPP, IAP2BT_SPP_UUID));
   }
   else if (BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf)
   {
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_HFP, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_PBDL, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_MSG, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_PAN, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_SPP, ""));
   }
   else if (BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)
   {
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_HFP, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_PBDL, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_MSG, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_PAN, ""));
      _ProtocolsToBeBlockedForProjectionDevice.push_back(Protocol(BM_PROTOCOL_ID_SPP, ""));
   }

   //! Set Protocols to be blocked for Non-Projection device
   if (BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
   {
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_HFP, ""));
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_PBDL, ""));
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_MSG, ""));
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_PAN, ""));
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_SPP, ""));
   }
   else if(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf)
   {
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_HFP, ""));

      //! For Wi-Fi 5 GHz, block AVP for Non-Projection devices based on project configuration
      if (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._blockAllProtocolsForNonProjectionDevices)
      {
         _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));
      }
   }
   else if (BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)
   {
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_HFP, ""));

      //! For USB, block AVP for Non-Projection devices based on project configuration
      if (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._blockAllProtocolsForNonProjectionDevices)
      {
         _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_AVP, ""));
      }

      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_PBDL, ""));
      _ProtocolsToBeBlockedForNonProjectionDevices.push_back(Protocol(BM_PROTOCOL_ID_MSG, ""));
   }

   ETG_TRACE_USR4(("SetProtocolsToBeBlockedForCarPlayOnActivating: Protocols to be blocked for Projection Device are"));
   VARTRACE(_ProtocolsToBeBlockedForProjectionDevice);

   ETG_TRACE_USR4(("SetProtocolsToBeBlockedForCarPlayOnActivating: Protocols to be blocked for Non Projection Devices are"));
   VARTRACE(_ProtocolsToBeBlockedForNonProjectionDevices);

}//!End of BtLimitationController::SetProtocolsToBeBlockedForCarPlayOnActivating()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::IsLimitationActionRequestAllowed(...)
 **************************************************************************/
BmResult BtLimitationController::IsLimitationActionRequestAllowed(const LimitationState CurrentLimitationState,
      const LimitationAction NewLimitationActionReq)
{
   ETG_TRACE_USR1(("IsLimitationActionRequestAllowed: entered - CurrentLimitationState = %d, NewLimitationAction = %d",
         ETG_CENUM(LimitationState, CurrentLimitationState), ETG_CENUM(LimitationAction, NewLimitationActionReq)));

   BmResult bmResult(BM_RESULT_OK);
   bool LimiActionNotAllowedInCurrentState = false;

   LimiActionNotAllowedInCurrentState |= ((BM_LIMITATION_STATE_PREPARING == CurrentLimitationState) &&
         ((BM_LIMITATION_ACTION_ACTIVATE == NewLimitationActionReq) || (BM_LIMITATION_ACTION_PREACTIVATE == NewLimitationActionReq)));

   LimiActionNotAllowedInCurrentState |= ((BM_LIMITATION_STATE_PREACTIVATING == CurrentLimitationState) &&
         ((BM_LIMITATION_ACTION_PREPARE == NewLimitationActionReq) || (BM_LIMITATION_ACTION_ACTIVATE == NewLimitationActionReq)));

   //!Allow Prepare request while in Activating state for CPW
   LimiActionNotAllowedInCurrentState |= ((BM_LIMITATION_STATE_ACTIVATING == CurrentLimitationState) &&
         (/*(BM_LIMITATION_ACTION_PREPARE == NewLimitationActionReq) ||*/ (BM_LIMITATION_ACTION_PREACTIVATE == NewLimitationActionReq)));

   LimiActionNotAllowedInCurrentState |= ((BM_LIMITATION_STATE_ACTIVE == CurrentLimitationState) &&
         ((BM_LIMITATION_ACTION_PREACTIVATE == NewLimitationActionReq)));

   LimiActionNotAllowedInCurrentState |= ((BM_LIMITATION_STATE_DEACTIVATING == CurrentLimitationState) &&
         ((BM_LIMITATION_ACTION_DEACTIVATE_WAIT == NewLimitationActionReq)));

   LimiActionNotAllowedInCurrentState |= ((BM_LIMITATION_STATE_ERROR == CurrentLimitationState) &&
         ((BM_LIMITATION_ACTION_DEACTIVATE == NewLimitationActionReq) || (BM_LIMITATION_ACTION_DEACTIVATE_WAIT == NewLimitationActionReq)));

   if (true == LimiActionNotAllowedInCurrentState)
   {
      ETG_TRACE_ERR(("IsLimitationActionRequestAllowed: Requested BT limitation action "
            "is not allowed in current BT limitation state"));
      bmResult = BM_RESULT_ERR_BT_LIMITATION_ACTION_NOT_ALLOWED;
   }//End of if (true == LimiActionNotAllowedInCurrentState)

   return bmResult;
}//!End of BmResult BtLimitationController::IsLimitationActionRequestAllowed(...)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::traceDeviceProtocolList()
 **************************************************************************/
void BtLimitationController::traceDeviceProtocolList(IN const std::string& callingMethodName,
      IN const DeviceProtocolList& deviceProtocolList) const
{
   ETG_TRACE_USR1(("traceDeviceProtocolList: calling method to trace device protocol list = %s",
         callingMethodName.c_str()));

   for (size_t idx = 0u; idx < deviceProtocolList.size(); ++idx)
   {
      ETG_TRACE_USR4(("traceDeviceProtocolList: device ID = %d, protocol ID = %d, protocol UUID = %s",
            deviceProtocolList[idx]._deviceId, ETG_CENUM(ProtocolId, deviceProtocolList[idx]._protocol._protocolId),
            deviceProtocolList[idx]._protocol._uuid.c_str()));
   }//End of for (size_t idx = 0u; idx < deviceProtocolList.size(); ++idx)
}//End of void BtLimitationController::traceDeviceProtocolList(IN const std::string& callingMethodName...)

/*************************************************************************
 ** FUNCTION:  BtLimitationController::checkAvpBlockStateProjectionDevice(void)
 **************************************************************************/
void BtLimitationController::checkAvpBlockStateOfProjectionDevice()
{
   ETG_TRACE_USR1(("checkAvpBlockStateOfProjectionDevice: entered with (bdAddress = \"%50s\")", _ProjectionDeviceBDAddress.c_str()));

   bool isLimitationModeActive = ((BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature)
         && ((BM_LIMITATION_COMMUNICATION_IF_USB == _CurrentLimitationMode._limitationCommunicationIf)
               || (BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
               || (BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf)));

   // projection device is in paired device list, check AVP block state
   if ((0u != _ProjectionDeviceBTDeviceId) && (true == isLimitationModeActive))
   {
      BlockState avpBlockState(BM_BLOCK_STATE_UNBLOCKED);
      Result result = LocalSpm::getDbManager().getProtocolBlockState(avpBlockState,
            _ProjectionDeviceBTDeviceId, Protocol(BM_PROTOCOL_ID_AVP, ""));

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR4(("checkAvpBlockStateOfProjectionDevice(bdAddress = \"%50s\"): AVP block state for given projection device is %d",
               _ProjectionDeviceBDAddress.c_str(), ETG_CENUM(BlockState, avpBlockState)));

         _bdAddressOfAvpBlockedDevice = (BM_BLOCK_STATE_BLOCKED == avpBlockState) ? _ProjectionDeviceBDAddress : "";
      }//End of if (CC_ERR_INT_NO_ERROR == result)
      else
      {
         ETG_TRACE_ERR(("checkAvpBlockStateProjectionDevice(bdAddress = \"%50s\"): could not get AVP block state "
               "for given projection device (error = %d)", _ProjectionDeviceBDAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }//End of else
   }//End of if ((0u != _ProjectionDeviceBTDeviceId)) && (BM_LIMITATION_FEATURE_CAR_PLAY == ...)
}//!End of Result BtLimitationController::checkAvpBlockStateOfProjectionDevice()

/*************************************************************************
 ** FUNCTION:  BtLimitationController::InitiateAutoConnection(void)
 **************************************************************************/
Result BtLimitationController::CheckIfDccIsAvailableForConnection(void)
{
   Result result(CC_ERR_INT_NO_ERROR);

   ETG_TRACE_USR1(("CheckIfDccIsAvailableForConnection: entered"));

   //!If CarPly Feature is restricting BT over Wi-Fi communication Interface(SPP needs to be connected)
   //!Therefore DCC should be made available.
   bool isWiFiCommActive = ((BM_LIMITATION_FEATURE_CAR_PLAY == _CurrentLimitationMode._limitationFeature)
         && ((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == _CurrentLimitationMode._limitationCommunicationIf)
               || (BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == _CurrentLimitationMode._limitationCommunicationIf)));

   //!If AA/CarLife/OnCar Feature is restricting BT over USB communication Interface(HFP needs to be connected)
   //!Therefore DCC should be made available.
   //!Dcc should be made available only for projection devices which require connection of some profile like HFP/SPP.
   if((true == IS_LIMIMODE_FOR_HANDSFREE_PROFILE(_CurrentLimitationMode._limitationFeature, _CurrentLimitationMode._limitationCommunicationIf))
         || (true == isWiFiCommActive))
   {
      DeviceConnectionControllerList dccInstancesInUse;
      result = LocalSpm::getBmController().getUsedDeviceConnectionControllers(dccInstancesInUse);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         //!Has maximum number of simultaneously connected devices reached
         if (dccInstancesInUse.size() >= LocalSpm::getDataProvider().getBmCoreConfiguration()._maxNumConnectedDevices)
         {
            ETG_TRACE_USR4(("CheckIfDccIsAvailableForConnection(bdAddress = \"%50s\"): maximum number of simultaneously used DCC is reached",
                  _ProjectionDeviceBDAddress.c_str()));

            bool projectionDeviceInUse(false);

            for (size_t idx = 0u; idx < dccInstancesInUse.size(); ++idx)
            {
               if ((0u != _ProjectionDeviceBTDeviceId) && (dccInstancesInUse[idx]->getDeviceId() == _ProjectionDeviceBTDeviceId))
               {
                  ETG_TRACE_USR1(("CheckIfDccIsAvailableForConnection(bdAddress = \"%50s\"): one DCC is used to handle the projection device",
                        _ProjectionDeviceBDAddress.c_str()));

                  projectionDeviceInUse = true;
                  break;
               }//!End of if ((0u != _ProjectionDeviceBTDeviceId) && ...)
            }//!End of for (size_t idx = 0u; idx < dccInstancesInUse.size(); ++idx)

            if (true == projectionDeviceInUse)
            {
               // projection device is in use, do not wait for a device to get disconnected
               _deviceIdWaitingForBeingDisconnected = 0u;
            }
            else
            {
               // projection device is not yet in use, wait for a device to get disconnected triggered by
               // subsequent protocol blocking
               _deviceIdWaitingForBeingDisconnected = 1u;
               ETG_TRACE_USR4(("CheckIfDccIsAvailableForConnection(bdAddress = \"%50s\"): none of the DCCs is already "
                     "assigned to the projection device", _ProjectionDeviceBDAddress.c_str()));
               ETG_TRACE_USR4(("CheckIfDccIsAvailableForConnection(bdAddress = \"%50s\"): waiting for a DCC getting released "
                     "triggered by subsequent protocol blocking",_ProjectionDeviceBDAddress.c_str()));
            }//!End of else
         }//!End of if (dccInstancesInUse.size() >= LocalSpm::getDataProvider()....)
      }//!End of if (CC_ERR_INT_NO_ERROR == result)
      else
      {
         ETG_TRACE_ERR(("CheckIfDccIsAvailableForConnection(bdAddress = \"%50s\"): could not get used DCCs from BmController (error = %d)",
               _ProjectionDeviceBDAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));
         result = CC_ERR_INT_GENERAL_ERROR;
      }//!End of else
   }//!End of if((true == isUSBCommActive) || (true == isWiFiCommActive))

   return result;
}//!End of Result BtLimitationController::CheckIfDccIsAvailableForConnection(void)

//!Static
/***************************************************************************
 ** FUNCTION:  BtLimitationController::CPWReconnectTimerCb(...)
 ***************************************************************************/
bool BtLimitationController::CPWReconnectTimerCb(timer_t rTimerID, void *pvObject,
      const void *pvUserData)
{
   (void)pvUserData;
   ETG_TRACE_USR1(("CPWReconnectTimerCb : Timeout for CPW Reconnect. Stop retrying to reconnect to CPW device "));

   BtLimitationController* LimitationController =
         static_cast<BtLimitationController*>(pvObject);

   if(NULL != LimitationController)
   {
      LimitationController->_LockCPWReconnectionTimer.lock();
      LimitationController->_CPWReconnectionTimer.CancelTimer(rTimerID);
      LimitationController->_CPWReconnectionTimerID = 0;
      LimitationController->_LockCPWReconnectionTimer.unlock();

      //!Reconnect Timer expired and iAP2 via SPP did not reconnect for CPW device.
      (void) LimitationController->sendSmEvent("CPWReconnectTimerCb", "NOTRECONNECTED");
   }//End of if(NULL != LimitationController)

   return true;
}//!End of bool BtLimitationController::CPWReconnectTimerCb(...)
}//! End of namespace bmcore
