/**
 * @file BmController.cpp
 *
 * @swcomponent BluetoothConnectionManagerCore
 *
 * @brief This file contains the definition of the class BmController
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup BmControllerModule
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BmController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#endif
#endif


#include "BmAllTypes.h"
#include "FunctionTracer.h"
#include "Dispatcher.h"
#include "BmVarTrace.h"
#include "BmController.h"
#include "DeviceConnectionController.h"
#include "ProtocolConnectionController.h"
#include "LocalSpm.h"
#include "BmUtils.h"
#include "BmCoreIfMessagesCreator.h"
#include "Timer.h"
#include "Lock.h"
#include "BtStackIfFactory.h"
#include "IBtStackIfConnectionRequest.h"
#include "BmGlobalLock.h"
#include <algorithm>
#include <math.h>
#include <cstdio>
#include <time.h>

namespace bmcore
{
   BmController::BmController(const ComponentId componentID) :
         BmControllerOnOffSm(),
         ILocalSpm(componentID),
         _firstStartUp(true),
         _deviceConnectionControllers(),
         _numberConnCtrlStopping(0u),
         _dccInstancesToBeReleased(),
         _requestQueue(),
         _autoConnectionBooked(false),
         _suppressAutoConnection(false),
         _linkQualityRequestTimeOut(),
         _linkQualityTimer(),
         _linkQualityTimerId()
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("BmController: is being created"));

      // threadless SM
      disableAllLocks(true);

      initialize();
   }

   BmController::~BmController()
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("~BmController"));
   }

/*
   void BmController::updateLinkQuality(IN const DeviceIdList& connectedDevices, IN const DeviceIdList& previousConnectedDevices, IN const bool considerAllConnectedDevices)
   {
      ENTRY

      ETG_TRACE_USR1(("updateLinkQuality: considerAllConnectedDevices = %50s",
            (true == considerAllConnectedDevices) ? "true" : "false"));

      VARTRACE(connectedDevices);

      Locker locker(&_linkQualityLock);

      ETG_TRACE_USR4(("updateLinkQuality(considerAllConnectedDevices = %50s): beginning: waiting list and intermediate link qualities:",
            (true == considerAllConnectedDevices) ? "true" : "false"));

      VARTRACE(_connectedDevicesWaitingForLinkQuality);
      VARTRACE(_intermediateLinkQuality);

      // if periodical link quality requesting is used and no devices are connected anymore,
      // cancel the link quality request timer
      if ((0u != (LocalSpm::getDataProvider().getBmCoreConfiguration()._linkQualityRequestTimeOut)) && (true == connectedDevices.empty()))
      {
         _linkQualityTimer.CancelTimer(_linkQualityTimerId);

         ETG_TRACE_USR1(("updateLinkQuality: no connected devices - canceled link quality request timer"));
      }

      // determine the connected devices for which link quality has to be requested
      DeviceIdList connectedDevicesLinkQualityHasToBeRequested;

      if (true == considerAllConnectedDevices)
      {
         connectedDevicesLinkQualityHasToBeRequested = connectedDevices;
      }
      else
      {
         for (size_t idx = 0u; idx < connectedDevices.size(); idx++)
         {
            if (previousConnectedDevices.end() == std::find(previousConnectedDevices.begin(),
                  previousConnectedDevices.end(), connectedDevices[idx]))
            {
               connectedDevicesLinkQualityHasToBeRequested.push_back(connectedDevices[idx]);
            }
         }
      }

      ETG_TRACE_USR4(("updateLinkQuality(considerAllConnectedDevices = %50s): connected devices for which link quality has to be requested:",
            (true == considerAllConnectedDevices) ? "true" : "false"));

      VARTRACE(connectedDevicesLinkQualityHasToBeRequested);

      // determine devices which are not connected anymore
      DeviceIdList devicesNotConnectedAnymore;

      for (size_t idx = 0u; idx < previousConnectedDevices.size(); idx++)
      {
         if (connectedDevices.end() == std::find(connectedDevices.begin(), connectedDevices.end(),
               previousConnectedDevices[idx]))
         {
            devicesNotConnectedAnymore.push_back(previousConnectedDevices[idx]);
         }
      }

      ETG_TRACE_USR4(("updateLinkQuality(considerAllConnectedDevices = %50s): devices which are not connected anymore:",
            (true == considerAllConnectedDevices) ? "true" : "false"));

      VARTRACE(devicesNotConnectedAnymore);

      // delete devices which are not connected anymore from intermediate link quality property
      for (size_t idx = 0u; idx < devicesNotConnectedAnymore.size(); idx++)
      {
         _intermediateLinkQuality._linkQualityInfoList.erase(devicesNotConnectedAnymore[idx]);

         DeviceIdList::iterator it = std::find(_connectedDevicesWaitingForLinkQuality.begin(),
               _connectedDevicesWaitingForLinkQuality.end(), devicesNotConnectedAnymore[idx]);

         if (_connectedDevicesWaitingForLinkQuality.end() != it)
         {
            _connectedDevicesWaitingForLinkQuality.erase(it);
         }
      }

      if (false == devicesNotConnectedAnymore.empty())
      {
         // at least one device has been removed from intermediate link quality property,
         // so update the link quality property

         ETG_TRACE_USR4(("updateLinkQuality(considerAllConnectedDevices = %50s): at least one device has been removed from intermediate link quality property, update link quality property",
               (true == considerAllConnectedDevices) ? "true" : "false"));

         _linkQualityPropHdl.set(_intermediateLinkQuality);
      }

      // request link quality
      for (size_t idx = 0u; idx < connectedDevicesLinkQualityHasToBeRequested.size(); idx++)
      {
         BdAddress bdAddress("");
         Result result = LocalSpm::getDbManager().getBdAddressByDeviceHandle(bdAddress, connectedDevicesLinkQualityHasToBeRequested[idx]);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("updateLinkQuality(considerAllConnectedDevices = %50s): requesting link quality from BtStackIf for device with BD address = \"%50s\"",
                  considerAllConnectedDevices ? "true" : "false", bdAddress.c_str()));

            getBtStackIfRequest().requestLinkQuality(bdAddress);

            // remember the new connected devices the link quality has been successfully requested for
            _connectedDevicesWaitingForLinkQuality.push_back(connectedDevicesLinkQualityHasToBeRequested[idx]);
         }
         else
         {
            ETG_TRACE_ERR(("updateLinkQuality: could not get BD address for device handle = %d from DB (error = %d)", connectedDevicesLinkQualityHasToBeRequested[idx], ETG_CENUM(CcErrorInternal, result)));
         }
      }

      // if periodical link quality requesting is used and there is a new connected device,
      // start the link quality request timer
      if ((0u != LocalSpm::getDataProvider().getBmCoreConfiguration()._linkQualityRequestTimeOut)
            && ((true == previousConnectedDevices.empty())) && (false == connectedDevicesLinkQualityHasToBeRequested.empty()))
      {
         long intervalMilliseconds = 1000L * static_cast<long>(LocalSpm::getDataProvider().getBmCoreConfiguration()._linkQualityRequestTimeOut);

         if (true == _linkQualityTimer.StartTimer(_linkQualityTimerId, intervalMilliseconds, intervalMilliseconds, this, BmController::linkQualityTimerCb, NULL))
         {
            ETG_TRACE_USR1(("updateLinkQuality: started link quality request timer (_linkQualityTimerId = 0x%p, intervalMilliseconds = %d)", _linkQualityTimerId, intervalMilliseconds));
         }
         else
         {
            ETG_TRACE_ERR(("updateLinkQuality: starting the link quality request timer failed"));
         }
      }

      ETG_TRACE_USR4(("updateLinkQuality(considerAllConnectedDevices = %50s): end: waiting list and intermediate link qualities:",
            (true == considerAllConnectedDevices) ? "true" : "false"));

      VARTRACE(_connectedDevicesWaitingForLinkQuality);
      VARTRACE(_intermediateLinkQuality);
   }
*/
   void BmController::startLinkQualityRequestTimer(void)
   {
      long intervalMilliseconds = 1000L * static_cast<long>(LocalSpm::getDataProvider().getBmCoreConfiguration()._linkQualityRequestTimeOut);

      if (true == _linkQualityTimer.StartTimer(_linkQualityTimerId, intervalMilliseconds, intervalMilliseconds, this, BmController::linkQualityTimerCb, NULL))
      {
         ETG_TRACE_USR1(("startLinkQualityRequestTimer: started link quality request timer (_linkQualityTimerId = 0x%p, intervalMilliseconds = %d)",
               _linkQualityTimerId, intervalMilliseconds));
      }
      else
      {
         ETG_TRACE_ERR(("startLinkQualityRequestTimer: starting the link quality request timer failed"));
      }
   }

   void BmController::cancelLinkQualityRequestTimer(void)
   {
      _linkQualityTimer.CancelTimer(_linkQualityTimerId);
   }

   void BmController::requestLinkQualityOnExpiredTimer(void)
   {
      ENTRY

      ETG_TRACE_USR1(("requestLinkQualityOnExpiredTimer"));

      BmCoreIfMessage_UpdateLinkQualityRequest* bmCoreIfMessage = getNewBmCoreIfMessage_UpdateLinkQualityRequest(0u,
            BM_CORE_IF_MSG_ORIGIN_INTERNAL);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("requestLinkQualityOnExpiredTimer: bmCoreIfMessage is 0"));
         return;
      }

      LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);

/*
      DeviceIdList connectedDeviceHandles;
      LocalSpm::getBmCorePropertyHandler().getConnectedDevices(connectedDeviceHandles);

      Locker locker(&_linkQualityLock);

      ETG_TRACE_USR4(("requestLinkQualityForAllConnectedDevices(internallyCalled = %50s): beginning: waiting list and intermediate link qualities:", internallyCalled ? "true" : "false"));

      VARTRACE(_connectedDevicesWaitingForLinkQuality);
      VARTRACE(_intermediateLinkQuality);

      if (false == internallyCalled)
      {
         _numberPendingUpdateLinkQualityRequests++;

         ETG_TRACE_USR4(("requestLinkQualityForAllConnectedDevices(internallyCalled = %50s): incremented number of pending updateLinkQuality requests to %d",
               internallyCalled ? "true" : "false", _numberPendingUpdateLinkQualityRequests));
      }

      for (size_t idx = 0u; idx < connectedDeviceHandles.size(); idx++)
      {
         if (_connectedDevicesWaitingForLinkQuality.end() == find(_connectedDevicesWaitingForLinkQuality.begin(),
               _connectedDevicesWaitingForLinkQuality.end(), connectedDeviceHandles[idx]))
         {
            // the connected device is not in the list of connected devices for which the link quality has already been requested

            ETG_TRACE_USR4(("requestLinkQualityForAllConnectedDevices(internallyCalled = %50s): found connected device with device handle = %d for which link quality has not yet been requested",
                  internallyCalled ? "true" : "false", connectedDeviceHandles[idx]));

            BdAddress bdAddress("");
            Result result = LocalSpm::getDbManager().getBdAddressByDeviceHandle(bdAddress, connectedDeviceHandles[idx]);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               ETG_TRACE_USR4(("requestLinkQualityForAllConnectedDevices(internallyCalled = %50s): requesting link quality from BtStackIf for device with BD address = \"%50s\"",
                                 internallyCalled ? "true" : "false", bdAddress.c_str()));

               getBtStackIfRequest().requestLinkQuality(bdAddress);

               // remember the new connected device the link quality has been successfully requested for
               _connectedDevicesWaitingForLinkQuality.push_back(connectedDeviceHandles[idx]);
            }
            else
            {
               ETG_TRACE_ERR(("requestLinkQualityForAllConnectedDevices(internallyCalled = %50s): could not get BD address for device handle = %d from DB (error = %d)",
                     internallyCalled ? "true" : "false", connectedDeviceHandles[idx], ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("requestLinkQualityForAllConnectedDevices(internallyCalled = %50s): already requested link quality for device with device handle = %d, still waiting for the response",
                  internallyCalled ? "true" : "false", connectedDeviceHandles[idx]));
         }
      }

      ETG_TRACE_USR4(("requestLinkQualityForAllConnectedDevices(internallyCalled = %50s): end: waiting list and intermediate link qualities:",
            internallyCalled ? "true" : "false"));

      VARTRACE(_connectedDevicesWaitingForLinkQuality);
      VARTRACE(_intermediateLinkQuality);

      if (true == _connectedDevicesWaitingForLinkQuality.empty())
      {
         ETG_TRACE_USR4(("requestLinkQualityForAllConnectedDevices(internallyCalled = %50s): waiting list is empty, responding %d updateLinkQuality request(s)",
               internallyCalled ? "true" : "false", _numberPendingUpdateLinkQualityRequests));

         for (unsigned int i = 0u; i < _numberPendingUpdateLinkQualityRequests; i++)
         {
            LocalSpm::getBmCoreCallbackIfWrapper().doUpdateLinkQualityResponse(BM_RESULT_OK);
         }

         _numberPendingUpdateLinkQualityRequests = 0u;
      }
*/
   }

   Result BmController::getUsedDeviceConnectionControllers(OUT DeviceConnectionControllerList& dccInstances)
   {
      ENTRY

      ETG_TRACE_USR1(("getUsedDeviceConnectionControllers: looking for used DCC instances"));

      Result result(CC_ERR_INT_NO_ERROR);

      dccInstances.clear();

      for (size_t index = 0u; index < _deviceConnectionControllers.size(); index++)
      {
         if (true == _deviceConnectionControllers[index]->isInUse())
         {
            dccInstances.push_back(_deviceConnectionControllers[index]);
         }
      }

      ETG_TRACE_USR1(("getUsedDeviceConnectionControllers: number of used DCC instances = %d", dccInstances.size()));

      return result;
   }

   Result BmController::destroyConnectionControllers(void)
   {
      ENTRY

      ETG_TRACE_USR1(("destroyConnectionControllers: destroying all Protocol and DCC instances"));

      (void) destroyProtocolConnectionControllers();

      (void) destroyDeviceConnectionControllers();

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::onDeviceConnectionControllerReleased(IN const DeviceConnectionController* dccInstance)
   {
      ENTRY

      ETG_TRACE_USR1(("onDeviceConnectionControllerReleased: dccInstance = 0x%p, number of DCC instances waiting for getting released = %d", (const void *) dccInstance, _dccInstancesToBeReleased.size()));

      if (false == _dccInstancesToBeReleased.empty())
      {
         DeviceConnectionControllerList::iterator it = std::find(_dccInstancesToBeReleased.begin(), _dccInstancesToBeReleased.end(), dccInstance);

         if (it != _dccInstancesToBeReleased.end())
         {
            _dccInstancesToBeReleased.erase(it);
            ETG_TRACE_USR1(("onDeviceConnectionControllerReleased(dccInstance = 0x%p): removed DCC instance from list of instances waiting for getting released", (const void *) dccInstance));
         }

         if (true == _dccInstancesToBeReleased.empty())
         {
            ETG_TRACE_USR1(("onDeviceConnectionControllerReleased(dccInstance = 0x%p): list of DCC instances waiting for getting released is empty, checking if a deferred request is available", (const void *) dccInstance));

            if (true == _requestQueue.empty())
            {
               ETG_TRACE_USR1(("onDeviceConnectionControllerReleased(dccInstance = 0x%p): no deferred request available", (const void *) dccInstance));
            }
            else
            {
               ETG_TRACE_USR1(("onDeviceConnectionControllerReleased(dccInstance = 0x%p): deferred request available", (const void *) dccInstance));

               BmResult bmResult = handleRequest(true);

               if (BM_RESULT_OK != bmResult)
               {
                  ETG_TRACE_ERR(("onDeviceConnectionControllerReleased(dccInstance = 0x%p): could not handle deferred request (bmResult = %d)",
                        (const void *) dccInstance, ETG_CENUM(BmResult, bmResult)));
               }
            }
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   DeviceId BmController::getDeviceIdOfDeferredRequest(void) const
   {
      if (false == this->isRequestQueueEmpty())
      {
         return static_cast<ConnectionRequestData*>(_requestQueue[0])->_deviceId;
      }

      return 0u;
   }

   Result BmController::addConnectionRequestDataToQueue(IN const bool internal, IN const ConnectionRequestOrigin origin,
         IN const DeviceId deviceId, IN const ProtocolList& protocolList, IN const PageTimeout pageTimeout, IN const bool delayConnection)
   {
      ENTRY

      ETG_TRACE_USR1(("addConnectionRequestDataToQueue: internal = %d, origin = %d, deviceId = %d, pageTimeout = %d, delayConnection = %10s",
            internal, origin, deviceId, pageTimeout, delayConnection ? "true" : "false"));

      VARTRACE(protocolList);

      Result result(CC_ERR_INT_NO_ERROR);

      RequestData* connectionRequestData = this->getNewConnectionRequestData(internal, origin, deviceId, protocolList, pageTimeout, delayConnection);

      if (0 != connectionRequestData)
      {
         _requestQueue.push_back(connectionRequestData);
         ETG_TRACE_USR1(("addConnectionRequestDataToQueue(internal = %d, origin = %d, deviceId = %d): added connection request to queue: ID = %d",
               internal, origin, deviceId, connectionRequestData->getRequestId()));
      }
      else
      {
         result = CC_ERR_INT_MEMORY_ALLOCATION_ERROR;
         ETG_TRACE_ERR(("addConnectionRequestDataToQueue(internal = %d, origin = %d, deviceId = %d): could not allocate memory for connection request data (error = %d)",
               internal, origin, deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result BmController::addConnectionRequestDataToQueue(IN const bool internal, IN const ConnectionRequestOrigin origin,
         IN const BdAddress& bdAddress, IN const ProtocolList& protocolList, IN const PageTimeout pageTimeout, IN const bool delayConnection)
   {
      ENTRY

      ETG_TRACE_USR1(("addConnectionRequestDataToQueue: internal = %d, origin = %d, bdAddress = \"%50s\", pageTimeout = %d, delayConnection = %10s",
            internal, origin, bdAddress.c_str(), pageTimeout, delayConnection ? "true" : "false"));

      VARTRACE(protocolList);

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);

      result = LocalSpm::getDbManager().getDeviceId(OUT deviceId, IN bdAddress);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         result = addConnectionRequestDataToQueue(internal, origin, deviceId, protocolList, pageTimeout, delayConnection);
      }
      else
      {
         ETG_TRACE_ERR(("addConnectionRequestDataToQueue(internal = %d, origin = %d, bdAddress = \"%50s\"): could not get deviceId for device with given BD address from DB (error = %d)",
               internal, origin, bdAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result BmController::removeRequestDataFromQueue(void)
   {
      ETG_TRACE_USR1(("removeRequestDataFromQueue"));

      Result result(CC_ERR_INT_NO_ERROR);

      if (false == _requestQueue.empty())
      {
         ETG_TRACE_USR1(("removeRequestDataFromQueue: removing request with ID = %d from queue", _requestQueue[0]->getRequestId()));

         delete _requestQueue[0];
         _requestQueue.erase(_requestQueue.begin());
      }
      else
      {
         ETG_TRACE_USR1(("removeRequestDataFromQueue: queue is already empty"));
      }

      return result;
   }

   Result BmController::cleanUpDeferredConnectionRequest(IN const ProtocolList& protocolList)
   {
      ETG_TRACE_USR1(("cleanUpDeferredConnectionRequest"));

      if (false == _requestQueue.empty())
      {
         ProtocolList::iterator it;
         ProtocolList modifiedProtocolList;
         ConnectionRequestData* connectionRequestData = static_cast<ConnectionRequestData*>(_requestQueue[0]);

         for (size_t index = 0u; index < connectionRequestData->_protocolList.size(); ++index)
         {
            if (std::find(protocolList.begin(), protocolList.end(),
                  connectionRequestData->_protocolList[index]) == protocolList.end())
            {
               modifiedProtocolList.push_back(connectionRequestData->_protocolList[index]);
            }
            else
            {
               ETG_TRACE_USR1(("cleanUpDeferredConnectionRequest: removed protocol (%d, \"%50s\") from protocol list of deferred connection request",
                     ETG_CENUM(ProtocolId, connectionRequestData->_protocolList[index]._protocolId),
                     connectionRequestData->_protocolList[index]._uuid.c_str()));
            }
         }

         connectionRequestData->_protocolList = modifiedProtocolList;
      }

      return CC_ERR_INT_NO_ERROR;
   }

   BmResult BmController::handleRequest(IN const bool deferredRequest, IN const bool ignoreStopAutoconnection)
   {
      ENTRY

      ETG_TRACE_USR1(("handleRequest: deferredRequest = %d ignoreStopAutoconnection = %d", deferredRequest, ignoreStopAutoconnection));

      Result result(CC_ERR_INT_NO_ERROR);
      BmResult bmResult(BM_RESULT_OK);
      bool keepRequestDataInQueue(false);

      if (false == _requestQueue.empty())
      {
         ETG_TRACE_USR1(("handleRequest: handling request with ID = %d", _requestQueue[0]->getRequestId()));

         if (true == isValidRequestType(_requestQueue[0]->_requestType))
         {
            // Avoid the connection request check if ignoreStopAutoconnection flag is true
            if ((false == _requestQueue[0]->_internal) && (false == ignoreStopAutoconnection))
            {
               // request has been made by an external component (BM Application Layer or BtStackIf)

               bool stopAutoConnection(false);

               ConnectionRequestData* connectionRequestData = static_cast<ConnectionRequestData*>(_requestQueue[0]);

               if (BM_REQUEST_TYPE_CONNECTION == _requestQueue[0]->_requestType)
               {

                  ProtocolList hmiUserSelectableProtocols;
                  result = LocalSpm::getDataProvider().getHmiUserSelectableProtocols(hmiUserSelectableProtocols);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     for (size_t i = 0u; i < connectionRequestData->_protocolList.size(); i++)
                     {
                        if (hmiUserSelectableProtocols.end() != std::find(hmiUserSelectableProtocols.begin(),
                              hmiUserSelectableProtocols.end(), connectionRequestData->_protocolList[i]))
                        {
                           // auto connection will be stopped if given protocol list includes a device connection protocol
                           stopAutoConnection = true;
                        }
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleRequest: could not get device connection protocols from DataProvider (error = %d) ",
                           ETG_CENUM(CcErrorInternal, result)));
                  }
               }

               if(true == stopAutoConnection)
               {
                  // if auto connection has to be stopped (i.e. given protocol list includes a device connection protocol)
                  // all DCCs currently busy with auto connection and/or reconnection of a lost device
                  // are to be released before handling the connection request
                  DeviceId sendBtLimitationDeactivateToDeviceId(0u);
                  result = getReleasableDeviceConnectionControllers(_dccInstancesToBeReleased, sendBtLimitationDeactivateToDeviceId);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     if(0u != sendBtLimitationDeactivateToDeviceId)
                     {
                        // Check the request deviceId is same as BTLimitationMode Projection DeviceId
                        if(connectionRequestData->_deviceId != sendBtLimitationDeactivateToDeviceId)
                        {
                           if(true == LocalSpm::getBtLimitationController().isHandlingLostDevice())
                           {
                              ETG_TRACE_USR4(("handleRequest: BTLimiationMode is handling lost device so send deactivate"));

                              BdAddress bdAddress;
                              result = LocalSpm::getDbManager().getBdAddress(bdAddress, sendBtLimitationDeactivateToDeviceId);

                              if (CC_ERR_INT_NO_ERROR != result)
                              {
                                 ETG_TRACE_USR1(("handleRequest(): could not get deviceId from DB (error = %d)",
                                       ETG_CENUM(CcErrorInternal, result)));
                              }
                              else
                              {
                                 DeviceConnectionController* dccInstance(0);
                                 result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, sendBtLimitationDeactivateToDeviceId);

                                 if (CC_ERR_INT_NO_ERROR == result)
                                 {
                                    _dccInstancesToBeReleased.push_back(dccInstance);

                                    // Deactivate  CPW
                                    LocalSpm::getBmCoreMainController().deactivateBtLimitationModeCPW(sendBtLimitationDeactivateToDeviceId);
                                 }
                                 else
                                 {
                                    ETG_TRACE_ERR(("handleRequest: could not get DCC instance handling device with ID = %d from DB (error = %d)",
                                          sendBtLimitationDeactivateToDeviceId, ETG_CENUM(CcErrorInternal, result)));
                                 }
                              }
                           }
                           else
                           {
                              ETG_TRACE_USR4(("handleRequest: BTLimiationMode is not handling lost device"));
                           }
                        }
                        else
                        {
                           ETG_TRACE_USR4(("handleRequest: connection request DeviceId is same as BTLimiationMode projectionDeviceId"));
                        }
                     }
                     else
                     {
                        ETG_TRACE_USR4(("handleRequest: BTLimiationMode is active or empty"));
                     }

                  if (false == _dccInstancesToBeReleased.empty())
                  {
                     // This is due to connection order remains same in the HMI
                     DeviceConnectionControllerList OrderedDccInstances;
                     (void) LocalSpm::getDbManager().getOrderedDeviceConnectionControllers(OrderedDccInstances, _dccInstancesToBeReleased);

                     for (size_t dccIdx = 0u; dccIdx < OrderedDccInstances.size(); dccIdx++)
                     {
                        ProtocolList disconnectProtocolList;

                        // check the dcc instance is already assigned to requested deviceId and also not handling lost scenario.
                        // if already assigned then send the disconnection protocol request for non-selected profiles
                        if((OrderedDccInstances[dccIdx]->getDeviceId() == connectionRequestData->_deviceId)
                              && (false == OrderedDccInstances[dccIdx]->isHandlingLostDevice()) && (true == stopAutoConnection))
                        {
                           bmResult = releaseDccInstance(connectionRequestData, OrderedDccInstances[dccIdx]);
                        }
                        else
                        {
                           bmResult = disconnectProtocols(OrderedDccInstances[dccIdx], disconnectProtocolList, true, BM_DISCONNECTED_REASON_AUTOMATIC);
                        }

                        if (BM_RESULT_OK != bmResult)
                        {
                           ETG_TRACE_ERR(("handleRequest: initiating device disconnection failed (bmResult = %d)",
                                 ETG_CENUM(BmResult, bmResult)));
                        }
                     }

                     if (false == _dccInstancesToBeReleased.empty())
                     {
                        keepRequestDataInQueue = true;

                        ETG_TRACE_USR1(("handleRequest: keeping request with ID = %d in queue for deferred processing", _requestQueue[0]->getRequestId()));
                     }
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleRequest: could not get releasable DCC instances (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }

                  ETG_TRACE_USR4(("handleRequest: sending event STOP_AUTO_CONNECTION to AutoConnectionControllerSm"));
                  result = LocalSpm::getAutoConnectionController().SendUrgentEvent(LocalSpm::getAutoConnectionController().STOP_AUTO_CONNECTION, 0);

                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_ERR(("handleRequest: could not send event STOP_AUTO_CONNECTION (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                  }
               }
            }

            if (false == keepRequestDataInQueue)
            {
               // no resources to be released before handling the current request

               if (BM_REQUEST_TYPE_CONNECTION == _requestQueue[0]->_requestType)
               {
                  result = handleConnectionRequest(deferredRequest);

                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_ERR(("handleRequest: could not handle connection request (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                  }
               }
               else
               {
                  // cannot be reached, see check for valid request type above
               }

               (void) removeRequestDataFromQueue();
            }
         }
         else
         {
            // error, invalid request type
            result = CC_ERR_INT_INVALID_PARAMETER;
            ETG_TRACE_ERR(("handleRequest: invalid value for request type (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         // error, no request available for processing
         result = CC_ERR_INT_GENERAL_ERROR;
         ETG_TRACE_ERR(("handleRequest: no request available for processing (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      if (CC_ERR_INT_NO_ERROR != result)
      {
         switch (result)
         {
            case CC_ERR_INT_DCC_ALL_IN_USE:
               bmResult = BM_RESULT_ERR_ALL_DEVICE_RESOURCES_IN_USE;
               break;
            case CC_ERR_INT_PCC_NO_RESOURCES:
               bmResult = BM_RESULT_ERR_ALL_PROTOCOL_RESOURCES_IN_USE;
               break;
            case CC_ERR_INT_MASTER_PROTOCOL_NOT_CONNECTED:
               bmResult = BM_RESULT_ERR_MASTER_PROTOCOL_NOT_CONNECTED;
               break;
            default:
               bmResult = BM_RESULT_ERR_GENERAL;
               break;
         }
      }

      return bmResult;
   }

   BmResult BmController::releaseDccInstance(IN ConnectionRequestData* connectionRequestData, IN DeviceConnectionController* dccInstance)
   {
      BmResult bmResult(BM_RESULT_OK);

      ProtocolList disconnectProtocolList;
      ProtocolList hmiUserSelectableProtocols;
      bool removeDccInstance = false;

      ETG_TRACE_USR3(("releaseDccInstance(dccInstance = 0x%p): entered", (const void *) dccInstance));

      (void) LocalSpm::getDataProvider().getHmiUserSelectableProtocols(hmiUserSelectableProtocols);

      for (size_t i = 0u; i < hmiUserSelectableProtocols.size(); i++)
      {
         if (connectionRequestData->_protocolList.end() == std::find(connectionRequestData->_protocolList.begin(),
               connectionRequestData->_protocolList.end(), hmiUserSelectableProtocols[i]))
         {
            disconnectProtocolList.push_back(hmiUserSelectableProtocols[i]);
         }
      }

      if(false == disconnectProtocolList.empty())
      {
         bmResult = disconnectProtocols(dccInstance, disconnectProtocolList, false, BM_DISCONNECTED_REASON_AUTOMATIC);

         if (BM_RESULT_OK != bmResult)
         {
            ETG_TRACE_ERR(("releaseDccInstance: initiating device disconnection failed (bmResult = %d)",
                  ETG_CENUM(BmResult, bmResult)));
         }
         else
         {
            removeDccInstance = true;
         }
      }
      else
      {
         removeDccInstance = true;
      }

      if(true == removeDccInstance)
      {
         DeviceConnectionControllerList::iterator it = std::find(_dccInstancesToBeReleased.begin(), _dccInstancesToBeReleased.end(),
               dccInstance);

         if (it != _dccInstancesToBeReleased.end())
         {
            _dccInstancesToBeReleased.erase(it);
            ETG_TRACE_USR3(("releaseDccInstance(dccInstance = 0x%p): removed DCC instance from list of instances waiting for getting released",
                  (const void *) dccInstance));
         }
      }

      return bmResult;
   }

   BmResult BmController::doRequestTestModeLinkQuality(void)
   {
      ENTRY

      ETG_TRACE_USR1(("doRequestTestModeLinkQuality"));

      BmResult bmResult(BM_RESULT_OK);

      if (true == LocalSpm::getBmCoreMainController().isFunctionalityPermitted(BM_RESTRICTION_GROUP_06))
      {
         SwitchStatus testMode;
         bmResult = LocalSpm::getBmCoreMainController().getTestMode(testMode);

         if (BM_RESULT_OK == bmResult)
         {
            if (SWITCH_STATE_SWITCHED_ON == testMode._switchState)
            {
               LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().requestTestModeLinkQuality();
            }
            else
            {
               ETG_TRACE_ERR(("doRequestTestModeLinkQuality: test mode is not switched on"));
               bmResult = BM_RESULT_ERR_NOT_IN_TEST_MODE;
            }
         }
         else
         {
            ETG_TRACE_ERR(("doRequestTestModeLinkQuality: could not get test mode switch state"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("doRequestTestModeLinkQuality: request is currently not allowed to be processed"));
         bmResult = BM_RESULT_ERR_NOT_ALLOWED;
      }

      if (BM_RESULT_OK != bmResult)
      {
         LocalSpm::getBmCoreCallbackIfWrapper().doRequestTestModeLinkQualityResponse(bmResult, LinkQualityInfo(127, 0xFFu));
      }

      return bmResult;
   }

   /*BmResult BmController::doGetLinkQualityTestMode(OUT TestModeLinkQuality& linkQuality)
   {
      ENTRY

      ETG_TRACE_USR1(("doGetLinkQualityTestMode"));

      BmResult bmResult(BM_RESULT_OK);

      if (true == LocalSpm::getBmCoreMainController().isFunctionalityPermitted(BM_RESTRICTION_GROUP_01))
      {
         _testModeLinkQualityPropHdl.get(linkQuality);

         VARTRACE(linkQuality);
      }
      else
      {
         ETG_TRACE_ERR(("doGetLinkQualityTestMode: request is currently not allowed to be processed"));
         bmResult = BM_RESULT_ERR_NOT_ALLOWED;
      }

      return bmResult;
   }*/

   Result BmController::messageNotConsumed()
   {
      ENTRY

      ETG_TRACE_USR1(("messageNotConsumed"));

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::startAutoConnection(IN const AutoConnectionStartMode autoConnectionStartMode)
   {
      ETG_TRACE_USR1(("startAutoConnection: autoConnectionStartMode = %d", ETG_CENUM(AutoConnectionStartMode, autoConnectionStartMode)));

      Result result(CC_ERR_INT_NO_ERROR);
      bool autoConnectionToBeTried(false);

      if (false == isValidAutoConnectionStartMode(autoConnectionStartMode))
      {
         result = CC_ERR_INT_INVALID_PARAMETER;
         ETG_TRACE_ERR(("startAutoConnection: invalid parameter value for autoConnectionStartMode (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }
      else
      {
         if (BM_AUTOCONNECTION_START_MODE_BOOK_AND_TRY_ONCE == autoConnectionStartMode)
         {
            _autoConnectionBooked = true;
            autoConnectionToBeTried = true;
         }

         if ((BM_AUTOCONNECTION_START_MODE_TRY_ONCE_ONLY_IF_BOOKED_BEFORE == autoConnectionStartMode)
               && (true == _autoConnectionBooked))
         {
            autoConnectionToBeTried = true;
         }

         if (BM_AUTOCONNECTION_START_MODE_TRY_ONCE == autoConnectionStartMode)
         {
            autoConnectionToBeTried = true;
         }

         ETG_TRACE_USR1(("startAutoConnection: auto connection is%10s requested to be started", (true == autoConnectionToBeTried) ? "" : " not"));

         if (false == _suppressAutoConnection)
         {
            if (true == autoConnectionToBeTried)
            {
               if (true == LocalSpm::getBmCoreMainController().isFunctionalityPermitted(BM_RESTRICTION_GROUP_06))
               {
                  ETG_TRACE_USR1(("startAutoConnection: starting auto connection sequence"));

                  AutoConnectionType autoConnectionType;
                  (void) LocalSpm::getBmCoreMainController().getAutoConnectionType(autoConnectionType);

                  char eventParams[20];

                  result = LocalSpm::getAutoConnectionController().ParameterSTART_AUTO_CONNECTION(OUT eventParams,
                        IN sizeof(eventParams), IN autoConnectionType._type);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     ETG_TRACE_USR4(("startAutoConnection: sending event START_AUTO_CONNECTION to AutoConnectionControllerSm"));
                     result = LocalSpm::getAutoConnectionController().SendEventByName("START_AUTO_CONNECTION", eventParams);

                     if (CC_ERR_INT_NO_ERROR != result)
                     {
                        ETG_TRACE_ERR(("startAutoConnection: could not send event START_AUTO_CONNECTION (error = %d)",
                              ETG_CENUM(CcErrorInternal, result)));
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("startAutoConnection: could not marshal event parameters for event START_AUTO_CONNECTION (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                  }
               }
               else
               {
                  ETG_TRACE_USR1(("startAutoConnection: auto connection is currently not allowed to be started"));
               }
            }
         }
         else
         {
            ETG_TRACE_USR1(("startAutoConnection: start of auto connection is currently suppressed"));
            _suppressAutoConnection = false;
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::stopAutoConnection(void)
   {
      ENTRY

      ETG_TRACE_USR1(("stopAutoConnection: stopping auto connection sequence"));

      ETG_TRACE_USR4(("startAutoConnection: sending event STOP_AUTO_CONNECTION to AutoConnectionControllerSm"));
      Result result = LocalSpm::getAutoConnectionController().SendUrgentEvent(LocalSpm::getAutoConnectionController().STOP_AUTO_CONNECTION, 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("stopAutoConnection: could not send event STOP_AUTO_CONNECTION (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::clearAutoConnectionBooking(void)
   {
      ENTRY

      ETG_TRACE_USR1(("clearAutoConnectionBooking"));

      _autoConnectionBooked = false;

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::isFirstStartUp()
   {
      ENTRY

      ETG_TRACE_USR1(("isFirstStartUp: this is %50s BmController start-up in the current power cycle", (true == _firstStartUp) ? "the first" : "not the first"));

      if (true == _firstStartUp)
      {
         _firstStartUp = false;

         return 1;
      }

      return 0;
   }

   Result BmController::initializeBtStackIf()
   {
      ENTRY

      ETG_TRACE_USR1(("initializeBtStackIf"));

      Result result(CC_ERR_INT_NO_ERROR);

      BTSLocalStackConfiguration config;

      // initializing local BD name

      LocalFriendlyName localFriendlyName;
      result = LocalSpm::getDbManager().getLocalBdName(localFriendlyName._localFriendlyName);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR1(("initializeBtStackIf: BD name stored in DB is \"%50s\"", localFriendlyName._localFriendlyName.c_str()));

         if (false == isValidLocalBdName(localFriendlyName._localFriendlyName))
         {
            localFriendlyName._localFriendlyName = LocalSpm::getDataProvider().getBmCoreConfiguration()._btLocalFriendlyName;

            ETG_TRACE_USR1(("initializeBtStackIf: BD name stored in DB is not valid (probably not yet set, using the configured default value \"%50s\")", localFriendlyName._localFriendlyName.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("initializeBtStackIf: getting local BD name from DB failed (error = %d)", ETG_CENUM(CcErrorInternal, result)));
      }

      config.name = static_cast<BTSDeviceName>(localFriendlyName._localFriendlyName);

      // set locally supported services

      ProtocolInfoMap localSupportedProtocols;
      result = LocalSpm::getDataProvider().getProtocolInfos(localSupportedProtocols, true, false, false);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ProtocolInfoMap::iterator it;
         for (it = localSupportedProtocols.begin(); it != localSupportedProtocols.end(); it++)
         {
            switch (it->first)
            {
               case BM_PROTOCOL_ID_HFP:
                  config.supportedServices.setBit(BTS_SUPP_SRV_HFP);
                  break;
               case BM_PROTOCOL_ID_AVP:
                  config.supportedServices.setBit(BTS_SUPP_SRV_A2DP);
                  config.supportedServices.setBit(BTS_SUPP_SRV_AVRCP);
                  break;
               case BM_PROTOCOL_ID_PBDL:
                  config.supportedServices.setBit(BTS_SUPP_SRV_PBAP);
                  break;
               case BM_PROTOCOL_ID_MSG:
                  config.supportedServices.setBit(BTS_SUPP_SRV_MAP);
                  break;
               case BM_PROTOCOL_ID_PAN:
                  config.supportedServices.setBit(BTS_SUPP_SRV_PAN);
                  break;
               case BM_PROTOCOL_ID_SPP:
                  config.supportedServices.setBit(BTS_SUPP_SRV_SPP);
                  break;
               default:
                  break;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("initializeBtStackIf: getting local protocol infos from DataProvider failed (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      config.deviceIdServiceRecord.primaryRecord = true;
      config.deviceIdServiceRecord.specificationID = LocalSpm::getDataProvider().getBmCoreConfiguration()._didSpecificationId;
      config.deviceIdServiceRecord.vendorID = LocalSpm::getDataProvider().getBmCoreConfiguration()._didVendorId;
      config.deviceIdServiceRecord.productID = LocalSpm::getDataProvider().getBmCoreConfiguration()._didProductId;
      config.deviceIdServiceRecord.version = LocalSpm::getDataProvider().getBmCoreConfiguration()._didVersion;
      config.deviceIdServiceRecord.vendorIDSource = LocalSpm::getDataProvider().getBmCoreConfiguration()._didVendorIdSource;
      config.deviceIdServiceRecord.clientExecutableURL = LocalSpm::getDataProvider().getBmCoreConfiguration()._didClientExecutableUrl;
      config.deviceIdServiceRecord.serviceDescription = LocalSpm::getDataProvider().getBmCoreConfiguration()._didServiceDescription;
      config.deviceIdServiceRecord.clientDocumentationURL = LocalSpm::getDataProvider().getBmCoreConfiguration()._didClientDocumentationUrl;

      config.carPlayWirelessEnabled = LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported;

      config.wbsEnabled = LocalSpm::getDataProvider().getBmCoreConfiguration()._wideBandSpeechEnabled;
      config.pbdlViaPbapEnabled = LocalSpm::getDataProvider().getBmCoreConfiguration()._pbdlViaPbapEnabled;
      config.pbdlViaHfpEnabled = LocalSpm::getDataProvider().getBmCoreConfiguration()._pbdlViaHfpEnabled;
      config.pbdlViaSppEnabled = LocalSpm::getDataProvider().getBmCoreConfiguration()._pbdlViaSppEnabled;
      config.pbdlViaSyncMlEnabled = LocalSpm::getDataProvider().getBmCoreConfiguration()._pbdlViaSyncMlEnabled;
      config.voiceRecognitionEnabled = LocalSpm::getDataProvider().getBmCoreConfiguration()._voiceRecognitionEnabled;
      config.enhancedCallControlEnabled = LocalSpm::getDataProvider().getBmCoreConfiguration()._enhancedCallControlEnabled;
      config.avrcpBrowsingEnabled = LocalSpm::getDataProvider().getBmCoreConfiguration()._avrcpBrowsingEnabled;
      config.audioCodecMp3Enabled = LocalSpm::getDataProvider().getBmCoreConfiguration()._audioCodecMp3Enabled;
      config.audioCodecAacEnabled = LocalSpm::getDataProvider().getBmCoreConfiguration()._audioCodecAacEnabled;//LocalSpm::getDataProvider().AudioCodecAacEnabled();

      config.initialDiscoverableMode = (TARGET_SWITCH_STATE_SWITCHED_ON == LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode) ? BTS_MODE_ENABLED : BTS_MODE_DISABLED;
      config.initialConnectableMode = (TARGET_SWITCH_STATE_SWITCHED_ON == LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode) ? BTS_MODE_ENABLED : BTS_MODE_DISABLED;

      config.sppServiceInfoList.clear();

      bool supportUuid = false;

      for (SppServiceInformation::const_iterator it = LocalSpm::getDataProvider().getBmCoreConfiguration()._sppServiceInformation.begin();
            it != LocalSpm::getDataProvider().getBmCoreConfiguration()._sppServiceInformation.end(); ++it)
      {
         supportUuid = true;

         if((false == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported) && (0 == (strcmp(it->_serviceName.c_str(), "CPW"))))
         {
            supportUuid = false;
            ETG_TRACE_USR4(("initializeBtStackIf: CPW is disabled in HU so not advertising the CPW uuid"));
         }
         else
         {
            if((false == LocalSpm::getDataProvider().getBmCoreConfiguration()._androidAutoWirelessSupported) && (0 == (strcmp(it->_serviceName.c_str(), "AAW"))))
            {
               supportUuid = false;
               ETG_TRACE_USR4(("initializeBtStackIf: AAW is disabled in HU so not advertising the AAW uuid"));
            }
         }

         if((true == supportUuid) && (0u < it->_maxNumInstances))
         {
            for(int index = 0u; index < it->_maxNumInstances; index++)
            {
               BTSSppServiceInfo sppServiceInfo;

               sppServiceInfo.serviceName = it->_serviceName;
               sppServiceInfo.localUuid = it->_localUuid;
               sppServiceInfo.remoteUuid = it->_remoteUuid;

               config.sppServiceInfoList.push_back(sppServiceInfo);

               ETG_TRACE_USR4(("initializeBtStackIf: sppServiceInfoList[%d]: maxNumInstances = %d, serviceName = %10s, localUuid = %50s, remoteUuid = %50s",
                     index, it->_maxNumInstances, sppServiceInfo.serviceName.c_str(), sppServiceInfo.localUuid.c_str(), sppServiceInfo.remoteUuid.c_str()));
            }
         }
      }

      BTSBDAddressList pairedDevices;
      result = LocalSpm::getDbManager().getAllBdAddresses(pairedDevices);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("initializeBtStackIf: could not get BD addresses of paired devices from DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
         pairedDevices.clear();
      }

      LocalSpm::getBmCoreMainController().initBtStackIfConnection(config, pairedDevices);

      BTSLocalWblConfiguration wblConfig;
      LocalSpm::getBmCoreMainController().initBtStackIfWbl(wblConfig);

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::initializeProperties()
   {
      ENTRY

      ETG_TRACE_USR1(("initializeProperties"));

      Result totalResult(CC_ERR_INT_NO_ERROR);

      LocalFriendlyName localFriendlyName;
      Result result = LocalSpm::getDbManager().getLocalBdName(localFriendlyName._localFriendlyName);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR1(("initializeProperties: BD name stored in DB is \"%50s\"", localFriendlyName._localFriendlyName.c_str()));

         if (false == isValidLocalBdName(localFriendlyName._localFriendlyName))
         {
            localFriendlyName._localFriendlyName = LocalSpm::getDataProvider().getBmCoreConfiguration()._btLocalFriendlyName;

            ETG_TRACE_USR1(("initializeProperties: BD name stored in DB is not valid (probably not yet set, using the configured default value \"%50s\")", localFriendlyName._localFriendlyName.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("initializeProperties: getting local BD name from DB failed (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      (void) LocalSpm::getBmCoreMainController().setBtLocalFriendlyNameIntForced(localFriendlyName._localFriendlyName);

      totalResult = result;

      // LocalInfo property (no BtStackIf interaction)
      result = LocalSpm::getBmCoreMainController().initLocalInfo();

      if (CC_ERR_INT_NO_ERROR == totalResult)
      {
         totalResult = result;
      }

      // LocalPairableMode and LocalConnectableMode
      result = LocalSpm::getBmCoreMainController().initLocalPairableMode();

      if (CC_ERR_INT_NO_ERROR == totalResult)
      {
         totalResult = result;
      }

      // LocalConnectableMode
      result = LocalSpm::getBmCoreMainController().initLocalConnectableMode();

      if (CC_ERR_INT_NO_ERROR == totalResult)
      {
         totalResult = result;
      }

      result = LocalSpm::getBmCoreMainController().initBmConfigData();

      if (CC_ERR_INT_NO_ERROR == totalResult)
      {
         totalResult = result;
      }

      result = LocalSpm::getBmCoreMainController().initResetToDefaultStatus();

      if (CC_ERR_INT_NO_ERROR == totalResult)
      {
         totalResult = result;
      }

      result = LocalSpm::getBmCoreMainController().initPairingPin();

      if (CC_ERR_INT_NO_ERROR == totalResult)
      {
         totalResult = result;
      }

      result = LocalSpm::getBmCoreMainController().initAutoConnectionType();

      if (CC_ERR_INT_NO_ERROR == totalResult)
      {
         totalResult = result;
      }

      // LocalInfo property (no BtStackIf interaction)
      result = LocalSpm::getBmCoreMainController().initMultiHFPSupport();

      if (CC_ERR_INT_NO_ERROR == totalResult)
      {
         totalResult = result;
      }

      result = LocalSpm::getBmCoreMainController().initBtSystemState();

      if (CC_ERR_INT_NO_ERROR == totalResult)
      {
         totalResult = result;
      }

      result = LocalSpm::getBmCoreMainController().initBlockStatus();

      if (CC_ERR_INT_NO_ERROR == totalResult)
      {
         totalResult = result;
      }

      result = LocalSpm::getBmCoreMainController().initPairedDeviceList();

      if (CC_ERR_INT_NO_ERROR == totalResult)
      {
         totalResult = result;
      }

      result = LocalSpm::getBmCoreMainController().initDiscoveryStatus();

      if (CC_ERR_INT_NO_ERROR == totalResult)
      {
         totalResult = result;
      }

      LocalSpm::getBmCoreMainController().initDiscoveredDeviceList();

      ETG_TRACE_USR4(("initializeProperties: sending event PROPERTIES_INITIALIZED to BmControllerOnOffSm"));
      result = this->SendEventByName("PROPERTIES_INITIALIZED", 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("initializeProperties: could not send event PROPERTIES_INITIALIZED (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::restoreBtStatus()
   {
      ENTRY

      ETG_TRACE_USR1(("restoreBtStatus"));

      Result result(CC_ERR_INT_NO_ERROR);
      TargetSwitchState btStatusTargetSwitchState(TARGET_SWITCH_STATE_SWITCHED_OFF);

      result = LocalSpm::getDbManager().getBtStatusTargetState(btStatusTargetSwitchState);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR1(("restoreBtStatus: BT Status' target switch state stored in DB is %d", ETG_CENUM(TargetSwitchState, btStatusTargetSwitchState)));

         if (false == isValidTargetSwitchState(btStatusTargetSwitchState))
         {
            btStatusTargetSwitchState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultBtStatus;

            ETG_TRACE_ERR(("restoreBtStatus: BT Status' target switch state stored in DB is not valid (using %d)",
                  ETG_CENUM(TargetSwitchState, btStatusTargetSwitchState)));

            result = LocalSpm::getDbManager().setBtStatusTargetState(btStatusTargetSwitchState);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("restoreBtStatus: storing BT Status' target switch state to DB failed (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("restoreBtStatus: getting BT Status' target switch state from DB failed (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      (void) LocalSpm::getBmCoreMainController().switchBtStatusInt(btStatusTargetSwitchState, false, false);

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::handleBmControllerActive()
   {
      ENTRY

      ETG_TRACE_USR1(("handleBmControllerActive"));

      (void) this->runDone(0);

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::startAutoConnectionAttempt()
   {
      ENTRY

      ETG_TRACE_USR1(("startAutoConnectionAttempt"));

      Result result = this->startAutoConnection(BM_AUTOCONNECTION_START_MODE_TRY_ONCE);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("startAutoConnectionAttempt: could not start auto connection attempt (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::disconnectAllDevices()
   {
      ENTRY

      ETG_TRACE_USR1(("disconnectAllDevices"));

      DeviceConnectionStatusList deviceConnectionStatusList;
      Result result(CC_ERR_INT_NO_ERROR);
      BmResult bmResult = LocalSpm::getBmCoreMainController().getDeviceConnectionStatusList(deviceConnectionStatusList);

      if (BM_RESULT_OK == bmResult)
      {
         if (true == deviceConnectionStatusList._deviceConnectionInfoList.empty())
         {
            checkAndReleaseReservedDCC();

            ETG_TRACE_USR1(("disconnectAllDevices: no connected devices found"));

            ETG_TRACE_USR4(("disconnectAllDevices: sending event ALL_DEVICES_DISCONNECTED to BmControllerOnOffSm"));
            result = this->SendEventByName("ALL_DEVICES_DISCONNECTED", 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("disconnectAllDevices: could not send event ALL_DEVICES_DISCONNECTED (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_USR1(("disconnectAllDevices: connected devices found, disconnecting all devices"));

            // Change for chery CRQ 90582: AVP DisconnectedReason to be set to FI_EN_E8INTERMEDIATE when entering 'STANDBY' state from 'IGNITION' state
            DisconnectedReason disconnectedReason = BM_DISCONNECTED_REASON_AUTOMATIC;

            BtSystemState btSystemState;
            BmResult bmResult = LocalSpm::getBmCoreMainController().getBtSystemState(btSystemState);

            if(BM_RESULT_OK == bmResult)
            {
               ETG_TRACE_USR1(("disconnectAllDevices: BluetoothSystemState = %d", ETG_CENUM(BluetoothSystemState, btSystemState._bluetoothSystemState)));

               if((BM_BT_SYSTEM_STATE_BLOCK == btSystemState._bluetoothSystemState) ||
                     (BM_BT_SYSTEM_STATE_OFF == btSystemState._bluetoothSystemState))
               {
                  disconnectedReason = BM_DISCONNECTED_REASON_TEMPORARILY;
               }
            }

            bmResult = LocalSpm::getBmCoreMainController().disconnectDeviceInt(0u, BM_DEVICE_HANDLE_TYPE_ALL, disconnectedReason);

            if (BM_RESULT_OK != bmResult)
            {
               ETG_TRACE_ERR(("disconnectAllDevices: disconnecting all devices failed (bmResult = %d)",
                     ETG_CENUM(BmResult, bmResult)));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("disconnectAllDevices: getting device connection status list failed (bmResult = %d)",
               ETG_CENUM(BmResult, bmResult)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   void BmController::checkAndReleaseReservedDCC()
   {
      ETG_TRACE_USR1(("checkAndReleaseReservedDCC: entered"));
      Result result(CC_ERR_INT_NO_ERROR);
      BmResult bmResult(BM_RESULT_OK);

      DeviceConnectionControllerList dccInstances;
      result = getUsedDeviceConnectionControllers(dccInstances);

      if ((CC_ERR_INT_NO_ERROR == result) && (0u != dccInstances.size()))
      {
         // This is due to connection order remains same in the HMI
         DeviceConnectionControllerList orderedDccInstances;
         result = LocalSpm::getDbManager().getOrderedDeviceConnectionControllers(orderedDccInstances, dccInstances);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            for (size_t index = 0u; index < orderedDccInstances.size(); index++)
            {
               BtLimitationModeInfo btLimitationModeInfo;
               result = LocalSpm::getBmCoreMainController().getBtLimitationModeInfo(btLimitationModeInfo, orderedDccInstances[index]->getDeviceId());

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  //Check the device is used for CPW. If the CPW session is not Active then Deactivate the CPW internally
                  result = LocalSpm::getBmCoreMainController().checkAndDeactivateCPW(btLimitationModeInfo);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     bmResult = disconnectProtocols(orderedDccInstances[index], ProtocolList(), true, BM_DISCONNECTED_REASON_AUTOMATIC);

                     if (BM_RESULT_OK != bmResult)
                     {
                        ETG_TRACE_ERR(("checkAndReleaseReservedDCC: disconnecting a device failed (bmResult = %d)",
                              ETG_CENUM(BmResult, bmResult)));
                     }
                  }
               }
               else
               {
                  bmResult = disconnectProtocols(orderedDccInstances[index], ProtocolList(), true, BM_DISCONNECTED_REASON_AUTOMATIC);

                  if (BM_RESULT_OK != bmResult)
                  {
                     ETG_TRACE_ERR(("checkAndReleaseReservedDCC: disconnecting a device failed (bmResult = %d)",
                           ETG_CENUM(BmResult, bmResult)));
                  }
               }
            }
         }
      }
   }

   Result BmController::initiateSwitchingBtOff()
   {
      ENTRY

      ETG_TRACE_USR1(("initiateSwitchingBtOff"));

//      BmResult bmResult = this->doSwitchBtStatus(TARGET_SWITCH_STATE_SWITCHED_OFF, false, true, false);
      BmResult bmResult = LocalSpm::getBmCoreMainController().switchBtStatusInt(TARGET_SWITCH_STATE_SWITCHED_OFF,
            false, false);

      if (BM_RESULT_OK != bmResult)
      {
         ETG_TRACE_ERR(("initiateSwitchingBtOff: initiating switching BT off failed (bmResult = %d)",
               ETG_CENUM(BmResult, bmResult)));

         ETG_TRACE_USR4(("initiateSwitchingBtOff: sending event SWITCH_BT_OFF"));
         Result result = this->SendEventByName("SWITCH_BT_OFF", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("initiateSwitchingBtOff: could not send event SWITCH_BT_OFF (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::switchBtOff()
   {
      ENTRY

      ETG_TRACE_USR1(("switchBtOff"));

      LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().switchBluetoothOnOff(BTS_BT_MODE_OFF, BTS_ADAPTER_MODE_APP);

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::stopAllConnectionControllers()
   {
      ETG_TRACE_USR1(("stopAllConnectionControllers"));

      _numberConnCtrlStopping = 0u;

      (void) this->stopDeviceConnectionControllers();

      (void) this->stopProtocolConnectionControllers();

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::areAllConnCtrlStopped()
   {
      if (0u < _numberConnCtrlStopping)
      {
         _numberConnCtrlStopping--;
      }

      ETG_TRACE_USR1(("areAllConnCtrlStopped: number of not yet stopped connection controllers: %u",
            _numberConnCtrlStopping));

      if (0u == _numberConnCtrlStopping)
      {
         ETG_TRACE_USR4(("areAllConnCtrlStopped: sending event ALL_CONN_CTRL_STOPPED to myself"));

         Result result = this->SendEventByName("ALL_CONN_CTRL_STOPPED", (char *) 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("areAllConnCtrlStopped: could not send event ALL_CONN_CTRL_STOPPED to myself (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::handleBmControllerStopped()
   {
      ETG_TRACE_USR1(("handleBmControllerStopped"));

      LocalSpm::getDbManager().switchOffDatabaseAccess();

      LocalSpm::getBmCoreMainController().stopThread();

      (void) this->stopDone(0);

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::terminateAllConnectionControllers()
   {
      ETG_TRACE_USR1(("terminateAllConnectionControllers"));

      _numberConnCtrlStopping = 0u;

      (void) terminateDeviceConnectionControllers();

      (void) terminateProtocolConnectionControllers();

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::areAllConnCtrlTerminated()
   {
      if (0u < _numberConnCtrlStopping)
      {
         _numberConnCtrlStopping--;
      }

      ETG_TRACE_USR1(("areAllConnCtrlTerminated: number of not yet terminated connection controllers: %u",
            _numberConnCtrlStopping));

      if (0u == _numberConnCtrlStopping)
      {
         ETG_TRACE_USR4(("areAllConnCtrlTerminated: sending event ALL_CONN_CTRL_TERMINATED to myself"));

         Result result = this->SendEventByName("ALL_CONN_CTRL_TERMINATED", (char *) 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("areAllConnCtrlTerminated: could not send event ALL_CONN_CTRL_TERMINATED to myself (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::handleBmControllerTerminated()
   {
      ENTRY

      ETG_TRACE_USR1(("handleBmControllerTerminated"));

      (void) this->doneDone(0);

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::StopDeviceDiscovery()
   {
      ENTRY

      ETG_TRACE_USR1(("StopDeviceDiscovery"));
      SwitchStatus deviceDiscoveryStatus;
      BmResult bmResult(BM_RESULT_OK);

      //!Check for device discovery status. If device discovery is switched ON then switch it OFF
      bmResult = LocalSpm::getBmCoreMainController().getDiscoveryStatus(deviceDiscoveryStatus);

      if((BM_RESULT_OK == bmResult) && ((SWITCH_STATE_SWITCHED_ON == deviceDiscoveryStatus._switchState)
    		  || (SWITCH_STATE_SWITCHING_ON == deviceDiscoveryStatus._switchState)))
      {
         (void) LocalSpm::getBmCoreMainController().switchDiscoveryStatusInt(TARGET_SWITCH_STATE_SWITCHED_OFF);
      }//End of if((BM_RESULT_OK == bmResult) && ((SWITCH_STATE_SWITCHED_ON == deviceDiscoveryStatus._switchState)..)

      //!If device discovery status is OFF or if GetDiscoveryStatus resulted into error.
      else if(((BM_RESULT_OK == bmResult) && (SWITCH_STATE_SWITCHED_OFF == deviceDiscoveryStatus._switchState))
    		  || (BM_RESULT_OK != bmResult))
      {
          Result result = this->SendEventByName("DEVICE_DISCOVERY_STOPPED", 0);

          if (CC_ERR_INT_NO_ERROR != result)
          {
             ETG_TRACE_ERR(("StopDeviceDiscovery: could not send event DEVICE_DISCOVERY_STOPPED (error = %d)",
                   ETG_CENUM(CcErrorInternal, result)));
          }//End of if (CC_ERR_INT_NO_ERROR != result)
      }//End of else if(((BM_RESULT_OK == bmResult)) && (SWITCH_STATE_SWITCHED_OFF == deviceDiscoveryStatus._switchState)..)

      //!Else wait for device discovery to stop in case state is SWITCH_STATE_SWITCHING_OFF
      return CC_ERR_INT_NO_ERROR;
   }//!End of Result BmController::StopDeviceDiscovery()

   void BmController::create()
   {
      ENTRY

      ETG_TRACE_USR1(("create: LocalSpm triggered component creation"));

      BmControllerOnOffSm::Create();

      createDone(0);
   }

   Result BmController::init(InitReason reason)
   {
      ENTRY

      ETG_TRACE_USR1(("init: LocalSpm triggered component initialization (reason = %d)",
            ETG_CENUM(InitReason, reason)));

      // Protocol Connection controllers and Device Connection controllers
      // have to be created after DB is opened and re-initialized which is
      // done in DbManager's init() method (called whenever LocalSpm's
      // stateChangeNormal() is called)
      (void) createProtocolConnectionControllers();

      (void) createDeviceConnectionControllers();

      (void) initProtocolConnectionControllers();

      (void) initDeviceConnectionControllers();

      srand(static_cast<unsigned int>(time(0)));

      BmControllerOnOffSm::Init();
      SetAnswerTimeout(2000);

      // Register state machine with dispatcher
      Dispatcher::GetInstance().Register(IN this);

      return initDone(0);
   }

   Result BmController::run()
   {
      ENTRY

      ETG_TRACE_USR1(("run: LocalSpm triggered component start"));

      (void) runProtocolConnectionControllers();

      (void) runDeviceConnectionControllers();

      ETG_TRACE_USR4(("run: sending event RUN to BmControllerOnOffSm"));
      Result result = this->SendEventByName("RUN", (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("run: could not send event RUN (error = %d)", ETG_CENUM(CcErrorInternal, result)));
      }

      return 0;
   }

   Result BmController::stop()
   {
      ENTRY

      ETG_TRACE_USR1(("stop: LocalSpm triggered component stop"));

      ETG_TRACE_USR4(("stop: sending event STOP to BmControllerOnOffSm"));
      Result result = this->SendEventByName("STOP", (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("stop: could not send event STOP (error = %d)", ETG_CENUM(CcErrorInternal, result)));
      }

      return 0;
   }

   Result BmController::done()
   {
      ENTRY

      ETG_TRACE_USR1(("done: LocalSpm triggered component termination"));

      // Deregister state machine with dispatcher
      Dispatcher::GetInstance().DeRegister(IN this);

      ETG_TRACE_USR4(("done: sending event DONE to BmControllerOnOffSm"));
      Result result = this->SendForceEvent(DONE, (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("done: could not send event DONE (error = %d)", ETG_CENUM(CcErrorInternal, result)));
      }

      return 0;
   }

   int BmController::statistics(OUT Statistics stat)
   {
      ENTRY

      (void) stat;

      return 0;
   }

   Result BmController::initialize()
   {
      ENTRY

      ETG_TRACE_USR1(("initialize"));

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::createProtocolConnectionControllers(void)
   {
      ENTRY

      ETG_TRACE_USR1(("createProtocolConnectionControllers: creating PCC instances"));

      Result result(CC_ERR_INT_NO_ERROR);
      ProtocolConnectionControllerList pccInstances;

      result = LocalSpm::getDbManager().getProtocolConnectionControllers(pccInstances);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         // check if PCCs were already created because PCCs are not destroyed during state change to OFF (because SMF does not provide destroy functionality (memory leak))
         if (0 < pccInstances.size())
         {
            ETG_TRACE_USR1(("createProtocolConnectionControllers: PCC instances already exist"));

            // reset PCC instances
            LocalSpm::getDbManager().resetProtocolConnectionController();
         }
         else
         {
            ProtocolInfoMap protocolInfoMap;
            result = LocalSpm::getDataProvider().getProtocolInfos(protocolInfoMap, true);

            ProtocolInfoMap::iterator it;

            for (it = protocolInfoMap.begin(); it != protocolInfoMap.end(); it++)
            {
               uint8_t maxNumInstancesConfigured(0u);
               uint8_t maxNumInstances(0u);

               result = LocalSpm::getDataProvider().getMaxNumProtocolInstances(maxNumInstancesConfigured, it->first);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  if (((BM_PROTOCOL_ID_PBDL == it->first) || (BM_PROTOCOL_ID_MSG == it->first)) && (0u < maxNumInstancesConfigured))
                  {
                     // for protocols PBDL and MSG create one PCC instance more than actually configured
                     // in order to be able to handle a protocol connection request for a device while a disconnection
                     // request is still handled for another device
                     maxNumInstances = (uint8_t)(maxNumInstancesConfigured + 1u);
                  }
                  else
                  {
                     maxNumInstances = maxNumInstancesConfigured;
                  }

                  for (uint8_t instCnt = 0u; instCnt < maxNumInstances; instCnt++)
                  {
                    result = LocalSpm::getDbManager().createProtocol(it->first, it->second._name);

                    if (CC_ERR_INT_NO_ERROR != result)
                       {
                          ETG_TRACE_ERR(("createProtocolConnectionControllers: creating protocol for protocol ID = %d in DB failed (error = %d)",
                                it->first, ETG_CENUM(CcErrorInternal, result)));
                       }
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("createProtocolConnectionControllers: could not get maximum number of protocol instances for protocol ID = %d (error = %d)",
                        it->first, ETG_CENUM(CcErrorInternal, result)));
               }
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("createProtocolConnectionControllers: could not get PCC instances from DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result BmController::initProtocolConnectionControllers(void)
   {
      ENTRY

      ETG_TRACE_USR1(("initProtocolConnectionControllers: initializing PCC instances"));

      Result result(CC_ERR_INT_NO_ERROR);
      ProtocolConnectionControllerList pccInstances;

      result = LocalSpm::getDbManager().getProtocolConnectionControllers(pccInstances);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         for (size_t instanceIdx = 0u; instanceIdx < pccInstances.size(); instanceIdx++)
         {
            if (pccInstances[instanceIdx])
            {
               result = pccInstances[instanceIdx]->init();

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("initProtocolConnectionControllers: could initialize PCC instance 0x%p (error = %d)",
                        (void *) pccInstances[instanceIdx], ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               result = CC_ERR_INT_NULL_POINTER;
               ETG_TRACE_ERR(("initProtocolConnectionControllers: PCC instance pointer is 0 (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("initProtocolConnectionControllers: could not get PCC instances from DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result BmController::runProtocolConnectionControllers(void)
   {
      ENTRY

      ETG_TRACE_USR1(("runProtocolConnectionControllers: starting PCC instances"));

      Result result(CC_ERR_INT_NO_ERROR);
      ProtocolConnectionControllerList pccInstances;

      result = LocalSpm::getDbManager().getProtocolConnectionControllers(pccInstances);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         for (size_t instanceIdx = 0u; instanceIdx < pccInstances.size(); instanceIdx++)
         {
            if (pccInstances[instanceIdx])
            {
               result = pccInstances[instanceIdx]->run();

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("runProtocolConnectionControllers: could start PCC instance 0x%p (error = %d)",
                        (void *) pccInstances[instanceIdx], ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               result = CC_ERR_INT_NULL_POINTER;
               ETG_TRACE_ERR(("runProtocolConnectionControllers: PCC instance pointer is 0 (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("runProtocolConnectionControllers: could not get PCC instances from DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result BmController::stopProtocolConnectionControllers(void)
   {
      ENTRY

      ETG_TRACE_USR1(("stopProtocolConnectionControllers: stopping PCC instances"));

      Result result(CC_ERR_INT_NO_ERROR);
      ProtocolConnectionControllerList pccInstances;

      result = LocalSpm::getDbManager().getProtocolConnectionControllers(pccInstances);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         for (size_t instanceIdx = 0u; instanceIdx < pccInstances.size(); instanceIdx++)
         {
            if (pccInstances[instanceIdx])
            {
               result = pccInstances[instanceIdx]->stop();

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  _numberConnCtrlStopping++;
               }
               else
               {
                  ETG_TRACE_ERR(("stopProtocolConnectionControllers: could not stop PCC instance 0x%p (error = %d)",
                        (void *) pccInstances[instanceIdx], ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               result = CC_ERR_INT_NULL_POINTER;
               ETG_TRACE_ERR(("stopProtocolConnectionControllers: PCC instance pointer is 0 (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("stopProtocolConnectionControllers: could not get PCC instances from DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result BmController::terminateProtocolConnectionControllers(void)
   {
      ENTRY

      ETG_TRACE_USR1(("terminateProtocolConnectionControllers: terminating PCC instances"));

      Result result(CC_ERR_INT_NO_ERROR);
      ProtocolConnectionControllerList pccInstances;

      result = LocalSpm::getDbManager().getProtocolConnectionControllers(pccInstances);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         for (size_t instanceIdx = 0u; instanceIdx < pccInstances.size(); instanceIdx++)
         {
            if (pccInstances[instanceIdx])
            {
               result = pccInstances[instanceIdx]->done();

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  _numberConnCtrlStopping++;
               }
               else
               {
                  ETG_TRACE_ERR(("terminateProtocolConnectionControllers: could not terminate PCC instance 0x%p (error = %d)",
                        (void *) pccInstances[instanceIdx], ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               result = CC_ERR_INT_NULL_POINTER;
               ETG_TRACE_ERR(("terminateProtocolConnectionControllers: PCC instance pointer is 0 (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("terminateProtocolConnectionControllers: could not get PCC instances from DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result BmController::destroyProtocolConnectionControllers(void)
   {
      ENTRY

      ETG_TRACE_USR1(("destroyProtocolConnectionControllers: destroying PCC instances"));

      // do not destroy PCCs because SMF does not provide destroy functionality (memory leak)

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::createDeviceConnectionControllers(void)
   {
      ENTRY

      ETG_TRACE_USR1(("createDeviceConnectionControllers: creating DCC instances"));

      Result result(CC_ERR_INT_NO_ERROR);

      // check if DCCs were already created because DCCs are not destroyed during state change to OFF (because SMF does not provide destroy functionality (memory leak))
      if (0 < _deviceConnectionControllers.size())
      {
         ETG_TRACE_USR1(("createDeviceConnectionControllers: DCC instances already exist"));
      }
      else
      {
         for (uint8_t instCnt = 0u; instCnt < LocalSpm::getDataProvider().getBmCoreConfiguration()._maxNumConnectedDevices; instCnt++)
         {
            DeviceConnectionController* dccInstance = this->getNewDeviceConnectionController();

            if (dccInstance)
            {
               dccInstance->create();
               _deviceConnectionControllers.push_back(dccInstance);
            }
            else
            {
               result = CC_ERR_INT_MEMORY_ALLOCATION_ERROR;
               ETG_TRACE_ERR(("createDeviceConnectionControllers: could not create DCC instance (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }

      return result;
   }

   Result BmController::initDeviceConnectionControllers(void)
   {
      ENTRY

      ETG_TRACE_USR1(("initDeviceConnectionControllers: initializing DCC instances"));

      Result result(CC_ERR_INT_NO_ERROR);

      for (size_t instCnt = 0u; instCnt < _deviceConnectionControllers.size(); instCnt++)
      {
         if (_deviceConnectionControllers[instCnt])
         {
            result = _deviceConnectionControllers[instCnt]->init();

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("initDeviceConnectionControllers: could initialize DCC instance 0x%p (error = %d)",
                     (void *) _deviceConnectionControllers[instCnt], ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            result = CC_ERR_INT_NULL_POINTER;
            ETG_TRACE_ERR(("initDeviceConnectionControllers: DCC instance pointer is 0 (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return result;
   }

   Result BmController::runDeviceConnectionControllers(void)
   {
      ENTRY

      ETG_TRACE_USR1(("runDeviceConnectionControllers: starting DCC instances"));

      Result result(CC_ERR_INT_NO_ERROR);

      for (size_t instCnt = 0u; instCnt < _deviceConnectionControllers.size(); instCnt++)
      {
         if (_deviceConnectionControllers[instCnt])
         {
            result = _deviceConnectionControllers[instCnt]->run();

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("runDeviceConnectionControllers: could start DCC instance 0x%p (error = %d)",
                     (void *) _deviceConnectionControllers[instCnt], ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            result = CC_ERR_INT_NULL_POINTER;
            ETG_TRACE_ERR(("runDeviceConnectionControllers: DCC instance pointer is 0 (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return result;
   }

   Result BmController::stopDeviceConnectionControllers(void)
   {
      ENTRY

      ETG_TRACE_USR1(("stopDeviceConnectionControllers: stopping DCC instances"));

      Result result(CC_ERR_INT_NO_ERROR);

      for (size_t instCnt = 0u; instCnt < _deviceConnectionControllers.size(); instCnt++)
      {
         if (_deviceConnectionControllers[instCnt])
         {
            result = _deviceConnectionControllers[instCnt]->stop();

            if (CC_ERR_INT_NO_ERROR == result)
            {
               _numberConnCtrlStopping++;
            }
            else
            {
               ETG_TRACE_ERR(("stopDeviceConnectionControllers: could not stop DCC instance 0x%p (error = %d)",
                     (void *) _deviceConnectionControllers[instCnt], ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            result = CC_ERR_INT_NULL_POINTER;
            ETG_TRACE_ERR(("stopDeviceConnectionControllers: DCC instance pointer is 0 (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return result;
   }

   Result BmController::terminateDeviceConnectionControllers(void)
   {
      ENTRY

      ETG_TRACE_USR1(("terminateDeviceConnectionControllers: terminating DCC instances"));

      Result result(CC_ERR_INT_NO_ERROR);

      for (size_t instCnt = 0u; instCnt < _deviceConnectionControllers.size(); instCnt++)
      {
         if (_deviceConnectionControllers[instCnt])
         {
            result = _deviceConnectionControllers[instCnt]->done();

            if (CC_ERR_INT_NO_ERROR == result)
            {
               _numberConnCtrlStopping++;
            }
            else
            {
               ETG_TRACE_ERR(("terminateDeviceConnectionControllers: could not terminate DCC instance 0x%p (error = %d)",
                     (void *) _deviceConnectionControllers[instCnt], ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            result = CC_ERR_INT_NULL_POINTER;
            ETG_TRACE_ERR(("terminateDeviceConnectionControllers: DCC instance pointer is 0 (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return result;
   }

   Result BmController::destroyDeviceConnectionControllers(void)
   {
      ENTRY

      ETG_TRACE_USR1(("destroyDeviceConnectionControllers: destroying DCC instances"));

      // do not destroy DCCs because SMF does not provide destroy functionality (memory leak)

      return CC_ERR_INT_NO_ERROR;
   }

   Result BmController::getUnusedDeviceConnectionController(OUT DeviceConnectionController** dccInstance)
   {
      ENTRY

      ETG_TRACE_USR1(("getUnusedDeviceConnectionController: looking for an unused DCC instance"));

      Result result(CC_ERR_INT_DCC_ALL_IN_USE);
      *dccInstance = 0;
      size_t index = 0u;

      //!Fix for CID 21104: Unused value (UNUSED_VALUE)
      for (index = 0u; index < _deviceConnectionControllers.size(); index++)
      {
          if (false == _deviceConnectionControllers[index]->isInUse())
          {
        	  *dccInstance = _deviceConnectionControllers[index];
        	  break;
          }
      }

      result = (0 == *dccInstance) ? (CC_ERR_INT_NULL_POINTER):(CC_ERR_INT_NO_ERROR);

      ETG_TRACE_USR4(("getUnusedDeviceConnectionController: unused DCC instance %sfound", (0 == *dccInstance) ? "not " : ""));

      return result;
   }

   Result BmController::getReleasableDeviceConnectionControllers(OUT DeviceConnectionControllerList& dccInstances, OUT DeviceId& sendBtLimitationDeactivateToDeviceId)
   {
      ENTRY

      ETG_TRACE_USR1(("getReleasableDeviceConnectionControllers: looking for DCC instances that can be or will get released soon"));

      Result result(CC_ERR_INT_NO_ERROR);
      Result dbResult(CC_ERR_INT_NO_ERROR);
      ConnectionStatus connectionStatus(BM_CONNECTION_STATUS_UNKNOWN);
      DisconnectedReason disconnectedReason(BM_DISCONNECTED_REASON_UNKNOWN);
      BtLimitationModeInfo btLimitationModeInfo;
      DeviceId deviceId(0u);

      dccInstances.clear();

      for (size_t index = 0u; index < _deviceConnectionControllers.size(); index++)
      {
         if (true == _deviceConnectionControllers[index]->isInUse())
         {
            ETG_TRACE_USR4(("getReleasableDeviceConnectionControllers: DCC 0x%p is in use", _deviceConnectionControllers[index]));

            if(true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported)
            {
               result = LocalSpm::getBmCoreMainController().getBtLimitationModeInfo(btLimitationModeInfo, _deviceConnectionControllers[index]->getDeviceId());

               if(CC_ERR_INT_NO_ERROR == result)
               {
                  if((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == btLimitationModeInfo._limitationMode._limitationCommunicationIf)
                        ||(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == btLimitationModeInfo._limitationMode._limitationCommunicationIf)
                        ||(BM_LIMITATION_COMMUNICATION_IF_WIFI == btLimitationModeInfo._limitationMode._limitationCommunicationIf))
                  {
                     if((BM_LIMITATION_FEATURE_CAR_PLAY == btLimitationModeInfo._limitationMode._limitationFeature) &&
                           ((BM_LIMITATION_STATE_ACTIVATING != btLimitationModeInfo._limitationState) &&
                                 (BM_LIMITATION_STATE_ACTIVE != btLimitationModeInfo._limitationState)))
                     {
                        deviceId = _deviceConnectionControllers[index]->getDeviceId();
                     }
                  }
               }
               else
               {
                  // set the default value for result as NO_ERROR if no limitationmode is available for the device
                  result = CC_ERR_INT_NO_ERROR;
               }
            }

            if(0u == deviceId)
            {
               // Check the Lost device
               if (true == _deviceConnectionControllers[index]->isHandlingLostDevice())
               {
                  ETG_TRACE_USR4(("getReleasableDeviceConnectionControllers: DCC 0x%p is busy with re-connection of a previously Bt lost device", (void *) _deviceConnectionControllers[index]));

                  dccInstances.push_back(_deviceConnectionControllers[index]);
               }
               else if (0 == strncmp(_deviceConnectionControllers[index]->GetCurrentState(), "Disconnecting", strlen(_deviceConnectionControllers[index]->GetCurrentState())))
               {
                  ETG_TRACE_USR4(("getReleasableDeviceConnectionControllers: DCC 0x%p is disconnecting a device (state = %50s)", (void *) _deviceConnectionControllers[index], _deviceConnectionControllers[index]->GetCurrentState()));

                  dccInstances.push_back(_deviceConnectionControllers[index]);
               }
               else if (0 == strncmp(_deviceConnectionControllers[index]->GetCurrentState(), "DisconnectionFailed", strlen(_deviceConnectionControllers[index]->GetCurrentState())))
               {
                  ETG_TRACE_USR4(("getReleasableDeviceConnectionControllers: DCC 0x%p is disconnecting a device (state = %50s)", (void *) _deviceConnectionControllers[index], _deviceConnectionControllers[index]->GetCurrentState()));

                  dccInstances.push_back(_deviceConnectionControllers[index]);
               }
            }
            else
            {
               sendBtLimitationDeactivateToDeviceId = deviceId;
               deviceId = 0u;
            }
         }
      }

      DeviceId deviceIdAutoConnection = LocalSpm::getAutoConnectionController().getDeviceIdConnectionInProgress();

      if ((0u != deviceIdAutoConnection) && (deviceIdAutoConnection != sendBtLimitationDeactivateToDeviceId))
      {
         dbResult = LocalSpm::getDbManager().getDeviceConnectionStatus(connectionStatus, disconnectedReason, deviceIdAutoConnection);

         if (CC_ERR_INT_NO_ERROR == dbResult)
         {
            if (BM_CONNECTION_STATUS_CONNECTED != connectionStatus)
            {
               // Usecase: Device is lost and BMApp retries to connect the device.
               // And autoconnection is also running for that device due to CP/AA USB disconnection.

               // Check the DCC instance if already is added in the list. If not available then added into the List.
               DeviceConnectionController* dccInstanceAutoConnection(0);
               dbResult = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstanceAutoConnection, deviceIdAutoConnection);

               if (CC_ERR_INT_NO_ERROR == dbResult)
               {
                  bool available = false;

                  for (size_t dccIdx = 0u; dccIdx < dccInstances.size(); dccIdx++)
                  {
                     if(dccInstances[dccIdx] == dccInstanceAutoConnection)
                     {
                        available = true;
                        break;
                     }
                  }

                  if(false == available)
                  {

                     ETG_TRACE_USR4(("getReleasableDeviceConnectionControllers: DCC 0x%p is busy with auto connecting a device and has to be released",
                           (void *) dccInstanceAutoConnection));
                     dccInstances.push_back(dccInstanceAutoConnection);
                  }
                  else
                  {
                     ETG_TRACE_USR4(("getReleasableDeviceConnectionControllers:  DCC 0x%p is busy with auto connecting a device is already added",
                           (void *) dccInstanceAutoConnection));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("getReleasableDeviceConnectionControllers: could not get DCC instance handling device with ID = %d from DB (error = %d)",
                        deviceIdAutoConnection, ETG_CENUM(CcErrorInternal, dbResult)));
               }
            }
         }
         else
         {
            ETG_TRACE_ERR(("getReleasableDeviceConnectionControllers: could not get device connection status of device with ID = %d from DB (error = %d)",
                  deviceIdAutoConnection, ETG_CENUM(CcErrorInternal, dbResult)));
         }
      }

      ETG_TRACE_USR4(("getReleasableDeviceConnectionControllers: number of DCC instances that can be or will get released soon: %d and sendBtLimitationDeactivateToDeviceId - %d",
            dccInstances.size(), sendBtLimitationDeactivateToDeviceId));

      return result;
   }

   Result BmController::assignDeviceConnectionController(OUT DeviceConnectionController** dccInstance, IN const DeviceId deviceId,
         IN const ProtocolList& protocolList, IN const ConnectionRequestOrigin origin)
   {
      ENTRY

      ETG_TRACE_USR1(("assignDeviceConnectionController: deviceId = %d", deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      *dccInstance = 0;

      result = LocalSpm::getDbManager().getDeviceConnectionController(dccInstance, deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         // In DB a DeviceConnectionController instance
         // is already assigned to the device with the ID deviceId

         ETG_TRACE_USR1(("assignDeviceConnectionController(deviceId = %d): found already assigned DCC instance", deviceId));
      }
      else if (CC_ERR_INT_DB_DCC_HANDLE_NOT_FOUND == result)
      {
         // In DB a DeviceConnectionController instance
         // is not assigned to the device with the device ID deviceId

         ETG_TRACE_USR1(("assignDeviceConnectionController(deviceId = %d): no DCC instance is currently assigned to the given device", deviceId));

         result = getUnusedDeviceConnectionController(dccInstance);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            bool assignDccInstance = false;
            bool available = false;

            // Check the protocol connection controller is available for the requested protocolList
            for (size_t idxGiven = 0u; idxGiven < protocolList.size(); ++idxGiven)
            {
               result = LocalSpm::getDbManager().isUnusedProtocolConnectionControllerAvailable(available,
                     protocolList[idxGiven]._protocolId, protocolList[idxGiven]._uuid);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  if(true == available)
                  {
                     BlockState protocolBlockState(BM_BLOCK_STATE_UNBLOCKED);

                     // Usecase: CPW is active for iPhone and Andriod device is paired. User switches OFF and ON the BT.
                     // BMApp starts autoconnection and send the connection request for Andriod device.

                     // DCC should be assigned only if the requested protocols are not blocked.
                     result = LocalSpm::getDbManager().getProtocolBlockState(protocolBlockState,
                           deviceId, protocolList[idxGiven]);

                     if (CC_ERR_INT_NO_ERROR == result)
                     {
                        if(BM_BLOCK_STATE_UNBLOCKED == protocolBlockState)
                        {
                           assignDccInstance = true;
                        }
                     }
                     else
                     {
                        ETG_TRACE_ERR(("assignDeviceConnectionController(deviceId = %d): could not get protocol's block state (error = %d), assuming protocol being unblocked",
                              deviceId, ETG_CENUM(CcErrorInternal, result)));
                     }
                  }
               }
            }

            if(true == assignDccInstance)
            {
               // An unused DeviceConnectionController
               // instance could be found

               ETG_TRACE_USR1(("assignDeviceConnectionController(deviceId = %d): found an unused DCC instance", deviceId));

               result = (*dccInstance)->assign(deviceId);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  ETG_TRACE_USR1(("assignDeviceConnectionController(deviceId = %d): assigned device ID = %d to DCC instance",
                        deviceId, deviceId));

                  // It is used to identifies the CPW need to start for this device or not.
                  (*dccInstance)->setConnectionRequestOrigin(origin);
               }
               else
               {
                  // unexpected serious error case
                  ETG_TRACE_ERR(("assignDeviceConnectionController(deviceId = %d): could not assign deviceId = %d to DCC instance (error = %d)",
                        deviceId, deviceId, ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               // All available ProtocolConnectionController instances
               // are already assigned to other devices
               ETG_TRACE_USR1(("assignDeviceConnectionController(deviceId = %d): all available DCC instances are already assigned to other devices",
                     deviceId));
               result = CC_ERR_INT_PCC_NO_RESOURCES;
            }
         }
         else if (CC_ERR_INT_DCC_ALL_IN_USE == result)
         {
            // All available DeviceConnectionController instances
            // are already assigned to other devices
            ETG_TRACE_USR1(("assignDeviceConnectionController(deviceId = %d): all available DCC instances are already assigned to other devices",
                  deviceId));
         }
         else
         {
            // unexpected serious error case
            ETG_TRACE_ERR(("assignDeviceConnectionController(deviceId = %d): getting an unused DCC instance failed (error = %d)",
                  deviceId, ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         // unexpected serious error case
         ETG_TRACE_ERR(("assignDeviceConnectionController(deviceId = %d): getting DCC instance from DB failed (error = %d)",
               deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result BmController::assignDeviceConnectionController(OUT DeviceConnectionController** dccInstance, IN const BdAddress& bdAddress,
         IN const ProtocolList& protocolList, IN const ConnectionRequestOrigin origin)
   {
      ENTRY

      ETG_TRACE_USR1(("assignDeviceConnectionController: bdAddress = \"%50s\"", bdAddress.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);
      DeviceId deviceId(0u);

      *dccInstance = 0;

      result = LocalSpm::getDbManager().getDeviceId(OUT deviceId, IN bdAddress);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         result = assignDeviceConnectionController(OUT dccInstance, IN deviceId, IN protocolList, IN origin);
      }
      else
      {
         ETG_TRACE_ERR(("assignDeviceConnectionController(bdAddress = \"%50s\"): could not get deviceId for device with BD address = \"%50s\" from DbManager (error = %d)",
               bdAddress.c_str(), bdAddress.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result BmController::handleConnectionRequest(IN const bool deferredRequest)
   {
      ENTRY

      ETG_TRACE_USR1(("handleConnectionRequest: deferredRequest = %s", deferredRequest ? "true" : "false"));

      Result result(CC_ERR_INT_NO_ERROR);

      if ((false == _requestQueue.empty()) && (BM_REQUEST_TYPE_CONNECTION == _requestQueue[0]->_requestType))
      {
         ConnectionRequestData* connectionRequestData = static_cast<ConnectionRequestData*>(_requestQueue[0]);
         DeviceConnectionController* dccInstance(0);

         ProtocolList givenProtocols = connectionRequestData->_protocolList;
         ProtocolList consideredProtocols;
         bool isSlaveProtocolId(false);
         bool considerProtocol(false);

         for (size_t idxGiven = 0u; idxGiven < givenProtocols.size(); ++idxGiven)
         {
            considerProtocol = false;
            isSlaveProtocolId = false;

            if (CC_ERR_INT_NO_ERROR == LocalSpm::getDataProvider().isSlaveProtocolId(isSlaveProtocolId, givenProtocols[idxGiven]._protocolId))
            {
               ETG_TRACE_USR1(("handleConnectionRequest(deferredRequest = %10s): given protocol = (%d, \"%50s\") is%10s a slave protocol",
                     deferredRequest ? "true" : "false", ETG_CENUM(ProtocolId, givenProtocols[idxGiven]._protocolId),
                     givenProtocols[idxGiven]._uuid.c_str(), isSlaveProtocolId ? "" : " NOT"));

               if (true == isSlaveProtocolId)
               {
                  // given protocol ID is a slave protocol ID

                  ProtocolIdList masterProtocolIds;

                  result = LocalSpm::getDataProvider().getMasterProtocolIds(masterProtocolIds, givenProtocols[idxGiven]._protocolId);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     ETG_TRACE_USR1(("handleConnectionRequest(deferredRequest = %10s): master protocol IDs for given slave protocol = (%d, \"%50s\"):",
                           deferredRequest ? "true" : "false", ETG_CENUM(ProtocolId, givenProtocols[idxGiven]._protocolId),
                           givenProtocols[idxGiven]._uuid.c_str()));

                     VARTRACE(masterProtocolIds);

                     bool masterProtocolIdFoundInGivenProtocols(false);

                     for (size_t idx = 0u; idx < givenProtocols.size(); ++idx)
                     {
                        if (std::find(masterProtocolIds.begin(), masterProtocolIds.end(), givenProtocols[idx]._protocolId)
                              != masterProtocolIds.end())
                        {
                           masterProtocolIdFoundInGivenProtocols = true;
                           break;
                        }
                     }

                     ETG_TRACE_USR1(("handleConnectionRequest(deferredRequest = %10s): a master protocol ID for given slave protocol = (%d, \"%50s\") has%10s been found in given protocol list",
                           deferredRequest ? "true" : "false", ETG_CENUM(ProtocolId, givenProtocols[idxGiven]._protocolId),
                           givenProtocols[idxGiven]._uuid.c_str(), masterProtocolIdFoundInGivenProtocols ? "" : " NOT"));

                     if (true == masterProtocolIdFoundInGivenProtocols)
                     {
                        // consider given slave protocol
                        considerProtocol = true;
                     }
                     else
                     {
                        result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, connectionRequestData->_deviceId);

                        if (CC_ERR_INT_NO_ERROR == result)
                        {
                           // In DB a DeviceConnectionController instance
                           // is already assigned to the device with the given device ID

                           ETG_TRACE_USR1(("handleConnectionRequest(deferredRequest = %10s): a DCC instance is already assigned to given device ID = %d",
                                 deferredRequest ? "true" : "false", connectionRequestData->_deviceId));

                           ProtocolConnectionControllerList pccInstances;

                           result = LocalSpm::getDbManager().getProtocolConnectionControllers(pccInstances, connectionRequestData->_deviceId);

                           if (CC_ERR_INT_NO_ERROR == result)
                           {
                              bool masterProtocolIdFoundInAssignedPccs(false);

                              for (size_t pccIdx = 0u; pccIdx < pccInstances.size(); ++pccIdx)
                              {
                                 if (std::find(masterProtocolIds.begin(), masterProtocolIds.end(), pccInstances[pccIdx]->getProtocolId())
                                       != masterProtocolIds.end())
                                 {
                                    ConnectionStatus connectionStatus = BM_CONNECTION_STATUS_UNKNOWN;
                                    DisconnectedReason disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;
                                    Uuid uuid = "";

                                    result = LocalSpm::getDbManager().getProtocolConnectionStatus(connectionStatus, disconnectedReason,
                                          connectionRequestData->_deviceId, pccInstances[pccIdx]->getProtocolId(), uuid);

                                    if (CC_ERR_INT_NO_ERROR == result)
                                    {
                                       if ((BM_CONNECTION_STATUS_CONNECTING == connectionStatus) || (BM_CONNECTION_STATUS_CONNECTED == connectionStatus))
                                       {
                                          masterProtocolIdFoundInAssignedPccs = true;
                                          break;
                                       }
                                    }
                                 }
                              }

                              ETG_TRACE_USR1(("handleConnectionRequest(deferredRequest = %10s): a PCC handling a master protocol for given slave protocol = (%d, \"%50s\") is%10s assigned to DCC handling the given device",
                                    deferredRequest ? "true" : "false", ETG_CENUM(ProtocolId, givenProtocols[idxGiven]._protocolId),
                                    givenProtocols[idxGiven]._uuid.c_str(), masterProtocolIdFoundInAssignedPccs ? "" : " NOT"));

                              if (true == masterProtocolIdFoundInAssignedPccs)
                              {
                                 // one of the already assigned PCCs handles a master protocol of the given slave protocol
                                 considerProtocol = true;
                              }
                              else
                              {
                                 // given slave protocol is NOT considered
                              }
                           }
                           else
                           {
                              ETG_TRACE_ERR(("handleConnectionRequest(deferredRequest = %10s): getting all PCC instances assigned to the given device failed (error = %d)",
                                    deferredRequest ? "true" : "false", ETG_CENUM(CcErrorInternal, result)));
                           }
                        }
                        else if (CC_ERR_INT_DB_DCC_HANDLE_NOT_FOUND == result)
                        {
                           // In DB a DeviceConnectionController instance
                           // is not assigned to the device with the given device ID

                           ETG_TRACE_USR1(("handleConnectionRequest(deferredRequest = %10s): no DCC instance is currently assigned to the given device ID",
                                 deferredRequest ? "true" : "false"));

                           result = CC_ERR_INT_NO_ERROR;

                           // given slave protocol is NOT considered
                        }
                        else
                        {
                           // unexpected serious error case
                           ETG_TRACE_ERR(("handleConnectionRequest(deferredRequest = %10s): getting DCC instance from DB failed (error = %d)",
                                 deferredRequest ? "true" : "false", ETG_CENUM(CcErrorInternal, result)));

                           // given slave protocol is NOT considered

                        }
                     }
                  }
                  else
                  {
                     // error: could not determine master protocol IDs for given slave protocol
                     ETG_TRACE_ERR(("handleConnectionRequest(deferredRequest = %10s): could not get master protocol IDs for given slave protocol = (%d, \"%50s\") (error = %d)",
                           deferredRequest ? "true" : "false", ETG_CENUM(ProtocolId, givenProtocols[idxGiven]._protocolId),
                           givenProtocols[idxGiven]._uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
                  }
                  // consider IAP2BT SPP profile
                  if((CC_ERR_INT_NO_ERROR == result) && (considerProtocol == false))
                  {
                     if(true == (checkSupportedUUID(connectionRequestData->_deviceId, givenProtocols[idxGiven]._protocolId, givenProtocols[idxGiven]._uuid, connectionRequestData->_origin, connectionRequestData->_internal )))
                     {
                        considerProtocol = true;
                     }
                  }
               }
               else
               {
                  // given protocol ID is NOT a slave protocol ID, consider given protocol
                  considerProtocol = true;
               }
            }
            else
            {
               // error: could not determine if given protocol ID is a slave protocol ID
               ETG_TRACE_ERR(("handleConnectionRequest(deferredRequest = %10s): could not determine if given protocol = (%d, \"%50s\") is a slave protocol (error = %d)",
                     deferredRequest ? "true" : "false", ETG_CENUM(ProtocolId, givenProtocols[idxGiven]._protocolId),
                     givenProtocols[idxGiven]._uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
            }

            if (true == considerProtocol)
            {
               consideredProtocols.push_back(givenProtocols[idxGiven]);
            }
            else
            {
               if (CC_ERR_INT_NO_ERROR == result)
               {
                  ETG_TRACE_ERR(("handleConnectionRequest(deferredRequest = %10s): given slave protocol = (%d, \"%50s\") is not allowed to get connected without an associated master protocol",
                        deferredRequest ? "true" : "false", ETG_CENUM(ProtocolId, givenProtocols[idxGiven]._protocolId),
                        givenProtocols[idxGiven]._uuid.c_str()));
               }
               else
               {
                  ETG_TRACE_ERR(("handleConnectionRequest(deferredRequest = %10s): given protocol = (%d, \"%50s\") is not considered for connection due to a general error",
                        deferredRequest ? "true" : "false", ETG_CENUM(ProtocolId, givenProtocols[idxGiven]._protocolId),
                        givenProtocols[idxGiven]._uuid.c_str()));
               }
            }
         }

         ETG_TRACE_USR1(("handleConnectionRequest(deferredRequest = %10s): protocols considered for connection:",
               deferredRequest ? "true" : "false"));

         VARTRACE(consideredProtocols);

         if (false == consideredProtocols.empty())
         {
            result = assignDeviceConnectionController(&dccInstance, connectionRequestData->_deviceId, consideredProtocols, connectionRequestData->_origin);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               if (true == connectionRequestData->_delayConnection)
               {
                  ETG_TRACE_USR1(("handleConnectionRequest(deviceId = %d, connectionRequestOrigin = %d): setting local protocol connection delay in DCC instance",
                        connectionRequestData->_deviceId, ETG_CENUM(ConnectionRequestOrigin, connectionRequestData->_origin)));

                  if (0u < LocalSpm::getDataProvider().getBmCoreConfiguration()._localProtocolConnectingDelaySeconds)
                  {
                     dccInstance->setLocalProtocolConnectionsToBeDelayed();
                  }
               }

               ETG_TRACE_USR1(("handleConnectionRequest(deviceId = %d, connectionRequestOrigin = %d): requesting DCC instance to handle protocol connection request(s)",
                     connectionRequestData->_deviceId, ETG_CENUM(ConnectionRequestOrigin, connectionRequestData->_origin)));

               result = dccInstance->requestProtocolConnections(consideredProtocols, connectionRequestData->_pageTimeout, connectionRequestData->_origin);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  ETG_TRACE_USR1(("handleConnectionRequest(deviceId = %d, connectionRequestOrigin = %d): protocol connection request(s) has(have) been accepted by DCC instance",
                        connectionRequestData->_deviceId, ETG_CENUM(ConnectionRequestOrigin, connectionRequestData->_origin)));
               }
               else if (CC_ERR_INT_PCC_NO_RESOURCES == result)
               {
                  ETG_TRACE_ERR(("handleConnectionRequest(deviceId = %d, connectionRequestOrigin = %d): all available PCC instances are already in use",
                        connectionRequestData->_deviceId, connectionRequestData->_origin));
               }
               else
               {
                  ETG_TRACE_ERR(("handleConnectionRequest(deviceId = %d, connectionRequestOrigin = %d): DCC instance could not handle the protocol connection request (error = %d)",
                        connectionRequestData->_deviceId, ETG_CENUM(ConnectionRequestOrigin, connectionRequestData->_origin),
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else if (CC_ERR_INT_DCC_ALL_IN_USE == result)
            {
               ETG_TRACE_ERR(("handleConnectionRequest(deviceId = %d, connectionRequestOrigin = %d): all available DCC instances are already in use",
                     connectionRequestData->_deviceId, connectionRequestData->_origin));
            }
            else
            {
               ETG_TRACE_ERR(("handleConnectionRequest(deviceId = %d, connectionRequestOrigin = %d): getting a DCC instance for handling the device connection request failed (error = %d)",
                     connectionRequestData->_deviceId, ETG_CENUM(ConnectionRequestOrigin, connectionRequestData->_origin),
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            result = CC_ERR_INT_MASTER_PROTOCOL_NOT_CONNECTED;
            ETG_TRACE_ERR(("handleConnectionRequest(deviceId = %d, connectionRequestOrigin = %d): no master protocol connected (error = %d)",
                  connectionRequestData->_deviceId, ETG_CENUM(ConnectionRequestOrigin, connectionRequestData->_origin),
                  ETG_CENUM(CcErrorInternal, result)));
         }

         if ((CC_ERR_INT_NO_ERROR != result) && (true == deferredRequest))
         {
            /*
         Result tmpResult = LocalSpm::getDbManager().setDeviceConnectionStatus(connectionRequestData->_deviceId, BM_CONNECTION_STATUS_CONNECTING, BM_DISCONNECTED_REASON_NOT_APPLICABLE);

         if (CC_ERR_INT_NO_ERROR != tmpResult)
         {
            ETG_TRACE_ERR(("handleConnectionRequest(deviceId = %d, connectionRequestOrigin = %d): could not set device's connection status in DB (error = %d)", connectionRequestData->_deviceId, ETG_CENUM(ConnectionRequestOrigin, connectionRequestData->_origin), tmpResult));
         }

         tmpResult = LocalSpm::getDbManager().setDeviceConnectionStatus(connectionRequestData->_deviceId, BM_CONNECTION_STATUS_DISCONNECTED, BM_DISCONNECTED_REASON_CONNECTION_FAILED);

         if (CC_ERR_INT_NO_ERROR != tmpResult)
         {
            ETG_TRACE_ERR(("handleConnectionRequest(deviceId = %d, connectionRequestOrigin = %d): could not set device's connection status in DB (error = %d)", connectionRequestData->_deviceId, ETG_CENUM(ConnectionRequestOrigin, connectionRequestData->_origin), tmpResult));
         }
             */
            if (BM_CONNECTION_REQUEST_ORIGIN_REMOTE == connectionRequestData->_origin)
            {
               BdAddress bdAddress("");
               Result tmpResult = LocalSpm::getDbManager().getBdAddress(bdAddress, connectionRequestData->_deviceId);

               if (CC_ERR_INT_NO_ERROR == tmpResult)
               {
                  BTSProtocolId btsProtocolId(BTS_PROTO_LAST);
                  tmpResult = getBtsProtocolIdFromBmProtocolId(btsProtocolId, connectionRequestData->_protocolList[0]._protocolId);

                  if (CC_ERR_INT_NO_ERROR == tmpResult)
                  {
                     ETG_TRACE_USR1(("handleConnectionRequest(deferredRequest = %10s): rejecting remote protocol connection request at BtStackIf",
                           deferredRequest ? "true" : "false"));

                     LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().rejectRemoteProtocolConnect(bdAddress, btsProtocolId,
                           connectionRequestData->_protocolList[0]._uuid);
                  }
                  else
                  {
                     ETG_TRACE_ERR(("handleConnectionRequest(deviceId = %d, connectionRequestOrigin = %d): could not map BM core's protocol ID to corresponding BtStackIf's protocol ID (error = %d)",
                           connectionRequestData->_deviceId, ETG_CENUM(ConnectionRequestOrigin, connectionRequestData->_origin),
                           ETG_CENUM(CcErrorInternal, tmpResult)));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("handleConnectionRequest(deviceId = %d, connectionRequestOrigin = %d): could not get BD address from DB (error = %d)",
                        connectionRequestData->_deviceId, ETG_CENUM(ConnectionRequestOrigin, connectionRequestData->_origin),
                        ETG_CENUM(CcErrorInternal, tmpResult)));
               }
            }
         }
      }
      else
      {
         result = CC_ERR_INT_GENERAL_ERROR;
         ETG_TRACE_ERR(("handleConnectionRequest: either request queue is empty or request is no connection request (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result BmController::getProtocolsForConnectionRestoring(OUT ProtocolList& protocolList, IN const DeviceId deviceId,
         bool ignoreProtocolsLastConnected)
   {
      ENTRY

      ETG_TRACE_USR1(("getProtocolsForConnectionRestoring: deviceId = %d, ignoreProtocolsLastConnected = %10s",
            deviceId, ignoreProtocolsLastConnected ? "true" : "false"));

      Result result(CC_ERR_INT_NO_ERROR);
      ProtocolList tmpProtocolList;
      bool ignorePanProtocolLastConnected = false;

      protocolList.clear();

      result = LocalSpm::getDbManager().getProtocolsExpectedToBeConnected(tmpProtocolList, deviceId);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("getProtocolsForConnectionRestoring: could not get protocols expected to be connected from DB (error = %d) ",
               ETG_CENUM(CcErrorInternal, result)));
      }

      if (true == tmpProtocolList.empty())
      {
         ETG_TRACE_USR3(("getProtocolsForConnectionRestoring: no protocols expected to be connected for device with given ID"));

         ignorePanProtocolLastConnected = true;

         if (false == ignoreProtocolsLastConnected)
         {
            ETG_TRACE_USR3(("getProtocolsForConnectionRestoring: considering all protocols"));

            result = LocalSpm::getDbManager().getProtocolsLastConnected(tmpProtocolList, deviceId);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("getProtocolsForConnectionRestoring: could not get last connected protocols from DB (error = %d) ",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }

      std::sort(tmpProtocolList.begin(), tmpProtocolList.end(), LocalSpm::getDbManager().compareProtocolIds);

      ProtocolIdList deviceConnectionProtocols;

      result = LocalSpm::getDataProvider().getDeviceConnectionProtocols(deviceConnectionProtocols);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR3(("getProtocolsForConnectionRestoring: taking only device connection protocols into account"));

         for (size_t i = 0u; i < tmpProtocolList.size(); i++)
         {
            if (std::find(deviceConnectionProtocols.begin(), deviceConnectionProtocols.end(),
                  tmpProtocolList[i]._protocolId) != deviceConnectionProtocols.end())
            {
               protocolList.push_back(tmpProtocolList[i]);
            }
            else
            {
               // Added the PAN protocol only if the PAN is expected to be connected
               if ((BM_PROTOCOL_ID_PAN == tmpProtocolList[i]._protocolId) && (false == ignorePanProtocolLastConnected))
               {
                  protocolList.push_back(tmpProtocolList[i]);
               }
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("getProtocolsForConnectionRestoring: could not get device connection protocols from DataProvider (error = %d) ",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   BmResult BmController::disconnectProtocols(IN DeviceConnectionController* const& dccInstance, IN const ProtocolList& protocolList,
         IN const bool disconnectAll, IN const DisconnectedReason disconnectedReason, IN const bool deleteDevice, IN const bool ignoreStopAutoconnection)
   {
      ENTRY

      ETG_TRACE_USR1(("disconnectProtocols: dccInstance = 0x%p, disconnectAll = %d, disconnectedReason = %d, deleteDevice = %d, ignoreStopAutoconection = %10s",
            (void *) dccInstance, disconnectAll, ETG_CENUM(DisconnectedReason, disconnectedReason), deleteDevice, ignoreStopAutoconnection ? "true" : "false"));
      ETG_TRACE_USR1(("disconnectProtocols: protocolList:"));
      VARTRACE(protocolList);

      Result result(CC_ERR_INT_NO_ERROR);
      BmResult bmResult(BM_RESULT_OK);

      if (0 != dccInstance)
      {
          if(ignoreStopAutoconnection == false)
          {
              bool stopAutoConnection(false);

              if (false == disconnectAll)
              {
                  ProtocolIdList deviceConnectionProtocols;
                  result = LocalSpm::getDataProvider().getDeviceConnectionProtocols(deviceConnectionProtocols);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                      for (size_t i = 0u; i < protocolList.size(); ++i)
                      {
                          if (std::find(deviceConnectionProtocols.begin(), deviceConnectionProtocols.end(),
                          protocolList[i]._protocolId) != deviceConnectionProtocols.end())
                          {
                              // auto connection will be stopped if given protocol list includes a device connection protocol
                              stopAutoConnection = true;
                          }
                      }
                  }
                  else
                  {
                      ETG_TRACE_ERR(("disconnectProtocols: could not get device connection protocols from DataProvider (error = %d)",
                      ETG_CENUM(CcErrorInternal, result)));
                  }
              }
              else
              {
                  stopAutoConnection = true;
              }

              if (true == stopAutoConnection)
              {
                  ETG_TRACE_USR4(("disconnectProtocols: sending event STOP_AUTO_CONNECTION to AutoConnectionControllerSm"));
                  result = LocalSpm::getAutoConnectionController().SendUrgentEvent(LocalSpm::getAutoConnectionController().STOP_AUTO_CONNECTION, (char *) 0);

                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                      ETG_TRACE_ERR(("disconnectProtocols: could not send event STOP_AUTO_CONNECTION (error = %d)",
                      ETG_CENUM(CcErrorInternal, result)));
                  }
              }
          }

          if (true == disconnectAll)
          {
              result = dccInstance->requestDeviceDisconnection(disconnectedReason, deleteDevice);

              if (CC_ERR_INT_NO_ERROR != result)
              {
                  ETG_TRACE_ERR(("disconnectProtocols(dccInstance = 0x%p): DCC instance could not handle the device disconnection request (error = %d)",
                  (void *) dccInstance, ETG_CENUM(CcErrorInternal, result)));
                  bmResult = BM_RESULT_ERR_GENERAL;
              }
          }
          else
          {
              result = dccInstance->requestProtocolDisconnections(protocolList, disconnectedReason);

              if (CC_ERR_INT_NO_ERROR == result)
              {
                  ETG_TRACE_USR1(("disconnectProtocols(dccInstance = 0x%p): protocol disconnection request(s) has(have) been accepted by DCC instance",
                  (void *) dccInstance));
              }
              else
              {
                  if (CC_ERR_INT_PCC_NONE_ASSIGNED == result)
                  {
                     ETG_TRACE_USR1(("disconnectProtocols(dccInstance = 0x%p): no PCC instances assigned for given protocols (error = %d)",
                       (void *) dccInstance, ETG_CENUM(CcErrorInternal, result)));
                     //bmResult = BM_RESULT_ERR_PROTOCOL_RESSOURCES_NOT_ASSIGNED;
                       bmResult = BM_RESULT_OK;
                  }
                  else
                  {
                      ETG_TRACE_ERR(("disconnectProtocols(dccInstance = 0x%p): DCC instance could not handle the protocol disconnection requests (error = %d)",
                      (void *) dccInstance, ETG_CENUM(CcErrorInternal, result)));
                      bmResult = BM_RESULT_ERR_GENERAL;
                  }
              }
          }
      }
      else
      {
         ETG_TRACE_ERR(("disconnectProtocols(dccInstance = 0): DCC instance pointer is 0 (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
         bmResult = BM_RESULT_ERR_DEVICE_RESSOURCES_NOT_ASSIGNED;
      }

      return bmResult;
   }

   BmResult BmController::disconnectProtocols(IN const DeviceId deviceId, IN const ProtocolList& protocolList,
         IN const bool disconnectAll, IN const DisconnectedReason disconnectedReason, IN const bool deleteDevice, IN const bool ignoreStopAutoconnection)
   {
      ENTRY

      ETG_TRACE_USR1(("disconnectProtocols: deviceId = %d, disconnectAll = %d, disconnectedReason = %d, deleteDevice = %d, ignoreStopAutoconection = %10s",
            deviceId, disconnectAll, ETG_CENUM(DisconnectedReason, disconnectedReason), deleteDevice, ignoreStopAutoconnection ? "true" : "false"));
      ETG_TRACE_USR1(("disconnectProtocols: protocolList:"));
      VARTRACE(protocolList);

      Result result(CC_ERR_INT_NO_ERROR);
      BmResult bmResult(BM_RESULT_OK);
      DeviceConnectionController* dccInstance(0);

      result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         bmResult = disconnectProtocols(dccInstance, protocolList, disconnectAll, disconnectedReason, deleteDevice, ignoreStopAutoconnection);
      }
      else
      {
         ETG_TRACE_USR1(("disconnectProtocols(deviceId = %d): no DCC instance is assigned to the given device ID", deviceId));
      }

      return bmResult;
   }

   bool BmController::linkQualityTimerCb(timer_t timerId , void *object, const void *userData)
   {
      ENTRY

      ETG_TRACE_USR1(("linkQualityTimerCb"));

      (void) timerId;
      (void) userData;

      static_cast<BmController*>(object)->requestLinkQualityOnExpiredTimer();

      return true;
   }

   DeviceConnectionController* BmController::getNewDeviceConnectionController(void)
   {
      return new DeviceConnectionController;
   }

   RequestData* BmController::getNewConnectionRequestData(IN const bool internal, IN const ConnectionRequestOrigin origin,
         IN const DeviceId deviceId, IN const ProtocolList& protocolList, IN const PageTimeout pageTimeout, IN const bool delayConnection)
   {
      RequestData* connectionRequestData = new ConnectionRequestData(internal, origin, deviceId, protocolList, pageTimeout, delayConnection);

      return connectionRequestData;
   }

   bool BmController::checkSupportedUUID(const DeviceId& deviceId, const ProtocolId& protocolId, const Uuid& uuid, const ConnectionRequestOrigin connectionRequestOrigin, bool isRequestInternal)
   {
      bool supportedUUID = false;
      Result result(CC_ERR_INT_NO_ERROR);

      ETG_TRACE_USR1(("checkSupportedUUID(): given protocol = (%d, \"%50s\") ConnectionRequestOrigin = %d isRequestInternal = %10s" ,
            ETG_CENUM(ProtocolId, protocolId), uuid.c_str(), ETG_CENUM(ConnectionRequestOrigin, connectionRequestOrigin), isRequestInternal ? "true" : "false"));

      // given slave protocol is considered only BtLimitationmode is prepared for Carplaywireless
      if(BM_PROTOCOL_ID_SPP == protocolId)
      {
         if(0 == (strcmp((const char *)IAP2BT_SPP_UUID, uuid.c_str())))
         {
            if((true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported) &&
                  (BM_CONNECTION_REQUEST_ORIGIN_REMOTE == connectionRequestOrigin))
            {
               supportedUUID = true;
               ETG_TRACE_USR4(("checkSupportedUUID(): Remote request for IAP2BT_SPP_UUID"));
            }
            else if((true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported) &&
                  (true == isRequestInternal))
            {
               BdAddress bdAddress = "";
               result = LocalSpm::getDbManager().getBdAddress(bdAddress, deviceId);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  BtLimitationModeInfo btLimitationModeInfo;
                  result = LocalSpm::getBmCoreMainController().getBtLimitationModeInfo(btLimitationModeInfo, bdAddress);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     if((BM_LIMITATION_FEATURE_CAR_PLAY == btLimitationModeInfo._limitationMode._limitationFeature) &&
                           ((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == btLimitationModeInfo._limitationMode._limitationCommunicationIf) ||
                                 (BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == btLimitationModeInfo._limitationMode._limitationCommunicationIf)))
                     {
                        supportedUUID = true;
                     }
                  }
                  else
                  {
                     ETG_TRACE_USR1(("CheckSupportedUUID: btLimitationMode CPW - BdAddress not matched with requested deviceID-BdAdress"));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("CheckSupportedUUID: getting bdAddress from DB failed (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               ETG_TRACE_ERR(("CheckSupportedUUID: CarPlay Wireless support is configured to be disabled, or connection is not of remote origin or is not an Internal request "));
            }
         }
         else if(0 == (strcmp((const char *)AAW_SPP_UUID, uuid.c_str())))
         {
            if((true == LocalSpm::getDataProvider().getBmCoreConfiguration()._androidAutoWirelessSupported) &&
                  (BM_CONNECTION_REQUEST_ORIGIN_REMOTE == connectionRequestOrigin))
            {
               supportedUUID = true;

               ETG_TRACE_USR4(("checkSupportedUUID(): Remote request for AAW_SPP_UUID"));
            }
            else if((true == LocalSpm::getDataProvider().getBmCoreConfiguration()._androidAutoWirelessSupported) &&
                  (true == isRequestInternal))
            {
               BdAddress bdAddress = "";
               result = LocalSpm::getDbManager().getBdAddress(bdAddress, deviceId);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  BtLimitationModeInfo btLimitationModeInfo;
                  result = LocalSpm::getBmCoreMainController().getBtLimitationModeInfo(btLimitationModeInfo, bdAddress);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     if((BM_LIMITATION_FEATURE_ANDROID_AUTO == btLimitationModeInfo._limitationMode._limitationFeature) &&
                           ((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == btLimitationModeInfo._limitationMode._limitationCommunicationIf) ||
                                 (BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == btLimitationModeInfo._limitationMode._limitationCommunicationIf)))
                     {
                        supportedUUID = true;
                     }
                  }
                  else
                  {
                     ETG_TRACE_USR4(("CheckSupportedUUID: btLimitationMode AAW - BdAddress not matched with requested deviceID-BdAdress"));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("CheckSupportedUUID: getting bdAddress from DB failed (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               ETG_TRACE_ERR(("CheckSupportedUUID: Android Auto Wireless support is configured to be disabled, or connection is not of remote origin or is not an Internal request "));
            }
         }
         else
         {
            ETG_TRACE_USR1(("checkSupportedUUID: given UUID is not matched with AAW/CPW SPP UUID"));
         }
      }
      else
      {
         ETG_TRACE_USR1(("checkSupportedUUID: given Protocol is not SPP"));
      }

      return supportedUUID;
   }
}
