//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FEATSTD_UTIL_VARIANT_TYPE_CONVERTER_H)
#define FEATSTD_UTIL_VARIANT_TYPE_CONVERTER_H

#include <FeatStd/Base.h>
#include <FeatStd/Util/FeatLimits.h>
#include <FeatStd/Util/Traits.h>
#include <FeatStd/Util/String.h>
#include <FeatStd/Util/Variant.h>
#include <FeatStd/Platform/String.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif
#include <FeatStd/Diagnostics/Log.h>

namespace FeatStd {
namespace Internal {

    template <SizeType v1, SizeType v2, bool isSourceTypeSigned, bool isTargetTypeSigned>
    struct IsDownCast2 {
        enum {
            Value = (v1 > v2) || ((v1 == v2) && ((0 == isSourceTypeSigned) && (0 != isTargetTypeSigned)))
        };
    };

    template <typename SourceType, typename TargetType>
    struct IsDownCast {
        enum {
            Value = IsDownCast2<sizeof(SourceType), sizeof(TargetType), IsSigned<SourceType>::Value, IsSigned<TargetType>::Value>::Value
        };
    };

    template<typename T>
    struct StringToAnyStaticCastStringScanfFormatTrait;

    template<>
    struct StringToAnyStaticCastStringScanfFormatTrait<FeatStd::Int64>
    {
        static const Char* StringScanfFormat()
        {
            return "%lld";
        }
    };

    template<>
    struct StringToAnyStaticCastStringScanfFormatTrait<FeatStd::UInt64>
    {
        static const Char* StringScanfFormat()
        {
            return "%llu";
        }
    };

    template<>
    struct StringToAnyStaticCastStringScanfFormatTrait<FeatStd::Float>
    {
        static const Char* StringScanfFormat()
        {
            return "%f";
        }
    };

    template<>
    struct StringToAnyStaticCastStringScanfFormatTrait<FeatStd::Double>
    {
        static const Char* StringScanfFormat()
        {
            return "%lf";
        }
    };

    template<typename T, bool isSigned>
    struct StringToAnyStaticCastStringIntegerScanfTypeTrait
    {
        typedef FeatStd::Int64 StringScanfType;
    };

    template<typename T>
    struct StringToAnyStaticCastStringIntegerScanfTypeTrait<T, false>
    {
        typedef FeatStd::UInt64 StringScanfType;
    };

    template<typename T, bool isFloatingPoint>
    struct StringToAnyStaticCastStringScanfTypeTrait
    {
        typedef T StringScanfType;
    };

    template<typename T>
    struct StringToAnyStaticCastStringScanfTypeTrait<T, false>
    {
        typedef typename StringToAnyStaticCastStringIntegerScanfTypeTrait<T, FeatStd::Internal::IsSigned<T>::Value>::StringScanfType StringScanfType;
    };

    template<typename T>
    struct StringToAnyStaticCastTrait
    {
        typedef typename StringToAnyStaticCastStringScanfTypeTrait<T, FeatStd::Internal::IsFloatingPoint<T>::Value>::StringScanfType StringScanfType;
    };

    template <typename T>
    struct StringToAnyStaticCast
    {
        static bool TryConvert(const FeatStd::String& sourceValue, T& value)
        {
            typename StringToAnyStaticCastTrait<T>::StringScanfType stringScanfValue = typename StringToAnyStaticCastTrait<T>::StringScanfType();
            const Char* format = StringToAnyStaticCastStringScanfFormatTrait<typename StringToAnyStaticCastTrait<T>::StringScanfType>::StringScanfFormat();
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(sourceValue.GetCriticalSection());
#endif
            bool ok = Internal::String::StringScanf(sourceValue.GetCString(), format, &stringScanfValue) > 0;
            value = static_cast<T>(stringScanfValue);
            return ok;
        }
    };

    template <>
    struct StringToAnyStaticCast<bool>
    {
        static bool TryConvert(const FeatStd::String& sourceValue, bool& value)
        {
            Int stringScanfValue = 0;
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(sourceValue.GetCriticalSection());
#endif
            bool ok = Internal::String::StringScanf(sourceValue.GetCString(), "%d", &stringScanfValue) > 0;
            value = (stringScanfValue != 0);
            return ok;
        }
    };

    template <typename T>
    struct AnyToStringStaticCast {
        static bool TryConvert(const T& sourceValue, FeatStd::String& value)
        {
            value = Variant(sourceValue).GetString();
            return true;
        }
    };

    template <typename SourceType, typename TargetType, bool isSourceTypeSigned, bool isTargetTypeSigned> struct StaticCast;

    template <typename SourceType, typename TargetType>
    struct SignedIntegerToSignedIntegerStaticDownCast
    {
        static bool TryConvert(const SourceType& sourceValue, TargetType& value)
        {
            if (sourceValue > FeatStd::Internal::Limits<TargetType>::Max()) {
                value = FeatStd::Internal::Limits<TargetType>::Max();
                return false;
            }
            if (sourceValue < FeatStd::Internal::Limits<TargetType>::Min()) {
                value = FeatStd::Internal::Limits<TargetType>::Min();
                return false;
            }
            value = static_cast<TargetType>(sourceValue);
            return true;
        }
    };

    template <typename SourceType, typename TargetType>
    struct SignedIntegerToSignedIntegerStaticUpCast
    {
        static bool TryConvert(const SourceType& sourceValue, TargetType& value)
        {
            value = static_cast<TargetType>(sourceValue);
            return true;
        }
    };

    template <typename SourceType, typename TargetType>
    struct UnsignedIntegerToUnsignedIntegerStaticDownCast
    {
        static bool TryConvert(const SourceType& sourceValue, TargetType& value)
        {
            if (sourceValue > FeatStd::Internal::Limits<TargetType>::Max()) {
                value = FeatStd::Internal::Limits<TargetType>::Max();
                return false;
            }
            value = static_cast<TargetType>(sourceValue);
            return true;
        }
    };

    template <typename SourceType, typename TargetType>
    struct UnsignedIntegerToUnsignedIntegerStaticUpCast
    {
        static bool TryConvert(const SourceType& sourceValue, TargetType& value)
        {
            value = static_cast<TargetType>(sourceValue);
            return true;
        }
    };

    template <typename SourceType, typename TargetType>
    struct SignedIntegerToUnsignedIntegerStaticDownCast
    {
        static bool TryConvert(const SourceType& sourceValue, TargetType& value)
        {
            if (sourceValue > FeatStd::Internal::Limits<TargetType>::Max()) {
                value = FeatStd::Internal::Limits<TargetType>::Max();
                return false;
            }
            if (sourceValue < SourceType()) {
                value = TargetType();
                return false;
            }
            value = static_cast<TargetType>(sourceValue);
            return true;
        }
    };

    template <typename SourceType, typename TargetType>
    struct SignedIntegerToUnsignedIntegerStaticUpCast
    {
        static bool TryConvert(const SourceType& sourceValue, TargetType& value)
        {
            if (sourceValue < SourceType()) {
                value = TargetType();
                return false;
            }
            value = static_cast<TargetType>(sourceValue);
            return true;
        }
    };

    template <typename SourceType, typename TargetType>
    struct UnsignedIntegerToSignedIntegerStaticDownCast
    {
        static bool TryConvert(const SourceType& sourceValue, TargetType& value)
        {
            if (sourceValue > static_cast<SourceType>(FeatStd::Internal::Limits<TargetType>::Max())) {
                value = FeatStd::Internal::Limits<TargetType>::Max();
                return false;
            }
            value = static_cast<TargetType>(sourceValue);
            return true;
        }
    };

    template <typename SourceType, typename TargetType>
    struct UnsignedIntegerToSignedIntegerStaticUpCast
    {
        static bool TryConvert(const SourceType& sourceValue, TargetType& value)
        {
            value = static_cast<TargetType>(sourceValue);
            return true;
        }
    };

    template <typename SourceType, typename TargetType>
    struct FloatToFloatStaticCast {
        static bool TryConvert(const SourceType& sourceValue, TargetType& value)
        {
            value = static_cast<TargetType>(sourceValue);
            return true;
        }
    };

    template <typename SourceType, typename TargetType>
    struct FloatToUnsignedIntegerStaticCast {
        static bool TryConvert(const SourceType& sourceValue, TargetType& value)
        {
            if (sourceValue < SourceType()) {
                value = TargetType();
                return false;
            }
            if (sourceValue > static_cast<SourceType>(FeatStd::Internal::Limits<TargetType>::Max())) {
                value = FeatStd::Internal::Limits<TargetType>::Max();
                return false;
            }
            if (sourceValue >= static_cast<SourceType>(FeatStd::Internal::Limits<TargetType>::Max())) {
                // static_cast seems to have an issue with max value
                // hence, the explicit check
                value = FeatStd::Internal::Limits<TargetType>::Max();
                return true;
            }
            value = static_cast<TargetType>(sourceValue);
            return true;
        }
    };

    template <typename SourceType, typename TargetType>
    struct FloatToSignedIntegerStaticCast {
        static bool TryConvert(const SourceType& sourceValue, TargetType& value)
        {
            if (sourceValue <= static_cast<SourceType>(FeatStd::Internal::Limits<TargetType>::Max())) {
                if (sourceValue >= static_cast<SourceType>(FeatStd::Internal::Limits<TargetType>::Max())) {
                    // static_cast seems to have an issue with max value
                    // hence, the explicit check
                    value = FeatStd::Internal::Limits<TargetType>::Max();
                    return true;
                }
                if (sourceValue > static_cast<SourceType>(FeatStd::Internal::Limits<TargetType>::Min())) {
                    if (sourceValue <= static_cast<SourceType>(FeatStd::Internal::Limits<TargetType>::Min())) {
                        // static_cast seems to have an issue with min value
                        // hence, the explicit check
                        value = FeatStd::Internal::Limits<TargetType>::Min();
                        return true;
                    }
                    value = static_cast<TargetType>(sourceValue);
                    return true;
                }
                value = FeatStd::Internal::Limits<TargetType>::Min();
            }
            else {
                value = FeatStd::Internal::Limits<TargetType>::Max();
            }
            return false;
        }
    };

    template <typename SourceType, typename TargetType>
    struct IntegerToFloatStaticCast {
        static bool TryConvert(const SourceType& sourceValue, TargetType& value)
        {
            value = static_cast<TargetType>(sourceValue);
            return true;
        }
    };

    template <typename SourceType>
    struct AnyToBoolStaticCast
    {
        static bool TryConvert(const SourceType& sourceValue, bool& value)
        {
            value = (sourceValue != SourceType());
            return true;
        }
    };


    template <typename TargetType>
    struct BoolToAnyStaticCast
    {
        static bool TryConvert(const bool& sourceValue, TargetType& value)
        {
            value = (sourceValue) ? static_cast<TargetType>(1) : TargetType();
            return true;
        }
    };

    template <typename T>
    struct SameTypeStaticCast {
        static bool TryConvert(const T& sourceValue, T& value)
        {
            value = sourceValue;
            return true;
        }
    };

    template <typename SourceType, typename TargetType, bool isSourceTypeSigned, bool isTargetTypeSigned, bool isDownCast>
    struct IntegerToIntegerStaticCastSelector;

    template <typename SourceType, typename TargetType>
    struct IntegerToIntegerStaticCastSelector<SourceType, TargetType, false, false, false> {
        typedef UnsignedIntegerToUnsignedIntegerStaticUpCast<SourceType, TargetType> StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct IntegerToIntegerStaticCastSelector<SourceType, TargetType, false, false, true> {
        typedef UnsignedIntegerToUnsignedIntegerStaticDownCast<SourceType, TargetType> StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct IntegerToIntegerStaticCastSelector<SourceType, TargetType, false, true, false> {
        typedef UnsignedIntegerToSignedIntegerStaticUpCast<SourceType, TargetType> StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct IntegerToIntegerStaticCastSelector<SourceType, TargetType, false, true, true> {
        typedef UnsignedIntegerToSignedIntegerStaticDownCast<SourceType, TargetType> StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct IntegerToIntegerStaticCastSelector<SourceType, TargetType, true, false, false> {
        typedef SignedIntegerToUnsignedIntegerStaticUpCast<SourceType, TargetType> StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct IntegerToIntegerStaticCastSelector<SourceType, TargetType, true, false, true> {
        typedef SignedIntegerToUnsignedIntegerStaticDownCast<SourceType, TargetType> StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct IntegerToIntegerStaticCastSelector<SourceType, TargetType, true, true, false> {
        typedef SignedIntegerToSignedIntegerStaticUpCast<SourceType, TargetType> StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct IntegerToIntegerStaticCastSelector<SourceType, TargetType, true, true, true> {
        typedef SignedIntegerToSignedIntegerStaticDownCast<SourceType, TargetType> StaticCast;
    };

    template <typename SourceType, typename TargetType, bool isTargetTypeSigned>
    struct FloatToIntegerStaticCastSelector;

    template <typename SourceType, typename TargetType>
    struct FloatToIntegerStaticCastSelector<SourceType, TargetType, true>
    {
        typedef FloatToSignedIntegerStaticCast<SourceType, TargetType> StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct FloatToIntegerStaticCastSelector<SourceType, TargetType, false>
    {
        typedef FloatToUnsignedIntegerStaticCast<SourceType, TargetType> StaticCast;
    };

    template <typename SourceType, typename TargetType, bool isSourceTypeFloat, bool isTargetTypeFloat>
    struct StaticCastSelector;

    template <typename SourceType, typename TargetType>
    struct StaticCastSelector<SourceType, TargetType, false, false> {
        typedef typename IntegerToIntegerStaticCastSelector<SourceType, TargetType, FeatStd::Internal::IsSigned<SourceType>::Value, FeatStd::Internal::IsSigned<TargetType>::Value, IsDownCast<SourceType, TargetType>::Value>::StaticCast StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct StaticCastSelector<SourceType, TargetType, false, true> {
        typedef IntegerToFloatStaticCast<SourceType, TargetType> StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct StaticCastSelector<SourceType, TargetType, true, false> {
        typedef typename FloatToIntegerStaticCastSelector<SourceType, TargetType, FeatStd::Internal::IsSigned<TargetType>::Value>::StaticCast StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct StaticCastSelector<SourceType, TargetType, true, true> {
        typedef FloatToFloatStaticCast<SourceType, TargetType> StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct StringStaticCastSelector;

    template <typename TargetType>
    struct StringStaticCastSelector<FeatStd::String, TargetType>
    {
        typedef StringToAnyStaticCast<TargetType> StaticCast;
    };

    template <typename SourceType>
    struct StringStaticCastSelector<SourceType, FeatStd::String>
    {
        typedef AnyToStringStaticCast<SourceType> StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct StringStaticCastSelector {
        typedef typename StaticCastSelector<SourceType, TargetType, FeatStd::Internal::IsFloatingPoint<SourceType>::Value, FeatStd::Internal::IsFloatingPoint<TargetType>::Value>::StaticCast StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct BoolStaticCastSelector;

    template <typename SourceType>
    struct BoolStaticCastSelector<SourceType, bool>
    {
        typedef AnyToBoolStaticCast<SourceType> StaticCast;
    };

    template <typename TargetType>
    struct BoolStaticCastSelector<bool, TargetType>
    {
        typedef BoolToAnyStaticCast<TargetType> StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct BoolStaticCastSelector {
        typedef typename StringStaticCastSelector<SourceType, TargetType>::StaticCast StaticCast;
    };

    template <typename SourceType, typename TargetType>
    struct VariantTypeConverterSelector
    {
        typedef typename BoolStaticCastSelector<SourceType, TargetType>::StaticCast StaticCast;
    };

    template <typename T>
    struct VariantTypeConverterSelector<T, T> {
        typedef SameTypeStaticCast<T> StaticCast;
    };

    template<typename T>
    struct VariantValueConverter
    {
        static T GetValue(const Variant& variant)
        {
            T value = T();
            static_cast<void>(TryGetValue(variant, value));
            return value;
        }

        static bool TryGetValue(const Variant& variant, T& value)
        {
            FEATSTD_LOG_SET_REALM(FeatStd::Diagnostics::LogRealm::FeatStdSystem);
            switch (variant.m_type)
            {
            case VariantValueType::BoolValue:
                return Internal::VariantTypeConverterSelector<bool, T>::StaticCast::TryConvert(variant.m_value.m_boolValue, value);
            case VariantValueType::FloatValue:
                return Internal::VariantTypeConverterSelector<Float, T>::StaticCast::TryConvert(variant.m_value.m_floatValue, value);
            case VariantValueType::DoubleValue:
                return Internal::VariantTypeConverterSelector<Double, T>::StaticCast::TryConvert(variant.m_value.m_doubleValue, value);
            case VariantValueType::Int8Value:
                return Internal::VariantTypeConverterSelector<Int8, T>::StaticCast::TryConvert(variant.m_value.m_int8Value, value);
            case VariantValueType::Int16Value:
                return Internal::VariantTypeConverterSelector<Int16, T>::StaticCast::TryConvert(variant.m_value.m_int16Value, value);
            case VariantValueType::Int32Value:
                return Internal::VariantTypeConverterSelector<Int32, T>::StaticCast::TryConvert(variant.m_value.m_int32Value, value);
            case VariantValueType::Int64Value:
                return Internal::VariantTypeConverterSelector<Int64, T>::StaticCast::TryConvert(variant.m_value.m_int64Value, value);
            case VariantValueType::UInt8Value:
                return Internal::VariantTypeConverterSelector<UInt8, T>::StaticCast::TryConvert(variant.m_value.m_uint8Value, value);
            case VariantValueType::UInt16Value:
                return Internal::VariantTypeConverterSelector<UInt16, T>::StaticCast::TryConvert(variant.m_value.m_uint16Value, value);
            case VariantValueType::UInt32Value:
                return Internal::VariantTypeConverterSelector<UInt64, T>::StaticCast::TryConvert(variant.m_value.m_uint32Value, value);
            case VariantValueType::UInt64Value:
                return Internal::VariantTypeConverterSelector<UInt64, T>::StaticCast::TryConvert(variant.m_value.m_uint64Value, value);
            case VariantValueType::StringValue:
                if (!Internal::VariantTypeConverterSelector<FeatStd::String, T>::StaticCast::TryConvert(variant.GetString(), value)) {
#ifdef FEATSTD_LOG_ENABLED
#ifdef FEATSTD_THREADSAFETY_ENABLED
                    FeatStd::Internal::CriticalSectionLocker lock(variant.GetString().GetCriticalSection());
#endif
                    FEATSTD_LOG_ERROR("failed to convert string '%s' into a value!", variant.GetString().GetCString());
#endif
                    return false;
                };
                return true;
            default:
                break;
            }
            return false;
        }
    };
} // namespace Internal
} // namespace FeatStd {
#endif //FEATSTD_UTIL_VARIANT_TYPE_CONVERTER_H
