//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "VariantTypeConverter.h"
#include "Variant.h"
#include <FeatStd/Util/String.h>
#include <FeatStd/Util/StringData.h>
#include <FeatStd/Platform/String.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

#include <FeatStd/Diagnostics/Log.h>


namespace FeatStd {

    FEATSTD_LINT_SYMBOL(751, FeatStd::mLogRealm, "Preparation for Log output")
    FEATSTD_LOG_SET_REALM(FeatStd::Diagnostics::LogRealm::FeatStdSystem);

    extern bool IsInRom(const void*);

#define FEATSTD_VARIANT_INVOKER(Class) \
    struct FEATSTD_CONCAT2(Class, Invoker) { template<typename T> struct Invoker : Class<T> { }; }

    template<typename T>
    struct VariantModuloAssignmentOperator
    {
        static void Invoke(T& self, const T& other)
        {
            self %= other;
        }
    };

    template<>
    struct VariantModuloAssignmentOperator<bool>
    {
        static void Invoke(bool& self, const bool& other)
        {
            self = self && other;
        }
    };

    template<typename FloatType>
    struct VariantModuloAssignmentOperatorFloat
    {
        static void Invoke(FloatType& self, const FloatType& other)
        {
            self -= static_cast<FloatType>(static_cast<Int64>(self / other)) * other;
        }
    };

    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantModuloAssignmentOperator<float>, "it is referenced! => false warning from LINT")
    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantModuloAssignmentOperator<double>, "it is referenced! => false warning from LINT")
    template<> struct VariantModuloAssignmentOperator<FeatStd::Float> : public VariantModuloAssignmentOperatorFloat<FeatStd::Float>{};
    template<> struct VariantModuloAssignmentOperator<FeatStd::Double> : public VariantModuloAssignmentOperatorFloat<FeatStd::Double>{};

    template<typename T>
    struct VariantMultiplicationAssignmentOperator
    {
        static void Invoke(T& self, const T& other)
        {
            self *= other;
        }
    };

    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantMultiplicationAssignmentOperator<bool>, "it is referenced! => false warning from LINT")
    template<>
    struct VariantMultiplicationAssignmentOperator<bool>
    {
        static void Invoke(bool& self, const bool& other)
        {
            self = self || other;
        }
    };

    template<typename T>
    struct VariantDivisionAssignmentOperator
    {
        static void Invoke(T& self, const T& other)
        {
            if (other != T()) {
                self /= other;
            }
            else {
                FEATSTD_LOG_FATAL("division by zero!");
            }
        }
    };

    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantDivisionAssignmentOperator<bool>, "it is referenced! => false warning from LINT")
    template<> struct VariantDivisionAssignmentOperator<bool> : public VariantModuloAssignmentOperator<bool>{};

    template<typename T>
    struct VariantAdditionAssignmentOperator
    {
        static void Invoke(T& self, const T& other)
        {
            self += other;
        }
    };

    template<>
    struct VariantAdditionAssignmentOperator<bool>
    {
        static void Invoke(bool& self, const bool& other)
        {
            self = (self != other);
        }
    };

    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantSubtractionAssignmentOperator<bool>, "it is referenced! => false warning from LINT")
    template<typename T>
    struct VariantSubtractionAssignmentOperator
    {
        static void Invoke(T& self, const T& other)
        {
            self -= other;
        }
    };

    template<> struct VariantSubtractionAssignmentOperator<bool> : public VariantAdditionAssignmentOperator<bool>{};

    template <typename T>
    struct VariantEqualComperator
    {
        static bool Invoke(const T& self, const T& other) {
            return self == other;
        }
    };

    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantEqualComperator<bool>, "it is referenced! => false warning from LINT")
    template <>
    struct VariantEqualComperator<bool>
    {
        static bool Invoke(const bool& self, const FeatStd::Int32& other) {
            return self == (0 != other);
        }
    };

    template <typename T>
    struct VariantLessComperator
    {
        static bool Invoke(const T& self, const T& other) {
            return self < other;
        }
    };

    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantLessComperator<bool>, "it is referenced! => false warning from LINT")
    template <>
    struct VariantLessComperator<bool>
    {
        static bool Invoke(const bool& self, const FeatStd::Int32& other) {
            return (self ? 1 : 0) < other;
        }
    };

    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantLessComperator<FeatStd::String>, "it is referenced! => false warning from LINT")
    template <>
    struct VariantLessComperator<FeatStd::String>
    {
        static bool Invoke(const FeatStd::String& self, const FeatStd::String& other) {
            return self.Compare(other) < 0;
        }
    };

    template <typename T>
    struct VariantGreaterComperator
    {
        static bool Invoke(const T& self, const T& other)
        {
            return self > other;
        }
    };

    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantGreaterComperator<bool>, "it is referenced! => false warning from LINT")
    template <>
    struct VariantGreaterComperator<bool>
    {
        static bool Invoke(const bool& self, const FeatStd::Int32& other)
        {
            return (self ? 1 : 0) > other;
        }
    };

    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantGreaterComperator<FeatStd::String>, "it is referenced! => false warning from LINT")
    template <>
    struct VariantGreaterComperator<FeatStd::String>
    {
        static bool Invoke(const FeatStd::String& self, const FeatStd::String& other) {
            return self.Compare(other) > 0;
        }
    };

    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantModuloAssignmentOperatorInvoker::Invoker<<1>>, "it is referenced! => false warning from LINT")
    FEATSTD_VARIANT_INVOKER(VariantModuloAssignmentOperator);
    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantMultiplicationAssignmentOperatorInvoker::Invoker<<1>>, "it is referenced! => false warning from LINT")
    FEATSTD_VARIANT_INVOKER(VariantMultiplicationAssignmentOperator);
    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantDivisionAssignmentOperatorInvoker::Invoker<<1>>, "it is referenced! => false warning from LINT")
    FEATSTD_VARIANT_INVOKER(VariantDivisionAssignmentOperator);
    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantAdditionAssignmentOperatorInvoker::Invoker<<1>>, "it is referenced! => false warning from LINT")
    FEATSTD_VARIANT_INVOKER(VariantAdditionAssignmentOperator);
    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantSubtractionAssignmentOperatorInvoker::Invoker<<1>>, "it is referenced! => false warning from LINT")
    FEATSTD_VARIANT_INVOKER(VariantSubtractionAssignmentOperator);
    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantEqualComperatorInvoker::Invoker<<1>>, "it is referenced! => false warning from LINT")
    FEATSTD_VARIANT_INVOKER(VariantEqualComperator);
    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantLessComperatorInvoker::Invoker<<1>>, "it is referenced! => false warning from LINT")
    FEATSTD_VARIANT_INVOKER(VariantLessComperator);
    FEATSTD_LINT_SYMBOL(753, FeatStd::VariantGreaterComperatorInvoker::Invoker<<1>>, "it is referenced! => false warning from LINT")
    FEATSTD_VARIANT_INVOKER(VariantGreaterComperator);

    template <typename Operator, typename T>
    static inline void VariantValueOperatorInvoke(T& targetValue, const T& sourceValue)
    {
        typedef typename Operator::template Invoker<T> TypeInvoker;
        TypeInvoker::Invoke(targetValue, sourceValue);
    }

    template <typename Comperator, typename TargetType, typename SourceType>
    static inline bool VariantValueComperatorInvoke(const TargetType& targetValue, const SourceType& sourceValue)
    {
        typedef typename Comperator::template Invoker<TargetType> TypeInvoker;
        return TypeInvoker::Invoke(targetValue, sourceValue);
    }

    struct VariantValueOperator
    {
        template <typename Operator>
        static Variant& Invoke(Variant& self, const Variant& otherVariant)
        {
            switch (self.m_type) {
            case VariantValueType::BoolValue:
                VariantValueOperatorInvoke<Operator, bool>(self.m_value.m_boolValue, otherVariant.GetBool());
                break;
            case VariantValueType::FloatValue:
                VariantValueOperatorInvoke<Operator, FeatStd::Float>(self.m_value.m_floatValue, otherVariant.GetFloat());
                break;
            case VariantValueType::DoubleValue:
                VariantValueOperatorInvoke<Operator, FeatStd::Double>(self.m_value.m_doubleValue, otherVariant.GetDouble());
                break;
            case VariantValueType::Int8Value:
                VariantValueOperatorInvoke<Operator, FeatStd::Int8>(self.m_value.m_int8Value, otherVariant.GetInt8());
                break;
            case VariantValueType::Int16Value:
                VariantValueOperatorInvoke<Operator, FeatStd::Int16>(self.m_value.m_int16Value, otherVariant.GetInt16());
                break;
            case VariantValueType::Int32Value:
                VariantValueOperatorInvoke<Operator, FeatStd::Int32>(self.m_value.m_int32Value, otherVariant.GetInt32());
                break;
            case VariantValueType::Int64Value:
                VariantValueOperatorInvoke<Operator, FeatStd::Int64>(self.m_value.m_int64Value, otherVariant.GetInt64());
                break;
            case VariantValueType::UInt8Value:
                VariantValueOperatorInvoke<Operator, FeatStd::UInt8>(self.m_value.m_uint8Value, otherVariant.GetUInt8());
                break;
            case VariantValueType::UInt16Value:
                VariantValueOperatorInvoke<Operator, FeatStd::UInt16>(self.m_value.m_uint16Value, otherVariant.GetUInt16());
                break;
            case VariantValueType::UInt32Value:
                VariantValueOperatorInvoke<Operator, FeatStd::UInt32>(self.m_value.m_uint32Value, otherVariant.GetUInt32());
                break;
            case VariantValueType::UInt64Value:
                VariantValueOperatorInvoke<Operator, FeatStd::UInt64>(self.m_value.m_uint64Value, otherVariant.GetUInt64());
                break;
            case VariantValueType::StringValue:
                FEATSTD_LOG_ERROR("operators are not defined for string variants!");
                self.DetachCurrent();
                break;
            default:
                break;
            }
            return self;
        }
    };

    struct VariantValueComperator {
        template <typename Comperator>
        static bool Invoke(const Variant& self, const Variant& otherVariant)
        {
            switch (self.m_type) {
            case VariantValueType::BoolValue:
                return VariantValueComperatorInvoke<Comperator, bool, FeatStd::Int32>(self.m_value.m_boolValue, otherVariant.GetInt32());
            case VariantValueType::FloatValue:
                return VariantValueComperatorInvoke<Comperator, FeatStd::Float, FeatStd::Float>(self.m_value.m_floatValue, otherVariant.GetFloat());
            case VariantValueType::DoubleValue:
                return VariantValueComperatorInvoke<Comperator, FeatStd::Double, FeatStd::Double>(self.m_value.m_doubleValue, otherVariant.GetDouble());
            case VariantValueType::Int8Value:
                return VariantValueComperatorInvoke<Comperator, FeatStd::Int8, FeatStd::Int8>(self.m_value.m_int8Value, otherVariant.GetInt8());
            case VariantValueType::Int16Value:
                return VariantValueComperatorInvoke<Comperator, FeatStd::Int16, FeatStd::Int16>(self.m_value.m_int16Value, otherVariant.GetInt16());
            case VariantValueType::Int32Value:
                return VariantValueComperatorInvoke<Comperator, FeatStd::Int32, FeatStd::Int32>(self.m_value.m_int32Value, otherVariant.GetInt32());
            case VariantValueType::Int64Value:
                return VariantValueComperatorInvoke<Comperator, FeatStd::Int64, FeatStd::Int64>(self.m_value.m_int64Value, otherVariant.GetInt64());
            case VariantValueType::UInt8Value:
                return VariantValueComperatorInvoke<Comperator, FeatStd::UInt8, FeatStd::UInt8>(self.m_value.m_uint8Value, otherVariant.GetUInt8());
            case VariantValueType::UInt16Value:
                return VariantValueComperatorInvoke<Comperator, FeatStd::UInt16, FeatStd::UInt16>(self.m_value.m_uint16Value, otherVariant.GetUInt16());
            case VariantValueType::UInt32Value:
                return VariantValueComperatorInvoke<Comperator, FeatStd::UInt32, FeatStd::UInt32>(self.m_value.m_uint32Value, otherVariant.GetUInt32());
            case VariantValueType::UInt64Value:
                return VariantValueComperatorInvoke<Comperator, FeatStd::UInt64, FeatStd::UInt64>(self.m_value.m_uint64Value, otherVariant.GetUInt64());
            case VariantValueType::StringValue:
                return VariantValueComperatorInvoke<Comperator, FeatStd::String, FeatStd::String>(self.GetString(), otherVariant.GetString());
            default:
                break;
            }
            return false;
        }
    };

    Variant::Variant() :
        m_type(VariantValueType::FloatValue)
    {
        m_value.m_floatValue = 0.0F;
    }

    Variant::Variant(FeatStd::Float value) :
        m_type(VariantValueType::FloatValue)
    {
        m_value.m_floatValue = value;
    }

    Variant::Variant(FeatStd::Double value) :
        m_type(VariantValueType::DoubleValue)
    {
        m_value.m_doubleValue = value;
    }

    Variant::Variant(FeatStd::Int8 value) :
        m_type(VariantValueType::Int8Value)
    {
        m_value.m_int8Value = value;
    }

    Variant::Variant(FeatStd::Int16 value) :
        m_type(VariantValueType::Int16Value)
    {
        m_value.m_int16Value = value;
    }

    Variant::Variant(FeatStd::Int32 value) :
        m_type(VariantValueType::Int32Value)
    {
        m_value.m_int32Value = value;
    }

    Variant::Variant(FeatStd::Int64 value) :
        m_type(VariantValueType::Int64Value)
    {
        m_value.m_int64Value = value;
    }

    Variant::Variant(FeatStd::UInt8 value) :
        m_type(VariantValueType::UInt8Value)
    {
        m_value.m_uint8Value = value;
    }

    Variant::Variant(FeatStd::UInt16 value) :
        m_type(VariantValueType::UInt16Value)
    {
        m_value.m_uint16Value = value;
    }

    Variant::Variant(FeatStd::UInt32 value) :
        m_type(VariantValueType::UInt32Value)
    {
        m_value.m_uint32Value = value;
    }

    Variant::Variant(FeatStd::UInt64 value) :
        m_type(VariantValueType::UInt64Value)
    {
        m_value.m_uint64Value = value;
    }

    Variant::Variant(bool value) :
        m_type(VariantValueType::BoolValue)
    {
        m_value.m_boolValue = value;
    }

    Variant::Variant(const FeatStd::String& value) :
        m_type(VariantValueType::StringValue)
    {
        AssignStringFrom(value);
    }

    Variant::~Variant()
    {
        DetachCurrent();
    }

    void Variant::DetachCurrent()
    {
        if (m_type == VariantValueType::StringValue) {
            if (!IsInRom(m_value.m_stringValue)) {
                String::DetachFrom(m_value.m_stringValue, Internal::StringData::TextToStringData(m_value.m_stringValue));
            }
        }
    }

    void Variant::AssignFrom(const Variant& variant)
    {
        m_type = variant.GetType();
        m_value.m_int64Value = variant.m_value.m_int64Value;
        if (m_type == VariantValueType::StringValue) {
            AssignStringFrom(variant.m_value.m_stringValue);
        }
    }

    void Variant::AssignStringFrom(const String& string)
    {
        AssignStringFrom(string.m_text);
    }

    void Variant::AssignStringFrom(const TChar* string)
    {
        m_value.m_stringValue = Internal::StringData::GetEmptyText();
        if (IsInRom(string)) {
            m_value.m_stringValue = const_cast<TChar*>(string);
        }
        else {
            (void)String::AttachTo(m_value.m_stringValue, Internal::StringData::TextToStringData(const_cast<TChar*>(string)));
        }
    }

    Variant::Variant(const Variant& variant)
    {
        AssignFrom(variant);
    }

    Variant& Variant::operator=(const Variant& otherVariant)
    {
        if (&otherVariant != this) {
            DetachCurrent();
            AssignFrom(otherVariant);
        }
        return *this;
    }

    Variant& Variant::operator%=(const Variant& otherVariant)
    {
        return VariantValueOperator::Invoke<VariantModuloAssignmentOperatorInvoker>(*this, otherVariant);
    }

    Variant& Variant::operator*=(const Variant& otherVariant)
    {
        return VariantValueOperator::Invoke<VariantMultiplicationAssignmentOperatorInvoker>(*this, otherVariant);
    }

    Variant& Variant::operator/=(const Variant& otherVariant)
    {
        return VariantValueOperator::Invoke<VariantDivisionAssignmentOperatorInvoker>(*this, otherVariant);
    }

    Variant& Variant::operator+=(const Variant& otherVariant)
    {
        return VariantValueOperator::Invoke<VariantAdditionAssignmentOperatorInvoker>(*this, otherVariant);
    }

    Variant& Variant::operator-=(const Variant& otherVariant)
    {
        return VariantValueOperator::Invoke<VariantSubtractionAssignmentOperatorInvoker>(*this, otherVariant);
    }

    Variant Variant::operator%(const Variant& otherVariant) const
    {
        return Variant(*this) %= otherVariant;
    }

    Variant Variant::operator*(const Variant& otherVariant) const
    {
        return Variant(*this) *= otherVariant;
    }

    Variant Variant::operator/(const Variant& otherVariant) const
    {
        return Variant(*this) /= otherVariant;
    }

    Variant Variant::operator+(const Variant& otherVariant) const
    {
        return Variant(*this) += otherVariant;
    }

    Variant Variant::operator-(const Variant& otherVariant) const
    {
        return Variant(*this) -= otherVariant;
    }

    bool Variant::operator==(const Variant& otherVariant) const
    {
        return VariantValueComperator::Invoke<VariantEqualComperatorInvoker>(*this, otherVariant);
    }

    bool Variant::operator!=(const Variant& otherVariant) const
    {
        return !operator==(otherVariant);
    }

    bool Variant::operator<(const Variant& otherVariant) const
    {
        return VariantValueComperator::Invoke<VariantLessComperatorInvoker>(*this, otherVariant);
    }

    bool Variant::operator>(const Variant& otherVariant) const
    {
        return VariantValueComperator::Invoke<VariantGreaterComperatorInvoker>(*this, otherVariant);
    }

    bool Variant::operator<=(const Variant& otherVariant) const
    {
        return !operator>(otherVariant);
    }

    bool Variant::operator>=(const Variant& otherVariant) const
    {
        return !operator<(otherVariant);
    }

    VariantValueType::Enum Variant::GetType() const
    {
        return m_type;
    }

    FeatStd::Float Variant::GetFloat() const
    {
        return Internal::VariantValueConverter<FeatStd::Float>::GetValue(*this);
    }

    bool Variant::TryGetFloat(FeatStd::Float& value) const
    {
        return Internal::VariantValueConverter<FeatStd::Float>::TryGetValue(*this, value);
    }

    FeatStd::Double Variant::GetDouble() const
    {
        return Internal::VariantValueConverter<FeatStd::Double>::GetValue(*this);
    }

    bool Variant::TryGetDouble(FeatStd::Double& value) const
    {
        return Internal::VariantValueConverter<FeatStd::Double>::TryGetValue(*this, value);
    }

    FeatStd::Int8 Variant::GetInt8() const
    {
        return Internal::VariantValueConverter<FeatStd::Int8>::GetValue(*this);
    }

    bool Variant::TryGetInt8(FeatStd::Int8& value) const
    {
        return Internal::VariantValueConverter<FeatStd::Int8>::TryGetValue(*this, value);
    }

    FeatStd::Int16 Variant::GetInt16() const
    {
        return Internal::VariantValueConverter<FeatStd::Int16>::GetValue(*this);
    }

    bool Variant::TryGetInt16(FeatStd::Int16& value) const
    {
        return Internal::VariantValueConverter<FeatStd::Int16>::TryGetValue(*this, value);
    }

    FeatStd::Int32 Variant::GetInt32() const
    {
        return Internal::VariantValueConverter<FeatStd::Int32>::GetValue(*this);
    }

    bool Variant::TryGetInt32(FeatStd::Int32& value) const
    {
        return Internal::VariantValueConverter<FeatStd::Int32>::TryGetValue(*this, value);
    }

    FeatStd::Int64 Variant::GetInt64() const
    {
        return Internal::VariantValueConverter<FeatStd::Int64>::GetValue(*this);
    }

    bool Variant::TryGetInt64(FeatStd::Int64& value) const
    {
        return Internal::VariantValueConverter<FeatStd::Int64>::TryGetValue(*this, value);
    }

    FeatStd::UInt8 Variant::GetUInt8() const
    {
        return Internal::VariantValueConverter<FeatStd::UInt8>::GetValue(*this);
    }

    bool Variant::TryGetUInt8(FeatStd::UInt8& value) const
    {
        return Internal::VariantValueConverter<FeatStd::UInt8>::TryGetValue(*this, value);
    }

    FeatStd::UInt16 Variant::GetUInt16() const
    {
        return Internal::VariantValueConverter<FeatStd::UInt16>::GetValue(*this);
    }

    bool Variant::TryGetUInt16(FeatStd::UInt16& value) const
    {
        return Internal::VariantValueConverter<FeatStd::UInt16>::TryGetValue(*this, value);
    }

    FeatStd::UInt32 Variant::GetUInt32() const
    {
        return Internal::VariantValueConverter<FeatStd::UInt32>::GetValue(*this);
    }

    bool Variant::TryGetUInt32(FeatStd::UInt32& value) const
    {
        return Internal::VariantValueConverter<FeatStd::UInt32>::TryGetValue(*this, value);
    }

    FeatStd::UInt64 Variant::GetUInt64() const
    {
        return Internal::VariantValueConverter<FeatStd::UInt64>::GetValue(*this);
    }

    bool Variant::TryGetUInt64(FeatStd::UInt64& value) const
    {
        return Internal::VariantValueConverter<FeatStd::UInt64>::TryGetValue(*this, value);
    }

    bool Variant::GetBool() const
    {
        return Internal::VariantValueConverter<bool>::GetValue(*this);
    }

    bool Variant::TryGetBool(bool& value) const
    {
        return Internal::VariantValueConverter<bool>::TryGetValue(*this, value);
    }

    template <typename T, bool isSigned>
    struct VariantIntegerToStringGetDigitHelper;

    template <typename T>
    struct VariantIntegerToStringGetDigitHelper<T, true> {
        static TChar Get(T value)
        {
            T d = T(value % 10);
            return TChar((d < 0) ? -d : d) + '0';
        }

        static bool IsNegativ(T value)
        {
            return value < 0;
        }
    };

    template <typename T>
    struct VariantIntegerToStringGetDigitHelper<T, false> {
        static TChar Get(T value)
        {
            TChar d = TChar(value % 10);
            return d + '0';
        }

        static bool IsNegativ(T)
        {
            return false;
        }
    };

    namespace VariantInternal {
        template <typename T>
        static TChar VariantIntegerToStringGetDigit(T value)
        {
            return VariantIntegerToStringGetDigitHelper<T, FeatStd::Internal::IsSigned<T>::Value>::Get(value);
        }

        template <typename T>
        static TChar* VariantIntegerToString(TChar* a_buffer, SizeType a_size, T value)
        {
            TChar * buffer = a_buffer;
            SizeType size = a_size;
            if (size > 0) {
                buffer[--size] = TChar(0);
                if (size > 0) {
                    bool sign = VariantIntegerToStringGetDigitHelper<T, FeatStd::Internal::IsSigned<T>::Value>::IsNegativ(value);
                    do {
                        buffer[--size] = VariantIntegerToStringGetDigit(value);
                        value /= 10;
                    } while ((value != 0) && (size > 0));
                    if (sign) {
                        if (size > 0) {
                            buffer[--size] = '-';
                        }
                        else {
                            buffer[0] = '-';
                            return buffer;
                        }
                    }
                    if (value == 0) {
                        return buffer + size;
                    }
                }
            }
            return buffer;
        }
    }

    FeatStd::String Variant::GetString() const
    {
        TChar buffer[128];
        buffer[0] = TChar(0);
        FEATSTD_LINT_CURRENT_SCOPE(1960, "return in case instead of break");
        switch (m_type)
        {
        case VariantValueType::BoolValue:
            return FeatStd::String(VariantInternal::VariantIntegerToString(buffer, sizeof(buffer), m_value.m_boolValue ? 1 : 0));
        case VariantValueType::FloatValue:
            static_cast<void>(Internal::String::StringPrintf(buffer, sizeof(buffer), "%.30g", m_value.m_floatValue));
            break;
        case VariantValueType::DoubleValue:
            static_cast<void>(Internal::String::StringPrintf(buffer, sizeof(buffer), "%.30g", m_value.m_doubleValue));
            break;
        case VariantValueType::Int8Value:
            return FeatStd::String(VariantInternal::VariantIntegerToString(buffer, sizeof(buffer), m_value.m_int8Value));
        case VariantValueType::Int16Value:
            return FeatStd::String(VariantInternal::VariantIntegerToString(buffer, sizeof(buffer), m_value.m_int16Value));
        case VariantValueType::Int32Value:
            return FeatStd::String(VariantInternal::VariantIntegerToString(buffer, sizeof(buffer), m_value.m_int32Value));
        case VariantValueType::Int64Value:
            return FeatStd::String(VariantInternal::VariantIntegerToString(buffer, sizeof(buffer), m_value.m_int64Value));
        case VariantValueType::UInt8Value:
            return FeatStd::String(VariantInternal::VariantIntegerToString(buffer, sizeof(buffer), m_value.m_uint8Value));
        case VariantValueType::UInt16Value:
            return FeatStd::String(VariantInternal::VariantIntegerToString(buffer, sizeof(buffer), m_value.m_uint16Value));
        case VariantValueType::UInt32Value:
            return FeatStd::String(VariantInternal::VariantIntegerToString(buffer, sizeof(buffer), m_value.m_uint32Value));
        case VariantValueType::UInt64Value:
            return FeatStd::String(VariantInternal::VariantIntegerToString(buffer, sizeof(buffer), m_value.m_uint64Value));
        case VariantValueType::StringValue:
            if (IsInRom(m_value.m_stringValue)) {
                return String(m_value.m_stringValue);
            }
            else {
                return String(Internal::StringData::TextToStringData(m_value.m_stringValue));
            }
        default:
            // not defined
            break;
        }
        return String(buffer);
    }

    bool Variant::SetValue(const Variant& otherVariant)
    {
        if (*this == otherVariant) {
            return true;
        }
        FEATSTD_LINT_CURRENT_SCOPE(1960, "return in case instead of break");
        switch (m_type)
        {
        case VariantValueType::BoolValue:
            return otherVariant.TryGetBool(m_value.m_boolValue);
        case VariantValueType::FloatValue:
            return otherVariant.TryGetFloat(m_value.m_floatValue);
        case VariantValueType::DoubleValue:
            return otherVariant.TryGetDouble(m_value.m_doubleValue);
        case VariantValueType::Int8Value:
            return otherVariant.TryGetInt8(m_value.m_int8Value);
        case VariantValueType::Int16Value:
            return otherVariant.TryGetInt16(m_value.m_int16Value);
        case VariantValueType::Int32Value:
            return otherVariant.TryGetInt32(m_value.m_int32Value);
        case VariantValueType::Int64Value:
            return otherVariant.TryGetInt64(m_value.m_int64Value);
        case VariantValueType::UInt8Value:
            return otherVariant.TryGetUInt8(m_value.m_uint8Value);
        case VariantValueType::UInt16Value:
            return otherVariant.TryGetUInt16(m_value.m_uint16Value);
        case VariantValueType::UInt32Value:
            return otherVariant.TryGetUInt32(m_value.m_uint32Value);
        case VariantValueType::UInt64Value:
            return otherVariant.TryGetUInt64(m_value.m_uint64Value);
        case VariantValueType::StringValue:
            DetachCurrent();
            AssignStringFrom(otherVariant.GetString());
            return true;
        default:
            break;
        }
        return false;
    }
}
