//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_StringEncoding_h)
#define FeatStd_Util_StringEncoding_h

#include <FeatStd/Base.h>

#include <FeatStd/Platform/String.h>

namespace FeatStd {
    namespace Internal {
        /// @addtogroup FEATSTD_UTILS
        /// @{
        /** cast given character to UInt32 without sign extension
            Defined Char to UInt32 conversion to avoid lint 571 suspicious cast warnings
            @param c the character to cast to UInt32
            @return c converted to UInt32 */
        inline UInt32 ToUInt32(Char c)
        {
            return UInt32(static_cast<UInt8>(c));
        }

        FEATSTD_LINT_SYMBOL(1790, FeatStd::Internal::TextEncodingBase, "TextEncodingBase has intentionally no virtual functions, no override intended")

        class TextEncodingBase {
            public:
                static bool EndOfString(const TChar* string)
                {
                    return *string == '\0';
                }
        };

        /** functions to parse and manipulate for ASCII (or more precise Extended ASCII) text */
        class AsciiEncoding : public TextEncodingBase {
            public:
                static UInt32 CodePointCount(const Char* string)
                {
                    return UInt32(FeatStd::Internal::String::Length(string));
                }

                static UInt32 CharCount(const Char* string)
                {
                    return UInt32(FeatStd::Internal::String::Length(string));
                }

                static UInt32 NCharCount(const Char *string, UInt32 size) 
                {
                    return UInt32(FeatStd::Internal::String::NLength(string, size));
                }

                static SizeType Copy(Char* dst, SizeType dstCharCount, const Char* src);
                static SizeType Copy(Char* dst, SizeType dstCharCount, const Char* src, SizeType codePointCount);

                static Int32 Compare(const Char* s1, const Char* s2)
                {
                    return FeatStd::Internal::String::CompareStrings(s1, s2);
                }

                static bool Match(const Char* s1, const Char* s2)
                {
                    return Compare(s1, s2) == 0;
                }

                static bool Match(const Char* s1, const Char* s2, UInt32 codePointCount)
                {
                    return FeatStd::Internal::String::ComparePartial(s1, s2, codePointCount) == 0;
                }

                static UInt32 CodeLength(const Char* codePoint)
                {
                    FEATSTD_UNUSED(codePoint);
                    return 1U;
                }

                static UInt32 NCodeLength(const Char *codePoint, UInt32 bufferSize)
                {
                    FEATSTD_UNUSED(codePoint);
                    return (bufferSize == 0U) ? 0U : 1U;
                }

                static UInt32 Advance(const Char*& p)
                {
                    if (*p == '\0') {
                        return 0;
                    }
                    ++p;
                    return 1U;
                }

                static UInt32 Ucs4(const Char* codePoint)
                {
                    return UInt32(UInt8(*codePoint));
                }

                static UInt32 NUcs4(const Char *codePoint, UInt32 codeLength)
                {
                    return (codeLength == 0) ? 0 : UInt32(UInt8(*codePoint));
                }

            private:
                FEATSTD_MAKE_CLASS_STATIC(AsciiEncoding);
        };

        /** functions to parse and manipulate for UTF-8 encoded text */
        class Utf8Encoding : public TextEncodingBase {
            public:
                static UInt32 CodePointCount(const Char* string);

                static UInt32 CharCount(const Char* string)
                {
                    return UInt32(FeatStd::Internal::String::Length(string));
                }
                static UInt32 NCharCount(const Char *string, UInt32 size) {
                    return UInt32(FeatStd::Internal::String::NLength(string, size));
                }

                static SizeType Copy(Char* dst, SizeType dstCharCount, const Char* src);
                static SizeType Copy(Char* dst, SizeType dstCharCount, const Char* src, SizeType codePointCount);

                static Int32 Compare(const Char* s1, const Char* s2)
                {
                    return FeatStd::Internal::String::CompareStrings(s1, s2);
                }

                static bool Match(const Char* s1, const Char* s2)
                {
                    return Compare(s1, s2) == 0;
                }

                static bool Match(const Char* s1, const Char* s2, UInt32 codePointCount);

                static UInt32 CodeLength(const Char* codePoint);
                static UInt32 NCodeLength(const Char *codePoint, UInt32 bufferSize);
                static UInt32 Advance(const Char*& codePoint);
                static UInt32 Ucs4(const Char* codePoint);
                static UInt32 NUcs4(const Char *codePoint, UInt32 codeLength);

            private:
                FEATSTD_MAKE_CLASS_STATIC(Utf8Encoding);
        };

        /** @brief Implements a code point iterator for UTF-8 encoded strings */
        template<typename Encoding> 
        class EncodedStringIterator {
            public:
                EncodedStringIterator(const TChar* string);

                /** Advances read position to next code point
                    It is assumed that the given string points to a valid code point and is overall a
                    valid UTF-8 encoded string. Invalid UTF-8 input data will yield undefined results.
                    @return number of bytes the read pointer has been moved */
                UInt32 Advance();

                /** returns the UCS4 (UTF-32) encoded code point at read position */
                UInt32 operator*() const;

                /** read position of the current code point */
                const Char* Pos() const;

                bool EndOfString() const;

            private:
                /** read position of the current code point */
                const Char* mString;
        };

        template<typename Encoding>
        EncodedStringIterator<Encoding>::EncodedStringIterator(const TChar* string)
        {
            mString = string;
        }

        template<typename Encoding>
        inline UInt32 EncodedStringIterator<Encoding>::Advance()
        {
            return Encoding::Advance(mString);
        }

        template<typename Encoding>
        inline UInt32 EncodedStringIterator<Encoding>::operator*() const
        {
            return Encoding::Ucs4(mString);
        }

        template<typename Encoding>
        inline const Char* EncodedStringIterator<Encoding>::Pos() const
        {
            return mString;
        }

        template<typename Encoding>
        bool EncodedStringIterator<Encoding>::EndOfString() const
        {
            return Encoding::EndOfString(mString);
        }


    /** @brief Implements a code point iterator for UTF-8 encoded strings */
    template<typename Encoding> class BoundEncodedStringIterator {
        public:
            BoundEncodedStringIterator(const TChar *string, Int32 length) :
                m_charPointSize(0),
                m_length(length),
                m_string(string)
            {
                if (length < 0) {
                    m_length = Encoding::CharCount(string);
                }
                Advance();
            }

            /** Advances read position to next code point
                It is assumed that the given string points to a valid code point and is overall a
                valid UTF-8 encoded string. Invalid UTF-8 input data will yield undefined results.
                @return number of Chars the read pointer has been moved */
            inline UInt32 Advance() {
                UInt32 oldCharPointSize = m_charPointSize;
                m_string += oldCharPointSize;
                m_length -= oldCharPointSize;
                m_charPointSize = Encoding::NCodeLength(m_string, m_length);
                return oldCharPointSize;
            }

            /** returns the UCS4 (UTF-32) encoded code point at read positon */
            inline UInt32 operator*() const {
                return Encoding::NUcs4(m_string, m_charPointSize);
            }

            /** read position of the current code point */
            const Char *Pos() const {
                return m_string;
            }

            bool EndOfString() const {
                return m_charPointSize == 0;
            }

            private:
                UInt32 m_charPointSize;
                UInt32 m_length;
                const Char *m_string;
    };

        typedef EncodedStringIterator<Utf8Encoding> Utf8TextIterator;
        typedef EncodedStringIterator<AsciiEncoding> AsciiTextIterator;

        typedef BoundEncodedStringIterator <Utf8Encoding> Utf8BoundTextIterator;
        typedef BoundEncodedStringIterator <AsciiEncoding> AsciiBoundTextIterator;

    

        /** FeatStd standard text encoding is UTF-8 */
        typedef Utf8Encoding TextEncoding;
        typedef EncodedStringIterator<Utf8Encoding> TextIterator;
        typedef BoundEncodedStringIterator<Utf8Encoding> BoundTextIterator;


        /// @}
    }
}
#endif
