//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FEATSTD_STRING_STRINGDATABASE_H)
#define FEATSTD_STRING_STRINGDATABASE_H

#include <FeatStd/Base.h>

#include <FeatStd/Platform/AtomicOp.h>

namespace FeatStd {
    class String;
}

FEATSTD_UNIT_TEST_TESTCASE_DECLARATION(String, String)
FEATSTD_UNIT_TEST_TESTCASE_DECLARATION(String, ConstructorTest)
FEATSTD_UNIT_TEST_TESTCASE_DECLARATION(StringData, StringDataBase)

namespace FeatStd { namespace Internal {
#ifdef FEATSTD_THREADSAFETY_ENABLED
    class CriticalSection;
#endif
    struct PreDefinedStringBuffer;
/// @addtogroup FEATSTD_UTILS
/// @{
    /**
     *  Base class for all string data objects which are attached to String.
     */
    class StringDataBase {
        public:

            /**
             *  Constructor
             */
            StringDataBase();

            /**
             * Destructor.
             */
            virtual ~StringDataBase() {}

            virtual void Destroy() = 0;
            /**
             *  Retrieves the character pointer of the string.
             *  @return     The pointer to the null-terminated character array.
             */
            virtual const TChar* GetCString() const = 0;

            /**
             *  Retrieves the unique identifier of the string.
             *  @return     The unique identifier of the string. If the string is not localized, 0 is returned.
             */
            virtual SizeType GetId() const { return 0; }

            /**
             *  @return     true, if the string has changed since the last call to GetCString().
             *              This only can happen if the string is localized. For static strings
             *              the return value is always false.
             */
            virtual bool HasChanged() const;

            /**
             * Determines, whether this string data object represents the empty string.
             *
             * @return      true, if empty, false otherwise.
             */
            virtual bool IsEmpty() const;

            /**
             *  Retrieves the TChar character count.
             *  @return Number of TChar instances in the string.
             */
            virtual UInt32 GetCharCount();

            /**
             *  Retrieves the number of code points.
             *  @return Number of code points in the string.
             */
            virtual UInt32 GetCodePointCount();

            /**
             *  Determines, whether this string data instance is referenced by multiple strings.
             *  @return true, if referenced by multiple strings, false otherwise.
             */
            bool HasMultipleRefs();

#ifdef FEATSTD_THREADSAFETY_ENABLED
            /**
             * @return the critical section that has to be used to protect the internal c-string from modification by other threads.
             */
            virtual FeatStd::Internal::CriticalSection* GetCriticalSection() const
            {
                return 0;
            }
#endif

        protected:
            void InvalidateCachedCharCount() const;
            void InvalidateCachedCodePointCount() const;

            void IncRefCount();
            void DecRefCount();
            virtual bool ShouldDecRefCount() const { return true; };

            TChar* GetText() { return reinterpret_cast<TChar*>(this + 1); }

        private:
            FeatStd::Internal::AtomicOp::Atomic m_refCounter;
            mutable UInt8 m_codePointCount;
            mutable UInt8 m_charCount;

            FEATSTD_UNIT_TEST_TESTCASE_FRIEND(String, String);
            FEATSTD_UNIT_TEST_TESTCASE_FRIEND(String, ConstructorTest);
            FEATSTD_UNIT_TEST_TESTCASE_FRIEND(StringData, StringDataBase);
            friend class FeatStd::String;
            friend struct PreDefinedStringBuffer;
    };
/// @}
}}
#endif  // FEATSTD_STRING_STRINGDATABASE_H
