//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "StringDataBase.h"

#include <FeatStd/Util/TextEncoding.h>

#include <FeatStd/MemoryManagement/Heap.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

namespace FeatStd { namespace Internal {

static const UInt8 cCharCountNotSet = 0;
static const UInt8 cInvalidCharCount = 0xFF;
static const UInt8 cCodePointCountNotSet = 0;
static const UInt8 cInvalidCodePointCount = 0xFF;

/******************************************************************************
 *  Constructor
 ******************************************************************************/
StringDataBase::StringDataBase() :
    m_codePointCount(cCodePointCountNotSet),
    m_charCount(cCharCountNotSet)
{
    *m_refCounter = 0;
}

/******************************************************************************
 *  HasChanged
 ******************************************************************************/
bool StringDataBase::HasChanged() const
{
    return false;
}

/******************************************************************************
 *  IsEmpty
 ******************************************************************************/
bool StringDataBase::IsEmpty() const
{
    return true;
}

/******************************************************************************
 *  IncRefCount
 ******************************************************************************/
void StringDataBase::IncRefCount()
{
    if (!ShouldDecRefCount()) {
        return;
    }

    (void) AtomicOp::Inc(m_refCounter);
}

/******************************************************************************
 *  DecRefCount
 ******************************************************************************/
void StringDataBase::DecRefCount()
{
    if (!ShouldDecRefCount()) {
        return;
    }

    if (AtomicOp::Dec(m_refCounter) == 0) {
        Destroy();
    }
}

/******************************************************************************
 *  GetCharCount
 ******************************************************************************/
UInt32 StringDataBase::GetCharCount()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(GetCriticalSection());
#endif
    const TChar* cString = GetCString();
    if ((m_charCount == cCharCountNotSet) && (cString != 0)) {
        UInt32 n = TextEncoding::CharCount(cString);
        m_charCount = (n >= cInvalidCharCount) ? cInvalidCharCount : UInt8(n);
    }

    return (m_charCount == cInvalidCharCount) ? TextEncoding::CharCount(cString) : m_charCount;
}

/******************************************************************************
 *  HasMultipleRefs
 ******************************************************************************/
bool StringDataBase::HasMultipleRefs() {
    return !AtomicOp::TestAndSet(m_refCounter, 1, 1);
}

/******************************************************************************
 *  GetCodePointCount
 ******************************************************************************/
UInt32 StringDataBase::GetCodePointCount()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(GetCriticalSection());
#endif
    const TChar* cString = GetCString();
    if ((m_codePointCount == cCodePointCountNotSet) && (cString != 0)) {
        UInt32 n = TextEncoding::CodePointCount(cString);
        m_codePointCount = (n >= cInvalidCodePointCount) ? cInvalidCodePointCount : UInt8(n);
    }

    return (m_codePointCount == cInvalidCodePointCount) ? TextEncoding::CodePointCount(cString) : m_codePointCount;
}

/******************************************************************************
 *  ResetCachedCharCount
 ******************************************************************************/
void StringDataBase::InvalidateCachedCharCount() const
{
    m_charCount = cCharCountNotSet;
}

/******************************************************************************
 *  ResetCachedCodePointCount
 ******************************************************************************/
void StringDataBase::InvalidateCachedCodePointCount() const
{
    m_codePointCount = cCodePointCountNotSet;
}
}}   // namespace FeatStd::Internal
