//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "StringData.h"

#include <FeatStd/Util/FeatLimits.h>

#include <FeatStd/MemoryManagement/Heap.h>
#include <FeatStd/Platform/Memory.h>

#include <FeatStd/Diagnostics/Log.h>
#include <FeatStd/Util/StaticObject.h>

namespace FeatStd { namespace Internal {

//FEATSTD_LOG_SET_REALM(FeatStd::Diagnostics::LogRealm::FeatStdSystem);

struct PreDefinedStringBuffer {
    StringData m_stringData;
    TChar m_char;

    PreDefinedStringBuffer() :
        m_char('\0')
    {
        *m_stringData.m_refCounter = 1;
    }
};

static PreDefinedStringBuffer& EmptyString()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(PreDefinedStringBuffer, s_EmptyString);
    return s_EmptyString;
}

static PreDefinedStringBuffer& s_EmptyText = EmptyString();


/******************************************************************************
 *  Constructors
 ******************************************************************************/

StringData::StringData()
{
}


StringData::~StringData()
{
}

/******************************************************************************
 *  GetCString
 ******************************************************************************/
const TChar* StringData::GetCString() const
{
    return const_cast<const TChar*>(const_cast<StringData*>(this)->GetText());
}

/******************************************************************************
 *  operator new
 ******************************************************************************/
StringData* StringData::Create(SizeType charCount, const TChar *text)
{
    FEATSTD_COMPILETIME_ASSERT((sizeof(StringData) % 4) == 0);

    if (charCount > (Limits<SizeType>::cMaxValue >> 3)) {
        return &(EmptyString().m_stringData);
    }

    SizeType n = sizeof(StringData) + ((charCount + 1) * sizeof(TChar)); // bytes
    n = (n + (n % sizeof(TChar))) / sizeof(TChar);             // TChars
    TChar* pInstance = FEATSTD_NEW_ARRAY(TChar, n);
    StringData *sd = MemoryManagement::ConstructObject<StringData>(pInstance);
    if ((sd != 0) && (text != 0)) {
        TChar* buffer = sd->GetText();
        Memory::Copy(buffer, text, charCount * sizeof(TChar));
        buffer[charCount] = TChar(0);
    }
    return sd;
}

void StringData::Destroy()
{
    MemoryManagement::DestructObject<StringData>(this);
    FEATSTD_DELETE_ARRAY(PointerToPointer<TChar*>(this));
}

bool StringData::IsEmpty() const
{
    return this == &(EmptyString().m_stringData);
}

bool StringData::ShouldDecRefCount() const
{
    return (this != &(EmptyString().m_stringData));
}

StringData* StringData::GetEmptyStringData() {
    return &(EmptyString().m_stringData);
}

TChar* StringData::GetEmptyText() {
    return &(EmptyString().m_char);
}

}

}   // namespace FeatStd::Internal
