//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FEATSTD_UTIL_STRINGBUFFER_H)
#define FEATSTD_UTIL_STRINGBUFFER_H

#include <FeatStd/Util/String.h>
#include <FeatStd/Util/FixedSizeString.h>
#include <FeatStd/Util/CharBuffer.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <FeatStd/Util/Variant.h>

namespace FeatStd {
    /// @addtogroup FEATSTD_UTILS
    /// @{

    /**
     * StringBuffer interface class is used to convert object into strings in a fast way.
     */
    class StringBuffer
    {
        public:
            /**
             * Append a string to the string buffer.
             * @param str The string that has to be appended.
             * @return Number of TChar instances that have been written. StringBuffer implementations are allowed to have a fixed size limit.
             */
            UInt32 Append(const TChar* str)
            {
                if (0 == str) {
                    return 0;
                }
                return Append(str, FeatStd::Internal::TextEncoding::CodePointCount(str));
            }

            /**
             * Append a sub string to the string buffer.
             * @param str The string that has to be appended.
             * @param codePointCount The number of code points that have to be appended.
             * @return Number of TChar instances that have been written. StringBuffer implementations are allowed to have a fixed size limit.
             */
            virtual UInt32 Append(const TChar* /*str*/, UInt32 /*codePointCount*/) { return 0; }

            /**
             * Append an object to the string buffer.
             * @param str The string that has to be appended.
             * @param codePointCount The number of code points that have to be appended.
             * @return Number of TChar instances that have been written. StringBuffer implementations are allowed to have a fixed size limit.
             */
#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
            template<typename T> UInt32 AppendObject(T const & object);
            UInt32 AppendPointer(void const * object);
#endif

        protected:
            /**
             * Protected default constructor. Only concrete implementation are allowed to be instantiated.
             */
            StringBuffer() { }
    };

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
    UInt32 inline StringBuffer::AppendPointer(const void* object)
    {
        FeatStd::Internal::CharBuffer<30> buffer;
        buffer.Format("0x%p", object);
        return Append(buffer.c_str());
    }
#endif

    template<UInt32 tcharCountMax> class FixedSizeStringBuffer : public StringBuffer
    {
        public:
            typedef FeatStd::Internal::FixedSizeString<tcharCountMax> StringType;
            /**
             * Create an FixedSizeStringBuffer that fills the FixedSizeString.
             * The FixedSizeStringBuffer will only append to the FixedSizeString.
             * Existing content has to be cleared before if the content has to be overwritten.
             * @param str The FixedSizeString that will be filled by the FixedSizeStringBuffer.
             */
            FixedSizeStringBuffer(StringType& str) :
                m_string(str)
            {
                m_tcharOffset = FeatStd::Internal::TextEncoding::CharCount(str.CStr());
            }

            /**
             * Append a sub string to the string buffer until the fixed size char buffer limit is reached.
             * @param str The string that has to be appended.
             * @param codePointCount The number of code points that have to be appended.
             * @return Number of TChar instances that have been written. StringBuffer implementations are allowed to have a fixed size limit.
             */
            virtual UInt32 Append(const TChar* str, UInt32 codePointCount);

        private:
            /**
             * Private default constructor. No default instances allowed.
             */
            FixedSizeStringBuffer();
            FixedSizeStringBuffer(const FixedSizeStringBuffer&);
            FixedSizeStringBuffer& operator=(const FixedSizeStringBuffer&);

            UInt32 m_tcharOffset;
            StringType& m_string;
    };

    template<UInt32 tcharCountMax> inline UInt32 FixedSizeStringBuffer<tcharCountMax>::Append(const TChar* str, UInt32 codePointCount)
    {
        if (0 == str) {
            return 0;
        }
        UInt32 tchars = FeatStd::Internal::TextEncoding::Copy(m_string.mString + m_tcharOffset, tcharCountMax - m_tcharOffset, str, codePointCount);
        m_tcharOffset += tchars;
        return tchars;
    }

    /**
     * Use this template to append an object to the string buffer.
     */
#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
    template<typename T> struct StringBufferAppender
    {
        /**
         * Use this method to append an object to the string buffer.
         * Implementations for the specific types have to be provided by more specific libraries and/or application code.
         * @param stringBuffer The StringBuffer to which the object has to be appended.
         * @param object The object that has to be appended.
         * @return Number of TChar instances that have been written. StringBuffer implementations are allowed to have a fixed size limit.
         */
        static UInt32 Append(StringBuffer& stringBuffer, T const & object);
    };

    template<UInt32 tcharCountMax> struct StringBufferAppender< FeatStd::Internal::FixedSizeString<tcharCountMax> >
    {
        /**
         * Use this method to append an object to the string buffer.
         * Implementations for the specific types have to be provided by more specific libraries and/or application code.
         * @param stringBuffer The StringBuffer to which the object has to be appended.
         * @param object The object that has to be appended.
         * @return Number of TChar instances that have been written. StringBuffer implementations are allowed to have a fixed size limit.
         */
        static UInt32 Append(StringBuffer& stringBuffer, FeatStd::Internal::FixedSizeString<tcharCountMax> const & object);
    };

    template<typename T> struct StringBufferAppender< FeatStd::MemoryManagement::SharedPointer<T> >
    {
        /**
         * Use this method to append an object to the string buffer.
         * Implementations for the specific types have to be provided by more specific libraries and/or application code.
         * @param stringBuffer The StringBuffer to which the object has to be appended.
         * @param object The object that has to be appended.
         * @return Number of TChar instances that have been written. StringBuffer implementations are allowed to have a fixed size limit.
         */
        static UInt32 Append(StringBuffer& stringBuffer, FeatStd::MemoryManagement::SharedPointer<T> const & object);
    };


    template<> inline UInt32 StringBufferAppender<bool>::Append(StringBuffer& stringBuffer, bool const & object) {
        return stringBuffer.Append((object) ? "true" : "false");
    }

    template<> inline UInt32 StringBufferAppender<TChar>::Append(StringBuffer& stringBuffer, TChar const & object)
    {
        return stringBuffer.Append(&object, 1);
    }

    template<> inline UInt32 StringBufferAppender<Int32>::Append(StringBuffer& stringBuffer, Int32 const & object)
    {
        FeatStd::Internal::CharBuffer<30> buffer;
        buffer.Format("%d", object);
        return stringBuffer.Append(buffer.c_str());
    }

    template<> inline UInt32 StringBufferAppender<UInt32>::Append(StringBuffer& stringBuffer, UInt32 const & object)
    {
        FeatStd::Internal::CharBuffer<30> buffer;
        buffer.Format("%u", object);
        return stringBuffer.Append(buffer.c_str());
    }

    template<> inline UInt32 StringBufferAppender<Int64>::Append(StringBuffer& stringBuffer, Int64 const & object)
    {
        FeatStd::Internal::CharBuffer<30> buffer;
        buffer.Format("%lld", object);
        return stringBuffer.Append(buffer.c_str());
    }

    template<> inline UInt32 StringBufferAppender<UInt64>::Append(StringBuffer& stringBuffer, UInt64 const & object)
    {
        FeatStd::Internal::CharBuffer<30> buffer;
        buffer.Format("%llu", object);
        return stringBuffer.Append(buffer.c_str());
    }

#ifdef FEATSTD_64BIT_PLATFORM
    template<> inline UInt32 StringBufferAppender<SizeType>::Append(StringBuffer& stringBuffer, SizeType const & object)
    {
        FeatStd::Internal::CharBuffer<30> buffer;
        buffer.Format("%u", object);
        return stringBuffer.Append(buffer.c_str());
    }
#endif

    template<> inline UInt32 StringBufferAppender<Int16>::Append(StringBuffer& stringBuffer, Int16 const & object)
    {
        return stringBuffer.AppendObject<Int32>(object);
    }

    template<> inline UInt32 StringBufferAppender<UInt16>::Append(StringBuffer& stringBuffer, UInt16 const & object)
    {
        return stringBuffer.AppendObject<UInt32>(object);
    }

    template<> inline UInt32 StringBufferAppender<Int8>::Append(StringBuffer& stringBuffer, Int8 const & object)
    {
        return stringBuffer.AppendObject<Int32>(object);
    }

    template<> inline UInt32 StringBufferAppender<UInt8>::Append(StringBuffer& stringBuffer, UInt8 const & object)
    {
        return stringBuffer.AppendObject<UInt32>(object);
    }

    template<> inline UInt32 StringBufferAppender<Double>::Append(StringBuffer& stringBuffer, Double const & object)
    {
        FeatStd::Internal::CharBuffer<30> buffer;
        buffer.Format("%f", object);
        return stringBuffer.Append(buffer.c_str());
    }

    template<> inline UInt32 StringBufferAppender<Float>::Append(StringBuffer& stringBuffer, Float const & object)
    {
        return stringBuffer.AppendObject<Double>(object);
    }

    template<> inline UInt32 StringBufferAppender<String>::Append(StringBuffer& stringBuffer, String const & object)
    {
        return stringBuffer.Append(object.GetCString());
    }

    template<> inline UInt32 StringBufferAppender<Variant>::Append(StringBuffer& stringBuffer, Variant const & object)
    {
        UInt32 bytes = stringBuffer.Append("FeatStd::Variant{ ");
        switch (object.GetType())
        {
        case VariantValueType::BoolValue:
            bytes += stringBuffer.Append(" bool, ");
            bytes += StringBufferAppender<bool>::Append(stringBuffer, object.GetBool());
            break;
        case VariantValueType::FloatValue:
            bytes += stringBuffer.Append(" Float, ");
            bytes += StringBufferAppender<Float>::Append(stringBuffer, object.GetFloat());
            break;
        case VariantValueType::Int32Value:
            bytes += stringBuffer.Append(" Int32, ");
            bytes += StringBufferAppender<Int32>::Append(stringBuffer, object.GetInt32());
            break;
        case VariantValueType::UInt32Value:
            bytes += stringBuffer.Append(" String, ");
            bytes += StringBufferAppender<UInt32>::Append(stringBuffer, object.GetUInt32());
            break;
        default:
            break;
        }
        bytes += stringBuffer.Append(" }");
        return bytes;
    }

    template<typename T> struct StringBufferAppender< T const * >
    {
        /**
         * Use this method to append an object to the string buffer.
         * Implementations for the specific types have to be provided by more specific libraries and/or application code.
         * @param stringBuffer The StringBuffer to which the object has to be appended.
         * @param object The object that has to be appended.
         * @return Number of TChar instances that have been written. StringBuffer implementations are allowed to have a fixed size limit.
         */
        static UInt32 Append(StringBuffer& stringBuffer, T const * const & object);
    };

    template<typename T> inline UInt32 StringBuffer::AppendObject(T const & object)
    {
        return StringBufferAppender<T>::Append(*this, object);
    }

    template<UInt32 tcharCountMax> inline UInt32 StringBufferAppender< FeatStd::Internal::FixedSizeString<tcharCountMax> >::Append(StringBuffer& stringBuffer, FeatStd::Internal::FixedSizeString<tcharCountMax> const & object)
    {
        return stringBuffer.Append(object.CStr());
    }

    template<typename T> inline UInt32 StringBufferAppender< FeatStd::MemoryManagement::SharedPointer<T> >::Append(StringBuffer& stringBuffer, FeatStd::MemoryManagement::SharedPointer<T> const & object)
    {
        UInt32 tcharCount = stringBuffer.AppendPointer(reinterpret_cast<const void*>(object.GetPointerToSharedInstance()));
        if (0 != object.GetPointerToSharedInstance()) {
            tcharCount += stringBuffer.Append(" { ");
            tcharCount += stringBuffer.AppendObject(*object);
            tcharCount += stringBuffer.Append(" }");
        }
        return tcharCount;
    }

    template<> inline UInt32 StringBufferAppender< TChar const * >::Append(StringBuffer& stringBuffer, TChar const * const & object)
    {
        UInt32 tcharCount = 0;
        if (0 == object) {
            tcharCount += stringBuffer.Append("0x00000000");
        }
        else {
            tcharCount += stringBuffer.Append("\"");
            tcharCount += stringBuffer.Append(object);
            tcharCount += stringBuffer.Append("\"");
        }
        return tcharCount;
    }

    template<typename T> inline UInt32 StringBufferAppender< T const * >::Append(StringBuffer& stringBuffer, T const * const & object)
    {
        UInt32 tcharCount = stringBuffer.AppendPointer(reinterpret_cast<void const *>(object));
        if (0 != object) {
            tcharCount += stringBuffer.Append(" { ");
            tcharCount += stringBuffer.AppendObject(*object);
            tcharCount += stringBuffer.Append(" }");
        }
        return tcharCount;
    }
#endif  // FEATSTD_STRINGBUFFER_APPENDER_ENABLED
    /// @}
}   // namespace FeatStd
#endif  // FEATSTD_UTIL_STRINGBUFFER_H
