//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FEATSTD_STRING_STRING_H)
#define FEATSTD_STRING_STRING_H

#include <FeatStd/Base.h>
#include <FeatStd/Util/TextId.h>

FEATSTD_UNIT_TEST_TESTCASE_DECLARATION(String, String)
FEATSTD_UNIT_TEST_TESTCASE_DECLARATION(String, ConstructorTest)
FEATSTD_UNIT_TEST_TESTCASE_DECLARATION(String, AssignmentOperator)
FEATSTD_UNIT_TEST_TESTCASE_DECLARATION(StringData, StringDataBase)

namespace FeatStd {
    namespace Internal {
        class StringData;
        class StringDataBase;
#ifdef FEATSTD_THREADSAFETY_ENABLED
        class CriticalSection;
#endif
    }

    /// @addtogroup FEATSTD_UTILS
    /// @{
    class CStringAccess;
    class Variant;

    /**
     * String class that will dynamically allocate string buffers. 
     * String buffers are reference counted.
     */
    class String {
        public:
            /**
             *  Default constructor.
             */
            String();

            /**
             *  Constructs a string object when having a character pointer.
             */
            String(const TChar*);

            /**
                Constructor.
                @param string The string.
                @param length The length of the string.
                @param isCodePoints Set to false if length is in bytes, instead of CodePoints(characters). Default is true.
             */
            String(const TChar *string, SizeType length, bool isCodePoints = true);
            
            /**
            *  Constructs a string object when having a hash value.
            *  @param hash     The hash value.
            */
            String(SizeType hash);

            /**
             *  Constructs a string object when having a string ID for a translatable text.
             *  @param id The id of the translatable text.
             */
            String(const TextId& id);

            /**
             *  Copy constructor.
             *  @param other The other string that is copied.
             */
            String(const String& other);

            /**
             *  Destructor.
             */
            ~String();

            String& operator=(const String& other);
            String& operator=(const TChar* other);

            bool operator==(const String& other) const;
            bool operator!=(const String& other) const;

            /**
            * Compares the first n characters of this string with the the first n characters of the other string.
            * return zero if the strings are identical, positive value if this string is greater, and negative value
            * if this string is less than the other string.
            * @param other The other string.
            * @return Zero if the strings are identical, positive value if this string is greater, and negative value
            *         if this string is less than the other string.
            */
            Int32 Compare(const String& other) const;

            /**
            * Compares this string with the other string.
            * return zero if the strings are identical, positive value if this string is greater, and negative value
            * if this string is less than the other string.
            * @param other The other string.
            * @param nChars Maximum number of characters to compare.
            * @return Zero if the strings are identical, positive value if this string is greater, and negative value
            *         if this string is less than the other string.
            */
            Int32 ComparePartial(const String& other, SizeType nChars) const;

            /**
            *  Retrieves CStringAccess object to facilitate accessing the char array in a safe way in multi-threaded environments.
            *  @note
            *  It is not recommended to use this function due to performance and memory penalty!
            *  @return A CStringAccess instance.
            */
            CStringAccess GetCStringAccess() const;

            /**
             *  Retrieves the character pointer of the string.
             *  @return The pointer to the null-terminated character array.
             */
            const TChar* GetCString() const;

            /**
                Retrieves the TChar character count.
                @return Number of TChar instances in the string.
             */
            SizeType GetCharCount() const;

            /**
                Retrieves the code point count.
                @return Number of code points in the string.
             */
            UInt32 GetCodePointCount() const;

            /**
             *  Retrieves the unique identifier of the string.
             *  @return Unique identifier of the string. If the string is not localized,
             *          0 is returned.
             */
            SizeType GetId() const;

            /**
             *  @return true, if the string has changed since the last call to GetCString().
             *          This only can happen if the string is localized. For static strings
             *          the return value is always false.
             */
            bool HasChanged() const;

            /**
             *  @return true, if the string is empty.
             */
            bool IsEmpty() const;

#ifdef FEATSTD_THREADSAFETY_ENABLED
            /**
            * @return the critical section that has to be used to protect the internal c-string from modification by other threads.
            */
            FeatStd::Internal::CriticalSection* GetCriticalSection() const;
#endif
    protected:
            explicit String(Internal::StringDataBase* strData);

        private:
            friend class Variant;
            TChar* m_text;

            static bool AttachTo(TChar*& text, Internal::StringDataBase* strData);
            static void DetachFrom(TChar*& text, Internal::StringDataBase* strData);

            bool AttachTo(Internal::StringDataBase* strData);
            void DetachFrom(Internal::StringDataBase* strData) const;
            void DetachCurrent() const;

            Internal::StringDataBase* GetData() const;
            void ClearData();
            Internal::StringData* AllocateStaticData(const TChar* string, SizeType stringCharCount) const;

            FEATSTD_UNIT_TEST_TESTCASE_FRIEND(String, String);
            FEATSTD_UNIT_TEST_TESTCASE_FRIEND(String, ConstructorTest);
            FEATSTD_UNIT_TEST_TESTCASE_FRIEND(String, AssignmentOperator);
            FEATSTD_UNIT_TEST_TESTCASE_FRIEND(StringData, StringDataBase);
    };
    /// @}
    /// @{
    /**
    * @brief The CStringAccess class is used for accessing the CString of a String.
    * The CString is copied safely and filled with a pattern ('@') before deletion in destructor.
    * The pattern shall facilitate debugging.
    * @note
    * It is not recommended to use this class due to performance and memory penalty!
    * @par
    * Usage:
    *     @code
    String myString = "Hello, World!";
    CStringAccess cstrAccess = myString.GetCStringAccess();

    const TChar* charString = cstrAccess.GetCString();
    *     @endcode
    * 
    * @note
    * The copy from the String instance is never updated! The CStringAccess instance needs to be retrieved again.
    * @par
    * A statement like the following will create a CStringAccess instance and immediately delete it after the statement.
    * The character pointer points to freed memory in this case!
    * @code
    String string1 = "Hello, World!";
    const TChar * charString = string1.GetCStringAccess().GetCString();
    // charString points to freed memory!
    // charString[0] equals '@', further access on charString will lead to memory violation
    * @endcode
    */
    class CStringAccess {
    public:

        ~CStringAccess();
        CStringAccess(const CStringAccess& cstr);
        CStringAccess& operator=(const CStringAccess& string);

        /**
        *  Retrieves the character pointer of the string.
        *  It is a copy of the string retrieved from String class instance.
        *  @return The pointer to the null-terminated character array.
        */
        inline const TChar* GetCString() { return m_copiedstring; }

    private:
        FEATSTD_MAKE_CLASS_STATIC(CStringAccess);
        friend class String;

        CStringAccess(const String& str);

        void InitString(const String& str);
        void ClearString();
        void CopyString(UInt32 length, const TChar* str);
        UInt32 m_length;
        TChar* m_copiedstring;
    };

    /// @}
}   // namespace FeatStd
#endif  // FEATSTD_STRING_STRING_H
