//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_StaticObject_h)
#define FeatStd_Util_StaticObject_h

#include <FeatStd/Base.h>
#include <FeatStd/Platform/CriticalSection.h>
#include <FeatStd/MemoryManagement/PlacementNew.h>
#include <FeatStd/Util/Traits.h>
#include <FeatStd/Util/Alignment.h>
#include <FeatStd/Util/ObjectBufferTypeProvider.h>
#include <FeatStd/Util/PodUtil.h>
#include <FeatStd/LifeTimeManager.h>

namespace FeatStd { namespace UnitTest {
    struct StaticObjectAccessor;
}}

namespace FeatStd { namespace Internal { namespace StaticObjectPrivate {
    // forward declarations
    template<typename T> class StaticObject;
    struct DtorListNode;    
}}}

    /// @addtogroup FEATSTD_UTILS
    /// @{

    // =====================================================================

#if defined(FEATSTD_DEBUG)
#   define FEATSTD_PRIVATE_STATIC_OBJECT_CONTEXT(var) FEATSTD__FILE__ "(" FEATSTD_STRINGIZE(__LINE__) "): " FEATSTD_STRINGIZE(var)
#else
#   define FEATSTD_PRIVATE_STATIC_OBJECT_CONTEXT(var) 
#endif

FEATSTD_LINT_MACRO(1960, FEATSTD_SYNCED_STATIC_OBJECT, "MISRA C++ 2008 Required Rule 16-3-1: false positive, none of order precedence conflicts apply")
FEATSTD_LINT_MACRO(1960, FEATSTD_UNSYNCED_STATIC_OBJECT, "MISRA C++ 2008 Required Rule 16-3-1: false positive, none of order precedence conflicts apply")

/** define a static object within function scope. 
    The object will be constructed thread safe and chained to the destruction list
    of StaticObjectManager. 
    @param name the name of the static variable 
    @param type the type of the static variable */
#define FEATSTD_SYNCED_STATIC_OBJECT(type, name, ...) \
    static FeatStd::Internal::StaticObjectPrivate::StaticObjectBuffer<type> FEATSTD_CONCAT2(name, ObjectBuffer) = { \
        { \
            &FEATSTD_CONCAT2(name, ObjectBuffer).mNode, \
            &FeatStd::MemoryManagement::DestructObject<type>, \
                FEATSTD_PRIVATE_STATIC_OBJECT_CONTEXT(name) \
            }, \
            { 0 } \
    }; \
    { \
        if (FEATSTD_CONCAT2(name, ObjectBuffer).mNode.mNext == &FEATSTD_CONCAT2(name, ObjectBuffer).mNode) { \
            FeatStd::Internal::FeatStdLocker lock; \
            if (FEATSTD_CONCAT2(name, ObjectBuffer).mNode.mNext == &FEATSTD_CONCAT2(name, ObjectBuffer).mNode) { \
            FEATSTD_LINT_NEXT_EXPRESSION(1713, "new T() - if POD Type is specified, it contributes to robustness if it is initialized with 0") \
            FEATSTD_LINT_NEXT_EXPRESSION(586, "placement operator new to construct a new object of type T regarded to be safe.") \
                (void) new(FeatStd::Internal::aligned_reinterpret_cast<FeatStd::MemoryManagement::Internal::PlacementNewType*>(FEATSTD_CONCAT2(name, ObjectBuffer).mObject)) type(__VA_ARGS__); \
                FEATSTD_CONCAT2(name, ObjectBuffer).mNode.Enlist(); \
            } \
        } \
    } \
    type &name = *FEATSTD_CONCAT2(name, ObjectBuffer)

/** define a static object within function scope. 
    The object will be constructed and chained to the destruction list
    of StaticObjectManager. 
    @param name the name of the static variable 
    @param type the type of the static variable */
#define FEATSTD_UNSYNCED_STATIC_OBJECT(type, name, ...) \
    static FeatStd::Internal::StaticObjectPrivate::StaticObjectBuffer<type> FEATSTD_CONCAT2(name, ObjectBuffer) = { \
        { \
            &FEATSTD_CONCAT2(name, ObjectBuffer).mNode, \
            &FeatStd::MemoryManagement::DestructObject<type>, \
            FEATSTD_PRIVATE_STATIC_OBJECT_CONTEXT(name) \
        }, \
        { 0 } \
    }; \
    if (FEATSTD_CONCAT2(name, ObjectBuffer).mNode.mNext == &FEATSTD_CONCAT2(name, ObjectBuffer).mNode) { \
        FEATSTD_LINT_NEXT_EXPRESSION(1713, "new T() - if POD is specified, it contributes to robustness if it is initialized with 0") \
        FEATSTD_LINT_NEXT_EXPRESSION(586, "placement operator new to construct a new object of type T regarded to be safe.") \
        (void) new(FeatStd::Internal::aligned_reinterpret_cast<FeatStd::MemoryManagement::Internal::PlacementNewType*>(FEATSTD_CONCAT2(name, ObjectBuffer).mObject)) type(__VA_ARGS__); \
        FEATSTD_CONCAT2(name, ObjectBuffer).mNode.Enlist(); \
    } \
    type &name = *FEATSTD_CONCAT2(name, ObjectBuffer)


namespace FeatStd {

    // =====================================================================

    /** StaticObjectManager is command control for static object destruction 
        Static objects defined with FEATSTD_SYNCED_STATIC_OBJECT or FEATSTD_UNSYNCED_STATIC_OBJECT 
        will be chained into the destruction list of StaticObjectManager. */
    class StaticObjectManager {
        private:
            static FeatStd::Internal::StaticObjectPrivate::DtorListNode *mListHead;      ///< head of destruction list
            
            static FeatStd::Internal::CriticalSection& GetLock();

            /** Destruct all static objects that have been enlisted */
            static void DestructStaticObjects();

            /** enlist given node to the destruction list */
            static void Enlist(FeatStd::Internal::StaticObjectPrivate::DtorListNode *objectNode);

            friend class FeatStd::Internal::LifeTimeManager;
            friend struct FeatStd::Internal::StaticObjectPrivate::DtorListNode;
            friend struct FeatStd::UnitTest::StaticObjectAccessor;

            FEATSTD_MAKE_CLASS_STATIC(StaticObjectManager);
    };

}

namespace FeatStd { namespace Internal { namespace StaticObjectPrivate {
    // =====================================================================

    /** prototype of the destruct function */
    typedef void (*DestructFn)(void*);

    /** node in the StaticObjectManager destruction list 
        The static object chained by the DtorListNode is immediately following
        in memory the node structure. */
    struct DtorListNode {
        void Enlist();

        DtorListNode *mNext;                            ///< next pointer to chain in StaticObjectList destruction list
        DestructFn mDestruct;                           ///< destruct function of the object 
#if defined(FEATSTD_DEBUG)
        const Char *mContext;                           ///< context (file + line) of the static object
#endif
    };

    // =====================================================================

    /** POD structure capable to store a DtorListNode and an
        object of type T. */
    template<typename T> struct StaticObjectBuffer {
        typedef typename ObjectBufferTypeProvider<T>::Type BufferType;

        StaticObjectPrivate::DtorListNode mNode;            ///< the node to chain to StaticObjectManager destruction list
        BufferType mObject;                                 ///< buffer to construct an object of type T

        inline T& operator*() {
            FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::Internal::ArrayTrait<T>::IsArray);
            return *aligned_reinterpret_cast<T*>(mObject);
        }

        inline const T& operator*() const {
            FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::Internal::ArrayTrait<T>::IsArray);
            return *aligned_reinterpret_cast<const T*>(mObject);
        }
    };





}}}

/** declare StaticObjectBuffer to be POD types (required for PodUtil macros) */
FEATSTD_LINT_NEXT_EXPRESSION(1960, "MISRA C++ 2008 Required Rule 7-3-1: removing the symbol from global name space violates backward compatibility. FeatStd prefix should be sufficient to avoid conflicts.")
FEATSTD_LINT_NEXT_EXPRESSION(1577, "MISRA C++ 2008 Required Rule 14-7-3: specialization only can be done with type declaration.")
template<typename T> struct FeatStdTypeTrait< FeatStd::Internal::StaticObjectPrivate::StaticObjectBuffer<T> > { 
    enum { IsPodType = true }; 
};

/// @}
#endif
