//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_RunLenghtEncoding_h)
#define FeatStd_Util_RunLenghtEncoding_h

#include <FeatStd/Base.h>

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_UTILS
/// @{

    /** Decompresses an run length encoded byte stream 
        The decompressed byte stream is handed to a user provided callback function.
        The RLE stream format is defined as follows:

        A run is a sequence of maximal 128 bytes. Each run is prefixed with a control
        byte. The control byte encodes the length of the run (7 least significant bits)
        and the run type (highest significant bit). As runs with 0 size are not valid, 
        the encoded length is the actual run length minus one (thus the maximum lenght of
        a run is 128 and not 127).
        If the highest significant bit in the control byte is not set, the run is a 
        sequence of identical bytes. Otherwise the run is a sequence of arbitrary bytes
        following the control byte.

        Examples:
        @code
        040a -> 0a0a0a0a0a
        800a -> 0a
        810a0b -> 0a0b
        @endcode
        */
    class RleDecompressor {
        public:
            /** identifies the operation to perform */
            enum Operation {
                Copy,                           ///< copy the given number bytes from the given source
                Fill                            ///< fill the given number of bytes with the given pattern byte
            };

            /** the signature of the write function for uncompressed data 
                @param op the operation to be performed (Copy or Fill)
                @param source if op == Fill, source will refer to the pattern byte to be used for the fill operation
                              if op == Copy, source will refer to the source buffer for the copy operation
                @param nBytes the number of bytes to copy or fill
                @param userData transparent value passed through from Decompress function */
            typedef void(*WriteSignature)(Operation op, const UInt8* source, SizeType nBytes, void *userData);

            /** setup decompressor to start decompression from the given control byte
                @param rleCtrlByte a pointer to a control byte in a valid RLE data stream 
                @param runOffset start decompression at the given offset within the run defined by rleCtrlByte */
            RleDecompressor(const void *rleCtrlByte, UInt8 runOffset = 0);

            /** decompress a given number of bytes 
                The decompressor will decompress the given number of bytes. The number of
                bytes refers to the number of actually decompressed bytes.
                @param nBytes the number of bytes to decompress 
                @param Write the callback function that will receive the decompressed 
                             byte stream
                @param userData user data which will be transparently passed to Write callback */
            void Decompress(SizeType nBytes, WriteSignature Write, void *userData);

        private:
            UInt8 mRunOffset;
            const UInt8 *mCtrlByte;

            FEATSTD_LINT_CURRENT_SCOPE(1712, "default intentionally not defined")
    };
/// @}
}}

#endif
