//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/Util/Traits.h>
#include "RunLengthEncoding.h"


namespace FeatStd { namespace Internal {


    const UInt8 cRunLenMask = 0x7f;         // mask to extract the run length from control byte
    const UInt8 cRunLenUnmask = 0x80;       // mask to extract the run type from control byte


    // ------------------------------------------------------------------------
    RleDecompressor::RleDecompressor(const void *rleCtrlByte, UInt8 runOffset) : 
        mRunOffset(runOffset),
        mCtrlByte(aligned_cast<const UInt8*>(rleCtrlByte))
    {           
    }

    // ------------------------------------------------------------------------
    void RleDecompressor::Decompress(SizeType nBytes, WriteSignature Write, void *userData)
    {
        // abandon if no write or data supplied
        if ((mCtrlByte == 0) || (Write == 0)) {
            return;
        }

        Operation op;

        for (;;) {

            // receives the length in bytes of the run to decompress
            UInt32 runLen;

            for (;;) {
                // determine op type - high bit set -> copy
                op = (((*mCtrlByte) & cRunLenUnmask) != 0) ? Copy : Fill;
                runLen = static_cast<UInt32>((*mCtrlByte) & cRunLenMask) + 1U;

                // the offset might refer to a position in the next run (this simplifies the
                // compressor implementation. 

                FEATSTD_DEBUG_ASSERT(mRunOffset < (runLen + 3U));
                if (runLen > mRunOffset) {
                    // if the offset is within the current run, stop here
                    break;
                }
                // advance to next run
                mCtrlByte += (op == Copy) ? (runLen + 1) : 2;
                mRunOffset -= static_cast<UInt8>(runLen);
            }

            // receives next position in stream 
            const UInt8 *mNextCtrlByte = mCtrlByte;
            UInt8 mNextRunOffset;

            // receives the byte count passed to the Write callback, set to rhe number
            // of bytes remaining in the current run
            UInt32 opByteCount = runLen - mRunOffset;
            if (opByteCount > nBytes) {
                // if all bytes to decompress are within the current run, 
                // advance nextPos within the current run
                mNextRunOffset = mRunOffset + static_cast<UInt8>(nBytes);

                // op byte count equals to requested nBytes
                opByteCount = static_cast<UInt32>(nBytes);
            }
            else {
                // more bytes then in the current run are requested ->
                // advance next pos to the next run and run offset 0
                mNextCtrlByte += (op == Copy) ? (runLen + 1) : 2U;
                mNextRunOffset = 0;
            }

            // receives the source pointer (either for copy or fill)
            const UInt8 *opSource;
            if (op == Copy) {
                // let source refer to first byte of the sequence to copy
                opSource = mCtrlByte + mRunOffset + 1U;
            }
            else {
                // let source refer to the pattern byte
                opSource = mCtrlByte + 1;
            }

            // invoke the Write callback
            Write(op, opSource, opByteCount, userData);

            // advance to next pos
            mCtrlByte = mNextCtrlByte;
            mRunOffset = mNextRunOffset;
            if (nBytes == opByteCount) {
                // stop if requested number of bytes have been decompressed
                break;
            }
            nBytes -= static_cast<SizeType>(opByteCount);
        }
    }

}}
