//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FEATSTD_RTTI_H)
#define FEATSTD_RTTI_H

#include <FeatStd/Platform/Types.h>
#include <FeatStd/Util/WarningMacros.h>


namespace FeatStd {

namespace MemoryManagement {
    template <typename T>
    class SharedPointer;
}

typedef const Char* TypeId;

namespace Internal {
/// @addtogroup Rtti
/// @{
    class TypeIdWrapper {
    public:
        TypeIdWrapper(): m_char('\0') {}
        TypeId GetTypeId() const { return &m_char; }
    private:
        Char m_char;
    };

    /**
     * This is used to grant access to the base class, even if declared in private section.
     */
    struct BaseClassRelation
    {
        template<typename BaseT, typename T>
        static inline bool IsTypeOf(T& object, ::FeatStd::TypeId typeId)
        {
            return object.BaseT::IsTypeOf(typeId);
        }
    };
}

/**
 *  Macros for Custom runtime type information (RTTI).
 *  GetTypeId() returns a unique type identifier.
 *  Check with IsTypeOf() if the type matches.
 *  The derivation hierarchy is also considered in IsTypeOf().
 */

/**
 *  Macro FEATSTD_RTTI_DECLARATION_BASE has to be used in the header file class declaration, public section.
 */
#define FEATSTD_RTTI_DECLARATION() \
    FEATSTD_LINT_SYMBOL(1511, *::GetTypeId, FEATSTD_LINT_REASON_RTTI) \
    static ::FeatStd::TypeId GetTypeId(); \
    virtual ::FeatStd::TypeId GetDynamicTypeId() const; \
    virtual bool IsTypeOf(::FeatStd::TypeId typeId) const

/**
 *  Macro FEATSTD_RTTI_BASECLASS_DEFINITION has to be used in cpp file whenever the class is a base class
 *  or its base class has no RTTI information.
 */
FEATSTD_LINT_MACRO(1960, FEATSTD_RTTI_BASECLASS_DEFINITION, "Only global scope is applicable to CanderaInjectedWidgetBase")
#define FEATSTD_RTTI_BASECLASS_DEFINITION(CLASS) \
    _FEATSTD_RTTI_DEFINITION_COMMON(CLASS, _FEATSTD_RTTI_GET_TYPE_ID_BASECLASS_IMPL)

/**
 *  Macro FEATSTD_RTTI_DEFINITION has to be used in cpp file whenever the class has a base class with RTTI information.
 */
#define FEATSTD_RTTI_DEFINITION(CLASS, BASECLASS) \
    _FEATSTD_RTTI_DEFINITION_COMMON(CLASS, _FEATSTD_RTTI_GET_TYPE_ID_IMPL(BASECLASS))

// --- for internal use only ----

/**
 * Used to grant access to base class, even if FEATSTD_RTTI_TYPEDEF_BASE is declared in private section.
 */

#define _FEATSTD_RTTI_GET_TYPE_ID_BASECLASS_IMPL \
    (GetTypeId() == typeId)

#define _FEATSTD_RTTI_GET_TYPE_ID_IMPL(BASECLASS) \
    (GetTypeId() == typeId) || ::FeatStd::Internal::BaseClassRelation::IsTypeOf< BASECLASS >(*this, typeId)

#define _FEATSTD_RTTI_DEFINITION_COMMON(CLASS, GET_TYPE_ID_IMPL) \
    ::FeatStd::TypeId CLASS::GetTypeId() \
    { \
        static const FeatStd::Char* const c_typeId = #CLASS; \
        return c_typeId; \
    } \
    ::FeatStd::TypeId CLASS::GetDynamicTypeId() const \
    { \
        return GetTypeId(); \
    } \
    bool CLASS::IsTypeOf(::FeatStd::TypeId typeId) const \
    { \
        return (GET_TYPE_ID_IMPL); \
    }

/**
 *  Macro FEATSTD_RTTI_DECLARATION has to be used in the header file class declaration, public section.
 */
#define FEATSTD_RTTI_TEMPLATE_DEFINITION(CLASS, BASECLASS) \
    FEATSTD_LINT_SYMBOL(1511, CLASS::GetTypeId, FEATSTD_LINT_REASON_RTTI) \
    static ::FeatStd::TypeId GetTypeId() \
    { \
        static ::FeatStd::Internal::TypeIdWrapper typeIdWrapper = ::FeatStd::Internal::TypeIdWrapper(); \
        return typeIdWrapper.GetTypeId(); \
    } \
    virtual ::FeatStd::TypeId GetDynamicTypeId() const \
    { \
        return GetTypeId(); \
    } \
    virtual bool IsTypeOf(::FeatStd::TypeId typeId) const \
    { \
        return (_FEATSTD_RTTI_GET_TYPE_ID_IMPL(BASECLASS)); \
    }

template<typename TOut, typename TIn> TOut Dynamic_Cast(TIn tIn);

namespace Internal
{
    // Make SharedPointer identifier visible inside
    using MemoryManagement::SharedPointer;

    template <typename TOut, typename TIn> class DynamicCastDispatcher;

    // Use this in case of non-const pointer to object
    // Foo f;
    // Foo* pf = &f;
    //E.g. Dynamic_Cast<DerivedFromFoo* >(pf)
    template <typename TOut, typename TIn>
    class DynamicCastDispatcher<TOut*, TIn*>
    {
    public:
        static TOut* Perform(TIn *tIn)
        {
            if ((tIn != 0) && (tIn->IsTypeOf(TOut::GetTypeId()))) {
                return static_cast<TOut* >(tIn);
            }
            else {
                return 0;
            }
        }
    };

    // Use this in case of shared pointer
    template <typename TOut, typename TIn>
    class DynamicCastDispatcher <SharedPointer<TOut>, SharedPointer<TIn> >
    {
    public:
        static SharedPointer<TOut> Perform(const SharedPointer<TIn> &tIn)
        {
            return SharedPointer<TOut>(Dynamic_Cast<TOut*>(tIn.GetPointerToSharedInstance()));
        }
    };
} // End of Internal namespace

template<typename TOut, typename TIn> TOut Dynamic_Cast(TIn tIn)
{
    return Internal::DynamicCastDispatcher<TOut, TIn>::Perform(tIn);
}

/// @}
}
#endif
