//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef FEATSTD_RANDOM_H
#define FEATSTD_RANDOM_H

#include <FeatStd/Platform/Types.h>
#include <FeatStd/Util/FeatLimits.h>

namespace FeatStd
{

/**
 * @brief  The Random class generates random numbers deterministically. It is based
 *         on the "Mother-of-all" generator from http://www.agner.org/random/
 */

class Random
{
public:
    /**
     *  Constructor
     *  @param seed  The seed for the generator to influence the sequence of deterministic random numbers.
     */
    Random(UInt32 seed = 0);

    /**
     *  Constructor
     *  @param seed  The seed for the generator to influence the sequence of deterministic random numbers.
     */
    void Init(UInt32 seed);
 
    /**
     *  Get a random integer number from the interval [0..2^32-1].
     *  @return  A random number.
     */
    UInt32 Get();

    /**
     *  Get a random floating point number with single precision from the interval [0..1].
     *  @return  A random floating point number with single precision from the interval [0..1].
     */
    Float GetFloat() { return static_cast<Float>(Get()) / static_cast<Float>(Internal::NativeTypeLimit<UInt32>::Max()); }

    /**
     *  Get a random floating point number with double precision from the interval [0..1].
     *  @return  A random floating point number with double precision from the interval [0..1].
     */
    Double GetDouble() { return static_cast<Double>(Get()) / static_cast<Double>(Internal::NativeTypeLimit<UInt32>::Max()); }

protected:
    UInt64 mX[5];
};

}

#endif
