//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_RBTree_h)
#define FeatStd_Util_RBTree_h

#include <FeatStd/Base.h>
#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/Util/FeatLimits.h>
#include <FeatStd/Util/PointerUtil.h>

namespace FeatStd { namespace UnitTest {
    struct RBTreeUnitTestAccessor;
}}

namespace FeatStd { namespace Internal { namespace RBTree {
/// @addtogroup FEATSTD_UTILS
/// @{

    namespace NodeColor {
        /** node color enumeration for RedBlackTreeBase */
        enum Enum {
            Red = 1,
            Black = 0
        };
    }

    /** node element of a RedBlackTreeBase 
        Each node has  node color (NodeColor::Enum) and a left and right link (child node) 
        The color is encoded in the lowest bit of the left child pointer (it is assumed that
        compund types like Node are at least four byte aligned). */
    class Node {
        public:
            /** node ctor */
            Node();
            /** node ctor 
                @param left pointer to the left child (may be 0 (null)) 
                @param right pointer to the right child (may be 0 (null)) */
            Node(Node *left, Node *right);
            ~Node();

            /** resets node to its initial state */
            void Reset() {
                mLinkVal[0] = cRedFlag;
                mLinkVal[1] = 0;
            }

            /** returns true if the node has red color.
                @return true if node is colored red, false when black colored */
            bool IsRed() const {
                return (mLinkVal[0] & cRedFlag) != 0;
            }

            /** returns the node color 
                @return the node color */
            NodeColor::Enum GetColor() const {
                return IsRed() ? NodeColor::Red : NodeColor::Black;
            }

            /** set the node color
                @param value the node color */
            void SetColor(NodeColor::Enum value) {
                if (value == NodeColor::Red) {
                    mLinkVal[0] |= cRedFlag;
                }
                else {
                    mLinkVal[0] &= ~cRedFlag;
                }
            }

            /** returns true if the node has no left and right link */
            bool IsLeave() const {
                return (mLinkVal[1] == 0) && ((mLinkVal[0] & cFlagsUnmask) == 0);
            }

            /** return the left child of the node 
                @return left child of node or 0 (null) if no child */
            Node* GetLeftLink() const {
                return ScalarToPointer<Node*>(mLinkVal[0] & cFlagsUnmask);
            }

            /** set the left link of the node 
                @param node the left pointer to the left child (may be 0 (zero)) */
            void SetLeftLink(Node *node) {
                mLinkVal[0] = (mLinkVal[0] & cFlagsMask) | PointerToScalar<PtrValType>(node);
            }

            /** return the right child of the node 
                @return right child of node or 0 (null) if no child */
            Node* GetRightLink() const {
                return ScalarToPointer<Node*>(mLinkVal[1]);
            }

            /** set the right link of the node 
                @param node the right pointer to the right child (may be 0 (zero)) */
            void SetRightLink(Node *node) {
                mLinkVal[1] = PointerToScalar<PtrValType>(node);
            }

            /** return the specified child link 
                @param index index 0 specifies left child, index 1 right child
                @return specified child pointer */
            Node* GetLink(UInt32 index) const {
                FEATSTD_DEBUG_ASSERT(index < 2);
                return (index == 0) ? GetLeftLink() : GetRightLink();
            }

            /** set the specified child link 
                @param index index 0 specifies left child, index 1 right child
                @param node the pointer to the child (may be 0 (zero)) */
            void SetLink(UInt32 index, Node *node) {
                if (index == 0) {
                    SetLeftLink(node);
                }
                else {
                    SetRightLink(node);
                }
            }

            /** traversing the tree requires additional information in the node 
                iterative traversal of the tree requires additional information in 
                the node
                @return true if tag is set, false otherwise */
            bool IsTagged() const {
                return (mLinkVal[0] & cTagFlag) != 0;
            }

            /** Set the tag on the node 
                @param value the tag value */
            void SetTag(bool value = true) {
                if (value) {
                    mLinkVal[0] |= cTagFlag;
                }
                else {
                    mLinkVal[0] &= ~cTagFlag;
                }
            }

        private:
            /// define integer type large enough to store a memory address
            typedef ByteCountToIntegerTypeMapper<sizeof(Node*)>::UnsignedType PtrValType;

            PtrValType mLinkVal[2];

            static const PtrValType cRedFlag = PtrValType(NodeColor::Red);
            static const PtrValType cTagFlag = PtrValType(0x02);
            static const PtrValType cFlagsMask = PtrValType(cRedFlag | cTagFlag);
            static const PtrValType cFlagsUnmask = PtrValType(~cFlagsMask);

            friend struct UnitTest::RBTreeUnitTestAccessor;
    };

    /** NodeSelector is a functor used to select a node in RedBlackTreeBase traversal */
    class NodeSelector {
        public:
            /** returns 0 if node should be selected, -1 the node to be selected is 
                smaller than the given node, 1 if the node to be selected is larger 
                the given node
                @param node the node to test */
            virtual Int32 operator()(const Node &node) = 0;
    };

    /** standard comparision of two nodes in a RedBlackTreeBase.
        compares left with right:
        - return 0 if left == right 
        - return -1 if left < right
        - return 1 if left > right
        @param left the left operand for comparison 
        @param right the right operand for comparison 
        @return comparison result. */
    typedef Int32 (*CompareNodeSignature)(const Node &left, const Node &right);

    /** VisitSignature - node traversal node callback
        @param node the currently visited node 
        @param userData pointer to user data
        @return true if tree traversal shall continue, false to stop tree traversal */
    typedef bool (*VisitSignature)(Node &node, void *userData);

    /** RedBlackTree functions */
    struct Functions {
        /** direction enumeration */
        enum Direction {
            Left = 0,       ///< left child 
            Right = 1       ///< right child
        };

        /** insert the given node with the given compare function
            @node the node to insert
            @CompareFn the compare function */
        static void Insert(Node *&root, Node *node, CompareNodeSignature CompareFn);

        /** remove the given node from the tree
            @param selector the node selector to use for the remove operation
            @return the node selected for removal, 0 if no node has been selected */
        static Node* Remove(Node *&root, NodeSelector &selector);

        /** traverse the tree for a node 
            @param selector the selector to select the node of interest
            @return the node selected by the selector, 0 if no node found */
        static Node* Find(Node *root, NodeSelector &selector);

        /** UnlinkAllNodes traverses through all nodes and removes them from the tree
            The node passed to the VisitFn is already removed from tree. It is not
            possible to stop tree traversal by returning false from VisitFn. This
            would leave the tree in an undefined state. 
            @param root the root node, will be set to 0 (null) 
            @param VisitFn the visit function
            @param userData optional user data that will be passed to the visit function */
        static void UnlinkAllNodes(Node *&root, VisitSignature VisitFn, void *userData = 0);

        /** Visit traverses through all nodes in sort order
            @param node the root node
            @param VisitFn the visit function
            @param userData optional user data that will be passed to the visit function 
            @return false if VisitFn abandoned traversal by returning false.*/
        static bool Visit(Node *node, VisitSignature VisitFn, void *userData = 0);

        /** change the given direction to the opposite direction 
            @param dir the given direction
            @return opposite direction, ie. Right->Left, Left->Right */
        static inline Direction ReverseDirection(Direction dir) {
            return static_cast<Direction>(1 - Int(dir));
        }

        /** check if the given node is colored red 
            @param node node pointer (may be 0)
            @return true if given node is Red, false if black or node pointer is 0. */
        static inline bool IsRed(const Node *node) {
            return (node != 0) && (node->IsRed());
        }

        /** apply a single rotation on the tree 
            @param root tree root
            @param dir rotation direction 
            @return new tree root */
        static Node* RotateSingle(Node *root, Direction dir);

        /** apply double rotation on the tree 
            @param root tree root
            @param dir rotation direction 
            @return new tree root */
        static Node* RotateDouble(Node *root, Direction dir);

        /** changes node color to Black and link colors to Red 
            @param node the node to repaint */
        static void PaintItBlack(Node *node);

        /** changes node color to Red and link colors to Black
            @param node the node to repaint */
        static void PaintItRed(Node *node);
    };

    FEATSTD_LINT_SYMBOL(1763, FeatStd::Internal::RBTree::RedBlackTreeBase::GetRoot, "intended behavior")

    /** base class for RedBlackTree implementations */
    class RedBlackTreeBase {
        public:
            /** returns true if items exist in the tree */
            bool HasItems() const {
                return mRoot != 0;
            }

        protected:
            RedBlackTreeBase();

            /** insert the given node with the given compare function
                @node the node to insert
                @CompareFn the compare function */
            void DoInsert(Node *node, CompareNodeSignature CompareFn) {
                Functions::Insert(mRoot, node, CompareFn);
            }

            /** remove the given node from the tree
                @param selector the node selector to use for the remove operation
                @return the node selected for removal, 0 if no node has been selected */
            Node* DoRemove(NodeSelector &selector) {
                return Functions::Remove(mRoot, selector);
            }

            /** traverse the tree for a node 
                @param selector the selector to select the node of interest
                @return the node selected by the selector, 0 if no node found */
            Node* DoFind(NodeSelector &selector) const {
                return Functions::Find(mRoot, selector);
            }

            /** Unlink all nodes from tree. 
                VisitFn will be invoked for each node to be unlinked. The node passed 
                to the VisitFn is already unlinked and can be eg. safely deleted.
                VisitFn must not return false to abandon traversal. 
                @param VisitFn the visit function for unlinked nodes 
                @param userData optional user data that will be passed to VisitFn */
            void DoUnlinkAllNodes(VisitSignature VisitFn, void *userData = 0) {
                Functions::UnlinkAllNodes(mRoot, VisitFn, userData);
            }

            /** Visit all tree nodes in sort order. 
                VisitFn will be invoked for each node. Node traversal will be
                continued until VisitFn returns false or all nodes have been
                visited.
                @param VisitFn the visit function 
                @param userData optional user data that will be passed to VisitFn 
                @return false if VisitFn abondoned traversal by returning false */
            bool DoVisit(VisitSignature VisitFn, void *userData = 0) {
                return Functions::Visit(mRoot, VisitFn, userData);
            }

            /** return the tree root node */
            Node* GetRoot() const {
                return mRoot;
            }

            /** return the tree root node */
            void SetRoot(Node *value) {
                mRoot = value;
            }

        private:
            /** the tree root node */
            Node *mRoot;

            friend struct UnitTest::RBTreeUnitTestAccessor;
    };

/// @}
}}}
#endif
