//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_PointerUtil_h)
#define FeatStd_Util_PointerUtil_h

#include <FeatStd/Util/Alignment.h>
#include <FeatStd/Util/Traits.h>
#include <FeatStd/Diagnostics/Diagnostics.h>

namespace FeatStd { namespace Internal { 
/// @addtogroup FEATSTD_UTILS
/// @{

    namespace PointerUtilPrivate {

        template<typename T, UInt32 sizeOfT = sizeof(T)> struct PointerToScalarMapper;

        template<typename T> struct PointerToScalarMapper<T, 2> {
            typedef UInt32 Scalar;
        };

        template<typename T> struct PointerToScalarMapper<T, 4> {
            typedef UInt32 Scalar;
        };

        template<typename T> struct PointerToScalarMapper<T, 8> {
            typedef UInt64 Scalar;
        };
    }

    /** Pointer support class
        Pointer arithmetic support. The union data member defines a pointer and
        scalar value. Arithmetic operations can be applied to the scalar member
        without the need of type casting the pointer. 
        Note that using this class might have implications to software portability. 
        @tparam T the pointer type */
    template<typename T = void*> struct ScalarPointerUnion {
        typedef typename PointerUtilPrivate::PointerToScalarMapper<T>::Scalar Scalar;

        /** default ctor to initialize scalar value 
            @param scalarValue the value mValue.mScalar will receive */
        ScalarPointerUnion(Scalar scalarValue = 0) {
            FEATSTD_COMPILETIME_ASSERT(FeatStd::Internal::PointerTrait<T>::IsPointer);
            mValue.mScalar = scalarValue;
        }

        /** default ctor to initialize pointer value 
            @param p the value mValue.mPointer will receive */
        explicit ScalarPointerUnion(T p) {
            FEATSTD_COMPILETIME_ASSERT(FeatStd::Internal::PointerTrait<T>::IsPointer);
            mValue.mPointer = p;
        }

        union {
            T mPointer;
            Scalar mScalar;
        } mValue;
    };

    /** Add nbytes to the given pointer and return result in the specified pointer type
        adding a given number of bytes to a pointer requires casting to UInt8*, adding
        the number of bytes, and casting back to the original pointer type. PointerAdd
        does all this and takes care that compilers and Lint are happy how it is done.
        @tparam D the type of the calculated pointer
        @tparam S the type of the source pointer
        @param p the pointer 
        @param nBytes the number of bytes to add to the pointer
        @return ((D) (((Uint8*) p) + nBytes)) */
    template<typename D, typename S> inline static D PointerAdd(S *p, OffsetType nBytes)
    {
        FEATSTD_DEBUG_ASSERT(p != 0);
        // transfer const, volatile from S to UInt8
        typedef typename FeatStd::Internal::TypeDecorationTransferer<S, UInt8>::DecoratedType P;
        return FeatStd::Internal::aligned_reinterpret_cast<D>(FeatStd::Internal::aligned_reinterpret_cast<P*>(p) + nBytes);
    }

    /** Add nbytes to the given pointer and return result
        adding a given number of bytes to a pointer requires casting to UInt8*, adding
        the number of bytes, and casting back to the original pointer type. PointerAdd
        does all this and takes care that compilers and Lint are happy how it is done.
        @tparam T the type of the pointer
        @param p the pointer 
        @param nBytes the number of bytes to add to the pointer
        @return ((T*) (((Uint8*) p) + nBytes)) */
    template<typename T> inline static T* PointerAdd(T *p, OffsetType nBytes)
    {
        return FeatStd::Internal::PointerAdd<T*>(p, nBytes);
    }

    /** Subtract nbytes to the given pointer and return result in the specified pointer type
        adding a given number of bytes to a pointer requires casting to UInt8*, adding
        the number of bytes, and casting back to the original pointer type. PointerAdd
        does all this and takes care that compilers and Lint are happy how it is done.
        @tparam D the type of the calculated pointer
        @tparam S the type of the source pointer
        @param p the pointer 
        @param nBytes the number of bytes to add to the pointer
        @return ((D) (((Uint8*) p) - nBytes)) */

    template<typename D, typename S> inline static D PointerSub(S *p, SizeType nBytes)
    {
        FEATSTD_DEBUG_ASSERT(p != 0);
        // transfer const, volatile from S to UInt8
        typedef typename FeatStd::Internal::TypeDecorationTransferer<S, UInt8>::DecoratedType P;
        return FeatStd::Internal::aligned_reinterpret_cast<D>(FeatStd::Internal::aligned_reinterpret_cast<P*>(p) - nBytes);
    }

    /** Subtract nbytes to the given pointer and return result
        adding a given number of bytes to a pointer requires casting to UInt8*, adding
        the number of bytes, and casting back to the original pointer type. PointerAdd
        does all this and takes care that compilers and Lint are happy how it is done.
        @tparam T the type of the pointer
        @param p the pointer 
        @param nBytes the number of bytes to add to the pointer
        @return ((T*) (((Uint8*) p) - nBytes)) */

    template<typename T> inline static T* PointerSub(T *p, SizeType nBytes)
    {
        // ArmCC needs whole namespace to identify overloaded template function.
        return FeatStd::Internal::PointerSub<T*>(p, nBytes);
    }

FEATSTD_SUPPRESS_GCC_WARNING_BEGIN("-Wstrict-aliasing", "Compile time assertion checks the size.")
    
    /** converts given pointer to scalar data type 
        @tparam Scalar the type receiving the pointer value
        @tparam T the pointer type
        @param pointer the pointer value 
        @return the pointer converted to the scalar value */
    template<typename Scalar, typename T>
    inline Scalar PointerToScalar(T pointer) 
    {
        FEATSTD_COMPILETIME_ASSERT(sizeof(T) == sizeof(Scalar));
        return *reinterpret_cast<Scalar*>(&pointer);
    }

    /** converts given scalar value to a pointer
        @tparam T the pointer type
        @tparam Scalar the type receiving the pointer value
        @param scalar the scalar holding the memory address of the pointer 
        @return the scalar value converted converted to a pointer */
    template<typename T, typename Scalar>
    inline T ScalarToPointer(Scalar scalar) 
    {
        FEATSTD_COMPILETIME_ASSERT(sizeof(T) == sizeof(Scalar));
        return reinterpret_cast<T>(scalar);
    }

    /** converts given scalar value to a pointer
        In contrast to ScalarToPointer this function will not check if the scalar value
        is able to store a pointer. Usage of this function requires detailed analysis
        of applicability.
        @tparam T the pointer type
        @tparam Scalar the type receiving the pointer value
        @param scalar the scalar holding the memory address of the pointer 
        @return the scalar value converted converted to a pointer */
    template<typename T, typename Scalar>
    inline T UnconditionalScalarToPointer(Scalar scalar) 
    {
        return reinterpret_cast<T>(scalar);
    }

    /** cast a pointer of type void to a given pointer type 
        @tparam T the pointer type to be returned
        @param pointer the pointer to void
        @return pointer converted to the requested pointer type */
    template<typename T>
    inline T PointerToPointer(void *pointer)
    {
        return static_cast<T>(pointer);
    }

    /** cast a pointer of type void to a given pointer type 
        @tparam T the pointer type to be returned
        @param pointer the pointer to void
        @return pointer converted to the requested pointer type */
    template<typename T>
    inline T PointerToPointer(const void *pointer)
    {
        return static_cast<T>(pointer);
    }

    /** returns the result of the operation (high - low)
        Note that high >= low and both pointer must point to the same memory region.
        @tparam T1 the type of the high argument
        @tparam T2 the type of the low argument
        @param high the high argument to (high - low)
        @param low the low argument to (high - low)
        @return the value of the operation (high - low) */
    template<typename T1, typename T2> inline OffsetType PointerDiff(T1 *high, T2 *low)
    {
        FEATSTD_DEBUG_ASSERT(high >= low);
        return high - low;
    }

    /** returns the result of operation (high - low) in *bytes*
        Note that high >= low and both pointer must point to the same memory region.
        @param high the high pointer
        @param low the low pointer 
        @return the value of operation (high - low) in *BYTES* */
    inline OffsetType PointerByteDiff(const void *high, const void *low)
    {
        FEATSTD_DEBUG_ASSERT(PointerToPointer<const UInt8 *>(high) >= PointerToPointer<const UInt8 *>(low));
        return PointerToPointer<const UInt8 *>(high) - PointerToPointer<const UInt8 *>(low);
    }

FEATSTD_SUPPRESS_GCC_WARNING_END()

/// @}
}}

#endif
