//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FEATSTD_OPTIONAL_H)
#define FEATSTD_OPTIONAL_H

#include <FeatStd/Config.h>
#include <FeatStd/Macros.h>

namespace FeatStd {

    /**
     *  Template class Initialized is a wrapper for representing initialized objects that have been initialized.
     */
    template<typename T> class Initialized
    {
    public:
        inline Initialized() :
            m_value(T())
        {
        }

        explicit inline Initialized(const T& value) :
            m_value(value)
        {
        }

        inline Initialized(const Initialized& value) :
            m_value(value.m_value)
        {
        }

        inline T& operator=(const Initialized& value)
        {
            return m_value = value.m_value;
        }

        inline T& operator=(const T& value)
        {
            return m_value = value;
        }

        inline operator T&()
        {
            return m_value;
        }

        inline operator const T&() const
        {
            return m_value;
        }

        inline bool operator==(const Initialized<T>& rhs) const
        {
            return (m_value == rhs.m_value);
        }

        inline bool operator!=(const Initialized<T>& rhs) const
        {
            return !(operator==(rhs));
        }
    private:
        T m_value;
    };
    
    /**
     *  @brief The template class Optional is a wrapper for representing optional objects that may not have been initialized.
     *
     *  Optional solves the issue of communicating whether or not a value is valid. While it may be possible to reserve a specific value of a data type 
     *  to describe an invalid value, such as NULL or EOF, this approach can only be used when the data type can hold all meaningful values additionally to the invalid value.
     *
     *  With Optional it is possible to determine whether or not a value was set by testing it against a boolean value:
     *  \code
     *  Optional<Int> opt;
     *  ...
     *  if(true == opt) {
     *    Int value = *opt; //opt has been set
     *  }
     *  else {
     *    //opt has not been set
     *  }
     *  \endcode
     *
     *  If the value is not set, it will hold the default initialized value for the specified type,
     *  hence it can only be used for basic types or types with an accessible default constructor.
     *
     */
    template<typename T> class Optional
    {
    public:

        inline Optional() :
            m_set(false),
            m_value()
        {
        }

        explicit inline Optional(const T& value) :
            m_set(true),
            m_value(value)
        {
        }

        inline Optional(const Optional& value) :
            m_set(value.m_set),
            m_value(value.m_value)
        {
        }

        inline Optional& operator=(const Optional& value)
        {
            m_set = value.m_set;
            m_value = value.m_value;
            return *this;
        }

        inline Optional& operator=(const T& value)
        {
            m_set = true;
            m_value = value;
            return *this;
        }

        bool IsSet() const
        {
            return m_set;
        }

#if(__cplusplus >= 201103L)
        // C++11 allows the use of explicit operators to avoid potentially unsafe operations.
        FEATSTD_DEPRECATED_3_5_0("Use IsSet instead!", inline operator bool() const);
#else
        // Implementation of the safe bool idiom to allow safe comparison of Optional to bool for pre C++11 compilers.
        typedef void (Optional::*SafeBool)() const;
        FEATSTD_DEPRECATED_3_5_0("Use IsSet instead!", inline operator SafeBool() const);
#endif

        FEATSTD_DEPRECATED_3_5_0("Use IsSet instead!", inline bool operator==(bool value) const);

        /**
         * Requires that instance of optional is initialized.
         */
        inline const T& operator*() const
        {
            return m_value;
        }

        inline bool operator==(const typename FeatStd::Optional<T>& rhs) const
        {
            if ((m_set) && (rhs.m_set))
            {
                // compare value
                return (m_value == rhs.m_value);
            }
            else
            {
                return (m_set == rhs.m_set);
            }
        }
        inline bool operator!=(const typename FeatStd::Optional<T>& rhs) const
        {
            return !(operator==(rhs));
        }

    private:
        void ComparisonIsNotSupported() const {};

        bool m_set;
        Initialized<T> m_value;
    };

#if(__cplusplus >= 201103L)
    template<typename T> inline Optional<T>::operator bool() const
    {
        return m_set;
    }
#else
    template<typename T> inline Optional<T>::operator typename Optional<T>::SafeBool() const
    {
        return m_set ? &Optional::ComparisonIsNotSupported : 0;
    }
#endif

    template<typename T> inline bool Optional<T>::operator==(bool value) const
    {
        return m_set == value;
    }

}   // namespace FeatStd

#endif // FEATSTD_OPTIONAL_H
