//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_NumericUtil_h)
#define FeatStd_Util_NumericUtil_h

#include <FeatStd/Util/FeatLimits.h>
#include <FeatStd/Util/Traits.h>

namespace FeatStd { namespace Internal { 
/// @addtogroup FEATSTD_UTILS
/// @{

    /**
    * Unsafe helper function to cast one integer type to another integer type.
    *
    * For debug builds it will assert that the source value can be represented in the Destination type.
    * This is done by testing whether the bit width of Destination is larger or equal to the bit width of Destination 
    * and if it is not whether the unsigned source value is smaller than the maximum unsigned value of the Destination type.
    *
    * The same caveats apply as for static_cast<T>.
    *
    * Usage: 
    * \code{.cpp}
    *  UInt64 largerInt = 0xFFffFFff; // Exactly 4 bytes used.
    *  FeatStd::Internal::NumericConversion<UInt32>(largerInt); // Valid.
    *  largerInt++; // Number now exceeds 4 bytes.
    *  FeatStd::Internal::NumericConversion<UInt32>(largerInt); // Invalid, debug assertion thrown.
    * \endcode
    *
    * @param source value of Source type.
    * @return value of source cast to Destination type.
    */
    template<typename Destination, typename Source>
    inline static Destination NumericConversion(Source source) {
#ifdef FEATSTD_DEBUG
        typedef typename MakeUnsigned<Source>::Type USource;
        typedef typename MakeUnsigned<Destination>::Type UDestination;
        const USource u_source = static_cast<USource>(source);
        const UDestination u_maxDestination = NativeTypeLimit<UDestination>::Max();
        FEATSTD_DEBUG_ASSERT(u_source <= u_maxDestination); // ensure That the value fits into the new type
#endif
        return static_cast<Destination>(source);
    }

/// @}
}}

#endif // !defined(FeatStd_Util_NumericUtil_h)
