//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Hash.h"
#include <FeatStd/Diagnostics/Debug.h>

#include <FeatStd/Util/TextEncoding.h>

namespace FeatStd {

    using namespace FeatStd::Internal;

    /* http://burtleburtle.net/bob/hash/doobs.html
       http://www.team5150.com/~andrew/noncryptohashzoo/OneAtATime.html
    */

    // ------------------------------------------------------------------------
    UInt32 Hash::CalcHash(register const Char *name, SizeType nameLen)
    {
        register const Char *last = name + nameLen;
        register UInt32 hash = 0;

        if (*name == cHashIndicator) {
            ++name;
            while ((*name != '\0') && (name != last)) {
                hash <<= 4U;
//! [FEATSTD_DEBUG_FAIL]
                if ((*name >= '0') && (*name <= '9')) {
                    hash += (ToUInt32(*name) - '0');
                }
                else if ((*name >= 'A') && (*name <= 'F')) {
                    hash += (ToUInt32(*name) - (static_cast<UInt32>('A') - 10));
                }
                else if ((*name >= 'a') && (*name <= 'f')) {
                    hash += (ToUInt32(*name) - (static_cast<UInt32>('a') - 10));
                }
                else {
                    // failure
                    FEATSTD_DEBUG_FAIL();
                }
//! [FEATSTD_DEBUG_FAIL]
                ++name;
            }
        }
        else {
            UInt32 seed = 11;
            for(;;) {
                while ((*name != '\0') && (name != last)) {
                    hash += ToUInt32(*name);
                    hash += (hash << 10);
                    hash ^= (hash >> 6);
                    ++name;
                }

                hash += (hash << 3);
                hash ^= (hash >> 11);
                hash += (hash << 15);

                // invalid hash value -> retry with a different initial seed value
                if (hash != cInvalidHashValue) {
                    break;
                }
                hash = seed;
                seed += 2;
            }
        }
        return hash;
    }

    // ------------------------------------------------------------------------
    UInt32 Hash::CalcBinBufferHash(register const UInt8 * buffer, SizeType bufferLen)
    {
        register const UInt8 *last = buffer + bufferLen;
        UInt32 hash = cInitialHashValue;
        UInt32 seed = cInitialSeed;
        do {
            UpdateBinBufferHash(buffer, last, hash);
        } while (!FinishBinBufferHash(hash, seed));
        return hash;
    }

    // ------------------------------------------------------------------------
    void Hash::UpdateBinBufferHash(register const UInt8 * buffer, register const UInt8 *end, UInt32& hash)
    {
        register UInt32 currentHash = hash;
        while (buffer != end) {
            currentHash += *buffer;
            currentHash += (currentHash << 10);
            currentHash ^= (currentHash >> 6);
            ++buffer;
        }
        hash = currentHash;
    }

    // ------------------------------------------------------------------------
    bool Hash::FinishBinBufferHash(UInt32& hash, UInt32& seed)
    {
        hash += (hash << 3);
        hash ^= (hash >> 11);
        hash += (hash << 15);

        // invalid hash value -> retry with a different initial seed value
        if ((hash != cInvalidHashValue) && (hash != 0)) {
            return true;
        }
        hash = seed;
        seed += 2;
        return false;
    }
}
