//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################
#if !(defined GUID_H)
#define GUID_H

#include <FeatStd/Platform/Memory.h>

namespace FeatStd {
    namespace Internal {
        /// @addtogroup FEATSTD_UTILS
        /// @{

        /**
        * @brief The Guid class is used to represent GUIDs as described in https://tools.ietf.org/html/rfc4122.html.
        * The Guid can be constructed without parameters to be a NIL GUID that can be set afterwards.
        * It is also possible to use text representation of the GUID to construct it.
        */
        class Guid {

        public:
            /**
            * Default ctor connstructs a NIL GUID
            */
            Guid();
            /**
            * Copy ctor
            * creates a deep copy of a Guid object.
            * @param other 
            */
            Guid(const Guid& other);
            /**
            * ctor
            * @param humanReadableGuid GUID in human readable format, null terminated c string in format "XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX"
            */
            explicit Guid(const FeatStd::Char*  humanReadableGuid);
            /**
            * dtor
            */
            ~Guid();
            /**
            *  ToString
            *  Converts the internal representation to a c string representation.
            *  @param dest destination c string. must at least have size bytes. Should have at least 37 bytes (32(numbers) + 4 (separators) +1(null termination))
            *  @param size capacity of dest.
            */
            void ToString(FeatStd::Char* dest, const FeatStd::SizeType  size) const;

            /**
            *  CopyBytes
            *  @param src Guid to copy identifier from
            *  @param dest Char[] to copy identifier to
            *  @param destSize maximum number of bytes to copy (size of dest)
            */
            static void CopyBytes(Guid const & src, FeatStd::UInt8* dest, FeatStd::SizeType destSize);

            /**
            *  IsValid
            *  Checks if the Guid is set to any value else than "00000000-0000-0000-0000-000000000000"
            */
            bool IsNil() const;

            /**
            * Equals operator
            */
            inline bool operator==(const Guid & other) const {
                return (0 == FeatStd::Internal::Memory::Compare(m_data, other.m_data, GuidSize));
            }
            /**
            * Not-equals operator
            */
            inline bool operator!=(const Guid & other) const {
                return (0 != FeatStd::Internal::Memory::Compare(m_data, other.m_data, GuidSize));
            }
            /**
            * Less-than operator
            */
            bool operator<(const Guid & other) const;
            /**
            * Assignment operator
            */
            Guid& operator=(const Guid & other);

        private:
            enum {
                GuidSize = 16 //no size related typos shall occur. As enum because it does not increase code size.
            };
            FeatStd::UInt8 m_data[GuidSize]; //128 bit GUID 8-4-4-4-12 

        };
        /// @}
    }
}

#endif // GUID_H

