//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef FEATSTD_GENERICHANDLE_H
#define FEATSTD_GENERICHANDLE_H

#include <FeatStd/Container/Vector.h>
#include <FeatStd/Diagnostics/Log.h>
#include <FeatStd/Platform/Types.h>

namespace FeatStd
{

namespace Internal
{

/// @addtogroup FEATSTD_UTILS
/// @{

/**
 *  @brief An instance of the template class GenericHandle references an object of the given type.
 *
 *  GenericHandles are created and resolved to pointers through their associated GenericHandleFactory.
 *  Unlike FeatStd::SharedPointer, a GenericHandle referencing an object does not impact the lifetime
 *  of the object. Unlike raw pointers that are dangling once the object they point to is freed,
 *  a GenericHandle resolves to a null pointer once the object is freed and removed from the GenericHandleFactory.
 *
 *  Depending on the template parameters used, a GenericHandle is either 32 bit or 64 bit in size. Default size is 32 bit.
 *  Assigning GenericHandles incurs no additional costs in terms of performance or memory. Resolving a GenericHandle to a
 *  pointer is 2 if-statements and 2 reads from memory.
 *
 *  @tparam Type         The type of the object the GenericHandle is used to reference.
 *  @tparam IndexBits    The number of bits assigned inside the GenericHandle for referencing.
 *                       2^IndexBits unique handles (not to be confused with unlimited copies of unique handles)
 *                       can be used at the same time. Creating more handles triggers an error log message and
 *                       returns a null handle.
 *  @tparam CounterBits  The number of bits assigned inside the GenericHandle for counting its reuse.
 */
template<typename Type, UInt IndexBits, UInt CounterBits>
class GenericHandle
{
public:
    /**
     *  Constructor creating 'null' handle.
     */
    GenericHandle() : m_index(0), m_counter(0) {}

    /**
     *  Check if the handle is the designated 'null' handle.
     *  @return  True, if the handle is a 'null' handle.
     *           False, if the handle is not a 'null' handle.
     */
    bool IsNull() const { return ((m_index == 0) && (m_counter == 0)); }

    /**
     *  Compare the component handles for equality.
     *  @param other  The second component handle
     *  @return  True, if equal, false otherwise.
     */
    bool operator== (const GenericHandle& other) const { return ((m_index == other.m_index) && (m_counter == other.m_counter)); }

    /**
     *  Compare the component handles for inequality.
     *  @param other  The second component handle
     *  @return  True, if not equal, false otherwise.
     */
    bool operator!= (const GenericHandle& other) const { return ((m_index != other.m_index) || (m_counter != other.m_counter)); }

protected:
    /**
     *  Constructor
     */
    GenericHandle(UInt32 index, UInt32 counter) : m_index(index), m_counter(counter) {}

    /**
     *  Get the index that is encoded in the handle.
     */
    UInt32 GetIndex() const { return m_index; }

    /**
     *  Get the counter that is encoded in the handle.
     */
    UInt32 GetCounter() const { return m_counter; }

private:
    template<typename TypeAlt, UInt IndexBitsAlt, UInt CounterBitsAlt> friend class GenericHandleFactory; // exclusively creates non-null handles.

    UInt32 m_index : IndexBits;
    UInt32 m_counter : CounterBits;
};

/**
 *  @brief The template class GenericHandleFactory creates GenericHandles for pointers of Type.
 *
 *  GenericHandleFactory is designed to be used by classes that perform explicit, centralized lifetime management of the
 *  objects they create. As SharedPointers cannot be used (because the implicitly manage lifetime), and raw pointers are
 *  not safe, this is where GenericHandles come in. For details regarding the concept see the GenericHandle description.
 *
 *  Example of a class using GenericHandleFactory:
 *
 *  \code
 *  // Class generating Objects for use and exposing them via handles instead of pointers.
 *  // -----------------------------------------------------------------------------------
 *  class SomeObjectFactory {
 *  public:
 *      typedef FeatStd::Internal::GenericHandleFactory<Object> ObjectHandleFactory;
 *      typedef ObjectHandleFactory::Handle Handle;
 *
 *      Handle Create() { // Create an Object on the heap and return a handle to it.
 *          Object* object = FEATSTD_NEW(FeatStd::Int);
 *          if (0 != object) {
 *              return m_handleFactory.Create(object);
 *          }
 *          return Handle(); // Allocation failed, return null handle.
 *      }
 *
 *      bool Destroy(Handle handle) { // Destroy the Object on the heap that is identified by the given handle.
 *          Object* object = m_handleFactory.Get(handle);
 *          if (0 != object) {
 *              FEATSTD_DELETE(object);
 *              return m_handleFactory.Remove(handle);
 *          }
 *          return false; // Given handle is invalid.
 *      }
 *
 *      void DoSomething(Handle handle) {
 *          Object* object = m_handleFactory.Get(handle);
 *          if (0 != object) {
 *              // Do something with the object.
 *          }
 *          else {
 *          // Given handle is invalid.
 *          }
 *      }
 *
 *      private:
 *          ObjectHandleFactory m_handleFactory;
 *      };
 *
 *  // How a user would use the class.
 *  // -----------------------------------------------------------------------------------
 *  SomeObjectFactory objectFactory;
 *  SomeObjectFactory::Handle handleA = objectFactory.Create(); // Creates an Object.
 *  SomeObjectFactory::Handle handleB = handleA;                // Assign and store copies anywhere you want.
 *  objectFactory.DoSomething(handleA);                         // DoSomething with Object referenced by handleA.
 *  objectFactory.Destroy(handleA);                             // Object no longer exists, handleA (and handleB) are dangling.
 *  objectFactory.DoSomething(handleA);                         // Will not crash, because dangling handle is resolved to 0.
 *  \endcode
 *
 *  @tparam Type         The type of the object the GenericHandleFactory manages GenericHandles for.
 *  @tparam IndexBits    The number of bits assigned inside the GenericHandle for referencing.
 *                       2^IndexBits unique handles (not to be confused with unlimited copies of unique handles)
 *                       can be used at the same time. Creating more handles triggers an error log message and
 *                       returns a null handle.
 *  @tparam CounterBits  The number of bits assigned inside the GenericHandle for counting its reuse.
 */
template<typename Type, UInt IndexBits = 16, UInt CounterBits = 16>
class GenericHandleFactory
{
public:
    typedef GenericHandle<Type, IndexBits, CounterBits> Handle;

    /**
     *  Constructor
     */
    GenericHandleFactory();

    /**
     *  Creates a handle for the given object.
     *  @param object  The object to create the handle for.
     *  @return  The handle that was created for the object.
     */
    Handle Create(Type* object);

    /**
     *  Removes the handle of an object. After the operation this or any duplicate handles
     *  will be dangling, and resolve to a null pointer.
     *  @param handle  The handle to remove.
     *  @return  True, if the handle was removed. False, if the handle was invalid.
     */
    bool Remove(Handle handle);

    /**
     *  Get the pointer of an object from its handle.
     *  @param handle  The handle of the object to get the pointer of.
     *  @return  The pointer associated with the handle, or 0 if the handle is dangling.
     */
    inline Type* Get(const Handle handle) const;

private:
    FEATSTD_LOG_SET_REALM(FeatStd::Diagnostics::LogRealm::FeatStdSystem);

    struct GenericHandleEntry
    {
        explicit GenericHandleEntry(UInt32 nextFreeIndex, UInt32 counter = 0)
            :
            m_nextFreeIndex(nextFreeIndex),
            m_counter(counter),
            m_object(0)
        {}

        UInt32 m_nextFreeIndex : IndexBits;
        UInt32 m_counter : CounterBits;
        Type* m_object;
    };

    Internal::Vector<GenericHandleEntry> m_handleEntries;
    SizeType m_firstFreeHandleEntry;
};

template<typename Type, UInt IndexBits, UInt CounterBits>
GenericHandleFactory<Type, IndexBits, CounterBits>::GenericHandleFactory()
    :
    m_firstFreeHandleEntry(0)
{
    Create(static_cast<Type*>(0)); // create entry for null pointer.
}

template<typename Type, UInt IndexBits, UInt CounterBits>
typename GenericHandleFactory<Type, IndexBits, CounterBits>::Handle GenericHandleFactory<Type, IndexBits, CounterBits>::Create(Type* object)
{
    if (m_handleEntries.Size() <= m_firstFreeHandleEntry) {
        const UInt32 newIndex = FeatStd::Internal::NumericConversion<UInt32>(m_handleEntries.Size());
        const UInt32 nextFreeIndex = newIndex + 1;
        if (nextFreeIndex > (1 << IndexBits)) {
            FEATSTD_LOG_ERROR("GenericHandleFactory::Create() failed because the number of handles created exceeds the number "
                "of bits used to index them. The template is used with %d IndexBit(s) equaling %d unique handles. Increase the number of IndexBits!", IndexBits, (1 << IndexBits));
            return Handle();
        }

        GenericHandleEntry handleEntry(nextFreeIndex);
        handleEntry.m_object = object;
        m_firstFreeHandleEntry = nextFreeIndex;
        return (m_handleEntries.Add(handleEntry)) ? Handle(newIndex, 0) : Handle();
    }

    const UInt32 newIndex = FeatStd::Internal::NumericConversion<UInt32>(m_firstFreeHandleEntry);
    m_firstFreeHandleEntry = m_handleEntries[newIndex].m_nextFreeIndex;
    ++(m_handleEntries[newIndex].m_counter);
    m_handleEntries[newIndex].m_object = object;

    return Handle(newIndex, m_handleEntries[newIndex].m_counter);
}

template<typename Type, UInt IndexBits, UInt CounterBits>
bool GenericHandleFactory<Type, IndexBits, CounterBits>::Remove(Handle handle)
{
    const SizeType index = static_cast<SizeType>(handle.GetIndex());
    if ((0 == index) || (index >= m_handleEntries.Size()) ||
        (m_handleEntries[index].m_counter != handle.GetCounter()) ||
        (0 == m_handleEntries[index].m_object)) {
        return false;
    }

    m_handleEntries[index].m_nextFreeIndex = m_firstFreeHandleEntry;
    m_handleEntries[index].m_object = 0;
    m_firstFreeHandleEntry = index;
    return true;
}

template<typename Type, UInt IndexBits, UInt CounterBits> inline
Type* GenericHandleFactory<Type, IndexBits, CounterBits>::Get(const Handle handle) const
{
    if (handle.GetIndex() < FeatStd::Internal::NumericConversion<UInt32>(m_handleEntries.Size())) {
        const GenericHandleEntry& handleEntry = m_handleEntries[handle.GetIndex()];
        if (handleEntry.m_counter == handle.GetCounter()) {
            return handleEntry.m_object;
        }
    }

    return 0;
}

/// @}

}

}
#endif
