//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_FixedSizeString_h)
#define FeatStd_Util_FixedSizeString_h

#include <FeatStd/Base.h>

#include <FeatStd/Util/TextEncoding.h>

namespace FeatStd {
// forward declaration
template<UInt32 tcharCountMax> class FixedSizeStringBuffer;

namespace Internal {
/// @addtogroup FEATSTD_UTILS
/// @{
    template<typename T> inline const TChar* ToCStr(const T &object) {
        return object.CStr();
    }

    inline const Char* ToCStr(const TChar *object) {
        return object;
    }

    template<UInt32 tcharCountMax> class FixedSizeString {
        public:
            inline FixedSizeString() {
                *mString = 0;
            }

            inline FixedSizeString(const TChar *string, UInt32 codePointCount) {
                Init(string, codePointCount);
            }

            template<typename T> inline FixedSizeString(const T &object) {
                Init(ToCStr(object));
            }

            static inline UInt32 TCharCapacity() {
                return tcharCountMax;
            }

            inline void Clear() {
                *mString = 0;
            }

            inline bool IsEmpty() const {
                return TextEncoding::EndOfString(mString);
            }

            inline UInt32 CodePointCount() const {
                return TextEncoding::CodePointCount(mString);
            }

            inline const TChar* CStr() const {
                return mString;
            }

            template<typename T> inline FixedSizeString& operator=(const T &object) {
                if (static_cast<const void*>(this) != static_cast<const void*>(&object)) {
                    Init(ToCStr(object));
                }
                return *this;
            }

            template<typename T> inline bool operator==(const T &object) const {
                return TextEncoding::Match(mString, ToCStr(object));
            }

            template<typename T> inline bool operator!=(const T &object) const {
                return !TextEncoding::Match(mString, ToCStr(object));
            }

        private:
            template<UInt32 _tcharCountMax> friend class ::FeatStd::FixedSizeStringBuffer;
            TChar mString[tcharCountMax];

            FEATSTD_LINT_INITIALIZER_FUNCTION(FeatStd::FixedSizeString::Init)

            inline void Init(const TChar *string) {
                TextEncoding::Copy(mString, tcharCountMax, string);
            }

            inline void Init(const TChar *string, UInt32 codePointCount) {
                TextEncoding::Copy(mString, tcharCountMax, string, codePointCount);
            }
    };

    // rule out zero sized strings
    template<> class FixedSizeString<0>;
    // rule out because we need trailing zero, so string does not make sense.
    template<> class FixedSizeString<1>;
/// @}
}}
#endif
