//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_BinarySearch_h)
#define FeatStd_Util_BinarySearch_h

#include <FeatStd/Base.h>

namespace FeatStd { namespace Internal { 

    /** Generic compare function 
        compares left and right (requires operator== and operator<) and will return
        0 in equality, -1 if left < right, and 1 if left > right 
        @param left left value of comparison
        @param right right value of comparison
        @return 0 in equality, -1 if left < right, and 1 if left > right */
    template<typename Left, typename Right> struct StdComparer {
        static inline Int Compare(Left left, Right right) {
            return (left == right) ? 0 : ((left < right) ? -1 : 1);
        }
    };

    /** Generic compare function
        Compares key and value (requires operator== and operator<) and will return
        0 in equality, -1 if key < value, and 1 if key > value. Value is retrieved 
        by accessing (requires operator[]) the passed index of the passed collection.
        @param key The key to search for.
        @param index The current index calculated in the binary search.
        @param collection A pointer to the collection that is processed.
        @return 0 in equality, -1 if key < value, and 1 if key > value */
    template<typename Key, typename Index, typename Collection> struct StdIndexedComparer {
        static inline Int Compare(Key key, Index index, const Collection& collection) {
            Key value = (&collection)[index];
            return (value == key) ? 0 : ((key < value) ? -1 : 1);
        }
    };

    /** generic binary search algorithm 
        @tparam Comparer is a type exposing a static Compare function 
                The Compare function compares the key against an 
                item in the collection (see FeatStd::Internal::StdComparer)
        @tparam Key is the key type used in comparison 
        @tparam Index the index type that shall be used
        @tparam Collection must be an indexable collection
        @param key the key to search in the collection
        @param collection the collection to search
        @param itemCount the number of items in the collection
        @param insertAtIndex optional pointer that will receive the insert 
                position in case the element has not been found. In case the element
                has been found in the collection, insertAtIndex will be set to the
                index of the found item.
        @return if key is in the collection, the index of key is returned. 
                Otherwise itemCount is returned */
    template<typename Comparer, typename Key, typename Index, typename Collection>
    static Index BinarySearch(const Key &key, const Collection &collection, Index itemCount, Index *insertAtIndex = 0) {
        Int rc = -1;
        Index fromIndex = 0;
        Index toIndex = itemCount;
        while (fromIndex < toIndex && rc != 0) {
            Index half = (fromIndex + toIndex) / 2;

            rc = Comparer::Compare(key, collection[half]);
            if (rc > 0) {
                fromIndex = half + 1;
            }
            else {
                toIndex = half;
            }
        }

        if (insertAtIndex != 0) {
            *insertAtIndex = toIndex;
        }
        return (rc == 0) ? toIndex : itemCount;
    }

    /** generic binary search algorithm
    This function passes the key, index (instead of value) and pointer to the collection.
    This enables access to the collection. For instance this is useful if adjacent items have to
    be taken into account for the comparison.
    @tparam IndexedComparer is a type exposing a static Compare function
            The Compare function compares the key against an
            item in the collection (see FeatStd::Internal::StdIndexedComparer)
    @tparam Key is the key type used in comparison
    @tparam Index the index type that shall be used
    @tparam Collection must be an indexable collection
    @param key the key to search in the collection
    @param collection the collection to search
    @param itemCount the number of items in the collection
    @param insertAtIndex optional pointer that will receive the insert
            position in case the element has not been found. In case the element
            has been found in the collection, insertAtIndex will be set to the
            index of the found item.
    @return if key is in the collection, the index of key is returned.
            Otherwise itemCount is returned */
    template<typename IndexedComparer, typename Key, typename Index, typename Collection>
    static Index BinarySearchIndexed(const Key &key, const Collection &collection, Index itemCount, Index *insertAtIndex = 0) {
        Int rc = -1;
        Index fromIndex = 0;
        Index toIndex = itemCount;
        while (fromIndex < toIndex && rc != 0) {
            Index half = (fromIndex + toIndex) / 2;

            rc = IndexedComparer::Compare(key, half, collection);
            if (rc > 0) {
                fromIndex = half + 1;
            }
            else {
                toIndex = half;
            }
        }

        if (insertAtIndex != 0) {
            *insertAtIndex = toIndex;
        }
        return (rc == 0) ? toIndex : itemCount;
    }

}}
#endif
