//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Utils_ArrayOperation_h)
#define FeatStd_Utils_ArrayOperation_h

#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/Platform/Memory.h>
#include <FeatStd/MemoryManagement/TypeTraits.h>
#include <FeatStd/MemoryManagement/Operators.h>

namespace FeatStd { namespace Internal { namespace ArrayOperationPrivate {
/// @addtogroup FEATSTD_UTILS
/// @{

    // =========================================================================

    /** ArrayOperations for POD types. Implemented with Memory::Move */
    template<typename T, bool isPOD = FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType> struct Helper {

        /** construct dest using items from src
            @param dest the buffer to be used to create the array items
            @param src the array to get the init values for the array items in dest
            @param nItems the number of items
            @return pointer to constructed array */
        static inline T* ConstructArrayFrom(void *dest, const T *src, SizeType nItems) {
            FeatStd::Internal::Memory::Copy(dest, src, nItems * sizeof(T));
            return static_cast<T*>(dest);
        }

        static void InsertWithoutConstruction(T *arr, SizeType nItems, SizeType atIndex, const T &item) {
            FEATSTD_DEBUG_ASSERT(arr != 0);
            if (atIndex > nItems) {
                atIndex = nItems;
            }
            if (atIndex < nItems) {
                FeatStd::Internal::Memory::Move(arr + atIndex + 1, arr + atIndex, (nItems - atIndex) * sizeof(T));
            }
            arr[atIndex] = item;
        }

        static inline void InsertWithConstruction(T *arr, SizeType nItems, SizeType atIndex, const T &item) {
            InsertWithoutConstruction(arr, nItems, atIndex, item);
        }

        static void RemoveWithoutDestruction(T *arr, SizeType nItems, SizeType atIndex) {
            FEATSTD_DEBUG_ASSERT((arr != 0) && (nItems > 0));
            --nItems;
            if (atIndex < nItems) {
                FeatStd::Internal::Memory::Move(arr + atIndex, arr + atIndex + 1, (nItems - atIndex) * sizeof(T));
            }
        }

        static inline void RemoveWithDestruction(T *arr, SizeType nItems, SizeType atIndex) {
            RemoveWithoutDestruction(arr, nItems, atIndex);
        }

        /** copy array items from src to dest
            @param dest the copy target 
            @param src the copy source
            @param nItems the number of items to copy */
        static inline void Copy(T *dest, const T *src, SizeType nItems) {
            FeatStd::Internal::Memory::Copy(dest, src, nItems * sizeof(T));
        }
    };

    // =========================================================================

    /** ArrayOperations for non POD types. Implemented with loops */
    template<typename T> struct Helper<T, false> {

        static void InsertWithoutConstruction(T *arr, SizeType nItems, SizeType atIndex, const T &item) {
            FEATSTD_DEBUG_ASSERT(arr != 0);
            if (atIndex >= nItems) {
                // append item
                arr[nItems] = item;
            }
            else {
                // shift up items
                for (SizeType i = nItems; i > atIndex; --i) {
                    arr[i] = arr[i - 1];
                }
                // assign item it arr[atIndex]
                arr[atIndex] = item;
            }
        }

        static void InsertWithConstruction(T *arr, SizeType nItems, SizeType atIndex, const T &item) {
            FEATSTD_DEBUG_ASSERT(arr != 0);
            if (atIndex >= nItems) {
                // append by constructing arr[nItems]
                FeatStd::MemoryManagement::ConstructObject<T>(&arr[nItems], item);
            }
            else {
                // nItems must be > 0 here, otherwise we would have entered the if branch
                // construct arr[nItem] with last existing item in arr 
                FeatStd::MemoryManagement::ConstructObject<T>(&arr[nItems], arr[nItems - 1]);
                // shift up remainging items 
                for (SizeType i = (nItems - 1); i > atIndex; --i) {
                    arr[i] = arr[i - 1];
                }
                // assign item it arr[atIndex]
                arr[atIndex] = item;
            }
        }

        static void RemoveWithoutDestruction(T *arr, SizeType nItems, SizeType atIndex) {
            FEATSTD_DEBUG_ASSERT(arr != 0 && nItems > 0);
            if (atIndex < nItems) {
                // shift down items
                for (SizeType i = atIndex; i < (nItems - 1); ++i) {
                    arr[i] = arr[i + 1];
                }
            }
        }

        static void RemoveWithDestruction(T *arr, SizeType nItems, SizeType atIndex) {
            FEATSTD_DEBUG_ASSERT(arr != 0 && nItems > 0);
            if (atIndex < nItems) {
                // shift down items
                for (SizeType i = atIndex; i < (nItems - 1); ++i) {
                    arr[i] = arr[i + 1];
                }
                // destruct last item in array
                FeatStd::MemoryManagement::DestructObject<T>(&arr[nItems - 1]);
            }
        }

        /** copy array items from src to dest
            @param dest the copy target 
            @param src the copy source
            @param nItems the number of items to copy */
        static inline void Copy(T *dest, const T *src, SizeType nItems) {
            T *last = dest + nItems;
            while (dest != last) {
                *dest = *src;
                ++dest;
                ++src;
            }
        }

        /** construct dest using items from src
            @param dest the buffer to be used to create the array items
            @param src the array to get the init values for the array items in dest
            @param nItems the number of items
            @return pointer to constructed array */
        static inline T* ConstructArrayFrom(void *dest, const T *src, SizeType nItems) {
            T *obj = static_cast<T*>(dest);
            for (SizeType i = 0; i < nItems; ++i) {
                FeatStd::MemoryManagement::ConstructObject<T>(obj + i, src[i]);
            }
            return obj;
        }
    };

/// @}
}}}

namespace FeatStd { namespace Internal { namespace ArrayOperation {
/// @addtogroup FEATSTD_UTILS
/// @{

    /** Construct the given array dest with elements from src
        items in dest will get constructed with according item in src.
        @tparam T the array item type
        @param dest buffer to be used to construct the array items
        @param src the source array to be used to construct the items in dest
        @param nItems the number of items. It must be ensured src has at least nItems items and 
                      dest is capable to store at least nItems 
        @return pointer to first item created in dest */
    template<typename T> T* ConstructFrom(void *dest, const T *src, SizeType nItems) {
        T *arr;
        if (dest == 0) {
            // nothing to be done here
            arr = 0;
        }
        else if (src == 0) {
            // stability - create array with default values
            arr = FeatStd::MemoryManagement::ConstructArray<T>(dest, nItems);
        }
        else {
            arr = ArrayOperationPrivate::Helper<T, FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::ConstructArrayFrom(dest, src, nItems);
        }
        return arr;
    }

    /** insert the given item into the array at given position
        insert the given item at atIndex in the array arr. The array must be capable
        to store at least (nItems + 1) items. It is assumed, that array item arr[nItem]
        does not need construction and can be simply assigned a value to.
        - shift up all elements starting from atIndex to nItem - 1 by one item
        - arr[atIndex] = item
        @tparam T the array item type
        @param arr the C/C++ array to insert into 
        @param nItems the number of items in the array 
        @param atIndex zero based insertion index, if atIndex greater or equal to nItems,
                       the item will be insert at the end of the array (arr[nItem] = item)
        @param item that arr[insertAt] will receive */
    template<typename T> inline void InsertWithoutConstruction(T *arr, SizeType nItems, SizeType atIndex, const T &item) {
        ArrayOperationPrivate::Helper<T>::InsertWithoutConstruction(arr, nItems, atIndex, item);
    }

    /** insert the given item into the array at given position
        insert the given item at atIndex in the array arr. The array must be capable
        to store at least (nItems + 1) items. It is assumed, that array item arr[nItem]
        is not constructed.
        - construct arr[nItems]
        - shift up all elements starting from atIndex to nItem - 1 by one item
        - arr[atIndex] = item
        @tparam T the array item type
        @param arr the C/C++ array to insert into 
        @param nItems the number of items in the array 
        @param atIndex zero based insertion index, if atIndex greater or equal to nItems,
                       the item will be insert at the end of the array (arr[nItem] = item)
        @param item that arr[insertAt] will receive */
    template<typename T> void InsertWithConstruction(T *arr, SizeType nItems, SizeType atIndex, const T &item) {
        ArrayOperationPrivate::Helper<T>::InsertWithConstruction(arr, nItems, atIndex, item);
    }

    /** remove item at given position from the given array
        the array item arr[nItem - 1] will not be destructed.
        - shift down all items starting with atIndex + 1 to nItems - 1 by one item
        @tparam T the array item type
        @param arr the C/C++ array to remove the item from  
        @param nItems the size (maximum number of items) of the array 
        @param atIndex zero based remove index, if atIndex >= nItems, no item will be removed */
    template<typename T> inline void RemoveWithoutDestruction(T *arr, SizeType nItems, SizeType atIndex) {
        ArrayOperationPrivate::Helper<T>::RemoveWithoutDestruction(arr, nItems, atIndex);
    }

    /** remove item at given position from the given C/C++ array
        the array item arr[nItem - 1] will be destructed.
        - shift down all items starting with atIndex + 1 to nItems - 1 by one item
        - destruct arr[nItems - 1]
        @tparam T the array item type
        @param arr the C/C++ array to remove the item from  
        @param nItems the size (maximum number of items) of the array 
        @param atIndex zero based remove index, if atIndex >= nItems, no item will be removed */
    template<typename T> inline void RemoveWithDestruction(T *arr, SizeType nItems, SizeType atIndex) {
        ArrayOperationPrivate::Helper<T>::RemoveWithDestruction(arr, nItems, atIndex);
    }

    /** Copy items from array src to array dest
        it is assumed that dest is capabable to store at least nItems items and src has
        at least nItems items. The elements will be copied to dest with the assignment,
        thus items in dest *must* be constructed before.
        @tparam T the array item type
        @param dest the array receiving the items to copy
        @param src the source array 
        @param nItems the number of items. It must be ensured src has at least nItems items and 
                      dest is capable to store at least nItems */
    template<typename T> void CopyWithoutConstruction(T *dest, const T *src, SizeType nItems) {
        ArrayOperationPrivate::Helper<T>::Copy(dest, src, nItems);
    }

    /** Copy items from array src to array dest
        it is assumed that dest is capabable to store at least nItems items and src has
        at least nItems items. The elements will be copied to dest with the copy ctor,
        thus items in dest *must not* be constructed before.
        @tparam T the array item type
        @param dest the array receiving the items to copy
        @param src the source array 
        @param nItems the number of items. It must be ensured src has at least nItems items and 
                      dest is capable to store at least nItems */
    template<typename T> void CopyWithConstruction(T *dest, const T *src, SizeType nItems) {
        (void) ConstructFrom(dest, src, nItems);
    }

/// @}
}}}

#endif // FeatStd_Utils_ArrayOperation_h
