//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_Ticks_h)
#define FeatStd_Platform_Ticks_h

#include <FeatStd/Platform/Base.h>

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_PLATFORM
/// @{
    namespace TicksType {
        /** Ticks object type to create. */
        enum Enum {
            ///
            Zero,
            ///
            Now,
            ///
            Infinite
        };
    }
/// @}
}}

#include FEATSTD_PLATFORM_INCLUDE(Ticks)

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_PLATFORM
/// @{


    FEATSTD_LINT_SYMBOL(1739, FeatStd::Internal::Ticks::operator+,"informational message - left unchanged for backward compatibility reasons")
    FEATSTD_LINT_SYMBOL(1739, FeatStd::Internal::Ticks::operator-,"informational message - left unchanged for backward compatibility reasons")
    FEATSTD_LINT_SYMBOL(1739, FeatStd::Internal::Ticks::operator==,"informational message - left unchanged for backward compatibility reasons")
    FEATSTD_LINT_SYMBOL(1739, FeatStd::Internal::Ticks::operator!=,"informational message - left unchanged for backward compatibility reasons")
    FEATSTD_LINT_SYMBOL(1739, FeatStd::Internal::Ticks::operator>,"informational message - left unchanged for backward compatibility reasons")
    FEATSTD_LINT_SYMBOL(1739, FeatStd::Internal::Ticks::operator<,"informational message - left unchanged for backward compatibility reasons")
    FEATSTD_LINT_SYMBOL(1739, FeatStd::Internal::Ticks::operator>=,"informational message - left unchanged for backward compatibility reasons")
    FEATSTD_LINT_SYMBOL(1739, FeatStd::Internal::Ticks::operator<=,"informational message - left unchanged for backward compatibility reasons")

    /** Ticks is a conversion class between OS specific tick values and milliseconds. */
    class Ticks : private Impl::Ticks {
        typedef Impl::Ticks Base;
    public:
        /** Initialization with zero. */
        Ticks(TicksType::Enum ticksType = TicksType::Zero);

        /** Initialization with time in milliseconds.
            @param timeMs     Time in milliseconds */
        Ticks(UInt32 timeMs);

        /// Initialization with a TickValue object to set the tick value instead of milliseconds.
        explicit Ticks(const TickValue & ticks);

        /// Copy constructor
        Ticks(const Ticks & ticks);

        /// Assignment
        Ticks & operator =(const Ticks & ticks);

        /** Resets ticks object to zero. */
        void SetZero();

        /** Check if ticks object is zero.
            @return     <em>true</em> if ticks object is zero.
                        <em>false</em> if ticks object is greater than zero.*/
        bool IsZero() const;

        /** Resets ticks object to infinite. */
        void SetInfinite();

        /** Check if ticks object is zero.
            @return     <em>true</em> if ticks object is infinite.
                        <em>false</em> if ticks object is not infinite.*/
        bool IsInfinite() const;

        /** Returns expired time in milliseconds till now.
            @return     Milliseconds of expired time till now.*/
        UInt32 GetExpiredTime() const;

        /** Returns expired time in milliseconds till given ticks.
            @param ticks    Ticks value to calculate expired time from.
            @return         Milliseconds of expired time till ticks. */
        UInt32 GetExpiredTime(const Ticks & ticks) const;

        /** Returns the time representation of the ticks object in milliseconds.
            @return     Ticks representation in milliseconds. */
        UInt32 GetTime() const;

        /** Initializes the ticks object based on the given time in milliseconds.
            @param timeMs Time to set in milliseconds. */
        inline void SetTime(UInt32 timeMs);

        /** Returns the tick value of the Ticks object.
            @return     Tick value. */
        const TickValue & GetTicks() const;

        /** Initializes the ticks object based with the given tick value.
            @param ticks     Ticks to set. */
        void SetTicks(const TickValue & ticks);

        ///
        Ticks operator +(const Ticks & ticks) const;

        ///
        Ticks & operator +=(const Ticks & ticks) { 
            mTicks += ticks.mTicks; return *this; 
        }

        ///
        Ticks operator -(const Ticks & ticks) const;

        ///
        Ticks & operator -=(const Ticks & ticks) { 
            mTicks -= ticks.mTicks; return *this; 
        }

        ///
        bool operator ==(const Ticks & ticks) const { 
            return mTicks == ticks.mTicks; 
        }

        ///
        bool operator !=(const Ticks & ticks) const { 
            return mTicks != ticks.mTicks; 
        }

        ///
        bool operator >(const Ticks & ticks) const { 
            return mTicks > ticks.mTicks; 
        }

        ///
        bool operator <(const Ticks & ticks) const { 
            return mTicks < ticks.mTicks; 
        }

        ///
        bool operator >=(const Ticks & ticks) const { 
            return mTicks >= ticks.mTicks; 
        }

        ///
        bool operator <=(const Ticks & ticks) const { 
            return mTicks <= ticks.mTicks; 
        }

    private:
        TickValue mTicks;
    };

    /*  --------------------------------------------------------------------
        Ticks
        -------------------------------------------------------------------- */
    inline Ticks::Ticks(UInt32 timeMs)
    {
        SetTime(timeMs);
    }

    /*  --------------------------------------------------------------------
        Ticks
        -------------------------------------------------------------------- */
    inline Ticks::Ticks(const TickValue & ticks) : mTicks(ticks)
    {
    }

    /*  --------------------------------------------------------------------
        Ticks
        -------------------------------------------------------------------- */
    inline Ticks::Ticks(const Ticks & ticks) : mTicks(ticks.mTicks)
    {
    }

    /*  --------------------------------------------------------------------
        operator =
        -------------------------------------------------------------------- */
    inline Ticks & Ticks::operator =(const Ticks & ticks)
    {
        if (&ticks != this) {
            mTicks = ticks.mTicks;
        }
        return *this;
    }

    /*  --------------------------------------------------------------------
        SetZero
        -------------------------------------------------------------------- */
    inline void Ticks::SetZero()
    {
        mTicks = 0;
    }

    /*  --------------------------------------------------------------------
        IsZero
        -------------------------------------------------------------------- */
    inline bool Ticks::IsZero() const
    {
        return (mTicks == 0);
    }

    /*  --------------------------------------------------------------------
        SetInfinite
        -------------------------------------------------------------------- */
    inline void Ticks::SetInfinite()
    {
        mTicks = Base::Infinite();
    }

    /*  --------------------------------------------------------------------
        IsInfinite
        -------------------------------------------------------------------- */
    inline bool Ticks::IsInfinite() const
    {
        return mTicks == Base::Infinite();
    }

    /*  --------------------------------------------------------------------
        GetExpiredTime
        -------------------------------------------------------------------- */
    inline UInt32 Ticks::GetExpiredTime() const
    {
        return Base::Ticks2Time(Base::GetSystemTicks() - mTicks);
    }

    /*  --------------------------------------------------------------------
        GetExpiredTime
        -------------------------------------------------------------------- */
    inline UInt32 Ticks::GetExpiredTime(const Ticks & ticks) const
    {
        return Base::Ticks2Time(ticks.mTicks - mTicks);
    }

    /*  --------------------------------------------------------------------
        GetTime
        -------------------------------------------------------------------- */
    inline UInt32 Ticks::GetTime() const
    {
        return Base::Ticks2Time(mTicks);
    }

    /*  --------------------------------------------------------------------
        SetTime
        -------------------------------------------------------------------- */
    inline void Ticks::SetTime(UInt32 timeMs)
    {
        mTicks = Base::Time2Ticks(timeMs);
    }

    /*  --------------------------------------------------------------------
        GetTime
        -------------------------------------------------------------------- */
    inline const TickValue & Ticks::GetTicks() const
    {
        return mTicks;
    }

    /*  --------------------------------------------------------------------
        SetTime
        -------------------------------------------------------------------- */
    inline void Ticks::SetTicks(const TickValue & ticks)
    {
        mTicks = ticks;
    }

    /*  --------------------------------------------------------------------
        operator +
        -------------------------------------------------------------------- */
    inline Ticks Ticks::operator +(const Ticks & ticks) const
    {
        Ticks sum;
        sum.mTicks = mTicks + ticks.mTicks;
        return sum;
    }

    /*  --------------------------------------------------------------------
        operator -
        -------------------------------------------------------------------- */
    inline Ticks Ticks::operator -(const Ticks & ticks) const
    {
        Ticks lDiff;
        lDiff.mTicks = mTicks - ticks.mTicks;
        return lDiff;
    }
/// @}
}}
#endif
